"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlowLog = exports.FlowLogDestination = exports.FlowLogResourceType = exports.FlowLogDestinationType = exports.FlowLogTrafficType = void 0;
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * The type of VPC traffic to log
 *
 * @experimental
 */
var FlowLogTrafficType;
(function (FlowLogTrafficType) {
    /**
     * Only log accepts
     */
    FlowLogTrafficType["ACCEPT"] = "ACCEPT";
    /**
     * Log all requests
     */
    FlowLogTrafficType["ALL"] = "ALL";
    /**
     * Only log rejects
     */
    FlowLogTrafficType["REJECT"] = "REJECT";
})(FlowLogTrafficType = exports.FlowLogTrafficType || (exports.FlowLogTrafficType = {}));
/**
 * The available destination types for Flow Logs
 * @experimental
 */
var FlowLogDestinationType;
(function (FlowLogDestinationType) {
    /**
     * Send flow logs to CloudWatch Logs Group
     */
    FlowLogDestinationType["CLOUD_WATCH_LOGS"] = "cloud-watch-logs";
    /**
     * Send flow logs to S3 Bucket
     */
    FlowLogDestinationType["S3"] = "s3";
})(FlowLogDestinationType = exports.FlowLogDestinationType || (exports.FlowLogDestinationType = {}));
/**
 * The type of resource to create the flow log for
 *
 * @experimental
 */
class FlowLogResourceType {
    /**
     * The subnet to attach the Flow Log to
     */
    static fromSubnet(subnet) {
        return {
            resourceType: 'Subnet',
            resourceId: subnet.subnetId,
        };
    }
    /**
     * The VPC to attach the Flow Log to
     */
    static fromVpc(vpc) {
        return {
            resourceType: 'VPC',
            resourceId: vpc.vpcId,
        };
    }
    /**
     * The Network Interface to attach the Flow Log to
     */
    static fromNetworkInterfaceId(id) {
        return {
            resourceType: 'NetworkInterface',
            resourceId: id,
        };
    }
}
exports.FlowLogResourceType = FlowLogResourceType;
/**
 * The destination type for the flow log
 *
 * @experimental
 */
class FlowLogDestination {
    /**
     * Use CloudWatch logs as the destination
     */
    static toCloudWatchLogs(logGroup, iamRole) {
        return new CloudWatchLogsDestination({
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        });
    }
    /**
     * Use S3 as the destination
     */
    static toS3(bucket) {
        return new S3Destination({
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket: bucket,
        });
    }
}
exports.FlowLogDestination = FlowLogDestination;
/**
 * @experimental
 */
class S3Destination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let s3Bucket;
        if (this.props.s3Bucket === undefined) {
            s3Bucket = new s3.Bucket(scope, 'Bucket', {
                encryption: s3.BucketEncryption.UNENCRYPTED,
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
        }
        else {
            s3Bucket = this.props.s3Bucket;
        }
        return {
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket,
        };
    }
}
/**
 * @experimental
 */
class CloudWatchLogsDestination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let iamRole;
        let logGroup;
        if (this.props.iamRole === undefined) {
            iamRole = new iam.Role(scope, 'IAMRole', {
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
            });
        }
        else {
            iamRole = this.props.iamRole;
        }
        if (this.props.logGroup === undefined) {
            logGroup = new logs.LogGroup(scope, 'LogGroup');
        }
        else {
            logGroup = this.props.logGroup;
        }
        iamRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogStream',
                'logs:PutLogEvents',
                'logs:DescribeLogStreams',
            ],
            effect: iam.Effect.ALLOW,
            resources: [logGroup.logGroupArn],
        }));
        iamRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            effect: iam.Effect.ALLOW,
            resources: [iamRole.roleArn],
        }));
        return {
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        };
    }
}
/**
 * The base class for a Flow Log
 *
 * @experimental
 */
class FlowLogBase extends core_1.Resource {
}
/**
 * A VPC flow log.
 * @resource AWS::EC2::FlowLog
 *
 * @experimental
 */
class FlowLog extends FlowLogBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.flowLogName,
        });
        const destination = props.destination || FlowLogDestination.toCloudWatchLogs();
        const destinationConfig = destination.bind(this, this);
        this.logGroup = destinationConfig.logGroup;
        this.bucket = destinationConfig.s3Bucket;
        this.iamRole = destinationConfig.iamRole;
        const flowLog = new ec2_generated_1.CfnFlowLog(this, 'FlowLog', {
            deliverLogsPermissionArn: this.iamRole ? this.iamRole.roleArn : undefined,
            logDestinationType: destinationConfig.logDestinationType,
            logGroupName: this.logGroup ? this.logGroup.logGroupName : undefined,
            resourceId: props.resourceType.resourceId,
            resourceType: props.resourceType.resourceType,
            trafficType: props.trafficType
                ? props.trafficType
                : FlowLogTrafficType.ALL,
            logDestination: this.bucket ? this.bucket.bucketArn : undefined,
        });
        this.flowLogId = flowLog.ref;
    }
    /**
     * Import a Flow Log by it's Id
     */
    static fromFlowLogId(scope, id, flowLogId) {
        class Import extends FlowLogBase {
            constructor() {
                super(...arguments);
                this.flowLogId = flowLogId;
            }
        }
        return new Import(scope, id);
    }
}
exports.FlowLog = FlowLog;
//# sourceMappingURL=data:application/json;base64,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