"""
Just another Travis CI (Python) API client.

Provide the access to the env vars resource type.

Author
    Nissar Chababy, @funilrys, contactTATAfunilrysTODTODcom

Project link
    https://github.com/funilrys/PyTravisCI

License
    ::


        MIT License

        Copyright (c) 2019 Nissar Chababy

        Permission is hereby granted, free of charge, to any person obtaining a copy
        of this software and associated documentation files (the "Software"), to deal
        in the Software without restriction, including without limitation the rights
        to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
        copies of the Software, and to permit persons to whom the Software is
        furnished to do so, subject to the following conditions:

        The above copyright notice and this permission notice shall be included in all
        copies or substantial portions of the Software.

        THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
        IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
        FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
        AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
        LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
        OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
        SOFTWARE.
"""

from ..communication import Communication
from ..exceptions import MissingArgument, TravisCIError
from .env_var import EnvVar


class EnvVars(Communication):
    """
    Provide the list of environment variables of a given :code:`repo_id_or_slug`.

    Official Travis CI API documentation
        - https://developer.travis-ci.org/resource/env_vars#Env%20vars
        - https://developer.travis-ci.org/resource/env_vars#for_repository
        - https://developer.travis-ci.org/resource/env_vars#create

    :param root:
        An initiated instance of :class:`~PyTravisCI.TravisCI`.
    :type root: :class:`~PyTravisCI.TravisCI`
    :param repo_id_or_slug:
        A repository ID or slug to get the information for.

        Can be
            :code:`{repository.id}`
                Value uniquely identifying the repository.
            :code:`{repository.slug}`
                Same as :code:`{repository.owner.name}/{repository.name}`

        .. warning::
            The Travis CI API `states`_:

                If querying using the repository slug, it must be formatted using
                `standard URL encoding`_, including any special characters.

            We do not except nor want that that from you.
            In fact, we do the encoding for you.

    :type repo_id_or_slug: str,int

    :ivar env_vars: List of env_vars.
    :vartype env_vars:
        Generator[:class:`~PyTravisCI.resource_types.env_var.EnvVar`, :code:`None`, :code:`None`]

    :raise MissingArgument:
        When :code:`repo_id_or_slug` is not given or empty.
    :raise TravisCIError:
        When something went wrong while communicating,
        getting or extracting data from or with the API.

    .. _standard URL encoding: http://www.w3schools.com/tags/ref_urlencode.asp
    .. _states: https://developer.travis-ci.org/resource/env_vars#env%20vars
    """

    __path_name_base__ = "repo"

    # pylint: disable=protected-access
    _parameters = EnvVar._parameters
    """
    Provide the parameters we can parse to the query string.

    .. seealso::
        Documentation of the parameters
            :attr:`~PyTravisCI.resource_types.env_var.EnvVar._parameters`.
    """

    def __init__(self, root, repo_id_or_slug):
        super(EnvVars, self).__init__(root)

        if not repo_id_or_slug:
            raise MissingArgument("repo_id_or_slug")

        self.__repo_id_or_slug = repo_id_or_slug

        self.__standard_endpoint_url = self.bind_path_name_to_access_point(
            self.access_point,
            f"{self.__path_name_base__}/{self.format_slug(self.__repo_id_or_slug)}/env_vars",
        )

        self._endpoint_url = self.__standard_endpoint_url
        self.__fetch_env_vars()

    def __fetch_env_vars(self):
        """
        Fetch all available environment variables.
        """

        response = self.standardize.it(self.get_request(follow_next_page=True))

        self.response_to_attribute(
            self, {x: y for x, y in response.items() if x != "env_vars"}
        )
        self.repository = self._root.repository(repo_id_or_slug=self.__repo_id_or_slug)
        self.env_vars = (
            self._root.env_var(self.repository.id, x["id"])
            for x in response["env_vars"]
        )

    def create(self, data):
        """
        Creates an environment variable for the given :code:`repo_id_or_slug`.

        :param dict data:
            A dict with the desired parameters.

            .. seealso::
                Available parameters
                    :attr:`~PyTravisCI.resource_types.env_vars.EnvVars._parameters`

        :rtype: bool

        :raise TravisCIError:
            When something went wrong while communicating,
            getting or extracting data from or with the API.

            .. warning::
                You may get :code:`duplicate_resource`
                into :meth:`..exceptions.TravisCIError.error_type` if you
                try to create a variable name which already exists.
        """

        self.filter_parameters(data)
        self._parameters = self.remove_not_needed_parameters(self._parameters)

        self._endpoint_url = self.__standard_endpoint_url

        try:
            self.post_request(data=self._parameters)
        except TravisCIError as exception:
            if "Created" in exception.error_message():
                self.__fetch_env_vars()

                return True
            raise exception
        return False
