"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process_1 = require("child_process");
const os_1 = require("os");
const tasks_1 = require("../../tasks");
const util_1 = require("../util");
test('no tasks, no tasks file', () => {
    const p = new util_1.TestProject();
    expect(synthTasksManifest(p)).toBeUndefined();
});
test('empty task', () => {
    const p = new util_1.TestProject();
    // WHEN
    p.addTask('empty');
    // THEN
    expectManifest(p, {
        tasks: {
            empty: {
                name: 'empty',
            },
        },
    });
});
test('remove task', () => {
    const p = new util_1.TestProject();
    // WHEN
    const task = p.addTask('task1');
    p.addTask('task2');
    const removeTask = p.removeTask('task1');
    // THEN
    expect(removeTask).toEqual(task);
    expectManifest(p, {
        tasks: {
            task2: {
                name: 'task2',
            },
        },
    });
});
test('re-add removed task', () => {
    const p = new util_1.TestProject();
    // WHEN
    p.addTask('task1');
    p.addTask('task2');
    const removeTask = p.removeTask('task2');
    p.addTask('task2');
    // THEN
    expect(removeTask).toBeTruthy();
    expectManifest(p, {
        tasks: {
            task1: {
                name: 'task1',
            },
            task2: {
                name: 'task2',
            },
        },
    });
});
test('throw when removing a dependent task', () => {
    const p = new util_1.TestProject();
    // WHEN
    const primary = p.addTask('primary');
    const dependent = p.addTask('dependent');
    primary.spawn(dependent);
    // THEN
    expect(() => p.removeTask('dependent'))
        .toThrowError('Unable to remove task "dependent" because the following tasks depend on it: primary');
});
test('remove already removed task', () => {
    const p = new util_1.TestProject();
    expect(p.removeTask('task1')).toBe(undefined);
});
test('multiple "exec" commands', () => {
    const p = new util_1.TestProject();
    // WHEN
    const task = p.addTask('hello', {
        description: 'hello, world',
        exec: 'echo hello',
        env: {
            FOO: 'bar',
        },
    });
    task.exec('echo world');
    task.exec('echo "with quotes"');
    task.env('BAR', 'baz');
    // THEN
    expectManifest(p, {
        tasks: {
            hello: {
                name: 'hello',
                description: 'hello, world',
                env: {
                    FOO: 'bar',
                    BAR: 'baz',
                },
                steps: [
                    { exec: 'echo hello' },
                    { exec: 'echo world' },
                    { exec: 'echo "with quotes"' },
                ],
            },
        },
    });
});
test('subtasks', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const hello = p.addTask('hello', { exec: 'echo hello' });
    const world = p.addTask('world');
    // WHEN
    world.exec('echo "running hello"');
    world.spawn(hello);
    // THEN
    expectManifest(p, {
        tasks: {
            hello: {
                name: 'hello',
                steps: [{ exec: 'echo hello' }],
            },
            world: {
                name: 'world',
                steps: [
                    { exec: 'echo "running hello"' },
                    { spawn: 'hello' },
                ],
            },
        },
    });
});
test('reset() can be used to reset task steps', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const t0 = p.addTask('your-task');
    const t = p.addTask('my-task');
    t.exec('line1');
    t.spawn(t0);
    t.exec('line2');
    // WHEN
    t.reset('line3');
    t.exec('line4', { cwd: 'foo' });
    // THEN
    expectManifest(p, {
        tasks: {
            'your-task': {
                name: 'your-task',
            },
            'my-task': {
                name: 'my-task',
                steps: [
                    { exec: 'line3' },
                    { cwd: 'foo', exec: 'line4' },
                ],
            },
        },
    });
});
test('prependXXX() can be used to add steps from the top', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const sub = p.addTask('my-sub-task', { exec: 'subexec' });
    const t = p.addTask('my-task');
    t.exec('line1');
    // WHEN
    t.prependExec('line2');
    t.prependSpawn(sub);
    t.prependSay('message');
    // THEN
    expectManifest(p, {
        tasks: {
            'my-sub-task': {
                name: 'my-sub-task',
                steps: [
                    { exec: 'subexec' },
                ],
            },
            'my-task': {
                name: 'my-task',
                steps: [
                    { say: 'message' },
                    { spawn: 'my-sub-task' },
                    { exec: 'line2' },
                    { exec: 'line1' },
                ],
            },
        },
    });
});
test('env() can be used to add environment variables', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const t = p.addTask('my-task', {
        env: {
            INITIAL: '123',
            ENV: '456',
        },
    });
    // WHEN
    t.env('FOO', 'BAR');
    t.env('HELLO', 'world');
    // THEN
    expectManifest(p, {
        tasks: {
            'my-task': {
                name: 'my-task',
                env: {
                    INITIAL: '123',
                    ENV: '456',
                    FOO: 'BAR',
                    HELLO: 'world',
                },
            },
        },
    });
});
test('.steps can be used to list all steps in the current task', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const t0 = p.addTask('your');
    const t = p.addTask('my');
    t.exec('step1');
    t.exec('step2');
    t.exec('step3');
    t.spawn(t0);
    t.exec('step4');
    // WHEN
    const steps = t.steps;
    // THEN
    expect(steps).toStrictEqual([
        { exec: 'step1' },
        { exec: 'step2' },
        { exec: 'step3' },
        { spawn: 'your' },
        { exec: 'step4' },
    ]);
});
test('"condition" can be used to define a command that will determine if a task should be skipped', () => {
    // GIVEN
    const p = new util_1.TestProject();
    p.addTask('foo', {
        condition: 'false',
        exec: 'foo bar',
    });
    // THEN
    expectManifest(p, {
        tasks: {
            foo: {
                name: 'foo',
                condition: 'false',
                steps: [
                    { exec: 'foo bar' },
                ],
            },
        },
    });
});
describe('toShellCommand()', () => {
    test('single step', () => {
        // GIVEN
        const p = new util_1.TestProject();
        // WHEN
        const t = p.addTask('foo', { exec: 'echo hi there' });
        // THEN
        expect(t.toShellCommand()).toMatchSnapshot();
        expect(shell(t)).toStrictEqual(['hi there']);
    });
    test('with a name', () => {
        // GIVEN
        const p = new util_1.TestProject();
        const t = p.addTask('foo');
        // WHEN
        t.exec('echo step 1', { name: 'STEP I' });
        // THEN
        expect(t.toShellCommand()).toMatchSnapshot();
        expect(shell(t)).toStrictEqual([
            'STEP I',
            'step 1',
        ]);
    });
    test('with a condition', () => {
        // GIVEN
        const p = new util_1.TestProject();
        const t = p.addTask('foo', {
            condition: '[ "${RUNME}" = "1" ]',
        });
        // WHEN
        t.exec('echo hello, world', { name: 'STEP I' });
        // THEN
        expect(t.toShellCommand()).toMatchSnapshot();
        expect(shell(t)).toStrictEqual([]);
        expect(shell(t, { RUNME: '1' })).toStrictEqual([
            'STEP I',
            'hello, world',
        ]);
    });
    test('multiple steps', () => {
        // GIVEN
        const p = new util_1.TestProject();
        const t = p.addTask('foo');
        // WHEN
        t.exec('echo step 1', { name: 'STEP I' });
        t.exec('echo step 2');
        t.exec('echo step 3', { name: 'STEP III' });
        // THEN
        expect(t.toShellCommand()).toMatchSnapshot();
        expect(shell(t)).toStrictEqual([
            'STEP I',
            'step 1',
            'step 2',
            'STEP III',
            'step 3',
        ]);
    });
    test('subtasks', () => {
        // GIVEN
        const p = new util_1.TestProject();
        const t1 = p.addTask('t1');
        const t2 = p.addTask('t2');
        // WHEN
        t1.exec('echo task1-step1');
        t1.exec('echo task1-step2');
        t2.exec('echo task2-step1');
        t2.spawn(t1, { name: 'spawning t1' });
        t2.exec('echo task2-step3');
        // THEN
        expect(t2.toShellCommand()).toMatchSnapshot();
        expect(shell(t2)).toStrictEqual([
            'task2-step1',
            'spawning t1',
            'task1-step1',
            'task1-step2',
            'task2-step3',
        ]);
    });
    test('with environment', () => {
        // GIVEN
        const p = new util_1.TestProject();
        p.tasks.addEnvironment('FOO', 'hello'); // global environment
        // WHEN
        const t1 = p.addTask('t1', {
            env: {
                BAR: '$(echo world)',
            },
            exec: 'echo $FOO, $BAR !',
        });
        // THEN
        expect(t1.toShellCommand()).toMatchSnapshot();
        expect(shell(t1)).toStrictEqual([
            'hello, world !',
        ]);
    });
});
function shell(t, env = {}) {
    const result = child_process_1.spawnSync(t.toShellCommand(), { shell: true, env: { ...process.env, ...env } });
    if (result.status !== 0) {
        throw new Error(`non-zero exit code ${result.status}: ${result.stderr.toString('utf-8')}`);
    }
    return result.stdout.toString('utf-8').trim().split(os_1.EOL).filter(x => x).map(x => x.trim());
}
function expectManifest(p, toStrictEqual) {
    const manifest = synthTasksManifest(p);
    delete manifest['//'];
    expect(manifest).toStrictEqual(toStrictEqual);
}
function synthTasksManifest(p) {
    return util_1.synthSnapshot(p)[tasks_1.Tasks.MANIFEST_FILE];
    ;
}
//# sourceMappingURL=data:application/json;base64,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