"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppSyncTransformer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_appsync_1 = require("@aws-cdk/aws-appsync");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const schema_transformer_1 = require("./transformer/schema-transformer");
const defaultAuthorizationConfig = {
    defaultAuthorization: {
        authorizationType: aws_appsync_1.AuthorizationType.API_KEY,
        apiKeyConfig: {
            description: 'Auto generated API Key from construct',
            name: 'dev',
        },
    },
};
/**
 * (experimental) AppSyncTransformer Construct.
 *
 * @experimental
 */
class AppSyncTransformer extends core_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j;
        super(scope, id);
        this.props = props;
        this.tableMap = {};
        this.isSyncEnabled = props.syncEnabled ? props.syncEnabled : false;
        this.pointInTimeRecovery = (_b = props.enableDynamoPointInTimeRecovery) !== null && _b !== void 0 ? _b : false;
        const transformerConfiguration = {
            schemaPath: props.schemaPath,
            syncEnabled: (_c = props.syncEnabled) !== null && _c !== void 0 ? _c : false,
        };
        // Combine the arrays so we only loop once
        // Test each transformer to see if it implements ITransformer
        const allCustomTransformers = [...(_d = props.preCdkTransformers) !== null && _d !== void 0 ? _d : [], ...(_e = props.postCdkTransformers) !== null && _e !== void 0 ? _e : []];
        if (allCustomTransformers && allCustomTransformers.length > 0) {
            allCustomTransformers.forEach(transformer => {
                if (transformer && !this.implementsITransformer(transformer)) {
                    throw new Error(`Transformer does not implement ITransformer from graphql-transformer-core: ${transformer}`);
                }
            });
        }
        const transformer = new schema_transformer_1.SchemaTransformer(transformerConfiguration);
        this.outputs = transformer.transform(props.preCdkTransformers, props.postCdkTransformers);
        const resolvers = transformer.getResolvers();
        this.functionResolvers = (_f = this.outputs.functionResolvers) !== null && _f !== void 0 ? _f : {};
        // Remove any function resolvers from the total list of resolvers
        // Otherwise it will add them twice
        for (const [_, functionResolvers] of Object.entries(this.functionResolvers)) {
            functionResolvers.forEach((resolver) => {
                switch (resolver.typeName) {
                    case 'Query':
                    case 'Mutation':
                    case 'Subscription':
                        delete resolvers[resolver.fieldName];
                        break;
                }
            });
        }
        this.httpResolvers = (_g = this.outputs.httpResolvers) !== null && _g !== void 0 ? _g : {};
        // Remove any http resolvers from the total list of resolvers
        // Otherwise it will add them twice
        for (const [_, httpResolvers] of Object.entries(this.httpResolvers)) {
            httpResolvers.forEach((resolver) => {
                switch (resolver.typeName) {
                    case 'Query':
                    case 'Mutation':
                    case 'Subscription':
                        delete resolvers[resolver.fieldName];
                        break;
                }
            });
        }
        this.resolvers = resolvers;
        this.nestedAppsyncStack = new core_1.NestedStack(this, 'appsync-nested-stack');
        // AppSync
        this.appsyncAPI = new aws_appsync_1.GraphqlApi(this.nestedAppsyncStack, `${id}-api`, {
            name: props.apiName ? props.apiName : `${id}-api`,
            authorizationConfig: props.authorizationConfig
                ? props.authorizationConfig
                : defaultAuthorizationConfig,
            logConfig: {
                fieldLogLevel: props.fieldLogLevel
                    ? props.fieldLogLevel
                    : aws_appsync_1.FieldLogLevel.NONE,
            },
            schema: aws_appsync_1.Schema.fromAsset('./appsync/schema.graphql'),
            xrayEnabled: (_h = props.xrayEnabled) !== null && _h !== void 0 ? _h : false,
        });
        let tableData = (_j = this.outputs.cdkTables) !== null && _j !== void 0 ? _j : {};
        // Check to see if sync is enabled
        if (tableData.DataStore) {
            this.isSyncEnabled = true;
            this.syncTable = this.createSyncTable(tableData.DataStore);
            delete tableData.DataStore; // We don't want to create this again below so remove it from the tableData map
        }
        this.tableNameMap = this.createTablesAndResolvers(tableData, resolvers);
        if (this.outputs.noneResolvers) {
            this.createNoneDataSourceAndResolvers(this.outputs.noneResolvers, resolvers);
        }
        this.createHttpResolvers();
        // Outputs so we can generate exports
        new core_1.CfnOutput(scope, 'appsyncGraphQLEndpointOutput', {
            value: this.appsyncAPI.graphqlUrl,
            description: 'Output for aws_appsync_graphqlEndpoint',
        });
    }
    /**
     * graphql-transformer-core needs to be jsii enabled to pull the ITransformer interface correctly.
     * Since it's not in peer dependencies it doesn't show up in the jsii deps list.
     * Since it's not jsii enabled it has to be bundled.
     * The package can't be in BOTH peer and bundled dependencies
     * So we do a fake test to make sure it implements these and hope for the best
     * @param transformer
     */
    implementsITransformer(transformer) {
        return 'name' in transformer
            && 'directive' in transformer
            && 'typeDefinitions' in transformer;
    }
    /**
     * Creates NONE data source and associated resolvers
     * @param noneResolvers The resolvers that belong to the none data source
     * @param resolvers The resolver map minus function resolvers
     */
    createNoneDataSourceAndResolvers(noneResolvers, resolvers) {
        const noneDataSource = this.appsyncAPI.addNoneDataSource('NONE');
        Object.keys(noneResolvers).forEach((resolverKey) => {
            const resolver = resolvers[resolverKey];
            new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                api: this.appsyncAPI,
                typeName: resolver.typeName,
                fieldName: resolver.fieldName,
                dataSource: noneDataSource,
                requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
            });
        });
    }
    /**
     * Creates each dynamodb table, gsis, dynamodb datasource, and associated resolvers
     * If sync is enabled then TTL configuration is added
     * Returns tableName: table map in case it is needed for lambda functions, etc
     * @param tableData The CdkTransformer table information
     * @param resolvers The resolver map minus function resolvers
     */
    createTablesAndResolvers(tableData, resolvers) {
        const tableNameMap = {};
        Object.keys(tableData).forEach((tableKey) => {
            const table = this.createTable(tableData[tableKey]);
            this.tableMap[tableKey] = table;
            const dataSource = this.appsyncAPI.addDynamoDbDataSource(tableKey, table);
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appsync-datasource-deltasyncconfig.html
            if (this.isSyncEnabled && this.syncTable) {
                //@ts-ignore - ds is the base CfnDataSource and the db config needs to be versioned - see CfnDataSource
                dataSource.ds.dynamoDbConfig.versioned = true;
                //@ts-ignore - ds is the base CfnDataSource - see CfnDataSource
                dataSource.ds.dynamoDbConfig.deltaSyncConfig = {
                    baseTableTtl: '43200',
                    deltaSyncTableName: this.syncTable.tableName,
                    deltaSyncTableTtl: '30',
                };
                // Need to add permission for our datasource service role to access the sync table
                dataSource.grantPrincipal.addToPolicy(new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: [
                        'dynamodb:*',
                    ],
                    resources: [this.syncTable.tableArn],
                }));
            }
            const dynamoDbConfig = dataSource.ds
                .dynamoDbConfig;
            tableNameMap[tableKey] = dynamoDbConfig.tableName;
            // Loop the basic resolvers
            tableData[tableKey].resolvers.forEach((resolverKey) => {
                let resolver = resolvers[resolverKey];
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: dataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
                });
            });
            // Loop the gsi resolvers
            tableData[tableKey].gsiResolvers.forEach((resolverKey) => {
                let resolver = resolvers.gsi[resolverKey];
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: dataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
                });
            });
        });
        return tableNameMap;
    }
    createTable(tableData) {
        var _b;
        // I do not want to force people to pass `TypeTable` - this way they are only passing the @model Type name
        const modelTypeName = tableData.tableName.replace('Table', '');
        const streamSpecification = this.props.dynamoDbStreamConfig && this.props.dynamoDbStreamConfig[modelTypeName];
        const tableProps = {
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                name: tableData.partitionKey.name,
                type: this.convertAttributeType(tableData.partitionKey.type),
            },
            pointInTimeRecovery: this.pointInTimeRecovery,
            sortKey: tableData.sortKey && tableData.sortKey.name
                ? {
                    name: tableData.sortKey.name,
                    type: this.convertAttributeType(tableData.sortKey.type),
                } : undefined,
            timeToLiveAttribute: ((_b = tableData === null || tableData === void 0 ? void 0 : tableData.ttl) === null || _b === void 0 ? void 0 : _b.enabled) ? tableData.ttl.attributeName : undefined,
            stream: streamSpecification,
        };
        const table = new aws_dynamodb_1.Table(this.nestedAppsyncStack, tableData.tableName, tableProps);
        tableData.localSecondaryIndexes.forEach((lsi) => {
            table.addLocalSecondaryIndex({
                indexName: lsi.indexName,
                sortKey: {
                    name: lsi.sortKey.name,
                    type: this.convertAttributeType(lsi.sortKey.type),
                },
                projectionType: this.convertProjectionType(lsi.projection.ProjectionType),
            });
        });
        tableData.globalSecondaryIndexes.forEach((gsi) => {
            table.addGlobalSecondaryIndex({
                indexName: gsi.indexName,
                partitionKey: {
                    name: gsi.partitionKey.name,
                    type: this.convertAttributeType(gsi.partitionKey.type),
                },
                sortKey: gsi.sortKey && gsi.sortKey.name
                    ? {
                        name: gsi.sortKey.name,
                        type: this.convertAttributeType(gsi.sortKey.type),
                    } : undefined,
                projectionType: this.convertProjectionType(gsi.projection.ProjectionType),
            });
        });
        return table;
    }
    /**
     * Creates the sync table for Amplify DataStore
     * https://docs.aws.amazon.com/appsync/latest/devguide/conflict-detection-and-sync.html
     * @param tableData The CdkTransformer table information
     */
    createSyncTable(tableData) {
        var _b;
        return new aws_dynamodb_1.Table(this, 'appsync-api-sync-table', {
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                name: tableData.partitionKey.name,
                type: this.convertAttributeType(tableData.partitionKey.type),
            },
            sortKey: {
                name: tableData.sortKey.name,
                type: this.convertAttributeType(tableData.sortKey.type),
            },
            timeToLiveAttribute: ((_b = tableData.ttl) === null || _b === void 0 ? void 0 : _b.attributeName) || '_ttl',
        });
    }
    convertAttributeType(type) {
        switch (type) {
            case 'N':
                return aws_dynamodb_1.AttributeType.NUMBER;
            case 'B':
                return aws_dynamodb_1.AttributeType.BINARY;
            case 'S': // Same as default
            default:
                return aws_dynamodb_1.AttributeType.STRING;
        }
    }
    convertProjectionType(type) {
        switch (type) {
            case 'INCLUDE':
                return aws_dynamodb_1.ProjectionType.INCLUDE;
            case 'KEYS_ONLY':
                return aws_dynamodb_1.ProjectionType.KEYS_ONLY;
            case 'ALL': // Same as default
            default:
                return aws_dynamodb_1.ProjectionType.ALL;
        }
    }
    createHttpResolvers() {
        for (const [endpoint, httpResolvers] of Object.entries(this.httpResolvers)) {
            const strippedEndpoint = endpoint.replace(/[^_0-9A-Za-z]/g, '');
            const httpDataSource = this.appsyncAPI.addHttpDataSource(`${strippedEndpoint}`, endpoint);
            httpResolvers.forEach((resolver) => {
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: httpDataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultRequestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultResponseMappingTemplate),
                });
            });
        }
    }
    /**
     * (experimental) Adds the function as a lambdaDataSource to the AppSync api Adds all of the functions resolvers to the AppSync api.
     *
     * @param functionName The function name specified in the.
     * @param id The id to give.
     * @param lambdaFunction The lambda function to attach.
     * @experimental
     * @function directive of the schema
     */
    addLambdaDataSourceAndResolvers(functionName, id, lambdaFunction, options) {
        const functionDataSource = this.appsyncAPI.addLambdaDataSource(id, lambdaFunction, options);
        for (const resolver of this.functionResolvers[functionName]) {
            new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                api: this.appsyncAPI,
                typeName: resolver.typeName,
                fieldName: resolver.fieldName,
                dataSource: functionDataSource,
                requestMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultRequestMappingTemplate),
                responseMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultResponseMappingTemplate),
            });
        }
        return functionDataSource;
    }
    /**
     * (experimental) Adds a stream to the dynamodb table associated with the type.
     *
     * @returns string - the stream arn token
     * @experimental
     */
    addDynamoDBStream(props) {
        const tableName = `${props.modelTypeName}Table`;
        const table = this.tableMap[tableName];
        if (!table)
            throw new Error(`Table with name '${tableName}' not found.`);
        const cfnTable = table.node.defaultChild;
        cfnTable.streamSpecification = {
            streamViewType: props.streamViewType,
        };
        return cfnTable.attrStreamArn;
    }
}
exports.AppSyncTransformer = AppSyncTransformer;
_a = JSII_RTTI_SYMBOL_1;
AppSyncTransformer[_a] = { fqn: "cdk-appsync-transformer.AppSyncTransformer", version: "1.77.14" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwc3luYy10cmFuc2Zvcm1lci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9hcHBzeW5jLXRyYW5zZm9ybWVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsc0RBVzhCO0FBRTlCLHdEQVErQjtBQUMvQiw4Q0FBMkQ7QUFFM0Qsd0NBQWtFO0FBVWxFLHlFQUcwQztBQXFFMUMsTUFBTSwwQkFBMEIsR0FBd0I7SUFDdEQsb0JBQW9CLEVBQUU7UUFDcEIsaUJBQWlCLEVBQUUsK0JBQWlCLENBQUMsT0FBTztRQUM1QyxZQUFZLEVBQUU7WUFDWixXQUFXLEVBQUUsdUNBQXVDO1lBQ3BELElBQUksRUFBRSxLQUFLO1NBQ1o7S0FDRjtDQUNGLENBQUM7Ozs7OztBQUtGLE1BQWEsa0JBQW1CLFNBQVEsZ0JBQVM7Ozs7SUFpRC9DLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBOEI7O1FBQ3RFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUM7UUFDbkIsSUFBSSxDQUFDLFFBQVEsR0FBRyxFQUFFLENBQUM7UUFDbkIsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUM7UUFDbkUsSUFBSSxDQUFDLG1CQUFtQixTQUFHLEtBQUssQ0FBQywrQkFBK0IsbUNBQUksS0FBSyxDQUFDO1FBRTFFLE1BQU0sd0JBQXdCLEdBQTJCO1lBQ3ZELFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtZQUM1QixXQUFXLFFBQUUsS0FBSyxDQUFDLFdBQVcsbUNBQUksS0FBSztTQUN4QyxDQUFDO1FBRUYsMENBQTBDO1FBQzFDLDZEQUE2RDtRQUM3RCxNQUFNLHFCQUFxQixHQUFHLENBQUMsU0FBRyxLQUFLLENBQUMsa0JBQWtCLG1DQUFJLEVBQUUsRUFBRSxTQUFHLEtBQUssQ0FBQyxtQkFBbUIsbUNBQUksRUFBRSxDQUFDLENBQUM7UUFDdEcsSUFBSSxxQkFBcUIsSUFBSSxxQkFBcUIsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQzdELHFCQUFxQixDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsRUFBRTtnQkFDMUMsSUFBSSxXQUFXLElBQUksQ0FBQyxJQUFJLENBQUMsc0JBQXNCLENBQUMsV0FBVyxDQUFDLEVBQUU7b0JBQzVELE1BQU0sSUFBSSxLQUFLLENBQUMsOEVBQThFLFdBQVcsRUFBRSxDQUFDLENBQUM7aUJBQzlHO1lBQ0gsQ0FBQyxDQUFDLENBQUM7U0FDSjtRQUVELE1BQU0sV0FBVyxHQUFHLElBQUksc0NBQWlCLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNwRSxJQUFJLENBQUMsT0FBTyxHQUFHLFdBQVcsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQzFGLE1BQU0sU0FBUyxHQUFHLFdBQVcsQ0FBQyxZQUFZLEVBQUUsQ0FBQztRQUU3QyxJQUFJLENBQUMsaUJBQWlCLFNBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsbUNBQUksRUFBRSxDQUFDO1FBRTlELGlFQUFpRTtRQUNqRSxtQ0FBbUM7UUFDbkMsS0FBSyxNQUFNLENBQUMsQ0FBQyxFQUFFLGlCQUFpQixDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FDakQsSUFBSSxDQUFDLGlCQUFpQixDQUN2QixFQUFFO1lBQ0QsaUJBQWlCLENBQUMsT0FBTyxDQUFDLENBQUMsUUFBYSxFQUFFLEVBQUU7Z0JBQzFDLFFBQVEsUUFBUSxDQUFDLFFBQVEsRUFBRTtvQkFDekIsS0FBSyxPQUFPLENBQUM7b0JBQ2IsS0FBSyxVQUFVLENBQUM7b0JBQ2hCLEtBQUssY0FBYzt3QkFDakIsT0FBTyxTQUFTLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDO3dCQUNyQyxNQUFNO2lCQUNUO1lBQ0gsQ0FBQyxDQUFDLENBQUM7U0FDSjtRQUVELElBQUksQ0FBQyxhQUFhLFNBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLG1DQUFJLEVBQUUsQ0FBQztRQUV0RCw2REFBNkQ7UUFDN0QsbUNBQW1DO1FBQ25DLEtBQUssTUFBTSxDQUFDLENBQUMsRUFBRSxhQUFhLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsRUFBRTtZQUNuRSxhQUFhLENBQUMsT0FBTyxDQUFDLENBQUMsUUFBYSxFQUFFLEVBQUU7Z0JBQ3RDLFFBQVEsUUFBUSxDQUFDLFFBQVEsRUFBRTtvQkFDekIsS0FBSyxPQUFPLENBQUM7b0JBQ2IsS0FBSyxVQUFVLENBQUM7b0JBQ2hCLEtBQUssY0FBYzt3QkFDakIsT0FBTyxTQUFTLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDO3dCQUNyQyxNQUFNO2lCQUNUO1lBQ0gsQ0FBQyxDQUFDLENBQUM7U0FDSjtRQUVELElBQUksQ0FBQyxTQUFTLEdBQUcsU0FBUyxDQUFDO1FBRTNCLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLGtCQUFXLENBQUMsSUFBSSxFQUFFLHNCQUFzQixDQUFDLENBQUM7UUFFeEUsVUFBVTtRQUNWLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSx3QkFBVSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxHQUFHLEVBQUUsTUFBTSxFQUFFO1lBQ3JFLElBQUksRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsTUFBTTtZQUNqRCxtQkFBbUIsRUFBRSxLQUFLLENBQUMsbUJBQW1CO2dCQUM1QyxDQUFDLENBQUMsS0FBSyxDQUFDLG1CQUFtQjtnQkFDM0IsQ0FBQyxDQUFDLDBCQUEwQjtZQUM5QixTQUFTLEVBQUU7Z0JBQ1QsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhO29CQUNoQyxDQUFDLENBQUMsS0FBSyxDQUFDLGFBQWE7b0JBQ3JCLENBQUMsQ0FBQywyQkFBYSxDQUFDLElBQUk7YUFDdkI7WUFDRCxNQUFNLEVBQUUsb0JBQU0sQ0FBQyxTQUFTLENBQUMsMEJBQTBCLENBQUM7WUFDcEQsV0FBVyxRQUFFLEtBQUssQ0FBQyxXQUFXLG1DQUFJLEtBQUs7U0FDeEMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxTQUFTLFNBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLG1DQUFJLEVBQUUsQ0FBQztRQUU3QyxrQ0FBa0M7UUFDbEMsSUFBSSxTQUFTLENBQUMsU0FBUyxFQUFFO1lBQ3ZCLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxDQUFDO1lBQzFCLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUM7WUFDM0QsT0FBTyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUMsK0VBQStFO1NBQzVHO1FBRUQsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3hFLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLEVBQUU7WUFDOUIsSUFBSSxDQUFDLGdDQUFnQyxDQUNuQyxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsRUFDMUIsU0FBUyxDQUNWLENBQUM7U0FDSDtRQUNELElBQUksQ0FBQyxtQkFBbUIsRUFBRSxDQUFDO1FBRTNCLHFDQUFxQztRQUNyQyxJQUFJLGdCQUFTLENBQUMsS0FBSyxFQUFFLDhCQUE4QixFQUFFO1lBQ25ELEtBQUssRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLFVBQVU7WUFDakMsV0FBVyxFQUFFLHdDQUF3QztTQUN0RCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNLLHNCQUFzQixDQUFDLFdBQWdCO1FBQzdDLE9BQU8sTUFBTSxJQUFJLFdBQVc7ZUFDdkIsV0FBVyxJQUFJLFdBQVc7ZUFDMUIsaUJBQWlCLElBQUksV0FBVyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssZ0NBQWdDLENBQ3RDLGFBQXlELEVBQ3pELFNBQWM7UUFFZCxNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLGlCQUFpQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBRWpFLE1BQU0sQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsV0FBZ0IsRUFBRSxFQUFFO1lBQ3RELE1BQU0sUUFBUSxHQUFHLFNBQVMsQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUN4QyxJQUFJLHNCQUFRLENBQ1YsSUFBSSxDQUFDLGtCQUFrQixFQUN2QixHQUFHLFFBQVEsQ0FBQyxRQUFRLElBQUksUUFBUSxDQUFDLFNBQVMsV0FBVyxFQUNyRDtnQkFDRSxHQUFHLEVBQUUsSUFBSSxDQUFDLFVBQVU7Z0JBQ3BCLFFBQVEsRUFBRSxRQUFRLENBQUMsUUFBUTtnQkFDM0IsU0FBUyxFQUFFLFFBQVEsQ0FBQyxTQUFTO2dCQUM3QixVQUFVLEVBQUUsY0FBYztnQkFDMUIsc0JBQXNCLEVBQUUsNkJBQWUsQ0FBQyxRQUFRLENBQzlDLFFBQVEsQ0FBQyxzQkFBc0IsQ0FDaEM7Z0JBQ0QsdUJBQXVCLEVBQUUsNkJBQWUsQ0FBQyxRQUFRLENBQy9DLFFBQVEsQ0FBQyx1QkFBdUIsQ0FDakM7YUFDRixDQUNGLENBQUM7UUFDSixDQUFDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSyx3QkFBd0IsQ0FDOUIsU0FBa0QsRUFDbEQsU0FBYztRQUVkLE1BQU0sWUFBWSxHQUFRLEVBQUUsQ0FBQztRQUU3QixNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFFBQWEsRUFBRSxFQUFFO1lBQy9DLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7WUFDcEQsSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsR0FBRyxLQUFLLENBQUM7WUFFaEMsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFFMUUsd0hBQXdIO1lBRXhILElBQUksSUFBSSxDQUFDLGFBQWEsSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFO2dCQUN4Qyx1R0FBdUc7Z0JBQ3ZHLFVBQVUsQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUM7Z0JBRTlDLCtEQUErRDtnQkFDL0QsVUFBVSxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsZUFBZSxHQUFHO29CQUM3QyxZQUFZLEVBQUUsT0FBTztvQkFDckIsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxTQUFTO29CQUM1QyxpQkFBaUIsRUFBRSxJQUFJO2lCQUN4QixDQUFDO2dCQUVGLGtGQUFrRjtnQkFDbEYsVUFBVSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQ25DLElBQUkseUJBQWUsQ0FBQztvQkFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsS0FBSztvQkFDcEIsT0FBTyxFQUFFO3dCQUNQLFlBQVk7cUJBQ2I7b0JBQ0QsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUM7aUJBQ3JDLENBQUMsQ0FDSCxDQUFDO2FBQ0g7WUFFRCxNQUFNLGNBQWMsR0FBRyxVQUFVLENBQUMsRUFBRTtpQkFDakMsY0FBc0QsQ0FBQztZQUMxRCxZQUFZLENBQUMsUUFBUSxDQUFDLEdBQUcsY0FBYyxDQUFDLFNBQVMsQ0FBQztZQUVsRCwyQkFBMkI7WUFDM0IsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxXQUFnQixFQUFFLEVBQUU7Z0JBQ3pELElBQUksUUFBUSxHQUFHLFNBQVMsQ0FBQyxXQUFXLENBQUMsQ0FBQztnQkFDdEMsSUFBSSxzQkFBUSxDQUNWLElBQUksQ0FBQyxrQkFBa0IsRUFDdkIsR0FBRyxRQUFRLENBQUMsUUFBUSxJQUFJLFFBQVEsQ0FBQyxTQUFTLFdBQVcsRUFDckQ7b0JBQ0UsR0FBRyxFQUFFLElBQUksQ0FBQyxVQUFVO29CQUNwQixRQUFRLEVBQUUsUUFBUSxDQUFDLFFBQVE7b0JBQzNCLFNBQVMsRUFBRSxRQUFRLENBQUMsU0FBUztvQkFDN0IsVUFBVSxFQUFFLFVBQVU7b0JBQ3RCLHNCQUFzQixFQUFFLDZCQUFlLENBQUMsUUFBUSxDQUM5QyxRQUFRLENBQUMsc0JBQXNCLENBQ2hDO29CQUNELHVCQUF1QixFQUFFLDZCQUFlLENBQUMsUUFBUSxDQUMvQyxRQUFRLENBQUMsdUJBQXVCLENBQ2pDO2lCQUNGLENBQ0YsQ0FBQztZQUNKLENBQUMsQ0FBQyxDQUFDO1lBRUgseUJBQXlCO1lBQ3pCLFNBQVMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUMsV0FBZ0IsRUFBRSxFQUFFO2dCQUM1RCxJQUFJLFFBQVEsR0FBRyxTQUFTLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxDQUFDO2dCQUMxQyxJQUFJLHNCQUFRLENBQ1YsSUFBSSxDQUFDLGtCQUFrQixFQUN2QixHQUFHLFFBQVEsQ0FBQyxRQUFRLElBQUksUUFBUSxDQUFDLFNBQVMsV0FBVyxFQUNyRDtvQkFDRSxHQUFHLEVBQUUsSUFBSSxDQUFDLFVBQVU7b0JBQ3BCLFFBQVEsRUFBRSxRQUFRLENBQUMsUUFBUTtvQkFDM0IsU0FBUyxFQUFFLFFBQVEsQ0FBQyxTQUFTO29CQUM3QixVQUFVLEVBQUUsVUFBVTtvQkFDdEIsc0JBQXNCLEVBQUUsNkJBQWUsQ0FBQyxRQUFRLENBQzlDLFFBQVEsQ0FBQyxzQkFBc0IsQ0FDaEM7b0JBQ0QsdUJBQXVCLEVBQUUsNkJBQWUsQ0FBQyxRQUFRLENBQy9DLFFBQVEsQ0FBQyx1QkFBdUIsQ0FDakM7aUJBQ0YsQ0FDRixDQUFDO1lBQ0osQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILE9BQU8sWUFBWSxDQUFDO0lBQ3RCLENBQUM7SUFFTyxXQUFXLENBQUMsU0FBOEI7O1FBQ2hELDBHQUEwRztRQUMxRyxNQUFNLGFBQWEsR0FBRyxTQUFTLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDL0QsTUFBTSxtQkFBbUIsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLG9CQUFvQixJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsb0JBQW9CLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDOUcsTUFBTSxVQUFVLEdBQWU7WUFDN0IsV0FBVyxFQUFFLDBCQUFXLENBQUMsZUFBZTtZQUN4QyxZQUFZLEVBQUU7Z0JBQ1osSUFBSSxFQUFFLFNBQVMsQ0FBQyxZQUFZLENBQUMsSUFBSTtnQkFDakMsSUFBSSxFQUFFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQzthQUM3RDtZQUNELG1CQUFtQixFQUFFLElBQUksQ0FBQyxtQkFBbUI7WUFDN0MsT0FBTyxFQUFFLFNBQVMsQ0FBQyxPQUFPLElBQUksU0FBUyxDQUFDLE9BQU8sQ0FBQyxJQUFJO2dCQUNsRCxDQUFDLENBQUM7b0JBQ0EsSUFBSSxFQUFFLFNBQVMsQ0FBQyxPQUFPLENBQUMsSUFBSTtvQkFDNUIsSUFBSSxFQUFFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQztpQkFDeEQsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUNmLG1CQUFtQixFQUFFLE9BQUEsU0FBUyxhQUFULFNBQVMsdUJBQVQsU0FBUyxDQUFFLEdBQUcsMENBQUUsT0FBTyxFQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUN0RixNQUFNLEVBQUUsbUJBQW1CO1NBQzVCLENBQUM7UUFFRixNQUFNLEtBQUssR0FBRyxJQUFJLG9CQUFLLENBQ3JCLElBQUksQ0FBQyxrQkFBa0IsRUFDdkIsU0FBUyxDQUFDLFNBQVMsRUFDbkIsVUFBVSxDQUNYLENBQUM7UUFFRixTQUFTLENBQUMscUJBQXFCLENBQUMsT0FBTyxDQUFDLENBQUMsR0FBUSxFQUFFLEVBQUU7WUFDbkQsS0FBSyxDQUFDLHNCQUFzQixDQUFDO2dCQUMzQixTQUFTLEVBQUUsR0FBRyxDQUFDLFNBQVM7Z0JBQ3hCLE9BQU8sRUFBRTtvQkFDUCxJQUFJLEVBQUUsR0FBRyxDQUFDLE9BQU8sQ0FBQyxJQUFJO29CQUN0QixJQUFJLEVBQUUsSUFBSSxDQUFDLG9CQUFvQixDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDO2lCQUNsRDtnQkFDRCxjQUFjLEVBQUUsSUFBSSxDQUFDLHFCQUFxQixDQUN4QyxHQUFHLENBQUMsVUFBVSxDQUFDLGNBQWMsQ0FDOUI7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILFNBQVMsQ0FBQyxzQkFBc0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxHQUFRLEVBQUUsRUFBRTtZQUNwRCxLQUFLLENBQUMsdUJBQXVCLENBQUM7Z0JBQzVCLFNBQVMsRUFBRSxHQUFHLENBQUMsU0FBUztnQkFDeEIsWUFBWSxFQUFFO29CQUNaLElBQUksRUFBRSxHQUFHLENBQUMsWUFBWSxDQUFDLElBQUk7b0JBQzNCLElBQUksRUFBRSxJQUFJLENBQUMsb0JBQW9CLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUM7aUJBQ3ZEO2dCQUNELE9BQU8sRUFBRSxHQUFHLENBQUMsT0FBTyxJQUFJLEdBQUcsQ0FBQyxPQUFPLENBQUMsSUFBSTtvQkFDdEMsQ0FBQyxDQUFDO3dCQUNBLElBQUksRUFBRSxHQUFHLENBQUMsT0FBTyxDQUFDLElBQUk7d0JBQ3RCLElBQUksRUFBRSxJQUFJLENBQUMsb0JBQW9CLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUM7cUJBQ2xELENBQUMsQ0FBQyxDQUFDLFNBQVM7Z0JBQ2YsY0FBYyxFQUFFLElBQUksQ0FBQyxxQkFBcUIsQ0FDeEMsR0FBRyxDQUFDLFVBQVUsQ0FBQyxjQUFjLENBQzlCO2FBQ0YsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssZUFBZSxDQUFDLFNBQThCOztRQUNwRCxPQUFPLElBQUksb0JBQUssQ0FBQyxJQUFJLEVBQUUsd0JBQXdCLEVBQUU7WUFDL0MsV0FBVyxFQUFFLDBCQUFXLENBQUMsZUFBZTtZQUN4QyxZQUFZLEVBQUU7Z0JBQ1osSUFBSSxFQUFFLFNBQVMsQ0FBQyxZQUFZLENBQUMsSUFBSTtnQkFDakMsSUFBSSxFQUFFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQzthQUM3RDtZQUNELE9BQU8sRUFBRTtnQkFDUCxJQUFJLEVBQUUsU0FBUyxDQUFDLE9BQVEsQ0FBQyxJQUFJO2dCQUM3QixJQUFJLEVBQUUsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFNBQVMsQ0FBQyxPQUFRLENBQUMsSUFBSSxDQUFDO2FBQ3pEO1lBQ0QsbUJBQW1CLEVBQUUsT0FBQSxTQUFTLENBQUMsR0FBRywwQ0FBRSxhQUFhLEtBQUksTUFBTTtTQUM1RCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sb0JBQW9CLENBQUMsSUFBWTtRQUN2QyxRQUFRLElBQUksRUFBRTtZQUNaLEtBQUssR0FBRztnQkFDTixPQUFPLDRCQUFhLENBQUMsTUFBTSxDQUFDO1lBQzlCLEtBQUssR0FBRztnQkFDTixPQUFPLDRCQUFhLENBQUMsTUFBTSxDQUFDO1lBQzlCLEtBQUssR0FBRyxDQUFDLENBQUMsa0JBQWtCO1lBQzVCO2dCQUNFLE9BQU8sNEJBQWEsQ0FBQyxNQUFNLENBQUM7U0FDL0I7SUFDSCxDQUFDO0lBRU8scUJBQXFCLENBQUMsSUFBWTtRQUN4QyxRQUFRLElBQUksRUFBRTtZQUNaLEtBQUssU0FBUztnQkFDWixPQUFPLDZCQUFjLENBQUMsT0FBTyxDQUFDO1lBQ2hDLEtBQUssV0FBVztnQkFDZCxPQUFPLDZCQUFjLENBQUMsU0FBUyxDQUFDO1lBQ2xDLEtBQUssS0FBSyxDQUFDLENBQUMsa0JBQWtCO1lBQzlCO2dCQUNFLE9BQU8sNkJBQWMsQ0FBQyxHQUFHLENBQUM7U0FDN0I7SUFDSCxDQUFDO0lBRU8sbUJBQW1CO1FBQ3pCLEtBQUssTUFBTSxDQUFDLFFBQVEsRUFBRSxhQUFhLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUNwRCxJQUFJLENBQUMsYUFBYSxDQUNuQixFQUFFO1lBQ0QsTUFBTSxnQkFBZ0IsR0FBRyxRQUFRLENBQUMsT0FBTyxDQUFDLGdCQUFnQixFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBQ2hFLE1BQU0sY0FBYyxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsaUJBQWlCLENBQ3RELEdBQUcsZ0JBQWdCLEVBQUUsRUFDckIsUUFBUSxDQUNULENBQUM7WUFFRixhQUFhLENBQUMsT0FBTyxDQUFDLENBQUMsUUFBb0MsRUFBRSxFQUFFO2dCQUM3RCxJQUFJLHNCQUFRLENBQ1YsSUFBSSxDQUFDLGtCQUFrQixFQUN2QixHQUFHLFFBQVEsQ0FBQyxRQUFRLElBQUksUUFBUSxDQUFDLFNBQVMsV0FBVyxFQUNyRDtvQkFDRSxHQUFHLEVBQUUsSUFBSSxDQUFDLFVBQVU7b0JBQ3BCLFFBQVEsRUFBRSxRQUFRLENBQUMsUUFBUTtvQkFDM0IsU0FBUyxFQUFFLFFBQVEsQ0FBQyxTQUFTO29CQUM3QixVQUFVLEVBQUUsY0FBYztvQkFDMUIsc0JBQXNCLEVBQUUsNkJBQWUsQ0FBQyxVQUFVLENBQ2hELFFBQVEsQ0FBQyw2QkFBNkIsQ0FDdkM7b0JBQ0QsdUJBQXVCLEVBQUUsNkJBQWUsQ0FBQyxVQUFVLENBQ2pELFFBQVEsQ0FBQyw4QkFBOEIsQ0FDeEM7aUJBQ0YsQ0FDRixDQUFDO1lBQ0osQ0FBQyxDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7Ozs7Ozs7Ozs7SUFVTSwrQkFBK0IsQ0FDcEMsWUFBb0IsRUFDcEIsRUFBVSxFQUNWLGNBQXlCLEVBQ3pCLE9BQTJCO1FBRTNCLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxtQkFBbUIsQ0FDNUQsRUFBRSxFQUNGLGNBQWMsRUFDZCxPQUFPLENBQ1IsQ0FBQztRQUVGLEtBQUssTUFBTSxRQUFRLElBQUksSUFBSSxDQUFDLGlCQUFpQixDQUFDLFlBQVksQ0FBQyxFQUFFO1lBQzNELElBQUksc0JBQVEsQ0FDVixJQUFJLENBQUMsa0JBQWtCLEVBQ3ZCLEdBQUcsUUFBUSxDQUFDLFFBQVEsSUFBSSxRQUFRLENBQUMsU0FBUyxXQUFXLEVBQ3JEO2dCQUNFLEdBQUcsRUFBRSxJQUFJLENBQUMsVUFBVTtnQkFDcEIsUUFBUSxFQUFFLFFBQVEsQ0FBQyxRQUFRO2dCQUMzQixTQUFTLEVBQUUsUUFBUSxDQUFDLFNBQVM7Z0JBQzdCLFVBQVUsRUFBRSxrQkFBa0I7Z0JBQzlCLHNCQUFzQixFQUFFLDZCQUFlLENBQUMsVUFBVSxDQUNoRCxRQUFRLENBQUMsNkJBQTZCLENBQ3ZDO2dCQUNELHVCQUF1QixFQUFFLDZCQUFlLENBQUMsVUFBVSxDQUNqRCxRQUFRLENBQUMsOEJBQThCLENBQ3hDO2FBQ0YsQ0FDRixDQUFDO1NBQ0g7UUFFRCxPQUFPLGtCQUFrQixDQUFDO0lBQzVCLENBQUM7Ozs7Ozs7SUFPTSxpQkFBaUIsQ0FBQyxLQUEwQjtRQUNqRCxNQUFNLFNBQVMsR0FBRyxHQUFHLEtBQUssQ0FBQyxhQUFhLE9BQU8sQ0FBQztRQUNoRCxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ3ZDLElBQUksQ0FBQyxLQUFLO1lBQUUsTUFBTSxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsU0FBUyxjQUFjLENBQUMsQ0FBQztRQUV6RSxNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFlBQXdCLENBQUM7UUFDckQsUUFBUSxDQUFDLG1CQUFtQixHQUFHO1lBQzdCLGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYztTQUNyQyxDQUFDO1FBRUYsT0FBTyxRQUFRLENBQUMsYUFBYSxDQUFDO0lBQ2hDLENBQUM7O0FBeGVILGdEQXllQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7XG4gIEdyYXBocWxBcGksXG4gIEF1dGhvcml6YXRpb25UeXBlLFxuICBGaWVsZExvZ0xldmVsLFxuICBNYXBwaW5nVGVtcGxhdGUsXG4gIENmbkRhdGFTb3VyY2UsXG4gIFJlc29sdmVyLFxuICBBdXRob3JpemF0aW9uQ29uZmlnLFxuICBTY2hlbWEsXG4gIERhdGFTb3VyY2VPcHRpb25zLFxuICBMYW1iZGFEYXRhU291cmNlLFxufSBmcm9tICdAYXdzLWNkay9hd3MtYXBwc3luYyc7XG5cbmltcG9ydCB7XG4gIENmblRhYmxlLFxuICBUYWJsZSxcbiAgQXR0cmlidXRlVHlwZSxcbiAgUHJvamVjdGlvblR5cGUsXG4gIEJpbGxpbmdNb2RlLFxuICBTdHJlYW1WaWV3VHlwZSxcbiAgVGFibGVQcm9wcyxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWR5bmFtb2RiJztcbmltcG9ydCB7IEVmZmVjdCwgUG9saWN5U3RhdGVtZW50IH0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgeyBJRnVuY3Rpb24gfSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCB7IENvbnN0cnVjdCwgTmVzdGVkU3RhY2ssIENmbk91dHB1dCB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG5pbXBvcnQge1xuICBDZGtUcmFuc2Zvcm1lclJlc29sdmVyLFxuICBDZGtUcmFuc2Zvcm1lckZ1bmN0aW9uUmVzb2x2ZXIsXG4gIENka1RyYW5zZm9ybWVySHR0cFJlc29sdmVyLFxuICBDZGtUcmFuc2Zvcm1lclRhYmxlLFxuICBTY2hlbWFUcmFuc2Zvcm1lck91dHB1dHMsXG59IGZyb20gJy4vdHJhbnNmb3JtZXInO1xuXG5pbXBvcnQge1xuICBTY2hlbWFUcmFuc2Zvcm1lcixcbiAgU2NoZW1hVHJhbnNmb3JtZXJQcm9wcyxcbn0gZnJvbSAnLi90cmFuc2Zvcm1lci9zY2hlbWEtdHJhbnNmb3JtZXInO1xuXG5leHBvcnQgaW50ZXJmYWNlIEFwcFN5bmNUcmFuc2Zvcm1lclByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2NoZW1hUGF0aDogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGF1dGhvcml6YXRpb25Db25maWc/OiBBdXRob3JpemF0aW9uQ29uZmlnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYXBpTmFtZT86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzeW5jRW5hYmxlZD86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW5hYmxlRHluYW1vUG9pbnRJblRpbWVSZWNvdmVyeT86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGZpZWxkTG9nTGV2ZWw/OiBGaWVsZExvZ0xldmVsO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHhyYXlFbmFibGVkPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZHluYW1vRGJTdHJlYW1Db25maWc/OiB7IFtuYW1lOiBzdHJpbmddOiBTdHJlYW1WaWV3VHlwZSB9O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5cbiAgcmVhZG9ubHkgcHJlQ2RrVHJhbnNmb3JtZXJzPzogYW55W107XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuXG4gIHJlYWRvbmx5IHBvc3RDZGtUcmFuc2Zvcm1lcnM/OiBhbnlbXTtcbn1cblxuY29uc3QgZGVmYXVsdEF1dGhvcml6YXRpb25Db25maWc6IEF1dGhvcml6YXRpb25Db25maWcgPSB7XG4gIGRlZmF1bHRBdXRob3JpemF0aW9uOiB7XG4gICAgYXV0aG9yaXphdGlvblR5cGU6IEF1dGhvcml6YXRpb25UeXBlLkFQSV9LRVksXG4gICAgYXBpS2V5Q29uZmlnOiB7XG4gICAgICBkZXNjcmlwdGlvbjogJ0F1dG8gZ2VuZXJhdGVkIEFQSSBLZXkgZnJvbSBjb25zdHJ1Y3QnLFxuICAgICAgbmFtZTogJ2RldicsXG4gICAgfSxcbiAgfSxcbn07XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIEFwcFN5bmNUcmFuc2Zvcm1lciBleHRlbmRzIENvbnN0cnVjdCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGFwcHN5bmNBUEk6IEdyYXBocWxBcGk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBuZXN0ZWRBcHBzeW5jU3RhY2s6IE5lc3RlZFN0YWNrO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSB0YWJsZU5hbWVNYXA6IHsgW25hbWU6IHN0cmluZ106IGFueSB9O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgdGFibGVNYXA6IHsgW25hbWU6IHN0cmluZ106IFRhYmxlIH07XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IG91dHB1dHM6IFNjaGVtYVRyYW5zZm9ybWVyT3V0cHV0cztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgcmVzb2x2ZXJzOiB7IFtuYW1lOiBzdHJpbmddOiBDZGtUcmFuc2Zvcm1lclJlc29sdmVyIH07XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgZnVuY3Rpb25SZXNvbHZlcnM6IHtcbiAgICBbbmFtZTogc3RyaW5nXTogQ2RrVHJhbnNmb3JtZXJGdW5jdGlvblJlc29sdmVyW107XG4gIH07XG5cbiAgcHVibGljIHJlYWRvbmx5IGh0dHBSZXNvbHZlcnM6IHtcbiAgICBbbmFtZTogc3RyaW5nXTogQ2RrVHJhbnNmb3JtZXJIdHRwUmVzb2x2ZXJbXTtcbiAgfTtcblxuICBwcml2YXRlIHByb3BzOiBBcHBTeW5jVHJhbnNmb3JtZXJQcm9wc1xuICBwcml2YXRlIGlzU3luY0VuYWJsZWQ6IGJvb2xlYW47XG4gIHByaXZhdGUgc3luY1RhYmxlOiBUYWJsZSB8IHVuZGVmaW5lZDtcbiAgcHJpdmF0ZSBwb2ludEluVGltZVJlY292ZXJ5OiBib29sZWFuO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBBcHBTeW5jVHJhbnNmb3JtZXJQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLnByb3BzID0gcHJvcHM7XG4gICAgdGhpcy50YWJsZU1hcCA9IHt9O1xuICAgIHRoaXMuaXNTeW5jRW5hYmxlZCA9IHByb3BzLnN5bmNFbmFibGVkID8gcHJvcHMuc3luY0VuYWJsZWQgOiBmYWxzZTtcbiAgICB0aGlzLnBvaW50SW5UaW1lUmVjb3ZlcnkgPSBwcm9wcy5lbmFibGVEeW5hbW9Qb2ludEluVGltZVJlY292ZXJ5ID8/IGZhbHNlO1xuXG4gICAgY29uc3QgdHJhbnNmb3JtZXJDb25maWd1cmF0aW9uOiBTY2hlbWFUcmFuc2Zvcm1lclByb3BzID0ge1xuICAgICAgc2NoZW1hUGF0aDogcHJvcHMuc2NoZW1hUGF0aCxcbiAgICAgIHN5bmNFbmFibGVkOiBwcm9wcy5zeW5jRW5hYmxlZCA/PyBmYWxzZSxcbiAgICB9O1xuXG4gICAgLy8gQ29tYmluZSB0aGUgYXJyYXlzIHNvIHdlIG9ubHkgbG9vcCBvbmNlXG4gICAgLy8gVGVzdCBlYWNoIHRyYW5zZm9ybWVyIHRvIHNlZSBpZiBpdCBpbXBsZW1lbnRzIElUcmFuc2Zvcm1lclxuICAgIGNvbnN0IGFsbEN1c3RvbVRyYW5zZm9ybWVycyA9IFsuLi5wcm9wcy5wcmVDZGtUcmFuc2Zvcm1lcnMgPz8gW10sIC4uLnByb3BzLnBvc3RDZGtUcmFuc2Zvcm1lcnMgPz8gW11dO1xuICAgIGlmIChhbGxDdXN0b21UcmFuc2Zvcm1lcnMgJiYgYWxsQ3VzdG9tVHJhbnNmb3JtZXJzLmxlbmd0aCA+IDApIHtcbiAgICAgIGFsbEN1c3RvbVRyYW5zZm9ybWVycy5mb3JFYWNoKHRyYW5zZm9ybWVyID0+IHtcbiAgICAgICAgaWYgKHRyYW5zZm9ybWVyICYmICF0aGlzLmltcGxlbWVudHNJVHJhbnNmb3JtZXIodHJhbnNmb3JtZXIpKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBUcmFuc2Zvcm1lciBkb2VzIG5vdCBpbXBsZW1lbnQgSVRyYW5zZm9ybWVyIGZyb20gZ3JhcGhxbC10cmFuc2Zvcm1lci1jb3JlOiAke3RyYW5zZm9ybWVyfWApO1xuICAgICAgICB9XG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBjb25zdCB0cmFuc2Zvcm1lciA9IG5ldyBTY2hlbWFUcmFuc2Zvcm1lcih0cmFuc2Zvcm1lckNvbmZpZ3VyYXRpb24pO1xuICAgIHRoaXMub3V0cHV0cyA9IHRyYW5zZm9ybWVyLnRyYW5zZm9ybShwcm9wcy5wcmVDZGtUcmFuc2Zvcm1lcnMsIHByb3BzLnBvc3RDZGtUcmFuc2Zvcm1lcnMpO1xuICAgIGNvbnN0IHJlc29sdmVycyA9IHRyYW5zZm9ybWVyLmdldFJlc29sdmVycygpO1xuXG4gICAgdGhpcy5mdW5jdGlvblJlc29sdmVycyA9IHRoaXMub3V0cHV0cy5mdW5jdGlvblJlc29sdmVycyA/PyB7fTtcblxuICAgIC8vIFJlbW92ZSBhbnkgZnVuY3Rpb24gcmVzb2x2ZXJzIGZyb20gdGhlIHRvdGFsIGxpc3Qgb2YgcmVzb2x2ZXJzXG4gICAgLy8gT3RoZXJ3aXNlIGl0IHdpbGwgYWRkIHRoZW0gdHdpY2VcbiAgICBmb3IgKGNvbnN0IFtfLCBmdW5jdGlvblJlc29sdmVyc10gb2YgT2JqZWN0LmVudHJpZXMoXG4gICAgICB0aGlzLmZ1bmN0aW9uUmVzb2x2ZXJzLFxuICAgICkpIHtcbiAgICAgIGZ1bmN0aW9uUmVzb2x2ZXJzLmZvckVhY2goKHJlc29sdmVyOiBhbnkpID0+IHtcbiAgICAgICAgc3dpdGNoIChyZXNvbHZlci50eXBlTmFtZSkge1xuICAgICAgICAgIGNhc2UgJ1F1ZXJ5JzpcbiAgICAgICAgICBjYXNlICdNdXRhdGlvbic6XG4gICAgICAgICAgY2FzZSAnU3Vic2NyaXB0aW9uJzpcbiAgICAgICAgICAgIGRlbGV0ZSByZXNvbHZlcnNbcmVzb2x2ZXIuZmllbGROYW1lXTtcbiAgICAgICAgICAgIGJyZWFrO1xuICAgICAgICB9XG4gICAgICB9KTtcbiAgICB9XG5cbiAgICB0aGlzLmh0dHBSZXNvbHZlcnMgPSB0aGlzLm91dHB1dHMuaHR0cFJlc29sdmVycyA/PyB7fTtcblxuICAgIC8vIFJlbW92ZSBhbnkgaHR0cCByZXNvbHZlcnMgZnJvbSB0aGUgdG90YWwgbGlzdCBvZiByZXNvbHZlcnNcbiAgICAvLyBPdGhlcndpc2UgaXQgd2lsbCBhZGQgdGhlbSB0d2ljZVxuICAgIGZvciAoY29uc3QgW18sIGh0dHBSZXNvbHZlcnNdIG9mIE9iamVjdC5lbnRyaWVzKHRoaXMuaHR0cFJlc29sdmVycykpIHtcbiAgICAgIGh0dHBSZXNvbHZlcnMuZm9yRWFjaCgocmVzb2x2ZXI6IGFueSkgPT4ge1xuICAgICAgICBzd2l0Y2ggKHJlc29sdmVyLnR5cGVOYW1lKSB7XG4gICAgICAgICAgY2FzZSAnUXVlcnknOlxuICAgICAgICAgIGNhc2UgJ011dGF0aW9uJzpcbiAgICAgICAgICBjYXNlICdTdWJzY3JpcHRpb24nOlxuICAgICAgICAgICAgZGVsZXRlIHJlc29sdmVyc1tyZXNvbHZlci5maWVsZE5hbWVdO1xuICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgIH1cbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHRoaXMucmVzb2x2ZXJzID0gcmVzb2x2ZXJzO1xuXG4gICAgdGhpcy5uZXN0ZWRBcHBzeW5jU3RhY2sgPSBuZXcgTmVzdGVkU3RhY2sodGhpcywgJ2FwcHN5bmMtbmVzdGVkLXN0YWNrJyk7XG5cbiAgICAvLyBBcHBTeW5jXG4gICAgdGhpcy5hcHBzeW5jQVBJID0gbmV3IEdyYXBocWxBcGkodGhpcy5uZXN0ZWRBcHBzeW5jU3RhY2ssIGAke2lkfS1hcGlgLCB7XG4gICAgICBuYW1lOiBwcm9wcy5hcGlOYW1lID8gcHJvcHMuYXBpTmFtZSA6IGAke2lkfS1hcGlgLFxuICAgICAgYXV0aG9yaXphdGlvbkNvbmZpZzogcHJvcHMuYXV0aG9yaXphdGlvbkNvbmZpZ1xuICAgICAgICA/IHByb3BzLmF1dGhvcml6YXRpb25Db25maWdcbiAgICAgICAgOiBkZWZhdWx0QXV0aG9yaXphdGlvbkNvbmZpZyxcbiAgICAgIGxvZ0NvbmZpZzoge1xuICAgICAgICBmaWVsZExvZ0xldmVsOiBwcm9wcy5maWVsZExvZ0xldmVsXG4gICAgICAgICAgPyBwcm9wcy5maWVsZExvZ0xldmVsXG4gICAgICAgICAgOiBGaWVsZExvZ0xldmVsLk5PTkUsXG4gICAgICB9LFxuICAgICAgc2NoZW1hOiBTY2hlbWEuZnJvbUFzc2V0KCcuL2FwcHN5bmMvc2NoZW1hLmdyYXBocWwnKSxcbiAgICAgIHhyYXlFbmFibGVkOiBwcm9wcy54cmF5RW5hYmxlZCA/PyBmYWxzZSxcbiAgICB9KTtcblxuICAgIGxldCB0YWJsZURhdGEgPSB0aGlzLm91dHB1dHMuY2RrVGFibGVzID8/IHt9O1xuXG4gICAgLy8gQ2hlY2sgdG8gc2VlIGlmIHN5bmMgaXMgZW5hYmxlZFxuICAgIGlmICh0YWJsZURhdGEuRGF0YVN0b3JlKSB7XG4gICAgICB0aGlzLmlzU3luY0VuYWJsZWQgPSB0cnVlO1xuICAgICAgdGhpcy5zeW5jVGFibGUgPSB0aGlzLmNyZWF0ZVN5bmNUYWJsZSh0YWJsZURhdGEuRGF0YVN0b3JlKTtcbiAgICAgIGRlbGV0ZSB0YWJsZURhdGEuRGF0YVN0b3JlOyAvLyBXZSBkb24ndCB3YW50IHRvIGNyZWF0ZSB0aGlzIGFnYWluIGJlbG93IHNvIHJlbW92ZSBpdCBmcm9tIHRoZSB0YWJsZURhdGEgbWFwXG4gICAgfVxuXG4gICAgdGhpcy50YWJsZU5hbWVNYXAgPSB0aGlzLmNyZWF0ZVRhYmxlc0FuZFJlc29sdmVycyh0YWJsZURhdGEsIHJlc29sdmVycyk7XG4gICAgaWYgKHRoaXMub3V0cHV0cy5ub25lUmVzb2x2ZXJzKSB7XG4gICAgICB0aGlzLmNyZWF0ZU5vbmVEYXRhU291cmNlQW5kUmVzb2x2ZXJzKFxuICAgICAgICB0aGlzLm91dHB1dHMubm9uZVJlc29sdmVycyxcbiAgICAgICAgcmVzb2x2ZXJzLFxuICAgICAgKTtcbiAgICB9XG4gICAgdGhpcy5jcmVhdGVIdHRwUmVzb2x2ZXJzKCk7XG5cbiAgICAvLyBPdXRwdXRzIHNvIHdlIGNhbiBnZW5lcmF0ZSBleHBvcnRzXG4gICAgbmV3IENmbk91dHB1dChzY29wZSwgJ2FwcHN5bmNHcmFwaFFMRW5kcG9pbnRPdXRwdXQnLCB7XG4gICAgICB2YWx1ZTogdGhpcy5hcHBzeW5jQVBJLmdyYXBocWxVcmwsXG4gICAgICBkZXNjcmlwdGlvbjogJ091dHB1dCBmb3IgYXdzX2FwcHN5bmNfZ3JhcGhxbEVuZHBvaW50JyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBncmFwaHFsLXRyYW5zZm9ybWVyLWNvcmUgbmVlZHMgdG8gYmUganNpaSBlbmFibGVkIHRvIHB1bGwgdGhlIElUcmFuc2Zvcm1lciBpbnRlcmZhY2UgY29ycmVjdGx5LlxuICAgKiBTaW5jZSBpdCdzIG5vdCBpbiBwZWVyIGRlcGVuZGVuY2llcyBpdCBkb2Vzbid0IHNob3cgdXAgaW4gdGhlIGpzaWkgZGVwcyBsaXN0LlxuICAgKiBTaW5jZSBpdCdzIG5vdCBqc2lpIGVuYWJsZWQgaXQgaGFzIHRvIGJlIGJ1bmRsZWQuXG4gICAqIFRoZSBwYWNrYWdlIGNhbid0IGJlIGluIEJPVEggcGVlciBhbmQgYnVuZGxlZCBkZXBlbmRlbmNpZXNcbiAgICogU28gd2UgZG8gYSBmYWtlIHRlc3QgdG8gbWFrZSBzdXJlIGl0IGltcGxlbWVudHMgdGhlc2UgYW5kIGhvcGUgZm9yIHRoZSBiZXN0XG4gICAqIEBwYXJhbSB0cmFuc2Zvcm1lclxuICAgKi9cbiAgcHJpdmF0ZSBpbXBsZW1lbnRzSVRyYW5zZm9ybWVyKHRyYW5zZm9ybWVyOiBhbnkpIHtcbiAgICByZXR1cm4gJ25hbWUnIGluIHRyYW5zZm9ybWVyXG4gICAgICAmJiAnZGlyZWN0aXZlJyBpbiB0cmFuc2Zvcm1lclxuICAgICAgJiYgJ3R5cGVEZWZpbml0aW9ucycgaW4gdHJhbnNmb3JtZXI7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBOT05FIGRhdGEgc291cmNlIGFuZCBhc3NvY2lhdGVkIHJlc29sdmVyc1xuICAgKiBAcGFyYW0gbm9uZVJlc29sdmVycyBUaGUgcmVzb2x2ZXJzIHRoYXQgYmVsb25nIHRvIHRoZSBub25lIGRhdGEgc291cmNlXG4gICAqIEBwYXJhbSByZXNvbHZlcnMgVGhlIHJlc29sdmVyIG1hcCBtaW51cyBmdW5jdGlvbiByZXNvbHZlcnNcbiAgICovXG4gIHByaXZhdGUgY3JlYXRlTm9uZURhdGFTb3VyY2VBbmRSZXNvbHZlcnMoXG4gICAgbm9uZVJlc29sdmVyczogeyBbbmFtZTogc3RyaW5nXTogQ2RrVHJhbnNmb3JtZXJSZXNvbHZlciB9LFxuICAgIHJlc29sdmVyczogYW55LFxuICApIHtcbiAgICBjb25zdCBub25lRGF0YVNvdXJjZSA9IHRoaXMuYXBwc3luY0FQSS5hZGROb25lRGF0YVNvdXJjZSgnTk9ORScpO1xuXG4gICAgT2JqZWN0LmtleXMobm9uZVJlc29sdmVycykuZm9yRWFjaCgocmVzb2x2ZXJLZXk6IGFueSkgPT4ge1xuICAgICAgY29uc3QgcmVzb2x2ZXIgPSByZXNvbHZlcnNbcmVzb2x2ZXJLZXldO1xuICAgICAgbmV3IFJlc29sdmVyKFxuICAgICAgICB0aGlzLm5lc3RlZEFwcHN5bmNTdGFjayxcbiAgICAgICAgYCR7cmVzb2x2ZXIudHlwZU5hbWV9LSR7cmVzb2x2ZXIuZmllbGROYW1lfS1yZXNvbHZlcmAsXG4gICAgICAgIHtcbiAgICAgICAgICBhcGk6IHRoaXMuYXBwc3luY0FQSSxcbiAgICAgICAgICB0eXBlTmFtZTogcmVzb2x2ZXIudHlwZU5hbWUsXG4gICAgICAgICAgZmllbGROYW1lOiByZXNvbHZlci5maWVsZE5hbWUsXG4gICAgICAgICAgZGF0YVNvdXJjZTogbm9uZURhdGFTb3VyY2UsXG4gICAgICAgICAgcmVxdWVzdE1hcHBpbmdUZW1wbGF0ZTogTWFwcGluZ1RlbXBsYXRlLmZyb21GaWxlKFxuICAgICAgICAgICAgcmVzb2x2ZXIucmVxdWVzdE1hcHBpbmdUZW1wbGF0ZSxcbiAgICAgICAgICApLFxuICAgICAgICAgIHJlc3BvbnNlTWFwcGluZ1RlbXBsYXRlOiBNYXBwaW5nVGVtcGxhdGUuZnJvbUZpbGUoXG4gICAgICAgICAgICByZXNvbHZlci5yZXNwb25zZU1hcHBpbmdUZW1wbGF0ZSxcbiAgICAgICAgICApLFxuICAgICAgICB9LFxuICAgICAgKTtcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGVhY2ggZHluYW1vZGIgdGFibGUsIGdzaXMsIGR5bmFtb2RiIGRhdGFzb3VyY2UsIGFuZCBhc3NvY2lhdGVkIHJlc29sdmVyc1xuICAgKiBJZiBzeW5jIGlzIGVuYWJsZWQgdGhlbiBUVEwgY29uZmlndXJhdGlvbiBpcyBhZGRlZFxuICAgKiBSZXR1cm5zIHRhYmxlTmFtZTogdGFibGUgbWFwIGluIGNhc2UgaXQgaXMgbmVlZGVkIGZvciBsYW1iZGEgZnVuY3Rpb25zLCBldGNcbiAgICogQHBhcmFtIHRhYmxlRGF0YSBUaGUgQ2RrVHJhbnNmb3JtZXIgdGFibGUgaW5mb3JtYXRpb25cbiAgICogQHBhcmFtIHJlc29sdmVycyBUaGUgcmVzb2x2ZXIgbWFwIG1pbnVzIGZ1bmN0aW9uIHJlc29sdmVyc1xuICAgKi9cbiAgcHJpdmF0ZSBjcmVhdGVUYWJsZXNBbmRSZXNvbHZlcnMoXG4gICAgdGFibGVEYXRhOiB7IFtuYW1lOiBzdHJpbmddOiBDZGtUcmFuc2Zvcm1lclRhYmxlIH0sXG4gICAgcmVzb2x2ZXJzOiBhbnksXG4gICk6IHsgW25hbWU6IHN0cmluZ106IHN0cmluZyB9IHtcbiAgICBjb25zdCB0YWJsZU5hbWVNYXA6IGFueSA9IHt9O1xuXG4gICAgT2JqZWN0LmtleXModGFibGVEYXRhKS5mb3JFYWNoKCh0YWJsZUtleTogYW55KSA9PiB7XG4gICAgICBjb25zdCB0YWJsZSA9IHRoaXMuY3JlYXRlVGFibGUodGFibGVEYXRhW3RhYmxlS2V5XSk7XG4gICAgICB0aGlzLnRhYmxlTWFwW3RhYmxlS2V5XSA9IHRhYmxlO1xuXG4gICAgICBjb25zdCBkYXRhU291cmNlID0gdGhpcy5hcHBzeW5jQVBJLmFkZER5bmFtb0RiRGF0YVNvdXJjZSh0YWJsZUtleSwgdGFibGUpO1xuXG4gICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtcHJvcGVydGllcy1hcHBzeW5jLWRhdGFzb3VyY2UtZGVsdGFzeW5jY29uZmlnLmh0bWxcblxuICAgICAgaWYgKHRoaXMuaXNTeW5jRW5hYmxlZCAmJiB0aGlzLnN5bmNUYWJsZSkge1xuICAgICAgICAvL0B0cy1pZ25vcmUgLSBkcyBpcyB0aGUgYmFzZSBDZm5EYXRhU291cmNlIGFuZCB0aGUgZGIgY29uZmlnIG5lZWRzIHRvIGJlIHZlcnNpb25lZCAtIHNlZSBDZm5EYXRhU291cmNlXG4gICAgICAgIGRhdGFTb3VyY2UuZHMuZHluYW1vRGJDb25maWcudmVyc2lvbmVkID0gdHJ1ZTtcblxuICAgICAgICAvL0B0cy1pZ25vcmUgLSBkcyBpcyB0aGUgYmFzZSBDZm5EYXRhU291cmNlIC0gc2VlIENmbkRhdGFTb3VyY2VcbiAgICAgICAgZGF0YVNvdXJjZS5kcy5keW5hbW9EYkNvbmZpZy5kZWx0YVN5bmNDb25maWcgPSB7XG4gICAgICAgICAgYmFzZVRhYmxlVHRsOiAnNDMyMDAnLCAvLyBHb3QgdGhpcyB2YWx1ZSBmcm9tIGFtcGxpZnkgLSAzMCBkYXlzIGluIG1pbnV0ZXNcbiAgICAgICAgICBkZWx0YVN5bmNUYWJsZU5hbWU6IHRoaXMuc3luY1RhYmxlLnRhYmxlTmFtZSxcbiAgICAgICAgICBkZWx0YVN5bmNUYWJsZVR0bDogJzMwJywgLy8gR290IHRoaXMgdmFsdWUgZnJvbSBhbXBsaWZ5IC0gMzAgbWludXRlc1xuICAgICAgICB9O1xuXG4gICAgICAgIC8vIE5lZWQgdG8gYWRkIHBlcm1pc3Npb24gZm9yIG91ciBkYXRhc291cmNlIHNlcnZpY2Ugcm9sZSB0byBhY2Nlc3MgdGhlIHN5bmMgdGFibGVcbiAgICAgICAgZGF0YVNvdXJjZS5ncmFudFByaW5jaXBhbC5hZGRUb1BvbGljeShcbiAgICAgICAgICBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgIGVmZmVjdDogRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgICAnZHluYW1vZGI6KicsIC8vIFRPRE86IFRoaXMgbWF5IGJlIHRvbyBwZXJtaXNzaXZlXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgcmVzb3VyY2VzOiBbdGhpcy5zeW5jVGFibGUudGFibGVBcm5dLFxuICAgICAgICAgIH0pLFxuICAgICAgICApO1xuICAgICAgfVxuXG4gICAgICBjb25zdCBkeW5hbW9EYkNvbmZpZyA9IGRhdGFTb3VyY2UuZHNcbiAgICAgICAgLmR5bmFtb0RiQ29uZmlnIGFzIENmbkRhdGFTb3VyY2UuRHluYW1vREJDb25maWdQcm9wZXJ0eTtcbiAgICAgIHRhYmxlTmFtZU1hcFt0YWJsZUtleV0gPSBkeW5hbW9EYkNvbmZpZy50YWJsZU5hbWU7XG5cbiAgICAgIC8vIExvb3AgdGhlIGJhc2ljIHJlc29sdmVyc1xuICAgICAgdGFibGVEYXRhW3RhYmxlS2V5XS5yZXNvbHZlcnMuZm9yRWFjaCgocmVzb2x2ZXJLZXk6IGFueSkgPT4ge1xuICAgICAgICBsZXQgcmVzb2x2ZXIgPSByZXNvbHZlcnNbcmVzb2x2ZXJLZXldO1xuICAgICAgICBuZXcgUmVzb2x2ZXIoXG4gICAgICAgICAgdGhpcy5uZXN0ZWRBcHBzeW5jU3RhY2ssXG4gICAgICAgICAgYCR7cmVzb2x2ZXIudHlwZU5hbWV9LSR7cmVzb2x2ZXIuZmllbGROYW1lfS1yZXNvbHZlcmAsXG4gICAgICAgICAge1xuICAgICAgICAgICAgYXBpOiB0aGlzLmFwcHN5bmNBUEksXG4gICAgICAgICAgICB0eXBlTmFtZTogcmVzb2x2ZXIudHlwZU5hbWUsXG4gICAgICAgICAgICBmaWVsZE5hbWU6IHJlc29sdmVyLmZpZWxkTmFtZSxcbiAgICAgICAgICAgIGRhdGFTb3VyY2U6IGRhdGFTb3VyY2UsXG4gICAgICAgICAgICByZXF1ZXN0TWFwcGluZ1RlbXBsYXRlOiBNYXBwaW5nVGVtcGxhdGUuZnJvbUZpbGUoXG4gICAgICAgICAgICAgIHJlc29sdmVyLnJlcXVlc3RNYXBwaW5nVGVtcGxhdGUsXG4gICAgICAgICAgICApLFxuICAgICAgICAgICAgcmVzcG9uc2VNYXBwaW5nVGVtcGxhdGU6IE1hcHBpbmdUZW1wbGF0ZS5mcm9tRmlsZShcbiAgICAgICAgICAgICAgcmVzb2x2ZXIucmVzcG9uc2VNYXBwaW5nVGVtcGxhdGUsXG4gICAgICAgICAgICApLFxuICAgICAgICAgIH0sXG4gICAgICAgICk7XG4gICAgICB9KTtcblxuICAgICAgLy8gTG9vcCB0aGUgZ3NpIHJlc29sdmVyc1xuICAgICAgdGFibGVEYXRhW3RhYmxlS2V5XS5nc2lSZXNvbHZlcnMuZm9yRWFjaCgocmVzb2x2ZXJLZXk6IGFueSkgPT4ge1xuICAgICAgICBsZXQgcmVzb2x2ZXIgPSByZXNvbHZlcnMuZ3NpW3Jlc29sdmVyS2V5XTtcbiAgICAgICAgbmV3IFJlc29sdmVyKFxuICAgICAgICAgIHRoaXMubmVzdGVkQXBwc3luY1N0YWNrLFxuICAgICAgICAgIGAke3Jlc29sdmVyLnR5cGVOYW1lfS0ke3Jlc29sdmVyLmZpZWxkTmFtZX0tcmVzb2x2ZXJgLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGFwaTogdGhpcy5hcHBzeW5jQVBJLFxuICAgICAgICAgICAgdHlwZU5hbWU6IHJlc29sdmVyLnR5cGVOYW1lLFxuICAgICAgICAgICAgZmllbGROYW1lOiByZXNvbHZlci5maWVsZE5hbWUsXG4gICAgICAgICAgICBkYXRhU291cmNlOiBkYXRhU291cmNlLFxuICAgICAgICAgICAgcmVxdWVzdE1hcHBpbmdUZW1wbGF0ZTogTWFwcGluZ1RlbXBsYXRlLmZyb21GaWxlKFxuICAgICAgICAgICAgICByZXNvbHZlci5yZXF1ZXN0TWFwcGluZ1RlbXBsYXRlLFxuICAgICAgICAgICAgKSxcbiAgICAgICAgICAgIHJlc3BvbnNlTWFwcGluZ1RlbXBsYXRlOiBNYXBwaW5nVGVtcGxhdGUuZnJvbUZpbGUoXG4gICAgICAgICAgICAgIHJlc29sdmVyLnJlc3BvbnNlTWFwcGluZ1RlbXBsYXRlLFxuICAgICAgICAgICAgKSxcbiAgICAgICAgICB9LFxuICAgICAgICApO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICByZXR1cm4gdGFibGVOYW1lTWFwO1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVUYWJsZSh0YWJsZURhdGE6IENka1RyYW5zZm9ybWVyVGFibGUpIHtcbiAgICAvLyBJIGRvIG5vdCB3YW50IHRvIGZvcmNlIHBlb3BsZSB0byBwYXNzIGBUeXBlVGFibGVgIC0gdGhpcyB3YXkgdGhleSBhcmUgb25seSBwYXNzaW5nIHRoZSBAbW9kZWwgVHlwZSBuYW1lXG4gICAgY29uc3QgbW9kZWxUeXBlTmFtZSA9IHRhYmxlRGF0YS50YWJsZU5hbWUucmVwbGFjZSgnVGFibGUnLCAnJyk7XG4gICAgY29uc3Qgc3RyZWFtU3BlY2lmaWNhdGlvbiA9IHRoaXMucHJvcHMuZHluYW1vRGJTdHJlYW1Db25maWcgJiYgdGhpcy5wcm9wcy5keW5hbW9EYlN0cmVhbUNvbmZpZ1ttb2RlbFR5cGVOYW1lXTtcbiAgICBjb25zdCB0YWJsZVByb3BzOiBUYWJsZVByb3BzID0ge1xuICAgICAgYmlsbGluZ01vZGU6IEJpbGxpbmdNb2RlLlBBWV9QRVJfUkVRVUVTVCxcbiAgICAgIHBhcnRpdGlvbktleToge1xuICAgICAgICBuYW1lOiB0YWJsZURhdGEucGFydGl0aW9uS2V5Lm5hbWUsXG4gICAgICAgIHR5cGU6IHRoaXMuY29udmVydEF0dHJpYnV0ZVR5cGUodGFibGVEYXRhLnBhcnRpdGlvbktleS50eXBlKSxcbiAgICAgIH0sXG4gICAgICBwb2ludEluVGltZVJlY292ZXJ5OiB0aGlzLnBvaW50SW5UaW1lUmVjb3ZlcnksXG4gICAgICBzb3J0S2V5OiB0YWJsZURhdGEuc29ydEtleSAmJiB0YWJsZURhdGEuc29ydEtleS5uYW1lXG4gICAgICAgID8ge1xuICAgICAgICAgIG5hbWU6IHRhYmxlRGF0YS5zb3J0S2V5Lm5hbWUsXG4gICAgICAgICAgdHlwZTogdGhpcy5jb252ZXJ0QXR0cmlidXRlVHlwZSh0YWJsZURhdGEuc29ydEtleS50eXBlKSxcbiAgICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICAgIHRpbWVUb0xpdmVBdHRyaWJ1dGU6IHRhYmxlRGF0YT8udHRsPy5lbmFibGVkID8gdGFibGVEYXRhLnR0bC5hdHRyaWJ1dGVOYW1lIDogdW5kZWZpbmVkLFxuICAgICAgc3RyZWFtOiBzdHJlYW1TcGVjaWZpY2F0aW9uLFxuICAgIH07XG5cbiAgICBjb25zdCB0YWJsZSA9IG5ldyBUYWJsZShcbiAgICAgIHRoaXMubmVzdGVkQXBwc3luY1N0YWNrLFxuICAgICAgdGFibGVEYXRhLnRhYmxlTmFtZSxcbiAgICAgIHRhYmxlUHJvcHMsXG4gICAgKTtcblxuICAgIHRhYmxlRGF0YS5sb2NhbFNlY29uZGFyeUluZGV4ZXMuZm9yRWFjaCgobHNpOiBhbnkpID0+IHtcbiAgICAgIHRhYmxlLmFkZExvY2FsU2Vjb25kYXJ5SW5kZXgoe1xuICAgICAgICBpbmRleE5hbWU6IGxzaS5pbmRleE5hbWUsXG4gICAgICAgIHNvcnRLZXk6IHtcbiAgICAgICAgICBuYW1lOiBsc2kuc29ydEtleS5uYW1lLFxuICAgICAgICAgIHR5cGU6IHRoaXMuY29udmVydEF0dHJpYnV0ZVR5cGUobHNpLnNvcnRLZXkudHlwZSksXG4gICAgICAgIH0sXG4gICAgICAgIHByb2plY3Rpb25UeXBlOiB0aGlzLmNvbnZlcnRQcm9qZWN0aW9uVHlwZShcbiAgICAgICAgICBsc2kucHJvamVjdGlvbi5Qcm9qZWN0aW9uVHlwZSxcbiAgICAgICAgKSxcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgdGFibGVEYXRhLmdsb2JhbFNlY29uZGFyeUluZGV4ZXMuZm9yRWFjaCgoZ3NpOiBhbnkpID0+IHtcbiAgICAgIHRhYmxlLmFkZEdsb2JhbFNlY29uZGFyeUluZGV4KHtcbiAgICAgICAgaW5kZXhOYW1lOiBnc2kuaW5kZXhOYW1lLFxuICAgICAgICBwYXJ0aXRpb25LZXk6IHtcbiAgICAgICAgICBuYW1lOiBnc2kucGFydGl0aW9uS2V5Lm5hbWUsXG4gICAgICAgICAgdHlwZTogdGhpcy5jb252ZXJ0QXR0cmlidXRlVHlwZShnc2kucGFydGl0aW9uS2V5LnR5cGUpLFxuICAgICAgICB9LFxuICAgICAgICBzb3J0S2V5OiBnc2kuc29ydEtleSAmJiBnc2kuc29ydEtleS5uYW1lXG4gICAgICAgICAgPyB7XG4gICAgICAgICAgICBuYW1lOiBnc2kuc29ydEtleS5uYW1lLFxuICAgICAgICAgICAgdHlwZTogdGhpcy5jb252ZXJ0QXR0cmlidXRlVHlwZShnc2kuc29ydEtleS50eXBlKSxcbiAgICAgICAgICB9IDogdW5kZWZpbmVkLFxuICAgICAgICBwcm9qZWN0aW9uVHlwZTogdGhpcy5jb252ZXJ0UHJvamVjdGlvblR5cGUoXG4gICAgICAgICAgZ3NpLnByb2plY3Rpb24uUHJvamVjdGlvblR5cGUsXG4gICAgICAgICksXG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHJldHVybiB0YWJsZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIHRoZSBzeW5jIHRhYmxlIGZvciBBbXBsaWZ5IERhdGFTdG9yZVxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvZGV2Z3VpZGUvY29uZmxpY3QtZGV0ZWN0aW9uLWFuZC1zeW5jLmh0bWxcbiAgICogQHBhcmFtIHRhYmxlRGF0YSBUaGUgQ2RrVHJhbnNmb3JtZXIgdGFibGUgaW5mb3JtYXRpb25cbiAgICovXG4gIHByaXZhdGUgY3JlYXRlU3luY1RhYmxlKHRhYmxlRGF0YTogQ2RrVHJhbnNmb3JtZXJUYWJsZSk6IFRhYmxlIHtcbiAgICByZXR1cm4gbmV3IFRhYmxlKHRoaXMsICdhcHBzeW5jLWFwaS1zeW5jLXRhYmxlJywge1xuICAgICAgYmlsbGluZ01vZGU6IEJpbGxpbmdNb2RlLlBBWV9QRVJfUkVRVUVTVCxcbiAgICAgIHBhcnRpdGlvbktleToge1xuICAgICAgICBuYW1lOiB0YWJsZURhdGEucGFydGl0aW9uS2V5Lm5hbWUsXG4gICAgICAgIHR5cGU6IHRoaXMuY29udmVydEF0dHJpYnV0ZVR5cGUodGFibGVEYXRhLnBhcnRpdGlvbktleS50eXBlKSxcbiAgICAgIH0sXG4gICAgICBzb3J0S2V5OiB7XG4gICAgICAgIG5hbWU6IHRhYmxlRGF0YS5zb3J0S2V5IS5uYW1lLCAvLyBXZSBrbm93IGl0IGhhcyBhIHNvcnRrZXkgYmVjYXVzZSB3ZSBmb3JjZWQgaXQgdG9cbiAgICAgICAgdHlwZTogdGhpcy5jb252ZXJ0QXR0cmlidXRlVHlwZSh0YWJsZURhdGEuc29ydEtleSEudHlwZSksIC8vIFdlIGtub3cgaXQgaGFzIGEgc29ydGtleSBiZWNhdXNlIHdlIGZvcmNlZCBpdCB0b1xuICAgICAgfSxcbiAgICAgIHRpbWVUb0xpdmVBdHRyaWJ1dGU6IHRhYmxlRGF0YS50dGw/LmF0dHJpYnV0ZU5hbWUgfHwgJ190dGwnLFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBjb252ZXJ0QXR0cmlidXRlVHlwZSh0eXBlOiBzdHJpbmcpOiBBdHRyaWJ1dGVUeXBlIHtcbiAgICBzd2l0Y2ggKHR5cGUpIHtcbiAgICAgIGNhc2UgJ04nOlxuICAgICAgICByZXR1cm4gQXR0cmlidXRlVHlwZS5OVU1CRVI7XG4gICAgICBjYXNlICdCJzpcbiAgICAgICAgcmV0dXJuIEF0dHJpYnV0ZVR5cGUuQklOQVJZO1xuICAgICAgY2FzZSAnUyc6IC8vIFNhbWUgYXMgZGVmYXVsdFxuICAgICAgZGVmYXVsdDpcbiAgICAgICAgcmV0dXJuIEF0dHJpYnV0ZVR5cGUuU1RSSU5HO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgY29udmVydFByb2plY3Rpb25UeXBlKHR5cGU6IHN0cmluZyk6IFByb2plY3Rpb25UeXBlIHtcbiAgICBzd2l0Y2ggKHR5cGUpIHtcbiAgICAgIGNhc2UgJ0lOQ0xVREUnOlxuICAgICAgICByZXR1cm4gUHJvamVjdGlvblR5cGUuSU5DTFVERTtcbiAgICAgIGNhc2UgJ0tFWVNfT05MWSc6XG4gICAgICAgIHJldHVybiBQcm9qZWN0aW9uVHlwZS5LRVlTX09OTFk7XG4gICAgICBjYXNlICdBTEwnOiAvLyBTYW1lIGFzIGRlZmF1bHRcbiAgICAgIGRlZmF1bHQ6XG4gICAgICAgIHJldHVybiBQcm9qZWN0aW9uVHlwZS5BTEw7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVIdHRwUmVzb2x2ZXJzKCkge1xuICAgIGZvciAoY29uc3QgW2VuZHBvaW50LCBodHRwUmVzb2x2ZXJzXSBvZiBPYmplY3QuZW50cmllcyhcbiAgICAgIHRoaXMuaHR0cFJlc29sdmVycyxcbiAgICApKSB7XG4gICAgICBjb25zdCBzdHJpcHBlZEVuZHBvaW50ID0gZW5kcG9pbnQucmVwbGFjZSgvW15fMC05QS1aYS16XS9nLCAnJyk7XG4gICAgICBjb25zdCBodHRwRGF0YVNvdXJjZSA9IHRoaXMuYXBwc3luY0FQSS5hZGRIdHRwRGF0YVNvdXJjZShcbiAgICAgICAgYCR7c3RyaXBwZWRFbmRwb2ludH1gLFxuICAgICAgICBlbmRwb2ludCxcbiAgICAgICk7XG5cbiAgICAgIGh0dHBSZXNvbHZlcnMuZm9yRWFjaCgocmVzb2x2ZXI6IENka1RyYW5zZm9ybWVySHR0cFJlc29sdmVyKSA9PiB7XG4gICAgICAgIG5ldyBSZXNvbHZlcihcbiAgICAgICAgICB0aGlzLm5lc3RlZEFwcHN5bmNTdGFjayxcbiAgICAgICAgICBgJHtyZXNvbHZlci50eXBlTmFtZX0tJHtyZXNvbHZlci5maWVsZE5hbWV9LXJlc29sdmVyYCxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBhcGk6IHRoaXMuYXBwc3luY0FQSSxcbiAgICAgICAgICAgIHR5cGVOYW1lOiByZXNvbHZlci50eXBlTmFtZSxcbiAgICAgICAgICAgIGZpZWxkTmFtZTogcmVzb2x2ZXIuZmllbGROYW1lLFxuICAgICAgICAgICAgZGF0YVNvdXJjZTogaHR0cERhdGFTb3VyY2UsXG4gICAgICAgICAgICByZXF1ZXN0TWFwcGluZ1RlbXBsYXRlOiBNYXBwaW5nVGVtcGxhdGUuZnJvbVN0cmluZyhcbiAgICAgICAgICAgICAgcmVzb2x2ZXIuZGVmYXVsdFJlcXVlc3RNYXBwaW5nVGVtcGxhdGUsXG4gICAgICAgICAgICApLFxuICAgICAgICAgICAgcmVzcG9uc2VNYXBwaW5nVGVtcGxhdGU6IE1hcHBpbmdUZW1wbGF0ZS5mcm9tU3RyaW5nKFxuICAgICAgICAgICAgICByZXNvbHZlci5kZWZhdWx0UmVzcG9uc2VNYXBwaW5nVGVtcGxhdGUsXG4gICAgICAgICAgICApLFxuICAgICAgICAgIH0sXG4gICAgICAgICk7XG4gICAgICB9KTtcbiAgICB9XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhZGRMYW1iZGFEYXRhU291cmNlQW5kUmVzb2x2ZXJzKFxuICAgIGZ1bmN0aW9uTmFtZTogc3RyaW5nLFxuICAgIGlkOiBzdHJpbmcsXG4gICAgbGFtYmRhRnVuY3Rpb246IElGdW5jdGlvbixcbiAgICBvcHRpb25zPzogRGF0YVNvdXJjZU9wdGlvbnMsXG4gICk6IExhbWJkYURhdGFTb3VyY2Uge1xuICAgIGNvbnN0IGZ1bmN0aW9uRGF0YVNvdXJjZSA9IHRoaXMuYXBwc3luY0FQSS5hZGRMYW1iZGFEYXRhU291cmNlKFxuICAgICAgaWQsXG4gICAgICBsYW1iZGFGdW5jdGlvbixcbiAgICAgIG9wdGlvbnMsXG4gICAgKTtcblxuICAgIGZvciAoY29uc3QgcmVzb2x2ZXIgb2YgdGhpcy5mdW5jdGlvblJlc29sdmVyc1tmdW5jdGlvbk5hbWVdKSB7XG4gICAgICBuZXcgUmVzb2x2ZXIoXG4gICAgICAgIHRoaXMubmVzdGVkQXBwc3luY1N0YWNrLFxuICAgICAgICBgJHtyZXNvbHZlci50eXBlTmFtZX0tJHtyZXNvbHZlci5maWVsZE5hbWV9LXJlc29sdmVyYCxcbiAgICAgICAge1xuICAgICAgICAgIGFwaTogdGhpcy5hcHBzeW5jQVBJLFxuICAgICAgICAgIHR5cGVOYW1lOiByZXNvbHZlci50eXBlTmFtZSxcbiAgICAgICAgICBmaWVsZE5hbWU6IHJlc29sdmVyLmZpZWxkTmFtZSxcbiAgICAgICAgICBkYXRhU291cmNlOiBmdW5jdGlvbkRhdGFTb3VyY2UsXG4gICAgICAgICAgcmVxdWVzdE1hcHBpbmdUZW1wbGF0ZTogTWFwcGluZ1RlbXBsYXRlLmZyb21TdHJpbmcoXG4gICAgICAgICAgICByZXNvbHZlci5kZWZhdWx0UmVxdWVzdE1hcHBpbmdUZW1wbGF0ZSxcbiAgICAgICAgICApLFxuICAgICAgICAgIHJlc3BvbnNlTWFwcGluZ1RlbXBsYXRlOiBNYXBwaW5nVGVtcGxhdGUuZnJvbVN0cmluZyhcbiAgICAgICAgICAgIHJlc29sdmVyLmRlZmF1bHRSZXNwb25zZU1hcHBpbmdUZW1wbGF0ZSxcbiAgICAgICAgICApLCAvLyBUaGlzIGRlZmF1bHRzIHRvIGFsbG93IGVycm9ycyB0byByZXR1cm4gdG8gdGhlIGNsaWVudCBpbnN0ZWFkIG9mIHRocm93aW5nXG4gICAgICAgIH0sXG4gICAgICApO1xuICAgIH1cblxuICAgIHJldHVybiBmdW5jdGlvbkRhdGFTb3VyY2U7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkRHluYW1vREJTdHJlYW0ocHJvcHM6IER5bmFtb0RCU3RyZWFtUHJvcHMpOiBzdHJpbmcge1xuICAgIGNvbnN0IHRhYmxlTmFtZSA9IGAke3Byb3BzLm1vZGVsVHlwZU5hbWV9VGFibGVgO1xuICAgIGNvbnN0IHRhYmxlID0gdGhpcy50YWJsZU1hcFt0YWJsZU5hbWVdO1xuICAgIGlmICghdGFibGUpIHRocm93IG5ldyBFcnJvcihgVGFibGUgd2l0aCBuYW1lICcke3RhYmxlTmFtZX0nIG5vdCBmb3VuZC5gKTtcblxuICAgIGNvbnN0IGNmblRhYmxlID0gdGFibGUubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuVGFibGU7XG4gICAgY2ZuVGFibGUuc3RyZWFtU3BlY2lmaWNhdGlvbiA9IHtcbiAgICAgIHN0cmVhbVZpZXdUeXBlOiBwcm9wcy5zdHJlYW1WaWV3VHlwZSxcbiAgICB9O1xuXG4gICAgcmV0dXJuIGNmblRhYmxlLmF0dHJTdHJlYW1Bcm47XG4gIH1cbn1cblxuZXhwb3J0IGludGVyZmFjZSBEeW5hbW9EQlN0cmVhbVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG1vZGVsVHlwZU5hbWU6IHN0cmluZztcbiAgcmVhZG9ubHkgc3RyZWFtVmlld1R5cGU6IFN0cmVhbVZpZXdUeXBlO1xufVxuIl19