"""
proxy for database connection

"""
import atexit
import copy
import datetime
import glob
import json
import logging
import math
import operator
import os
import random
import re
import socket
import sys
import time
import traceback
import uuid
import warnings

from pandacommon.pandalogger.LogWrapper import LogWrapper
from pandacommon.pandalogger.PandaLogger import PandaLogger

from pandaserver.config import panda_config
from pandaserver.srvcore import CoreUtils, srv_msg_utils
from pandaserver.taskbuffer import (
    ErrorCode,
    EventServiceUtils,
    GlobalShares,
    JobUtils,
    PrioUtil,
    ProcessGroups,
    SiteSpec,
    task_split_rules,
)
from pandaserver.taskbuffer.DatasetSpec import DatasetSpec
from pandaserver.taskbuffer.DdmSpec import DdmSpec
from pandaserver.taskbuffer.FileSpec import FileSpec
from pandaserver.taskbuffer.HarvesterMetricsSpec import HarvesterMetricsSpec
from pandaserver.taskbuffer.JobSpec import JobSpec, push_status_changes
from pandaserver.taskbuffer.ResourceSpec import (
    BASIC_RESOURCE_TYPE,
    ResourceSpec,
    ResourceSpecMapper,
)
from pandaserver.taskbuffer.SupErrors import SupErrors
from pandaserver.taskbuffer.Utils import create_shards
from pandaserver.taskbuffer.WorkerSpec import WorkerSpec
from pandaserver.taskbuffer.WrappedCursor import WrappedCursor

try:
    import idds.common.constants
    import idds.common.utils
    from idds.client.client import Client as iDDS_Client
except ImportError:
    pass

if panda_config.backend == "oracle":
    import oracledb

    from . import wrapped_oracle_conn

    oracledb.init_oracle_client()
    varNUMBER = oracledb.NUMBER

elif panda_config.backend == "postgres":
    import psycopg2 as psycopg
    import psycopg2.errorcodes as psycopg_errorcodes

    from . import WrappedPostgresConn

    varNUMBER = int

else:
    import MySQLdb

    varNUMBER = int

warnings.filterwarnings("ignore")

# logger
_logger = PandaLogger().getLogger("DBProxy")
_loggerFiltered = PandaLogger().getLogger("DBProxyFiltered")

# add handlers
for handler in _loggerFiltered.handlers:
    handler.setLevel(logging.INFO)
    _logger.addHandler(handler)
    _loggerFiltered.removeHandler(handler)


# get mb proxies used in DBProxy methods
def get_mb_proxy_dict():
    try:
        if hasattr(panda_config, "mq_configFile") and panda_config.mq_configFile:
            # delay import to open logger file inside python daemon
            from pandaserver.taskbuffer.PanDAMsgProcessor import MsgProcAgent

            out_q_list = [
                "panda_jobstatus",
                "panda_jedi",
                "panda_pilot_topic",
                "panda_pilot_queue",
            ]
            mp_agent = MsgProcAgent(config_file=panda_config.mq_configFile)
            mb_proxy_dict = mp_agent.start_passive_mode(in_q_list=[], out_q_list=out_q_list)
            # stop with atexit
            atexit.register(mp_agent.stop_passive_mode)
            # return
            return mb_proxy_dict
    except Exception as exc:
        _logger.error(f"Error on get_mb_proxy_dict : {traceback.format_exc()}")
    return {}


# convert dict to bind variable dict
def convert_dict_to_bind_vars(item):
    ret = dict()
    for k in item:
        ret[f":{k}"] = item[k]
    return ret


# topics in SQL_QUEUE
SQL_QUEUE_TOPIC_async_dataset_update = "async_dataset_update"


# proxy
class DBProxy:
    # constructor
    def __init__(self, useOtherError=False):
        # connection object
        self.conn = None
        # cursor object
        self.cur = None
        # host name
        self.hostname = None
        # retry count
        self.nTry = 5
        # use special error codes for reconnection in querySQL
        self.useOtherError = useOtherError
        # pledge resource ratio
        self.beyondPledgeRatio = {}
        # update time for pledge resource ratio
        self.updateTimeForPledgeRatio = None

        # hostname
        self.myHostName = socket.getfqdn()
        self.backend = panda_config.backend

        # global share variables
        self.tree = None  # Pointer to the root of the global shares tree
        self.leave_shares = None  # Pointer to the list with leave shares
        self.__t_update_shares = None  # Timestamp when the shares were last updated
        self.__hs_distribution = None  # HS06s distribution of sites
        self.__t_update_distribution = None  # Timestamp when the HS06s distribution was last updated

        # resource type mapper
        # if you want to use it, you need to call __reload_resource_spec_mapper first
        self.__resource_spec_mapper = None
        self.__t_update_resource_type_mapper = None

        # priority boost
        self.job_prio_boost_dict = None
        self.job_prio_boost_dict_update_time = None

        # keep type
        self.__orig_type = type

        # mb proxy
        self.mb_proxy_dict = None

    # connect to DB
    def connect(
        self,
        dbhost=panda_config.dbhost,
        dbpasswd=panda_config.dbpasswd,
        dbuser=panda_config.dbuser,
        dbname=panda_config.dbname,
        dbtimeout=panda_config.dbtimeout,
        reconnect=False,
        dbport=panda_config.dbport,
    ):
        _logger.debug(f"connect : re={reconnect}")
        # keep parameters for reconnect
        if not reconnect:
            self.dbhost = dbhost
            self.dbpasswd = dbpasswd
            self.dbuser = dbuser
            self.dbname = dbname
            self.dbtimeout = dbtimeout
            self.dbport = dbport
        # close old connection
        if reconnect:
            _logger.debug("closing old connection")
            try:
                self.conn.close()
            except Exception:
                _logger.debug("failed to close old connection")
        # connect
        try:
            if self.backend == "oracle":
                conn = oracledb.connect(dsn=self.dbhost, user=self.dbuser, password=self.dbpasswd)

                def OutputTypeHandler(cursor, name, defaultType, size, precision, scale):
                    if defaultType == oracledb.CLOB:
                        return cursor.var(oracledb.LONG_STRING, arraysize=cursor.arraysize)

                conn.outputtypehandler = OutputTypeHandler
                self.conn = wrapped_oracle_conn.WrappedOracleConn(conn)

            elif self.backend == "postgres":
                dsn = {"dbname": self.dbname, "user": self.dbuser, "keepalives_idle": 30, "keepalives_interval": 30, "keepalives": 1}
                if self.dbpasswd:
                    dsn["password"] = self.dbpasswd
                if self.dbhost:
                    dsn["host"] = self.dbhost
                if self.dbport:
                    dsn["port"] = self.dbport
                conn = psycopg.connect(**dsn)
                self.conn = WrappedPostgresConn.WrappedPostgresConn(conn)
            else:
                self.conn = MySQLdb.connect(
                    host=self.dbhost,
                    db=self.dbname,
                    port=self.dbport,
                    connect_timeout=self.dbtimeout,
                    user=self.dbuser,
                    passwd=self.dbpasswd,
                    charset="utf8",
                )
            self.cur = WrappedCursor(self.conn)
            try:
                # use SQL dumper
                if panda_config.dump_sql:
                    from pandaserver.taskbuffer.SQLDumper import SQLDumper

                    self.cur = SQLDumper(self.cur)
            except Exception:
                pass
            self.hostname = self.cur.initialize()
            if not reconnect:
                atexit.register(self.close_connection)
            _logger.debug(f"connect : re={reconnect} ready")
            return True
        except Exception as e:
            _logger.error(f"connect : {str(e)}")
            return False

    # close connection
    def close_connection(self):
        if self.conn:
            try:
                self.conn.close()
            except Exception:
                pass
        return

    # cleanup
    def cleanup(self):
        _logger.debug("cleanup start")
        self.close_connection()
        atexit.unregister(self.close_connection)
        _logger.debug("cleanup done")

    def getvalue_corrector(self, value):
        """
        Needed to support old and new versions of cx_Oracle
        :return:
        """
        if isinstance(value, list):  # cx_Oracle version >= 6.3
            return value[0]
        else:  # cx_Oracle version < 6.3
            return value

    # Internal caching of a result. Use only for information with low update frequency and low memory footprint
    def memoize(f):
        memo = {}
        kwd_mark = object()

        def helper(self, *args, **kwargs):
            now = datetime.datetime.now()
            key = args + (kwd_mark,) + tuple(sorted(kwargs.items()))
            if key not in memo or memo[key]["timestamp"] < now - datetime.timedelta(hours=1):
                tmp_data = {"value": f(self, *args, **kwargs), "timestamp": now}
                memo[key] = tmp_data
            return memo[key]["value"]

        return helper

    # query an SQL
    def querySQL(self, sql, arraySize=1000):
        comment = " /* DBProxy.querySQL */"
        try:
            _logger.debug(f"querySQL : {sql} ")
            # begin transaction
            self.conn.begin()
            self.cur.arraysize = arraySize
            self.cur.execute(sql + comment)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            return res
        except Exception:
            # roll back
            self._rollback(self.useOtherError)
            type, value, traceBack = sys.exc_info()
            _logger.error(f"querySQL : {sql} ")
            _logger.error(f"querySQL : {type} {value}")
            return None

    # query an SQL return Status
    def querySQLS(self, sql, varMap, arraySize=1000):
        comment = " /* DBProxy.querySQLS */"
        try:
            # begin transaction
            self.conn.begin()
            self.cur.arraysize = arraySize
            ret = self.cur.execute(sql + comment, varMap)
            if ret:
                ret = True
            if sql.startswith("INSERT") or sql.startswith("UPDATE") or sql.startswith("DELETE"):
                res = self.cur.rowcount
            else:
                res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            return ret, res
        except Exception as e:
            # roll back
            self._rollback(self.useOtherError)
            _logger.error(f"querySQLS : {sql} {str(varMap)}")
            _logger.error(f"querySQLS : {str(e)}")
            return -1, None

    # execute an SQL return with executemany
    def executemanySQL(self, sql, varMaps, arraySize=1000):
        comment = " /* DBProxy.executemanySQL */"
        try:
            # begin transaction
            self.conn.begin()
            self.cur.arraysize = arraySize
            ret = self.cur.executemany(sql + comment, varMaps)
            if ret:
                ret = True
            if sql.startswith("INSERT") or sql.startswith("UPDATE") or sql.startswith("DELETE"):
                res = self.cur.rowcount
            else:
                raise RuntimeError("Operation unsupported. Only INSERT, UPDATE, DELETE are allowed")
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            return res
        except Exception as e:
            # roll back
            self._rollback(self.useOtherError)
            _logger.error(f"executemanySQL : {sql} {str(varMaps)}")
            _logger.error(f"executemanySQL : {str(e)}")
            return None

    # get CLOB
    def getClobObj(self, sql, varMap, arraySize=10000, use_commit=True):
        comment = " /* DBProxy.getClobObj */"
        try:
            # begin transaction
            if use_commit:
                self.conn.begin()
                self.cur.arraysize = arraySize
            ret = self.cur.execute(sql + comment, varMap)
            if ret:
                ret = True
            res = []
            for items in self.cur:
                resItem = []
                for item in items:
                    # read CLOB
                    try:
                        itemRead = item.read()
                    except AttributeError:
                        itemRead = item
                    resItem.append(itemRead)
                # append
                res.append(resItem)
            # commit
            if use_commit:
                if not self._commit():
                    raise RuntimeError("Commit error")
            return ret, res
        except Exception as e:
            # roll back
            if use_commit:
                self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error(f"getClobObj : {sql} {str(varMap)}")
            _logger.error(f"getClobObj : {str(e)}")
            return -1, None

    # get configuration value. cached for an hour
    @memoize
    def getConfigValue(self, component, key, app="pandaserver", vo=None):
        comment = " /* DBProxy.getConfigValue */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        varMap = {":component": component, ":key": key, ":app": app}
        sql = """
        SELECT value, type FROM ATLAS_PANDA.CONFIG
        WHERE component=:component
        AND key=:key
        AND app=:app
        """

        # If VO is specified, select only the config values for this VO or VO independent values
        if vo:
            varMap[":vo"] = vo
            sql += "AND (vo=:vo or vo IS NULL)"

        self.cur.execute(sql + comment, varMap)

        try:
            value_str, type = self.cur.fetchone()
        except TypeError:
            error_message = f"Specified key={key} not found for component={component} app={app}"
            _logger.debug(error_message)
            return None

        try:
            if type.lower() in ("str", "string"):
                return value_str
            elif type.lower() in ("int", "integer"):
                return int(value_str)
            elif type.lower() == "float":
                return float(value_str)
            elif type.lower() in ("bool", "boolean"):
                if value_str.lower() == "true":
                    return True
                else:
                    return False
            else:
                raise ValueError
        except ValueError:
            error_message = f"Wrong value/type pair. Value: {value_str}, Type: {type}"
            _logger.debug(error_message)
            return None

    # insert job to jobsDefined
    def insertNewJob(
        self,
        job,
        user,
        serNum,
        weight=0.0,
        priorityOffset=0,
        userVO=None,
        toPending=False,
        origEsJob=False,
        eventServiceInfo=None,
        oldPandaIDs=None,
        relationType=None,
        fileIDPool=[],
        origSpecialHandling=None,
        unprocessedMap=None,
        prio_reduction=True,
        no_late_bulk_exec=True,
        extracted_sqls=None,
        new_jobset_id=None,
    ):
        comment = " /* DBProxy.insertNewJob */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <JediTaskID={job.jediTaskID} idPool={len(fileIDPool)}>"
        tmp_log = LogWrapper(_logger, methodName)

        # insert jobs to jobsDefined4 or jobsWaiting4
        if not toPending:
            sql1 = f"INSERT INTO ATLAS_PANDA.jobsDefined4 ({JobSpec.columnNames()}) "
        else:
            sql1 = f"INSERT INTO ATLAS_PANDA.jobsWaiting4 ({JobSpec.columnNames()}) "
        sql1 += JobSpec.bindValuesExpression(useSeq=False)

        # job status
        if not toPending:
            job.jobStatus = "defined"
        else:
            if job.computingSite == EventServiceUtils.siteIdForWaitingCoJumboJobs:
                # put waiting co-jumbo jobs to waiting
                job.jobStatus = "waiting"
            else:
                job.jobStatus = "pending"

        # host and time information
        job.modificationHost = self.hostname
        job.creationTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
        job.modificationTime = job.creationTime
        job.stateChangeTime = job.creationTime
        job.prodDBUpdateTime = job.creationTime
        # DN
        if job.prodUserID == "NULL" or job.prodSourceLabel in ["user", "panda"]:
            job.prodUserID = user

        # compact username
        job.prodUserName = self.cleanUserID(job.prodUserID)
        if job.prodUserName in ["", "NULL"]:
            # use prodUserID as compact user name
            job.prodUserName = job.prodUserID

        # VO
        job.VO = userVO

        # priority
        if job.assignedPriority != "NULL":
            job.currentPriority = job.assignedPriority
        if job.prodSourceLabel == "install":
            job.currentPriority = 4100
        elif job.prodUserName in ["artprod"] and job.prodSourceLabel in [
            "user",
            "panda",
        ]:
            job.currentPriority = 7000
        elif job.prodSourceLabel == "user":
            if job.processingType == "pmerge" and job.currentPriority not in ["NULL", None]:
                # avoid prio reduction for merge jobs
                pass
            else:
                if not prio_reduction:
                    job.currentPriority = priorityOffset
                    if job.isScoutJob():
                        job.currentPriority += 1
                elif job.currentPriority not in ["NULL", None] and (job.isScoutJob() or job.currentPriority >= JobUtils.priorityTasksToJumpOver):
                    pass
                else:
                    job.currentPriority = PrioUtil.calculatePriority(priorityOffset, serNum, weight)
                    if "express" in job.specialHandling:
                        job.currentPriority = 6000
        elif job.prodSourceLabel == "panda":
            job.currentPriority = 2000 + priorityOffset
            if "express" in job.specialHandling:
                job.currentPriority = 6500

        # set attempt numbers
        if job.prodSourceLabel in ["user", "panda"] + JobUtils.list_ptest_prod_sources:
            if job.attemptNr in [None, "NULL", ""]:
                job.attemptNr = 0
            if job.maxAttempt in [None, "NULL", ""]:
                job.maxAttempt = 0
            # set maxAttempt to attemptNr to disable server/pilot retry
            if job.maxAttempt == -1:
                job.maxAttempt = job.attemptNr
            else:
                # set maxAttempt to have server/pilot retries for retried jobs
                if job.maxAttempt <= job.attemptNr:
                    job.maxAttempt = job.attemptNr + 2

        # obtain the share and resource type
        if job.gshare in ("NULL", None, ""):
            job.gshare = self.get_share_for_job(job)
        tmp_log.debug(f"resource_type is set to {job.resource_type}")
        tmp_log.debug(f"jediTaskID={job.jediTaskID} SH={origSpecialHandling} origEsJob={origEsJob} eInfo={eventServiceInfo}")
        if job.resource_type in ("NULL", None, ""):
            try:
                job.resource_type = self.get_resource_type_job(job)
                tmp_log.debug(f"reset resource_type to {job.resource_type}")
            except Exception:
                job.resource_type = "Undefined"
                tmp_log.error(f"reset resource_type excepted with: {traceback.format_exc()}")

        try:
            # use JEDI
            if hasattr(panda_config, "useJEDI") and panda_config.useJEDI is True and job.lockedby == "jedi":
                useJEDI = True
            else:
                useJEDI = False

            # begin transaction
            if no_late_bulk_exec:
                self.conn.begin()

            # get jobsetID for event service
            if origEsJob:
                if not no_late_bulk_exec:
                    job.jobsetID = new_jobset_id
                else:
                    if self.backend == "mysql":
                        # fake sequence
                        sql = " INSERT INTO ATLAS_PANDA.JOBSDEFINED4_PANDAID_SEQ (col) VALUES (NULL) "
                        self.cur.arraysize = 10
                        self.cur.execute(sql + comment, {})
                        sql2 = """ SELECT LAST_INSERT_ID() """
                        self.cur.execute(sql2 + comment, {})
                        (job.jobsetID,) = self.cur.fetchone()
                    else:
                        sqlESS = "SELECT ATLAS_PANDA.JOBSDEFINED4_PANDAID_SEQ.nextval FROM dual "
                        self.cur.arraysize = 10
                        self.cur.execute(sqlESS + comment, {})
                        (job.jobsetID,) = self.cur.fetchone()

            # get originPandaID
            originPandaID = None
            if oldPandaIDs is not None and len(oldPandaIDs) > 0:
                varMap = {}
                varMap[":jediTaskID"] = job.jediTaskID
                varMap[":pandaID"] = oldPandaIDs[0]
                sqlOrigin = f"SELECT originPandaID FROM {panda_config.schemaJEDI}.JEDI_Job_Retry_History "
                sqlOrigin += "WHERE jediTaskID=:jediTaskID AND newPandaID=:pandaID "
                sqlOrigin += "AND (relationType IS NULL OR NOT relationType IN ("
                for tmpType in EventServiceUtils.relationTypesForJS:
                    tmpKey = f":{tmpType}"
                    sqlOrigin += f"{tmpKey},"
                    varMap[tmpKey] = tmpType
                sqlOrigin = sqlOrigin[:-1]
                sqlOrigin += ")) "
                self.cur.execute(sqlOrigin + comment, varMap)
                resOrigin = self.cur.fetchone()
                if resOrigin is not None:
                    (originPandaID,) = resOrigin
                else:
                    originPandaID = oldPandaIDs[0]
            if originPandaID is None:
                originPandaID = job.PandaID
            newJobName = re.sub("\$ORIGINPANDAID", str(originPandaID), job.jobName)
            # update jobName
            if newJobName != job.jobName:
                job.jobName = newJobName

            # insert job
            if no_late_bulk_exec:
                varMap = job.valuesMap(useSeq=False)
                self.cur.execute(sql1 + comment, varMap)
            else:
                extracted_sqls["job"] = {"sql": sql1 + comment, "vars": [job.valuesMap(useSeq=False)]}

            # get jobsetID
            if job.jobsetID in [None, "NULL", -1]:
                jobsetID = 0
            else:
                jobsetID = job.jobsetID
            jobsetID = "%06d" % jobsetID
            try:
                strJediTaskID = str(job.jediTaskID)
            except Exception:
                strJediTaskID = ""

            # reset changed attribute list
            job.resetChangedList()
            # insert files
            tmp_log.debug(f"inserted {job.PandaID} label:{job.prodSourceLabel} prio:{job.currentPriority} jediTaskID:{job.jediTaskID}")
            # sql with SEQ
            sqlFile = f"INSERT INTO ATLAS_PANDA.filesTable4 ({FileSpec.columnNames()}) "
            sqlFile += FileSpec.bindValuesExpression(useSeq=True)
            sqlFile += " RETURNING row_ID INTO :newRowID"
            # sql without SEQ
            sqlFileW = f"INSERT INTO ATLAS_PANDA.filesTable4 ({FileSpec.columnNames()}) "
            sqlFileW += FileSpec.bindValuesExpression(useSeq=False)
            dynNumEvents = EventServiceUtils.isDynNumEventsSH(job.specialHandling)
            dynFileMap = {}
            dynLfnIdMap = {}
            totalInputEvents = 0
            indexFileID = 0
            varMapsForFile = []
            nFilesWaitingMap = {}
            nEventsToProcess = 0

            # failed related ES jobs
            if origEsJob and eventServiceInfo is not None and not job.notDiscardEvents():
                self.updateRelatedEventServiceJobs(job, killEvents=False, forceFailed=True)
            for file in job.Files:
                file.row_ID = None
                if file.status not in ["ready", "cached"]:
                    file.status = "unknown"
                # replace $PANDAID with real PandaID
                file.lfn = re.sub("\$PANDAID", "%05d" % job.PandaID, file.lfn)
                # replace $JOBSETID with real jobsetID
                if job.prodSourceLabel not in ["managed"]:
                    file.lfn = re.sub("\$JOBSETID", jobsetID, file.lfn)
                    try:
                        file.lfn = re.sub("\$JEDITASKID", strJediTaskID, file.lfn)
                    except Exception:
                        pass
                # avoid duplicated files for dynamic number of events
                toSkipInsert = False
                if dynNumEvents and file.type in ["input", "pseudo_input"]:
                    if file.lfn not in dynFileMap:
                        dynFileMap[file.lfn] = set()
                    else:
                        toSkipInsert = True
                        dynFileMap[file.lfn].add(
                            (
                                file.jediTaskID,
                                file.datasetID,
                                file.fileID,
                                file.attemptNr,
                            )
                        )
                # set scope
                if file.type in ["output", "log"] and job.VO in ["atlas"]:
                    file.scope = self.extractScope(file.dataset)
                # insert
                if not toSkipInsert:
                    if indexFileID < len(fileIDPool):
                        file.row_ID = fileIDPool[indexFileID]
                        varMap = file.valuesMap(useSeq=False)
                        varMapsForFile.append(varMap)
                        indexFileID += 1
                    else:
                        varMap = file.valuesMap(useSeq=True)
                        varMap[":newRowID"] = self.cur.var(varNUMBER)
                        self.cur.execute(sqlFile + comment, varMap)
                        # get rowID
                        val = self.getvalue_corrector(self.cur.getvalue(varMap[":newRowID"]))
                        file.row_ID = int(val)
                    dynLfnIdMap[file.lfn] = file.row_ID
                    # reset changed attribute list
                    file.resetChangedList()
                # update JEDI table
                if useJEDI:
                    # skip if no JEDI
                    if file.fileID == "NULL":
                        continue
                    # input for waiting co-jumbo jobs
                    isWaiting = None
                    isFileForWaitingCoJumbo = False
                    if file.type not in ["output", "log"]:
                        if job.computingSite == EventServiceUtils.siteIdForWaitingCoJumboJobs:
                            isFileForWaitingCoJumbo = True
                        # check is_waiting
                        sqlJediFileIsW = "SELECT is_waiting FROM ATLAS_PANDA.JEDI_Dataset_Contents "
                        sqlJediFileIsW += " WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                        varMap = {}
                        varMap[":fileID"] = file.fileID
                        varMap[":jediTaskID"] = file.jediTaskID
                        varMap[":datasetID"] = file.datasetID
                        self.cur.execute(sqlJediFileIsW + comment, varMap)
                        resJediFileIsW = self.cur.fetchone()
                        if resJediFileIsW is not None:
                            (isWaiting,) = resJediFileIsW
                    # update Dataset_Contents table
                    varMap = {}
                    varMap[":fileID"] = file.fileID
                    if isFileForWaitingCoJumbo:
                        # not change status for wating co-jumbo jobs to allow new jobs to pickup files
                        varMap[":status"] = "picked"
                        varMap[":is_waiting"] = "Y"
                    else:
                        varMap[":status"] = "running"
                    varMap[":oldStatusI"] = "picked"
                    varMap[":oldStatusO"] = "defined"
                    varMap[":attemptNr"] = file.attemptNr
                    varMap[":datasetID"] = file.datasetID
                    varMap[":keepTrack"] = 1
                    varMap[":jediTaskID"] = file.jediTaskID
                    if isFileForWaitingCoJumbo:
                        varMap[":PandaID"] = job.jobsetID
                    else:
                        varMap[":PandaID"] = file.PandaID
                    varMap[":jobsetID"] = job.jobsetID
                    sqlJediFile = "UPDATE /*+ INDEX_RS_ASC(JEDI_DATASET_CONTENTS (JEDI_DATASET_CONTENTS.JEDITASKID JEDI_DATASET_CONTENTS.DATASETID JEDI_DATASET_CONTENTS.FILEID)) */ ATLAS_PANDA.JEDI_Dataset_Contents SET status=:status,PandaID=:PandaID,jobsetID=:jobsetID"
                    if file.type in ["output", "log"]:
                        sqlJediFile += ",outPandaID=:PandaID"
                    if isFileForWaitingCoJumbo:
                        sqlJediFile += ",is_waiting=:is_waiting"
                    else:
                        sqlJediFile += ",is_waiting=NULL"
                    sqlJediFile += " WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                    sqlJediFile += "AND attemptNr=:attemptNr AND status IN (:oldStatusI,:oldStatusO) AND keepTrack=:keepTrack "
                    self.cur.execute(sqlJediFile + comment, varMap)
                    # get number of inputs for waiting co-jumbo jobs
                    if (isFileForWaitingCoJumbo or isWaiting is not None) and self.cur.rowcount > 0:
                        if file.datasetID not in nFilesWaitingMap:
                            nFilesWaitingMap[file.datasetID] = 0
                        if isFileForWaitingCoJumbo and isWaiting is None:
                            nFilesWaitingMap[file.datasetID] += 1
                        elif not isFileForWaitingCoJumbo and isWaiting is not None:
                            nFilesWaitingMap[file.datasetID] -= 1
                    # no insert for dynamic number of events
                    if toSkipInsert:
                        continue
                    # insert events for ES
                    if origEsJob and eventServiceInfo is not None and file.lfn in eventServiceInfo:
                        # discard old successful event ranges
                        sqlJediOdEvt = (
                            "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                        )
                        sqlJediOdEvt += f"{panda_config.schemaJEDI}.JEDI_Events tab "
                        sqlJediOdEvt += "SET status=:newStatus "
                        sqlJediOdEvt += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                        sqlJediOdEvt += "AND status IN (:esFinished,:esDone) "
                        varMap = {}
                        varMap[":jediTaskID"] = file.jediTaskID
                        varMap[":datasetID"] = file.datasetID
                        varMap[":fileID"] = file.fileID
                        if not job.notDiscardEvents():
                            varMap[":newStatus"] = EventServiceUtils.ST_discarded
                        else:
                            varMap[":newStatus"] = EventServiceUtils.ST_done
                        varMap[":esDone"] = EventServiceUtils.ST_done
                        varMap[":esFinished"] = EventServiceUtils.ST_finished
                        _logger.debug(sqlJediOdEvt + comment + str(varMap))
                        self.cur.execute(sqlJediOdEvt + comment, varMap)
                        # cancel old unprocessed event ranges
                        sqlJediCEvt = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                        sqlJediCEvt += f"{panda_config.schemaJEDI}.JEDI_Events tab "
                        sqlJediCEvt += "SET status=:newStatus "
                        sqlJediCEvt += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                        sqlJediCEvt += "AND NOT status IN (:esFinished,:esDone,:esDiscarded,:esCancelled,:esFailed,:esFatal,:esCorrupted) "
                        sqlJediCEvt += "AND (is_jumbo IS NULL OR (is_jumbo=:isJumbo AND status NOT IN (:esSent,:esRunning))) "
                        varMap[":newStatus"] = EventServiceUtils.ST_cancelled
                        varMap[":esDiscarded"] = EventServiceUtils.ST_discarded
                        varMap[":esCancelled"] = EventServiceUtils.ST_cancelled
                        varMap[":esCorrupted"] = EventServiceUtils.ST_corrupted
                        varMap[":esFatal"] = EventServiceUtils.ST_fatal
                        varMap[":esFailed"] = EventServiceUtils.ST_failed
                        varMap[":esSent"] = EventServiceUtils.ST_sent
                        varMap[":esRunning"] = EventServiceUtils.ST_running
                        varMap[":isJumbo"] = EventServiceUtils.eventTableIsJumbo
                        tmp_log.debug(sqlJediCEvt + comment + str(varMap))
                        self.cur.execute(sqlJediCEvt + comment, varMap)
                        # unset processed_upto for old failed events
                        sqlJediFEvt = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                        sqlJediFEvt += f"{panda_config.schemaJEDI}.JEDI_Events tab "
                        sqlJediFEvt += "SET processed_upto_eventID=NULL "
                        sqlJediFEvt += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                        sqlJediFEvt += "AND status=:esFailed AND processed_upto_eventID IS NOT NULL "
                        varMap = {}
                        varMap[":jediTaskID"] = file.jediTaskID
                        varMap[":datasetID"] = file.datasetID
                        varMap[":fileID"] = file.fileID
                        varMap[":esFailed"] = EventServiceUtils.ST_failed
                        tmp_log.debug(sqlJediFEvt + comment + str(varMap))
                        self.cur.execute(sqlJediFEvt + comment, varMap)
                        # event range with offset
                        withOffset = False
                        if "offset" in eventServiceInfo[file.lfn] and eventServiceInfo[file.lfn]["offset"] != -1:
                            withOffset = True
                        # get sucessful event ranges
                        okRanges = set()
                        if job.notDiscardEvents():
                            sqlJediOks = (
                                "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                            )
                            sqlJediOks += f"jediTaskID,fileID,job_processID FROM {panda_config.schemaJEDI}.JEDI_Events tab "
                            sqlJediOks += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                            sqlJediOks += "AND (status=:esDone OR (is_jumbo=:isJumbo AND status IN (:esSent,:esRunning))) "
                            varMap = {}
                            varMap[":jediTaskID"] = file.jediTaskID
                            varMap[":datasetID"] = file.datasetID
                            varMap[":fileID"] = file.fileID
                            varMap[":esDone"] = EventServiceUtils.ST_done
                            varMap[":esSent"] = EventServiceUtils.ST_sent
                            varMap[":esRunning"] = EventServiceUtils.ST_running
                            varMap[":isJumbo"] = EventServiceUtils.eventTableIsJumbo
                            self.cur.execute(sqlJediOks + comment, varMap)
                            resOks = self.cur.fetchall()
                            for (
                                tmpOk_jediTaskID,
                                tmpOk_fileID,
                                tmpOk_job_processID,
                            ) in resOks:
                                okRanges.add(f"{tmpOk_jediTaskID}-{tmpOk_fileID}-{tmpOk_job_processID}")
                        # insert new ranges
                        sqlJediEvent = f"INSERT INTO {panda_config.schemaJEDI}.JEDI_Events "
                        sqlJediEvent += "(jediTaskID,datasetID,PandaID,fileID,attemptNr,status,"
                        sqlJediEvent += "job_processID,def_min_eventID,def_max_eventID,processed_upto_eventID,"
                        sqlJediEvent += "event_offset"
                        sqlJediEvent += ") "
                        sqlJediEvent += "VALUES(:jediTaskID,:datasetID,:pandaID,:fileID,:attemptNr,:eventStatus,"
                        sqlJediEvent += ":startEvent,:startEvent,:lastEvent,:processedEvent,"
                        sqlJediEvent += ":eventOffset"
                        sqlJediEvent += ") "
                        varMaps = []
                        iEvent = 1
                        while iEvent <= eventServiceInfo[file.lfn]["nEvents"]:
                            varMap = {}
                            varMap[":jediTaskID"] = file.jediTaskID
                            varMap[":datasetID"] = file.datasetID
                            varMap[":pandaID"] = job.jobsetID
                            varMap[":fileID"] = file.fileID
                            varMap[":attemptNr"] = eventServiceInfo[file.lfn]["maxAttempt"]
                            varMap[":eventStatus"] = EventServiceUtils.ST_ready
                            varMap[":processedEvent"] = 0
                            varMap[":startEvent"] = eventServiceInfo[file.lfn]["startEvent"] + iEvent
                            iEvent += eventServiceInfo[file.lfn]["nEventsPerRange"]
                            if iEvent > eventServiceInfo[file.lfn]["nEvents"]:
                                iEvent = eventServiceInfo[file.lfn]["nEvents"] + 1
                            lastEvent = eventServiceInfo[file.lfn]["startEvent"] + iEvent - 1
                            varMap[":lastEvent"] = lastEvent
                            # add offset for positional event numbers
                            if not job.inFilePosEvtNum():
                                varMap[":startEvent"] += totalInputEvents
                                varMap[":lastEvent"] += totalInputEvents
                            # keep jobsetID
                            varMap[":eventOffset"] = job.jobsetID
                            # skip if already succeeded
                            tmpKey = f"{varMap[':jediTaskID']}-{varMap[':fileID']}-{varMap[':startEvent']}"
                            if tmpKey in okRanges:
                                continue
                            varMaps.append(varMap)
                            nEventsToProcess += 1
                        tmp_log.debug(f"{job.PandaID} insert {len(varMaps)} event ranges jediTaskID:{job.jediTaskID}")
                        if no_late_bulk_exec:
                            self.cur.executemany(sqlJediEvent + comment, varMaps)
                        else:
                            extracted_sqls["event"] = {"sql": sqlJediEvent + comment, "vars": varMaps}
                        tmp_log.debug(f"{job.PandaID} inserted {len(varMaps)} event ranges jediTaskID:{job.jediTaskID}")
                        totalInputEvents += eventServiceInfo[file.lfn]["nEvents"]
            if job.notDiscardEvents() and origEsJob and nEventsToProcess == 0:
                job.setAllOkEvents()
                if not toPending:
                    sqlJediJSH = "UPDATE ATLAS_PANDA.jobsDefined4 "
                else:
                    sqlJediJSH = "UPDATE ATLAS_PANDA.jobsWaiting4 "
                sqlJediJSH += "SET specialHandling=:specialHandling WHERE PandaID=:PandaID "
                varMap = dict()
                varMap[":specialHandling"] = job.specialHandling
                varMap[":PandaID"] = job.PandaID
                self.cur.execute(sqlJediJSH + comment, varMap)
            # use score if not so many events are available
            if origEsJob and unprocessedMap is not None:
                unprocessedMap[job.jobsetID] = nEventsToProcess
            if EventServiceUtils.isEventServiceJob(job) and not EventServiceUtils.isJobCloningJob(job) and unprocessedMap is not None:
                if job.coreCount not in [None, "", "NULL"] and job.coreCount > 1:
                    minUnprocessed = self.getConfigValue("dbproxy", "AES_MINEVENTSFORMCORE")
                    if minUnprocessed is not None:
                        minUnprocessed = max(minUnprocessed, job.coreCount)
                        if unprocessedMap[job.jobsetID] < minUnprocessed and unprocessedMap[job.jobsetID] > 0:
                            self.setScoreSiteToEs(job, f"insertNewJob : {job.PandaID}", comment)
            # bulk insert files
            if len(varMapsForFile) > 0:
                tmp_log.debug(f"{job.PandaID} bulk insert {len(varMapsForFile)} files for jediTaskID:{job.jediTaskID}")
                if no_late_bulk_exec:
                    self.cur.executemany(sqlFileW + comment, varMapsForFile)
                else:
                    extracted_sqls["file"] = {"sql": sqlFileW + comment, "vars": varMapsForFile}
            # update nFilesWaiting
            if len(nFilesWaitingMap) > 0:
                sqlJediNFW = f"UPDATE {panda_config.schemaJEDI}.JEDI_Datasets "
                sqlJediNFW += "SET nFilesWaiting=nFilesWaiting+:nDiff "
                sqlJediNFW += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                for tmpDatasetID in nFilesWaitingMap:
                    nDiff = nFilesWaitingMap[tmpDatasetID]
                    varMap = {}
                    varMap[":jediTaskID"] = job.jediTaskID
                    varMap[":datasetID"] = tmpDatasetID
                    varMap[":nDiff"] = nDiff
                    self.cur.execute(sqlJediNFW + comment, varMap)
            # insert events for dynamic number of events
            if dynFileMap != {}:
                # insert new ranges
                sqlJediEvent = f"INSERT INTO {panda_config.schemaJEDI}.JEDI_Events "
                sqlJediEvent += "(jediTaskID,datasetID,PandaID,fileID,attemptNr,status,"
                sqlJediEvent += "job_processID,def_min_eventID,def_max_eventID,processed_upto_eventID) "
                sqlJediEvent += "VALUES(:jediTaskID,:datasetID,:pandaID,:fileID,:attemptNr,:eventStatus,"
                sqlJediEvent += ":processID,:startEvent,:lastEvent,:processedEvent) "
                varMaps = []
                for tmpLFN in dynFileMap:
                    dynFiles = dynFileMap[tmpLFN]
                    for (
                        tmpJediTaskID,
                        tmpDatasetID,
                        tmpFileID,
                        tmpAttemptNr,
                    ) in dynFiles:
                        varMap = {}
                        varMap[":jediTaskID"] = tmpJediTaskID
                        varMap[":datasetID"] = tmpDatasetID
                        varMap[":pandaID"] = job.PandaID
                        varMap[":fileID"] = tmpFileID
                        varMap[":attemptNr"] = tmpAttemptNr + 1  # to avoid 0
                        varMap[":eventStatus"] = EventServiceUtils.ST_discarded
                        varMap[":processID"] = dynLfnIdMap[tmpLFN]
                        varMap[":processedEvent"] = -1
                        varMap[":startEvent"] = 0
                        varMap[":lastEvent"] = 0
                        varMaps.append(varMap)
                if no_late_bulk_exec:
                    self.cur.executemany(sqlJediEvent + comment, varMaps)
                else:
                    extracted_sqls["dynamic"] = {"sql": sqlJediEvent + comment, "vars": varMaps}
                tmp_log.debug(f"{job.PandaID} inserted {len(varMaps)} dyn events jediTaskID:{job.jediTaskID}")
            # update t_task
            if useJEDI and job.prodSourceLabel not in ["panda"] and job.processingType != "pmerge":
                varMap = {}
                varMap[":jediTaskID"] = job.jediTaskID
                varMap[":nJobs"] = 1
                schemaDEFT = self.getSchemaDEFT()
                sqlTtask = f"UPDATE {schemaDEFT}.T_TASK "
                sqlTtask += "SET total_req_jobs=total_req_jobs+:nJobs,timestamp=CURRENT_DATE "
                sqlTtask += "WHERE taskid=:jediTaskID "
                if no_late_bulk_exec:
                    tmp_log.debug(sqlTtask + comment + str(varMap))
                    self.cur.execute(sqlTtask + comment, varMap)
                else:
                    extracted_sqls["t_task"] = {"sql": sqlTtask + comment, "vars": [varMap]}
                tmp_log.debug(f"{job.PandaID} updated T_TASK jediTaskID:{job.jediTaskID}")
            # metadata
            if job.prodSourceLabel in ["user", "panda"] and job.metadata != "":
                sqlMeta = "INSERT INTO ATLAS_PANDA.metaTable (PandaID,metaData) VALUES (:PandaID,:metaData)"
                varMap = {}
                varMap[":PandaID"] = job.PandaID
                varMap[":metaData"] = job.metadata
                tmp_log.debug(f"{job.PandaID} inserting meta jediTaskID:{job.jediTaskID}")
                if no_late_bulk_exec:
                    self.cur.execute(sqlMeta + comment, varMap)
                else:
                    extracted_sqls["meta"] = {"sql": sqlMeta + comment, "vars": [varMap]}
                tmp_log.debug(f"{job.PandaID} inserted meta jediTaskID:{job.jediTaskID}")
            # job parameters
            if job.prodSourceLabel not in ["managed"]:
                job.jobParameters = re.sub("\$JOBSETID", jobsetID, job.jobParameters)
                try:
                    job.jobParameters = re.sub("\$JEDITASKID", strJediTaskID, job.jobParameters)
                except Exception:
                    pass
            sqlJob = "INSERT INTO ATLAS_PANDA.jobParamsTable (PandaID,jobParameters) VALUES (:PandaID,:param)"
            varMap = {}
            varMap[":PandaID"] = job.PandaID
            varMap[":param"] = job.jobParameters
            tmp_log.debug(f"{job.PandaID} inserting jobParam jediTaskID:{job.jediTaskID}")
            if no_late_bulk_exec:
                self.cur.execute(sqlJob + comment, varMap)
            else:
                extracted_sqls["jobparams"] = {"sql": sqlJob + comment, "vars": [varMap]}
            tmp_log.debug(f"{job.PandaID} inserted jobParam jediTaskID:{job.jediTaskID}")
            # update input
            if (
                useJEDI
                and not EventServiceUtils.isJumboJob(job)
                and job.computingSite != EventServiceUtils.siteIdForWaitingCoJumboJobs
                and not (EventServiceUtils.isEventServiceJob(job) and not origEsJob)
            ):
                self.updateInputStatusJedi(job.jediTaskID, job.PandaID, "queued", no_late_bulk_exec=no_late_bulk_exec, extracted_sqls=extracted_sqls)
            # record retry history
            if oldPandaIDs is not None and len(oldPandaIDs) > 0:
                tmp_log.debug(f"{job.PandaID} recording history nOld={len(oldPandaIDs)} jediTaskID:{job.jediTaskID}")
                self.recordRetryHistoryJEDI(job.jediTaskID, job.PandaID, oldPandaIDs, relationType, no_late_bulk_exec, extracted_sqls)
                tmp_log.debug(f"{job.PandaID} recorded history jediTaskID:{job.jediTaskID}")
            # record jobset
            if origEsJob:
                self.recordRetryHistoryJEDI(
                    job.jediTaskID,
                    job.PandaID,
                    [job.jobsetID],
                    EventServiceUtils.relationTypeJS_ID,
                    no_late_bulk_exec,
                    extracted_sqls,
                )
                # record jobset history
                if oldPandaIDs is not None and len(oldPandaIDs) > 0:
                    # get old jobsetID
                    for oldPandaID in oldPandaIDs:
                        oldJobsetID = self.getJobsetIDforPandaID(oldPandaID, job.jediTaskID)
                        if oldJobsetID is not None:
                            self.recordRetryHistoryJEDI(
                                job.jediTaskID,
                                job.jobsetID,
                                [oldJobsetID],
                                EventServiceUtils.relationTypeJS_Retry,
                                no_late_bulk_exec,
                                extracted_sqls,
                            )
            # record jobset mapping for event service
            if EventServiceUtils.isEventServiceJob(job) and EventServiceUtils.isResurrectConsumers(job.specialHandling):
                self.recordRetryHistoryJEDI(
                    job.jediTaskID,
                    job.jobsetID,
                    [job.PandaID],
                    EventServiceUtils.relationTypeJS_Map,
                    no_late_bulk_exec,
                    extracted_sqls,
                )
            if no_late_bulk_exec:
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                tmp_log.debug(f"{job.PandaID} all OK jediTaskID:{job.jediTaskID}")
                # record status change
                try:
                    self.recordStatusChange(job.PandaID, job.jobStatus, jobInfo=job)
                except Exception:
                    tmp_log.error("recordStatusChange in insertNewJob")
                self.push_job_status_message(job, job.PandaID, job.jobStatus, job.jediTaskID, origSpecialHandling)
            else:
                self.recordStatusChange(job.PandaID, job.jobStatus, jobInfo=job, no_late_bulk_exec=False, extracted_sqls=extracted_sqls)
            if unprocessedMap is not None:
                return True, unprocessedMap
            return True
        except Exception:
            # roll back
            if no_late_bulk_exec:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            if unprocessedMap is not None:
                return False, unprocessedMap
            return False

    # bulk insert new jobs
    def bulk_insert_new_jobs(self, jedi_task_id, arg_list, new_jobset_id_list, special_handling_list):
        method_name = f"bulk_insert_new_jobs < jediTaskID={jedi_task_id} > "
        tmp_log = LogWrapper(_logger, method_name)
        try:
            start_time = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            tmp_log.debug("start")
            sql_key_list = ["job", "event", "file", "dynamic", "t_task", "meta", "jobparams", "retry_history", "state_change", "jedi_input"]
            self.conn.begin()
            return_list = []
            extracted_sqls = {}
            for args, kwargs in arg_list:
                tmp_extracted_sqls = {}
                new_kwargs = {
                    "no_late_bulk_exec": False,
                    "extracted_sqls": tmp_extracted_sqls,
                }
                if kwargs.get("origEsJob"):
                    new_kwargs["new_jobset_id"] = new_jobset_id_list.pop(0)
                kwargs.update(new_kwargs)
                ret = self.insertNewJob(*args, **kwargs)
                job = args[0]
                if kwargs.get("unprocessedMap") is not None:
                    tmp_ret, _ = ret
                else:
                    tmp_ret = ret
                if not tmp_ret:
                    job.PandaID = None
                else:
                    # combine SQLs
                    for target_key in sql_key_list:
                        if target_key in tmp_extracted_sqls:
                            extracted_sqls.setdefault(target_key, {"sqls": [], "vars": {}})
                            if tmp_extracted_sqls[target_key]["sql"] not in extracted_sqls[target_key]["sqls"]:
                                extracted_sqls[target_key]["sqls"].append(tmp_extracted_sqls[target_key]["sql"])
                                extracted_sqls[target_key]["vars"][tmp_extracted_sqls[target_key]["sql"]] = []
                            extracted_sqls[target_key]["vars"][tmp_extracted_sqls[target_key]["sql"]] += tmp_extracted_sqls[target_key]["vars"]
                return_list.append([job, ret])
            # consolidate SQLs for t_task
            if "t_task" in extracted_sqls:
                for sql in extracted_sqls["t_task"]["sqls"]:
                    old_vars = extracted_sqls["t_task"]["vars"][sql]
                    n_jobs_map = {}
                    for var in old_vars:
                        n_jobs_map.setdefault(var[":jediTaskID"], 0)
                        n_jobs_map[var[":jediTaskID"]] += var[":nJobs"]
                    extracted_sqls["t_task"]["vars"][sql] = []
                    for k, v in n_jobs_map.items():
                        extracted_sqls["t_task"]["vars"][sql].append({":jediTaskID": k, ":nJobs": v})
            # bulk execution
            tmp_log.debug(f"bulk execution for {len(arg_list)} jobs")
            for target_key in sql_key_list:
                if target_key not in extracted_sqls:
                    continue
                for sql in extracted_sqls[target_key]["sqls"]:
                    self.cur.executemany(sql, extracted_sqls[target_key]["vars"][sql])
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # send messages
            for (job, _), special_handling in zip(return_list, special_handling_list):
                self.push_job_status_message(job, job.PandaID, job.jobStatus, job.jediTaskID, special_handling)
            exec_time = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - start_time
            tmp_log.debug("done OK. took %s.%03d sec" % (exec_time.seconds, exec_time.microseconds / 1000))
            return True, return_list
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, method_name)
            exec_time = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - start_time
            tmp_log.debug("done NG. took %s.%03d sec" % (exec_time.seconds, exec_time.microseconds / 1000))
            return False, None

    # activate job. move job from jobsDefined to jobsActive
    def activateJob(self, job):
        comment = " /* DBProxy.activateJob */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        if job is None:
            tmp_id = None
        else:
            tmp_id = job.PandaID
        tmpLog = LogWrapper(_logger, methodName + f" < PandaID={tmp_id} >")
        updatedFlag = False
        if job is None:
            tmpLog.debug("skip job=None")
            return True
        tmpLog.debug("start")
        sql0 = "SELECT row_ID FROM ATLAS_PANDA.filesTable4 WHERE PandaID=:PandaID AND type=:type AND NOT status IN (:status1,:status2) "
        sql1 = "DELETE FROM ATLAS_PANDA.jobsDefined4 "
        sql1 += "WHERE PandaID=:PandaID AND (jobStatus=:oldJobStatus1 OR jobStatus=:oldJobStatus2) AND commandToPilot IS NULL"
        sql2 = f"INSERT INTO ATLAS_PANDA.jobsActive4 ({JobSpec.columnNames()}) "
        sql2 += JobSpec.bindValuesExpression()
        # host and time information
        job.modificationTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
        # set stateChangeTime for defined->activated but not for assigned->activated
        if job.jobStatus in ["defined"]:
            job.stateChangeTime = job.modificationTime
        nTry = 3
        to_push = False
        for iTry in range(nTry):
            try:
                # check if all files are ready
                allOK = True
                for file in job.Files:
                    if file.type == "input" and file.status not in ["ready", "cached"]:
                        allOK = False
                        break
                # begin transaction
                self.conn.begin()
                # check all inputs are ready
                varMap = {}
                varMap[":type"] = "input"
                varMap[":status1"] = "ready"
                varMap[":status2"] = "cached"
                varMap[":PandaID"] = job.PandaID
                self.cur.arraysize = 100
                self.cur.execute(sql0 + comment, varMap)
                res = self.cur.fetchall()
                if len(res) == 0 or allOK:
                    # check resource share
                    job.jobStatus = "activated"

                    # delete
                    varMap = {}
                    varMap[":PandaID"] = job.PandaID
                    varMap[":oldJobStatus1"] = "assigned"
                    varMap[":oldJobStatus2"] = "defined"
                    self.cur.execute(sql1 + comment, varMap)
                    n = self.cur.rowcount
                    if n == 0:
                        # already killed or activated
                        tmpLog.debug("Job not found to activate")
                    else:
                        # insert
                        self.cur.execute(sql2 + comment, job.valuesMap())
                        # update files
                        datasetContentsStat = {}
                        for file in job.Files:
                            sqlF = f"UPDATE ATLAS_PANDA.filesTable4 SET {file.bindUpdateChangesExpression()}" + "WHERE row_ID=:row_ID"
                            varMap = file.valuesMap(onlyChanged=True)
                            if varMap != {}:
                                varMap[":row_ID"] = file.row_ID
                                _logger.debug(sqlF + comment + str(varMap))
                                self.cur.execute(sqlF + comment, varMap)
                        # job parameters
                        sqlJob = "UPDATE ATLAS_PANDA.jobParamsTable SET jobParameters=:param WHERE PandaID=:PandaID"
                        varMap = {}
                        varMap[":PandaID"] = job.PandaID
                        varMap[":param"] = job.jobParameters
                        self.cur.execute(sqlJob + comment, varMap)
                        updatedFlag = True
                        to_push = job.is_push_job()
                else:
                    # update job
                    sqlJ = (
                        f"UPDATE ATLAS_PANDA.jobsDefined4 SET {job.bindUpdateChangesExpression()} "
                    ) + "WHERE PandaID=:PandaID AND (jobStatus=:oldJobStatus1 OR jobStatus=:oldJobStatus2)"
                    varMap = job.valuesMap(onlyChanged=True)
                    varMap[":PandaID"] = job.PandaID
                    varMap[":oldJobStatus1"] = "assigned"
                    varMap[":oldJobStatus2"] = "defined"
                    _logger.debug(sqlJ + comment + str(varMap))
                    self.cur.execute(sqlJ + comment, varMap)
                    n = self.cur.rowcount
                    if n == 0:
                        # already killed or activated
                        tmpLog.debug("Job not found to update")
                    else:
                        # update files
                        for file in job.Files:
                            sqlF = f"UPDATE ATLAS_PANDA.filesTable4 SET {file.bindUpdateChangesExpression()}" + "WHERE row_ID=:row_ID"
                            varMap = file.valuesMap(onlyChanged=True)
                            if varMap != {}:
                                varMap[":row_ID"] = file.row_ID
                                _logger.debug(sqlF + comment + str(varMap))
                                self.cur.execute(sqlF + comment, varMap)
                        # job parameters
                        sqlJob = "UPDATE ATLAS_PANDA.jobParamsTable SET jobParameters=:param WHERE PandaID=:PandaID"
                        varMap = {}
                        varMap[":PandaID"] = job.PandaID
                        varMap[":param"] = job.jobParameters
                        self.cur.execute(sqlJob + comment, varMap)
                        updatedFlag = True
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                # record status change
                try:
                    if updatedFlag:
                        self.recordStatusChange(job.PandaID, job.jobStatus, jobInfo=job)
                except Exception:
                    tmpLog.error("recordStatusChange failed")
                self.push_job_status_message(job, job.PandaID, job.jobStatus)
                # push job
                if to_push:
                    mb_proxy_queue = self.get_mb_proxy("panda_pilot_queue")
                    mb_proxy_topic = self.get_mb_proxy("panda_pilot_topic")
                    if mb_proxy_queue and mb_proxy_topic:
                        tmpLog.debug("push job")
                        srv_msg_utils.send_job_message(mb_proxy_queue, mb_proxy_topic, job.jediTaskID, job.PandaID)
                    else:
                        tmpLog.debug("message queue/topic not configured")
                tmpLog.debug("done")
                return True
            except Exception as e:
                # roll back
                self._rollback()
                if iTry + 1 < nTry:
                    tmpLog.debug(f"retry: {iTry}")
                    time.sleep(random.randint(10, 20))
                    continue
                tmpLog.error(f"failed with {str(e)}")
                return False

    # send job to jobsWaiting
    def keepJob(self, job):
        comment = " /* DBProxy.keepJob */"
        _logger.debug(f"keepJob : {job.PandaID}")
        sql1 = "DELETE FROM ATLAS_PANDA.jobsDefined4 "
        sql1 += "WHERE PandaID=:PandaID AND (jobStatus=:oldJobStatus1 OR jobStatus=:oldJobStatus2) AND commandToPilot IS NULL"
        sql2 = f"INSERT INTO ATLAS_PANDA.jobsWaiting4 ({JobSpec.columnNames()}) "
        sql2 += JobSpec.bindValuesExpression()
        # time information
        job.modificationTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
        job.stateChangeTime = job.modificationTime
        updatedFlag = False
        nTry = 3
        for iTry in range(nTry):
            try:
                # begin transaction
                self.conn.begin()
                # delete
                varMap = {}
                varMap[":PandaID"] = job.PandaID
                varMap[":oldJobStatus1"] = "assigned"
                varMap[":oldJobStatus2"] = "defined"
                self.cur.execute(sql1 + comment, varMap)
                n = self.cur.rowcount
                if n == 0:
                    # already killed
                    _logger.debug(f"keepJob : Not found {job.PandaID}")
                else:
                    # set status
                    job.jobStatus = "waiting"
                    # insert
                    self.cur.execute(sql2 + comment, job.valuesMap())
                    # update files
                    for file in job.Files:
                        sqlF = f"UPDATE ATLAS_PANDA.filesTable4 SET {file.bindUpdateChangesExpression()}" + "WHERE row_ID=:row_ID"
                        varMap = file.valuesMap(onlyChanged=True)
                        if varMap != {}:
                            varMap[":row_ID"] = file.row_ID
                            _logger.debug(sqlF + comment + str(varMap))
                            self.cur.execute(sqlF + comment, varMap)
                    # update parameters
                    sqlJob = "UPDATE ATLAS_PANDA.jobParamsTable SET jobParameters=:param WHERE PandaID=:PandaID"
                    varMap = {}
                    varMap[":PandaID"] = job.PandaID
                    varMap[":param"] = job.jobParameters
                    self.cur.execute(sqlJob + comment, varMap)
                    updatedFlag = True
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                # record status change
                try:
                    if updatedFlag:
                        self.recordStatusChange(job.PandaID, job.jobStatus, jobInfo=job)
                        self.push_job_status_message(job, job.PandaID, job.jobStatus)
                except Exception:
                    _logger.error("recordStatusChange in keepJob")
                return True
            except Exception:
                # roll back
                self._rollback()
                if iTry + 1 < nTry:
                    _logger.debug(f"keepJob : {job.PandaID} retry : {iTry}")
                    time.sleep(random.randint(10, 20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error(f"keepJob : {type} {value}")
                return False

    # archive job to jobArchived and remove the job from jobsActive or jobsDefined
    def archiveJob(
        self,
        job,
        fromJobsDefined,
        useCommit=True,
        extraInfo=None,
        fromJobsWaiting=False,
        async_params=None,
    ):
        comment = " /* DBProxy.archiveJob */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName = methodName + f" < PandaID={job.PandaID} jediTaskID={job.jediTaskID} >"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug(f"start status={job.jobStatus} label={job.prodSourceLabel} " f"type={job.processingType} async_params={async_params}")
        start_time = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
        if fromJobsDefined:
            sql0 = "SELECT jobStatus FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID "
            sql1 = "DELETE FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID AND (jobStatus=:oldJobStatus1 OR jobStatus=:oldJobStatus2)"
        elif fromJobsWaiting:
            sql0 = "SELECT jobStatus FROM ATLAS_PANDA.jobsWaiting4 WHERE PandaID=:PandaID "
            sql1 = "DELETE FROM ATLAS_PANDA.jobsWaiting4 WHERE PandaID=:PandaID"
        else:
            sql0 = "SELECT jobStatus FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID FOR UPDATE "
            sql1 = "DELETE FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID"
        sql2 = f"INSERT INTO ATLAS_PANDA.jobsArchived4 ({JobSpec.columnNames()}) "
        sql2 += JobSpec.bindValuesExpression()
        updatedJobList = []
        nTry = 1
        for iTry in range(nTry):
            try:
                # begin transaction
                if useCommit:
                    self.conn.begin()
                # check if JEDI is used
                useJEDI = False
                if (
                    hasattr(panda_config, "useJEDI")
                    and panda_config.useJEDI is True
                    and job.lockedby == "jedi"
                    and self.checkTaskStatusJEDI(job.jediTaskID, self.cur)
                ):
                    useJEDI = True
                if useCommit:
                    if not self._commit():
                        raise RuntimeError("Commit error")
                # delete downstream jobs first
                ddmIDs = []
                newJob = None
                ddmAttempt = 0
                if job.prodSourceLabel == "panda" and job.jobStatus == "failed":
                    # look for outputs
                    upOutputs = []
                    for file in job.Files:
                        if file.type == "output":
                            upOutputs.append(file.lfn)
                    toBeClosedSubList = {}
                    topUserDsList = []
                    # look for downstream jobs
                    sqlD = "SELECT PandaID FROM ATLAS_PANDA.filesTable4 WHERE type=:type AND lfn=:lfn GROUP BY PandaID"
                    sqlDJS = f"SELECT {JobSpec.columnNames()} "
                    sqlDJS += "FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID"
                    sqlDJD = "DELETE FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID"
                    sqlDJI = f"INSERT INTO ATLAS_PANDA.jobsArchived4 ({JobSpec.columnNames()}) "
                    sqlDJI += JobSpec.bindValuesExpression()
                    sqlDFup = "UPDATE ATLAS_PANDA.filesTable4 SET status=:status WHERE PandaID=:PandaID AND type IN (:type1,:type2)"
                    sqlFMod = "UPDATE ATLAS_PANDA.filesTable4 SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
                    sqlMMod = "UPDATE ATLAS_PANDA.metaTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
                    sqlPMod = "UPDATE ATLAS_PANDA.jobParamsTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
                    sqlGetSub = "SELECT DISTINCT destinationDBlock FROM ATLAS_PANDA.filesTable4 WHERE type=:type AND PandaID=:PandaID"
                    sqlCloseSub = 'UPDATE /*+ INDEX_RS_ASC(TAB("DATASETS"."NAME")) */ ATLAS_PANDA.Datasets tab '
                    sqlCloseSub += "SET status=:status,modificationDate=CURRENT_DATE WHERE name=:name"
                    sqlDFile = f"SELECT {FileSpec.columnNames()} FROM ATLAS_PANDA.filesTable4 "
                    sqlDFile += "WHERE PandaID=:PandaID"
                    for upFile in upOutputs:
                        tmpLog.debug(f"look for downstream jobs for {upFile}")
                        if useCommit:
                            self.conn.begin()
                        # select PandaID
                        varMap = {}
                        varMap[":lfn"] = upFile
                        varMap[":type"] = "input"
                        self.cur.arraysize = 100000
                        self.cur.execute(sqlD + comment, varMap)
                        res = self.cur.fetchall()
                        if useCommit:
                            if not self._commit():
                                raise RuntimeError("Commit error")
                        iDownJobs = 0
                        nDownJobs = len(res)
                        nDownChunk = 20
                        inTransaction = False
                        tmpLog.debug(f"found {nDownJobs} downstream jobs for {upFile}")
                        # loop over all downstream IDs
                        for (downID,) in res:
                            if useCommit:
                                if not inTransaction:
                                    self.conn.begin()
                                    inTransaction = True
                            tmpLog.debug(f"delete : {downID} ({iDownJobs}/{nDownJobs})")
                            iDownJobs += 1
                            # select jobs
                            varMap = {}
                            varMap[":PandaID"] = downID
                            self.cur.arraysize = 10
                            self.cur.execute(sqlDJS + comment, varMap)
                            resJob = self.cur.fetchall()
                            if len(resJob) == 0:
                                if useCommit and (iDownJobs % nDownChunk) == 0:
                                    if not self._commit():
                                        raise RuntimeError("Commit error")
                                    inTransaction = False
                                continue
                            # instantiate JobSpec
                            dJob = JobSpec()
                            dJob.pack(resJob[0])
                            # delete
                            varMap = {}
                            varMap[":PandaID"] = downID
                            self.cur.execute(sqlDJD + comment, varMap)
                            retD = self.cur.rowcount
                            if retD == 0:
                                if useCommit and (iDownJobs % nDownChunk) == 0:
                                    if not self._commit():
                                        raise RuntimeError("Commit error")
                                    inTransaction = False
                                continue
                            # error code
                            dJob.jobStatus = "cancelled"
                            dJob.endTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
                            dJob.taskBufferErrorCode = ErrorCode.EC_Kill
                            dJob.taskBufferErrorDiag = "killed by Panda server : upstream job failed"
                            dJob.modificationTime = dJob.endTime
                            dJob.stateChangeTime = dJob.endTime
                            # insert
                            self.cur.execute(sqlDJI + comment, dJob.valuesMap())
                            # update file status
                            varMap = {}
                            varMap[":PandaID"] = downID
                            varMap[":status"] = "failed"
                            varMap[":type1"] = "output"
                            varMap[":type2"] = "log"
                            self.cur.execute(sqlDFup + comment, varMap)
                            # update files,metadata,parametes
                            varMap = {}
                            varMap[":PandaID"] = downID
                            varMap[":modificationTime"] = dJob.modificationTime
                            self.cur.execute(sqlFMod + comment, varMap)
                            self.cur.execute(sqlMMod + comment, varMap)
                            self.cur.execute(sqlPMod + comment, varMap)
                            # collect to record state change
                            updatedJobList.append(dJob)
                            # update JEDI tables
                            if useJEDI:
                                # read files
                                varMap = {}
                                varMap[":PandaID"] = downID
                                self.cur.arraysize = 100000
                                self.cur.execute(sqlDFile + comment, varMap)
                                resDFiles = self.cur.fetchall()
                                for resDFile in resDFiles:
                                    tmpDFile = FileSpec()
                                    tmpDFile.pack(resDFile)
                                    dJob.addFile(tmpDFile)
                                self.propagateResultToJEDI(dJob, self.cur)
                            # set tobeclosed to sub datasets
                            if dJob.jobDefinitionID not in toBeClosedSubList:
                                # init
                                toBeClosedSubList[dJob.jobDefinitionID] = []
                                # get sub datasets
                                varMap = {}
                                varMap[":type"] = "output"
                                varMap[":PandaID"] = downID
                                self.cur.arraysize = 1000
                                self.cur.execute(sqlGetSub + comment, varMap)
                                resGetSub = self.cur.fetchall()
                                if len(resGetSub) == 0:
                                    if useCommit and (iDownJobs % nDownChunk) == 0:
                                        if not self._commit():
                                            raise RuntimeError("Commit error")
                                        inTransaction = False
                                    continue
                                # loop over all sub datasets
                                for (tmpDestinationDBlock,) in resGetSub:
                                    if re.search("_sub\d+$", tmpDestinationDBlock) is None:
                                        continue
                                    if tmpDestinationDBlock not in toBeClosedSubList[dJob.jobDefinitionID]:
                                        # set tobeclosed
                                        varMap = {}
                                        varMap[":status"] = "tobeclosed"
                                        varMap[":name"] = tmpDestinationDBlock
                                        self.cur.execute(sqlCloseSub + comment, varMap)
                                        tmpLog.debug(f"set tobeclosed for {tmpDestinationDBlock}")
                                        # append
                                        toBeClosedSubList[dJob.jobDefinitionID].append(tmpDestinationDBlock)
                                        # close top-level user dataset
                                        topUserDsName = re.sub("_sub\d+$", "", tmpDestinationDBlock)
                                        if not useJEDI and topUserDsName != tmpDestinationDBlock and topUserDsName not in topUserDsList:
                                            # set tobeclosed
                                            varMap = {}
                                            if dJob.processingType.startswith("gangarobot") or dJob.processingType.startswith("hammercloud"):
                                                varMap[":status"] = "completed"
                                            else:
                                                varMap[":status"] = "tobeclosed"
                                            varMap[":name"] = topUserDsName
                                            self.cur.execute(sqlCloseSub + comment, varMap)
                                            tmpLog.debug(f"set {varMap[':status']} for {topUserDsName}")
                                            # append
                                            topUserDsList.append(topUserDsName)
                            if useCommit and (iDownJobs % nDownChunk) == 0:
                                if not self._commit():
                                    raise RuntimeError("Commit error")
                                inTransaction = False
                        if useCommit and inTransaction:
                            if not self._commit():
                                raise RuntimeError("Commit error")

                # main job
                if useCommit:
                    self.conn.begin()
                oldJobSubStatus = None
                # get current status
                currentJobStatus = None
                if fromJobsDefined or fromJobsWaiting:
                    varMap = {}
                    varMap[":PandaID"] = job.PandaID
                    self.cur.execute(sql0 + comment, varMap)
                    res0 = self.cur.fetchone()
                    if res0 is not None:
                        (currentJobStatus,) = res0
                else:
                    # lock job so that events are not dispatched during the processing
                    varMap = {}
                    varMap[":PandaID"] = job.PandaID
                    self.cur.execute(sql0 + comment, varMap)
                    res0 = self.cur.fetchone()
                # check input status for ES merge
                if useJEDI and EventServiceUtils.isEventServiceMerge(job) and job.jobStatus == "finished":
                    retInputStat = self.checkInputFileStatusInJEDI(job, useCommit=False, withLock=True)
                    tmpLog.debug(f"checkInput for ES merge -> {retInputStat}")
                    if retInputStat is None:
                        raise RuntimeError(f"archiveJob : {job.PandaID} failed to check input")
                    if retInputStat is False:
                        tmpLog.debug("set jobStatus=failed due to inconsisten input")
                        job.jobStatus = "failed"
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceInconsistentIn
                        job.taskBufferErrorDiag = "inconsistent file status between Panda and JEDI"
                        for fileSpec in job.Files:
                            if fileSpec.type in ["output", "log"]:
                                fileSpec.status = "failed"
                # actions for jobs without tasks
                if not useJEDI:
                    # update HS06sec for non-JEDI jobs (e.g. HC)
                    hs06sec = self.setHS06sec(job.PandaID, inActive=True)
                    tmpLog.debug(f"calculated hs06sec {hs06sec}")
                    if hs06sec is not None:
                        job.hs06sec = hs06sec

                    # update the g of CO2 emitted by the job
                    try:
                        gco2_regional, gco2_global = self.set_co2_emissions(job.PandaID, in_active=True)
                        tmpLog.debug(f"calculated gCO2 regional {gco2_regional} and global {gco2_global}")
                        if gco2_regional is not None:
                            job.gco2_regional = gco2_regional
                        if gco2_global is not None:
                            job.gco2_global = gco2_global
                    except Exception:
                        tmpLog.error(f"failed calculating gCO2 with {traceback.format_exc()}")

                # actions for successful normal ES jobs
                if useJEDI and EventServiceUtils.isEventServiceJob(job) and not EventServiceUtils.isJobCloningJob(job):
                    # update some job attributes
                    hs06sec = self.setHS06sec(job.PandaID, inActive=True)
                    if hs06sec is not None:
                        job.hs06sec = hs06sec

                    # update the g of CO2 emitted by the job
                    try:
                        gco2_regional, gco2_global = self.set_co2_emissions(job.PandaID, in_active=True)
                        tmpLog.debug(f"calculated gCO2 regional {gco2_regional} and global {gco2_global}")
                        if gco2_regional is not None:
                            job.gco2_regional = gco2_regional
                        if gco2_global is not None:
                            job.gco2_global = gco2_global
                    except Exception:
                        tmpLog.error(f"failed calculating gCO2 with {traceback.format_exc()}")

                    # post processing
                    oldJobSubStatus = job.jobSubStatus
                    if oldJobSubStatus == "NULL":
                        oldJobSubStatus = None
                    retEvS, retNewPandaID = self.ppEventServiceJob(job, currentJobStatus, False)
                    tmpLog.debug(f"ppE -> {retEvS}")
                    # DB error
                    if retEvS is None:
                        raise RuntimeError("Faied to retry for Event Service")
                    elif retEvS == 0:
                        # retry event ranges
                        job.jobStatus = "merging"
                        job.jobSubStatus = "es_retry"
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceRetried
                        job.taskBufferErrorDiag = f"closed to retry unprocessed even ranges in PandaID={retNewPandaID}"
                    elif retEvS in [2, 10]:
                        # goes to merging
                        if retEvS == 2:
                            job.jobStatus = "merging"
                        else:
                            job.jobStatus = "closed"
                        job.jobSubStatus = "es_merge"
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceMerge
                        job.taskBufferErrorDiag = f"closed to merge pre-merged files in PandaID={retNewPandaID}"
                        # kill unused event service consumers
                        self.killUnusedEventServiceConsumers(job, False, killAll=True)
                    elif retEvS == 3:
                        # maximum attempts reached
                        job.jobStatus = "failed"
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceMaxAttempt
                        job.taskBufferErrorDiag = "maximum event attempts reached"
                        # kill other consumers
                        self.killEventServiceConsumers(job, False, False)
                        self.killUnusedEventServiceConsumers(job, False, killAll=True, checkAttemptNr=True)
                    elif retEvS == 4:
                        # other consumers are running
                        job.jobStatus = "merging"
                        job.jobSubStatus = "es_wait"
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceWaitOthers
                        job.taskBufferErrorDiag = "no further action since other Event Service consumers were still running"
                    elif retEvS == 5:
                        # didn't process any event ranges
                        job.jobStatus = "closed"
                        job.jobSubStatus = "es_inaction"
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceUnprocessed
                        job.taskBufferErrorDiag = "didn't process any events on WN or reached last job attempt and take no further action"
                    elif retEvS == 6:
                        # didn't process any event ranges and last consumer
                        job.jobStatus = "failed"
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceLastUnprocessed
                        job.taskBufferErrorDiag = "didn't process any events on WN and give up since this is the last consumer"
                    elif retEvS == 7:
                        # all event ranges failed
                        job.jobStatus = "failed"
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceAllFailed
                        job.taskBufferErrorDiag = "all event ranges failed"
                    elif retEvS == 8:
                        # retry event ranges but no events were processed
                        job.jobStatus = "closed"
                        job.jobSubStatus = "es_noevent"
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceNoEvent
                        job.taskBufferErrorDiag = f"didn't process any events on WN and retry unprocessed even ranges in PandaID={retNewPandaID}"
                    elif retEvS == 9:
                        # closed in bad job status
                        job.jobStatus = "closed"
                        job.jobSubStatus = "es_badstatus"
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceBadStatus
                        job.taskBufferErrorDiag = "cloned in bad jobStatus like defined and pending"
                    # additional actions when retry
                    codeListWithRetry = [0, 4, 5, 8, 9]
                    if retEvS in codeListWithRetry and job.computingSite != EventServiceUtils.siteIdForWaitingCoJumboJobs:
                        # check jumbo flag
                        sqlJumbo = f"SELECT useJumbo FROM {panda_config.schemaJEDI}.JEDI_Tasks "
                        sqlJumbo += "WHERE jediTaskID=:jediTaskID "
                        varMap = {}
                        varMap[":jediTaskID"] = job.jediTaskID
                        self.cur.execute(sqlJumbo + comment, varMap)
                        resJumbo = self.cur.fetchone()
                        if resJumbo is not None:
                            (useJumbo,) = resJumbo
                        else:
                            useJumbo = None
                        tmpLog.debug(f"useJumbo={useJumbo}")
                        # no new jobs
                        if retNewPandaID is None and (retEvS != 4 or EventServiceUtils.isCoJumboJob(job) or useJumbo is not None):
                            nActiveConsumers = self.getActiveConsumers(job.jediTaskID, job.jobsetID, job.PandaID)
                            # create a fake cojumbo
                            if (
                                nActiveConsumers == 0
                                and retEvS in [4, 5]
                                and (EventServiceUtils.isCoJumboJob(job) or useJumbo is not None)
                                and job.computingSite != EventServiceUtils.siteIdForWaitingCoJumboJobs
                            ):
                                nActiveConsumers = self.makeFakeCoJumbo(job)
                            # no ES queues for retry
                            if nActiveConsumers == 0:
                                job.jobStatus = "failed"
                                job.taskBufferErrorCode = ErrorCode.EC_EventServiceNoEsQueues
                                job.taskBufferErrorDiag = "no ES queues available for new consumers"
                                tmpLog.debug(f"set {job.jobStatus} since {job.taskBufferErrorDiag}")
                    # kill unused event ranges
                    if job.jobStatus == "failed":
                        if not job.notDiscardEvents():
                            self.killUnusedEventRanges(job.jediTaskID, job.jobsetID)
                        self.updateRelatedEventServiceJobs(job, True)
                elif useJEDI and EventServiceUtils.isEventServiceJob(job) and EventServiceUtils.isJobCloningJob(job):
                    # check for cloned jobs
                    retJC = self.checkClonedJob(job, False)
                    # DB error
                    if retJC is None:
                        raise RuntimeError("Faied to take post-action for cloned job")
                    elif retJC["lock"] is True:
                        # kill other clones if the job done after locking semaphore
                        self.killEventServiceConsumers(job, False, False)
                        self.killUnusedEventServiceConsumers(job, False, killAll=True)
                    else:
                        # failed to lock semaphore
                        if retJC["last"] is False:
                            # set closed if it is not the last clone
                            job.jobStatus = "closed"
                            job.jobSubStatus = "jc_unlock"
                            job.taskBufferErrorCode = ErrorCode.EC_JobCloningUnlock
                            if retJC["win"] is not None:
                                job.taskBufferErrorDiag = f"closed since another clone PandaID={retJC['win']} got semaphore"
                            else:
                                job.taskBufferErrorDiag = "closed since failed to lock semaphore"
                elif useJEDI and EventServiceUtils.is_fine_grained_job(job):
                    # fine-grained
                    n_done, n_remain = self.check_fine_grained_processing(job)
                    if n_done > 0 or n_remain == 0:
                        job.jobStatus = "finished"
                        if n_remain == 0:
                            job.jobSubStatus = "fg_done"
                        else:
                            job.jobSubStatus = "fg_partial"
                    else:
                        job.jobSubStatus = "fg_stumble"
                # release unprocessed samples for HPO
                if job.is_hpo_workflow():
                    self.release_unprocessed_events(job.jediTaskID, job.PandaID)
                # delete from jobsDefined/Active
                varMap = {}
                varMap[":PandaID"] = job.PandaID
                if fromJobsDefined:
                    varMap[":oldJobStatus1"] = "assigned"
                    varMap[":oldJobStatus2"] = "defined"
                self.cur.execute(sql1 + comment, varMap)
                n = self.cur.rowcount
                if n == 0:
                    # already deleted
                    raise RuntimeError(f"PandaID={job.PandaID} already deleted")
                else:
                    # insert
                    job.modificationTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
                    job.stateChangeTime = job.modificationTime
                    if job.endTime == "NULL":
                        job.endTime = job.modificationTime
                    self.cur.execute(sql2 + comment, job.valuesMap())
                    # update files
                    for file in job.Files:
                        sqlF = f"UPDATE ATLAS_PANDA.filesTable4 SET {file.bindUpdateChangesExpression()}" + "WHERE row_ID=:row_ID"
                        varMap = file.valuesMap(onlyChanged=True)
                        if varMap != {}:
                            varMap[":row_ID"] = file.row_ID
                            tmpLog.debug(sqlF + comment + str(varMap))
                            self.cur.execute(sqlF + comment, varMap)
                    # update metadata and parameters
                    sqlFMod = "UPDATE ATLAS_PANDA.filesTable4 SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
                    sqlMMod = "UPDATE ATLAS_PANDA.metaTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
                    sqlPMod = "UPDATE ATLAS_PANDA.jobParamsTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
                    varMap = {}
                    varMap[":PandaID"] = job.PandaID
                    varMap[":modificationTime"] = job.modificationTime
                    self.cur.execute(sqlFMod + comment, varMap)
                    self.cur.execute(sqlMMod + comment, varMap)
                    self.cur.execute(sqlPMod + comment, varMap)
                    # increment the number of failed jobs in _dis
                    myDisList = []
                    if job.jobStatus == "failed" and job.prodSourceLabel in [
                        "managed",
                        "test",
                    ]:
                        for tmpFile in job.Files:
                            if tmpFile.type == "input" and tmpFile.dispatchDBlock not in ["", "NULL", None] and tmpFile.dispatchDBlock not in myDisList:
                                varMap = {}
                                varMap[":name"] = tmpFile.dispatchDBlock
                                # check currentfiles
                                sqlGetCurFiles = """SELECT /*+ BEGIN_OUTLINE_DATA """
                                sqlGetCurFiles += """INDEX_RS_ASC(@"SEL$1" "TAB"@"SEL$1" ("DATASETS"."NAME")) """
                                sqlGetCurFiles += """OUTLINE_LEAF(@"SEL$1") ALL_ROWS """
                                sqlGetCurFiles += """IGNORE_OPTIM_EMBEDDED_HINTS """
                                sqlGetCurFiles += """END_OUTLINE_DATA */ """
                                sqlGetCurFiles += "currentfiles,vuid FROM ATLAS_PANDA.Datasets tab WHERE name=:name"
                                self.cur.execute(sqlGetCurFiles + comment, varMap)
                                resCurFiles = self.cur.fetchone()
                                tmpLog.debug(f"{str(resCurFiles)}")
                                if resCurFiles is not None:
                                    # increment currentfiles only for the first failed job since that is enough
                                    tmpCurrentFiles, tmpVUID = resCurFiles
                                    tmpLog.debug(f"{tmpFile.dispatchDBlock} currentfiles={tmpCurrentFiles}")
                                    if tmpCurrentFiles == 0:
                                        tmpLog.debug(f"{tmpFile.dispatchDBlock} update currentfiles")
                                        varMap = {}
                                        varMap[":vuid"] = tmpVUID
                                        sqlFailedInDis = "UPDATE ATLAS_PANDA.Datasets "
                                        sqlFailedInDis += "SET currentfiles=currentfiles+1 WHERE vuid=:vuid"
                                        self.cur.execute(sqlFailedInDis + comment, varMap)
                                myDisList.append(tmpFile.dispatchDBlock)
                    # collect to record state change
                    updatedJobList.append(job)
                    # update JEDI tables unless it is an ES consumer job which was successful but waits for merging or other running consumers
                    if useJEDI and not (EventServiceUtils.isEventServiceJob(job) and (job.isCancelled() or job.jobStatus == "merging")):
                        self.propagateResultToJEDI(
                            job,
                            self.cur,
                            extraInfo=extraInfo,
                            async_params=async_params,
                        )
                    # update related ES jobs when ES-merge job is done
                    if (
                        useJEDI
                        and EventServiceUtils.isEventServiceMerge(job)
                        and job.taskBufferErrorCode not in [ErrorCode.EC_PilotRetried]
                        and not job.isCancelled()
                    ):
                        if job.jobStatus == "failed":
                            self.updateRelatedEventServiceJobs(job, True)
                        else:
                            self.updateRelatedEventServiceJobs(job)
                # propagate successful result to unmerge job
                if useJEDI and job.processingType == "pmerge" and job.jobStatus == "finished":
                    self.updateUnmergedJobs(job, async_params=async_params)
                # overwrite job status
                tmpJobStatus = job.jobStatus
                sqlPRE = "SELECT /* use_json_type */ scj.data.pledgedcpu FROM ATLAS_PANDA.schedconfig_json scj WHERE scj.panda_queue=:siteID "

                sqlOJS = "UPDATE ATLAS_PANDA.jobsArchived4 SET jobStatus=:jobStatus,jobSubStatus=:jobSubStatus WHERE PandaID=:PandaID "
                if (
                    oldJobSubStatus in ["pilot_failed", "es_heartbeat"]
                    or oldJobSubStatus == "pilot_killed"
                    and job.jobSubStatus in ["es_noevent", "es_inaction"]
                ):
                    # check if preemptable
                    isPreemptable = False
                    varMap = {}
                    varMap[":siteID"] = job.computingSite
                    self.cur.execute(sqlPRE + comment, varMap)
                    resPRE = self.cur.fetchone()
                    if resPRE is not None:
                        try:
                            if int(resPRE[0]) == -1:
                                isPreemptable = True
                        except Exception:
                            pass
                    # overwrite job status
                    varMap = {}
                    varMap[":PandaID"] = job.PandaID
                    if isPreemptable and oldJobSubStatus not in ["pilot_failed"]:
                        varMap[":jobStatus"] = "closed"
                        varMap[":jobSubStatus"] = "es_preempted"
                    else:
                        varMap[":jobStatus"] = "failed"
                        varMap[":jobSubStatus"] = oldJobSubStatus
                    self.cur.execute(sqlOJS + comment, varMap)
                    tmpJobStatus = varMap[":jobStatus"]
                if EventServiceUtils.isEventServiceJob(job):
                    if (
                        job.jobStatus in ["failed", "closed"]
                        and job.taskBufferErrorCode
                        in [
                            ErrorCode.EC_EventServiceLastUnprocessed,
                            ErrorCode.EC_EventServiceUnprocessed,
                        ]
                        and job.nEvents > 0
                    ):
                        varMap = {}
                        varMap[":PandaID"] = job.PandaID
                        varMap[":jobStatus"] = "merging"
                        if oldJobSubStatus in ["es_toolong"]:
                            varMap[":jobSubStatus"] = oldJobSubStatus
                        else:
                            varMap[":jobSubStatus"] = "es_wait"
                        self.cur.execute(sqlOJS + comment, varMap)
                        tmpJobStatus = varMap[":jobStatus"]
                        tmpLog.debug("change failed to merging")
                    elif (
                        job.jobStatus in ["failed"]
                        and job.taskBufferErrorCode
                        in [
                            ErrorCode.EC_EventServiceLastUnprocessed,
                            ErrorCode.EC_EventServiceUnprocessed,
                        ]
                        and (
                            oldJobSubStatus in ["pilot_noevents"]
                            or (job.pilotErrorCode == 0 and job.ddmErrorCode == 0 and job.supErrorCode == 0 and job.jobDispatcherErrorCode == 0)
                        )
                    ):
                        varMap = {}
                        varMap[":PandaID"] = job.PandaID
                        varMap[":jobStatus"] = "closed"
                        varMap[":jobSubStatus"] = oldJobSubStatus
                        self.cur.execute(sqlOJS + comment, varMap)
                        tmpJobStatus = varMap[":jobStatus"]
                        tmpLog.debug(f"change failed to closed for {oldJobSubStatus}")
                # commit
                if useCommit:
                    if not self._commit():
                        raise RuntimeError("Commit error")
                # record status change
                try:
                    for tmpJob in updatedJobList:
                        self.recordStatusChange(
                            tmpJob.PandaID,
                            tmpJobStatus,
                            jobInfo=tmpJob,
                            useCommit=useCommit,
                        )
                        extra_info_dict = {
                            "job_nevents": tmpJob.nEvents,
                            "job_ninputfiles": tmpJob.nInputFiles,
                            "job_noutputdatafiles": tmpJob.nOutputDataFiles,
                            "job_ninputdatafiles": tmpJob.nInputDataFiles,
                            "job_inputfilebytes": tmpJob.inputFileBytes,
                            "job_outputfilebytes": tmpJob.outputFileBytes,
                            "job_hs06sec": tmpJob.hs06sec,
                        }
                        self.push_job_status_message(
                            tmpJob,
                            tmpJob.PandaID,
                            tmpJobStatus,
                            extra_data=extra_info_dict,
                        )
                except Exception:
                    tmpLog.error("recordStatusChange in archiveJob")
                exec_time = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - start_time
                tmpLog.debug("done OK. took %s.%03d sec" % (exec_time.seconds, exec_time.microseconds / 1000))
                return True, ddmIDs, ddmAttempt, newJob
            except Exception:
                # roll back
                if useCommit:
                    self._rollback(True)
                # error
                self.dumpErrorMessage(_logger, methodName)
                exec_time = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - start_time
                tmpLog.debug("done NG. took %s.%03d sec" % (exec_time.seconds, exec_time.microseconds / 1000))
                if not useCommit:
                    raise RuntimeError("archiveJob failed")
                return False, [], 0, None

    # finalize pending jobs
    def finalizePendingJobs(self, prodUserName, jobDefinitionID, waitLock=False):
        comment = " /* DBProxy.finalizePendingJobs */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(
            _logger,
            methodName + f" < user={prodUserName} jobdefID={jobDefinitionID} >",
        )
        tmpLog.debug("start")
        sql0 = "SELECT PandaID,lockedBy,jediTaskID FROM ATLAS_PANDA.jobsActive4 "
        sql0 += "WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID "
        sql0 += "AND prodSourceLabel=:prodSourceLabel AND jobStatus=:jobStatus "
        sqlU = "UPDATE ATLAS_PANDA.jobsActive4 SET jobStatus=:newJobStatus "
        sqlU += "WHERE PandaID=:PandaID AND jobStatus=:jobStatus "
        sql1 = f"SELECT {JobSpec.columnNames()} FROM ATLAS_PANDA.jobsActive4 "
        sql1 += "WHERE PandaID=:PandaID AND jobStatus=:jobStatus "
        sql2 = "DELETE FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID AND jobStatus=:jobStatus "
        sql3 = f"INSERT INTO ATLAS_PANDA.jobsArchived4 ({JobSpec.columnNames()}) "
        sql3 += JobSpec.bindValuesExpression()
        sqlFMod = "UPDATE ATLAS_PANDA.filesTable4 SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
        sqlMMod = "UPDATE ATLAS_PANDA.metaTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
        sqlPMod = "UPDATE ATLAS_PANDA.jobParamsTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
        try:
            # begin transaction
            self.conn.begin()
            self.cur.arraysize = 100000
            # select
            varMap = {}
            varMap[":jobStatus"] = "failed"
            varMap[":prodUserName"] = prodUserName
            varMap[":jobDefinitionID"] = jobDefinitionID
            varMap[":prodSourceLabel"] = "user"
            self.cur.execute(sql0 + comment, varMap)
            resPending = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # lock
            pPandaIDs = []
            lockedBy = None
            jediTaskID = None
            for pandaID, tmpLockedBy, tmpJediTaskID in resPending:
                if lockedBy is None:
                    lockedBy = tmpLockedBy
                if jediTaskID is None:
                    jediTaskID = tmpJediTaskID
                pPandaIDs.append(pandaID)
            # check if JEDI is used
            useJEDI = False
            if hasattr(panda_config, "useJEDI") and panda_config.useJEDI is True and lockedBy == "jedi" and self.checkTaskStatusJEDI(jediTaskID, self.cur):
                useJEDI = True
            # loop over all PandaIDs
            for pandaID in pPandaIDs:
                # begin transaction
                self.conn.begin()
                # lock
                varMap = {}
                varMap[":jobStatus"] = "failed"
                varMap[":newJobStatus"] = "holding"
                varMap[":PandaID"] = pandaID
                self.cur.execute(sqlU + comment, varMap)
                retU = self.cur.rowcount
                if retU == 0:
                    # commit
                    if not self._commit():
                        raise RuntimeError("Commit error")
                # get job
                varMap = {}
                varMap[":PandaID"] = pandaID
                varMap[":jobStatus"] = "holding"
                self.cur.arraysize = 10
                self.cur.execute(sql1 + comment, varMap)
                res = self.cur.fetchall()
                if len(res) == 0:
                    tmpLog.debug(f"PandaID {pandaID} not found")
                    # commit
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    continue
                job = JobSpec()
                job.pack(res[0])
                job.jobStatus = "failed"
                job.modificationTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
                # delete
                self.cur.execute(sql2 + comment, varMap)
                n = self.cur.rowcount
                if n == 0:
                    # already killed
                    tmpLog.debug(f"Not found {pandaID}")
                else:
                    tmpLog.debug(f"finalizing {pandaID}")
                    # insert
                    self.cur.execute(sql3 + comment, job.valuesMap())
                    # update files,metadata,parametes
                    varMap = {}
                    varMap[":PandaID"] = pandaID
                    varMap[":modificationTime"] = job.modificationTime
                    self.cur.execute(sqlFMod + comment, varMap)
                    self.cur.execute(sqlMMod + comment, varMap)
                    self.cur.execute(sqlPMod + comment, varMap)
                    # update JEDI tables
                    if useJEDI:
                        # read files
                        sqlFile = f"SELECT {FileSpec.columnNames()} FROM ATLAS_PANDA.filesTable4 "
                        sqlFile += "WHERE PandaID=:PandaID"
                        varMap = {}
                        varMap[":PandaID"] = pandaID
                        self.cur.arraysize = 100000
                        self.cur.execute(sqlFile + comment, varMap)
                        resFs = self.cur.fetchall()
                        for resF in resFs:
                            tmpFile = FileSpec()
                            tmpFile.pack(resF)
                            job.addFile(tmpFile)
                        self.propagateResultToJEDI(job, self.cur, finishPending=True, waitLock=waitLock)
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                self.push_job_status_message(job, pandaID, varMap[":newJobStatus"])
            tmpLog.debug(f"done {len(pPandaIDs)} jobs")
            return True
        except Exception as e:
            # roll back
            self._rollback()
            tmpLog.error(f"failed with {str(e)}")
            return False

    # update Job status in jobsActive
    def updateJobStatus(self, pandaID, jobStatus, param, updateStateChange=False, attemptNr=None):
        comment = " /* DBProxy.updateJobStatus */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        method_name += f" <PandaID={pandaID}> "
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug(f"attemptNr={attemptNr} status={jobStatus}")
        sql0 = "SELECT commandToPilot,endTime,specialHandling,jobStatus,computingSite,cloud,prodSourceLabel,lockedby,jediTaskID,"
        sql0 += "jobsetID,jobDispatcherErrorDiag,supErrorCode,eventService,batchID "
        sql0 += "FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
        varMap0 = {}
        varMap0[":PandaID"] = pandaID
        sql1 = "UPDATE ATLAS_PANDA.jobsActive4 SET jobStatus=:jobStatus"
        varMap = {}
        presetEndTime = False
        for key in list(param):
            if key in ["corruptedFiles"]:
                continue
            if param[key] is not None or key in ["jobDispatcherErrorDiag"]:
                param[key] = JobSpec.truncateStringAttr(key, param[key])
                sql1 += f",{key}=:{key}"
                varMap[f":{key}"] = param[key]
                if key == "endTime":
                    presetEndTime = True
                try:
                    # store positive error code even for pilot retry
                    if key == "pilotErrorCode" and param[key].startswith("-"):
                        varMap[f":{key}"] = param[key][1:]
                except Exception:
                    pass
            if key == "jobMetrics":
                # extract the memory leak from the pilot jobMetrics
                try:
                    tmpM = re.search("leak=(-?\d+\.*\d+)", param[key])
                    if tmpM is not None:
                        memoryLeak = int(float(tmpM.group(1)))
                        tmpKey = "memory_leak"
                        sql1 += ",{0}=:{0}".format(tmpKey)
                        varMap[f":{tmpKey}"] = memoryLeak
                except Exception:
                    pass

                # extract the chi2 measurement for the memory leak fitting
                try:
                    tmpM = re.search("chi2=(-?\d+\.*\d+)", param[key])
                    if tmpM is not None:
                        # keep measurement under 11 digits because of DB declaration
                        memory_leak_x2 = min(float(tmpM.group(1)), 10**11 - 1)
                        tmpKey = "memory_leak_x2"
                        sql1 += ",{0}=:{0}".format(tmpKey)
                        varMap[f":{tmpKey}"] = memory_leak_x2
                except Exception:
                    pass
        sql1W = " WHERE PandaID=:PandaID "
        varMap[":PandaID"] = pandaID
        if attemptNr is not None:
            sql0 += "AND attemptNr=:attemptNr "
            sql1W += "AND attemptNr=:attemptNr "
            varMap[":attemptNr"] = attemptNr
            varMap0[":attemptNr"] = attemptNr
        # prevent change from holding to transferring which doesn't register files to sub/tid
        if jobStatus == "transferring":
            sql1W += "AND NOT jobStatus=:ngStatus "
            varMap[":ngStatus"] = "holding"
        updatedFlag = False
        action_in_downstream = None
        nTry = 1
        for iTry in range(nTry):
            try:
                # begin transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10
                self.cur.execute(sql0 + comment, varMap0)
                res = self.cur.fetchone()
                if res is not None:
                    ret = ""
                    (
                        commandToPilot,
                        endTime,
                        specialHandling,
                        oldJobStatus,
                        computingSite,
                        cloud,
                        prodSourceLabel,
                        lockedby,
                        jediTaskID,
                        jobsetID,
                        jobDispatcherErrorDiag,
                        supErrorCode,
                        eventService,
                        batchID,
                    ) = res
                    # check debug mode and job cloning with runonce
                    is_job_cloning = False
                    if specialHandling:
                        tmpJobSpec = JobSpec()
                        tmpJobSpec.specialHandling = specialHandling
                        if tmpJobSpec.is_debug_mode():
                            ret += "debug,"
                        if EventServiceUtils.getJobCloningType(tmpJobSpec) == "runonce":
                            is_job_cloning = True
                    # FIXME
                    # else:
                    #    ret += 'debugoff,'
                    # kill command
                    if commandToPilot not in [None, ""]:
                        # soft kill
                        if supErrorCode in [ErrorCode.EC_EventServicePreemption]:
                            # commandToPilot = 'softkill'
                            pass
                        ret += f"{commandToPilot},"
                    ret = ret[:-1]
                    # convert empty to NULL
                    if ret == "":
                        ret = "NULL"
                    if oldJobStatus == "failed" and jobStatus in [
                        "holding",
                        "transferring",
                        "starting",
                        "running",
                    ]:
                        tmp_log.debug(f"skip to set {jobStatus} since it is already {oldJobStatus}")
                        ret = "alreadydone"
                    elif oldJobStatus == "transferring" and jobStatus == "holding" and jobDispatcherErrorDiag in [None, ""]:
                        # skip transferring -> holding
                        tmp_log.debug("skip to set holding since it is alredy in transferring")
                        ret = "alreadydone"
                    elif (
                        oldJobStatus == "holding"
                        and jobStatus == "holding"
                        and ("jobDispatcherErrorDiag" not in param or param["jobDispatcherErrorDiag"] not in [None, ""])
                    ):
                        # just ignore hearbeats for job recovery
                        tmp_log.debug("skip to reset holding")
                    elif (
                        oldJobStatus == "holding"
                        and jobStatus == "holding"
                        and jobDispatcherErrorDiag in [None, ""]
                        and "jobDispatcherErrorDiag" in param
                        and param["jobDispatcherErrorDiag"] in [None, ""]
                    ):
                        # special return to avoid duplicated XMLs
                        tmp_log.debug("skip to set holding since it was already set to holding by the final heartbeat")
                        ret = "alreadydone"
                    elif oldJobStatus == "merging":
                        # don't update merging
                        tmp_log.debug("skip to change from merging")
                    elif oldJobStatus in ["holding", "transferring"] and jobStatus == "starting":
                        # don't update holding
                        tmp_log.debug(f"skip to change {oldJobStatus} to {jobStatus} to avoid inconsistency")
                    elif oldJobStatus == "holding" and jobStatus == "running":
                        # don't update holding
                        tmp_log.debug(f"skip to change {oldJobStatus} to {jobStatus} not to return to active")
                    elif (
                        batchID not in ["", None]
                        and "batchID" in param
                        and param["batchID"] not in ["", None]
                        and batchID != param["batchID"]
                        and re.search("^\d+\.*\d+$", batchID) is None
                        and re.search("^\d+\.*\d+$", param["batchID"]) is None
                    ):
                        # invalid batchID
                        tmp_log.debug(
                            "to be killed since batchID mismatch old {} in {} vs new {} in {}".format(
                                batchID.replace("\n", ""),
                                oldJobStatus,
                                param["batchID"].replace("\n", ""),
                                jobStatus,
                            )
                        )
                        ret = "tobekilled"
                        # set supErrorCode and supErrorDiag
                        varMap = {}
                        varMap[":PandaID"] = pandaID
                        varMap[":code"] = SupErrors.error_codes["INVALID_BATCH_ID"]
                        varMap[":diag"] = "got an update request with invalid batchID={0}".format(param["batchID"].replace("\n", ""))
                        varMap[":diag"] = JobSpec.truncateStringAttr("supErrorDiag", varMap[":diag"])
                        sqlSUP = "UPDATE ATLAS_PANDA.jobsActive4 SET supErrorCode=:code,supErrorDiag=:diag "
                        sqlSUP += "WHERE PandaID=:PandaID "
                        self.cur.execute(sqlSUP + comment, varMap)
                    else:
                        # change starting to running
                        if oldJobStatus == "running" and jobStatus == "starting":
                            tmp_log.debug(f"changed to {oldJobStatus} from {jobStatus} to avoid inconsistent update")
                            jobStatus = oldJobStatus
                        # update stateChangeTime
                        if updateStateChange or (jobStatus != oldJobStatus):
                            sql1 += ",stateChangeTime=CURRENT_DATE"
                        # set endTime if undefined for holding
                        if (jobStatus == "holding" or (jobStatus == "transferring" and oldJobStatus == "running")) and endTime is None and not presetEndTime:
                            sql1 += ",endTime=CURRENT_DATE "
                        # update startTime
                        if oldJobStatus in ["sent", "starting"] and jobStatus == "running" and ":startTime" not in varMap:
                            sql1 += ",startTime=CURRENT_DATE"
                        # update modification time
                        sql1 += ",modificationTime=CURRENT_DATE"
                        # update
                        varMap[":jobStatus"] = jobStatus
                        self.cur.execute(sql1 + sql1W + comment, varMap)
                        nUp = self.cur.rowcount
                        tmp_log.debug(f"attemptNr={attemptNr} nUp={nUp} old={oldJobStatus} new={jobStatus}")
                        if nUp == 1:
                            updatedFlag = True
                        if nUp == 0 and jobStatus == "transferring":
                            tmp_log.debug("ignore to update for transferring")
                        # update waiting ES jobs not to get reassigned
                        if updatedFlag and EventServiceUtils.isEventServiceSH(specialHandling):
                            # sql to update ES jobs
                            sqlUEA = "SELECT PandaID FROM ATLAS_PANDA.jobsActive4 "
                            sqlUEA += "WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID AND jobStatus=:jobStatus "
                            sqlUEL = "SELECT modificationTime FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
                            sqlUEL += "FOR UPDATE NOWAIT "
                            sqlUE = "UPDATE ATLAS_PANDA.jobsActive4 SET modificationTime=CURRENT_DATE "
                            sqlUE += "WHERE PandaID=:PandaID "
                            varMap = {}
                            varMap[":jediTaskID"] = jediTaskID
                            varMap[":jobsetID"] = jobsetID
                            varMap[":jobStatus"] = "activated"
                            self.cur.execute(sqlUEA + comment, varMap)
                            resUEA = self.cur.fetchall()
                            nUE = 0
                            for (ueaPandaID,) in resUEA:
                                varMap = {}
                                varMap[":PandaID"] = ueaPandaID
                                try:
                                    # lock with NOWAIT
                                    self.cur.execute(sqlUEL + comment, varMap)
                                    resUEL = self.cur.fetchone()
                                    if resUEL is None:
                                        continue
                                except Exception:
                                    tmp_log.debug(f"skip to update associated ES={ueaPandaID}")
                                    continue
                                self.cur.execute(sqlUE + comment, varMap)
                                nUE += self.cur.rowcount
                            tmp_log.debug(f"updated {nUE} ES jobs")
                        # update fake co-jumbo jobs
                        if updatedFlag and eventService == EventServiceUtils.jumboJobFlagNumber:
                            # sql to update fake co-jumbo
                            sqlIFL = "SELECT PandaID FROM ATLAS_PANDA.jobsWaiting4 "
                            sqlIFL += "WHERE jediTaskID=:jediTaskID AND eventService=:eventService AND jobStatus=:jobStatus "
                            sqlIFL += "FOR UPDATE NOWAIT "
                            sqlIF = "UPDATE ATLAS_PANDA.jobsWaiting4 SET modificationTime=CURRENT_DATE "
                            sqlIF += "WHERE jediTaskID=:jediTaskID AND eventService=:eventService AND jobStatus=:jobStatus "
                            varMap = {}
                            varMap[":jediTaskID"] = jediTaskID
                            varMap[":eventService"] = EventServiceUtils.coJumboJobFlagNumber
                            varMap[":jobStatus"] = "waiting"
                            try:
                                # lock with NOWAIT
                                self.cur.execute(sqlIFL + comment, varMap)
                                resIFL = self.cur.fetchall()
                                self.cur.execute(sqlIF + comment, varMap)
                                nUE = self.cur.rowcount
                                tmp_log.debug(f"updated {nUE} fake co-jumbo jobs")
                            except Exception:
                                tmp_log.debug("skip to update fake co-jumbo jobs")
                        # update nFilesOnHold for JEDI RW calculation
                        if (
                            updatedFlag
                            and jobStatus == "transferring"
                            and oldJobStatus == "holding"
                            and hasattr(panda_config, "useJEDI")
                            and panda_config.useJEDI is True
                            and lockedby == "jedi"
                            and self.checkTaskStatusJEDI(jediTaskID, self.cur)
                        ):
                            # SQL to get file list from Panda
                            sqlJediFP = "SELECT datasetID,fileID,attemptNr FROM ATLAS_PANDA.filesTable4 "
                            sqlJediFP += "WHERE PandaID=:pandaID AND type IN (:type1,:type2) ORDER BY datasetID,fileID "
                            # SQL to check JEDI files
                            sqlJediFJ = "SELECT /*+ INDEX_RS_ASC(JEDI_DATASET_CONTENTS (JEDI_DATASET_CONTENTS.JEDITASKID JEDI_DATASET_CONTENTS.DATASETID JEDI_DATASET_CONTENTS.FILEID)) */ 1 FROM ATLAS_PANDA.JEDI_Dataset_Contents "
                            sqlJediFJ += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                            sqlJediFJ += "AND attemptNr=:attemptNr AND status=:status AND keepTrack=:keepTrack "
                            # get file list
                            varMap = {}
                            varMap[":pandaID"] = pandaID
                            varMap[":type1"] = "input"
                            varMap[":type2"] = "pseudo_input"
                            self.cur.arraysize = 100000
                            self.cur.execute(sqlJediFP + comment, varMap)
                            resJediFile = self.cur.fetchall()
                            datasetContentsStat = {}
                            # loop over all files
                            for tmpDatasetID, tmpFileID, tmpAttemptNr in resJediFile:
                                # check file in JEDI
                                varMap = {}
                                varMap[":jediTaskID"] = jediTaskID
                                varMap[":datasetID"] = tmpDatasetID
                                varMap[":fileID"] = tmpFileID
                                varMap[":attemptNr"] = tmpAttemptNr
                                varMap[":status"] = "running"
                                varMap[":keepTrack"] = 1
                                self.cur.execute(sqlJediFJ + comment, varMap)
                                res = self.cur.fetchone()
                                if res is not None:
                                    if tmpDatasetID not in datasetContentsStat:
                                        datasetContentsStat[tmpDatasetID] = 0
                                    if jobStatus == "transferring":
                                        # increment nOnHold
                                        datasetContentsStat[tmpDatasetID] += 1
                                    else:
                                        # decrement nOnHold
                                        datasetContentsStat[tmpDatasetID] -= 1
                            # loop over all datasets
                            tmpDatasetIDs = sorted(datasetContentsStat)
                            for tmpDatasetID in tmpDatasetIDs:
                                diffNum = datasetContentsStat[tmpDatasetID]
                                # no difference
                                if diffNum == 0:
                                    continue
                                # SQL to lock
                                sqlJediDL = "SELECT nFilesOnHold FROM ATLAS_PANDA.JEDI_Datasets "
                                sqlJediDL += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                                sqlJediDL += "FOR UPDATE NOWAIT "
                                varMap = {}
                                varMap[":jediTaskID"] = jediTaskID
                                varMap[":datasetID"] = tmpDatasetID
                                tmp_log.debug(sqlJediDL + comment + str(varMap))
                                self.cur.execute(sqlJediDL + comment, varMap)
                                # SQL to update
                                sqlJediDU = "UPDATE ATLAS_PANDA.JEDI_Datasets SET "
                                if diffNum > 0:
                                    sqlJediDU += "nFilesOnHold=nFilesOnHold+:diffNum "
                                else:
                                    sqlJediDU += "nFilesOnHold=nFilesOnHold-:diffNum "
                                sqlJediDU += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                                sqlJediDU += "AND NOT type IN (:ngType1,:ngType2) "
                                varMap = {}
                                varMap[":jediTaskID"] = jediTaskID
                                varMap[":datasetID"] = tmpDatasetID
                                varMap[":diffNum"] = abs(diffNum)
                                varMap[":ngType1"] = "trn_log"
                                varMap[":ngType2"] = "trn_output"
                                tmp_log.debug(sqlJediDU + comment + str(varMap))
                                self.cur.execute(sqlJediDU + comment, varMap)
                        # update lastStart
                        if oldJobStatus in ("starting", "sent") and jobStatus == "running":
                            sqlLS = "UPDATE ATLAS_PANDAMETA.siteData SET lastStart=CURRENT_DATE "
                            sqlLS += "WHERE site=:site AND hours=:hours AND flag IN (:flag1,:flag2) "
                            varMap = {}
                            varMap[":site"] = computingSite
                            varMap[":hours"] = 3
                            varMap[":flag1"] = "production"
                            varMap[":flag2"] = "analysis"
                            self.cur.execute(sqlLS + comment, varMap)
                            tmp_log.debug("updated lastStart")
                        # update input
                        if updatedFlag and jediTaskID is not None and jobStatus == "running" and oldJobStatus != jobStatus:
                            self.updateInputStatusJedi(jediTaskID, pandaID, jobStatus)
                        # register corrupted zip files
                        if updatedFlag and "corruptedFiles" in param and eventService == EventServiceUtils.esMergeJobFlagNumber:
                            # get exiting files
                            sqlCorF = "SELECT lfn FROM ATLAS_PANDA.filesTable4 WHERE PandaID=:PandaID AND type=:type "
                            varMap = {}
                            varMap[":PandaID"] = pandaID
                            varMap[":type"] = "zipinput"
                            self.cur.execute(sqlCorF + comment, varMap)
                            resCorF = self.cur.fetchall()
                            exCorFiles = set()
                            for (tmpLFN,) in resCorF:
                                exCorFiles.add(tmpLFN)
                            # register files
                            tmpJobSpec = JobSpec()
                            tmpJobSpec.PandaID = pandaID
                            sqlCorIN = f"INSERT INTO ATLAS_PANDA.filesTable4 ({FileSpec.columnNames()}) "
                            sqlCorIN += FileSpec.bindValuesExpression(useSeq=True)
                            for tmpLFN in param["corruptedFiles"].split(","):
                                tmpLFN = tmpLFN.strip()
                                if tmpLFN in exCorFiles or tmpLFN == "":
                                    continue
                                tmpFileSpec = FileSpec()
                                tmpFileSpec.jediTaskID = jediTaskID
                                tmpFileSpec.fsize = 0
                                tmpFileSpec.lfn = tmpLFN
                                tmpFileSpec.type = "zipinput"
                                tmpFileSpec.status = "corrupted"
                                tmpJobSpec.addFile(tmpFileSpec)
                                varMap = tmpFileSpec.valuesMap(useSeq=True)
                                self.cur.execute(sqlCorIN + comment, varMap)
                        # add params to execute getEventRanges later
                        if updatedFlag and is_job_cloning and jobStatus == "running" and oldJobStatus in ["sent", "starting"]:
                            action_in_downstream = {"action": "get_event", "pandaID": pandaID, "jobsetID": jobsetID, "jediTaskID": jediTaskID}
                            tmp_log.debug(f'take action={action_in_downstream["action"]} in downstream')
                        # try to update the lastupdate column in the harvester_rel_job_worker table to propagate
                        # changes to Elastic Search
                        sqlJWU = "UPDATE ATLAS_PANDA.Harvester_Rel_Jobs_Workers SET lastUpdate=:lastUpdate "
                        sqlJWU += "WHERE PandaID=:PandaID "
                        varMap = {
                            ":PandaID": pandaID,
                            ":lastUpdate": datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None),
                        }
                        self.cur.execute(sqlJWU + comment, varMap)
                        nRow = self.cur.rowcount
                        tmp_log.debug(f"{nRow} workers updated")

                        try:
                            # try to update the computing element from the harvester worker table
                            sql_ce = """
                                     UPDATE ATLAS_PANDA.jobsActive4
                                     SET computingelement = (SELECT * FROM (
                                       SELECT computingelement FROM ATLAS_PANDA.harvester_workers hw, ATLAS_PANDA.Harvester_Rel_Jobs_Workers hrjw
                                       WHERE hw.workerid = hrjw.workerid AND hw.harvesterid = hrjw.harvesterid AND hrjw.pandaid = :PandaID ORDER BY hw.workerid DESC
                                       ) WHERE rownum=1)
                                     where PandaID=:PandaID
                                     """
                            varMap = {":PandaID": pandaID}
                            self.cur.execute(sql_ce + comment, varMap)
                            nRow = self.cur.rowcount
                            tmp_log.debug(f"succeeded to update CE from harvester table (rowcount={nRow})")
                        except Exception:
                            tmp_log.error(f"updateJobStatus : failed to update CE from harvester table with {traceback.format_exc()}")
                    # push status change
                    self.push_job_status_message(None, pandaID, jobStatus, jediTaskID, specialHandling, extra_data={"computingsite": computingSite})
                else:
                    tmp_log.debug("not found")
                    # already deleted or bad attempt number
                    ret = "tobekilled"
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                # record status change
                try:
                    if updatedFlag and oldJobStatus is not None and oldJobStatus != jobStatus:
                        self.recordStatusChange(
                            pandaID,
                            jobStatus,
                            infoMap={
                                "computingSite": computingSite,
                                "cloud": cloud,
                                "prodSourceLabel": prodSourceLabel,
                            },
                        )
                except Exception:
                    tmp_log.error("recordStatusChange in updateJobStatus")
                tmp_log.debug("done")
                return ret, action_in_downstream
            except Exception as e:
                # roll back
                self._rollback(True)
                if iTry + 1 < nTry:
                    tmp_log.debug(f"retry : {iTry}")
                    time.sleep(random.randint(10, 20))
                    continue
                # dump error
                self.dumpErrorMessage(_logger, method_name)
                return False, None

    # update job information in jobsActive or jobsDefined
    def updateJob(self, job, inJobsDefined, oldJobStatus=None, extraInfo=None):
        comment = " /* DBProxy.updateJob */"
        _logger.debug(f"updateJob : {job.PandaID}")
        updatedFlag = False
        nTry = 3
        for iTry in range(nTry):
            try:
                job.modificationTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
                # set stateChangeTime for defined->assigned
                if inJobsDefined:
                    job.stateChangeTime = job.modificationTime
                # make SQL
                if inJobsDefined:
                    sql1 = f"UPDATE ATLAS_PANDA.jobsDefined4 SET {job.bindUpdateChangesExpression()} "
                    sql_last_jobstatus = "SELECT jobStatus FROM ATLAS_PANDA.jobsDefined4 "
                else:
                    sql1 = f"UPDATE ATLAS_PANDA.jobsActive4 SET {job.bindUpdateChangesExpression()} "
                    sql_last_jobstatus = "SELECT jobStatus FROM ATLAS_PANDA.jobsActive4 "
                sql1 += "WHERE PandaID=:PandaID "
                sql_last_jobstatus += "WHERE PandaID=:PandaID "
                if inJobsDefined:
                    sql1 += " AND (jobStatus=:oldJobStatus1 OR jobStatus=:oldJobStatus2) "
                # begin transaction
                self.conn.begin()
                # get jobstatus before update
                varMap = {":PandaID": job.PandaID}
                _logger.debug(sql_last_jobstatus + comment + str(varMap))
                self.cur.execute(sql_last_jobstatus + comment, varMap)
                res_last_jobstatus = self.cur.fetchall()
                last_jobstatus = None
                for (js,) in res_last_jobstatus:
                    last_jobstatus = js
                    break
                # update
                varMap = job.valuesMap(onlyChanged=True)
                varMap[":PandaID"] = job.PandaID
                if inJobsDefined:
                    varMap[":oldJobStatus1"] = "assigned"
                    varMap[":oldJobStatus2"] = "defined"
                _logger.debug(sql1 + comment + str(varMap))
                self.cur.execute(sql1 + comment, varMap)
                n = self.cur.rowcount
                if n == 0:
                    # already killed or activated
                    _logger.debug(f"updateJob : Not found {job.PandaID}")
                else:
                    # check if JEDI is used
                    useJEDI = False
                    if (
                        oldJobStatus != job.jobStatus
                        and (job.jobStatus in ["transferring", "merging"] or oldJobStatus in ["transferring", "merging"])
                        and hasattr(panda_config, "useJEDI")
                        and panda_config.useJEDI is True
                        and job.lockedby == "jedi"
                        and self.checkTaskStatusJEDI(job.jediTaskID, self.cur)
                    ):
                        useJEDI = True
                    # SQL to check JEDI files
                    sqlJediFJ = "SELECT /*+ INDEX_RS_ASC(JEDI_DATASET_CONTENTS (JEDI_DATASET_CONTENTS.JEDITASKID JEDI_DATASET_CONTENTS.DATASETID JEDI_DATASET_CONTENTS.FILEID)) */ 1 FROM ATLAS_PANDA.JEDI_Dataset_Contents "
                    sqlJediFJ += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                    sqlJediFJ += "AND attemptNr=:attemptNr AND status=:status AND keepTrack=:keepTrack "
                    datasetContentsStat = {}
                    # loop over all files
                    for file in job.Files:
                        sqlF = f"UPDATE ATLAS_PANDA.filesTable4 SET {file.bindUpdateChangesExpression()}" + "WHERE row_ID=:row_ID"
                        varMap = file.valuesMap(onlyChanged=True)
                        if varMap != {}:
                            varMap[":row_ID"] = file.row_ID
                            _logger.debug(sqlF + comment + str(varMap))
                            self.cur.execute(sqlF + comment, varMap)
                        # actions for JEDI
                        if (
                            useJEDI
                            and (job.jobStatus == "transferring" or oldJobStatus == "transferring")
                            and file.type in ["input", "pseudo_input"]
                            and job.processingType != "pmerge"
                        ):
                            # check file in JEDI
                            varMap = {}
                            varMap[":jediTaskID"] = file.jediTaskID
                            varMap[":datasetID"] = file.datasetID
                            varMap[":fileID"] = file.fileID
                            varMap[":attemptNr"] = file.attemptNr
                            varMap[":status"] = "running"
                            varMap[":keepTrack"] = 1
                            self.cur.execute(sqlJediFJ + comment, varMap)
                            res = self.cur.fetchone()
                            if res is not None:
                                if file.datasetID not in datasetContentsStat:
                                    datasetContentsStat[file.datasetID] = {
                                        "diff": 0,
                                        "cType": "hold",
                                    }
                                if job.jobStatus == "transferring":
                                    # increment nOnHold
                                    datasetContentsStat[file.datasetID]["diff"] += 1
                                else:
                                    # decrement nOnHold
                                    datasetContentsStat[file.datasetID]["diff"] -= 1
                        elif useJEDI and job.jobStatus == "merging" and file.type in ["log", "output"] and file.status != "nooutput":
                            # SQL to update JEDI files
                            varMap = {}
                            varMap[":fileID"] = file.fileID
                            varMap[":attemptNr"] = file.attemptNr
                            varMap[":datasetID"] = file.datasetID
                            varMap[":keepTrack"] = 1
                            varMap[":jediTaskID"] = file.jediTaskID
                            varMap[":status"] = "ready"
                            varMap[":boundaryID"] = job.PandaID
                            varMap[":maxAttempt"] = file.attemptNr + 3
                            sqlJFile = "UPDATE ATLAS_PANDA.JEDI_Dataset_Contents "
                            sqlJFile += "SET status=:status,boundaryID=:boundaryID,maxAttempt=:maxAttempt"
                            for tmpKey in ["lfn", "GUID", "fsize", "checksum"]:
                                tmpVal = getattr(file, tmpKey)
                                if tmpVal == "NULL":
                                    if tmpKey in file._zeroAttrs:
                                        tmpVal = 0
                                    else:
                                        tmpVal = None
                                tmpMapKey = f":{tmpKey}"
                                sqlJFile += f",{tmpKey}={tmpMapKey}"
                                varMap[tmpMapKey] = tmpVal
                            sqlJFile += " WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                            sqlJFile += "AND attemptNr=:attemptNr AND keepTrack=:keepTrack "
                            # update JEDI file
                            _logger.debug(sqlJFile + comment + str(varMap))
                            self.cur.execute(sqlJFile + comment, varMap)
                            nRow = self.cur.rowcount
                            if nRow == 1:
                                if file.datasetID not in datasetContentsStat:
                                    datasetContentsStat[file.datasetID] = {
                                        "diff": 0,
                                        "cType": "hold",
                                    }
                                datasetContentsStat[file.datasetID]["diff"] += 1
                        # update metadata in JEDI
                        if useJEDI and file.type in ["output", "log"] and extraInfo is not None:
                            varMap = {}
                            sqlFileMeta = ""
                            if "nevents" in extraInfo and file.lfn in extraInfo["nevents"]:
                                tmpKey = "nEvents"
                                tmpMapKey = f":{tmpKey}"
                                sqlFileMeta += f"{tmpKey}={tmpMapKey},"
                                varMap[tmpMapKey] = extraInfo["nevents"][file.lfn]
                            if "lbnr" in extraInfo and file.lfn in extraInfo["lbnr"]:
                                tmpKey = "lumiBlockNr"
                                tmpMapKey = f":{tmpKey}"
                                sqlFileMeta += f"{tmpKey}={tmpMapKey},"
                                varMap[tmpMapKey] = extraInfo["lbnr"][file.lfn]
                            if varMap != {}:
                                # update
                                varMap[":fileID"] = file.fileID
                                varMap[":attemptNr"] = file.attemptNr
                                varMap[":datasetID"] = file.datasetID
                                varMap[":jediTaskID"] = file.jediTaskID
                                varMap[":keepTrack"] = 1
                                sqlFileMeta = "UPDATE ATLAS_PANDA.JEDI_Dataset_Contents SET " + sqlFileMeta
                                sqlFileMeta = sqlFileMeta[:-1]
                                sqlFileMeta += " "
                                sqlFileMeta += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                                sqlFileMeta += "AND attemptNr=:attemptNr AND keepTrack=:keepTrack "
                                _logger.debug(sqlFileMeta + comment + str(varMap))
                                self.cur.execute(sqlFileMeta + comment, varMap)
                    # loop over all JEDI datasets
                    tmpDatasetIDs = sorted(datasetContentsStat)
                    for tmpDatasetID in tmpDatasetIDs:
                        valMap = datasetContentsStat[tmpDatasetID]
                        diffNum = valMap["diff"]
                        cType = valMap["cType"]
                        # no difference
                        if diffNum == 0:
                            continue
                        # SQL to check lock
                        varMap = {}
                        varMap[":jediTaskID"] = job.jediTaskID
                        varMap[":datasetID"] = tmpDatasetID
                        sqlJediCL = "SELECT nFilesTobeUsed,nFilesOnHold,status FROM ATLAS_PANDA.JEDI_Datasets "
                        sqlJediCL += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                        sqlJediCL += "FOR UPDATE NOWAIT "
                        _logger.debug(sqlJediCL + comment + str(varMap))
                        self.cur.execute(sqlJediCL + comment, varMap)
                        # SQL to update dataset
                        varMap = {}
                        varMap[":jediTaskID"] = job.jediTaskID
                        varMap[":datasetID"] = tmpDatasetID
                        varMap[":diffNum"] = abs(diffNum)
                        sqlJediDU = "UPDATE ATLAS_PANDA.JEDI_Datasets SET "
                        if cType == "hold":
                            if diffNum > 0:
                                sqlJediDU += "nFilesOnHold=nFilesOnHold+:diffNum "
                            else:
                                sqlJediDU += "nFilesOnHold=nFilesOnHold-:diffNum "
                        elif cType == "touse":
                            varMap[":status"] = "ready"
                            sqlJediDU += "nFilesTobeUsed=nFilesTobeUsed+:diffNum,status=:status "
                        sqlJediDU += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                        _logger.debug(sqlJediDU + comment + str(varMap))
                        self.cur.execute(sqlJediDU + comment, varMap)
                    # update job parameters
                    sqlJobP = "UPDATE ATLAS_PANDA.jobParamsTable SET jobParameters=:param WHERE PandaID=:PandaID"
                    varMap = {}
                    varMap[":PandaID"] = job.PandaID
                    varMap[":param"] = job.jobParameters
                    self.cur.execute(sqlJobP + comment, varMap)
                    updatedFlag = True
                    # update input
                    if useJEDI and job.jobStatus in ["transferring"]:
                        self.updateInputStatusJedi(job.jediTaskID, job.PandaID, job.jobStatus)
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                # record status change
                try:
                    if updatedFlag and job.jobStatus != last_jobstatus:
                        self.recordStatusChange(job.PandaID, job.jobStatus, jobInfo=job)
                        self.push_job_status_message(job, job.PandaID, job.jobStatus)
                except Exception:
                    _logger.error("recordStatusChange in updateJob")
                return True
            except Exception:
                # roll back
                self._rollback(True)
                if iTry + 1 < nTry:
                    _logger.debug(f"updateJob : {job.PandaID} retry : {iTry}")
                    time.sleep(random.randint(3, 10))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error(f"updateJob : {type} {value}")
                return False

    # update the worker status as seen by the pilot
    def updateWorkerPilotStatus(self, workerID, harvesterID, status, node_id):
        comment = " /* DBProxy.updateWorkerPilotStatus */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_logger = LogWrapper(
            _logger,
            method_name + f"< harvesterID={harvesterID} workerID={workerID} >",
        )

        timestamp_utc = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
        var_map = {
            ":status": status,
            ":harvesterID": harvesterID,
            ":workerID": workerID,
            ":nodeID": node_id,
        }
        sql = "UPDATE ATLAS_PANDA.harvester_workers SET pilotStatus=:status,nodeID=:nodeID "

        tmp_logger.debug(f"Updating to status={status} nodeID={node_id} at {timestamp_utc}")

        # add the start or end time
        if status == "started":
            sql += ", pilotStartTime=:now "
            var_map[":now"] = timestamp_utc
        elif status == "finished":
            sql += ", pilotEndTime=:now "
            var_map[":now"] = timestamp_utc

        sql += "WHERE workerID=:workerID AND harvesterID=:harvesterID "

        try:
            self.conn.begin()
            self.cur.execute(sql + comment, var_map)
            retD = self.cur.rowcount
            if not self._commit():
                raise RuntimeError("Commit error")
            tmp_logger.debug(f"Updated successfully with {retD}")
            return True

        except Exception:
            # roll back
            self._rollback(True)
            err_type, err_value = sys.exc_info()[:2]
            tmp_logger.error(f"updateWorkerPilotStatus : {err_type} {err_value}")
            return False

    # retry analysis job
    def retryJob(
        self,
        pandaID,
        param,
        failedInActive=False,
        changeJobInMem=False,
        inMemJob=None,
        getNewPandaID=False,
        attemptNr=None,
        recoverableEsMerge=False,
    ):
        comment = " /* DBProxy.retryJob */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" < PandaID={pandaID} >"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug(f"inActive={failedInActive}")
        sql1 = f"SELECT {JobSpec.columnNames()} FROM ATLAS_PANDA.jobsActive4 "
        sql1 += "WHERE PandaID=:PandaID "
        if failedInActive:
            sql1 += "AND jobStatus=:jobStatus "
        updatedFlag = False
        nTry = 3
        for iTry in range(nTry):
            try:
                retValue = False
                if not changeJobInMem:
                    # begin transaction
                    self.conn.begin()
                    # select
                    varMap = {}
                    varMap[":PandaID"] = pandaID
                    if failedInActive:
                        varMap[":jobStatus"] = "failed"
                    self.cur.arraysize = 10
                    self.cur.execute(sql1 + comment, varMap)
                    res = self.cur.fetchall()
                    if len(res) == 0:
                        tmpLog.debug("PandaID not found")
                        self._rollback()
                        return retValue
                    job = JobSpec()
                    job.pack(res[0])
                else:
                    job = inMemJob
                # don't use getNewPandaID for buildJob since the order of PandaIDs is broken
                if getNewPandaID and job.prodSourceLabel in ["panda"]:
                    if not changeJobInMem:
                        # commit
                        if not self._commit():
                            raise RuntimeError("Commit error")
                    # return
                    return retValue
                # convert attemptNr to int
                try:
                    attemptNr = int(attemptNr)
                except Exception:
                    tmpLog.debug(f"attemptNr={attemptNr} non-integer")
                    attemptNr = -999
                # check attemptNr
                if attemptNr is not None:
                    if job.attemptNr != attemptNr:
                        tmpLog.debug(f"bad attemptNr job.{job.attemptNr} != pilot.{attemptNr}")
                        if not changeJobInMem:
                            # commit
                            if not self._commit():
                                raise RuntimeError("Commit error")
                        # return
                        return retValue
                # check if already retried
                if job.taskBufferErrorCode in [
                    ErrorCode.EC_Reassigned,
                    ErrorCode.EC_Retried,
                    ErrorCode.EC_PilotRetried,
                ]:
                    tmpLog.debug(f"already retried {job.taskBufferErrorCode}")
                    if not changeJobInMem:
                        # commit
                        if not self._commit():
                            raise RuntimeError("Commit error")
                    # return
                    return retValue
                # use JEDI
                useJEDI = False
                if (
                    hasattr(panda_config, "useJEDI")
                    and panda_config.useJEDI is True
                    and job.lockedby == "jedi"
                    and self.checkTaskStatusJEDI(job.jediTaskID, self.cur)
                ):
                    useJEDI = True
                # check pilot retry
                usePilotRetry = False
                if (
                    job.prodSourceLabel in ["user", "panda"] + JobUtils.list_ptest_prod_sources
                    and "pilotErrorCode" in param
                    and param["pilotErrorCode"].startswith("-")
                    and job.maxAttempt > job.attemptNr
                    and (not job.processingType.startswith("gangarobot") or job.processingType == "gangarobot-rctest")
                    and not job.processingType.startswith("hammercloud")
                ):
                    usePilotRetry = True
                # retry for ES merge
                if recoverableEsMerge and EventServiceUtils.isEventServiceMerge(job) and job.maxAttempt > job.attemptNr:
                    usePilotRetry = True
                # check if it's analysis job # FIXME once pilot retry works correctly the conditions below will be cleaned up
                if (
                    (
                        (job.prodSourceLabel == "user" or job.prodSourceLabel == "panda")
                        and not job.processingType.startswith("gangarobot")
                        and not job.processingType.startswith("hammercloud")
                        and "pilotErrorCode" in param
                        and param["pilotErrorCode"] in ["1200", "1201", "1213"]
                        and (not job.computingSite.startswith("ANALY_LONG_"))
                        and job.attemptNr < 2
                    )
                    or failedInActive
                    or usePilotRetry
                ) and job.commandToPilot != "tobekilled":
                    # check attemptNr for JEDI
                    moreRetryForJEDI = True
                    if useJEDI:
                        moreRetryForJEDI = self.checkMoreRetryJEDI(job)
                    # OK in JEDI
                    if moreRetryForJEDI:
                        tmpLog.debug(f"reset PandaID:{job.PandaID} #{job.attemptNr}")
                        if not changeJobInMem:
                            # job parameters
                            sqlJobP = "SELECT jobParameters FROM ATLAS_PANDA.jobParamsTable WHERE PandaID=:PandaID"
                            varMap = {}
                            varMap[":PandaID"] = job.PandaID
                            self.cur.execute(sqlJobP + comment, varMap)
                            for (clobJobP,) in self.cur:
                                try:
                                    job.jobParameters = clobJobP.read()
                                except AttributeError:
                                    job.jobParameters = str(clobJobP)
                                break
                        # reset job
                        job.jobStatus = "activated"
                        job.startTime = None
                        job.modificationTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
                        job.attemptNr = job.attemptNr + 1
                        if usePilotRetry:
                            job.currentPriority -= 10
                        job.endTime = None
                        job.transExitCode = None
                        job.batchID = None
                        for attr in job._attributes:
                            if attr.endswith("ErrorCode") or attr.endswith("ErrorDiag"):
                                setattr(job, attr, None)
                        # remove flag regarding to pledge-resource handling
                        if job.specialHandling not in [None, "NULL", ""]:
                            newSpecialHandling = re.sub(",*localpool", "", job.specialHandling)
                            if newSpecialHandling == "":
                                job.specialHandling = None
                            else:
                                job.specialHandling = newSpecialHandling
                        # send it to long queue for analysis jobs
                        oldComputingSite = job.computingSite
                        if not changeJobInMem:
                            if job.computingSite.startswith("ANALY"):
                                longSite = None
                                tmpLongSiteList = []
                                tmpLongSite = re.sub("^ANALY_", "ANALY_LONG_", job.computingSite)
                                tmpLongSite = re.sub("_\d+$", "", tmpLongSite)
                                tmpLongSiteList.append(tmpLongSite)
                                tmpLongSite = job.computingSite + "_LONG"
                                tmpLongSiteList.append(tmpLongSite)
                                tmpLongSite = re.sub("SHORT", "LONG", job.computingSite)
                                if tmpLongSite != job.computingSite:
                                    tmpLongSiteList.append(tmpLongSite)
                                # loop over all possible long sitenames
                                for tmpLongSite in tmpLongSiteList:
                                    varMap = {}
                                    varMap[":siteID"] = tmpLongSite
                                    varMap[":status"] = "online"
                                    sqlSite = "SELECT /* use_json_type */ COUNT(*) FROM ATLAS_PANDA.schedconfig_json scj WHERE scj.panda_queue=:siteID AND scj.data.status=:status"
                                    self.cur.execute(sqlSite + comment, varMap)
                                    resSite = self.cur.fetchone()
                                    if resSite is not None and resSite[0] > 0:
                                        longSite = tmpLongSite
                                        break
                                # use long site if exists
                                if longSite is not None:
                                    tmpLog.debug(f"sending PandaID:{job.PandaID} to {longSite}")
                                    job.computingSite = longSite
                                    # set destinationSE if queue is changed
                                    if oldComputingSite == job.destinationSE:
                                        job.destinationSE = job.computingSite
                        if not changeJobInMem:
                            # select files
                            varMap = {}
                            varMap[":PandaID"] = job.PandaID
                            if not getNewPandaID:
                                varMap[":type1"] = "log"
                                varMap[":type2"] = "output"
                            sqlFile = f"SELECT {FileSpec.columnNames()} FROM ATLAS_PANDA.filesTable4 "
                            if not getNewPandaID:
                                sqlFile += "WHERE PandaID=:PandaID AND (type=:type1 OR type=:type2)"
                            else:
                                sqlFile += "WHERE PandaID=:PandaID"
                            self.cur.arraysize = 100
                            self.cur.execute(sqlFile + comment, varMap)
                            resFs = self.cur.fetchall()
                        else:
                            # get log or output files only
                            resFs = []
                            for tmpFile in job.Files:
                                if tmpFile.type in ["log", "output"]:
                                    resFs.append(tmpFile)
                        # loop over all files
                        for resF in resFs:
                            if not changeJobInMem:
                                # set PandaID
                                file = FileSpec()
                                file.pack(resF)
                                job.addFile(file)
                            else:
                                file = resF
                            # set new GUID
                            if file.type == "log":
                                file.GUID = str(uuid.uuid4())
                            # don't change input or lib.tgz, or ES merge output/log since it causes a problem with input name construction
                            if (
                                file.type in ["input", "pseudo_input"]
                                or (file.type == "output" and job.prodSourceLabel == "panda")
                                or (file.type == "output" and file.lfn.endswith(".lib.tgz") and job.prodSourceLabel in JobUtils.list_ptest_prod_sources)
                            ):
                                continue
                            # append attemptNr to LFN
                            oldName = file.lfn
                            file.lfn = re.sub("\.\d+$", "", file.lfn)
                            file.lfn = f"{file.lfn}.{job.attemptNr}"
                            newName = file.lfn
                            # set destinationSE
                            if oldComputingSite == file.destinationSE:
                                file.destinationSE = job.computingSite
                            # modify jobParameters
                            if not recoverableEsMerge:
                                sepPatt = "('|\"|%20|:)" + oldName + "('|\"|%20| )"
                            else:
                                sepPatt = "('|\"| |:|=)" + oldName + "('|\"| |<|$)"
                            matches = re.findall(sepPatt, job.jobParameters)
                            for match in matches:
                                oldPatt = match[0] + oldName + match[-1]
                                newPatt = match[0] + newName + match[-1]
                                job.jobParameters = re.sub(oldPatt, newPatt, job.jobParameters)
                            if not changeJobInMem and not getNewPandaID:
                                # reset file status
                                if file.type in ["output", "log"]:
                                    file.status = "unknown"
                                # update files
                                sqlFup = f"UPDATE ATLAS_PANDA.filesTable4 SET {file.bindUpdateChangesExpression()}" + "WHERE row_ID=:row_ID"
                                varMap = file.valuesMap(onlyChanged=True)
                                if varMap != {}:
                                    varMap[":row_ID"] = file.row_ID
                                    self.cur.execute(sqlFup + comment, varMap)
                        # set site to ES merger job
                        if recoverableEsMerge and EventServiceUtils.isEventServiceMerge(job):
                            self.setSiteForEsMerge(job, False, methodName, comment)
                        if not changeJobInMem:
                            # reuse original PandaID
                            if not getNewPandaID:
                                # update job
                                sql2 = f"UPDATE ATLAS_PANDA.jobsActive4 SET {job.bindUpdateChangesExpression()} "
                                sql2 += "WHERE PandaID=:PandaID "
                                varMap = job.valuesMap(onlyChanged=True)
                                varMap[":PandaID"] = job.PandaID
                                self.cur.execute(sql2 + comment, varMap)
                                # update job parameters
                                sqlJobP = "UPDATE ATLAS_PANDA.jobParamsTable SET jobParameters=:param WHERE PandaID=:PandaID"
                                varMap = {}
                                varMap[":PandaID"] = job.PandaID
                                varMap[":param"] = job.jobParameters
                                self.cur.execute(sqlJobP + comment, varMap)
                                updatedFlag = True
                            else:
                                # read metadata
                                sqlMeta = "SELECT metaData FROM ATLAS_PANDA.metaTable WHERE PandaID=:PandaID"
                                varMap = {}
                                varMap[":PandaID"] = job.PandaID
                                self.cur.execute(sqlMeta + comment, varMap)
                                for (clobJobP,) in self.cur:
                                    try:
                                        job.metadata = clobJobP.read()
                                    except AttributeError:
                                        job.metadata = str(clobJobP)
                                    break
                                # insert job with new PandaID
                                sql1 = f"INSERT INTO ATLAS_PANDA.jobsActive4 ({JobSpec.columnNames()}) "
                                sql1 += JobSpec.bindValuesExpression(useSeq=True)
                                sql1 += " RETURNING PandaID INTO :newPandaID"
                                # set parentID
                                job.parentID = job.PandaID
                                job.creationTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
                                job.modificationTime = job.creationTime
                                varMap = job.valuesMap(useSeq=True)
                                varMap[":newPandaID"] = self.cur.var(varNUMBER)
                                # insert
                                retI = self.cur.execute(sql1 + comment, varMap)
                                # set PandaID
                                val = self.getvalue_corrector(self.cur.getvalue(varMap[":newPandaID"]))
                                job.PandaID = int(val)
                                tmpLog.debug(f"Generate new PandaID {job.parentID} -> {job.PandaID} #{job.attemptNr}")
                                # insert files
                                sqlFile = f"INSERT INTO ATLAS_PANDA.filesTable4 ({FileSpec.columnNames()}) "
                                sqlFile += FileSpec.bindValuesExpression(useSeq=True)
                                sqlFile += " RETURNING row_ID INTO :newRowID"
                                for file in job.Files:
                                    # reset rowID
                                    file.row_ID = None
                                    # insert
                                    varMap = file.valuesMap(useSeq=True)
                                    varMap[":newRowID"] = self.cur.var(varNUMBER)
                                    self.cur.execute(sqlFile + comment, varMap)
                                    val = self.getvalue_corrector(self.cur.getvalue(varMap[":newRowID"]))
                                    file.row_ID = int(val)
                                # job parameters
                                sqlJob = "INSERT INTO ATLAS_PANDA.jobParamsTable (PandaID,jobParameters) VALUES (:PandaID,:param)"
                                varMap = {}
                                varMap[":PandaID"] = job.PandaID
                                varMap[":param"] = job.jobParameters
                                self.cur.execute(sqlJob + comment, varMap)
                                # set error code to original job to avoid being retried by another process
                                sqlE = "UPDATE ATLAS_PANDA.jobsActive4 SET taskBufferErrorCode=:errCode,taskBufferErrorDiag=:errDiag WHERE PandaID=:PandaID"
                                varMap = {}
                                varMap[":PandaID"] = job.parentID
                                varMap[":errCode"] = ErrorCode.EC_PilotRetried
                                varMap[":errDiag"] = f"retrying at the same site. new PandaID={job.PandaID}"
                                self.cur.execute(sqlE + comment, varMap)
                                # propagate change to JEDI
                                if useJEDI:
                                    self.updateForPilotRetryJEDI(job, self.cur)
                        # set return
                        if not getNewPandaID:
                            retValue = True
                if not changeJobInMem:
                    # commit
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    # record status change
                    try:
                        if updatedFlag:
                            self.recordStatusChange(job.PandaID, job.jobStatus, jobInfo=job)
                            self.push_job_status_message(job, job.PandaID, job.jobStatus)
                    except Exception:
                        tmpLog.error("recordStatusChange in retryJob")
                return retValue
            except Exception:
                # roll back
                self._rollback()
                if iTry + 1 < nTry:
                    _logger.debug(f"retryJob : {pandaID} retry : {iTry}")
                    time.sleep(random.randint(10, 20))
                    continue
                # error report
                self.dumpErrorMessage(_logger, methodName)
                return False

    def construct_where_clause(
        self,
        site_name,
        mem,
        disk_space,
        background,
        resource_type,
        prod_source_label,
        computing_element,
        is_gu,
        job_type,
        prod_user_id,
        task_id,
        average_memory_limit,
    ):
        get_val_map = {":oldJobStatus": "activated", ":computingSite": site_name}

        sql_where_clause = "WHERE jobStatus=:oldJobStatus AND computingSite=:computingSite "

        if mem not in [0, "0"]:
            sql_where_clause += "AND (minRamCount<=:minRamCount OR minRamCount=0) "
            get_val_map[":minRamCount"] = mem

        if disk_space not in [0, "0"]:
            sql_where_clause += "AND (maxDiskCount<=:maxDiskCount OR maxDiskCount=0) "
            get_val_map[":maxDiskCount"] = disk_space

        if background is True:
            sql_where_clause += "AND jobExecutionID=1 "

        if resource_type is not None:
            sql_where_clause += "AND resource_type=:resourceType "
            get_val_map[":resourceType"] = resource_type

        if prod_source_label == "user":
            sql_where_clause += "AND prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2,:prodSourceLabel3) "
            get_val_map[":prodSourceLabel1"] = "user"
            get_val_map[":prodSourceLabel2"] = "panda"
            get_val_map[":prodSourceLabel3"] = "install"
        elif prod_source_label in [None, "managed"]:
            sql_where_clause += "AND prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2,:prodSourceLabel3,:prodSourceLabel4) "
            get_val_map[":prodSourceLabel1"] = "managed"
            get_val_map[":prodSourceLabel2"] = "test"
            get_val_map[":prodSourceLabel3"] = "prod_test"
            get_val_map[":prodSourceLabel4"] = "install"
        elif prod_source_label == "test" and computing_element is not None:
            if is_gu and job_type == "user":
                sql_where_clause += "AND processingType=:processingType1 "
                get_val_map[":processingType1"] = "gangarobot"
            else:
                sql_where_clause += "AND (processingType=:processingType1 OR prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2,:prodSourceLabel3)) "
                get_val_map[":processingType1"] = "gangarobot"
                get_val_map[":prodSourceLabel1"] = "prod_test"
                get_val_map[":prodSourceLabel2"] = "install"
                get_val_map[":prodSourceLabel3"] = "test"
        elif prod_source_label == "unified":
            sql_where_clause += (
                "AND prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2,:prodSourceLabel3,:prodSourceLabel4,:prodSourceLabel5,:prodSourceLabel6) "
            )
            get_val_map[":prodSourceLabel1"] = "managed"
            get_val_map[":prodSourceLabel2"] = "test"
            get_val_map[":prodSourceLabel3"] = "prod_test"
            get_val_map[":prodSourceLabel4"] = "install"
            get_val_map[":prodSourceLabel5"] = "user"
            get_val_map[":prodSourceLabel6"] = "panda"
        else:
            sql_where_clause += "AND prodSourceLabel=:prodSourceLabel "
            get_val_map[":prodSourceLabel"] = prod_source_label

        if prod_user_id is not None:
            compact_dn = self.cleanUserID(prod_user_id)
            if compact_dn in ["", "NULL", None]:
                compact_dn = prod_user_id
            sql_where_clause += "AND prodUserName=:prodUserName "
            get_val_map[":prodUserName"] = compact_dn

        if task_id not in [None, "NULL"]:
            sql_where_clause += "AND jediTaskID=:taskID "
            get_val_map[":taskID"] = task_id

        if average_memory_limit:
            sql_where_clause += "AND minramcount / NVL(corecount, 1)<=:average_memory_limit "
            get_val_map[":average_memory_limit"] = average_memory_limit

        return sql_where_clause, get_val_map

    # get jobs
    def getJobs(
        self,
        nJobs,
        siteName,
        prodSourceLabel,
        mem,
        diskSpace,
        node,
        timeout,
        computingElement,
        prodUserID,
        taskID,
        background,
        resourceType,
        harvester_id,
        worker_id,
        schedulerID,
        jobType,
        is_gu,
        via_topic,
    ):
        """
        1. Construct where clause (sql_where_clause) based on applicable filters for request
        2. Select n jobs with the highest priorities and the lowest pandaids
        3. Update the jobs to status SENT
        4. Pack the files and if jobs are AES also the event ranges
        """
        comment = " /* DBProxy.getJobs */"
        timeStart = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
        tmpLog = LogWrapper(_logger, f"getJobs : {siteName} {datetime.datetime.isoformat(timeStart)} -> ")
        tmpLog.debug("Start")

        # Number of PanDAIDs that will be tried
        if hasattr(panda_config, "nJobsInGetJob"):
            maxAttemptIDx = panda_config.nJobsInGetJob
        else:
            maxAttemptIDx = 10

        # get the configuration for maximum workers of each type
        is_push_queue = False
        average_memory_target = None
        average_memory_limit = None
        pq_data_des = self.get_config_for_pq(siteName)
        if not pq_data_des:
            tmpLog.debug("Error retrieving queue configuration from DB, limits can not be applied")
        else:
            try:
                if pq_data_des["meanrss"] != 0:
                    average_memory_target = pq_data_des["meanrss"]
            except KeyError:
                pass
            try:
                workflow = pq_data_des["workflow"]
                if workflow and workflow.startswith("push"):
                    is_push_queue = True
            except KeyError:
                pass

        if is_push_queue and average_memory_target:
            average_memory_jobs_running_submitted, average_memory_jobs_running = self.get_average_memory_jobs(siteName, average_memory_target)
            if average_memory_jobs_running_submitted > average_memory_target or average_memory_jobs_running > average_memory_target:
                average_memory_limit = average_memory_target
                tmpLog.info(f"Queue {siteName} meanRSS will be throttled to jobs under {average_memory_limit}MB")

        # generate the WHERE clauses based on the requirements for the job
        sql_where_clause, getValMap = self.construct_where_clause(
            site_name=siteName,
            mem=mem,
            disk_space=diskSpace,
            background=background,
            resource_type=resourceType,
            prod_source_label=prodSourceLabel,
            computing_element=computingElement,
            is_gu=is_gu,
            job_type=jobType,
            prod_user_id=prodUserID,
            task_id=taskID,
            average_memory_limit=average_memory_limit,
        )

        # get the sorting criteria (global shares, age, etc.)
        sorting_sql, sorting_varmap = self.getSortingCriteria(siteName, maxAttemptIDx)
        if sorting_varmap:  # copy the var map, but not the sql, since it has to be at the very end
            for tmp_key in sorting_varmap:
                getValMap[tmp_key] = sorting_varmap[tmp_key]

        retJobs = []
        nSent = 0
        getValMapOrig = copy.copy(getValMap)

        try:
            timeLimit = datetime.timedelta(seconds=timeout - 10)

            # get nJobs
            for iJob in range(nJobs):
                getValMap = copy.copy(getValMapOrig)
                pandaID = 0

                nTry = 1
                for iTry in range(nTry):
                    # set siteID
                    tmpSiteID = siteName
                    # get file lock
                    tmpLog.debug("lock")
                    if (datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - timeStart) < timeLimit:
                        toGetPandaIDs = True
                        pandaIDs = []
                        specialHandlingMap = {}

                        if toGetPandaIDs:
                            # get PandaIDs
                            sqlP = "SELECT /*+ INDEX_RS_ASC(tab (PRODSOURCELABEL COMPUTINGSITE JOBSTATUS) ) */ PandaID,currentPriority,specialHandling FROM ATLAS_PANDA.jobsActive4 tab "
                            sqlP += sql_where_clause

                            if sorting_sql:
                                sqlP = "SELECT * FROM (" + sqlP
                                sqlP += sorting_sql

                            tmpLog.debug(sqlP + comment + str(getValMap))
                            # start transaction
                            self.conn.begin()
                            # select
                            self.cur.arraysize = 100000
                            self.cur.execute(sqlP + comment, getValMap)
                            resIDs = self.cur.fetchall()
                            # commit
                            if not self._commit():
                                raise RuntimeError("Commit error")

                            for (
                                tmpPandaID,
                                tmpCurrentPriority,
                                tmpSpecialHandling,
                            ) in resIDs:
                                pandaIDs.append(tmpPandaID)
                                specialHandlingMap[tmpPandaID] = tmpSpecialHandling

                        if pandaIDs == []:
                            tmpLog.debug("no PandaIDs")
                            retU = 0  # retU: return from update
                        else:
                            # update
                            for indexID, tmpPandaID in enumerate(pandaIDs):
                                # max attempts
                                if indexID > maxAttemptIDx:
                                    break
                                # lock first
                                sqlPL = "SELECT jobStatus FROM ATLAS_PANDA.jobsActive4 " "WHERE PandaID=:PandaID FOR UPDATE NOWAIT "
                                # update
                                sqlJ = "UPDATE ATLAS_PANDA.jobsActive4 "
                                sqlJ += "SET jobStatus=:newJobStatus,modificationTime=CURRENT_DATE,modificationHost=:modificationHost,startTime=CURRENT_DATE"
                                varMap = {}
                                varMap[":PandaID"] = tmpPandaID
                                varMap[":newJobStatus"] = "sent"
                                varMap[":oldJobStatus"] = "activated"
                                varMap[":modificationHost"] = node
                                # set CE
                                if computingElement is not None:
                                    sqlJ += ",computingElement=:computingElement"
                                    varMap[":computingElement"] = computingElement
                                # set schedulerID
                                if schedulerID is not None:
                                    sqlJ += ",schedulerID=:schedulerID"
                                    varMap[":schedulerID"] = schedulerID

                                # background flag
                                if background is not True:
                                    sqlJ += ",jobExecutionID=0"
                                sqlJ += " WHERE PandaID=:PandaID AND jobStatus=:oldJobStatus"
                                # SQL to get nSent
                                sentLimit = timeStart - datetime.timedelta(seconds=60)
                                sqlSent = "SELECT count(*) FROM ATLAS_PANDA.jobsActive4 WHERE jobStatus=:jobStatus "
                                sqlSent += "AND prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2) "
                                sqlSent += "AND computingSite=:computingSite "
                                sqlSent += "AND modificationTime>:modificationTime "
                                varMapSent = {}
                                varMapSent[":jobStatus"] = "sent"
                                varMapSent[":computingSite"] = tmpSiteID
                                varMapSent[":modificationTime"] = sentLimit
                                varMapSent[":prodSourceLabel1"] = "managed"
                                varMapSent[":prodSourceLabel2"] = "test"

                                # start transaction
                                self.conn.begin()
                                # pre-lock
                                prelocked = False
                                try:
                                    varMapPL = {":PandaID": tmpPandaID}
                                    tmpLog.debug(sqlPL + comment + str(varMapPL))
                                    self.cur.execute(sqlPL + comment, varMapPL)
                                    prelocked = True
                                except Exception:
                                    tmpLog.debug("cannot pre-lock")
                                # update
                                if prelocked:
                                    tmpLog.debug(sqlJ + comment + str(varMap))
                                    self.cur.execute(sqlJ + comment, varMap)
                                    retU = self.cur.rowcount
                                    tmpLog.debug(f"retU={retU}")
                                else:
                                    retU = 0
                                if retU != 0:
                                    # get nSent for production jobs
                                    if prodSourceLabel in [None, "managed"]:
                                        tmpLog.debug(sqlSent + comment + str(varMapSent))
                                        self.cur.execute(sqlSent + comment, varMapSent)
                                        resSent = self.cur.fetchone()
                                        if resSent is not None:
                                            (nSent,) = resSent
                                    # insert job and worker mapping
                                    if harvester_id is not None and worker_id is not None:
                                        # insert worker if missing
                                        self.updateWorkers(
                                            harvester_id,
                                            [
                                                {
                                                    "workerID": worker_id,
                                                    "nJobs": 1,
                                                    "status": "running",
                                                    "lastUpdate": datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None),
                                                }
                                            ],
                                            useCommit=False,
                                        )
                                        # insert mapping
                                        sqlJWH = "SELECT 1 FROM ATLAS_PANDA.Harvester_Instances WHERE harvester_ID=:harvesterID "
                                        sqlJWC = "SELECT PandaID FROM ATLAS_PANDA.Harvester_Rel_Jobs_Workers "
                                        sqlJWC += "WHERE harvesterID=:harvesterID AND workerID=:workerID AND PandaID=:PandaID "
                                        sqlJWI = "INSERT INTO ATLAS_PANDA.Harvester_Rel_Jobs_Workers (harvesterID,workerID,PandaID,lastUpdate) "
                                        sqlJWI += "VALUES (:harvesterID,:workerID,:PandaID,:lastUpdate) "
                                        sqlJWU = "UPDATE ATLAS_PANDA.Harvester_Rel_Jobs_Workers SET lastUpdate=:lastUpdate "
                                        sqlJWU += "WHERE harvesterID=:harvesterID AND workerID=:workerID AND PandaID=:PandaID "
                                        varMap = dict()
                                        varMap[":harvesterID"] = harvester_id
                                        self.cur.execute(sqlJWH + comment, varMap)
                                        resJWH = self.cur.fetchone()
                                        if resJWH is None:
                                            tmpLog.debug(f"getJobs : Site {tmpSiteID} harvester_id={harvester_id} not found")
                                        else:
                                            varMap = dict()
                                            varMap[":harvesterID"] = harvester_id
                                            varMap[":workerID"] = worker_id
                                            varMap[":PandaID"] = tmpPandaID
                                            self.cur.execute(sqlJWC + comment, varMap)
                                            resJWC = self.cur.fetchone()
                                            varMap = dict()
                                            varMap[":harvesterID"] = harvester_id
                                            varMap[":workerID"] = worker_id
                                            varMap[":PandaID"] = tmpPandaID
                                            varMap[":lastUpdate"] = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
                                            if resJWC is None:
                                                # insert
                                                self.cur.execute(sqlJWI + comment, varMap)
                                            else:
                                                # update
                                                self.cur.execute(sqlJWU + comment, varMap)
                                # commit
                                if not self._commit():
                                    raise RuntimeError("Commit error")
                                # succeeded
                                if retU != 0:
                                    pandaID = tmpPandaID
                                    break
                    else:
                        tmpLog.debug("do nothing")
                        retU = 0
                    # release file lock
                    tmpLog.debug("unlock")
                    # succeeded
                    if retU != 0:
                        break
                    if iTry + 1 < nTry:
                        # time.sleep(0.5)
                        pass
                # failed to UPDATE
                if retU == 0:
                    # reset pandaID
                    pandaID = 0
                tmpLog.debug(f"retU {retU} : PandaID {pandaID} - {prodSourceLabel}")
                if pandaID == 0:
                    break

                # start transaction
                self.conn.begin()
                # query to get the DB entry for a specific PanDA ID
                sql_select_job = f"SELECT {JobSpec.columnNames()} FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID"
                varMap = {}
                varMap[":PandaID"] = pandaID
                self.cur.arraysize = 10
                self.cur.execute(sql_select_job + comment, varMap)
                res = self.cur.fetchone()
                if len(res) == 0:
                    # commit
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    break
                # instantiate Job
                job = JobSpec()
                job.pack(res)

                # sql to read range
                sqlRR = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                sqlRR += "PandaID,job_processID,attemptNr,objStore_ID,zipRow_ID,path_convention "
                sqlRR += f"FROM {panda_config.schemaJEDI}.JEDI_Events tab "
                sqlRR += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID AND status=:eventStatus "
                # sql to read log bucket IDs
                sqlLBK = "SELECT jobMetrics FROM ATLAS_PANDA.jobsArchived4 WHERE PandaID=:PandaID "
                sqlLBK += "UNION "
                sqlLBK += "SELECT jobMetrics FROM ATLAS_PANDAARCH.jobsArchived WHERE PandaID=:PandaID AND modificationTime>(CURRENT_DATE-30) "
                # read files
                sqlFile = f"SELECT {FileSpec.columnNames()} FROM ATLAS_PANDA.filesTable4 "
                sqlFile += "WHERE PandaID=:PandaID ORDER BY row_ID "
                # read LFN and dataset name for output files
                sqlFileOut = "SELECT lfn,dataset FROM ATLAS_PANDA.filesTable4 "
                sqlFileOut += "WHERE PandaID=:PandaID AND type=:type "
                # read files from JEDI for jumbo jobs
                sqlFileJEDI = "SELECT lfn,GUID,fsize,checksum "
                sqlFileJEDI += f"FROM {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
                sqlFileJEDI += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                sqlFileJEDI += "ORDER BY lfn "
                # read zip file
                sqlZipFile = "SELECT lfn,destinationSE,fsize,checksum FROM ATLAS_PANDA.filesTable4 "
                sqlZipFile += "WHERE row_ID=:row_ID "
                sqlZipFile += "UNION "
                sqlZipFile += "SELECT lfn,destinationSE,fsize,checksum FROM ATLAS_PANDAARCH.filesTable_ARCH "
                sqlZipFile += "WHERE row_ID=:row_ID "
                self.cur.arraysize = 10000
                self.cur.execute(sqlFile + comment, varMap)
                resFs = self.cur.fetchall()
                eventRangeIDs = {}
                esDonePandaIDs = []
                esOutputZipMap = {}
                esZipRow_IDs = set()
                esOutputFileMap = {}
                # use new file format for ES
                useNewFileFormatForES = False
                if job.AtlasRelease is not None:
                    try:
                        tmpMajorVer = job.AtlasRelease.split("-")[-1].split(".")[0]
                        if int(tmpMajorVer) == 20:
                            useNewFileFormatForES = True
                    except Exception:
                        pass
                for resF in resFs:
                    file = FileSpec()
                    file.pack(resF)
                    # add files except event service merge or jumbo
                    if (not EventServiceUtils.isEventServiceMerge(job) and not EventServiceUtils.isJumboJob(job)) or file.type in ["output", "log"]:
                        job.addFile(file)
                    # read real input files for jumbo jobs
                    elif EventServiceUtils.isJumboJob(job):
                        # get files
                        varMap = {}
                        varMap[":jediTaskID"] = file.jediTaskID
                        varMap[":datasetID"] = file.datasetID
                        self.cur.execute(sqlFileJEDI + comment, varMap)
                        resFileJEDI = self.cur.fetchall()
                        for tmpLFN, tmpGUID, tmpFsize, tmpChecksum in resFileJEDI:
                            newFileSpec = FileSpec()
                            newFileSpec.pack(resF)
                            newFileSpec.lfn = tmpLFN
                            newFileSpec.GUID = tmpGUID
                            newFileSpec.fsize = tmpFsize
                            newFileSpec.checksum = tmpChecksum
                            # add file
                            job.addFile(newFileSpec)
                        continue
                    # construct input files from event ragnes for event service merge
                    if EventServiceUtils.isEventServiceMerge(job):
                        # only for input
                        if file.type not in ["output", "log"]:
                            # get ranges
                            varMap = {}
                            varMap[":jediTaskID"] = file.jediTaskID
                            varMap[":datasetID"] = file.datasetID
                            varMap[":fileID"] = file.fileID
                            varMap[":eventStatus"] = EventServiceUtils.ST_done
                            self.cur.execute(sqlRR + comment, varMap)
                            resRR = self.cur.fetchall()
                            for (
                                esPandaID,
                                job_processID,
                                attemptNr,
                                objStoreID,
                                zipRow_ID,
                                pathConvention,
                            ) in resRR:
                                tmpEventRangeID = self.makeEventRangeID(
                                    file.jediTaskID,
                                    esPandaID,
                                    file.fileID,
                                    job_processID,
                                    attemptNr,
                                )
                                if file.fileID not in eventRangeIDs:
                                    eventRangeIDs[file.fileID] = {}
                                addFlag = False
                                if job_processID not in eventRangeIDs[file.fileID]:
                                    addFlag = True
                                else:
                                    oldEsPandaID = eventRangeIDs[file.fileID][job_processID]["pandaID"]
                                    if esPandaID > oldEsPandaID:
                                        addFlag = True
                                        if oldEsPandaID in esDonePandaIDs:
                                            esDonePandaIDs.remove(oldEsPandaID)
                                if addFlag:
                                    # append
                                    if pathConvention is not None:
                                        objStoreID = f"{objStoreID}/{pathConvention}"
                                    eventRangeIDs[file.fileID][job_processID] = {
                                        "pandaID": esPandaID,
                                        "eventRangeID": tmpEventRangeID,
                                        "objStoreID": objStoreID,
                                    }
                                    # zip file in jobMetrics
                                    if esPandaID not in esDonePandaIDs:
                                        esDonePandaIDs.append(esPandaID)
                                        # get jobMetrics
                                        varMap = {}
                                        varMap[":PandaID"] = esPandaID
                                        self.cur.execute(sqlLBK + comment, varMap)
                                        resLBK = self.cur.fetchone()
                                        if resLBK is not None and resLBK[0] is not None:
                                            outputZipBucketID = None
                                            tmpPatch = re.search("outputZipBucketID=(\d+)", resLBK[0])
                                            if tmpPatch is not None:
                                                outputZipBucketID = tmpPatch.group(1)
                                            outputZipName = None
                                            tmpPatch = re.search("outputZipName=([^ ]+)", resLBK[0])
                                            if tmpPatch is not None:
                                                outputZipName = tmpPatch.group(1)
                                            if outputZipBucketID is not None and outputZipName is not None:
                                                if esPandaID not in esOutputZipMap:
                                                    esOutputZipMap[esPandaID] = []
                                                esOutputZipMap[esPandaID].append(
                                                    {
                                                        "name": outputZipName,
                                                        "osid": outputZipBucketID,
                                                    }
                                                )
                                    # output LFN and dataset
                                    if esPandaID not in esOutputFileMap:
                                        esOutputFileMap[esPandaID] = dict()
                                        varMap = {}
                                        varMap[":PandaID"] = esPandaID
                                        varMap[":type"] = "output"
                                        self.cur.execute(sqlFileOut + comment, varMap)
                                        resFileOut = self.cur.fetchall()
                                        for tmpOutLFN, tmpOutDataset in resFileOut:
                                            esOutputFileMap[esPandaID][tmpOutDataset] = tmpOutLFN
                                # zip file in fileTable
                                if zipRow_ID is not None and zipRow_ID not in esZipRow_IDs:
                                    esZipRow_IDs.add(zipRow_ID)
                                    varMap = {}
                                    varMap[":row_ID"] = zipRow_ID
                                    self.cur.execute(sqlZipFile + comment, varMap)
                                    resZip = self.cur.fetchone()
                                    if resZip is not None:
                                        (
                                            outputZipName,
                                            outputZipBucketID,
                                            outputZipFsize,
                                            outputZipChecksum,
                                        ) = resZip
                                        if esPandaID not in esOutputZipMap:
                                            esOutputZipMap[esPandaID] = []
                                        esOutputZipMap[esPandaID].append(
                                            {
                                                "name": outputZipName,
                                                "osid": outputZipBucketID,
                                                "fsize": outputZipFsize,
                                                "checksum": outputZipChecksum,
                                            }
                                        )
                # make input for event service output merging
                mergeInputOutputMap = {}
                mergeInputFiles = []
                mergeFileObjStoreMap = {}
                mergeZipPandaIDs = []
                for tmpFileID in eventRangeIDs:
                    tmpMapEventRangeID = eventRangeIDs[tmpFileID]
                    jobProcessIDs = sorted(tmpMapEventRangeID)
                    # make input
                    for jobProcessID in jobProcessIDs:
                        for tmpFileSpec in job.Files:
                            if tmpFileSpec.type not in ["output"]:
                                continue
                            esPandaID = tmpMapEventRangeID[jobProcessID]["pandaID"]
                            tmpInputFileSpec = copy.copy(tmpFileSpec)
                            tmpInputFileSpec.type = "input"
                            outLFN = tmpInputFileSpec.lfn
                            # change LFN
                            if esPandaID in esOutputFileMap and tmpInputFileSpec.dataset in esOutputFileMap[esPandaID]:
                                tmpInputFileSpec.lfn = esOutputFileMap[esPandaID][tmpInputFileSpec.dataset]
                            # change attemptNr back to the original, which could have been changed by ES merge retry
                            if not useNewFileFormatForES:
                                origLFN = re.sub("\.\d+$", ".1", tmpInputFileSpec.lfn)
                                outLFN = re.sub("\.\d+$", ".1", outLFN)
                            else:
                                origLFN = re.sub("\.\d+$", ".1_000", tmpInputFileSpec.lfn)
                                outLFN = re.sub("\.\d+$", ".1_000", outLFN)
                            # append eventRangeID as suffix
                            tmpInputFileSpec.lfn = origLFN + "." + tmpMapEventRangeID[jobProcessID]["eventRangeID"]
                            # make input/output map
                            if outLFN not in mergeInputOutputMap:
                                mergeInputOutputMap[outLFN] = []
                            mergeInputOutputMap[outLFN].append(tmpInputFileSpec.lfn)
                            # add file
                            if esPandaID not in esOutputZipMap:
                                # no zip
                                mergeInputFiles.append(tmpInputFileSpec)
                                # mapping for ObjStore
                                mergeFileObjStoreMap[tmpInputFileSpec.lfn] = tmpMapEventRangeID[jobProcessID]["objStoreID"]
                            elif esPandaID not in mergeZipPandaIDs:
                                # zip
                                mergeZipPandaIDs.append(esPandaID)
                                for tmpEsOutZipFile in esOutputZipMap[esPandaID]:
                                    # copy for zip
                                    tmpZipInputFileSpec = copy.copy(tmpInputFileSpec)
                                    # add prefix
                                    tmpZipInputFileSpec.lfn = "zip://" + tmpEsOutZipFile["name"]
                                    if "fsize" in tmpEsOutZipFile:
                                        tmpZipInputFileSpec.fsize = tmpEsOutZipFile["fsize"]
                                    if "checksum" in tmpEsOutZipFile:
                                        tmpZipInputFileSpec.checksum = tmpEsOutZipFile["checksum"]
                                    mergeInputFiles.append(tmpZipInputFileSpec)
                                    # mapping for ObjStore
                                    mergeFileObjStoreMap[tmpZipInputFileSpec.lfn] = tmpEsOutZipFile["osid"]
                for tmpInputFileSpec in mergeInputFiles:
                    job.addFile(tmpInputFileSpec)

                # job parameters
                sqlJobP = "SELECT jobParameters FROM ATLAS_PANDA.jobParamsTable WHERE PandaID=:PandaID"
                varMap = {}
                varMap[":PandaID"] = job.PandaID
                self.cur.execute(sqlJobP + comment, varMap)
                for (clobJobP,) in self.cur:
                    try:
                        job.jobParameters = clobJobP.read()
                    except AttributeError:
                        job.jobParameters = str(clobJobP)
                    break

                # remove or extract parameters for merge
                if EventServiceUtils.isEventServiceJob(job) or EventServiceUtils.isJumboJob(job) or EventServiceUtils.isCoJumboJob(job):
                    try:
                        job.jobParameters = re.sub(
                            "<PANDA_ESMERGE_.+>.*</PANDA_ESMERGE_.+>",
                            "",
                            job.jobParameters,
                        )
                    except Exception:
                        pass
                    # sort files since file order is important for positional event number
                    job.sortFiles()
                elif EventServiceUtils.isEventServiceMerge(job):
                    try:
                        origJobParameters = job.jobParameters
                        tmpMatch = re.search(
                            "<PANDA_ESMERGE_JOBP>(.*)</PANDA_ESMERGE_JOBP>",
                            origJobParameters,
                        )
                        job.jobParameters = tmpMatch.group(1)
                        tmpMatch = re.search(
                            "<PANDA_ESMERGE_TRF>(.*)</PANDA_ESMERGE_TRF>",
                            origJobParameters,
                        )
                        job.transformation = tmpMatch.group(1)
                    except Exception:
                        pass
                    # pass in/out map for merging via metadata
                    job.metadata = [mergeInputOutputMap, mergeFileObjStoreMap]

                # read task parameters
                if job.lockedby == "jedi":
                    sqlTP = f"SELECT ioIntensity,ioIntensityUnit FROM {panda_config.schemaJEDI}.JEDI_Tasks WHERE jediTaskID=:jediTaskID "
                    varMap = {}
                    varMap[":jediTaskID"] = job.jediTaskID
                    self.cur.execute(sqlTP + comment, varMap)
                    resTP = self.cur.fetchone()
                    if resTP is not None:
                        ioIntensity, ioIntensityUnit = resTP
                        job.set_task_attribute("ioIntensity", ioIntensity)
                        job.set_task_attribute("ioIntensityUnit", ioIntensityUnit)

                if not self._commit():
                    raise RuntimeError("Commit error")

                # append the job to the returned list
                retJobs.append(job)

                # record status change
                try:
                    self.recordStatusChange(job.PandaID, job.jobStatus, jobInfo=job)
                except Exception:
                    tmpLog.error("recordStatusChange in getJobs")
                self.push_job_status_message(job, job.PandaID, job.jobStatus)
                if via_topic and job.is_push_job():
                    tmpLog.debug("delete job message")
                    mb_proxy_queue = self.get_mb_proxy("panda_pilot_queue")
                    srv_msg_utils.delete_job_message(mb_proxy_queue, job.PandaID)
            return retJobs, nSent
        except Exception as e:
            errStr = f"getJobs : {str(e)} {traceback.format_exc()}"
            if tmpLog:
                tmpLog.error(errStr)
            else:
                _logger.error(errStr)
            # roll back
            self._rollback()
            return [], 0

    # reset job in jobsActive or jobsWaiting
    def resetJob(
        self,
        pandaID,
        activeTable=True,
        keepSite=False,
        getOldSubs=False,
        forPending=True,
    ):
        comment = " /* DBProxy.resetJob */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <PandaID={pandaID}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug(f"activeTable={activeTable}")
        # select table
        table = "ATLAS_PANDA.jobsWaiting4"
        if activeTable:
            table = "ATLAS_PANDA.jobsActive4"
        sql1 = f"SELECT {JobSpec.columnNames()} FROM {table} "
        sql1 += "WHERE PandaID=:PandaID"
        sql2 = f"DELETE FROM {table} "
        sql2 += "WHERE PandaID=:PandaID AND (jobStatus=:oldJobStatus1 OR jobStatus=:oldJobStatus2)"
        sql3 = f"INSERT INTO ATLAS_PANDA.jobsDefined4 ({JobSpec.columnNames()}) "
        sql3 += JobSpec.bindValuesExpression()
        try:
            # transaction causes Request ndbd time-out in ATLAS_PANDA.jobsActive4
            self.conn.begin()
            # select
            varMap = {}
            varMap[":PandaID"] = pandaID
            self.cur.arraysize = 10
            self.cur.execute(sql1 + comment, varMap)
            res = self.cur.fetchone()
            # not found
            if res is None:
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                # return
                return None
            # instantiate Job
            job = JobSpec()
            job.pack(res)
            # if already running
            if job.jobStatus != "waiting" and job.jobStatus != "activated" and (forPending and job.jobStatus != "pending"):
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                # return
                return None
            # do nothing for analysis jobs
            if job.prodSourceLabel in ["user", "panda"] and not forPending and job.jobStatus != "pending":
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                # return
                return None
            # delete
            varMap = {}
            varMap[":PandaID"] = pandaID
            if not forPending:
                varMap[":oldJobStatus1"] = "waiting"
            else:
                varMap[":oldJobStatus1"] = "pending"
            varMap[":oldJobStatus2"] = "activated"
            self.cur.execute(sql2 + comment, varMap)
            retD = self.cur.rowcount
            # delete failed
            tmpLog.debug(f"retD = {retD}")
            if retD != 1:
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                return None
            # delete from jobsDefined4 just in case
            varMap = {}
            varMap[":PandaID"] = pandaID
            sqlD = "DELETE FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID"
            self.cur.execute(sqlD + comment, varMap)
            # increase priority
            if job.jobStatus == "activated" and job.currentPriority < 100:
                job.currentPriority = 100
            # reset computing site and dispatchDBlocks
            job.jobStatus = "defined"
            if job.prodSourceLabel not in ["user", "panda"]:
                job.dispatchDBlock = None
                # erase old assignment
                if (not keepSite) and job.relocationFlag not in [1, 2]:
                    job.computingSite = None
                job.computingElement = None
            # host and time information
            job.modificationHost = self.hostname
            job.modificationTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            job.stateChangeTime = job.modificationTime
            # reset
            job.brokerageErrorDiag = None
            job.brokerageErrorCode = None
            # insert
            self.cur.execute(sql3 + comment, job.valuesMap())
            # job parameters
            sqlJobP = "SELECT jobParameters FROM ATLAS_PANDA.jobParamsTable WHERE PandaID=:PandaID"
            self.cur.execute(sqlJobP + comment, varMap)
            for (clobJobP,) in self.cur:
                try:
                    job.jobParameters = clobJobP.read()
                except AttributeError:
                    job.jobParameters = str(clobJobP)
                break
            # Files
            oldSubList = []
            sqlFile = f"SELECT {FileSpec.columnNames()} FROM ATLAS_PANDA.filesTable4 "
            sqlFile += "WHERE PandaID=:PandaID"
            self.cur.arraysize = 10000
            self.cur.execute(sqlFile + comment, varMap)
            resFs = self.cur.fetchall()
            for resF in resFs:
                file = FileSpec()
                file.pack(resF)
                # reset GUID to trigger LRC/LFC scanning
                if file.status == "missing":
                    file.GUID = None
                # collect old subs
                if job.prodSourceLabel in ["managed", "test"] and file.type in ["output", "log"] and re.search("_sub\d+$", file.destinationDBlock) is not None:
                    if file.destinationDBlock not in oldSubList:
                        oldSubList.append(file.destinationDBlock)
                # reset status, destinationDBlock and dispatchDBlock
                if job.lockedby != "jedi":
                    file.status = "unknown"
                if job.prodSourceLabel not in ["user", "panda"]:
                    file.dispatchDBlock = None
                file.destinationDBlock = re.sub("_sub\d+$", "", file.destinationDBlock)
                # add file
                job.addFile(file)
                # update files
                sqlF = f"UPDATE ATLAS_PANDA.filesTable4 SET {file.bindUpdateChangesExpression()}" + "WHERE row_ID=:row_ID"
                varMap = file.valuesMap(onlyChanged=True)
                if varMap != {}:
                    varMap[":row_ID"] = file.row_ID
                    tmpLog.debug(sqlF + comment + str(varMap))
                    self.cur.execute(sqlF + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # record status change
            try:
                self.recordStatusChange(job.PandaID, job.jobStatus, jobInfo=job)
            except Exception:
                _logger.error("recordStatusChange in resetJobs")
            self.push_job_status_message(job, job.PandaID, job.jobStatus)
            if getOldSubs:
                return job, oldSubList
            tmpLog.debug("done")
            return job
        except Exception:
            # roll back
            self._rollback()
            # error report
            type, value, traceBack = sys.exc_info()
            tmpLog.error(f"{type} {value}")
            return None

    # reset jobs in jobsDefined
    def resetDefinedJob(self, pandaID, keepSite=False, getOldSubs=False):
        comment = " /* DBProxy.resetDefinedJob */"
        _logger.debug(f"resetDefinedJob : {pandaID}")
        sql1 = "UPDATE ATLAS_PANDA.jobsDefined4 SET "
        sql1 += "jobStatus=:newJobStatus,"
        sql1 += "modificationTime=CURRENT_DATE,"
        sql1 += "dispatchDBlock=NULL,"
        sql1 += "computingElement=NULL"
        sql1 += " WHERE PandaID=:PandaID AND (jobStatus=:oldJobStatus1 OR jobStatus=:oldJobStatus2)"
        sql2 = f"SELECT {JobSpec.columnNames()} FROM ATLAS_PANDA.jobsDefined4 "
        sql2 += "WHERE PandaID=:PandaID"
        try:
            oldSubList = []
            # begin transaction
            self.conn.begin()
            # update
            varMap = {}
            varMap[":PandaID"] = pandaID
            varMap[":newJobStatus"] = "defined"
            varMap[":oldJobStatus1"] = "assigned"
            varMap[":oldJobStatus2"] = "defined"
            self.cur.execute(sql1 + comment, varMap)
            retU = self.cur.rowcount
            # not found
            updatedFlag = False
            job = None
            if retU == 0:
                _logger.debug(f"resetDefinedJob : Not found {pandaID}")
            else:
                # select
                varMap = {}
                varMap[":PandaID"] = pandaID
                self.cur.arraysize = 10
                self.cur.execute(sql2 + comment, varMap)
                res = self.cur.fetchone()
                # not found
                if res is None:
                    raise RuntimeError(f"Could not SELECT : PandaID={pandaID}")
                # instantiate Job
                job = JobSpec()
                job.pack(res)
                # do nothing for analysis jobs
                if job.prodSourceLabel in ["user", "panda"]:
                    _logger.debug(f"resetDefinedJob : rollback since PandaID={pandaID} is analysis job")
                    # roll back
                    self._rollback()
                    return None
                job.dispatchDBlock = None
                if (not keepSite) and job.relocationFlag not in [1, 2]:
                    # erase old assignment
                    job.computingSite = None
                job.computingElement = None
                # job parameters
                sqlJobP = "SELECT jobParameters FROM ATLAS_PANDA.jobParamsTable WHERE PandaID=:PandaID"
                self.cur.execute(sqlJobP + comment, varMap)
                for (clobJobP,) in self.cur:
                    try:
                        job.jobParameters = clobJobP.read()
                    except AttributeError:
                        job.jobParameters = str(clobJobP)
                    break
                # Files
                sqlFile = f"SELECT {FileSpec.columnNames()} FROM ATLAS_PANDA.filesTable4 "
                sqlFile += "WHERE PandaID=:PandaID"
                self.cur.arraysize = 10000
                self.cur.execute(sqlFile + comment, varMap)
                resFs = self.cur.fetchall()
                for resF in resFs:
                    file = FileSpec()
                    file.pack(resF)
                    # collect old subs
                    if (
                        job.prodSourceLabel in ["managed", "test"]
                        and file.type in ["output", "log"]
                        and re.search("_sub\d+$", file.destinationDBlock) is not None
                    ):
                        if file.destinationDBlock not in oldSubList:
                            oldSubList.append(file.destinationDBlock)
                    # reset status, destinationDBlock and dispatchDBlock
                    if job.lockedby != "jedi":
                        file.status = "unknown"
                    file.dispatchDBlock = None
                    file.destinationDBlock = re.sub("_sub\d+$", "", file.destinationDBlock)
                    # add file
                    job.addFile(file)
                    # update files
                    sqlF = f"UPDATE ATLAS_PANDA.filesTable4 SET {file.bindUpdateChangesExpression()}" + "WHERE row_ID=:row_ID"
                    varMap = file.valuesMap(onlyChanged=True)
                    if varMap != {}:
                        varMap[":row_ID"] = file.row_ID
                        _logger.debug(sqlF + comment + str(varMap))
                        self.cur.execute(sqlF + comment, varMap)
                updatedFlag = True
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # record status change
            try:
                if updatedFlag:
                    self.recordStatusChange(job.PandaID, job.jobStatus, jobInfo=job)
                    self.push_job_status_message(job, job.PandaID, job.jobStatus)
            except Exception:
                _logger.error("recordStatusChange in resetDefinedJobs")
            if getOldSubs:
                return job, oldSubList
            return job
        except Exception:
            # error report
            type, value, traceBack = sys.exc_info()
            _logger.error(f"resetDefinedJobs : {type} {value}")
            # roll back
            self._rollback()
            return None

    # kill job
    def killJob(
        self,
        pandaID,
        user,
        code,
        prodManager,
        getUserInfo=False,
        wgProdRole=[],
        killOpts=[],
    ):
        # code
        # 2  : expire
        # 3  : aborted
        # 4  : expire in waiting
        # 7  : retry by server
        # 8  : rebrokerage
        # 9  : force kill
        # 10 : fast rebrokerage in overloaded PQ
        # 50 : kill by JEDI
        # 51 : reassigned by JEDI
        # 52 : force kill by JEDI
        # 55 : killed since task is (almost) done
        # 60 : workload was terminated by the pilot without actual work
        # 91 : kill user jobs with prod role
        # 99 : force kill user jobs with prod role
        comment = " /* DBProxy.killJob */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <PandaID={pandaID}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug(f"code={code} role={prodManager} user={user} wg={wgProdRole} opts={killOpts}")
        timeStart = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
        # check PandaID
        try:
            int(pandaID)
        except Exception:
            tmpLog.error(f"not an integer : {pandaID}")
            if getUserInfo:
                return False, {}
            return False
        sql0 = "SELECT prodUserID,prodSourceLabel,jobDefinitionID,jobsetID,workingGroup,specialHandling,jobStatus,taskBufferErrorCode,eventService FROM %s "
        sql0 += "WHERE PandaID=:PandaID "
        sql0 += "FOR UPDATE NOWAIT "
        sql1 = "UPDATE %s SET commandToPilot=:commandToPilot,taskBufferErrorDiag=:taskBufferErrorDiag WHERE PandaID=:PandaID "
        sql1 += "AND (commandToPilot IS NULL OR commandToPilot<>'tobekilled') "
        sql1F = "UPDATE %s SET commandToPilot=:commandToPilot,taskBufferErrorDiag=:taskBufferErrorDiag WHERE PandaID=:PandaID "
        sql2 = f"SELECT {JobSpec.columnNames()} "
        sql2 += "FROM %s WHERE PandaID=:PandaID AND jobStatus<>:jobStatus"
        sql3 = "DELETE FROM %s WHERE PandaID=:PandaID"
        sqlU = "DELETE FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID AND (jobStatus=:oldJobStatus1 OR jobStatus=:oldJobStatus2)"
        sql4 = f"INSERT INTO ATLAS_PANDA.jobsArchived4 ({JobSpec.columnNames()}) "
        sql4 += JobSpec.bindValuesExpression()
        sqlF = "UPDATE ATLAS_PANDA.filesTable4 SET status=:status WHERE PandaID=:PandaID AND type IN (:type1,:type2)"
        sqlFMod = "UPDATE ATLAS_PANDA.filesTable4 SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
        sqlMMod = "UPDATE ATLAS_PANDA.metaTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
        sqlPMod = "UPDATE ATLAS_PANDA.jobParamsTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
        sqlFile = f"SELECT {FileSpec.columnNames()} FROM ATLAS_PANDA.filesTable4 "
        sqlFile += "WHERE PandaID=:PandaID"
        try:
            flagCommand = False
            flagKilled = False
            userProdUserID = ""
            userProdSourceLabel = ""
            userJobDefinitionID = ""
            userJobsetID = ""
            updatedFlag = False
            # begin transaction
            self.conn.begin()
            for table in (
                "ATLAS_PANDA.jobsDefined4",
                "ATLAS_PANDA.jobsActive4",
                "ATLAS_PANDA.jobsWaiting4",
            ):
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                # begin transaction
                self.conn.begin()
                # get DN if user is not production DN
                varMap = {}
                varMap[":PandaID"] = pandaID
                self.cur.arraysize = 10
                self.cur.execute((sql0 + comment) % table, varMap)
                res = self.cur.fetchone()
                # not found
                if res is None:
                    continue

                # owner?
                def getCN(dn):
                    distinguishedName = ""
                    for line in dn.split("/"):
                        if line.startswith("CN="):
                            distinguishedName = re.sub("^CN=", "", line)
                            distinguishedName = re.sub("\d+$", "", distinguishedName)
                            distinguishedName = distinguishedName.strip()
                            break
                    if distinguishedName == "":
                        distinguishedName = dn
                    return distinguishedName

                # prevent prod proxy from killing analysis jobs
                (
                    userProdUserID,
                    userProdSourceLabel,
                    userJobDefinitionID,
                    userJobsetID,
                    workingGroup,
                    specialHandling,
                    jobStatusInDB,
                    taskBufferErrorCode,
                    eventService,
                ) = res
                # check group prod role
                validGroupProdRole = False
                if res[1] in ["managed", "test"] and workingGroup != "":
                    for tmpGroupProdRole in wgProdRole:
                        if tmpGroupProdRole == "":
                            continue
                        if re.search("(^|_)" + tmpGroupProdRole + "$", workingGroup, re.I) is not None:
                            validGroupProdRole = True
                            break
                if prodManager:
                    if res[1] in ["user", "panda"] and (
                        code
                        not in [
                            "2",
                            "4",
                            "7",
                            "8",
                            "9",
                            "50",
                            "51",
                            "52",
                            "91",
                            "10",
                            "99",
                        ]
                    ):
                        tmpLog.debug(f"ignored -> prod proxy tried to kill analysis job type={res[1]}")
                        break
                    tmpLog.debug("using prod role")
                elif validGroupProdRole:
                    # WGs with prod role
                    tmpLog.debug(f"using group prod role for workingGroup={workingGroup}")
                    pass
                else:
                    cn1 = getCN(res[0])
                    cn2 = getCN(user)
                    tmpLog.debug(f"Owner:{cn1}  - Requester:{cn2} ")
                    if cn1 != cn2:
                        tmpLog.debug("ignored  -> Owner != Requester")
                        break
                # event service
                useEventService = EventServiceUtils.isEventServiceSH(specialHandling) or eventService in [
                    EventServiceUtils.jumboJobFlagNumber,
                    EventServiceUtils.coJumboJobFlagNumber,
                ]
                useEventServiceMerge = EventServiceUtils.isEventServiceMergeSH(specialHandling)
                # update
                varMap = {}
                varMap[":PandaID"] = pandaID
                varMap[":commandToPilot"] = "tobekilled"
                varMap[":taskBufferErrorDiag"] = f"killed by {user}"
                if code in ["2", "9", "10", "52", "51", "60", "99"]:
                    # ignore commandToPilot for force kill
                    self.cur.execute((sql1F + comment) % table, varMap)
                elif useEventService or jobStatusInDB in ["merging"]:
                    # use force kill for event service or merging
                    self.cur.execute((sql1F + comment) % table, varMap)
                else:
                    self.cur.execute((sql1 + comment) % table, varMap)
                retU = self.cur.rowcount
                if retU == 0:
                    continue
                # set flag
                flagCommand = True
                # select
                varMap = {}
                varMap[":PandaID"] = pandaID
                if ((userProdSourceLabel in ["managed", "test", None] or "test" in userProdSourceLabel) and code in ["9", "52"]) or (
                    prodManager and code == "99"
                ):
                    # use dummy for force kill
                    varMap[":jobStatus"] = "dummy"
                elif (useEventService and not EventServiceUtils.isJobCloningSH(specialHandling)) or jobStatusInDB in ["merging"]:
                    # use dummy for force kill
                    varMap[":jobStatus"] = "dummy"
                else:
                    varMap[":jobStatus"] = "running"
                self.cur.arraysize = 10
                self.cur.execute((sql2 + comment) % table, varMap)
                res = self.cur.fetchall()
                if len(res) == 0:
                    continue
                # instantiate JobSpec
                job = JobSpec()
                job.pack(res[0])
                # delete
                if table == "ATLAS_PANDA.jobsDefined4":
                    varMap = {}
                    varMap[":PandaID"] = pandaID
                    varMap[":oldJobStatus1"] = "assigned"
                    varMap[":oldJobStatus2"] = "defined"
                    self.cur.execute(sqlU + comment, varMap)
                else:
                    varMap = {}
                    varMap[":PandaID"] = pandaID
                    self.cur.execute((sql3 + comment) % table, varMap)
                retD = self.cur.rowcount
                if retD == 0:
                    continue
                oldJobStatus = job.jobStatus
                # error code
                if job.jobStatus != "failed":
                    currentTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
                    # set status etc for non-failed jobs
                    if job.endTime in [None, "NULL"]:
                        job.endTime = currentTime
                    # reset startTime for aCT where starting jobs don't acutally get started
                    if job.jobStatus == "starting":
                        job.startTime = job.endTime
                    job.modificationTime = currentTime
                    if code in ["2", "4"]:
                        # expire
                        job.jobStatus = "closed"
                        job.jobSubStatus = "toreassign"
                        job.taskBufferErrorCode = ErrorCode.EC_Expire
                        job.taskBufferErrorDiag = f"expired in {oldJobStatus}. status unchanged since {str(job.stateChangeTime)}"
                    elif code == "3":
                        # aborted
                        job.taskBufferErrorCode = ErrorCode.EC_Aborted
                        job.taskBufferErrorDiag = "aborted by ExtIF"
                    elif code == "8":
                        # reassigned by rebrokeage
                        job.taskBufferErrorCode = ErrorCode.EC_Reassigned
                        job.taskBufferErrorDiag = f"reassigned to another site by rebrokerage. new {user}"
                        job.commandToPilot = None
                    elif code in ["50", "52"]:
                        # killed by JEDI
                        job.taskBufferErrorCode = ErrorCode.EC_Kill
                        job.taskBufferErrorDiag = user
                    elif code == "51":
                        # reassigned by JEDI
                        job.jobStatus = "closed"
                        job.jobSubStatus = "toreassign"
                        job.taskBufferErrorCode = ErrorCode.EC_Kill
                        job.taskBufferErrorDiag = "reassigned by JEDI"
                    elif code == "55":
                        # killed since task is (almost) done
                        job.jobStatus = "closed"
                        job.jobSubStatus = "taskdone"
                        job.taskBufferErrorCode = ErrorCode.EC_Kill
                        job.taskBufferErrorDiag = "killed since task is (almost) done"
                    elif code == "60":
                        # terminated by the pilot. keep jobSubStatus reported by the pilot
                        job.jobStatus = "closed"
                        job.taskBufferErrorCode = ErrorCode.EC_Kill
                        job.taskBufferErrorDiag = "closed by the pilot"
                    elif code == "10":
                        job.jobStatus = "closed"
                        job.taskBufferErrorCode = ErrorCode.EC_FastRebrokerage
                        job.taskBufferErrorDiag = "fast rebrokerage due to Nq/Nr overshoot"
                    else:
                        # killed
                        job.taskBufferErrorCode = ErrorCode.EC_Kill
                        job.taskBufferErrorDiag = f"killed by {user}"
                    # set job status
                    if job.jobStatus != "closed":
                        job.jobStatus = "cancelled"
                else:
                    # keep status for failed jobs
                    job.modificationTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
                    if code == "7":
                        # retried by server
                        job.taskBufferErrorCode = ErrorCode.EC_Retried
                        job.taskBufferErrorDiag = f"retrying at another site. new {user}"
                        job.commandToPilot = None
                job.stateChangeTime = job.modificationTime
                # insert
                self.cur.execute(sql4 + comment, job.valuesMap())
                # update file
                varMap = {}
                varMap[":PandaID"] = pandaID
                varMap[":status"] = "failed"
                varMap[":type1"] = "output"
                varMap[":type2"] = "log"
                self.cur.execute(sqlF + comment, varMap)
                # update files,metadata,parametes
                varMap = {}
                varMap[":PandaID"] = pandaID
                varMap[":modificationTime"] = job.modificationTime
                self.cur.execute(sqlFMod + comment, varMap)
                self.cur.execute(sqlMMod + comment, varMap)
                self.cur.execute(sqlPMod + comment, varMap)
                flagKilled = True
                updatedFlag = True
                # update JEDI tables
                if (
                    hasattr(panda_config, "useJEDI")
                    and panda_config.useJEDI is True
                    and job.lockedby == "jedi"
                    and self.checkTaskStatusJEDI(job.jediTaskID, self.cur)
                ):
                    # read files
                    varMap = {}
                    varMap[":PandaID"] = pandaID
                    self.cur.arraysize = 10000
                    self.cur.execute(sqlFile + comment, varMap)
                    resFs = self.cur.fetchall()
                    for resF in resFs:
                        fileSpec = FileSpec()
                        fileSpec.pack(resF)
                        job.addFile(fileSpec)
                    # actions for event service unless it was already retried
                    if taskBufferErrorCode not in [
                        ErrorCode.EC_Reassigned,
                        ErrorCode.EC_Retried,
                        ErrorCode.EC_PilotRetried,
                    ]:
                        # kill associated consumers for event service
                        if useEventService:
                            self.killEventServiceConsumers(job, True, False)
                            if job.computingSite != EventServiceUtils.siteIdForWaitingCoJumboJobs:
                                self.killUnusedEventServiceConsumers(job, False, killAll=True, checkAttemptNr=True)
                            self.updateRelatedEventServiceJobs(job, True)
                            if not job.notDiscardEvents():
                                self.killUnusedEventRanges(job.jediTaskID, job.jobsetID)
                            if eventService == EventServiceUtils.jumboJobFlagNumber:
                                self.hasDoneEvents(job.jediTaskID, job.PandaID, job, False)
                        elif useEventServiceMerge:
                            self.updateRelatedEventServiceJobs(job, True)
                    # disable reattempt
                    if job.processingType == "pmerge" and "keepUnmerged" not in killOpts and code != "51":
                        self.disableFurtherReattempt(job)
                    # update JEDI
                    self.propagateResultToJEDI(job, self.cur, oldJobStatus)
                break
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            timeDelta = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - timeStart
            tmpLog.debug(f"com={flagCommand} kill={flagKilled} time={timeDelta.seconds}")
            # record status change
            try:
                if updatedFlag:
                    self.recordStatusChange(job.PandaID, job.jobStatus, jobInfo=job)
                    self.push_job_status_message(job, job.PandaID, job.jobStatus)
            except Exception:
                tmpLog.error("recordStatusChange in killJob")
            if getUserInfo:
                return (flagCommand or flagKilled), {
                    "prodUserID": userProdUserID,
                    "prodSourceLabel": userProdSourceLabel,
                    "jobDefinitionID": userJobDefinitionID,
                    "jobsetID": userJobsetID,
                }
            return flagCommand or flagKilled
        except Exception:
            self.dumpErrorMessage(_logger, methodName)
            # roll back
            self._rollback()
            timeDelta = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - timeStart
            tmpLog.debug(f"time={timeDelta.seconds}")
            if getUserInfo:
                return False, {}
            return False

    # peek at job
    def peekJob(self, pandaID, fromDefined, fromActive, fromArchived, fromWaiting, forAnal=False):
        comment = " /* DBProxy.peekJob */"
        _logger.debug(f"peekJob : {pandaID}")
        # return None for NULL PandaID
        if pandaID in ["NULL", "", "None", None]:
            return None
        # only int
        try:
            tmpID = int(pandaID)
        except Exception:
            _logger.debug(f"peekJob : return None for {pandaID}:non-integer")
            return None
        sql1_0 = "SELECT %s FROM %s "
        sql1_1 = "WHERE PandaID=:PandaID"
        nTry = 3
        for iTry in range(nTry):
            try:
                tables = []
                if fromDefined:
                    tables.append("ATLAS_PANDA.jobsDefined4")
                if fromActive:
                    tables.append("ATLAS_PANDA.jobsActive4")
                if fromArchived:
                    tables.append("ATLAS_PANDA.jobsArchived4")
                if fromWaiting:
                    tables.append("ATLAS_PANDA.jobsWaiting4")
                if fromDefined:
                    # for jobs which are just reset
                    tables.append("ATLAS_PANDA.jobsDefined4")
                # select
                varMap = {}
                varMap[":PandaID"] = pandaID
                for table in tables:
                    # start transaction
                    self.conn.begin()
                    # select
                    sql = sql1_0 % (JobSpec.columnNames(), table) + sql1_1
                    self.cur.arraysize = 10
                    self.cur.execute(sql + comment, varMap)
                    res = self.cur.fetchall()
                    # commit
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    if len(res) != 0:
                        # Job
                        job = JobSpec()
                        job.pack(res[0])
                        # Files
                        # start transaction
                        self.conn.begin()
                        # select
                        sqlFile = f"SELECT {FileSpec.columnNames()} FROM ATLAS_PANDA.filesTable4 "
                        sqlFile += "WHERE PandaID=:PandaID"
                        self.cur.arraysize = 10000
                        self.cur.execute(sqlFile + comment, varMap)
                        resFs = self.cur.fetchall()
                        # metadata
                        resMeta = None
                        if table == "ATLAS_PANDA.jobsArchived4" or forAnal:
                            # read metadata only for finished/failed production jobs
                            sqlMeta = "SELECT metaData FROM ATLAS_PANDA.metaTable WHERE PandaID=:PandaID"
                            self.cur.execute(sqlMeta + comment, varMap)
                            for (clobMeta,) in self.cur:
                                if clobMeta is not None:
                                    try:
                                        resMeta = clobMeta.read()
                                    except AttributeError:
                                        resMeta = str(clobMeta)
                                break
                        # job parameters
                        job.jobParameters = None
                        sqlJobP = "SELECT jobParameters FROM ATLAS_PANDA.jobParamsTable WHERE PandaID=:PandaID"
                        varMap = {}
                        varMap[":PandaID"] = job.PandaID
                        self.cur.execute(sqlJobP + comment, varMap)
                        for (clobJobP,) in self.cur:
                            if clobJobP is not None:
                                try:
                                    job.jobParameters = clobJobP.read()
                                except AttributeError:
                                    job.jobParameters = str(clobJobP)
                            break
                        # commit
                        if not self._commit():
                            raise RuntimeError("Commit error")
                        # set files
                        for resF in resFs:
                            file = FileSpec()
                            file.pack(resF)
                            job.addFile(file)
                        # set metadata
                        job.metadata = resMeta
                        return job
                _logger.debug(f"peekJob() : PandaID {pandaID} not found")
                return None
            except Exception:
                # roll back
                self._rollback()
                if iTry + 1 < nTry:
                    _logger.debug(f"peekJob : {pandaID} retry : {iTry}")
                    time.sleep(random.randint(10, 20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error(f"peekJob : {pandaID} {type} {value}")
                # return None for analysis
                if forAnal:
                    return None
                # return 'unknown'
                job = JobSpec()
                job.PandaID = pandaID
                job.jobStatus = "unknown"
                return job

    # get PandaIDs with TaskID
    def getPandaIDsWithTaskID(self, jediTaskID):
        comment = " /* DBProxy.getPandaIDsWithTaskID */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" <jediTaskID={jediTaskID}>")
        tmpLog.debug("start")
        # SQL
        sql = "SELECT PandaID FROM ATLAS_PANDA.jobsWaiting4 "
        sql += "WHERE jediTaskID=:jediTaskID "
        sql += "UNION "
        sql += "SELECT PandaID FROM ATLAS_PANDA.jobsDefined4 "
        sql += "WHERE jediTaskID=:jediTaskID "
        sql += "UNION "
        sql += "SELECT PandaID FROM ATLAS_PANDA.jobsActive4 "
        sql += "WHERE jediTaskID=:jediTaskID "
        sql += "UNION "
        sql += "SELECT PandaID FROM ATLAS_PANDA.jobsArchived4 "
        sql += "WHERE jediTaskID=:jediTaskID "
        varMap = {}
        varMap[":jediTaskID"] = jediTaskID
        try:
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 1000000
            self.cur.execute(sql + comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            retList = []
            for (pandaID,) in res:
                retList.append(pandaID)

            tmpLog.debug(f"found {len(retList)} IDs")
            return retList
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return []

    # get express jobs
    def getExpressJobs(self, dn):
        comment = " /* DBProxy.getExpressJobs */"
        _logger.debug(f"getExpressJobs : {dn}")
        sqlX = "SELECT specialHandling,COUNT(*) FROM %s "
        sqlX += "WHERE prodUserName=:prodUserName AND prodSourceLabel=:prodSourceLabel1 "
        sqlX += "AND specialHandling IS NOT NULL "
        sqlXJob = "SELECT PandaID,jobStatus,prodSourceLabel,modificationTime,jobDefinitionID,jobsetID,startTime,endTime FROM %s "
        sqlXJob += "WHERE prodUserName=:prodUserName AND prodSourceLabel=:prodSourceLabel1 "
        sqlXJob += "AND specialHandling IS NOT NULL AND specialHandling=:specialHandling "
        sqlQ = sqlX
        sqlQ += "GROUP BY specialHandling "
        sqlQJob = sqlXJob
        sqlA = sqlX
        sqlA += "AND modificationTime>:modificationTime GROUP BY specialHandling "
        sqlAJob = sqlXJob
        sqlAJob += "AND modificationTime>:modificationTime "
        try:
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ["", "NULL", None]:
                compactDN = dn
            expressStr = "express"
            activeExpressU = []
            timeUsageU = datetime.timedelta(0)
            executionTimeU = datetime.timedelta(hours=1)
            jobCreditU = 3
            timeCreditU = executionTimeU * jobCreditU
            timeNow = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            timeLimit = timeNow - datetime.timedelta(hours=6)
            # loop over tables
            for table in [
                "ATLAS_PANDA.jobsDefined4",
                "ATLAS_PANDA.jobsActive4",
                "ATLAS_PANDA.jobsArchived4",
            ]:
                varMap = {}
                varMap[":prodUserName"] = compactDN
                varMap[":prodSourceLabel1"] = "user"
                if table == "ATLAS_PANDA.jobsArchived4":
                    varMap[":modificationTime"] = timeLimit
                    sql = sqlA % table
                    sqlJob = sqlAJob % table
                else:
                    sql = sqlQ % table
                    sqlJob = sqlQJob % table
                # start transaction
                self.conn.begin()
                # get the number of jobs for each specialHandling
                self.cur.arraysize = 10
                _logger.debug(sql + comment + str(varMap))
                self.cur.execute(sql + comment, varMap)
                res = self.cur.fetchall()
                _logger.debug(f"getExpressJobs {str(res)}")
                for specialHandling, countJobs in res:
                    if specialHandling is None:
                        continue
                    # look for express jobs
                    if expressStr in specialHandling:
                        varMap[":specialHandling"] = specialHandling
                        self.cur.arraysize = 1000
                        self.cur.execute(sqlJob + comment, varMap)
                        resJobs = self.cur.fetchall()
                        _logger.debug(f"getExpressJobs {str(resJobs)}")
                        for (
                            tmp_PandaID,
                            tmp_jobStatus,
                            tmp_prodSourceLabel,
                            tmp_modificationTime,
                            tmp_jobDefinitionID,
                            tmp_jobsetID,
                            tmp_startTime,
                            tmp_endTime,
                        ) in resJobs:
                            # collect active jobs
                            if tmp_jobStatus not in [
                                "finished",
                                "failed",
                                "cancelled",
                                "closed",
                            ]:
                                activeExpressU.append((tmp_PandaID, tmp_jobsetID, tmp_jobDefinitionID))
                            # get time usage
                            if tmp_jobStatus not in ["defined", "activated"]:
                                # check only jobs which actually use or used CPU on WN
                                if tmp_startTime is not None:
                                    # running or not
                                    if tmp_endTime is None:
                                        # job got started before/after the time limit
                                        if timeLimit > tmp_startTime:
                                            timeDelta = timeNow - timeLimit
                                        else:
                                            timeDelta = timeNow - tmp_startTime
                                    else:
                                        # job got started before/after the time limit
                                        if timeLimit > tmp_startTime:
                                            timeDelta = tmp_endTime - timeLimit
                                        else:
                                            timeDelta = tmp_endTime - tmp_startTime
                                    # add
                                    if timeDelta > datetime.timedelta(0):
                                        timeUsageU += timeDelta
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
            # check quota
            rRet = True
            rRetStr = ""
            rQuota = 0
            if len(activeExpressU) >= jobCreditU:
                rRetStr += f"The number of queued runXYZ exceeds the limit = {jobCreditU}. "
                rRet = False
            if timeUsageU >= timeCreditU:
                rRetStr += f"The total execution time for runXYZ exceeds the limit = {timeCreditU.seconds / 60} min. "
                rRet = False
            # calculate available quota
            if rRet:
                tmpQuota = jobCreditU - len(activeExpressU) - timeUsageU.seconds / executionTimeU.seconds
                if tmpQuota < 0:
                    rRetStr += "Quota for runXYZ exceeds. "
                    rRet = False
                else:
                    rQuota = tmpQuota
            # return
            retVal = {
                "status": rRet,
                "quota": rQuota,
                "output": rRetStr,
                "usage": timeUsageU,
                "jobs": activeExpressU,
            }
            _logger.debug(f"getExpressJobs : {str(retVal)}")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            errtype, errvalue = sys.exc_info()[:2]
            _logger.error(f"getExpressJobs : {errtype} {errvalue}")
            return None

    # get active debug jobs
    def getActiveDebugJobs(self, dn=None, workingGroup=None, prodRole=False):
        comment = " /* DBProxy.getActiveDebugJobs */"
        _logger.debug(f"getActiveDebugJobs : DN={dn} wg={workingGroup} prodRole={prodRole}")
        varMap = {}
        sqlX = "SELECT PandaID,jobStatus,specialHandling FROM %s "
        sqlX += "WHERE "
        if prodRole:
            pass
        elif workingGroup is not None:
            sqlX += "UPPER(workingGroup) IN (:wg1,:wg2) AND "
            varMap[":wg1"] = f"AP_{workingGroup.upper()}"
            varMap[":wg2"] = f"GP_{workingGroup.upper()}"
        else:
            sqlX += "prodUserName=:prodUserName AND "
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ["", "NULL", None]:
                compactDN = dn
            varMap[":prodUserName"] = compactDN
        sqlX += "specialHandling IS NOT NULL "
        try:
            debugStr = "debug"
            activeDebugJobs = []
            # loop over tables
            for table in ["ATLAS_PANDA.jobsDefined4", "ATLAS_PANDA.jobsActive4"]:
                sql = sqlX % table
                # start transaction
                self.conn.begin()
                # get jobs with specialHandling
                self.cur.arraysize = 100000
                self.cur.execute(sql + comment, varMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                # loop over all PandaIDs
                for pandaID, jobStatus, specialHandling in res:
                    if specialHandling is None:
                        continue
                    # only active jobs
                    if jobStatus not in [
                        "defined",
                        "activated",
                        "running",
                        "sent",
                        "starting",
                    ]:
                        continue
                    # look for debug jobs
                    if debugStr in specialHandling and pandaID not in activeDebugJobs:
                        activeDebugJobs.append(pandaID)
            # return
            activeDebugJobs.sort()
            _logger.debug(f"getActiveDebugJobs : DN={dn} -> {str(activeDebugJobs)}")
            return activeDebugJobs
        except Exception:
            # roll back
            self._rollback()
            errtype, errvalue = sys.exc_info()[:2]
            _logger.error(f"getActiveDebugJobs : {errtype} {errvalue}")
            return None

    # set debug mode
    def setDebugMode(self, dn, pandaID, prodManager, modeOn, workingGroup):
        comment = " /* DBProxy.setDebugMode */"
        _logger.debug(f"turnDebugModeOn : dn={dn} id={pandaID} prod={prodManager} wg={workingGroup} mode={modeOn}")
        sqlX = "SELECT prodUserName,jobStatus,specialHandling,workingGroup FROM %s "
        sqlX += "WHERE PandaID=:PandaID "
        sqlU = "UPDATE %s SET specialHandling=:specialHandling "
        sqlU += "WHERE PandaID=:PandaID "
        try:
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ["", "NULL", None]:
                compactDN = dn
            debugStr = "debug"
            retStr = ""
            retCode = False
            # loop over tables
            for table in ["ATLAS_PANDA.jobsDefined4", "ATLAS_PANDA.jobsActive4"]:
                varMap = {}
                varMap[":PandaID"] = pandaID
                sql = sqlX % table
                # start transaction
                self.conn.begin()
                # get jobs with specialHandling
                self.cur.arraysize = 10
                self.cur.execute(sql + comment, varMap)
                res = self.cur.fetchone()
                # not found
                if res is None:
                    retStr = f"PandaID={pandaID} not found in active DB"
                    # commit
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    continue
                prodUserName, jobStatus, specialHandling, wGroup = res
                # not active
                changeableState = [
                    "defined",
                    "activated",
                    "running",
                    "sent",
                    "starting",
                    "assigned",
                ]
                if jobStatus not in changeableState:
                    retStr = f"Cannot set debugMode since the job status is {jobStatus} which is not in one of {str(changeableState)}"
                    # commit
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    break
                # extract workingGroup
                try:
                    wGroup = wGroup.split("_")[-1]
                    wGroup = wGroup.lower()
                except Exception:
                    pass
                # not owner
                notOwner = False
                if not prodManager:
                    if workingGroup is not None:
                        if workingGroup.lower() != wGroup:
                            retStr = f"Permission denied. Not the production manager for workingGroup={wGroup}"
                            notOwner = True
                    else:
                        if prodUserName != compactDN:
                            retStr = "Permission denied. Not the owner or production manager"
                            notOwner = True
                    if notOwner:
                        # commit
                        if not self._commit():
                            raise RuntimeError("Commit error")
                        break
                # set specialHandling
                updateSH = True
                if specialHandling in [None, ""]:
                    if modeOn:
                        # set debug mode
                        specialHandling = debugStr
                    else:
                        # already disabled debug mode
                        updateSH = False
                elif debugStr in specialHandling:
                    if modeOn:
                        # already in debug mode
                        updateSH = False
                    else:
                        # disable debug mode
                        specialHandling = re.sub(debugStr, "", specialHandling)
                        specialHandling = re.sub(",,", ",", specialHandling)
                        specialHandling = re.sub("^,", "", specialHandling)
                        specialHandling = re.sub(",$", "", specialHandling)
                else:
                    if modeOn:
                        # set debug mode
                        specialHandling = debugStr
                    else:
                        # already disabled debug mode
                        updateSH = False

                # no update
                if not updateSH:
                    retStr = "Already set accordingly"
                    # commit
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    break
                # update
                varMap = {}
                varMap[":PandaID"] = pandaID
                varMap[":specialHandling"] = specialHandling
                self.cur.execute((sqlU + comment) % table, varMap)
                retD = self.cur.rowcount
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                if retD == 0:
                    retStr = "Failed to update DB"
                else:
                    retStr = "Succeeded"
                    break
            # return
            _logger.debug(f"setDebugMode : {dn} {pandaID} -> {retStr}")
            return retStr
        except Exception:
            # roll back
            self._rollback()
            errtype, errvalue = sys.exc_info()[:2]
            _logger.error(f"setDebugMode : {errtype} {errvalue}")
            return None

    # lock jobs for reassign
    def lockJobsForReassign(
        self,
        tableName,
        timeLimit,
        statList,
        labels,
        processTypes,
        sites,
        clouds,
        useJEDI=False,
        onlyReassignable=False,
        useStateChangeTime=False,
        getEventService=False,
    ):
        comment = " /* DBProxy.lockJobsForReassign */"
        _logger.debug(f"lockJobsForReassign : {tableName} {timeLimit} {statList} {labels} {processTypes} {sites} {clouds} {useJEDI}")
        try:
            # make sql
            if not useJEDI:
                sql = f"SELECT PandaID FROM {tableName} "
            elif getEventService:
                sql = f"SELECT PandaID,lockedby,eventService,attemptNr,computingSite FROM {tableName} "
            else:
                sql = f"SELECT PandaID,lockedby FROM {tableName} "
            if not useStateChangeTime:
                sql += "WHERE modificationTime<:modificationTime "
            else:
                sql += "WHERE stateChangeTime<:modificationTime "
            varMap = {}
            varMap[":modificationTime"] = timeLimit
            if statList != []:
                sql += "AND jobStatus IN ("
                tmpIdx = 0
                for tmpStat in statList:
                    tmpKey = f":stat{tmpIdx}"
                    varMap[tmpKey] = tmpStat
                    sql += f"{tmpKey},"
                    tmpIdx += 1
                sql = sql[:-1]
                sql += ") "
            if labels != []:
                sql += "AND prodSourceLabel IN ("
                tmpIdx = 0
                for tmpStat in labels:
                    tmpKey = f":label{tmpIdx}"
                    varMap[tmpKey] = tmpStat
                    sql += f"{tmpKey},"
                    tmpIdx += 1
                sql = sql[:-1]
                sql += ") "
            if processTypes != []:
                sql += "AND processingType IN ("
                tmpIdx = 0
                for tmpStat in processTypes:
                    tmpKey = f":processType{tmpIdx}"
                    varMap[tmpKey] = tmpStat
                    sql += f"{tmpKey},"
                    tmpIdx += 1
                sql = sql[:-1]
                sql += ") "
            if sites != []:
                sql += "AND computingSite IN ("
                tmpIdx = 0
                for tmpStat in sites:
                    tmpKey = f":site{tmpIdx}"
                    varMap[tmpKey] = tmpStat
                    sql += f"{tmpKey},"
                    tmpIdx += 1
                sql = sql[:-1]
                sql += ") "
            if clouds != []:
                sql += "AND cloud IN ("
                tmpIdx = 0
                for tmpStat in clouds:
                    tmpKey = f":cloud{tmpIdx}"
                    varMap[tmpKey] = tmpStat
                    sql += f"{tmpKey},"
                    tmpIdx += 1
                sql = sql[:-1]
                sql += ") "
            if onlyReassignable:
                sql += "AND (relocationFlag IS NULL OR relocationFlag<>:relocationFlag) "
                varMap[":relocationFlag"] = 2
            # sql for lock
            if not useStateChangeTime:
                sqlLock = f"UPDATE {tableName} SET modificationTime=CURRENT_DATE WHERE PandaID=:PandaID"
            else:
                sqlLock = f"UPDATE {tableName} SET stateChangeTime=CURRENT_DATE WHERE PandaID=:PandaID"
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 1000000
            _logger.debug(sql + comment + str(varMap))
            self.cur.execute(sql + comment, varMap)
            resList = self.cur.fetchall()
            _logger.debug(f"lockJobsForReassign : found {len(resList)}")
            retList = []
            # lock
            for tmpItem in resList:
                tmpID = tmpItem[0]
                varLock = {":PandaID": tmpID}
                self.cur.execute(sqlLock + comment, varLock)
                retList.append(tmpItem)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # sort
            retList.sort()
            _logger.debug(f"lockJobsForReassign : return {len(retList)}")
            return True, retList
        except Exception:
            # roll back
            self._rollback()
            errType, errValue = sys.exc_info()[:2]
            _logger.error(f"lockJobsForReassign : {errType} {errValue}")
            # return empty
            return False, []

    # lock jobs for finisher
    def lockJobsForFinisher(self, timeNow, rownum, highPrio):
        comment = " /* DBProxy.lockJobsForFinisher */"
        _logger.debug(f"lockJobsForFinisher : {timeNow} {rownum} {highPrio}")
        try:
            varMap = {}
            varMap[":jobStatus"] = "transferring"
            varMap[":currentPriority"] = 800
            varMap[":pLabel1"] = "managed"
            varMap[":pLabel2"] = "test"
            varMap[":esJumbo"] = EventServiceUtils.jumboJobFlagNumber
            # make sql
            sql = "SELECT PandaID FROM ATLAS_PANDA.jobsActive4 "
            sql += "WHERE jobStatus=:jobStatus AND modificationTime<:modificationTime AND prodSourceLabel IN (:pLabel1,:pLabel2) "
            sql += "AND (eventService IS NULL OR eventService<>:esJumbo) "
            if highPrio:
                varMap[":modificationTime"] = timeNow - datetime.timedelta(hours=1)
                sql += f"AND currentPriority>=:currentPriority AND rownum<={rownum} "
            else:
                sql += f"AND currentPriority<:currentPriority AND rownum<={rownum} "
                varMap[":modificationTime"] = timeNow - datetime.timedelta(hours=2)
            sql += "FOR UPDATE "
            # sql for lock
            sqlLock = "UPDATE ATLAS_PANDA.jobsActive4 SET modificationTime=CURRENT_DATE WHERE PandaID=:PandaID"
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 1000
            self.cur.execute(sql + comment, varMap)
            resList = self.cur.fetchall()
            retList = []
            # lock
            for (tmpID,) in resList:
                varLock = {":PandaID": tmpID}
                self.cur.execute(sqlLock + comment, varLock)
                retList.append(tmpID)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # sort
            retList.sort()
            _logger.debug(f"lockJobsForFinisher : {len(retList)}")
            return True, retList
        except Exception:
            # roll back
            self._rollback()
            errType, errValue = sys.exc_info()[:2]
            _logger.error(f"lockJobsForFinisher : {errType} {errValue}")
            # return empty
            return False, []

    # lock jobs for activator
    def lockJobsForActivator(self, timeLimit, rownum, prio):
        comment = " /* DBProxy.lockJobsForActivator */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            varMap = {}
            varMap[":jobStatus"] = "assigned"
            if prio > 0:
                varMap[":currentPriority"] = prio
            varMap[":timeLimit"] = timeLimit
            # make sql
            sql = "SELECT PandaID FROM ATLAS_PANDA.jobsDefined4 "
            sql += "WHERE jobStatus=:jobStatus AND (prodDBUpdateTime IS NULL OR prodDBUpdateTime<:timeLimit) "
            if prio > 0:
                sql += "AND currentPriority>=:currentPriority "
            sql += f"AND rownum<={rownum} "
            sql += "FOR UPDATE "
            # sql for lock
            sqlLock = "UPDATE ATLAS_PANDA.jobsDefined4 SET prodDBUpdateTime=CURRENT_DATE WHERE PandaID=:PandaID"
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 1000
            self.cur.execute(sql + comment, varMap)
            resList = self.cur.fetchall()
            retList = []
            # lock
            for (tmpID,) in resList:
                varLock = {":PandaID": tmpID}
                self.cur.execute(sqlLock + comment, varLock)
                retList.append(tmpID)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # sort
            retList.sort()
            tmpLog.debug(f"locked {len(retList)} jobs")
            return True, retList
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            # return empty
            return False, []

    # add metadata
    def addMetadata(self, pandaID, metadata, newStatus):
        comment = " /* DBProxy.addMetaData */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" <PandaID={pandaID}>")
        tmpLog.debug(f"start {newStatus}")
        # discard metadata for failed jobs
        if newStatus == "failed":
            tmpLog.debug("skip")
            return True
        sqlJ = "SELECT jobStatus FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
        sqlJ += "UNION "
        sqlJ += "SELECT jobStatus FROM ATLAS_PANDA.jobsArchived4 WHERE PandaID=:PandaID "
        sql0 = "SELECT PandaID FROM ATLAS_PANDA.metaTable WHERE PandaID=:PandaID"
        sql1 = "INSERT INTO ATLAS_PANDA.metaTable (PandaID,metaData) VALUES (:PandaID,:metaData)"
        nTry = 1
        regStart = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
        for iTry in range(nTry):
            try:
                # autocommit on
                self.conn.begin()
                self.cur.arraysize = 10
                # check job status
                varMap = {}
                varMap[":PandaID"] = pandaID
                self.cur.execute(sqlJ + comment, varMap)
                resJ = self.cur.fetchone()
                if resJ is not None:
                    (jobStatus,) = resJ
                else:
                    jobStatus = "unknown"
                if jobStatus in ["unknown"]:
                    tmpLog.debug(f"skip jobStatus={jobStatus}")
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    return False
                # skip if in final state
                if jobStatus in ["cancelled", "closed", "finished", "failed"]:
                    tmpLog.debug(f"skip jobStatus={jobStatus}")
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    # return True so that subsequent procedure can keep going
                    return True
                # select
                varMap = {}
                varMap[":PandaID"] = pandaID
                self.cur.arraysize = 10
                self.cur.execute(sql0 + comment, varMap)
                res = self.cur.fetchone()
                # already exist
                if res is not None:
                    tmpLog.debug(f"skip duplicated during jobStatus={jobStatus}")
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    return True
                # truncate
                if metadata is not None:
                    origSize = len(metadata)
                else:
                    origSize = 0
                maxSize = 1024 * 1024
                if newStatus in ["failed"] and origSize > maxSize:
                    metadata = metadata[:maxSize]
                # insert
                varMap = {}
                varMap[":PandaID"] = pandaID
                varMap[":metaData"] = metadata
                self.cur.execute(sql1 + comment, varMap)
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                regTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - regStart
                msgStr = f"done in jobStatus={jobStatus}->{newStatus} took {regTime.seconds} sec"
                if metadata is not None:
                    msgStr += f" for {len(metadata)} (orig {origSize}) bytes"
                tmpLog.debug(msgStr)
                return True
            except Exception:
                # roll back
                self._rollback()
                if iTry + 1 < nTry:
                    tmpLog.debug(f"retry : {iTry}")
                    time.sleep(random.randint(10, 20))
                    continue
                self.dumpErrorMessage(_logger, methodName)
                return False

    # add stdout
    def addStdOut(self, pandaID, stdOut):
        comment = " /* DBProxy.addStdOut */"
        _logger.debug(f"addStdOut : {pandaID} start")
        sqlJ = "SELECT PandaID FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID FOR UPDATE "
        sqlC = "SELECT PandaID FROM ATLAS_PANDA.jobsDebug WHERE PandaID=:PandaID "
        sqlI = "INSERT INTO ATLAS_PANDA.jobsDebug (PandaID,stdOut) VALUES (:PandaID,:stdOut) "
        sqlU = "UPDATE ATLAS_PANDA.jobsDebug SET stdOut=:stdOut WHERE PandaID=:PandaID "
        try:
            # autocommit on
            self.conn.begin()
            # select
            varMap = {}
            varMap[":PandaID"] = pandaID
            self.cur.arraysize = 10
            # check job table
            self.cur.execute(sqlJ + comment, varMap)
            res = self.cur.fetchone()
            if res is None:
                _logger.debug(f"addStdOut : {pandaID} non active")
            else:
                # check debug table
                self.cur.execute(sqlC + comment, varMap)
                res = self.cur.fetchone()
                # already exist
                if res is not None:
                    # update
                    sql = sqlU
                else:
                    # insert
                    sql = sqlI
                # write stdout
                varMap = {}
                varMap[":PandaID"] = pandaID
                varMap[":stdOut"] = stdOut
                self.cur.execute(sql + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            return True
        except Exception:
            # roll back
            self._rollback()
            errtype, errvalue = sys.exc_info()[:2]
            _logger.error(f"addStdOut : {errtype} {errvalue}")
            return False

    # insert sandbox file info
    def insertSandboxFileInfo(self, userName, hostName, fileName, fileSize, checkSum):
        comment = " /* DBProxy.insertSandboxFileInfo */"
        _logger.debug(f"insertSandboxFileInfo : {userName} {hostName} {fileName} {fileSize} {checkSum}")
        sqlC = "SELECT userName,fileSize,checkSum FROM ATLAS_PANDAMETA.userCacheUsage "
        sqlC += "WHERE hostName=:hostName AND fileName=:fileName FOR UPDATE"
        sql = "INSERT INTO ATLAS_PANDAMETA.userCacheUsage "
        sql += "(userName,hostName,fileName,fileSize,checkSum,creationTime,modificationTime) "
        sql += "VALUES (:userName,:hostName,:fileName,:fileSize,:checkSum,CURRENT_DATE,CURRENT_DATE) "
        try:
            # begin transaction
            self.conn.begin()
            # check if it already exists
            varMap = {}
            varMap[":hostName"] = hostName
            varMap[":fileName"] = fileName
            self.cur.arraysize = 10
            self.cur.execute(sqlC + comment, varMap)
            res = self.cur.fetchall()
            if len(res) != 0:
                _logger.debug(f"insertSandboxFileInfo : skip {hostName} {fileName} since already exists")
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                return "WARNING: file exist"
            # insert
            varMap = {}
            varMap[":userName"] = userName
            varMap[":hostName"] = hostName
            varMap[":fileName"] = fileName
            varMap[":fileSize"] = fileSize
            varMap[":checkSum"] = checkSum
            self.cur.execute(sql + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            return "OK"
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error(f"insertSandboxFileInfo : {type} {value}")
            return "ERROR: DB failure"

    # get and lock sandbox files
    def getLockSandboxFiles(self, time_limit, n_files):
        comment = " /* DBProxy.getLockSandboxFiles */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName)
        sqlC = (
            "SELECT * FROM ("
            "SELECT userName,hostName,fileName,creationTime,modificationTime FROM ATLAS_PANDAMETA.userCacheUsage "
            "WHERE modificationTime<:timeLimit AND (fileName like 'sources%' OR fileName like 'jobO%') ) "
            "WHERE rownum<:nRows "
        )
        sqlU = "UPDATE ATLAS_PANDAMETA.userCacheUsage SET modificationTime=CURRENT_DATE " "WHERE userName=:userName AND fileName=:fileName "
        try:
            tmpLog.debug("start")
            # begin transaction
            self.conn.begin()
            # check if it already exists
            varMap = {}
            varMap[":timeLimit"] = time_limit
            varMap[":nRows"] = n_files
            self.cur.execute(sqlC + comment, varMap)
            res = self.cur.fetchall()
            retList = []
            for userName, hostName, fileName, creationTime, modificationTime in res:
                retList.append((userName, hostName, fileName, creationTime, modificationTime))
                # lock
                varMap = dict()
                varMap[":userName"] = userName
                varMap[":fileName"] = fileName
                self.cur.execute(sqlU + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"locked {len(retList)} files")
            return retList
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None

    # check duplicated sandbox file
    def checkSandboxFile(self, dn, fileSize, checkSum):
        comment = " /* DBProxy.checkSandboxFile */"
        _logger.debug(f"checkSandboxFile : {dn} {fileSize} {checkSum}")
        sqlC = "SELECT hostName,fileName FROM ATLAS_PANDAMETA.userCacheUsage "
        sqlC += "WHERE userName=:userName AND fileSize=:fileSize AND checkSum=:checkSum "
        sqlC += "AND hostName<>:ngHostName AND creationTime>CURRENT_DATE-3 "
        sqlC += "AND creationTime>CURRENT_DATE-3 "
        try:
            retStr = "NOTFOUND"
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ["", "NULL", None]:
                compactDN = dn
            # begin transaction
            self.conn.begin()
            # check if it already exists
            varMap = {}
            varMap[":userName"] = compactDN
            varMap[":fileSize"] = fileSize
            varMap[":checkSum"] = str(checkSum)
            varMap[":ngHostName"] = "localhost.localdomain"
            self.cur.arraysize = 10
            self.cur.execute(sqlC + comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            if len(res) != 0:
                hostName, fileName = res[0]
                retStr = f"FOUND:{hostName}:{fileName}"
            _logger.debug(f"checkSandboxFile -> {retStr}")
            return retStr
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error(f"checkSandboxFile : {type} {value}")
            return "ERROR: DB failure"

    # insert dataset
    def insertDataset(self, dataset, tablename="ATLAS_PANDA.Datasets"):
        comment = " /* DBProxy.insertDataset */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" <dataset={dataset.name}>")
        tmpLog.debug("start")
        sql0 = f"SELECT COUNT(*) FROM {tablename} WHERE vuid=:vuid "
        sql1 = f"INSERT INTO {tablename} "
        sql1 += f"({DatasetSpec.columnNames()}) "
        sql1 += DatasetSpec.bindValuesExpression()
        sql2 = f"SELECT name FROM {tablename} WHERE vuid=:vuid "
        # time information
        dataset.creationdate = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
        dataset.modificationdate = dataset.creationdate
        try:
            # subtype
            if dataset.subType in ["", "NULL", None]:
                # define using name
                if re.search("_dis\d+$", dataset.name) is not None:
                    dataset.subType = "dis"
                elif re.search("_sub\d+$", dataset.name) is not None:
                    dataset.subType = "sub"
                else:
                    dataset.subType = "top"
            # begin transaction
            self.conn.begin()
            # check if it already exists
            varMap = {}
            varMap[":vuid"] = dataset.vuid
            self.cur.execute(sql0 + comment, varMap)
            (nDS,) = self.cur.fetchone()
            tmpLog.debug(f"nDS={nDS} with {dataset.vuid}")
            if nDS == 0:
                # insert
                tmpLog.debug(sql1 + comment + str(dataset.valuesMap()))
                self.cur.execute(sql1 + comment, dataset.valuesMap())
                # check name in DB
                varMap = {}
                varMap[":vuid"] = dataset.vuid
                self.cur.execute(sql2 + comment, varMap)
                (nameInDB,) = self.cur.fetchone()
                tmpLog.debug(f"inDB -> {nameInDB} {dataset.name == nameInDB}")
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False

    # get and lock dataset with a query
    def getLockDatasets(self, sqlQuery, varMapGet, modTimeOffset="", getVersion=False):
        comment = " /* DBProxy.getLockDatasets */"
        _logger.debug(f"getLockDatasets({sqlQuery},{str(varMapGet)},{modTimeOffset})")
        sqlGet = (
            "SELECT /*+ INDEX_RS_ASC(tab(STATUS,TYPE,MODIFICATIONDATE)) */ vuid,name,modificationdate,version,transferStatus FROM ATLAS_PANDA.Datasets tab WHERE "
            + sqlQuery
        )
        sqlLock = "UPDATE ATLAS_PANDA.Datasets SET modificationdate=CURRENT_DATE"
        if modTimeOffset != "":
            sqlLock += f"+{modTimeOffset}"
        sqlLock += ",transferStatus=MOD(transferStatus+1,10)"
        if getVersion:
            sqlLock += ",version=:version"
        sqlLock += " WHERE vuid=:vuid AND transferStatus=:transferStatus"
        retList = []
        try:
            # begin transaction
            self.conn.begin()
            # get datasets
            self.cur.arraysize = 1000000
            self.cur.execute(sqlGet + comment, varMapGet)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # loop over all datasets
            if res is not None and len(res) != 0:
                for vuid, name, modificationdate, version, transferStatus in res:
                    # lock
                    varMapLock = {}
                    varMapLock[":vuid"] = vuid
                    varMapLock[":transferStatus"] = transferStatus
                    if getVersion:
                        try:
                            varMapLock[":version"] = str(int(version) + 1)
                        except Exception:
                            varMapLock[":version"] = str(1)
                    # begin transaction
                    self.conn.begin()
                    # update for lock
                    self.cur.execute(sqlLock + comment, varMapLock)
                    retU = self.cur.rowcount
                    # commit
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    if retU > 0:
                        # append
                        if not getVersion:
                            retList.append((vuid, name, modificationdate))
                        else:
                            retList.append((vuid, name, modificationdate, version))
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # retrun
            return retList
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error(f"getLockDatasets : {type} {value}")
            return []

    # query dataset with map
    def queryDatasetWithMap(self, map):
        comment = " /* DBProxy.queryDatasetWithMap */"
        _logger.debug(f"queryDatasetWithMap({map})")
        if "name" in map:
            sql1 = """SELECT /*+ BEGIN_OUTLINE_DATA """
            sql1 += """INDEX_RS_ASC(@"SEL$1" "TAB"@"SEL$1" ("DATASETS"."NAME")) """
            sql1 += """OUTLINE_LEAF(@"SEL$1") ALL_ROWS """
            sql1 += """IGNORE_OPTIM_EMBEDDED_HINTS """
            sql1 += """END_OUTLINE_DATA */ """
            sql1 += f"{DatasetSpec.columnNames()} FROM ATLAS_PANDA.Datasets tab"
        else:
            sql1 = f"SELECT {DatasetSpec.columnNames()} FROM ATLAS_PANDA.Datasets"
        varMap = {}
        for key in map:
            if len(varMap) == 0:
                sql1 += f" WHERE {key}=:{key}"
            else:
                sql1 += f" AND {key}=:{key}"
            varMap[f":{key}"] = map[key]
        try:
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 100
            _logger.debug(sql1 + comment + str(varMap))
            self.cur.execute(sql1 + comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # instantiate Dataset
            if res is not None and len(res) != 0:
                dataset = DatasetSpec()
                dataset.pack(res[0])
                return dataset
            _logger.error(f"queryDatasetWithMap({map}) : dataset not found")
            return None
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error(f"queryDatasetWithMap({map}) : {type} {value}")
            return None

    # update dataset
    def updateDataset(self, datasets, withLock, withCriteria, criteriaMap):
        comment = " /* DBProxy.updateDataset */"
        _logger.debug("updateDataset()")
        sql1 = f"UPDATE ATLAS_PANDA.Datasets SET {DatasetSpec.bindUpdateExpression()} "
        sql1 += "WHERE vuid=:vuid"
        if withCriteria != "":
            sql1 += f" AND {withCriteria}"
        retList = []
        try:
            # start transaction
            self.conn.begin()
            for dataset in datasets:
                _logger.debug(f"updateDataset({dataset.name},{dataset.status})")
                # time information
                dataset.modificationdate = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
                # update
                varMap = dataset.valuesMap()
                varMap[":vuid"] = dataset.vuid
                for cKey in criteriaMap:
                    varMap[cKey] = criteriaMap[cKey]
                _logger.debug(sql1 + comment + str(varMap))
                self.cur.execute(sql1 + comment, varMap)
                retU = self.cur.rowcount
                if retU != 0 and retU != 1:
                    raise RuntimeError(f"Invalid retrun {retU}")
                retList.append(retU)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            _logger.debug(f"updateDataset() ret:{retList}")
            return retList
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error(f"updateDataset() : {type} {value}")
            return []

    # trigger cleanup of internal datasets used by a task
    def trigger_cleanup_internal_datasets(self, task_id: int) -> bool:
        """
        Set deleting flag to dispatch datasets used by a task, which triggers deletion in datasetManager
        """
        comment = " /* DBProxy.trigger_cleanup_internal_datasets */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, f"{method_name} < jediTaskID={task_id} >")
        tmp_log.debug("start")
        sql1 = (
            f"UPDATE {panda_config.schemaPANDA}.Datasets SET status=:newStatus,modificationdate=CURRENT_DATE "
            "WHERE type=:type AND MoverID=:taskID AND status IN (:status_d,:status_c) "
        )
        try:
            # start transaction
            self.conn.begin()
            # update
            var_map = {
                ":type": "dispatch",
                ":newStatus": "deleting",
                ":taskID": task_id,
                ":status_d": "defined",
                ":status_c": "completed",
            }
            self.cur.execute(sql1 + comment, var_map)
            ret_u = self.cur.rowcount
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmp_log.debug(f"set flag to {ret_u} dispatch datasets")
            return True
        except Exception:
            # roll back
            self._rollback()
            self.dumpErrorMessage(_logger, method_name)
            return False

    # get serial number for dataset, insert dummy datasets to increment SN
    def getSerialNumber(self, datasetname, definedFreshFlag=None):
        comment = " /* DBProxy.getSerialNumber */"
        try:
            _logger.debug(f"getSerialNumber({datasetname},{definedFreshFlag})")
            if isinstance(datasetname, str):
                datasetname = datasetname.encode("ascii", "ignore")
                _logger.debug(f"getSerialNumber converted unicode for {datasetname}")
            # start transaction
            self.conn.begin()
            # check freshness
            if definedFreshFlag is None:
                # select
                varMap = {}
                varMap[":name"] = datasetname
                varMap[":type"] = "output"
                sql = "SELECT /*+ INDEX_RS_ASC(TAB (DATASETS.NAME)) */ COUNT(*) FROM ATLAS_PANDA.Datasets tab WHERE type=:type AND name=:name"
                self.cur.arraysize = 100
                self.cur.execute(sql + comment, varMap)
                res = self.cur.fetchone()
                # fresh dataset or not
                if res is not None and len(res) != 0 and res[0] > 0:
                    freshFlag = False
                else:
                    freshFlag = True
            else:
                # use predefined flag
                freshFlag = definedFreshFlag
            # get serial number
            if self.backend == "oracle":
                sql = "SELECT ATLAS_PANDA.SUBCOUNTER_SUBID_SEQ.nextval FROM dual"
                self.cur.arraysize = 100
                self.cur.execute(sql + comment, {})
                (sn,) = self.cur.fetchone()
            elif self.backend == "postgres":
                sql = f"SELECT {panda_config.schemaPANDA}.SUBCOUNTER_SUBID_SEQ.nextval"
                self.cur.arraysize = 100
                self.cur.execute(sql + comment, {})
                (sn,) = self.cur.fetchone()
            else:
                # panda_config.backend == 'mysql'
                # fake sequence
                sql = " INSERT INTO ATLAS_PANDA.SUBCOUNTER_SUBID_SEQ (col) VALUES (NULL) "
                self.cur.arraysize = 100
                self.cur.execute(sql + comment, {})
                sql2 = """ SELECT LAST_INSERT_ID() """
                self.cur.execute(sql2 + comment, {})
                (sn,) = self.cur.fetchone()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            _logger.debug(f"getSerialNumber : {sn} {freshFlag}")
            return (sn, freshFlag)
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error(f"getSerialNumber() : {type} {value}")
            return (-1, False)

    # count the number of files with map
    def countFilesWithMap(self, map):
        comment = " /* DBProxy.countFilesWithMap */"
        sql1 = "SELECT /*+ index(tab FILESTABLE4_DESTDBLOCK_IDX) */ COUNT(*) FROM ATLAS_PANDA.filesTable4 tab"
        varMap = {}
        for key in map:
            if len(varMap) == 0:
                sql1 += f" WHERE {key}=:{key}"
            else:
                sql1 += f" AND {key}=:{key}"
            varMap[f":{key}"] = map[key]
        nTry = 3
        for iTry in range(nTry):
            try:
                # start transaction
                self.conn.begin()
                # select
                _logger.debug(f"countFilesWithMap() : {sql1} {str(map)}")
                self.cur.arraysize = 10
                retS = self.cur.execute(sql1 + comment, varMap)
                res = self.cur.fetchone()
                _logger.debug(f"countFilesWithMap() : {retS} {str(res)}")
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                nFiles = 0
                if res is not None:
                    nFiles = res[0]
                return nFiles
            except Exception:
                # roll back
                self._rollback()
                if iTry + 1 < nTry:
                    _logger.debug(f"countFilesWithMap() retry : {iTry}")
                    time.sleep(random.randint(10, 20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error(f"countFilesWithMap({map}) : {type} {value}")
                return -1

    # update input files and return corresponding PandaIDs
    def updateInFilesReturnPandaIDs(self, dataset, status, fileLFN=""):
        comment = " /* DBProxy.updateInFilesReturnPandaIDs */"
        _logger.debug(f"updateInFilesReturnPandaIDs({dataset},{fileLFN})")
        sql0 = "SELECT /*+ index(tab FILESTABLE4_DISPDBLOCK_IDX) */ row_ID,PandaID FROM ATLAS_PANDA.filesTable4 tab WHERE status<>:status AND dispatchDBlock=:dispatchDBlock"
        sql1 = "UPDATE /*+ index(tab FILESTABLE4_DISPDBLOCK_IDX) */ ATLAS_PANDA.filesTable4 tab SET status=:status WHERE status<>:status AND dispatchDBlock=:dispatchDBlock"
        varMap = {}
        varMap[":status"] = status
        varMap[":dispatchDBlock"] = dataset
        if fileLFN != "":
            sql0 += " AND lfn=:lfn"
            sql1 += " AND lfn=:lfn"
            varMap[":lfn"] = fileLFN
        for iTry in range(self.nTry):
            try:
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10000
                retS = self.cur.execute(sql0 + comment, varMap)
                resS = self.cur.fetchall()
                # update
                retU = self.cur.execute(sql1 + comment, varMap)
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                # collect PandaIDs
                retList = []
                for tmpRowID, tmpPandaID in resS:
                    # append
                    if tmpPandaID not in retList:
                        retList.append(tmpPandaID)
                # return
                _logger.debug(f"updateInFilesReturnPandaIDs : {str(retList)}")
                return retList
            except Exception:
                # roll back
                self._rollback()
                # error report
                if iTry + 1 < self.nTry:
                    _logger.debug(f"updateInFilesReturnPandaIDs retry : {iTry}")
                    time.sleep(random.randint(10, 20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error(f"updateInFilesReturnPandaIDs : {type} {value}")
        return []

    # update output files and return corresponding PandaIDs
    def updateOutFilesReturnPandaIDs(self, dataset, fileLFN=""):
        comment = " /* DBProxy.updateOutFilesReturnPandaIDs */"
        _logger.debug(f"updateOutFilesReturnPandaIDs({dataset},{fileLFN})")
        sql0 = "SELECT /*+ index(tab FILESTABLE4_DESTDBLOCK_IDX) */ row_ID,PandaID FROM ATLAS_PANDA.filesTable4 tab WHERE destinationDBlock=:destinationDBlock AND status=:status"
        sql1 = "UPDATE /*+ index(tab FILESTABLE4_DESTDBLOCK_IDX) */ ATLAS_PANDA.filesTable4 tab SET status='ready' WHERE destinationDBlock=:destinationDBlock AND status=:status"
        varMap = {}
        varMap[":status"] = "transferring"
        varMap[":destinationDBlock"] = dataset
        if fileLFN != "":
            sql0 += " AND lfn=:lfn"
            sql1 += " AND lfn=:lfn"
            varMap[":lfn"] = fileLFN
        for iTry in range(self.nTry):
            try:
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10000
                retS = self.cur.execute(sql0 + comment, varMap)
                resS = self.cur.fetchall()
                # update
                retList = []
                retU = self.cur.execute(sql1 + comment, varMap)
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                # collect PandaIDs
                retList = []
                for tmpRowID, tmpPandaID in resS:
                    # append
                    if tmpPandaID not in retList:
                        retList.append(tmpPandaID)
                # return
                _logger.debug(f"updateOutFilesReturnPandaIDs : {str(retList)}")
                return retList
            except Exception:
                # roll back
                self._rollback()
                # error report
                if iTry + 1 < self.nTry:
                    _logger.debug(f"updateOutFilesReturnPandaIDs retry : {iTry}")
                    time.sleep(random.randint(10, 20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error(f"updateOutFilesReturnPandaIDs : {type} {value}")
        return []

    # get _dis datasets associated to _sub
    def getAssociatedDisDatasets(self, subDsName):
        comment = " /* DBProxy.getAssociatedDisDatasets */"
        _logger.debug(f"getAssociatedDisDatasets({subDsName})")
        sqlF = (
            "SELECT /*+ index(tab FILESTABLE4_DESTDBLOCK_IDX) */ distinct PandaID FROM ATLAS_PANDA.filesTable4 tab WHERE destinationDBlock=:destinationDBlock"
        )
        sqlJ = "SELECT distinct dispatchDBlock FROM ATLAS_PANDA.filesTable4 WHERE PandaID=:PandaID AND type=:type"
        try:
            # start transaction
            self.conn.begin()
            # get PandaIDs
            varMap = {}
            varMap[":destinationDBlock"] = subDsName
            self.cur.arraysize = 10000
            self.cur.execute(sqlF + comment, varMap)
            resS = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # loop over all PandaIDs
            retList = []
            for (pandaID,) in resS:
                # start transaction
                self.conn.begin()
                # get _dis name
                varMap = {}
                varMap[":type"] = "input"
                varMap[":PandaID"] = pandaID
                self.cur.arraysize = 1000
                self.cur.execute(sqlJ + comment, varMap)
                resD = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                # append
                for (disName,) in resD:
                    if disName is not None and disName not in retList:
                        retList.append(disName)
            # return
            _logger.debug(f"getAssociatedDisDatasets : {str(retList)}")
            return retList
        except Exception:
            # roll back
            self._rollback()
            errType, errValue = sys.exc_info()[:2]
            _logger.error(f"getAssociatedDisDatasets : {subDsName} : {errType} {errValue}")
            return []

    # set GUIDs
    def setGUIDs(self, files):
        comment = " /* DBProxy.setGUIDs */"
        _logger.debug(f"setGUIDs({files})")
        sql0 = "UPDATE ATLAS_PANDA.filesTable4 SET GUID=:GUID,fsize=:fsize,checksum=:checksum,scope=:scope WHERE lfn=:lfn"
        for iTry in range(self.nTry):
            try:
                # start transaction
                self.conn.begin()
                self.cur.arraysize = 1000000
                # update
                for file in files:
                    varMap = {}
                    varMap[":GUID"] = file["guid"]
                    varMap[":lfn"] = file["lfn"]
                    if file["checksum"] in ["", "NULL"]:
                        varMap[":checksum"] = None
                    else:
                        varMap[":checksum"] = file["checksum"]
                    varMap[":fsize"] = file["fsize"]
                    if "scope" not in file or file["scope"] in ["", "NULL"]:
                        varMap[":scope"] = None
                    else:
                        varMap[":scope"] = file["scope"]
                    self.cur.execute(sql0 + comment, varMap)
                    retU = self.cur.rowcount
                    _logger.debug(f"setGUIDs : retU {retU}")
                    if retU < 0:
                        raise RuntimeError("SQL error")
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                return True
            except Exception:
                # roll back
                self._rollback()
                # error report
                if iTry + 1 < self.nTry:
                    _logger.debug(f"setGUIDs retry : {iTry}")
                    time.sleep(random.randint(10, 20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error(f"setGUIDs : {type} {value}")
        return False

    # get job statistics
    def getJobStatistics(
        self,
        archived=False,
        predefined=False,
        workingGroup="",
        countryGroup="",
        jobType="",
        forAnal=None,
        minPriority=None,
    ):
        comment = " /* DBProxy.getJobStatistics */"
        _logger.debug(f"getJobStatistics({archived},{predefined},'{workingGroup}','{countryGroup}','{jobType}',{forAnal},{minPriority})")
        timeLimit = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(hours=12)
        sql0 = "SELECT computingSite,jobStatus,COUNT(*) FROM %s "
        # processingType
        tmpJobTypeMap = {}
        sqlJobType = ""
        useWhereInSQL = True
        if forAnal is None or jobType != "":
            useWhereInSQL = False
        elif forAnal is True:
            tmpJobTypeMap[":prodSourceLabel1"] = "user"
            tmpJobTypeMap[":prodSourceLabel2"] = "panda"
            sql0 += "WHERE prodSourceLabel IN ("
            sqlJobType = ":prodSourceLabel1,:prodSourceLabel2) "
        else:
            tmpJobTypeMap[":prodSourceLabel1"] = "managed"
            sql0 += "WHERE prodSourceLabel IN ("
            sqlJobType = ":prodSourceLabel1) "
        sql0 += sqlJobType
        # predefined
        if predefined:
            if useWhereInSQL:
                sql0 += "AND relocationFlag=1 "
            else:
                sql0 += "WHERE relocationFlag=1 "
                useWhereInSQL = True
        # working group
        tmpGroupMap = {}
        sqlGroups = ""
        if workingGroup != "":
            if useWhereInSQL:
                sqlGroups += "AND workingGroup IN ("
            else:
                sqlGroups += "WHERE workingGroup IN ("
                useWhereInSQL = True
            # loop over all groups
            idxWG = 1
            for tmpWG in workingGroup.split(","):
                tmpWGkey = f":workingGroup{idxWG}"
                sqlGroups += f"{tmpWGkey},"
                tmpGroupMap[tmpWGkey] = tmpWG
                idxWG += 1
            sqlGroups = sqlGroups[:-1] + ") "

        sql0 += sqlGroups
        # minimum priority
        sqlPrio = ""
        tmpPrioMap = {}
        if minPriority is not None:
            if useWhereInSQL:
                sqlPrio = "AND currentPriority>=:minPriority "
            else:
                sqlPrio = "WHERE currentPriority>=:minPriority "
                useWhereInSQL = True
            tmpPrioMap[":minPriority"] = minPriority
        sql0 += sqlPrio
        sql0 += "GROUP BY computingSite,jobStatus"
        sqlA = "SELECT /*+ INDEX_RS_ASC(tab (MODIFICATIONTIME PRODSOURCELABEL)) */ computingSite,jobStatus,COUNT(*) "
        sqlA += f"FROM {panda_config.schemaPANDA}.jobsArchived4 tab WHERE modificationTime>:modificationTime "
        if sqlJobType != "":
            sqlA += "AND prodSourceLabel IN ("
            sqlA += sqlJobType
        if predefined:
            sqlA += "AND relocationFlag=1 "
        sqlA += sqlGroups
        sqlA += sqlPrio
        sqlA += "GROUP BY computingSite,jobStatus"
        jobs_active_4_table = f"{panda_config.schemaPANDA}.jobsActive4"
        jobs_defined_4_table = f"{panda_config.schemaPANDA}.jobsDefined4"
        jobs_archived_4_table = f"{panda_config.schemaPANDA}.jobsArchived4"
        tables = [jobs_active_4_table, jobs_defined_4_table]
        if archived:
            tables.append(jobs_archived_4_table)
        # sql for materialized view
        sqlMV = re.sub("COUNT\(\*\)", "SUM(num_of_jobs)", sql0)
        sqlMV = re.sub(":minPriority", "TRUNC(:minPriority,-1)", sqlMV)
        sqlMV = re.sub("SELECT ", "SELECT /*+ RESULT_CACHE */ ", sqlMV)
        ret = {}
        nTry = 3
        for iTry in range(nTry):
            try:
                for table in tables:
                    # start transaction
                    self.conn.begin()
                    # select
                    varMap = {}
                    for tmpJobType in tmpJobTypeMap:
                        varMap[tmpJobType] = tmpJobTypeMap[tmpJobType]
                    for tmpGroup in tmpGroupMap:
                        varMap[tmpGroup] = tmpGroupMap[tmpGroup]
                    for tmpPrio in tmpPrioMap:
                        varMap[tmpPrio] = tmpPrioMap[tmpPrio]
                    if table != jobs_archived_4_table:
                        self.cur.arraysize = 10000
                        if table == jobs_active_4_table:
                            sqlExeTmp = (sqlMV + comment) % f"{panda_config.schemaPANDA}.MV_JOBSACTIVE4_STATS"
                        else:
                            sqlExeTmp = (sql0 + comment) % table
                        _logger.debug(f"getJobStatistics : {sqlExeTmp} {str(varMap)}")
                        self.cur.execute(sqlExeTmp, varMap)
                    else:
                        varMap[":modificationTime"] = timeLimit
                        self.cur.arraysize = 10000
                        self.cur.execute(sqlA + comment, varMap)
                    res = self.cur.fetchall()
                    # commit
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    # create map
                    for computingSite, jobStatus, nJobs in res:
                        # FIXME
                        # ignore some job status since they break APF
                        if jobStatus in ["merging"]:
                            continue
                        if computingSite not in ret:
                            ret[computingSite] = {}
                        if jobStatus not in ret[computingSite]:
                            ret[computingSite][jobStatus] = 0
                        ret[computingSite][jobStatus] += nJobs
                # for zero
                stateList = ["assigned", "activated", "running"]
                if archived:
                    stateList += ["finished", "failed"]
                for site in ret:
                    for state in stateList:
                        if state not in ret[site]:
                            ret[site][state] = 0
                # return
                _logger.debug(f"getJobStatistics -> {str(ret)}")
                return ret
            except Exception:
                # roll back
                self._rollback()
                if iTry + 1 < nTry:
                    _logger.debug(f"getJobStatistics() retry : {iTry}")
                    time.sleep(2)
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error(f"getJobStatistics : {type} {value}")
                return {}

    # get the number of job for a user
    def getNumberJobsUser(self, dn, workingGroup=None):
        comment = " /* DBProxy.getNumberJobsUser */"
        _logger.debug(f"getNumberJobsUsers({dn},{workingGroup})")
        # get compact DN
        compactDN = self.cleanUserID(dn)
        if compactDN in ["", "NULL", None]:
            compactDN = dn
        if workingGroup is not None:
            sql0 = "SELECT COUNT(*) FROM %s WHERE prodUserName=:prodUserName AND prodSourceLabel=:prodSourceLabel AND workingGroup=:workingGroup "
        else:
            sql0 = "SELECT COUNT(*) FROM %s WHERE prodUserName=:prodUserName AND prodSourceLabel=:prodSourceLabel AND workingGroup IS NULL "
        sql0 += "AND NOT jobStatus IN (:failed,:merging) "
        nTry = 1
        nJob = 0
        for iTry in range(nTry):
            try:
                for table in ("ATLAS_PANDA.jobsActive4", "ATLAS_PANDA.jobsDefined4"):
                    # start transaction
                    self.conn.begin()
                    # select
                    varMap = {}
                    varMap[":prodUserName"] = compactDN
                    varMap[":prodSourceLabel"] = "user"
                    varMap[":failed"] = "failed"
                    varMap[":merging"] = "merging"
                    if workingGroup is not None:
                        varMap[":workingGroup"] = workingGroup
                    self.cur.arraysize = 10
                    self.cur.execute((sql0 + comment) % table, varMap)
                    res = self.cur.fetchall()
                    # commit
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    # create map
                    if len(res) != 0:
                        nJob += res[0][0]
                # return
                _logger.debug(f"getNumberJobsUsers({dn}) : {nJob}")
                return nJob
            except Exception:
                # roll back
                self._rollback()
                if iTry + 1 < nTry:
                    time.sleep(2)
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error(f"getNumberJobsUsers : {type} {value}")
                return 0

    # get job statistics for ExtIF
    def getJobStatisticsForExtIF(self, sourcetype=None):
        comment = " /* DBProxy.getJobStatisticsForExtIF */"
        _logger.debug("getJobStatisticsForExtIF()")
        timeLimit = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(hours=12)
        if sourcetype == "analysis":
            sql0 = "SELECT jobStatus,COUNT(*), cloud FROM %s WHERE prodSourceLabel IN (:prodSourceLabel1, :prodSourceLabel2) GROUP BY jobStatus, cloud"

            sqlA = "SELECT /* use_json_type */ /*+ INDEX_RS_ASC(tab (MODIFICATIONTIME PRODSOURCELABEL)) */ jobStatus,COUNT(*), tabS.data.cloud FROM %s tab, ATLAS_PANDA.schedconfig_json tabS "
            sqlA += "WHERE prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2) AND tab.computingSite=tabS.panda_queue "
        else:
            sql0 = "SELECT /* use_json_type */ tab.jobStatus, COUNT(*), tabS.data.cloud FROM %s tab, ATLAS_PANDA.schedconfig_json tabS "
            sql0 += "WHERE prodSourceLabel IN (:prodSourceLabel1,"
            for tmpLabel in JobUtils.list_ptest_prod_sources:
                tmpKey = f":prodSourceLabel_{tmpLabel}"
                sql0 += tmpKey
                sql0 += ","
            sql0 = sql0[:-1]
            sql0 += ") AND tab.computingSite=tabS.panda_queue GROUP BY tab.jobStatus, tabS.data.cloud"
            sqlA = "SELECT /* use_json_type */ /*+ INDEX_RS_ASC(tab (MODIFICATIONTIME PRODSOURCELABEL)) */ jobStatus, COUNT(*), tabS.data.cloud FROM %s tab, ATLAS_PANDA.schedconfig_json tabS "
            sqlA += "WHERE prodSourceLabel IN (:prodSourceLabel1,"
            for tmpLabel in JobUtils.list_ptest_prod_sources:
                tmpKey = f":prodSourceLabel_{tmpLabel}"
                sqlA += tmpKey
                sqlA += ","
            sqlA = sqlA[:-1]
            sqlA += ") AND tab.computingSite=tabS.panda_queue "
        sqlA += "AND modificationTime>:modificationTime GROUP BY tab.jobStatus,tabS.data.cloud"
        # sql for materialized view
        sqlMV = re.sub("COUNT\(\*\)", "SUM(num_of_jobs)", sql0)
        sqlMV = re.sub("SELECT ", "SELECT /*+ RESULT_CACHE */ ", sqlMV)
        ret = {}
        try:
            for table in (
                "ATLAS_PANDA.jobsActive4",
                "ATLAS_PANDA.jobsWaiting4",
                "ATLAS_PANDA.jobsArchived4",
                "ATLAS_PANDA.jobsDefined4",
            ):
                # start transaction
                self.conn.begin()
                # select
                varMap = {}
                if sourcetype == "analysis":
                    varMap[":prodSourceLabel1"] = "user"
                    varMap[":prodSourceLabel2"] = "panda"
                else:
                    varMap[":prodSourceLabel1"] = "managed"
                    for tmpLabel in JobUtils.list_ptest_prod_sources:
                        tmpKey = f":prodSourceLabel_{tmpLabel}"
                        varMap[tmpKey] = tmpLabel
                if table != "ATLAS_PANDA.jobsArchived4":
                    self.cur.arraysize = 10000
                    if table == "ATLAS_PANDA.jobsActive4":
                        self.cur.execute(
                            (sqlMV + comment) % "ATLAS_PANDA.MV_JOBSACTIVE4_STATS",
                            varMap,
                        )
                    else:
                        self.cur.execute((sql0 + comment) % table, varMap)
                else:
                    varMap[":modificationTime"] = timeLimit
                    self.cur.arraysize = 10000
                    self.cur.execute((sqlA + comment) % table, varMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                # change NULL to US for old jobs
                newRes = []
                usMap = {}
                for jobStatus, count, cloud in res:
                    ret.setdefault(cloud, dict())
                    ret[cloud].setdefault(jobStatus, 0)
                    ret[cloud][jobStatus] += count
            # return
            _logger.debug(f"getJobStatisticsForExtIF -> {str(ret)}")
            return ret
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error(f"getJobStatisticsForExtIF : {type} {value}")
            return {}

    # get job statistics per processingType
    def getJobStatisticsPerProcessingType(self, useMorePG=False):
        comment = " /* DBProxy.getJobStatisticsPerProcessingType */"
        timeLimit = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(hours=12)
        _logger.debug("getJobStatisticsPerProcessingType()")
        if useMorePG is False:
            sqlN = "SELECT /* use_json_type */ jobStatus, COUNT(*), tabS.data.cloud, processingType "
            sqlN += "FROM %s tab, ATLAS_PANDA.schedconfig_json tabS "
            sqlN += "WHERE prodSourceLabel IN (:prodSourceLabel1,"
            for tmpLabel in JobUtils.list_ptest_prod_sources:
                tmpKey = f":prodSourceLabel_{tmpLabel}"
                sqlN += tmpKey
                sqlN += ","
            sqlN = sqlN[:-1]
            sqlN += ") AND computingSite=tabS.panda_queue "
            sqlN += "GROUP BY jobStatus,tabS.data.cloud,processingType "

            sqlA = (
                "SELECT /* use_json_type */ /*+ INDEX_RS_ASC(tab (MODIFICATIONTIME PRODSOURCELABEL)) */ jobStatus, COUNT(*), tabS.data.cloud, processingType "
            )
            sqlA += "FROM %s tab, ATLAS_PANDA.schedconfig_json tabS "
            sqlA += "WHERE prodSourceLabel IN (:prodSourceLabel1,"
            for tmpLabel in JobUtils.list_ptest_prod_sources:
                tmpKey = f":prodSourceLabel_{tmpLabel}"
                sqlA += tmpKey
                sqlA += ","
            sqlA = sqlA[:-1]
            sqlA += ") AND modificationTime>:modificationTime "
            sqlA += "AND computingSite=tabS.panda_queue "
            sqlA += "GROUP BY jobStatus,tabS.data.cloud,processingType"

        else:
            sqlN = "SELECT jobStatus,COUNT(*), cloud, processingType, coreCount, workingGroup FROM %s "
            sqlN += "WHERE prodSourceLabel IN (:prodSourceLabel1,"
            for tmpLabel in JobUtils.list_ptest_prod_sources:
                tmpKey = f":prodSourceLabel_{tmpLabel}"
                sqlN += tmpKey
                sqlN += ","
            sqlN = sqlN[:-1]
            sqlN += ") "
            sqlN += "GROUP BY jobStatus, cloud, processingType, coreCount, workingGroup"

            sqlA = "SELECT /*+ INDEX_RS_ASC(tab (MODIFICATIONTIME PRODSOURCELABEL)) */ "
            sqlA += "jobStatus, COUNT(*), cloud, processingType, coreCount, workingGroup FROM %s tab "
            sqlA += "WHERE prodSourceLabel IN (:prodSourceLabel1,"
            for tmpLabel in JobUtils.list_ptest_prod_sources:
                tmpKey = f":prodSourceLabel_{tmpLabel}"
                sqlA += tmpKey
                sqlA += ","
            sqlA = sqlA[:-1]
            sqlA += ") AND modificationTime>:modificationTime "
            sqlA += "GROUP BY jobStatus, cloud, processingType, coreCount, workingGroup"

        # sql for materialized view
        sqlMV = re.sub("COUNT\(\*\)", "SUM(num_of_jobs)", sqlN)
        sqlMV = re.sub("SELECT ", "SELECT /*+ RESULT_CACHE */ ", sqlMV)
        ret = {}
        try:
            for table in (
                "ATLAS_PANDA.jobsActive4",
                "ATLAS_PANDA.jobsWaiting4",
                "ATLAS_PANDA.jobsArchived4",
                "ATLAS_PANDA.jobsDefined4",
            ):
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10000
                # select
                varMap = {}
                varMap[":prodSourceLabel1"] = "managed"
                for tmpLabel in JobUtils.list_ptest_prod_sources:
                    tmpKey = f":prodSourceLabel_{tmpLabel}"
                    varMap[tmpKey] = tmpLabel
                if table == "ATLAS_PANDA.jobsArchived4":
                    varMap[":modificationTime"] = timeLimit
                    self.cur.execute((sqlA + comment) % table, varMap)
                else:
                    if table == "ATLAS_PANDA.jobsActive4" and useMorePG is False:
                        self.cur.execute(
                            (sqlMV + comment) % "ATLAS_PANDA.MV_JOBSACTIVE4_STATS",
                            varMap,
                        )
                    else:
                        # use real table since coreCount is unavailable in MatView
                        self.cur.execute((sqlN + comment) % table, varMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                # create map
                for tmpItem in res:
                    if useMorePG is False:
                        jobStatus, count, cloud, processingType = tmpItem
                    else:
                        (
                            jobStatus,
                            count,
                            cloud,
                            processingType,
                            coreCount,
                            workingGroup,
                        ) = tmpItem
                        # convert cloud and processingType for extended process group
                        if useMorePG == ProcessGroups.extensionLevel_1:
                            # extension level 1
                            cloud, processingType = ProcessGroups.converCPTforEPG(cloud, processingType, coreCount)
                        else:
                            # extension level 2
                            cloud, processingType = ProcessGroups.converCPTforEPG(cloud, processingType, coreCount, workingGroup)

                    # add cloud
                    if cloud not in ret:
                        ret[cloud] = {}
                    # add processingType
                    if processingType not in ret[cloud]:
                        ret[cloud][processingType] = {}
                    # add status
                    if jobStatus not in ret[cloud][processingType]:
                        ret[cloud][processingType][jobStatus] = 0
                    ret[cloud][processingType][jobStatus] += count
            # return
            _logger.debug(f"getJobStatisticsPerProcessingType -> {str(ret)}")
            return ret
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error(f"getJobStatisticsPerProcessingType : {type} {value}")
            return {}

    # update site data
    def updateSiteData(self, hostID, pilotRequests, interval):
        comment = " /* DBProxy.updateSiteData */"
        _logger.debug("updateSiteData start")

        sqlDel = "DELETE FROM ATLAS_PANDAMETA.SiteData WHERE LASTMOD<:LASTMOD"

        sqlRst = "UPDATE ATLAS_PANDAMETA.SiteData "
        sqlRst += "SET GETJOB=:GETJOB,UPDATEJOB=:UPDATEJOB,NOJOB=:NOJOB,GETJOBABS=:GETJOBABS,UPDATEJOBABS=:UPDATEJOBABS,NOJOBABS=:NOJOBABS "
        sqlRst += "WHERE HOURS=:HOURS AND LASTMOD<:LASTMOD"

        sqlCh = "SELECT count(*) FROM ATLAS_PANDAMETA.SiteData WHERE FLAG=:FLAG AND HOURS=:HOURS AND SITE=:SITE"

        sqlIn = "INSERT INTO ATLAS_PANDAMETA.SiteData (SITE,FLAG,HOURS,GETJOB,UPDATEJOB,NOJOB,GETJOBABS,UPDATEJOBABS,NOJOBABS,"
        sqlIn += "LASTMOD,NSTART,FINISHED,FAILED,DEFINED,ASSIGNED,WAITING,ACTIVATED,HOLDING,RUNNING,TRANSFERRING) "
        sqlIn += "VALUES (:SITE,:FLAG,:HOURS,:GETJOB,:UPDATEJOB,:NOJOB,:GETJOBABS,:UPDATEJOBABS,:NOJOBABS,CURRENT_DATE,"
        sqlIn += "0,0,0,0,0,0,0,0,0,0)"

        sqlUp = "UPDATE ATLAS_PANDAMETA.SiteData SET GETJOB=:GETJOB,UPDATEJOB=:UPDATEJOB,NOJOB=:NOJOB,"
        sqlUp += "GETJOBABS=:GETJOBABS,UPDATEJOBABS=:UPDATEJOBABS,NOJOBABS=:NOJOBABS,LASTMOD=CURRENT_DATE "
        sqlUp += "WHERE FLAG=:FLAG AND HOURS=:HOURS AND SITE=:SITE"

        sqlAll = "SELECT getJob,updateJob,noJob,getJobAbs,updateJobAbs,noJobAbs,FLAG "
        sqlAll += "FROM ATLAS_PANDAMETA.SiteData WHERE HOURS=:HOURS AND SITE=:SITE"

        try:
            timeNow = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            self.conn.begin()
            # delete old records
            varMap = {}
            varMap[":LASTMOD"] = timeNow - datetime.timedelta(hours=48)
            self.cur.execute(sqlDel + comment, varMap)
            # set 0 to old records
            varMap = {}
            varMap[":HOURS"] = interval
            varMap[":GETJOB"] = 0
            varMap[":UPDATEJOB"] = 0
            varMap[":NOJOB"] = 0
            varMap[":GETJOBABS"] = 0
            varMap[":UPDATEJOBABS"] = 0
            varMap[":NOJOBABS"] = 0
            varMap[":LASTMOD"] = timeNow - datetime.timedelta(hours=interval)
            self.cur.execute(sqlRst + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # shuffle to avoid concatenation
            tmpSiteList = list(pilotRequests)
            random.shuffle(tmpSiteList)
            # loop over all sites
            for tmpSite in tmpSiteList:
                tmpVal = pilotRequests[tmpSite]
                # start transaction
                self.conn.begin()
                # check individual host info first
                varMap = {}
                varMap[":FLAG"] = hostID
                varMap[":SITE"] = tmpSite
                varMap[":HOURS"] = interval
                self.cur.arraysize = 10
                self.cur.execute(sqlCh + comment, varMap)
                res = self.cur.fetchone()
                # row exists or not
                if res[0] == 0:
                    sql = sqlIn
                else:
                    sql = sqlUp

                # getJob, updateJob and noJob entries contain the number of slots/nodes that submitted the request
                # getJobAbs, updateJobAbs and noJobAbs entries contain the absolute number of requests
                if "getJob" in tmpVal:
                    varMap[":GETJOB"] = len(tmpVal["getJob"])
                    getJobAbs = 0
                    for node in tmpVal["getJob"]:
                        getJobAbs += tmpVal["getJob"][node]
                    varMap[":GETJOBABS"] = getJobAbs
                else:
                    varMap[":GETJOB"] = 0
                    varMap[":GETJOBABS"] = 0

                if "updateJob" in tmpVal:
                    varMap[":UPDATEJOB"] = len(tmpVal["updateJob"])
                    updateJobAbs = 0
                    for node in tmpVal["updateJob"]:
                        updateJobAbs += tmpVal["updateJob"][node]
                    varMap[":UPDATEJOBABS"] = updateJobAbs
                else:
                    varMap[":UPDATEJOB"] = 0
                    varMap[":UPDATEJOBABS"] = 0

                if "noJob" in tmpVal:
                    varMap[":NOJOB"] = len(tmpVal["noJob"])
                    noJobAbs = 0
                    for node in tmpVal["noJob"]:
                        noJobAbs += tmpVal["noJob"][node]
                    varMap[":NOJOBABS"] = noJobAbs
                else:
                    varMap[":NOJOB"] = 0
                    varMap[":NOJOBABS"] = 0

                # update
                self.cur.execute(sql + comment, varMap)
                # get all info
                sumExist = False
                varMap = {}
                varMap[":SITE"] = tmpSite
                varMap[":HOURS"] = interval
                self.cur.arraysize = 100
                self.cur.execute(sqlAll + comment, varMap)
                res = self.cur.fetchall()
                # get total getJob/updateJob
                varMap[":GETJOB"] = 0
                varMap[":UPDATEJOB"] = 0
                varMap[":NOJOB"] = 0
                varMap[":GETJOBABS"] = 0
                varMap[":UPDATEJOBABS"] = 0
                varMap[":NOJOBABS"] = 0
                nCol = 0
                for (
                    tmpGetJob,
                    tmpUpdateJob,
                    tmpNoJob,
                    tmpGetJobAbs,
                    tmpUpdateJobAbs,
                    tmpNoJobAbs,
                    tmpFlag,
                ) in res:
                    # don't use summed info
                    if tmpFlag == "production":
                        sumExist = True
                        continue
                    if tmpFlag == "analysis":
                        if tmpSite.startswith("ANALY_"):
                            sumExist = True
                        continue
                    if tmpFlag in ["test"]:
                        continue

                    if tmpGetJob is None:
                        tmpGetJob = 0
                    if tmpUpdateJob is None:
                        tmpUpdateJob = 0
                    if tmpNoJob is None:
                        tmpNoJob = 0
                    if tmpGetJobAbs is None:
                        tmpGetJobAbs = 0
                    if tmpUpdateJobAbs is None:
                        tmpUpdateJobAbs = 0
                    if tmpNoJobAbs is None:
                        tmpNoJobAbs = 0

                    # sum
                    varMap[":GETJOB"] += tmpGetJob
                    varMap[":UPDATEJOB"] += tmpUpdateJob
                    varMap[":NOJOB"] += tmpNoJob
                    varMap[":GETJOBABS"] += tmpGetJobAbs
                    varMap[":UPDATEJOBABS"] += tmpUpdateJobAbs
                    varMap[":NOJOBABS"] += tmpNoJobAbs
                    nCol += 1
                # get average
                if nCol != 0:
                    if varMap[":GETJOB"] >= nCol:
                        varMap[":GETJOB"] /= nCol
                    if varMap[":UPDATEJOB"] >= nCol:
                        varMap[":UPDATEJOB"] /= nCol
                    if varMap[":NOJOB"] >= nCol:
                        varMap[":NOJOB"] /= nCol
                    if varMap[":GETJOBABS"] >= nCol:
                        varMap[":GETJOBABS"] /= nCol
                    if varMap[":UPDATEJOBABS"] >= nCol:
                        varMap[":UPDATEJOBABS"] /= nCol
                    if varMap[":NOJOBABS"] >= nCol:
                        varMap[":NOJOBABS"] /= nCol

                if tmpSite.startswith("ANALY_"):
                    varMap[":FLAG"] = "analysis"
                else:
                    varMap[":FLAG"] = "production"
                # row exists or not
                if sumExist:
                    sql = sqlUp
                else:
                    sql = sqlIn
                # update
                self.cur.execute(sql + comment, varMap)
                _logger.debug(
                    "updateSiteData : %s hours=%s getJob=%s updateJob=%s, noJob=%s, getJobAbs=%s updateJobAbs=%s, noJobAbs=%s"
                    % (
                        tmpSite,
                        interval,
                        varMap[":GETJOB"],
                        varMap[":UPDATEJOB"],
                        varMap[":NOJOB"],
                        varMap[":GETJOBABS"],
                        varMap[":UPDATEJOBABS"],
                        varMap[":NOJOBABS"],
                    )
                )
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
            _logger.debug("updateSiteData done")
            return True
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error(f"updateSiteData : {type} {value}")
            return False

    # get site data
    def getCurrentSiteData(self):
        comment = " /* DBProxy.getCurrentSiteData */"
        _logger.debug("getCurrentSiteData")
        sql = "SELECT SITE,getJob,updateJob,FLAG FROM ATLAS_PANDAMETA.SiteData WHERE FLAG IN (:FLAG1,:FLAG2) and HOURS=3"
        varMap = {}
        varMap[":FLAG1"] = "production"
        varMap[":FLAG2"] = "analysis"
        try:
            # set autocommit on
            self.conn.begin()
            # select
            self.cur.arraysize = 10000
            self.cur.execute(sql + comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            ret = {}
            for site, getJob, updateJob, flag in res:
                if site.startswith("ANALY_"):
                    if flag != "analysis":
                        continue
                else:
                    if flag != "production":
                        continue
                ret[site] = {"getJob": getJob, "updateJob": updateJob}
            return ret
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error(f"getCurrentSiteData : {type} {value}")
            # roll back
            self._rollback()
            return {}

    # insert nRunning in site data
    def insertnRunningInSiteData(self):
        comment = " /* DBProxy.insertnRunningInSiteData */"
        _logger.debug("insertnRunningInSiteData start")
        sqlDel = "DELETE FROM ATLAS_PANDAMETA.SiteData WHERE FLAG IN (:FLAG1,:FLAG2) AND LASTMOD<CURRENT_DATE-1"
        sqlRun = "SELECT COUNT(*),computingSite FROM ATLAS_PANDA.jobsActive4 "
        sqlRun += "WHERE prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2) "
        sqlRun += "AND jobStatus=:jobStatus GROUP BY computingSite"
        sqlCh = "SELECT COUNT(*) FROM ATLAS_PANDAMETA.SiteData WHERE FLAG=:FLAG AND HOURS=:HOURS AND SITE=:SITE"
        sqlIn = "INSERT INTO ATLAS_PANDAMETA.SiteData (SITE,FLAG,HOURS,GETJOB,UPDATEJOB,LASTMOD,"
        sqlIn += "NSTART,FINISHED,FAILED,DEFINED,ASSIGNED,WAITING,ACTIVATED,HOLDING,RUNNING,TRANSFERRING) "
        sqlIn += "VALUES (:SITE,:FLAG,:HOURS,0,0,CURRENT_DATE,"
        sqlIn += "0,0,0,0,0,0,0,0,:RUNNING,0)"
        sqlUp = "UPDATE ATLAS_PANDAMETA.SiteData SET RUNNING=:RUNNING,LASTMOD=CURRENT_DATE "
        sqlUp += "WHERE FLAG=:FLAG AND HOURS=:HOURS AND SITE=:SITE"
        sqlMax = "SELECT SITE,MAX(RUNNING) FROM ATLAS_PANDAMETA.SiteData "
        sqlMax += "WHERE FLAG=:FLAG GROUP BY SITE"
        try:
            # use offset(1000)+minites for :HOURS
            timeNow = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            nHours = 1000 + timeNow.hour * 60 + timeNow.minute
            # delete old records
            varMap = {}
            varMap[":FLAG1"] = "max"
            varMap[":FLAG2"] = "snapshot"
            self.conn.begin()
            self.cur.execute(sqlDel + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # get nRunning
            varMap = {}
            varMap[":jobStatus"] = "running"
            varMap[":prodSourceLabel1"] = "user"
            varMap[":prodSourceLabel2"] = "panda"
            self.conn.begin()
            self.cur.arraysize = 10000
            self.cur.execute(sqlRun + comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # loop over all sites
            for nRunning, computingSite in res:
                # only ANALY_ sites
                if not computingSite.startswith("ANALY_"):
                    continue
                # check if the row is already there
                varMap = {}
                varMap[":FLAG"] = "snapshot"
                varMap[":SITE"] = computingSite
                varMap[":HOURS"] = nHours
                # start transaction
                self.conn.begin()
                self.cur.arraysize = 10
                self.cur.execute(sqlCh + comment, varMap)
                res = self.cur.fetchone()
                # row exists or not
                if res[0] == 0:
                    sql = sqlIn
                else:
                    sql = sqlUp
                # set current nRunning
                varMap = {}
                varMap[":FLAG"] = "snapshot"
                varMap[":SITE"] = computingSite
                varMap[":HOURS"] = nHours
                varMap[":RUNNING"] = nRunning
                # insert or update
                self.cur.execute(sql + comment, varMap)
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
            # get max nRunning
            varMap = {}
            varMap[":FLAG"] = "snapshot"
            self.conn.begin()
            self.cur.arraysize = 10000
            self.cur.execute(sqlMax + comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # loop over all sites
            for computingSite, maxnRunning in res:
                # start transaction
                self.conn.begin()
                # check if the row is already there
                varMap = {}
                varMap[":FLAG"] = "max"
                varMap[":SITE"] = computingSite
                varMap[":HOURS"] = 0
                self.cur.arraysize = 10
                self.cur.execute(sqlCh + comment, varMap)
                res = self.cur.fetchone()
                # row exists or not
                if res[0] == 0:
                    sql = sqlIn
                else:
                    sql = sqlUp
                # set max nRunning
                varMap = {}
                varMap[":FLAG"] = "max"
                varMap[":SITE"] = computingSite
                varMap[":HOURS"] = 0
                varMap[":RUNNING"] = maxnRunning
                self.cur.execute(sql + comment, varMap)
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
            _logger.debug("insertnRunningInSiteData done")
            return True
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error(f"insertnRunningInSiteData : {type} {value}")
            return False

    # get site info
    def getSiteInfo(self):
        comment = " /* DBProxy.getSiteInfo */"
        _logger.debug("getSiteInfo start")
        methodName = comment.split(" ")[-2].split(".")[-1]
        try:
            # get DDM endpoints
            pandaEndpointMap = self.getDdmEndpoints()

            # sql to get site spec
            sql = """
                   SELECT /* use_json_type */ panda_queue, data, b.site_name, c.role
                   FROM (ATLAS_PANDA.schedconfig_json a
                   LEFT JOIN ATLAS_PANDA.panda_site b ON a.panda_queue = b.panda_site_name)
                   LEFT JOIN ATLAS_PANDA.site c ON b.site_name = c.site_name
                   WHERE panda_queue IS NOT NULL
                   """
            self.cur.arraysize = 10000
            # self.cur.execute(sql+comment)
            # resList = self.cur.fetchall()
            ret, resList = self.getClobObj(sql, {})
            if not resList:
                _logger.error("Empty site list!")

            # set autocommit on
            self.conn.begin()
            # sql to get num slots
            sqlSL = "SELECT pandaQueueName, gshare, resourcetype, numslots FROM ATLAS_PANDA.Harvester_Slots "
            sqlSL += "WHERE (expirationTime IS NULL OR expirationTime>CURRENT_DATE) "

            num_slots_by_site = {}
            self.cur.execute(sqlSL + comment)
            resSL = self.cur.fetchall()

            for sl_queuename, sl_gshare, sl_resourcetype, sl_numslots in resSL:
                if sl_numslots < 0:
                    continue
                num_slots_by_site.setdefault(sl_queuename, {})
                num_slots_by_site[sl_queuename].setdefault(sl_gshare, {})
                num_slots_by_site[sl_queuename][sl_gshare][sl_resourcetype] = sl_numslots

            retList = {}
            if resList is not None:
                # loop over all results
                for res in resList:
                    try:  # don't let a problem with one queue break the whole map
                        # change None to ''
                        resTmp = []
                        for tmpItem in res:
                            if tmpItem is None:
                                tmpItem = ""
                            resTmp.append(tmpItem)

                        siteid, queue_data_json, pandasite, role = resTmp
                        try:
                            if isinstance(queue_data_json, dict):
                                queue_data = queue_data_json
                            else:
                                queue_data = json.loads(queue_data_json)
                        except Exception:
                            _logger.error(f"loading json for queue {siteid} excepted. json was: {queue_data_json}")
                            continue

                        # skip invalid siteid
                        if siteid in [None, "", "ALL"] or not queue_data:
                            if siteid != "ALL":  # skip noisy error message for ALL
                                _logger.error(f"siteid {siteid} had no queue_data {queue_data}")
                            continue

                        _logger.debug(f"processing queue {siteid}")

                        # instantiate SiteSpec
                        ret = SiteSpec.SiteSpec()
                        ret.sitename = siteid
                        ret.pandasite = pandasite
                        ret.role = role

                        ret.type = queue_data.get("type", "production")
                        ret.nickname = queue_data.get("nickname")
                        try:
                            ret.ddm = queue_data.get("ddm", "").split(",")[0]
                        except AttributeError:
                            ret.ddm = ""
                        try:
                            ret.cloud = queue_data.get("cloud", "").split(",")[0]
                        except AttributeError:
                            ret.cloud = ""
                        ret.memory = queue_data.get("memory")
                        ret.maxrss = queue_data.get("maxrss")
                        ret.minrss = queue_data.get("minrss")
                        ret.maxtime = queue_data.get("maxtime")
                        ret.status = queue_data.get("status")
                        ret.space = queue_data.get("space")
                        ret.maxinputsize = queue_data.get("maxinputsize")
                        ret.comment = queue_data.get("comment_")
                        ret.statusmodtime = queue_data.get("lastmod")
                        ret.catchall = queue_data.get("catchall")
                        ret.tier = queue_data.get("tier")
                        ret.jobseed = queue_data.get("jobseed")
                        ret.capability = queue_data.get("capability")
                        ret.workflow = queue_data.get("workflow")
                        ret.maxDiskio = queue_data.get("maxdiskio")
                        ret.pandasite_state = "ACTIVE"
                        ret.fairsharePolicy = queue_data.get("fairsharepolicy")
                        ret.defaulttoken = queue_data.get("defaulttoken")

                        ret.direct_access_lan = queue_data.get("direct_access_lan") is True
                        ret.direct_access_wan = queue_data.get("direct_access_wan") is True

                        ret.iscvmfs = queue_data.get("is_cvmfs") is True

                        if queue_data.get("corepower") is None:
                            ret.corepower = 0
                        else:
                            ret.corepower = queue_data.get("corepower")

                        ret.wnconnectivity = queue_data.get("wnconnectivity")
                        if ret.wnconnectivity == "":
                            ret.wnconnectivity = None

                        # maxwdir
                        try:
                            if queue_data.get("maxwdir") is None:
                                ret.maxwdir = 0
                            else:
                                ret.maxwdir = int(queue_data["maxwdir"])
                        except Exception:
                            if ret.maxinputsize in [0, None]:
                                ret.maxwdir = 0
                            else:
                                try:
                                    ret.maxwdir = ret.maxinputsize + 2000
                                except Exception:
                                    ret.maxwdir = 16336

                        # mintime
                        if queue_data.get("mintime") is not None:
                            ret.mintime = queue_data["mintime"]
                        else:
                            ret.mintime = 0

                        # reliability
                        ret.reliabilityLevel = None

                        # pledged CPUs
                        ret.pledgedCPU = 0
                        if queue_data.get("pledgedcpu") not in ["", None]:
                            try:
                                ret.pledgedCPU = int(queue_data["pledgedcpu"])
                            except Exception:
                                pass

                        # core count
                        ret.coreCount = 0
                        if queue_data.get("corecount") not in ["", None]:
                            try:
                                ret.coreCount = int(queue_data["corecount"])
                            except Exception:
                                pass

                        # convert releases to list
                        ret.releases = []
                        if queue_data.get("releases"):
                            ret.releases = queue_data["releases"]

                        # convert validatedreleases to list
                        ret.validatedreleases = []
                        if queue_data.get("validatedreleases"):
                            for tmpRel in queue_data["validatedreleases"].split("|"):
                                # remove white space
                                tmpRel = tmpRel.strip()
                                if tmpRel != "":
                                    ret.validatedreleases.append(tmpRel)

                        # limit of the number of transferring jobs
                        ret.transferringlimit = 0
                        if queue_data.get("transferringlimit") not in ["", None]:
                            try:
                                ret.transferringlimit = int(queue_data["transferringlimit"])
                            except Exception:
                                pass

                        # FAX
                        ret.allowfax = False
                        try:
                            if queue_data.get("catchall") is not None and "allowfax" in queue_data["catchall"]:
                                ret.allowfax = True
                            if queue_data.get("allowfax") is True:
                                ret.allowfax = True
                        except Exception:
                            pass

                        # DDM endpoints
                        ret.ddm_endpoints_input = {}
                        ret.ddm_endpoints_output = {}
                        if siteid in pandaEndpointMap:
                            for scope in pandaEndpointMap[siteid]:
                                if "input" in pandaEndpointMap[siteid][scope]:
                                    ret.ddm_endpoints_input[scope] = pandaEndpointMap[siteid][scope]["input"]
                                if "output" in pandaEndpointMap[siteid][scope]:
                                    ret.ddm_endpoints_output[scope] = pandaEndpointMap[siteid][scope]["output"]
                        else:
                            # empty
                            ret.ddm_endpoints_input["default"] = DdmSpec()
                            ret.ddm_endpoints_output["default"] = DdmSpec()

                        # initialize dictionary fields
                        ret.setokens_input = {}
                        ret.setokens_output = {}
                        ret.ddm_input = {}
                        for scope in ret.ddm_endpoints_input:
                            # mapping between token and endpoints
                            ret.setokens_input[scope] = ret.ddm_endpoints_input[scope].getTokenMap("input")
                            # set DDM to the default endpoint
                            ret.ddm_input[scope] = ret.ddm_endpoints_input[scope].getDefaultRead()

                        ret.ddm_output = {}
                        for scope in ret.ddm_endpoints_output:
                            # mapping between token and endpoints
                            ret.setokens_output[scope] = ret.ddm_endpoints_output[scope].getTokenMap("output")
                            # set DDM to the default endpoint
                            ret.ddm_output[scope] = ret.ddm_endpoints_output[scope].getDefaultWrite()

                        # object stores
                        try:
                            ret.objectstores = queue_data["objectstores"]
                        except Exception:
                            ret.objectstores = []

                        # default unified flag
                        ret.is_unified = False

                        # num slots
                        ret.num_slots_map = num_slots_by_site.get(siteid, {})

                        # append
                        retList[ret.nickname] = ret
                    except Exception:
                        _logger.error(f"getSiteInfo exception in queue: {traceback.format_exc()}")
                        continue
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            _logger.debug("getSiteInfo done")
            return retList
        except Exception as e:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return {}

    def getDdmEndpoints(self):
        """
        get list of ddm input endpoints
        """
        comment = " /* DBProxy.getDdmEndpoints */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        _logger.debug(f"{methodName} start")

        # get all ddm endpoints
        sql_ddm = "SELECT * FROM ATLAS_PANDA.ddm_endpoint "
        self.cur.arraysize = 10000
        self.cur.execute(f"{sql_ddm}{comment}")
        results_ddm = self.cur.fetchall()

        # extract the column names from the query
        column_names = [i[0].lower() for i in self.cur.description]

        # save the endpoints into a dictionary
        endpoint_dict = {}
        for ddm_endpoint_row in results_ddm:
            tmp_endpoint = {}
            # unzip the ddm_endpoint row into a dictionary
            for column_name, column_val in zip(column_names, ddm_endpoint_row):
                tmp_endpoint[column_name] = column_val

            # ignore TEST
            # if tmp_endpoint['type'] == 'TEST':
            #    continue

            endpoint_dict[tmp_endpoint["ddm_endpoint_name"]] = tmp_endpoint

        # get relationship between panda sites and ddm endpoints
        sql_panda_ddm = """
               SELECT pdr.panda_site_name, pdr.ddm_endpoint_name, pdr.is_local, de.ddm_spacetoken_name,
                      de.is_tape, pdr.default_read, pdr.default_write, pdr.roles, pdr.order_read, pdr.order_write,
                      nvl(pdr.scope, 'default') as scope, de.blacklisted_read
               FROM ATLAS_PANDA.panda_ddm_relation pdr, ATLAS_PANDA.ddm_endpoint de
               WHERE pdr.ddm_endpoint_name = de.ddm_endpoint_name
               """
        if self.backend == "mysql":
            sql_panda_ddm = """
               SELECT pdr.panda_site_name, pdr.ddm_endpoint_name, pdr.is_local, de.ddm_spacetoken_name,
                      de.is_tape, pdr.default_read, pdr.default_write, pdr.roles, pdr.order_read, pdr.order_write,
                      ifnull(pdr.scope, 'default') as scope, de.blacklisted
               FROM ATLAS_PANDA.panda_ddm_relation pdr, ATLAS_PANDA.ddm_endpoint de
               WHERE pdr.ddm_endpoint_name = de.ddm_endpoint_name
               """

        self.cur.execute(f"{sql_panda_ddm}{comment}")
        results_panda_ddm = self.cur.fetchall()
        column_names = [i[0].lower() for i in self.cur.description]

        # save the panda ddm relations into a dictionary
        panda_endpoint_map = {}
        for panda_ddm_row in results_panda_ddm:
            tmp_relation = {}
            for column_name, column_val in zip(column_names, panda_ddm_row):
                # Default unavailable endpoint space to 0
                if column_name.startswith("space_") and column_val is None:
                    column_val = 0
                tmp_relation[column_name] = column_val

            # add the relations to the panda endpoint map
            panda_site_name = tmp_relation["panda_site_name"]
            scope = tmp_relation["scope"]
            panda_endpoint_map.setdefault(panda_site_name, {})
            panda_endpoint_map[panda_site_name].setdefault(scope, {})

            if panda_site_name not in panda_endpoint_map:
                panda_endpoint_map[panda_site_name] = {
                    "input": DdmSpec(),
                    "output": DdmSpec(),
                }
            if "read_lan" in tmp_relation["roles"] and tmp_relation["blacklisted_read"] != "Y":
                panda_endpoint_map[panda_site_name][scope].setdefault("input", DdmSpec())
                panda_endpoint_map[panda_site_name][scope]["input"].add(tmp_relation, endpoint_dict)
            if "write_lan" in tmp_relation["roles"]:
                panda_endpoint_map[panda_site_name][scope].setdefault("output", DdmSpec())
                panda_endpoint_map[panda_site_name][scope]["output"].add(tmp_relation, endpoint_dict)

        _logger.debug(f"{methodName} done")
        return panda_endpoint_map

    # get dispatch sorting criteria
    def getSortingCriteria(self, site_name, max_jobs):
        method_name = "getSortingCriteria"
        # throw the dice to decide the algorithm
        random_number = random.randrange(100)

        sloppy_ratio = self.getConfigValue("jobdispatch", "SLOPPY_DISPATCH_RATIO")
        if not sloppy_ratio:
            sloppy_ratio = 10

        _logger.debug(f"{method_name} random_number: {random_number} sloppy_ratio: {sloppy_ratio}")

        if random_number <= sloppy_ratio:
            # generate the age sorting
            _logger.debug(f"{method_name} sorting by age")
            return self.getCriteriaByAge(site_name, max_jobs)
        else:
            # generate the global share sorting
            _logger.debug(f"{method_name} sorting by gshare")
            return self.getCriteriaForGlobalShares(site_name, max_jobs)

    # get selection criteria for share of production activities
    def getCriteriaForGlobalShares(self, site_name, max_jobs):
        method_name = "getCriteriaForGlobalShare"
        # return for no criteria
        var_map = {}
        ret_empty = "", {}

        try:
            # Get the share leaves sorted by order of under-pledging
            _logger.debug(f"{method_name} Going to call get sorted leaves")
            t_before = time.time()
            sorted_leaves = self.get_sorted_leaves()
            t_after = time.time()
            total = t_after - t_before
            _logger.debug(f"{method_name} Sorting leaves took {total}s")

            i = 0
            tmp_list = []
            for leave in sorted_leaves:
                var_map[f":leave{i}"] = leave.name
                if leave.name == "Test":
                    # Test share will bypass others for the moment
                    tmp_list.append(f"WHEN gshare=:leave{i} THEN 0")
                else:
                    tmp_list.append("WHEN gshare=:leave{0} THEN {0}".format(i))
                i += 1

            # Only get max_jobs, to avoid getting all activated jobs from the table
            var_map[":njobs"] = max_jobs

            # We want to sort by global share, highest priority and lowest pandaid
            leave_bindings = " ".join(tmp_list)
            ret_sql = f"""
                      ORDER BY (CASE {leave_bindings} ELSE {len(sorted_leaves)} END), currentpriority desc, pandaid asc)
                      WHERE ROWNUM <= :njobs
                      """

            _logger.debug(f"{method_name} ret_sql: {ret_sql}")
            _logger.debug(f"{method_name} var_map: {var_map}")
            return ret_sql, var_map

        except Exception:
            err_type, err_value = sys.exc_info()[:2]
            err_str = f"{method_name} {site_name} : {err_type} {err_value}"
            err_str.strip()
            err_str += traceback.format_exc()
            _logger.error(err_str)
            # roll back
            self._rollback()
            return ret_empty

    # get selection criteria for share of production activities
    def getCriteriaByAge(self, site_name, max_jobs):
        comment = " /* DBProxy.getCriteriaByAge */"
        # return for no criteria
        ret_sql = ""
        var_map = {}
        ret_empty = "", {}

        try:
            # Only get max_jobs, to avoid getting all activated jobs from the table
            var_map[":njobs"] = max_jobs

            # We want to ignore global share and just take the oldest pandaid
            ret_sql = """
                      ORDER BY pandaid asc)
                      WHERE ROWNUM <= :njobs
                      """

            _logger.debug(f"ret_sql: {ret_sql}")
            _logger.debug(f"var_map: {var_map}")
            return ret_sql, var_map

        except Exception:
            err_type, err_value = sys.exc_info()[:2]
            err_str = f"getCriteriaByAge {site_name} : {err_type} {err_value}"
            err_str.strip()
            err_str += traceback.format_exc()
            _logger.error(err_str)
            # roll back
            self._rollback()
            return ret_empty

    def get_cloud_list(self):
        """
        Get a list of distinct cloud names from the database.
        """
        comment = " /* DBProxy.get_cloud_list */"
        method_name = comment.split(" ")[-2].split(".")[-1]

        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        try:
            with self.conn:
                sql = (
                    f"SELECT /* use_json_type */ DISTINCT sj.data.cloud AS cloud "
                    f"FROM {panda_config.schemaPANDA}.schedconfig_json sj "
                    f"UNION "
                    f"SELECT 'WORLD' AS cloud "
                    f"FROM dual "
                    f"ORDER BY cloud"
                )
                self.cur.arraysize = 100
                self.cur.execute(sql + comment)
                results = self.cur.fetchall()
                clouds = [result[0] for result in results]

            tmp_log.debug("done")
            return clouds
        except Exception:
            type, value, traceBack = sys.exc_info()
            tmp_log.error(f"failed with: {type} {value}")
            self._rollback()
            return []

    # get special dispatcher parameters
    def get_special_dispatch_params(self):
        """
        Get the following special parameters for dispatcher.
          Authorized name lists for proxy, key-pair, and token-key retrieval
          Key pairs
          Token keys
        """
        comment = " /* DBProxy.get_special_dispatch_params */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        try:
            return_map = {}
            # set autocommit on
            self.conn.begin()
            self.cur.arraysize = 100000
            # get token keys
            token_keys = {}
            sql = f"SELECT dn, credname FROM {panda_config.schemaMETA}.proxykey WHERE expires>:limit ORDER BY expires DESC "
            var_map = {":limit": datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)}
            self.cur.execute(sql + comment, var_map)
            res_list = self.cur.fetchall()
            for client_name, token_key in res_list:
                token_keys.setdefault(client_name, {"fullList": [], "latest": token_key})
                token_keys[client_name]["fullList"].append(token_key)
            return_map["tokenKeys"] = token_keys
            tmp_list = [f"""{k}:{len(token_keys[k]["fullList"])}""" for k in token_keys]
            tmp_log.debug(f"""got token keys {",".join(tmp_list)}""")
            # select to get the list of authorized users
            allow_key = []
            allow_proxy = []
            allow_token = []
            sql = "SELECT DISTINCT name, gridpref FROM ATLAS_PANDAMETA.users " "WHERE (status IS NULL OR status<>:ngStatus) AND gridpref IS NOT NULL "
            var_map = {":ngStatus": "disabled"}
            self.cur.execute(sql + comment, var_map)
            res_list = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            for compactDN, gridpref in res_list:
                # users authorized for proxy retrieval
                if PrioUtil.PERMISSION_PROXY in gridpref:
                    if compactDN not in allow_proxy:
                        allow_proxy.append(compactDN)
                # users authorized for key-pair retrieval
                if PrioUtil.PERMISSION_KEY in gridpref:
                    if compactDN not in allow_key:
                        allow_key.append(compactDN)
                # users authorized for token-key retrieval
                if PrioUtil.PERMISSION_TOKEN_KEY in gridpref:
                    if compactDN not in allow_token:
                        allow_token.append(compactDN)
            return_map["allowKeyPair"] = allow_key
            return_map["allowProxy"] = allow_proxy
            return_map["allowTokenKey"] = allow_token
            tmp_log.debug(
                f"got authed users key-pair:{len(return_map['allowKeyPair'])}, proxy:{len(return_map['allowProxy'])}, token-key:{len(return_map['allowTokenKey'])}"
            )
            # read key pairs
            keyPair = {}
            try:
                keyFileNames = glob.glob(panda_config.keyDir + "/*")
                for keyName in keyFileNames:
                    tmpF = open(keyName)
                    keyPair[os.path.basename(keyName)] = tmpF.read()
                    tmpF.close()
            except Exception as e:
                tmp_log.error(f"failed read key-pairs with {str(e)}")
            return_map["keyPair"] = keyPair
            tmp_log.debug(f"got {len(return_map['keyPair'])} key-pair files")
            tmp_log.debug("done")
            return return_map
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, method_name)
            return {}

    # extract name from DN
    def cleanUserID(self, id):
        return CoreUtils.clean_user_id(id)

    # extract scope from dataset name
    def extractScope(self, name):
        try:
            if name.lower().startswith("user") or name.lower().startswith("group"):
                # return None if there are not enough fields
                if len(name.split(".")) < 2:
                    return None
                return name.lower().split(".")[0] + "." + name.lower().split(".")[1]
            return name.split(".")[0]
        except Exception:
            return None

    # check quota
    def checkQuota(self, dn):
        comment = " /* DBProxy.checkQuota */"
        _logger.debug(f"checkQuota {dn}")
        try:
            # set autocommit on
            self.conn.begin()
            # select
            name = self.cleanUserID(dn)
            sql = "SELECT cpua1, cpua7, cpua30, quotaa1, quotaa7, quotaa30 FROM ATLAS_PANDAMETA.users WHERE name=:name"
            varMap = {}
            varMap[":name"] = name
            self.cur.arraysize = 10
            self.cur.execute(sql + comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            weight = 0.0
            if res is not None and len(res) != 0:
                item = res[0]
                # cpu and quota
                cpu1 = item[0]
                cpu7 = item[1]
                cpu30 = item[2]
                if item[3] in [0, None]:
                    quota1 = 0
                else:
                    quota1 = item[3] * 3600
                if item[4] in [0, None]:
                    quota7 = 0
                else:
                    quota7 = item[4] * 3600
                if item[5] in [0, None]:
                    quota30 = 0
                else:
                    quota30 = item[5] * 3600
                # CPU usage
                if cpu1 is None:
                    cpu1 = 0.0
                # weight
                if quota1 > 0:
                    weight = float(cpu1) / float(quota1)
                # not exceeded the limit
                weight = 0.0
                _logger.debug(f"checkQuota {dn} Weight:{weight} Quota:{quota1} CPU:{cpu1}")
            else:
                _logger.debug(f"checkQuota cannot found {dn}")
            return weight
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error(f"checkQuota : {type} {value}")
            # roll back
            self._rollback()
            return 0.0

    # check if super user
    def isSuperUser(self, userName):
        comment = " /* DBProxy.isSuperUser */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" <{userName}>")
        tmpLog.debug("start")
        try:
            isSU = False
            isSG = False
            # start transaction
            self.conn.begin()
            # check gridpref
            name = self.cleanUserID(userName)
            sql = "SELECT gridpref FROM ATLAS_PANDAMETA.users WHERE name=:name"
            varMap = {}
            varMap[":name"] = name
            self.cur.arraysize = 10
            self.cur.execute(sql + comment, varMap)
            res = self.cur.fetchone()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # check if s in gridpref
            if res is not None:
                (gridpref,) = res
                if gridpref is not None:
                    if PrioUtil.PERMISSION_SUPER_USER in gridpref:
                        isSU = True
                    if PrioUtil.PERMISSION_SUPER_GROUP in gridpref:
                        isSG = True
            tmpLog.debug(f"done with superUser={isSU} superGroup={isSG}")
            return isSU, isSG
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False, False

    # get serialize JobID and status
    def getUserParameter(self, dn, jobID, jobsetID):
        comment = " /* DBProxy.getUserParameter */"
        _logger.debug(f"getUserParameter {dn} JobID={jobID} JobsetID={jobsetID}")
        try:
            # set initial values
            retStatus = True
            if jobsetID == -1:
                # generate new jobsetID
                retJobsetID = jobID
                # new jobID = 1 + new jobsetID
                retJobID = retJobsetID + 1
            elif jobsetID in ["NULL", None, 0]:
                # no jobsetID
                retJobsetID = None
                retJobID = jobID
            else:
                # user specified jobsetID
                retJobsetID = jobsetID
                retJobID = jobID
            # set autocommit on
            self.conn.begin()
            # select
            name = self.cleanUserID(dn)
            sql = "SELECT jobid,status FROM ATLAS_PANDAMETA.users WHERE name=:name "
            sql += "FOR UPDATE "
            sqlAdd = "INSERT INTO ATLAS_PANDAMETA.users "
            sqlAdd += "(ID,NAME,LASTMOD,FIRSTJOB,LATESTJOB,CACHETIME,NCURRENT,JOBID) "
            sqlAdd += "VALUES(ATLAS_PANDAMETA.USERS_ID_SEQ.nextval,:name,"
            sqlAdd += "CURRENT_DATE,CURRENT_DATE,CURRENT_DATE,CURRENT_DATE,0,1) "
            varMap = {}
            varMap[":name"] = name
            self.cur.execute(sql + comment, varMap)
            self.cur.arraysize = 10
            res = self.cur.fetchall()
            # insert if no record
            if res is None or len(res) == 0:
                try:
                    self.cur.execute(sqlAdd + comment, varMap)
                    retI = self.cur.rowcount
                    _logger.debug(f"getUserParameter {dn} inserted new row with {retI}")
                    # emulate DB response
                    res = [[1, ""]]
                except Exception:
                    errType, errValue = sys.exc_info()[:2]
                    _logger.error(f"getUserParameter {dn} failed to insert new row with {errType}:{errValue}")
            if res is not None and len(res) != 0:
                item = res[0]
                # JobID in DB
                dbJobID = item[0]
                # check status
                if item[1] in ["disabled"]:
                    retStatus = False
                # use larger JobID
                if dbJobID >= int(retJobID) or (jobsetID == -1 and dbJobID >= int(retJobsetID)):
                    if jobsetID == -1:
                        # generate new jobsetID = 1 + exsiting jobID
                        retJobsetID = dbJobID + 1
                        # new jobID = 1 + new jobsetID
                        retJobID = retJobsetID + 1
                    else:
                        # new jobID = 1 + exsiting jobID
                        retJobID = dbJobID + 1
                # update DB
                varMap = {}
                varMap[":name"] = name
                varMap[":jobid"] = retJobID
                sql = "UPDATE ATLAS_PANDAMETA.users SET jobid=:jobid WHERE name=:name"
                self.cur.execute(sql + comment, varMap)
                _logger.debug(f"getUserParameter set JobID={retJobID} for {dn}")
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            _logger.debug(f"getUserParameter {dn} return JobID={retJobID} JobsetID={retJobsetID} Status={retStatus}")
            return retJobID, retJobsetID, retStatus
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error(f"getUserParameter : {type} {value}")
            # roll back
            self._rollback()
            return retJobID, retJobsetID, retStatus

    # check ban user
    def checkBanUser(self, dn, sourceLabel, jediCheck=False):
        comment = " /* DBProxy.checkBanUser */"
        try:
            methodName = "checkBanUser"
            # set initial values
            retStatus = True
            name = self.cleanUserID(dn)
            methodName += f" {name}"
            _logger.debug(f"{methodName} start dn={dn} label={sourceLabel} jediCheck={jediCheck}")
            # set autocommit on
            self.conn.begin()
            # select
            sql = "SELECT status,dn FROM ATLAS_PANDAMETA.users WHERE name=:name"
            varMap = {}
            varMap[":name"] = name
            self.cur.execute(sql + comment, varMap)
            self.cur.arraysize = 10
            res = self.cur.fetchone()
            if res is not None:
                # check status
                tmpStatus, dnInDB = res
                if tmpStatus in ["disabled"]:
                    retStatus = False
                elif jediCheck and (dnInDB in ["", None] or dnInDB != dn):
                    # add DN
                    sqlUp = "UPDATE ATLAS_PANDAMETA.users SET dn=:dn WHERE name=:name "
                    varMap = {}
                    varMap[":name"] = name
                    varMap[":dn"] = dn
                    self.cur.execute(sqlUp + comment, varMap)
                    retI = self.cur.rowcount
                    _logger.debug(f"{methodName} update DN with Status={retI}")
                    if retI != 1:
                        retStatus = 1
            else:
                # new user
                if jediCheck:
                    name = self.cleanUserID(dn)
                    sqlAdd = "INSERT INTO ATLAS_PANDAMETA.users "
                    sqlAdd += "(ID,NAME,DN,LASTMOD,FIRSTJOB,LATESTJOB,CACHETIME,NCURRENT,JOBID) "
                    sqlAdd += "VALUES(ATLAS_PANDAMETA.USERS_ID_SEQ.nextval,:name,:dn,"
                    sqlAdd += "CURRENT_DATE,CURRENT_DATE,CURRENT_DATE,CURRENT_DATE,0,1) "
                    varMap = {}
                    varMap[":name"] = name
                    varMap[":dn"] = dn
                    self.cur.execute(sqlAdd + comment, varMap)
                    retI = self.cur.rowcount
                    _logger.debug(f"{methodName} inserted new row with Status={retI}")
                    if retI != 1:
                        retStatus = 2
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            _logger.debug(f"{methodName} done with Status={retStatus}")
            return retStatus
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return retStatus

    # get email address for a user
    def getEmailAddr(self, name, withDN=False, withUpTime=False):
        comment = " /* DBProxy.getEmailAddr */"
        _logger.debug(f"get email for {name}")
        # sql
        if withDN:
            failedRet = "", "", None
            sql = "SELECT email,dn,location FROM ATLAS_PANDAMETA.users WHERE name=:name"
        elif withUpTime:
            failedRet = "", None
            sql = "SELECT email,location FROM ATLAS_PANDAMETA.users WHERE name=:name"
        else:
            failedRet = ""
            sql = "SELECT email FROM ATLAS_PANDAMETA.users WHERE name=:name"
        try:
            # set autocommit on
            self.conn.begin()
            # select
            varMap = {}
            varMap[":name"] = name
            self.cur.execute(sql + comment, varMap)
            self.cur.arraysize = 10
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            if res is not None and len(res) != 0:
                if withDN or withUpTime:
                    if withDN:
                        email, dn, upTime = res[0]
                    else:
                        email, upTime = res[0]
                    # convert time
                    try:
                        upTime = datetime.datetime.strptime(upTime, "%Y-%m-%d %H:%M:%S")
                    except Exception:
                        upTime = None
                    if withDN:
                        return email, dn, upTime
                    else:
                        return email, upTime
                else:
                    return res[0][0]
            # return empty string
            return failedRet
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error(f"getEmailAddr : {type} {value}")
            # roll back
            self._rollback()
            return failedRet

    # set email address for a user
    def setEmailAddr(self, userName, emailAddr):
        comment = " /* DBProxy.setEmailAddr */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        _logger.debug(f"{methodName} {userName} to {emailAddr}")
        # sql
        sql = "UPDATE ATLAS_PANDAMETA.users SET email=:email,location=:uptime WHERE name=:name "
        try:
            # set autocommit on
            self.conn.begin()
            # set
            varMap = {}
            varMap[":name"] = userName
            varMap[":email"] = emailAddr
            varMap[":uptime"] = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None).strftime("%Y-%m-%d %H:%M:%S")
            self.cur.execute(sql + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False

    # get ban users
    def get_ban_users(self):
        comment = " /* DBProxy.get_ban_user */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        # sql
        sql = "SELECT name FROM ATLAS_PANDAMETA.users WHERE status=:status "
        try:
            # set autocommit on
            self.conn.begin()
            varMap = {}
            varMap[":status"] = "disabled"
            self.cur.execute(sql + comment, varMap)
            self.cur.arraysize = 10
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            retVal = {name: False for name, in res}
            tmpLog.debug(f"got {retVal}")
            return True, retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False, None

    # get client version
    def getPandaClientVer(self):
        comment = " /* DBProxy.getPandaClientVer */"
        _logger.debug("getPandaClientVer")
        try:
            # set autocommit on
            self.conn.begin()
            # select
            sql = "SELECT pathena FROM ATLAS_PANDAMETA.pandaconfig WHERE name=:name"
            varMap = {}
            varMap[":name"] = "current"
            self.cur.execute(sql + comment, varMap)
            self.cur.arraysize = 10
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            retStr = ""
            if res is not None and len(res) != 0:
                retStr = res[0][0]
            _logger.debug(f"getPandaClientVer -> {retStr}")
            return retStr
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error(f"getPandaClientVer : {type} {value}")
            return ""

    # register token key
    def register_token_key(self, client_name: str, lifetime: int) -> bool:
        """
        Register token key for a client with a lifetime and delete expired tokens

        :param client_name: client name who owns the token key
        :param lifetime: lifetime of the token key in hours

        :return: True if succeeded. False otherwise
        """
        comment = " /* DBProxy.register_token_key */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        method_name += f" < client={client_name} >"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        try:
            # set autocommit on
            self.conn.begin()
            time_now = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            expire_at = time_now + datetime.timedelta(hours=lifetime)
            # check if a new key was registered recently
            sql = f"SELECT 1 FROM {panda_config.schemaMETA}.proxykey WHERE dn=:dn AND expires>:limit "
            var_map = {":dn": client_name, ":limit": expire_at - datetime.timedelta(hours=1)}
            self.cur.execute(sql + comment, var_map)
            res = self.cur.fetchone()
            if res:
                tmp_log.debug("skip as a new key was registered recently")
            else:
                # get max ID
                sql = "SELECT MAX(ID) FROM ATLAS_PANDAMETA.proxykey "
                self.cur.execute(sql + comment, {})
                res = self.cur.fetchone()
                if not res:
                    max_id = 0
                else:
                    (max_id,) = res
                max_id += 1
                max_id %= 10000000
                # register a key
                sql = (
                    f"INSERT INTO {panda_config.schemaMETA}.proxykey (ID,DN,CREDNAME,CREATED,EXPIRES,ORIGIN,MYPROXY) "
                    "VALUES(:id,:dn,:credname,:created,:expires,:origin,:myproxy) "
                )
                var_map = {
                    ":id": max_id,
                    ":dn": client_name,
                    ":credname": str(uuid.uuid4()),
                    ":created": time_now,
                    ":expires": expire_at,
                    ":origin": "panda",
                    ":myproxy": "NA",
                }
                try:
                    self.cur.execute(sql + comment, var_map)
                    tmp_log.debug(f"registered a new key with id={max_id}")
                except Exception as e:
                    # ignore ID duplication error
                    tmp_log.debug(f"ignoring registration failure with {str(e)}")
            # delete obsolete keys
            sql = "DELETE FROM ATLAS_PANDAMETA.proxykey WHERE expires<:limit "
            var_map = {":limit": time_now}
            self.cur.execute(sql + comment, var_map)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # return True
            tmp_log.debug("done")
            return True
        except Exception:
            # roll back
            self._rollback()
            # dump error
            self.dumpErrorMessage(_logger, method_name)
            return False

    # get list of archived tables
    def getArchiveTables(self):
        # return
        return ["ATLAS_PANDAARCH.jobsArchived"]

    # peek at job
    def peekJobLog(self, pandaID, days=None):
        comment = " /* DBProxy.peekJobLog */"
        _logger.debug(f"peekJobLog : {pandaID} days={days}")
        # return None for NULL PandaID
        if pandaID in ["NULL", "", "None", None]:
            return None
        sql1_0 = "SELECT %s FROM %s "
        sql1_1 = "WHERE PandaID=:PandaID AND modificationTime>(CURRENT_DATE-:days) "
        # select
        varMap = {}
        varMap[":PandaID"] = pandaID
        if days is None:
            days = 30
        varMap[":days"] = days
        nTry = 1
        for iTry in range(nTry):
            try:
                # get list of archived tables
                tables = self.getArchiveTables()
                # select
                for table in tables:
                    # start transaction
                    self.conn.begin()
                    # select
                    sql = sql1_0 % (JobSpec.columnNames(), table) + sql1_1
                    self.cur.arraysize = 10
                    self.cur.execute(sql + comment, varMap)
                    res = self.cur.fetchall()
                    # commit
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    if len(res) != 0:
                        # Job
                        job = JobSpec()
                        job.pack(res[0])
                        # Files
                        # start transaction
                        self.conn.begin()
                        # select
                        fileTableName = re.sub("jobsArchived", "filesTable_ARCH", table)
                        sqlFile = f"SELECT /*+ INDEX(tab FILES_ARCH_PANDAID_IDX)*/ {FileSpec.columnNames()} "
                        sqlFile += f"FROM {fileTableName} tab "
                        # put constraint on modificationTime to avoid full table scan
                        sqlFile += "WHERE PandaID=:PandaID AND modificationTime>(CURRENT_DATE-:days)"
                        self.cur.arraysize = 10000
                        self.cur.execute(sqlFile + comment, varMap)
                        resFs = self.cur.fetchall()
                        # metadata
                        varMap = {}
                        varMap[":PandaID"] = job.PandaID
                        job.metadata = None
                        metaTableName = re.sub("jobsArchived", "metaTable_ARCH", table)
                        sqlMeta = f"SELECT metaData FROM {metaTableName} WHERE PandaID=:PandaID"
                        self.cur.execute(sqlMeta + comment, varMap)
                        for (clobMeta,) in self.cur:
                            if clobMeta is not None:
                                try:
                                    job.metadata = clobMeta.read()
                                except AttributeError:
                                    job.metadata = str(clobMeta)
                            break
                        # job parameters
                        job.jobParameters = None
                        jobParamTableName = re.sub("jobsArchived", "jobParamsTable_ARCH", table)
                        sqlJobP = f"SELECT jobParameters FROM {jobParamTableName} WHERE PandaID=:PandaID"
                        varMap = {}
                        varMap[":PandaID"] = job.PandaID
                        self.cur.execute(sqlJobP + comment, varMap)
                        for (clobJobP,) in self.cur:
                            if clobJobP is not None:
                                try:
                                    job.jobParameters = clobJobP.read()
                                except AttributeError:
                                    job.jobParameters = str(clobJobP)
                            break
                        # commit
                        if not self._commit():
                            raise RuntimeError("Commit error")
                        # set files
                        for resF in resFs:
                            file = FileSpec()
                            file.pack(resF)
                            # remove redundant white spaces
                            try:
                                file.md5sum = file.md5sum.strip()
                            except Exception:
                                pass
                            try:
                                file.checksum = file.checksum.strip()
                            except Exception:
                                pass
                            job.addFile(file)
                        return job
                _logger.debug(f"peekJobLog() : PandaID {pandaID} not found")
                return None
            except Exception:
                # roll back
                self._rollback()
                if iTry + 1 < nTry:
                    _logger.error(f"peekJobLog : {pandaID}")
                    time.sleep(random.randint(10, 20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error(f"peekJobLog : {type} {value}")
                # return None
                return None

    # record status change
    def recordStatusChange(self, pandaID, jobStatus, jobInfo=None, infoMap={}, useCommit=True, no_late_bulk_exec=True, extracted_sqls=None):
        comment = " /* DBProxy.recordStatusChange */"
        # check config
        if not hasattr(panda_config, "record_statuschange") or panda_config.record_statuschange is not True:
            return
        # get job info
        varMap = {}
        varMap[":PandaID"] = pandaID
        varMap[":jobStatus"] = jobStatus
        varMap[":modificationHost"] = self.myHostName
        if jobInfo is not None:
            varMap[":computingSite"] = jobInfo.computingSite
            varMap[":cloud"] = jobInfo.cloud
            varMap[":prodSourceLabel"] = jobInfo.prodSourceLabel
        elif infoMap is not None:
            varMap[":computingSite"] = infoMap["computingSite"]
            varMap[":cloud"] = infoMap["cloud"]
            varMap[":prodSourceLabel"] = infoMap["prodSourceLabel"]
        else:
            # no info
            return
        # convert NULL to None
        for tmpKey in varMap:
            if varMap[tmpKey] == "NULL":
                varMap[tmpKey] = None
        # insert
        sql = "INSERT INTO ATLAS_PANDA.jobs_StatusLog "
        sql += "(PandaID,modificationTime,jobStatus,prodSourceLabel,cloud,computingSite,modificationHost,modiftime_extended) "
        sql += "VALUES (:PandaID,CURRENT_DATE,:jobStatus,:prodSourceLabel,:cloud,:computingSite,:modificationHost,CURRENT_TIMESTAMP) "
        try:
            # start transaction
            if no_late_bulk_exec:
                if useCommit:
                    self.conn.begin()
                self.cur.execute(sql + comment, varMap)
                # commit
                if useCommit:
                    if not self._commit():
                        raise RuntimeError("Commit error")
            else:
                extracted_sqls.setdefault("state_change", {"sql": sql + comment, "vars": []})
                extracted_sqls["state_change"]["vars"].append(varMap)
        except Exception:
            # roll back
            if useCommit and no_late_bulk_exec:
                self._rollback()
            errType, errValue = sys.exc_info()[:2]
            _logger.error(f"recordStatusChange {pandaID} {jobStatus}: {errType} {errValue}")
            if not useCommit:
                raise RuntimeError("recordStatusChange failed")
        return

    def push_job_status_message(
        self,
        job_spec,
        panda_id,
        status,
        jedi_task_id=None,
        special_handling=None,
        extra_data=None,
    ):
        if not (hasattr(panda_config, "mq_configFile") and panda_config.mq_configFile):
            # skip if not configured
            return
        to_push = False
        if special_handling is not None:
            to_push = push_status_changes(special_handling)
        elif job_spec is not None:
            to_push = job_spec.push_status_changes()
        # only run if to push status change
        if not to_push:
            return
        # skip statuses unnecessary to push
        if status in ["sent", "holding", "merging"]:
            return
        # skip if no mb to push to
        mb_proxy = self.get_mb_proxy("panda_jobstatus")
        if not mb_proxy:
            return
        if to_push:
            # push job status change
            try:
                now_time = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
                now_ts = int(now_time.timestamp())
                # init
                inputs = []
                computingsite = None
                error_tmp_dict = {}
                # info from job spec
                if job_spec is not None:
                    # task id
                    if jedi_task_id is None:
                        jedi_task_id = job_spec.jediTaskID
                    # inputs
                    if job_spec.Files is not None:
                        for file_spec in job_spec.Files:
                            if file_spec.type in ["input", "pseudo_input"]:
                                inputs.append(file_spec.lfn)
                    # computing site
                    if job_spec.computingSite is not None:
                        computingsite = job_spec.computingSite
                    # error codes and diags
                    error_tmp_dict["piloterrorcode"] = job_spec.pilotErrorCode
                    error_tmp_dict["exeerrorcode"] = job_spec.exeErrorCode
                    error_tmp_dict["superrorcode"] = job_spec.supErrorCode
                    error_tmp_dict["ddmerrorcode"] = job_spec.ddmErrorCode
                    error_tmp_dict["brokerageerrorcode"] = job_spec.brokerageErrorCode
                    error_tmp_dict["jobdispatchererrorcode"] = job_spec.jobDispatcherErrorCode
                    error_tmp_dict["taskbuffererrorcode"] = job_spec.taskBufferErrorCode
                    error_tmp_dict["piloterrordiag"] = job_spec.pilotErrorDiag
                    error_tmp_dict["exeerrordiag"] = job_spec.exeErrorDiag
                    error_tmp_dict["superrordiag"] = job_spec.supErrorDiag
                    error_tmp_dict["ddmerrordiag"] = job_spec.ddmErrorDiag
                    error_tmp_dict["brokerageerrordiag"] = job_spec.brokerageErrorDiag
                    error_tmp_dict["jobdispatchererrordiag"] = job_spec.jobDispatcherErrorDiag
                    error_tmp_dict["taskbuffererrordiag"] = job_spec.taskBufferErrorDiag
                # message
                orig_msg_dict = {
                    "msg_type": "job_status",
                    "jobid": panda_id,
                    "taskid": jedi_task_id,
                    "status": status,
                    "timestamp": now_ts,
                }
                update_msg_dict = {
                    "computingsite": computingsite,
                    "inputs": inputs if inputs else None,
                }
                update_msg_dict.update(error_tmp_dict)
                msg_dict = update_msg_dict.copy()
                if extra_data:
                    msg_dict.update(extra_data)
                msg_dict.update(orig_msg_dict)
                msg = json.dumps(msg_dict)
                if mb_proxy.got_disconnected:
                    mb_proxy.restart()
                mb_proxy.send(msg)
                _logger.debug(f"push_job_status_message: sent message: {msg}")
            except Exception:
                errType, errValue = sys.exc_info()[:2]
                _logger.error(f"push_job_status_message {panda_id} {status}: {errType} {errValue}")

    # propagate result to JEDI
    def propagateResultToJEDI(
        self,
        jobSpec,
        cur,
        oldJobStatus=None,
        extraInfo=None,
        finishPending=False,
        waitLock=False,
        async_params=None,
    ):
        comment = " /* DBProxy.propagateResultToJEDI */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" < PandaID={jobSpec.PandaID} jediTaskID={jobSpec.jediTaskID} >"
        tmpLog = LogWrapper(_logger, methodName)
        datasetContentsStat = {}
        # loop over all files
        finishUnmerge = set()
        hasInput = False
        tmpLog.debug(f"waitLock={waitLock} async_params={async_params}")
        # make pseudo files for dynamic number of events
        if EventServiceUtils.isDynNumEventsSH(jobSpec.specialHandling):
            pseudoFiles = self.create_pseudo_files_for_dyn_num_events(jobSpec, tmpLog)
        else:
            pseudoFiles = []
        # flag for job cloning
        useJobCloning = False
        if EventServiceUtils.isEventServiceJob(jobSpec) and EventServiceUtils.isJobCloningJob(jobSpec):
            useJobCloning = True
        # set delete flag to events
        if (EventServiceUtils.isEventServiceJob(jobSpec) or EventServiceUtils.isEventServiceMerge(jobSpec)) and jobSpec.jobStatus in [
            "finished",
            "failed",
            "cancelled",
        ]:
            # sql to check attemptNr
            sqlDelC = f"SELECT attemptNr FROM {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
            sqlDelC += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            # sql to set delete flag
            sqlDelE = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlDelE += f"{panda_config.schemaJEDI}.JEDI_Events tab "
            sqlDelE += "SET file_not_deleted=CASE WHEN objStore_ID IS NULL THEN NULL ELSE :delFlag END "
            if jobSpec.jobStatus == "finished":
                sqlDelE += ",status=CASE WHEN status=:st_done THEN :st_merged ELSE status END "
            sqlDelE += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            for fileSpec in jobSpec.Files:
                if fileSpec.type not in ["input", "pseudo_input"]:
                    continue
                # check attemptNr
                varMap = {}
                varMap[":jediTaskID"] = fileSpec.jediTaskID
                varMap[":datasetID"] = fileSpec.datasetID
                varMap[":fileID"] = fileSpec.fileID
                self.cur.execute(sqlDelC + comment, varMap)
                (tmpAttemptNr,) = self.cur.fetchone()
                if fileSpec.attemptNr != tmpAttemptNr:
                    tmpLog.debug(f"skip to set Y for datasetID={fileSpec.datasetID} fileID={fileSpec.fileID} due to attemprNr mismatch")
                    continue
                # set del flag
                varMap = {}
                varMap[":jediTaskID"] = fileSpec.jediTaskID
                varMap[":datasetID"] = fileSpec.datasetID
                varMap[":fileID"] = fileSpec.fileID
                varMap[":delFlag"] = "Y"
                if jobSpec.jobStatus == "finished":
                    varMap[":st_done"] = EventServiceUtils.ST_done
                    varMap[":st_merged"] = EventServiceUtils.ST_merged
                tmpLog.debug(sqlDelE + comment + str(varMap))
                self.cur.execute(sqlDelE + comment, varMap)
                retDelE = self.cur.rowcount
                tmpLog.debug(f"set Y to {retDelE} event ranges")
        # loop over all files to update file or dataset attributes
        for fileSpec in jobSpec.Files + pseudoFiles:
            # skip if no JEDI
            if fileSpec.fileID == "NULL":
                continue
            # do nothing for unmerged output/log files when merged job successfully finishes,
            # since they were already updated by merged job
            if jobSpec.jobStatus == "finished" and fileSpec.isUnMergedOutput():
                continue
            # check file status
            varMap = {}
            varMap[":fileID"] = fileSpec.fileID
            varMap[":datasetID"] = fileSpec.datasetID
            varMap[":jediTaskID"] = jobSpec.jediTaskID
            # no attemptNr check for premerge since attemptNr can be incremented by pmerge
            if not (jobSpec.isCancelled() and fileSpec.isUnMergedOutput()):
                varMap[":attemptNr"] = fileSpec.attemptNr
            sqlFileStat = "SELECT status,is_waiting FROM ATLAS_PANDA.JEDI_Dataset_Contents "
            sqlFileStat += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            if not (jobSpec.isCancelled() and fileSpec.isUnMergedOutput()):
                sqlFileStat += "AND attemptNr=:attemptNr "
            sqlFileStat += "FOR UPDATE "
            if not waitLock:
                sqlFileStat += "NOWAIT "
            n_try = 5
            for i_try in range(n_try):
                try:
                    tmpLog.debug(f"Trying to lock file {i_try+1}/{n_try} sql:{sqlFileStat} var:{str(varMap)}")
                    cur.execute(sqlFileStat + comment, varMap)
                    break
                except Exception as e:
                    if i_try + 1 == n_try:
                        raise e
                    time.sleep(1)
            resFileStat = self.cur.fetchone()
            if resFileStat is not None:
                oldFileStatus, oldIsWaiting = resFileStat
            else:
                oldFileStatus, oldIsWaiting = None, None
            # skip if already cancelled
            if oldFileStatus in ["cancelled"]:
                continue
            # update Dataset Contents table
            updateMetadata = False
            updateAttemptNr = False
            updateNumEvents = False
            updateFailedAttempt = False
            varMap = {}
            varMap[":fileID"] = fileSpec.fileID
            varMap[":datasetID"] = fileSpec.datasetID
            varMap[":keepTrack"] = 1
            varMap[":jediTaskID"] = jobSpec.jediTaskID
            if not (jobSpec.isCancelled() and fileSpec.isUnMergedOutput()):
                varMap[":attemptNr"] = fileSpec.attemptNr
            # set file status
            if fileSpec.type in ["input", "pseudo_input"]:
                hasInput = True
                updateAttemptNr = True
                if (
                    (
                        (jobSpec.jobStatus == "finished" and not EventServiceUtils.is_fine_grained_job(jobSpec))
                        or (jobSpec.jobSubStatus == "fg_done" and EventServiceUtils.is_fine_grained_job(jobSpec))
                    )
                    and not jobSpec.is_hpo_workflow()
                    and fileSpec.status != "skipped"
                ):
                    varMap[":status"] = "finished"
                    if fileSpec.type in ["input", "pseudo_input"]:
                        updateNumEvents = True
                else:
                    # set ready for next attempt
                    varMap[":status"] = "ready"
                    if jobSpec.jobStatus == "failed" and not jobSpec.is_hpo_workflow():
                        updateFailedAttempt = True
            else:
                if jobSpec.isCancelled():
                    # use only cancelled for all flavors
                    varMap[":status"] = "cancelled"
                else:
                    varMap[":status"] = jobSpec.jobStatus
                if fileSpec.status == "nooutput":
                    varMap[":status"] = fileSpec.status
                elif jobSpec.jobStatus == "finished":
                    varMap[":status"] = "finished"
                    # update metadata
                    updateMetadata = True
                    # update nEvents
                    updateNumEvents = True
                elif fileSpec.status == "merging":
                    # set ready to merge files for failed jobs
                    varMap[":status"] = "ready"
                    # update metadata
                    updateMetadata = True
            sqlFile = "UPDATE /*+ INDEX_RS_ASC(JEDI_DATASET_CONTENTS (JEDI_DATASET_CONTENTS.JEDITASKID JEDI_DATASET_CONTENTS.DATASETID JEDI_DATASET_CONTENTS.FILEID)) */ ATLAS_PANDA.JEDI_Dataset_Contents SET status=:status,is_waiting=NULL"
            # attempt number
            if updateAttemptNr is True:
                # increment attemptNr for next attempt
                if not jobSpec.is_hpo_workflow():
                    sqlFile += ",attemptNr=attemptNr+1"
                else:
                    sqlFile += ",attemptNr=MOD(attemptNr+1,maxAttempt)"
            # failed attempts
            if updateFailedAttempt is True:
                sqlFile += ",failedAttempt=failedAttempt+1"
            # set correct PandaID for job cloning
            if useJobCloning:
                varMap[":PandaID"] = jobSpec.PandaID
                if fileSpec.type in ["log", "output"]:
                    sqlFile += ",outPandaID=:PandaID,PandaID=:PandaID"
                else:
                    sqlFile += ",PandaID=:PandaID"
            # metadata
            if updateMetadata:
                # set file metadata
                for tmpKey in ["lfn", "GUID", "fsize", "checksum"]:
                    tmpVal = getattr(fileSpec, tmpKey)
                    if tmpVal == "NULL":
                        if tmpKey in fileSpec._zeroAttrs:
                            tmpVal = 0
                        else:
                            tmpVal = None
                    tmpMapKey = f":{tmpKey}"
                    sqlFile += f",{tmpKey}={tmpMapKey}"
                    varMap[tmpMapKey] = tmpVal
                # extra metadata
                if extraInfo is not None:
                    # nevents
                    if "nevents" in extraInfo and fileSpec.lfn in extraInfo["nevents"]:
                        tmpKey = "nEvents"
                        tmpMapKey = f":{tmpKey}"
                        sqlFile += f",{tmpKey}={tmpMapKey}"
                        varMap[tmpMapKey] = extraInfo["nevents"][fileSpec.lfn]
                    # LB number
                    if "lbnr" in extraInfo and fileSpec.lfn in extraInfo["lbnr"]:
                        tmpKey = "lumiBlockNr"
                        tmpMapKey = f":{tmpKey}"
                        sqlFile += f",{tmpKey}={tmpMapKey}"
                        varMap[tmpMapKey] = extraInfo["lbnr"][fileSpec.lfn]
                # reset keepTrack unless merging
                if fileSpec.status != "merging":
                    sqlFile += ",keepTrack=NULL"
                else:
                    # set boundaryID for merging
                    sqlFile += ",boundaryID=:boundaryID"
                    varMap[":boundaryID"] = jobSpec.PandaID
                    # set max attempt
                    sqlFile += ",maxAttempt=attemptNr+3"
            sqlFile += " WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            sqlFile += "AND keepTrack=:keepTrack "
            if not (jobSpec.isCancelled() and fileSpec.isUnMergedOutput()):
                sqlFile += "AND attemptNr=:attemptNr "
            tmpLog.debug(sqlFile + comment + str(varMap))
            cur.execute(sqlFile + comment, varMap)
            nRow = cur.rowcount
            if nRow == 1 and fileSpec.status not in ["nooutput"]:
                datasetID = fileSpec.datasetID
                fileStatus = varMap[":status"]
                if datasetID not in datasetContentsStat:
                    datasetContentsStat[datasetID] = {
                        "nFilesUsed": 0,
                        "nFilesFinished": 0,
                        "nFilesFailed": 0,
                        "nFilesOnHold": 0,
                        "nFilesTobeUsed": 0,
                        "nEvents": 0,
                        "nEventsUsed": 0,
                        "nFilesWaiting": 0,
                    }
                # read nEvents
                if updateNumEvents:
                    sqlEVT = "SELECT nEvents,startEvent,endEvent,keepTrack FROM ATLAS_PANDA.JEDI_Dataset_Contents "
                    sqlEVT += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                    if not waitLock:
                        sqlEVT += "FOR UPDATE NOWAIT "
                    varMap = {}
                    varMap[":fileID"] = fileSpec.fileID
                    varMap[":datasetID"] = fileSpec.datasetID
                    varMap[":jediTaskID"] = jobSpec.jediTaskID
                    tmpLog.debug(sqlEVT + comment + str(varMap))
                    cur.execute(sqlEVT + comment, varMap)
                    resEVT = self.cur.fetchone()
                    if resEVT is not None:
                        tmpNumEvents, tmpStartEvent, tmpEndEvent, tmpKeepTrack = resEVT
                        if tmpNumEvents is not None:
                            try:
                                if fileSpec.type in ["input", "pseudo_input"]:
                                    if tmpKeepTrack == 1:
                                        # keep track on how many events successfully used
                                        if tmpStartEvent is not None and tmpEndEvent is not None:
                                            datasetContentsStat[datasetID]["nEventsUsed"] += tmpEndEvent - tmpStartEvent + 1
                                        else:
                                            datasetContentsStat[datasetID]["nEventsUsed"] += tmpNumEvents
                                else:
                                    datasetContentsStat[datasetID]["nEvents"] += tmpNumEvents
                            except Exception:
                                pass
                # update file counts
                isDone = False
                if fileSpec.status == "merging" and (finishPending or jobSpec.prodSourceLabel not in ["user", "panda"]):
                    # files to be merged for pending failed jobs
                    datasetContentsStat[datasetID]["nFilesOnHold"] += 1
                elif fileStatus == "ready":
                    # check attemptNr and maxAttempt when the file failed (ready = input failed)
                    # skip secondary datasets which have maxAttempt=None
                    sqlAttNr = "SELECT attemptNr,maxAttempt,failedAttempt,maxFailure FROM ATLAS_PANDA.JEDI_Dataset_Contents "
                    sqlAttNr += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                    varMap = {}
                    varMap[":fileID"] = fileSpec.fileID
                    varMap[":datasetID"] = fileSpec.datasetID
                    varMap[":jediTaskID"] = jobSpec.jediTaskID
                    tmpLog.debug(sqlAttNr + comment + str(varMap))
                    cur.execute(sqlAttNr + comment, varMap)
                    resAttNr = self.cur.fetchone()
                    if resAttNr is not None:
                        newAttemptNr, maxAttempt, failedAttempt, maxFailure = resAttNr
                        if maxAttempt is not None:
                            if maxAttempt > newAttemptNr and (maxFailure is None or maxFailure > failedAttempt):
                                if oldFileStatus == "ready":
                                    # don't change nFilesUsed when fake co-jumbo is done
                                    pass
                                elif fileSpec.status != "merging":
                                    # decrement nUsed to trigger reattempt
                                    datasetContentsStat[datasetID]["nFilesUsed"] -= 1
                                else:
                                    # increment nTobeUsed to trigger merging
                                    datasetContentsStat[datasetID]["nFilesTobeUsed"] += 1
                            else:
                                # no more reattempt
                                datasetContentsStat[datasetID]["nFilesFailed"] += 1
                                isDone = True
                                # merge job failed
                                if jobSpec.processingType == "pmerge":
                                    # update unmerged file
                                    sqlUmFile = "UPDATE ATLAS_PANDA.JEDI_Dataset_Contents SET status=:status,keepTrack=NULL "
                                    sqlUmFile += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                                    varMap = {}
                                    varMap[":fileID"] = fileSpec.fileID
                                    varMap[":datasetID"] = fileSpec.datasetID
                                    varMap[":jediTaskID"] = jobSpec.jediTaskID
                                    varMap[":status"] = "notmerged"
                                    tmpLog.debug(sqlUmFile + comment + str(varMap))
                                    cur.execute(sqlUmFile + comment, varMap)
                                    # set flag to update unmerged jobs
                                    finishUnmerge.add(fileSpec.fileID)
                elif fileStatus in ["finished", "lost"]:
                    # successfully used or produced, or lost
                    datasetContentsStat[datasetID]["nFilesFinished"] += 1
                    isDone = True
                else:
                    # failed to produce the file
                    datasetContentsStat[datasetID]["nFilesFailed"] += 1
                # changed from transferring
                if fileSpec.type in ["input", "pseudo_input"]:
                    if oldJobStatus == "transferring":
                        datasetContentsStat[datasetID]["nFilesOnHold"] -= 1
                # reset is_waiting
                if oldIsWaiting is not None:
                    datasetContentsStat[datasetID]["nFilesWaiting"] -= 1
                    if isDone:
                        datasetContentsStat[datasetID]["nFilesUsed"] += 1
                # killed during merging
                if jobSpec.isCancelled() and oldJobStatus == "merging" and fileSpec.isUnMergedOutput():
                    # get corresponding sub
                    varMap = {}
                    varMap[":pandaID"] = jobSpec.PandaID
                    varMap[":fileID"] = fileSpec.fileID
                    varMap[":datasetID"] = fileSpec.datasetID
                    varMap[":jediTaskID"] = jobSpec.jediTaskID
                    sqlGetDest = "SELECT destinationDBlock FROM ATLAS_PANDA.filesTable4 "
                    sqlGetDest += "WHERE pandaID=:pandaID AND jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                    tmpLog.debug(sqlGetDest + comment + str(varMap))
                    cur.execute(sqlGetDest + comment, varMap)
                    (preMergedDest,) = self.cur.fetchone()
                    # check if corresponding sub is closed
                    varMap = {}
                    varMap[":name"] = preMergedDest
                    varMap[":subtype"] = "sub"
                    sqlCheckDest = "SELECT status FROM ATLAS_PANDA.Datasets "
                    sqlCheckDest += "WHERE name=:name AND subtype=:subtype "
                    tmpLog.debug(sqlCheckDest + comment + str(varMap))
                    cur.execute(sqlCheckDest + comment, varMap)
                    tmpResDestStat = self.cur.fetchone()
                    if tmpResDestStat is not None:
                        (preMergedDestStat,) = tmpResDestStat
                    else:
                        preMergedDestStat = "notfound"
                        tmpLog.debug(f"{preMergedDest} not found for datasetID={datasetID}")
                    if preMergedDestStat not in ["tobeclosed", "completed"]:
                        datasetContentsStat[datasetID]["nFilesOnHold"] -= 1
                    else:
                        tmpLog.debug(f"not change nFilesOnHold for datasetID={datasetID} since sub is in {preMergedDestStat}")
                        # increment nUsed when mergeing is killed before merge job is generated
                        if oldFileStatus == "ready":
                            datasetContentsStat[datasetID]["nFilesUsed"] += 1
        # update JEDI_Datasets table
        nOutEvents = 0
        if datasetContentsStat != {}:
            tmpDatasetIDs = sorted(datasetContentsStat)
            for tmpDatasetID in tmpDatasetIDs:
                tmpLog.debug(f"trying to lock datasetID={tmpDatasetID}")
                tmpContentsStat = datasetContentsStat[tmpDatasetID]
                sqlJediDL = "SELECT nFilesUsed,nFilesFailed,nFilesTobeUsed,nFilesFinished," "nFilesOnHold,type,masterID,status FROM ATLAS_PANDA.JEDI_Datasets "
                sqlJediDL += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                sqlJediDLnoL = sqlJediDL
                sqlJediDL += "FOR UPDATE "
                if not waitLock:
                    sqlJediDL += "NOWAIT "
                varMap = {}
                varMap[":jediTaskID"] = jobSpec.jediTaskID
                varMap[":datasetID"] = tmpDatasetID
                if async_params is None:
                    cur.execute(sqlJediDL + comment, varMap)
                else:
                    cur.execute(sqlJediDLnoL + comment, varMap)
                tmpResJediDL = self.cur.fetchone()
                (
                    t_nFilesUsed,
                    t_nFilesFailed,
                    t_nFilesTobeUsed,
                    t_nFilesFinished,
                    t_nFilesOnHold,
                    t_type,
                    t_masterID,
                    t_status,
                ) = tmpResJediDL
                tmpLog.debug(
                    f"datasetID={tmpDatasetID} had nFilesTobeUsed={t_nFilesTobeUsed} "
                    f"nFilesUsed={t_nFilesUsed} nFilesFinished={t_nFilesFinished} "
                    f"nFilesFailed={t_nFilesFailed} status={t_status}"
                )
                if async_params is not None:
                    self.insert_to_query_pool(
                        SQL_QUEUE_TOPIC_async_dataset_update,
                        async_params["PandaID"],
                        async_params["jediTaskID"],
                        sqlJediDL,
                        varMap,
                        async_params["exec_order"],
                    )
                    async_params["exec_order"] += 1
                # sql to update nFiles info
                toUpdateFlag = False
                eventsToRead = False
                sqlJediDS = "UPDATE ATLAS_PANDA.JEDI_Datasets SET "
                for tmpStatKey in tmpContentsStat:
                    tmpStatVal = tmpContentsStat[tmpStatKey]
                    if tmpStatVal == 0:
                        continue
                    if tmpStatVal > 0:
                        sqlJediDS += f"{tmpStatKey}={tmpStatKey}+{tmpStatVal},"
                    else:
                        sqlJediDS += f"{tmpStatKey}={tmpStatKey}-{abs(tmpStatVal)},"
                    toUpdateFlag = True
                    if tmpStatKey == "nEvents" and tmpStatVal > nOutEvents:
                        nOutEvents = tmpStatVal
                sqlJediDS = sqlJediDS[:-1]
                sqlJediDS += " WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                varMap = {}
                varMap[":jediTaskID"] = jobSpec.jediTaskID
                varMap[":datasetID"] = tmpDatasetID
                # update
                if toUpdateFlag:
                    tmpLog.debug(sqlJediDS + comment + str(varMap))
                    if async_params is not None:
                        self.insert_to_query_pool(
                            SQL_QUEUE_TOPIC_async_dataset_update,
                            async_params["PandaID"],
                            async_params["jediTaskID"],
                            sqlJediDS,
                            varMap,
                            async_params["exec_order"],
                        )
                        async_params["exec_order"] += 1
                    else:
                        cur.execute(sqlJediDS + comment, varMap)
                    # update events in corrupted input files
                    if (
                        EventServiceUtils.isEventServiceMerge(jobSpec)
                        and jobSpec.jobStatus == "failed"
                        and jobSpec.pilotErrorCode in EventServiceUtils.PEC_corruptedInputFiles + EventServiceUtils.PEC_corruptedInputFilesTmp
                        and t_type in ["input", "pseudo_input"]
                        and t_masterID is None
                        and (tmpContentsStat["nFilesUsed"] < 0 or tmpContentsStat["nFilesFailed"] > 0)
                    ):
                        toSet = True
                        if jobSpec.pilotErrorCode in EventServiceUtils.PEC_corruptedInputFilesTmp:
                            # check failure count for temporary errors
                            toSet = self.checkFailureCountWithCorruptedFiles(jobSpec.jediTaskID, jobSpec.PandaID)
                        if toSet:
                            self.setCorruptedEventRanges(jobSpec.jediTaskID, jobSpec.PandaID)
        # add jobset info for job cloning
        if useJobCloning:
            self.recordRetryHistoryJEDI(
                jobSpec.jediTaskID,
                jobSpec.PandaID,
                [jobSpec.jobsetID],
                EventServiceUtils.relationTypeJS_ID,
            )
        # update jumbo flag
        if jobSpec.eventService == EventServiceUtils.jumboJobFlagNumber:
            # check site
            varMap = {}
            varMap[":jediTaskID"] = jobSpec.jediTaskID
            sqlJumboS = f"SELECT site FROM {panda_config.schemaJEDI}.JEDI_Tasks WHERE jediTaskID=:jediTaskID "
            cur.execute(sqlJumboS + comment, varMap)
            tmpResS = self.cur.fetchone()
            (jumboSite,) = tmpResS
            # count number of events for jumbo
            newUseJumbo = "L"
            """
            varMap = {}
            varMap[':jediTaskID'] = jobSpec.jediTaskID
            varMap[':eventStatus']  = EventServiceUtils.ST_ready
            varMap[':minAttemptNr'] = 0
            sqlJumboC = "SELECT COUNT(*) FROM {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            sqlJumboC += "WHERE jediTaskID=:jediTaskID AND status=:eventStatus AND attemptNr>:minAttemptNr ".format(panda_config.schemaJEDI)
            cur.execute(sqlJumboC+comment,varMap)
            tmpResC = self.cur.fetchone()
            if tmpResC is not None:
                nEventsJumbo, = tmpResC
                tmpLog.debug('{0} event ranges available for jumbo'.format(nEventsJumbo))
                # no more events
                if nEventsJumbo == 0 and jumboSite is None:
                    newUseJumbo = 'D'
            """
            # update flag
            varMap = {}
            varMap[":jediTaskID"] = jobSpec.jediTaskID
            varMap[":newJumbo"] = newUseJumbo
            varMap[":notUseJumbo"] = "D"
            sqlJumboF = f"UPDATE {panda_config.schemaJEDI}.JEDI_Tasks "
            sqlJumboF += "SET useJumbo=:newJumbo WHERE jediTaskID=:jediTaskID "
            sqlJumboF += "AND useJumbo IS NOT NULL AND useJumbo<>:notUseJumbo "
            cur.execute(sqlJumboF + comment, varMap)
            nRow = cur.rowcount
            tmpLog.debug(f"set task.useJumbo={varMap[':newJumbo']} with {nRow}")
        # update input
        if (
            not EventServiceUtils.isJumboJob(jobSpec)
            and not (jobSpec.computingSite == EventServiceUtils.siteIdForWaitingCoJumboJobs and not jobSpec.isCancelled())
            and jobSpec.taskBufferErrorCode not in [ErrorCode.EC_PilotRetried]
        ):
            self.updateInputStatusJedi(jobSpec.jediTaskID, jobSpec.PandaID, jobSpec.jobStatus)
        # update t_task
        if jobSpec.jobStatus == "finished" and jobSpec.prodSourceLabel not in ["panda"]:
            varMap = {}
            varMap[":jediTaskID"] = jobSpec.jediTaskID
            varMap[":noutevents"] = nOutEvents
            schemaDEFT = self.getSchemaDEFT()
            sqlTtask = f"UPDATE {schemaDEFT}.T_TASK "
            if jobSpec.processingType != "pmerge":
                updateNumDone = True
                sqlTtask += "SET total_done_jobs=total_done_jobs+1,timestamp=CURRENT_DATE,total_events=LEAST(9999999999,total_events+:noutevents) "
            else:
                updateNumDone = False
                sqlTtask += "SET timestamp=CURRENT_DATE,total_events=LEAST(9999999999,total_events+:noutevents) "
            sqlTtask += "WHERE taskid=:jediTaskID "
            tmpLog.debug(sqlTtask + comment + str(varMap))
            cur.execute(sqlTtask + comment, varMap)
            nRow = cur.rowcount
            # get total_done_jobs
            if updateNumDone and nRow == 1:
                varMap = {}
                varMap[":jediTaskID"] = jobSpec.jediTaskID
                sqlNumDone = f"SELECT total_done_jobs FROM {schemaDEFT}.T_TASK "
                sqlNumDone += "WHERE taskid=:jediTaskID "
                cur.execute(sqlNumDone + comment, varMap)
                tmpResNumDone = self.cur.fetchone()
                if tmpResNumDone is not None:
                    (numDone,) = tmpResNumDone
                    if numDone in [5, 100]:
                        # reset walltimeUnit to recalcurate task parameters
                        varMap = {}
                        varMap[":jediTaskID"] = jobSpec.jediTaskID
                        sqlRecal = "UPDATE ATLAS_PANDA.JEDI_Tasks SET walltimeUnit=NULL WHERE jediTaskId=:jediTaskID "
                        msgStr = "trigger recalcuration of task parameters "
                        msgStr += f"with nDoneJobs={numDone} for jediTaskID={jobSpec.jediTaskID}"
                        tmpLog.debug(msgStr)
                        cur.execute(sqlRecal + comment, varMap)
        # propagate failed result to unmerge job
        if len(finishUnmerge) > 0:
            self.updateUnmergedJobs(jobSpec, finishUnmerge, async_params=async_params)
        # update some job attributes
        self.setHS06sec(jobSpec.PandaID)

        # update the g of CO2 emitted by the job
        try:
            gco2_regional, gco2_global = self.set_co2_emissions(jobSpec.PandaID)
            tmpLog.debug(f"calculated gCO2 regional {gco2_regional} and global {gco2_global}")
        except Exception:
            tmpLog.error(f"failed calculating gCO2 with {traceback.format_exc()}")

        # return
        return True

    # check if task is active
    def checkTaskStatusJEDI(self, jediTaskID, cur):
        comment = " /* DBProxy.checkTaskStatusJEDI */"
        retVal = False
        curStat = None
        if jediTaskID not in ["NULL", None]:
            sql = "SELECT status FROM ATLAS_PANDA.JEDI_Tasks WHERE jediTaskID=:jediTaskID "
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            cur.execute(sql + comment, varMap)
            res = cur.fetchone()
            if res is not None:
                curStat = res[0]
                if curStat not in [
                    "done",
                    "finished",
                    "failed",
                    "broken",
                    "aborted",
                    "prepared",
                ]:
                    retVal = True
        _logger.debug(f"checkTaskStatusJEDI jediTaskID={jediTaskID} in {curStat} with {retVal}")
        return retVal

    # update JEDI for pilot retry
    def updateForPilotRetryJEDI(self, job, cur, onlyHistory=False, relationType=None):
        comment = " /* DBProxy.updateForPilotRetryJEDI */"
        # sql to update file
        sqlFJI = f"UPDATE {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
        sqlFJI += "SET attemptNr=attemptNr+1,failedAttempt=failedAttempt+1,PandaID=:PandaID "
        sqlFJI += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
        sqlFJI += "AND attemptNr=:attemptNr AND keepTrack=:keepTrack "
        sqlFJO = f"UPDATE {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
        sqlFJO += "SET attemptNr=attemptNr+1,failedAttempt=failedAttempt+1,PandaID=:PandaID,outPandaID=:PandaID "
        sqlFJO += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
        sqlFJO += "AND attemptNr=:attemptNr AND keepTrack=:keepTrack "
        sqlFP = "UPDATE ATLAS_PANDA.filesTable4 SET attemptNr=attemptNr+1 "
        sqlFP += "WHERE row_ID=:row_ID "
        if not onlyHistory:
            for tmpFile in job.Files:
                # skip if no JEDI
                if tmpFile.fileID == "NULL":
                    continue
                # update JEDI contents
                varMap = {}
                varMap[":jediTaskID"] = tmpFile.jediTaskID
                varMap[":datasetID"] = tmpFile.datasetID
                varMap[":fileID"] = tmpFile.fileID
                varMap[":attemptNr"] = tmpFile.attemptNr
                varMap[":PandaID"] = tmpFile.PandaID
                varMap[":keepTrack"] = 1
                if tmpFile.type in ["output", "log"]:
                    sqlFJ = sqlFJO
                else:
                    sqlFJ = sqlFJI
                _logger.debug(sqlFJ + comment + str(varMap))
                cur.execute(sqlFJ + comment, varMap)
                nRow = cur.rowcount
                if nRow == 1:
                    # update fileTable if JEDI contents was updated
                    varMap = {}
                    varMap[":row_ID"] = tmpFile.row_ID
                    _logger.debug(sqlFP + comment + str(varMap))
                    cur.execute(sqlFP + comment, varMap)
        # get origin
        originIDs = self.getOriginPandaIDsJEDI(job.parentID, job.jediTaskID, cur)
        # sql to record retry history
        sqlRH = f"INSERT INTO {panda_config.schemaJEDI}.JEDI_Job_Retry_History "
        sqlRH += "(jediTaskID,oldPandaID,newPandaID,originPandaID,relationType) "
        sqlRH += "VALUES(:jediTaskID,:oldPandaID,:newPandaID,:originPandaID,:relationType) "
        # record retry history
        for originID in originIDs:
            varMap = {}
            varMap[":jediTaskID"] = job.jediTaskID
            varMap[":oldPandaID"] = job.parentID
            varMap[":newPandaID"] = job.PandaID
            varMap[":originPandaID"] = originID
            if relationType is None:
                varMap[":relationType"] = "retry"
            else:
                varMap[":relationType"] = relationType
            cur.execute(sqlRH + comment, varMap)
        # record jobset
        if EventServiceUtils.isEventServiceMerge(job) and relationType is None:
            varMap = {}
            varMap[":jediTaskID"] = job.jediTaskID
            varMap[":oldPandaID"] = job.jobsetID
            varMap[":newPandaID"] = job.PandaID
            varMap[":originPandaID"] = job.jobsetID
            varMap[":relationType"] = EventServiceUtils.relationTypeJS_ID
            cur.execute(sqlRH + comment, varMap)
        return

    # get origin PandaIDs
    def getOriginPandaIDsJEDI(self, pandaID, jediTaskID, cur):
        comment = " /* DBProxy.getOriginPandaIDsJEDI */"
        # get parent IDs
        varMap = {}
        varMap[":jediTaskID"] = jediTaskID
        varMap[":newPandaID"] = pandaID
        sqlFJ = f"SELECT MIN(originPandaID) FROM {panda_config.schemaJEDI}.JEDI_Job_Retry_History "
        sqlFJ += "WHERE jediTaskID=:jediTaskID AND newPandaID=:newPandaID "
        sqlFJ += "AND (relationType IS NULL OR NOT relationType IN ("
        for tmpType in EventServiceUtils.relationTypesForJS:
            tmpKey = f":{tmpType}"
            sqlFJ += f"{tmpKey},"
            varMap[tmpKey] = tmpType
        sqlFJ = sqlFJ[:-1]
        sqlFJ += ")) "
        cur.execute(sqlFJ + comment, varMap)
        resT = cur.fetchone()
        retList = []
        if resT is None:
            # origin
            retList.append(pandaID)
        else:
            # use only one origin since tracking the whole tree brings too many origins
            (originPandaID,) = resT
            if originPandaID is None:
                # origin
                retList.append(pandaID)
            else:
                retList.append(originPandaID)
        # return
        return retList

    # get jobsetID for PandaID
    def getJobsetIDforPandaID(self, pandaID, jediTaskID):
        comment = " /* DBProxy. */"
        # get parent IDs
        varMap = {}
        varMap[":jediTaskID"] = jediTaskID
        varMap[":newPandaID"] = pandaID
        varMap[":relationType"] = EventServiceUtils.relationTypeJS_ID
        sqlFJ = f"SELECT oldPandaID FROM {panda_config.schemaJEDI}.JEDI_Job_Retry_History "
        sqlFJ += "WHERE jediTaskID=:jediTaskID AND newPandaID=:newPandaID "
        sqlFJ += "AND relationType=:relationType "
        self.cur.execute(sqlFJ + comment, varMap)
        resT = self.cur.fetchone()
        if resT is not None:
            return resT[0]
        return None

    # get original consumers
    def getOriginalConsumers(self, jediTaskID, jobsetID, pandaID):
        comment = " /* DBProxy.getOriginalConsumers */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" < jediTaskID={jediTaskID} jobsetID={jobsetID} PandaID={pandaID} >"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            # sql to get sites where consumers are active
            sqlA = "SELECT computingSite FROM ATLAS_PANDA.jobsActive4 WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID "
            sqlA += "UNION "
            sqlA += "SELECT computingSite FROM ATLAS_PANDA.jobsDefined4 WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID "
            # sql to get original IDs
            sqlG = f"SELECT oldPandaID FROM {panda_config.schemaJEDI}.JEDI_Job_Retry_History "
            sqlG += "WHERE jediTaskID=:jediTaskID AND newPandaID=:jobsetID AND relationType=:relationType "
            # sql to check computingSite
            sqlC1 = "SELECT computingSite FROM ATLAS_PANDA.jobsArchived4 WHERE PandaID=:PandaID "
            sqlC2 = "SELECT computingSite FROM ATLAS_PANDAARCH.jobsArchived WHERE PandaID=:PandaID AND modificationTime>(CURRENT_DATE-30) "
            # sql to get job info
            sqlJ = f"SELECT {JobSpec.columnNames()} "
            sqlJ += "FROM {0} "
            sqlJ += "WHERE PandaID=:PandaID AND modificationTime>(CURRENT_DATE-30) "
            sqlF = f"SELECT {FileSpec.columnNames()} FROM ATLAS_PANDA.filesTable4 "
            sqlF += "WHERE PandaID=:PandaID "
            sqlP = "SELECT jobParameters FROM {0} WHERE PandaID=:PandaID "
            # get sites
            aSites = set()
            varMap = dict()
            varMap[":jediTaskID"] = jediTaskID
            varMap[":jobsetID"] = jobsetID
            self.cur.execute(sqlA + comment, varMap)
            resA = self.cur.fetchall()
            for (computingSite,) in resA:
                aSites.add(computingSite)
            # get original IDs
            varMap = dict()
            varMap[":jediTaskID"] = jediTaskID
            varMap[":jobsetID"] = jobsetID
            varMap[":relationType"] = EventServiceUtils.relationTypeJS_Map
            self.cur.execute(sqlG + comment, varMap)
            resG = self.cur.fetchall()
            jobList = []
            for (pandaID,) in resG:
                # check computingSite
                varMap = dict()
                varMap[":PandaID"] = pandaID
                self.cur.execute(sqlC1 + comment, varMap)
                resC = self.cur.fetchone()
                if resC is None:
                    # try archived
                    self.cur.execute(sqlC2 + comment, varMap)
                    resC = self.cur.fetchone()
                    inArchived = True
                else:
                    inArchived = False
                # skip since it is not yet archived and thus is still active
                if resC is None:
                    continue
                (computingSite,) = resC
                # skip since there is an active consumer at the site
                if computingSite in aSites:
                    continue
                # get job
                if inArchived:
                    self.cur.execute(sqlJ.format("ATLAS_PANDAARCH.jobsArchived") + comment, varMap)
                else:
                    self.cur.execute(sqlJ.format("ATLAS_PANDA.jobsArchived4") + comment, varMap)
                resJ = self.cur.fetchone()
                if resJ is not None:
                    jobSpec = JobSpec()
                    jobSpec.pack(resJ)
                    # get files
                    self.cur.execute(sqlF + comment, varMap)
                    resFs = self.cur.fetchall()
                    if len(resFs) == 0:
                        continue
                    for resF in resFs:
                        fileSpec = FileSpec()
                        fileSpec.pack(resF)
                        jobSpec.addFile(fileSpec)
                    # get job params
                    if inArchived:
                        self.cur.execute(
                            sqlP.format("ATLAS_PANDAARCH.jobParamsTable_ARCH") + comment,
                            varMap,
                        )
                    else:
                        self.cur.execute(sqlP.format("ATLAS_PANDA.jobParamsTable") + comment, varMap)
                    for (clobJobP,) in self.cur:
                        if clobJobP is not None:
                            try:
                                jobSpec.jobParameters = clobJobP.read()
                            except AttributeError:
                                jobSpec.jobParameters = str(clobJobP)
                        break
                    # add
                    jobList.append(jobSpec)
                    aSites.add(computingSite)
            tmpLog.debug(f"got {len(jobList)} consumers")
            return jobList
        except Exception:
            # error
            self.dumpErrorMessage(_logger, methodName)
            return []

    # wake up connection
    def wakeUp(self):
        for iTry in range(5):
            try:
                # check if the connection is working
                self.conn.ping()
                return
            except Exception:
                type, value, traceBack = sys.exc_info()
                _logger.debug("wakeUp %d : %s %s" % (iTry, type, value))
                # wait for reconnection
                time.sleep(1)
                self.connect(reconnect=True)

    # commit
    def _commit(self):
        try:
            self.conn.commit()
            return True
        except Exception:
            _logger.error("commit error")
            return False

    # rollback
    def _rollback(self, useOtherError=False):
        return_value = True
        # rollback
        err_code = None
        _logger.debug("rollback")
        try:
            self.conn.rollback()
        except Exception:
            _logger.error("rollback error")
            return_value = False
        # reconnect if needed
        try:
            err_type, err_value = sys.exc_info()[:2]
            # get error code for postgres
            if self.backend == "postgres":
                try:
                    err_code = err_value.pgcode
                except Exception:
                    pass
            # get ORA ErrorCode
            if err_code is None:
                err_code = str(err_value).split()[0]
                err_code = err_code[:-1]
            err_msg = f"rollback EC:{err_code} {err_value}"
            _logger.debug(err_msg)
            # error codes for connection error
            if self.backend == "oracle":
                error_list_for_reconnect = [
                    "ORA-01012",
                    "ORA-01033",
                    "ORA-01034",
                    "ORA-01089",
                    "ORA-03113",
                    "ORA-03114",
                    "ORA-12203",
                    "ORA-12500",
                    "ORA-12571",
                    "ORA-03135",
                    "ORA-25402",
                ]
                # other errors are apparently given when connection lost contact
                if useOtherError:
                    error_list_for_reconnect += ["ORA-01861", "ORA-01008"]
            elif self.backend == "postgres":
                error_list_for_reconnect = [
                    psycopg_errorcodes.CONNECTION_EXCEPTION,
                    psycopg_errorcodes.SQLSERVER_REJECTED_ESTABLISHMENT_OF_SQLCONNECTION,
                    psycopg_errorcodes.CONNECTION_DOES_NOT_EXIST,
                    psycopg_errorcodes.SQLCLIENT_UNABLE_TO_ESTABLISH_SQLCONNECTION,
                    psycopg_errorcodes.CONNECTION_FAILURE,
                    psycopg_errorcodes.READ_ONLY_SQL_TRANSACTION,
                ]
            else:
                # mysql error codes for connection error
                from MySQLdb.constants.CR import (
                    CONN_HOST_ERROR,
                    CONNECTION_ERROR,
                    LOCALHOST_CONNECTION,
                    SERVER_LOST,
                )
                from MySQLdb.constants.ER import (
                    ACCESS_DENIED_ERROR,
                    DBACCESS_DENIED_ERROR,
                    ILLEGAL_VALUE_FOR_TYPE,
                    SERVER_SHUTDOWN,
                )

                error_list_for_reconnect = [
                    ACCESS_DENIED_ERROR,
                    DBACCESS_DENIED_ERROR,
                    SERVER_SHUTDOWN,
                    CONNECTION_ERROR,
                    CONN_HOST_ERROR,
                    LOCALHOST_CONNECTION,
                    SERVER_LOST,
                ]
                # other errors are apparently given when connection lost contact
                if useOtherError:
                    error_list_for_reconnect += [ILLEGAL_VALUE_FOR_TYPE]
            if err_code in error_list_for_reconnect:
                # reconnect
                reconnect_stat = self.connect(reconnect=True)
                _logger.debug(f"rollback reconnected {reconnect_stat}")
        except Exception:
            pass
        # return
        return return_value

    # dump error message
    def dumpErrorMessage(self, tmpLog, methodName):
        # error
        errtype, errvalue = sys.exc_info()[:2]
        errStr = f"{methodName}: {errtype.__name__} {errvalue}"
        errStr.strip()
        errStr += " "
        errStr += traceback.format_exc()
        tmpLog.error(errStr)

    # get DEFT schema
    def getSchemaDEFT(self):
        if not hasattr(panda_config, "schemaDEFT"):
            return "ATLAS_DEFT"
        return panda_config.schemaDEFT

    # get working group with production role
    def getWorkingGroup(self, fqans):
        for fqan in fqans:
            # check production role
            match = re.search("/[^/]+/([^/]+)/Role=production", fqan)
            if match is not None:
                return match.group(1)
        return None

    # insert TaskParams
    def insertTaskParamsPanda(
        self,
        taskParams,
        dn,
        prodRole,
        fqans,
        parent_tid,
        properErrorCode=False,
        allowActiveTask=False,
    ):
        comment = " /* JediDBProxy.insertTaskParamsPanda */"
        try:
            methodName = "insertTaskParamsPanda"
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ["", "NULL", None]:
                compactDN = dn
            methodName += f" <{compactDN}>"
            _logger.debug(f"{methodName} start")
            # decode json
            taskParamsJson = PrioUtil.decodeJSON(taskParams)
            # set user name
            if not prodRole or "userName" not in taskParamsJson:
                taskParamsJson["userName"] = compactDN
            # identify parent
            if "parentTaskName" in taskParamsJson:
                parent_tid = self.get_parent_task_id_with_name(taskParamsJson["userName"], taskParamsJson["parentTaskName"])
                if not parent_tid:
                    tmpMsg = f"failed to find parent with user=\"{taskParamsJson['userName']}\" name={taskParamsJson['parentTaskName']}"
                    _logger.debug(f"{methodName} {tmpMsg}")
                    return 11, tmpMsg
                else:
                    _logger.debug(f"{methodName} found parent {parent_tid} with user=\"{taskParamsJson['userName']}\" name={taskParamsJson['parentTaskName']}")
            # set task type
            if not prodRole or "taskType" not in taskParamsJson:
                taskParamsJson["taskType"] = "anal"
                taskParamsJson["taskPriority"] = 1000
                # extract working group
                if "official" in taskParamsJson and taskParamsJson["official"] is True:
                    workingGroup = self.getWorkingGroup(fqans)
                    if workingGroup is not None:
                        taskParamsJson["workingGroup"] = workingGroup

            _logger.debug(f"{methodName} taskName={taskParamsJson['taskName']}")
            schemaDEFT = self.getSchemaDEFT()
            # sql to check task duplication for user
            sqlTDU = f"SELECT jediTaskID,status FROM {panda_config.schemaJEDI}.JEDI_Tasks "
            sqlTDU += "WHERE vo=:vo AND prodSourceLabel=:prodSourceLabel AND userName=:userName AND taskName=:taskName "
            sqlTDU += "ORDER BY jediTaskID DESC FOR UPDATE "
            # sql to check task duplication for group
            sqlTDW = f"SELECT jediTaskID,status FROM {panda_config.schemaJEDI}.JEDI_Tasks "
            sqlTDW += "WHERE vo=:vo AND prodSourceLabel=:prodSourceLabel AND taskName=:taskName "
            sqlTDW += "ORDER BY jediTaskID DESC FOR UPDATE "
            # sql to check DEFT table for user
            sqlCDU = f"SELECT taskid FROM {schemaDEFT}.T_TASK "
            sqlCDU += "WHERE vo=:vo AND prodSourceLabel=:prodSourceLabel AND userName=:userName AND taskName=:taskName "
            sqlCDU += "ORDER BY taskid DESC FOR UPDATE "
            # sql to check DEFT table for group
            sqlCDW = f"SELECT taskid FROM {schemaDEFT}.T_TASK "
            sqlCDW += "WHERE vo=:vo AND prodSourceLabel=:prodSourceLabel AND taskName=:taskName "
            sqlCDW += "ORDER BY taskid DESC FOR UPDATE "
            # sql to insert task parameters
            sqlT = f"INSERT INTO {schemaDEFT}.T_TASK "
            sqlT += "(taskid,status,submit_time,vo,prodSourceLabel,userName,taskName,jedi_task_parameters,priority,current_priority,parent_tid) VALUES "
            varMap = {}
            if self.backend in ["oracle", "postgres"]:
                sqlT += f"({schemaDEFT}.PRODSYS2_TASK_ID_SEQ.nextval,"
            else:
                # panda_config.backend == 'mysql':
                # fake sequence
                sql = " INSERT INTO PRODSYS2_TASK_ID_SEQ (col) VALUES (NULL) "
                self.cur.arraysize = 100
                self.cur.execute(sql + comment, {})
                sql2 = """ SELECT LAST_INSERT_ID() """
                self.cur.execute(sql2 + comment, {})
                (nextval,) = self.cur.fetchone()
                sqlT += "( :nextval ,".format(schemaDEFT)
                varMap[":nextval"] = nextval
            sqlT += ":status,CURRENT_DATE,:vo,:prodSourceLabel,:userName,:taskName,:param,:priority,:current_priority,"
            if parent_tid is None:
                if self.backend in ["oracle", "postgres"]:
                    sqlT += f"{schemaDEFT}.PRODSYS2_TASK_ID_SEQ.currval) "
                else:
                    # panda_config.backend == 'mysql':
                    # fake sequence
                    sql = " SELECT MAX(COL) FROM PRODSYS2_TASK_ID_SEQ "
                    self.cur.arraysize = 100
                    self.cur.execute(sql + comment, {})
                    (currval,) = self.cur.fetchone()
                    sqlT += " :currval ) "
                    varMap[":currval"] = currval
            else:
                sqlT += ":parent_tid) "
            sqlT += "RETURNING TASKID INTO :jediTaskID"
            # sql to delete command
            sqlDC = f"DELETE FROM {schemaDEFT}.PRODSYS_COMM "
            sqlDC += "WHERE COMM_TASK=:jediTaskID "
            # sql to insert command
            sqlIC = f"INSERT INTO {schemaDEFT}.PRODSYS_COMM (COMM_TASK,COMM_OWNER,COMM_CMD,COMM_PARAMETERS) "
            sqlIC += "VALUES (:jediTaskID,:comm_owner,:comm_cmd,:comm_parameters) "
            max_n_tasks = self.getConfigValue(
                "dbproxy",
                f"MAX_ACTIVE_TASKS_PER_USER_{taskParamsJson['prodSourceLabel']}",
            )
            # begin transaction
            self.conn.begin()
            # check max
            if max_n_tasks is not None:
                sqlTOT = (
                    "SELECT COUNT(*) "
                    "FROM {0}.JEDI_Tasks tabT,{0}.JEDI_AUX_Status_MinTaskID tabA "
                    "WHERE tabT.status=tabA.status AND tabT.jediTaskID>=tabA.min_jediTaskID "
                    "AND tabT.prodSourceLabel=:prodSourceLabel AND tabT.userName=:userName "
                    "AND tabT.status IN ("
                ).format(panda_config.schemaJEDI)
                varMapTot = {}
                varMapTot[":prodSourceLabel"] = taskParamsJson["prodSourceLabel"]
                varMapTot[":userName"] = taskParamsJson["userName"]
                for st in [
                    "registered",
                    "defined",
                    "ready",
                    "scouting",
                    "running",
                    "paused",
                    "throttled",
                ]:
                    key = f":{st}"
                    sqlTOT += f"{key},"
                    varMapTot[key] = st
                sqlTOT = sqlTOT[:-1]
                sqlTOT += ") "
                self.cur.execute(sqlTOT + comment, varMapTot)
                resTOT = self.cur.fetchone()
                if resTOT is not None and resTOT[0] > max_n_tasks:
                    # commit
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    tmpMsg = f"Too many active tasks for {taskParamsJson['userName']} {resTOT[0]}>{max_n_tasks}"
                    _logger.debug(f"{methodName} {tmpMsg}")
                    return 10, tmpMsg
            # check duplication
            goForward = True
            retFlag = False
            retVal = None
            errorCode = 0
            if taskParamsJson["taskType"] == "anal" and (("uniqueTaskName" in taskParamsJson and taskParamsJson["uniqueTaskName"] is True) or allowActiveTask):
                if "official" in taskParamsJson and taskParamsJson["official"] is True:
                    isOfficial = True
                else:
                    isOfficial = False
                # check JEDI
                varMap[":vo"] = taskParamsJson["vo"]
                if isOfficial:
                    pass
                else:
                    varMap[":userName"] = taskParamsJson["userName"]
                varMap[":taskName"] = taskParamsJson["taskName"]
                varMap[":prodSourceLabel"] = taskParamsJson["prodSourceLabel"]
                if isOfficial:
                    self.cur.execute(sqlTDW + comment, varMap)
                else:
                    self.cur.execute(sqlTDU + comment, varMap)
                resDT = self.cur.fetchone()
                if resDT is None:
                    # check DEFT table
                    varMap = {}
                    varMap[":vo"] = taskParamsJson["vo"]
                    if isOfficial:
                        pass
                    else:
                        varMap[":userName"] = taskParamsJson["userName"]
                    varMap[":taskName"] = taskParamsJson["taskName"]
                    varMap[":prodSourceLabel"] = taskParamsJson["prodSourceLabel"]
                    if isOfficial:
                        self.cur.execute(sqlCDW + comment, varMap)
                    else:
                        self.cur.execute(sqlCDU + comment, varMap)
                    resCD = self.cur.fetchone()
                    if resCD is not None:
                        # task is already in DEFT
                        (jediTaskID,) = resCD
                        _logger.debug(f"{methodName} old jediTaskID={jediTaskID} with taskName={varMap[':taskName']} in DEFT table")
                        goForward = False
                        retVal = f"jediTaskID={jediTaskID} is already queued for outDS={taskParamsJson['taskName']}. "
                        retVal += "You cannot submit duplicated tasks. "
                        _logger.debug(f"{methodName} skip since old task is already queued in DEFT")
                        errorCode = 1
                else:
                    # task is already in JEDI table
                    jediTaskID, taskStatus = resDT
                    _logger.debug(f"{methodName} old jediTaskID={jediTaskID} with taskName={varMap[':taskName']} in status={taskStatus}")
                    # check task status
                    if taskStatus not in [
                        "finished",
                        "failed",
                        "aborted",
                        "done",
                        "exhausted",
                    ] and not (allowActiveTask and taskStatus in ["running", "scouting", "pending"] and taskParamsJson["prodSourceLabel"] in ["user"]):
                        # still active
                        goForward = False
                        retVal = f"jediTaskID={jediTaskID} is in the {taskStatus} state for outDS={taskParamsJson['taskName']}. "
                        retVal += "You can re-submit the task with new parameters for the same or another input "
                        retVal += "once it goes into finished/failed/done. "
                        retVal += "Or you can retry the task once it goes into running/finished/failed/done. "
                        retVal += "Note that retry != resubmission accoring to "
                        retVal += "https://twiki.cern.ch/twiki/bin/view/PanDA/PandaJEDI#Task_retry_and_resubmission "
                        _logger.debug(f"{methodName} skip since old task is not yet finalized")
                        errorCode = 2
                    else:
                        # extract several params for incremental execution
                        newTaskParams = {}
                        newRamCount = None
                        for tmpKey in taskParamsJson:
                            tmpVal = taskParamsJson[tmpKey]
                            # dataset names
                            # site limitation
                            # command line parameters
                            # splitting hints
                            # fixed source code
                            if tmpKey.startswith("dsFor") or tmpKey in [
                                "site",
                                "cloud",
                                "includedSite",
                                "excludedSite",
                                "cliParams",
                                "nFilesPerJob",
                                "nFiles",
                                "nEvents",
                                "nGBPerJob",
                                "fixedSandbox",
                                "ignoreMissingInDS",
                                "currentPriority",
                                "priority",
                                "nMaxFilesPerJob",
                                "ramCount",
                                "noLoopingCheck",
                                "loopingCheck",
                                "avoidVP",
                            ]:
                                if tmpKey == "priority":
                                    tmpKey = "currentPriority"
                                if tmpKey == "loopingCheck":
                                    tmpKey == "noLoopingCheck"
                                    if tmpVal:
                                        tmpVal = False
                                    else:
                                        tmpVal = True
                                newTaskParams[tmpKey] = tmpVal
                                if tmpKey == "fixedSandbox" and "sourceURL" in taskParamsJson:
                                    newTaskParams["sourceURL"] = taskParamsJson["sourceURL"]
                                elif tmpKey == "ramCount":
                                    newRamCount = tmpVal
                        # send command to reactivate the task
                        if not allowActiveTask or taskStatus in [
                            "finished",
                            "failed",
                            "aborted",
                            "done",
                            "exhausted",
                        ]:
                            # set new RAM count
                            if newRamCount is not None:
                                sqlRAM = f"UPDATE {panda_config.schemaJEDI}.JEDI_Dataset_Contents SET ramCount=:ramCount "
                                sqlRAM += "WHERE jediTaskID=:jediTaskID AND (ramCount IS NOT NULL AND ramCount>:ramCount) "
                                sqlRAM += f"AND datasetID IN (SELECT datasetID FROM {panda_config.schemaJEDI}.JEDI_Datasets "
                                sqlRAM += "WHERE jediTaskID=:jediTaskID AND type IN (:type1,:type2)) "
                                varMap = {}
                                varMap[":jediTaskID"] = jediTaskID
                                varMap[":type1"] = "input"
                                varMap[":type2"] = "pseudo_input"
                                varMap[":ramCount"] = newRamCount
                                self.cur.execute(sqlRAM + comment, varMap)
                                sqlRAMT = f"UPDATE {panda_config.schemaJEDI}.JEDI_Tasks SET ramCount=:ramCount WHERE jediTaskID=:jediTaskID "
                                varMap = {}
                                varMap[":jediTaskID"] = jediTaskID
                                varMap[":ramCount"] = newRamCount
                                self.cur.execute(sqlRAMT + comment, varMap)
                            # delete command just in case
                            varMap = {}
                            varMap[":jediTaskID"] = jediTaskID
                            self.cur.execute(sqlDC + comment, varMap)
                            # insert command
                            varMap = {}
                            varMap[":jediTaskID"] = jediTaskID
                            varMap[":comm_cmd"] = "incexec"
                            varMap[":comm_owner"] = "DEFT"
                            varMap[":comm_parameters"] = json.dumps(newTaskParams)
                            self.cur.execute(sqlIC + comment, varMap)
                            _logger.info(f"{methodName} {varMap[':comm_cmd']} jediTaskID={jediTaskID} with {str(newTaskParams)}")
                            retVal = "reactivation accepted. "
                            retVal += f"jediTaskID={jediTaskID} (currently in {taskStatus} state) will be re-executed with old and/or new input"
                            errorCode = 3
                        else:
                            # sql to read task params
                            sqlTP = f"SELECT taskParams FROM {panda_config.schemaJEDI}.JEDI_TaskParams WHERE jediTaskID=:jediTaskID "
                            varMap = {}
                            varMap[":jediTaskID"] = jediTaskID
                            self.cur.execute(sqlTP + comment, varMap)
                            tmpStr = ""
                            for (tmpItem,) in self.cur:
                                try:
                                    tmpStr = tmpItem.read()
                                except AttributeError:
                                    tmpStr = str(tmpItem)
                                break
                            # decode json
                            taskParamsJson = json.loads(tmpStr)
                            # just change some params for active task
                            for tmpKey in newTaskParams:
                                tmpVal = newTaskParams[tmpKey]
                                taskParamsJson[tmpKey] = tmpVal
                            # update params
                            sqlTU = f"UPDATE {panda_config.schemaJEDI}.JEDI_TaskParams SET taskParams=:taskParams "
                            sqlTU += "WHERE jediTaskID=:jediTaskID "
                            varMap = {}
                            varMap[":jediTaskID"] = jediTaskID
                            varMap[":taskParams"] = json.dumps(taskParamsJson)
                            self.cur.execute(sqlTU + comment, varMap)
                            _logger.debug(f"{methodName} add new params for jediTaskID={jediTaskID} with {str(newTaskParams)}")
                            retVal = f"{taskStatus}. new tasks params have been set to jediTaskID={jediTaskID}. "
                            errorCode = 5
                        goForward = False
                        retFlag = True
            if goForward:
                # insert task parameters
                taskParams = json.dumps(taskParamsJson)
                varMap = {}
                varMap[":param"] = taskParams
                varMap[":status"] = "waiting"
                varMap[":vo"] = taskParamsJson["vo"]
                varMap[":userName"] = taskParamsJson["userName"]
                varMap[":taskName"] = taskParamsJson["taskName"]
                if parent_tid is not None:
                    varMap[":parent_tid"] = parent_tid
                varMap[":prodSourceLabel"] = taskParamsJson["prodSourceLabel"]
                varMap[":jediTaskID"] = self.cur.var(varNUMBER)
                if "taskPriority" in taskParamsJson:
                    varMap[":priority"] = taskParamsJson["taskPriority"]
                else:
                    varMap[":priority"] = 100
                varMap[":current_priority"] = varMap[":priority"]
                self.cur.execute(sqlT + comment, varMap)
                val = self.getvalue_corrector(self.cur.getvalue(varMap[":jediTaskID"]))
                jediTaskID = int(val)
                if properErrorCode:
                    retVal = f"succeeded. new jediTaskID={jediTaskID}"
                else:
                    retVal = jediTaskID
                _logger.debug(f"{methodName} inserted new jediTaskID={jediTaskID}")
                retFlag = True
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            _logger.debug(f"{methodName} done")
            if properErrorCode:
                return errorCode, retVal
            return retFlag, retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            errorCode = 4
            retVal = "failed to register task"
            if properErrorCode:
                return errorCode, retVal
            return False, retVal

    # send command to task through DEFT
    def sendCommandTaskPanda(
        self,
        jediTaskID,
        dn,
        prodRole,
        comStr,
        comComment=None,
        useCommit=True,
        properErrorCode=False,
        comQualifier=None,
        broadcast=False,
    ):
        comment = " /* JediDBProxy.sendCommandTaskPanda */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" < jediTaskID={jediTaskID} >"
        tmpLog = LogWrapper(_logger, methodName)
        try:
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ["", "NULL", None]:
                compactDN = dn
            tmpLog.debug(f"start com={comStr} DN={compactDN} prod={prodRole} comment={comComment} qualifier={comQualifier} broadcast={broadcast}")
            # sql to check status and owner
            sqlTC = f"SELECT status,userName,prodSourceLabel FROM {panda_config.schemaJEDI}.JEDI_Tasks "
            sqlTC += "WHERE jediTaskID=:jediTaskID FOR UPDATE "
            # sql to delete command
            schemaDEFT = self.getSchemaDEFT()
            sqlT = f"DELETE FROM {schemaDEFT}.PRODSYS_COMM "
            sqlT += "WHERE COMM_TASK=:jediTaskID "
            # sql to insert command
            sqlC = f"INSERT INTO {schemaDEFT}.PRODSYS_COMM (COMM_TASK,COMM_OWNER,COMM_CMD,COMM_COMMENT) "
            sqlC += "VALUES (:jediTaskID,:comm_owner,:comm_cmd,:comm_comment) "
            goForward = True
            retStr = ""
            retCode = 0
            sendMsgToPilot = False
            # begin transaction
            if useCommit:
                self.conn.begin()
            # get task status and owner
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            self.cur.execute(sqlTC + comment, varMap)
            resTC = self.cur.fetchone()
            if resTC is None:
                # task not found
                retStr = f"jediTaskID={jediTaskID} not found"
                tmpLog.debug(retStr)
                goForward = False
                retCode = 2
            else:
                taskStatus, userName, prodSourceLabel = resTC
                tmpLog.debug(f"status={taskStatus}")
            # check owner
            if goForward:
                if not prodRole and compactDN != userName:
                    retStr = "Permission denied: not the task owner or no production role"
                    tmpLog.debug(retStr)
                    goForward = False
                    retCode = 3
            # check task status
            if goForward:
                add_msg = ""
                if comStr in ["kill", "finish"]:
                    sendMsgToPilot = broadcast
                    if taskStatus in [
                        "finished",
                        "done",
                        "prepared",
                        "broken",
                        "aborted",
                        "aborted",
                        "toabort",
                        "aborting",
                        "failed",
                    ]:
                        goForward = False
                if comStr == "retry":
                    if taskStatus not in ["finished", "failed", "exhausted"]:
                        goForward = False
                    elif taskStatus == "exhausted" and not prodRole:
                        goForward = False
                        add_msg = "and production role is missing"
                if comStr == "incexec":
                    if taskStatus not in [
                        "finished",
                        "failed",
                        "done",
                        "exhausted",
                    ]:
                        goForward = False
                if comStr == "reassign":
                    if taskStatus not in [
                        "registered",
                        "defined",
                        "ready",
                        "running",
                        "scouting",
                        "scouted",
                        "pending",
                        "assigning",
                        "exhausted",
                    ]:
                        goForward = False
                if comStr == "pause":
                    if taskStatus in [
                        "finished",
                        "failed",
                        "done",
                        "aborted",
                        "broken",
                        "paused",
                    ]:
                        goForward = False
                if comStr == "resume":
                    if taskStatus not in ["paused", "throttled", "staging"]:
                        goForward = False
                if comStr == "avalanche":
                    if taskStatus not in ["scouting"]:
                        goForward = False
                if comStr == "release":
                    if taskStatus not in ["scouting", "pending", "running", "ready", "assigning", "defined"]:
                        goForward = False
                if not goForward:
                    retStr = f"Command rejected: the {comStr} command is not accepted " f"if the task is in {taskStatus} status {add_msg}"
                    _logger.debug(f"{methodName} : {retStr}")
                    retCode = 4
                    # retry for failed analysis jobs
                    if comStr == "retry" and properErrorCode and taskStatus in ["running", "scouting", "pending"] and prodSourceLabel in ["user"]:
                        retCode = 5
                        retStr = taskStatus
            if goForward:
                # delete command just in case
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                self.cur.execute(sqlT + comment, varMap)
                # insert command
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":comm_cmd"] = comStr
                varMap[":comm_owner"] = "DEFT"
                if comComment is None:
                    tmpStr = ""
                    if comQualifier not in ["", None]:
                        tmpStr += f"{comQualifier} "
                    tmpStr += f"{comStr} by {compactDN}"
                    varMap[":comm_comment"] = tmpStr
                else:
                    varMap[":comm_comment"] = comComment
                self.cur.execute(sqlC + comment, varMap)
                retStr = f"command={comStr} is registered. will be executed in a few minutes"
                tmpLog.info(f"{retStr}")
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError("Commit error")
            # send command to the pilot
            if sendMsgToPilot:
                mb_proxy_topic = self.get_mb_proxy("panda_pilot_topic")
                if mb_proxy_topic:
                    tmpLog.debug(f"push {comStr}")
                    srv_msg_utils.send_task_message(mb_proxy_topic, comStr, jediTaskID)
                else:
                    tmpLog.debug("message topic not configured")
            if properErrorCode:
                return retCode, retStr
            else:
                if retCode == 0:
                    return True, retStr
                else:
                    return False, retStr
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            if properErrorCode:
                return 1, "failed to register command"
            else:
                return False, "failed to register command"

    # update unmerged jobs
    def updateUnmergedJobs(self, job, fileIDs=None, async_params=None):
        comment = " /* JediDBProxy.updateUnmergedJobs */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" < PandaID={job.PandaID} async_params={async_params} >"
        tmpLog = LogWrapper(_logger, methodName)
        # get PandaID which produced unmerged files
        umPandaIDs = []
        umCheckedIDs = []
        if fileIDs is None:
            fileIDs = set()
        # sql to get PandaIDs
        sqlUMP = "SELECT PandaID,attemptNr FROM ATLAS_PANDA.filesTable4 "
        sqlUMP += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
        sqlUMP += "AND type IN (:type1,:type2) ORDER BY attemptNr DESC "
        # sql to check job status
        sqlUMS = "SELECT jobStatus FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
        # look for unmerged files
        for tmpFile in job.Files:
            if tmpFile.isUnMergedInput():
                # only fileIDs which reach max attempt
                if len(fileIDs) > 0 and tmpFile.fileID not in fileIDs:
                    continue
                varMap = {}
                varMap[":jediTaskID"] = tmpFile.jediTaskID
                varMap[":datasetID"] = tmpFile.datasetID
                varMap[":fileID"] = tmpFile.fileID
                varMap[":type1"] = "output"
                varMap[":type2"] = "log"
                self.cur.arraysize = 100
                self.cur.execute(sqlUMP + comment, varMap)
                resUMP = self.cur.fetchall()
                # loop for job in merging state
                for tmpPandaID, tmpAttemptNr in resUMP:
                    # skip checked PandaIDs
                    if tmpPandaID in umCheckedIDs:
                        continue
                    # append to avoid redundant check
                    umCheckedIDs.append(tmpPandaID)
                    # check job status
                    varMap = {}
                    varMap[":PandaID"] = tmpPandaID
                    self.cur.execute(sqlUMS + comment, varMap)
                    resUMS = self.cur.fetchone()
                    # unmerged job should be in merging state
                    if resUMS is not None and resUMS[0] == "merging":
                        # append
                        umPandaIDs.append(tmpPandaID)
                        break
        # finish unmerge jobs
        sqlJFJ = f"SELECT {JobSpec.columnNames()} "
        sqlJFJ += "FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID"
        sqlJFF = f"SELECT {FileSpec.columnNames()} FROM ATLAS_PANDA.filesTable4 "
        sqlJFF += "WHERE PandaID=:PandaID"
        for tmpPandaID in umPandaIDs:
            # read job
            varMap = {}
            varMap[":PandaID"] = tmpPandaID
            self.cur.arraysize = 10
            self.cur.execute(sqlJFJ + comment, varMap)
            resJFJ = self.cur.fetchone()
            umJob = JobSpec()
            umJob.pack(resJFJ)
            umJob.jobStatus = job.jobStatus
            if umJob.jobStatus in ["failed"] or umJob.isCancelled():
                umJob.taskBufferErrorCode = ErrorCode.EC_MergeFailed
                umJob.taskBufferErrorDiag = f"merge job {umJob.jobStatus}"
                umJob.jobSubStatus = f"merge_{umJob.jobStatus}"
            # read files
            self.cur.arraysize = 10000
            self.cur.execute(sqlJFF + comment, varMap)
            resJFFs = self.cur.fetchall()
            for resJFF in resJFFs:
                umFile = FileSpec()
                umFile.pack(resJFF)
                if umFile.status not in ["nooutput"]:
                    umFile.status = umJob.jobStatus
                umJob.addFile(umFile)
            # finish
            tmpLog.debug(f"update unmerged PandaID={umJob.PandaID}")
            self.archiveJob(umJob, False, useCommit=False, async_params=async_params)
        return

    # disable further reattempt for pmerge
    def disableFurtherReattempt(self, jobSpec):
        comment = " /* JediDBProxy.disableFurtherReattempt */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <PandaID={jobSpec.PandaID}>"
        # sql to update file
        sqlFJ = f"UPDATE {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
        sqlFJ += "SET maxAttempt=attemptNr-1 "
        sqlFJ += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
        sqlFJ += "AND attemptNr=:attemptNr AND keepTrack=:keepTrack "
        nRow = 0
        for tmpFile in jobSpec.Files:
            # skip if no JEDI
            if tmpFile.fileID == "NULL":
                continue
            # only input
            if tmpFile.type not in ["input", "pseudo_input"]:
                continue
            # update JEDI contents
            varMap = {}
            varMap[":jediTaskID"] = tmpFile.jediTaskID
            varMap[":datasetID"] = tmpFile.datasetID
            varMap[":fileID"] = tmpFile.fileID
            varMap[":attemptNr"] = tmpFile.attemptNr
            varMap[":keepTrack"] = 1
            self.cur.execute(sqlFJ + comment, varMap)
            nRow += self.cur.rowcount
        # finish
        _logger.debug(f"{methodName} done with nRows={nRow}")
        return

    # update unmerged datasets to trigger merging
    def updateUnmergedDatasets(self, job, finalStatusDS, updateCompleted=False):
        comment = " /* JediDBProxy.updateUnmergedDatasets */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <PandaID={job.PandaID}>"
        # get PandaID which produced unmerged files
        umPandaIDs = []
        umCheckedIDs = []
        # sql to get file counts
        sqlGFC = "SELECT status,PandaID,outPandaID FROM ATLAS_PANDA.JEDI_Dataset_Contents "
        sqlGFC += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND PandaID IS NOT NULL "
        # sql to update nFiles in JEDI datasets
        sqlUNF = "UPDATE ATLAS_PANDA.JEDI_Datasets "
        sqlUNF += "SET nFilesOnHold=0,nFiles=:nFiles,"
        sqlUNF += "nFilesUsed=:nFilesUsed,nFilesTobeUsed=:nFilesTobeUsed "
        sqlUNF += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
        # sql to check nFiles
        sqlUCF = "SELECT nFilesTobeUsed,nFilesUsed FROM ATLAS_PANDA.JEDI_Datasets "
        sqlUCF += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
        # sql to update dataset status
        sqlUDS = "UPDATE ATLAS_PANDA.JEDI_Datasets "
        sqlUDS += "SET status=:status,modificationTime=CURRENT_DATE "
        sqlUDS += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
        # sql to update dataset status in panda
        sqlUDP = "UPDATE ATLAS_PANDA.Datasets "
        sqlUDP += "SET status=:status "
        sqlUDP += "WHERE vuid=:vuid AND NOT status IN (:statusR,:statusD) "
        try:
            _logger.debug(f"{methodName} start")
            # begin transaction
            self.conn.begin()
            # update dataset in panda
            toSkip = False
            for datasetSpec in finalStatusDS:
                varMap = {}
                varMap[":vuid"] = datasetSpec.vuid
                varMap[":status"] = "tobeclosed"
                varMap[":statusR"] = "tobeclosed"
                if not updateCompleted:
                    varMap[":statusD"] = "completed"
                else:
                    varMap[":statusD"] = "dummy"
                _logger.debug(methodName + " " + sqlUDP + comment + str(varMap))
                self.cur.execute(sqlUDP + comment, varMap)
                nRow = self.cur.rowcount
                if nRow != 1:
                    toSkip = True
                    _logger.debug(f"{methodName} failed to lock {datasetSpec.name}")
            # look for unmerged files
            if not toSkip:
                updatedDS = []
                for tmpFile in job.Files:
                    if tmpFile.isUnMergedOutput():
                        if tmpFile.datasetID in updatedDS:
                            continue
                        updatedDS.append(tmpFile.datasetID)
                        # get file counts
                        varMap = {}
                        varMap[":jediTaskID"] = tmpFile.jediTaskID
                        varMap[":datasetID"] = tmpFile.datasetID
                        self.cur.arraysize = 100000
                        _logger.debug(sqlGFC + comment + str(varMap))
                        self.cur.execute(sqlGFC + comment, varMap)
                        resListGFC = self.cur.fetchall()
                        varMap = {}
                        tmpNumFiles = 0
                        tmpNumReady = 0
                        for tmpFileStatus, tmpPandaID, tmpOutPandaID in resListGFC:
                            if tmpFileStatus in [
                                "finished",
                                "failed",
                                "cancelled",
                                "notmerged",
                                "ready",
                                "lost",
                                "broken",
                                "picked",
                                "nooutput",
                            ]:
                                pass
                            elif tmpFileStatus == "running" and tmpPandaID != tmpOutPandaID:
                                pass
                            else:
                                continue
                            tmpNumFiles += 1
                            if tmpFileStatus in ["ready"]:
                                tmpNumReady += 1
                        # update nFiles
                        varMap = {}
                        varMap[":jediTaskID"] = tmpFile.jediTaskID
                        varMap[":datasetID"] = tmpFile.datasetID
                        varMap[":nFiles"] = tmpNumFiles
                        varMap[":nFilesTobeUsed"] = tmpNumFiles
                        varMap[":nFilesUsed"] = tmpNumFiles - tmpNumReady
                        self.cur.arraysize = 10
                        _logger.debug(sqlUNF + comment + str(varMap))
                        self.cur.execute(sqlUNF + comment, varMap)
                        nRow = self.cur.rowcount
                        if nRow == 1:
                            # check nFilesTobeUsed
                            varMap = {}
                            varMap[":jediTaskID"] = tmpFile.jediTaskID
                            varMap[":datasetID"] = tmpFile.datasetID
                            self.cur.execute(sqlUCF + comment, varMap)
                            resUCF = self.cur.fetchone()
                            if resUCF is not None:
                                nFilesTobeUsed, nFilesUsed = resUCF
                                varMap = {}
                                varMap[":jediTaskID"] = tmpFile.jediTaskID
                                varMap[":datasetID"] = tmpFile.datasetID
                                if nFilesTobeUsed - nFilesUsed > 0:
                                    varMap[":status"] = "ready"
                                else:
                                    varMap[":status"] = "done"
                                # update dataset status
                                _logger.debug(methodName + " " + sqlUDS + comment + str(varMap))
                                self.cur.execute(sqlUDS + comment, varMap)
                        else:
                            _logger.debug(f"{methodName} skip jediTaskID={tmpFile.jediTaskID} datasetID={tmpFile.datasetID}")
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            _logger.debug(f"{methodName} done")
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False

    # get active JediTasks in a time range
    def getJediTasksInTimeRange(self, dn, timeRange, fullFlag=False, minTaskID=None, task_type="user"):
        comment = " /* DBProxy.getJediTasksInTimeRange */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        _logger.debug(f"{methodName} : DN={dn} range={timeRange.strftime('%Y-%m-%d %H:%M:%S')} full={fullFlag}")
        try:
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ["", "NULL", None]:
                compactDN = dn
            # make sql
            attrList = [
                "jediTaskID",
                "modificationTime",
                "status",
                "processingType",
                "transUses",
                "transHome",
                "architecture",
                "reqID",
                "creationDate",
                "site",
                "cloud",
                "taskName",
            ]
            sql = "SELECT "
            if fullFlag:
                sql += "* FROM (SELECT "
            for tmpAttr in attrList:
                sql += f"{tmpAttr},"
            sql = sql[:-1]
            sql += f" FROM {panda_config.schemaJEDI}.JEDI_Tasks "
            sql += "WHERE userName=:userName AND modificationTime>=:modificationTime AND prodSourceLabel=:prodSourceLabel "
            varMap = {}
            varMap[":userName"] = compactDN
            varMap[":prodSourceLabel"] = task_type
            varMap[":modificationTime"] = timeRange
            if minTaskID is not None:
                sql += "AND jediTaskID>:minTaskID "
                varMap[":minTaskID"] = minTaskID
            if fullFlag:
                sql += "ORDER BY jediTaskID) WHERE rownum<=500 "
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10000
            _logger.debug(sql + comment + str(varMap))
            self.cur.execute(sql + comment, varMap)
            resList = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # append
            retTasks = {}
            for tmpRes in resList:
                tmpDict = {}
                for tmpIdx, tmpAttr in enumerate(attrList):
                    tmpDict[tmpAttr] = tmpRes[tmpIdx]
                if fullFlag:
                    # additional info
                    addInfo = self.getJediTaskDigest(tmpDict["jediTaskID"])
                    for k in addInfo:
                        v = addInfo[k]
                        tmpDict[k] = v
                retTasks[tmpDict["reqID"]] = tmpDict
            _logger.debug(f"{methodName} : {str(retTasks)}")
            return retTasks
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return {}

    # get details of JediTask
    def getJediTaskDetails(self, jediTaskID, fullFlag, withTaskInfo):
        comment = " /* DBProxy.getJediTaskDetails */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        _logger.debug(f"{methodName} : jediTaskID={jediTaskID} full={fullFlag}")
        try:
            retDict = {
                "inDS": "",
                "outDS": "",
                "statistics": "",
                "PandaID": set(),
                "mergeStatus": None,
                "mergePandaID": set(),
            }
            # sql to get task status
            sqlT = f"SELECT status FROM {panda_config.schemaJEDI}.JEDI_Tasks WHERE jediTaskID=:jediTaskID "
            # sql to get datasets
            sqlD = "SELECT datasetID,datasetName,containerName,type,nFiles,nFilesTobeUsed,nFilesFinished,nFilesFailed,masterID,nFilesUsed,nFilesOnHold "
            sqlD += f"FROM {panda_config.schemaJEDI}.JEDI_Datasets "
            sqlD += "WHERE jediTaskID=:jediTaskID "
            # sql to get PandaIDs
            sqlP = f"SELECT PandaID,COUNT(*) FROM {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
            sqlP += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND PandaID IS NOT NULL "
            sqlP += "GROUP BY PandaID "
            # sql to get job status
            sqlJS = "SELECT PandaID,jobStatus,processingType FROM ATLAS_PANDA.jobsDefined4 "
            sqlJS += "WHERE jediTaskID=:jediTaskID AND prodSourceLabel=:prodSourceLabel "
            sqlJS += "UNION "
            sqlJS = "SELECT PandaID,jobStatus,processingType FROM ATLAS_PANDA.jobsActive4 "
            sqlJS += "WHERE jediTaskID=:jediTaskID AND prodSourceLabel=:prodSourceLabel "
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 100000
            # get task status
            if withTaskInfo:
                self.cur.execute(sqlT + comment, varMap)
                resT = self.cur.fetchone()
                if resT is None:
                    raise RuntimeError("No task info")
                retDict["status"] = resT[0]
            # get datasets
            self.cur.execute(sqlD + comment, varMap)
            resList = self.cur.fetchall()
            # get job status
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            varMap[":prodSourceLabel"] = "user"
            self.cur.execute(sqlJS + comment, varMap)
            resJS = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # make jobstatus map
            jobStatPandaIDs = {}
            for tmpPandaID, tmpJobStatus, tmpProcessingType in resJS:
                # ignore merge jobs
                if tmpProcessingType == "pmerge":
                    continue
                jobStatPandaIDs[tmpPandaID] = tmpJobStatus
            # append
            inDSs = []
            outDSs = []
            totalNumFiles = 0
            totalTobeDone = 0
            totalFinished = 0
            totalFailed = 0
            totalStatMap = {}
            for (
                datasetID,
                datasetName,
                containerName,
                datasetType,
                nFiles,
                nFilesTobeUsed,
                nFilesFinished,
                nFilesFailed,
                masterID,
                nFilesUsed,
                nFilesOnHold,
            ) in resList:
                # primay input
                if datasetType in ["input", "pseudo_input", "trn_log"] and masterID is None:
                    # unmerge dataset
                    if datasetType == "trn_log":
                        unmergeFlag = True
                    else:
                        unmergeFlag = False
                    # collect input dataset names
                    if datasetType == "input":
                        # use container name if not empty
                        if containerName not in [None, ""]:
                            targetName = containerName
                        else:
                            targetName = datasetName
                        if targetName not in inDSs:
                            inDSs.append(targetName)
                            retDict["inDS"] += f"{targetName},"
                    # statistics
                    if datasetType in ["input", "pseudo_input"]:
                        totalNumFiles += nFiles
                        totalFinished += nFilesFinished
                        totalFailed += nFilesFailed
                        totalTobeDone += nFiles - nFilesUsed
                    # collect PandaIDs
                    self.conn.begin()
                    varMap = {}
                    varMap[":jediTaskID"] = jediTaskID
                    varMap[":datasetID"] = datasetID
                    self.cur.execute(sqlP + comment, varMap)
                    resP = self.cur.fetchall()
                    # commit
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    for tmpPandaID, tmpNumFiles in resP:
                        if not unmergeFlag:
                            retDict["PandaID"].add(tmpPandaID)
                        else:
                            retDict["mergePandaID"].add(tmpPandaID)
                        # map to job status
                        if datasetType in ["input", "pseudo_input"]:
                            if tmpPandaID in jobStatPandaIDs:
                                tmpJobStatus = jobStatPandaIDs[tmpPandaID]
                                if tmpJobStatus not in totalStatMap:
                                    totalStatMap[tmpJobStatus] = 0
                                totalStatMap[tmpJobStatus] += tmpNumFiles
                # output
                if datasetType.endswith("output") or datasetType.endswith("log"):
                    # ignore transient datasets
                    if "trn_" in datasetType:
                        continue
                    # use container name if not empty
                    if containerName not in [None, ""]:
                        targetName = containerName
                    else:
                        targetName = datasetName
                    if targetName not in outDSs:
                        outDSs.append(targetName)
                        retDict["outDS"] += f"{targetName},"
            retDict["inDS"] = retDict["inDS"][:-1]
            retDict["outDS"] = retDict["outDS"][:-1]
            # statistics
            statStr = ""
            nPicked = totalNumFiles
            if totalTobeDone > 0:
                statStr += f"tobedone*{totalTobeDone},"
                nPicked -= totalTobeDone
            if totalFinished > 0:
                statStr += f"finished*{totalFinished},"
                nPicked -= totalFinished
            if totalFailed > 0:
                statStr += f"failed*{totalFailed},"
                nPicked -= totalFailed
            for tmpJobStatus in totalStatMap:
                tmpNumFiles = totalStatMap[tmpJobStatus]
                # skip active failed
                if tmpJobStatus == "failed":
                    continue
                statStr += f"{tmpJobStatus}*{tmpNumFiles},"
                nPicked -= tmpNumFiles
            if nPicked > 0:
                statStr += f"picked*{nPicked},"
            retDict["statistics"] = statStr[:-1]
            # command line parameters
            if fullFlag:
                # sql to read task params
                sql = f"SELECT taskParams FROM {panda_config.schemaJEDI}.JEDI_TaskParams WHERE jediTaskID=:jediTaskID "
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                # begin transaction
                self.conn.begin()
                self.cur.execute(sql + comment, varMap)
                retStr = ""
                for (tmpItem,) in self.cur:
                    try:
                        retStr = tmpItem.read()
                    except AttributeError:
                        retStr = str(tmpItem)
                    break
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                # decode json
                taskParamsJson = json.loads(retStr)
                if "cliParams" in taskParamsJson:
                    retDict["cliParams"] = taskParamsJson["cliParams"]
                else:
                    retDict["cliParams"] = ""
            retDict["PandaID"] = list(retDict["PandaID"])
            retDict["mergePandaID"] = list(retDict["mergePandaID"])
            _logger.debug(f"{methodName} : {str(retDict)}")
            return retDict
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return {}

    # get JediTask digest
    def getJediTaskDigest(self, jediTaskID):
        comment = " /* DBProxy.getJediTaskDigest */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" < jediTaskID={jediTaskID} >"
        tmpLog = LogWrapper(_logger, methodName)
        try:
            retDict = {
                "inDS": "",
                "outDS": "",
                "statistics": "",
                "PandaID": [],
                "mergeStatus": None,
                "mergePandaID": [],
            }
            # sql to get datasets
            sqlD = "SELECT datasetName,containerName,type "
            sqlD += f"FROM {panda_config.schemaJEDI}.JEDI_Datasets "
            sqlD += "WHERE jediTaskID=:jediTaskID AND ((type IN (:in1,:in2) AND masterID IS NULL) OR type IN (:out1,:out2)) "
            sqlD += "GROUP BY datasetName,containerName,type "
            # sql to get job status
            sqlJS = "SELECT proc_status,COUNT(*) FROM {0}.JEDI_Datasets d,{0}.JEDI_Dataset_Contents c ".format(panda_config.schemaJEDI)
            sqlJS += "WHERE c.jediTaskID=d.jediTaskID AND c.datasetID=d.datasetID AND d.jediTaskID=:jediTaskID "
            sqlJS += "AND d.type IN (:in1,:in2) AND d.masterID IS NULL "
            sqlJS += "GROUP BY proc_status "
            # sql to read task params
            sqlTP = f"SELECT taskParams FROM {panda_config.schemaJEDI}.JEDI_TaskParams WHERE jediTaskID=:jediTaskID "
            # start transaction
            self.conn.begin()
            self.cur.arraysize = 100000
            # get datasets
            inDSs = set()
            outDSs = set()
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            varMap[":in1"] = "input"
            varMap[":in2"] = "pseudo_input"
            varMap[":out1"] = "output"
            varMap[":out2"] = "tmpl_output"
            self.cur.execute(sqlD + comment, varMap)
            resList = self.cur.fetchall()
            for datasetName, containerName, datasetType in resList:
                # use container name if not empty
                if containerName not in [None, ""]:
                    targetName = containerName
                else:
                    targetName = datasetName
                if "output" in datasetType:
                    outDSs.add(targetName)
                else:
                    inDSs.add(targetName)
            inDSs = sorted(inDSs)
            retDict["inDS"] = ",".join(inDSs)
            outDSs = sorted(outDSs)
            retDict["outDS"] = ",".join(outDSs)
            # get job status
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            varMap[":in1"] = "input"
            varMap[":in2"] = "pseudo_input"
            self.cur.execute(sqlJS + comment, varMap)
            resJS = self.cur.fetchall()
            jobStatMap = dict()
            for proc_status, ninputs in resJS:
                jobStatMap[proc_status] = ninputs
            psList = sorted(jobStatMap)
            retDict["statistics"] = ",".join([f"{j}*{jobStatMap[j]}" for j in psList])
            # command line parameters
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            self.cur.execute(sqlTP + comment, varMap)
            retStr = ""
            for (tmpItem,) in self.cur:
                try:
                    retStr = tmpItem.read()
                except AttributeError:
                    retStr = str(tmpItem)
                break
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # decode json
            taskParamsJson = json.loads(retStr)
            if "cliParams" in taskParamsJson:
                retDict["cliParams"] = taskParamsJson["cliParams"]
            else:
                retDict["cliParams"] = ""
            tmpLog.debug(f"{str(retDict)}")
            return retDict
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return {}

    # make event range ID for event service
    def makeEventRangeID(self, jediTaskID, pandaID, fileID, job_processID, attemptNr):
        return f"{jediTaskID}-{pandaID}-{fileID}-{job_processID}-{attemptNr}"

    # get a list of even ranges for a PandaID
    def getEventRanges(self, pandaID, jobsetID, jediTaskID, nRanges, acceptJson, scattered, segment_id):
        comment = " /* DBProxy.getEventRanges */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <PandaID={pandaID} jobsetID={jobsetID} jediTaskID={jediTaskID}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug(f"start nRanges={nRanges} scattered={scattered} segment={segment_id}")
        try:
            regStart = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            # convert to int
            try:
                nRanges = int(nRanges)
            except Exception:
                nRanges = 8
            try:
                pandaID = int(pandaID)
            except Exception:
                pass
            try:
                jobsetID = int(jobsetID)
            except Exception:
                pass
            try:
                jediTaskID = int(jediTaskID)
            except Exception:
                jediTaskID = None
            iRanges = 0
            # sql to get job
            sqlJ = f"SELECT jobStatus,commandToPilot,eventService,jediTaskID FROM {panda_config.schemaPANDA}.jobsActive4 "
            sqlJ += "WHERE PandaID=:pandaID FOR UPDATE "
            # sql to find a file to lock
            sqlFF = f"SELECT jediTaskID,datasetID,fileID FROM {panda_config.schemaPANDA}.filesTable4 "
            sqlFF += "WHERE PandaID=:pandaID AND type IN (:type1,:type2) "
            sqlFF += "ORDER BY fileID "
            # sql to use a dataset as lock
            sqlLD = f"SELECT status FROM {panda_config.schemaJEDI}.JEDI_Datasets "
            sqlLD += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
            sqlLD += "FOR UPDATE "
            # sql to use a file as lock
            sqlLK = f"SELECT status FROM {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
            sqlLK += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            sqlLK += "FOR UPDATE "
            # sql to get ranges with jediTaskID
            sqlW = f"UPDATE {panda_config.schemaJEDI}.JEDI_Events tab "
            sqlW += "SET PandaID=:pandaID,status=:newEventStatus "
            sqlW += "WHERE (jediTaskID,PandaID,fileID,job_processID,attemptNr) IN ("
            sqlW += "SELECT jediTaskID,PandaID,fileID,job_processID,attemptNr FROM ("
            sqlW += "SELECT jediTaskID,PandaID,fileID,job_processID,attemptNr FROM "
            sqlW += "/* sorted by JEDITASKID, PANDAID, FILEID to take advantage of the IOT table structure*/ "
            sqlW += f"{panda_config.schemaJEDI}.JEDI_Events tab "
            sqlW += "WHERE jediTaskID=:jediTaskID AND PandaID=:jobsetID AND status=:eventStatus AND attemptNr>:minAttemptNr "
            if segment_id is not None:
                sqlW += "AND datasetID=:datasetID "
            sqlW += "ORDER BY jediTaskID,PandaID,fileID "
            sqlW += f") WHERE rownum<={nRanges + 1}) "
            # sql to get ranges for jumbo
            sqlJM = f"UPDATE {panda_config.schemaJEDI}.JEDI_Events tab "
            sqlJM += "SET PandaID=:pandaID,status=:newEventStatus "
            sqlJM += "WHERE (jediTaskID,PandaID,fileID,job_processID,attemptNr) IN ("
            sqlJM += "SELECT jediTaskID,PandaID,fileID,job_processID,attemptNr FROM ("
            sqlJM += "SELECT jediTaskID,PandaID,fileID,job_processID,attemptNr FROM "
            sqlJM += "/* sorted by JEDITASKID, PANDAID, FILEID to take advantage of the IOT table structure*/ "
            sqlJM += f"{panda_config.schemaJEDI}.JEDI_Events tab "
            sqlJM += "WHERE jediTaskID=:jediTaskID AND status=:eventStatus AND attemptNr>:minAttemptNr "
            if scattered:
                pass
            else:
                sqlJM += "ORDER BY jediTaskID,PandaID,fileID "
            sqlJM += f") WHERE rownum<={nRanges + 1}) "
            # sql to get ranges
            sqlRR = "SELECT jediTaskID,datasetID,fileID,attemptNr,job_processID,def_min_eventID,def_max_eventID,event_offset "
            sqlRR += f"FROM {panda_config.schemaJEDI}.JEDI_Events tab "
            sqlRR += "WHERE jediTaskID=:jediTaskID AND PandaID=:PandaID AND status=:eventStatus "
            # sql to get datasets
            sqlGD = f"SELECT datasetID FROM {panda_config.schemaJEDI}.JEDI_Datasets "
            sqlGD += "WHERE jediTaskID=:jediTaskID AND type IN (:type1,:type2) "
            # sql to update files in the jobset
            sqlJS = f"UPDATE {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
            sqlJS += "SET status=:newStatus,is_waiting=NULL "
            sqlJS += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
            sqlJS += "AND status=:oldStatus AND keepTrack=:keepTrack AND PandaID IN ("
            # sql to update dataset
            sqlUD = f"UPDATE {panda_config.schemaJEDI}.JEDI_Datasets "
            sqlUD += "SET nFilesUsed=nFilesUsed+:nDiff,nFilesWaiting=nFilesWaiting-:nDiff "
            sqlUD += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
            # sql to get file info
            sqlF = f"SELECT lfn,GUID,scope FROM {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
            sqlF += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            # sql to lock range
            sqlU = f"UPDATE {panda_config.schemaJEDI}.JEDI_Events "
            sqlU += "SET status=:eventStatus,is_jumbo=:isJumbo "
            sqlU += "WHERE jediTaskID=:jediTaskID AND PandaID=:pandaID "
            sqlU += "AND status=:oldEventStatus "
            # sql to release range
            sqlRS = f"UPDATE {panda_config.schemaJEDI}.JEDI_Events "
            sqlRS += "SET PandaID=event_offset,status=:eventStatus "
            sqlRS += "WHERE jediTaskID=:jediTaskID AND fileID=:fileID AND PandaID=:pandaID "
            sqlRS += "AND job_processID=:job_processID AND attemptNr=:attemptNr "
            sqlRS += "AND status=:oldEventStatus "
            # start transaction
            self.conn.begin()
            self.cur.arraysize = 100000
            # get job
            varMap = {}
            varMap[":pandaID"] = pandaID
            self.cur.execute(sqlJ + comment, varMap)
            resJ = self.cur.fetchone()
            toSkip = True
            retRanges = []
            noMoreEvents = False
            if resJ is None:
                # job not found
                tmpLog.debug("skip job not found")
            elif resJ[0] not in ["sent", "running", "starting"]:
                # wrong job status
                tmpLog.debug(f"skip wrong job status in {resJ[0]}")
            elif resJ[1] == "tobekilled":
                # job is being killed
                tmpLog.debug("skip job is being killed")
            else:
                toSkip = False
                # jumbo
                if resJ[2] == EventServiceUtils.jumboJobFlagNumber:
                    isJumbo = True
                else:
                    isJumbo = False
                # get jediTaskID
                if jediTaskID is None:
                    jediTaskID = resJ[3]
                # find a file to lock
                varMap = dict()
                varMap[":pandaID"] = pandaID
                varMap[":type1"] = "input"
                varMap[":type2"] = "pseudo_input"
                self.cur.execute(sqlFF + comment, varMap)
                resFF = self.cur.fetchone()
                if resFF is not None:
                    ffJediTask, ffDatasetID, ffFileID = resFF
                    varMap = dict()
                    varMap[":jediTaskID"] = ffJediTask
                    varMap[":datasetID"] = ffDatasetID
                    if isJumbo:
                        self.cur.execute(sqlLD + comment, varMap)
                        tmpLog.debug(f"locked datasetID={ffDatasetID}")
                # prelock event ranges
                varMap = {}
                varMap[":eventStatus"] = EventServiceUtils.ST_ready
                varMap[":minAttemptNr"] = 0
                varMap[":jediTaskID"] = jediTaskID
                varMap[":pandaID"] = pandaID
                varMap[":eventStatus"] = EventServiceUtils.ST_ready
                varMap[":newEventStatus"] = EventServiceUtils.ST_reserved_get
                if segment_id is not None:
                    varMap[":datasetID"] = segment_id
                if not isJumbo:
                    varMap[":jobsetID"] = jobsetID
                if isJumbo:
                    tmpLog.debug(sqlJM + comment + str(varMap))
                    self.cur.execute(sqlJM + comment, varMap)
                else:
                    self.cur.execute(sqlW + comment, varMap)
                nRow = self.cur.rowcount
                tmpLog.debug(f"pre-locked {nRow} events")
                # get event ranges
                varMap = dict()
                varMap[":jediTaskID"] = jediTaskID
                varMap[":PandaID"] = pandaID
                varMap[":eventStatus"] = EventServiceUtils.ST_reserved_get
                tmpLog.debug(sqlRR + comment + str(varMap))
                self.cur.execute(sqlRR + comment, varMap)
                resList = self.cur.fetchall()
                if len(resList) > nRanges:
                    # release the last event range
                    (
                        tmpJediTaskID,
                        datasetID,
                        fileID,
                        attemptNr,
                        job_processID,
                        startEvent,
                        lastEvent,
                        tmpJobsetID,
                    ) = resList[-1]
                    varMap = {}
                    varMap[":jediTaskID"] = tmpJediTaskID
                    varMap[":fileID"] = fileID
                    varMap[":job_processID"] = job_processID
                    varMap[":pandaID"] = pandaID
                    varMap[":attemptNr"] = attemptNr
                    varMap[":eventStatus"] = EventServiceUtils.ST_ready
                    varMap[":oldEventStatus"] = EventServiceUtils.ST_reserved_get
                    self.cur.execute(sqlRS + comment, varMap)
                    resList = resList[:nRanges]
                else:
                    noMoreEvents = True
                # make dict
                fileInfo = {}
                jobsetList = {}
                for (
                    tmpJediTaskID,
                    datasetID,
                    fileID,
                    attemptNr,
                    job_processID,
                    startEvent,
                    lastEvent,
                    tmpJobsetID,
                ) in resList:
                    # get file info
                    if fileID not in fileInfo:
                        varMap = {}
                        varMap[":jediTaskID"] = tmpJediTaskID
                        varMap[":datasetID"] = datasetID
                        varMap[":fileID"] = fileID
                        self.cur.execute(sqlF + comment, varMap)
                        resF = self.cur.fetchone()
                        # not found
                        if resF is None:
                            resF = (None, None, None)
                            tmpLog.warning(f"file info is not found for fileID={fileID}")
                        fileInfo[fileID] = resF
                    # get LFN and GUID
                    tmpLFN, tmpGUID, tmpScope = fileInfo[fileID]
                    # make dict
                    tmpDict = {
                        "eventRangeID": self.makeEventRangeID(tmpJediTaskID, pandaID, fileID, job_processID, attemptNr),
                        "startEvent": startEvent,
                        "lastEvent": lastEvent,
                        "LFN": tmpLFN,
                        "GUID": tmpGUID,
                        "scope": tmpScope,
                    }
                    # append
                    retRanges.append(tmpDict)
                    iRanges += 1
                    if tmpJediTaskID not in jobsetList:
                        jobsetList[tmpJediTaskID] = []
                    jobsetList[tmpJediTaskID].append(tmpJobsetID)
                tmpLog.debug(f"got {len(retRanges)} events")
                # lock events
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":pandaID"] = pandaID
                varMap[":eventStatus"] = EventServiceUtils.ST_sent
                varMap[":oldEventStatus"] = EventServiceUtils.ST_reserved_get
                if isJumbo:
                    varMap[":isJumbo"] = EventServiceUtils.eventTableIsJumbo
                else:
                    varMap[":isJumbo"] = None
                self.cur.execute(sqlU + comment, varMap)
                nRow = self.cur.rowcount
                tmpLog.debug(f"locked {nRow} events")
                # kill unused consumers
                if not isJumbo and not toSkip and (retRanges == [] or noMoreEvents) and jediTaskID is not None and segment_id is None:
                    tmpLog.debug("kill unused consumers")
                    tmpJobSpec = JobSpec()
                    tmpJobSpec.PandaID = pandaID
                    tmpJobSpec.jobsetID = jobsetID
                    tmpJobSpec.jediTaskID = jediTaskID
                    self.killUnusedEventServiceConsumers(tmpJobSpec, False, checkAttemptNr=True)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            regTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - regStart
            tmpLog.debug(f"done with {iRanges} event ranges. took {regTime.seconds} sec")
            if not acceptJson:
                return json.dumps(retRanges)
            return retRanges
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None

    # update even ranges
    def updateEventRanges(self, eventDictParam, version=0):
        # version 0: normal event service
        # version 1: jumbo jobs with zip file support
        # version 2: fine-grained processing where events can be updated before being dispatched
        comment = " /* DBProxy.updateEventRanges */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <{datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None).isoformat('/')}>"
        tmpLog = LogWrapper(_logger, methodName)
        try:
            regStart = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            retList = []
            jobAttrs = {}
            commandMap = {}
            # sql to update status
            sqlU = f"UPDATE {panda_config.schemaJEDI}.JEDI_Events "
            sqlU += "SET status=:eventStatus,objstore_ID=:objstoreID,error_code=:errorCode," "path_convention=:pathConvention,error_diag=:errorDiag"
            if version != 0:
                sqlU += ",zipRow_ID=:zipRow_ID"
            sqlU += " WHERE jediTaskID=:jediTaskID AND pandaID=:pandaID AND fileID=:fileID "
            sqlU += "AND job_processID=:job_processID AND attemptNr=:attemptNr "
            if version == 2:
                sqlU += "AND status IN (:esSent, :esRunning, :esReady) "
            else:
                sqlU += "AND status IN (:esSent, :esRunning) "
            # sql to get event range
            sqlC = f"SELECT splitRule FROM {panda_config.schemaJEDI}.JEDI_Tasks "
            sqlC += "WHERE jediTaskID=:jediTaskID "
            # sql to get nEvents
            sqlE = "SELECT jobStatus,nEvents,commandToPilot,supErrorCode,specialHandling FROM ATLAS_PANDA.jobsActive4 "
            sqlE += "WHERE PandaID=:pandaID "
            if version == 2:
                sqlE += "OR jobsetID=:pandaID "
            # sql to set nEvents
            sqlS = "UPDATE ATLAS_PANDA.jobsActive4 "
            sqlS += f"SET nEvents=(SELECT COUNT(1) FROM {panda_config.schemaJEDI}.JEDI_Events "
            sqlS += "WHERE jediTaskID=:jediTaskID AND PandaID=:PandaID AND status IN (:esFinished,:esDone,:esMerged))*:nEvents "
            sqlS += "WHERE PandaID=:pandaID "
            if version == 2:
                sqlS += "OR jobsetID=:pandaID "
            # sql to check zip file
            sqlFC = "SELECT row_ID FROM ATLAS_PANDA.filesTable4 "
            sqlFC += "WHERE PandaID=:pandaID AND lfn=:lfn "
            # sql to insert zip file
            sqlF = f"INSERT INTO ATLAS_PANDA.filesTable4 ({FileSpec.columnNames()}) "
            sqlF += FileSpec.bindValuesExpression(useSeq=True)
            sqlF += " RETURNING row_ID INTO :newRowID"
            # sql for fatal events
            sqlFA = f"UPDATE {panda_config.schemaJEDI}.JEDI_Events "
            sqlFA += "SET attemptNr=:newAttemptNr "
            sqlFA += " WHERE jediTaskID=:jediTaskID AND pandaID=:pandaID AND fileID=:fileID "
            sqlFA += "AND job_processID=:job_processID AND attemptNr=:oldAttemptNr "
            sqlFA += "AND status=:eventStatus "
            # params formatting with version
            if version == 0:
                # format without zip
                eventDictList = eventDictParam
            else:
                # format with zip
                eventDictList = []
                for eventDictChunk in eventDictParam:
                    # get zip file if any
                    if "zipFile" in eventDictChunk:
                        zipFile = eventDictChunk["zipFile"]
                    else:
                        zipFile = None
                    # collect all dicts
                    if "eventRanges" in eventDictChunk:
                        for eventDict in eventDictChunk["eventRanges"]:
                            # add zip file
                            eventDict["zipFile"] = zipFile
                            # append
                            eventDictList.append(eventDict)
                    else:
                        eventDictList.append(eventDictChunk)
            # update events
            tmpLog.debug(f"total {len(eventDictList)} events")
            zipRowIdMap = {}
            nEventsMap = dict()
            iEvents = 0
            maxEvents = 100000
            iSkipped = 0
            ok_job_status = ["sent", "running", "starting", "transferring"]
            if version == 2:
                ok_job_status += ["activated"]
            # start transaction
            self.conn.begin()
            # loop over all events
            varMapListU = []
            varMapListFA = []
            for eventDict in eventDictList:
                # avoid too many events
                iEvents += 1
                if iEvents > maxEvents:
                    retList.append(None)
                    iSkipped += 1
                    continue
                # get event range ID
                if "eventRangeID" not in eventDict:
                    tmpLog.error(f"eventRangeID is missing in {str(eventDict)}")
                    retList.append(False)
                    continue
                eventRangeID = eventDict["eventRangeID"]
                # decompose eventRangeID
                try:
                    tmpItems = eventRangeID.split("-")
                    jediTaskID, pandaID, fileID, job_processID, attemptNr = tmpItems
                    jediTaskID = int(jediTaskID)
                    pandaID = int(pandaID)
                    fileID = int(fileID)
                    job_processID = int(job_processID)
                    attemptNr = int(attemptNr)
                except Exception:
                    _logger.error(f"{methodName} : wrongly formatted eventRangeID")
                    retList.append(False)
                    continue
                # get event status
                if "eventStatus" not in eventDict:
                    tmpLog.error(f"<eventRangeID={eventRangeID}> eventStatus is missing in {str(eventDict)}")
                    retList.append(False)
                    continue
                eventStatus = eventDict["eventStatus"]
                # map string status to int
                isFatal = False
                if eventStatus == "running":
                    intEventStatus = EventServiceUtils.ST_running
                elif eventStatus == "transferring":
                    intEventStatus = EventServiceUtils.ST_running
                elif eventStatus == "finished":
                    intEventStatus = EventServiceUtils.ST_finished
                elif eventStatus == "failed":
                    intEventStatus = EventServiceUtils.ST_failed
                elif eventStatus == "fatal":
                    intEventStatus = EventServiceUtils.ST_failed
                    isFatal = True
                else:
                    tmpLog.error(f"<eventRangeID={eventRangeID}> unknown status {eventStatus}")
                    retList.append(False)
                    continue
                # only final status
                if eventStatus not in ["finished", "failed", "fatal"]:
                    retList.append(None)
                    iSkipped += 1
                    tmpLog.debug(f"<eventRangeID={eventRangeID}> eventStatus={eventStatus} skipped")
                    continue
                # core count
                coreCount = eventDict.get("coreCount")
                # CPU consumption
                cpuConsumptionTime = eventDict.get("cpuConsumptionTime")
                # objectstore ID
                objstoreID = eventDict.get("objstoreID")
                # error code
                errorCode = eventDict.get("errorCode")
                # path convention
                pathConvention = eventDict.get("pathConvention")
                # error diag
                errorDiag = eventDict.get("errorDiag")
                isOK = True
                # get job attributes
                if pandaID not in jobAttrs:
                    varMap = {}
                    varMap[":pandaID"] = pandaID
                    self.cur.execute(sqlE + comment, varMap)
                    resE = self.cur.fetchone()
                    jobAttrs[pandaID] = resE
                    tmpLog.debug(f"PandaID={pandaID}")
                resE = jobAttrs[pandaID]
                if resE is None:
                    tmpLog.error(f"<eventRangeID={eventRangeID}> unknown PandaID")
                    retList.append(False)
                    isOK = False
                    commandToPilot = "tobekilled"
                else:
                    # check job status
                    (
                        jobStatus,
                        nEventsOld,
                        commandToPilot,
                        supErrorCode,
                        specialHandling,
                    ) = resE
                    if jobStatus not in ok_job_status:
                        tmpLog.error(f"<eventRangeID={eventRangeID}> wrong jobStatus={jobStatus}")
                        _logger.debug(f"{methodName} : wrong jobStatus={jobStatus}")
                        retList.append(False)
                        isOK = False
                    else:
                        # insert zip
                        zipRow_ID = None
                        if "zipFile" in eventDict and eventDict["zipFile"] is not None:
                            if eventDict["zipFile"]["lfn"] in zipRowIdMap:
                                zipRow_ID = zipRowIdMap[eventDict["zipFile"]["lfn"]]
                            else:
                                # check zip
                                varMap = dict()
                                varMap[":pandaID"] = pandaID
                                varMap[":lfn"] = eventDict["zipFile"]["lfn"]
                                self.cur.execute(sqlFC + comment, varMap)
                                resFC = self.cur.fetchone()
                                if resFC is not None:
                                    (zipRow_ID,) = resFC
                                else:
                                    # insert new file
                                    zipJobSpec = JobSpec()
                                    zipJobSpec.PandaID = pandaID
                                    zipJobSpec.specialHandling = specialHandling
                                    zipFileSpec = FileSpec()
                                    zipFileSpec.jediTaskID = jediTaskID
                                    zipFileSpec.lfn = eventDict["zipFile"]["lfn"]
                                    zipFileSpec.GUID = str(uuid.uuid4())
                                    if "fsize" in eventDict["zipFile"]:
                                        zipFileSpec.fsize = int(eventDict["zipFile"]["fsize"])
                                    else:
                                        zipFileSpec.fsize = 0
                                    if "adler32" in eventDict["zipFile"]:
                                        zipFileSpec.checksum = f"ad:{eventDict['zipFile']['adler32']}"
                                    if "numEvents" in eventDict["zipFile"]:
                                        zipFileSpec.dispatchDBlockToken = eventDict["zipFile"]["numEvents"]
                                    zipFileSpec.type = "zipoutput"
                                    zipFileSpec.status = "ready"
                                    zipFileSpec.destinationSE = eventDict["zipFile"]["objstoreID"]
                                    if "pathConvention" in eventDict["zipFile"]:
                                        zipFileSpec.destinationSE = f"{zipFileSpec.destinationSE}/{eventDict['zipFile']['pathConvention']}"
                                    zipJobSpec.addFile(zipFileSpec)
                                    varMap = zipFileSpec.valuesMap(useSeq=True)
                                    varMap[":newRowID"] = self.cur.var(varNUMBER)
                                    self.cur.execute(sqlF + comment, varMap)
                                    val = self.getvalue_corrector(self.cur.getvalue(varMap[":newRowID"]))
                                    zipRow_ID = int(val)
                                    zipRowIdMap[eventDict["zipFile"]["lfn"]] = zipRow_ID
                                    # make an empty file to trigger registration for zip files in Adder
                                    if zipJobSpec.registerEsFiles():
                                        # tmpFileName = '{0}_{1}_{2}'.format(pandaID, EventServiceUtils.esRegStatus,
                                        #                                    uuid.uuid3(uuid.NAMESPACE_DNS,''))
                                        # tmpFileName = os.path.join(panda_config.logdir, tmpFileName)
                                        # try:
                                        #     open(tmpFileName, 'w').close()
                                        # except Exception:
                                        #     pass
                                        # sql to insert
                                        sqlI = (
                                            "INSERT INTO {0}.Job_Output_Report "
                                            "(PandaID, prodSourceLabel, jobStatus, attemptNr, data, timeStamp) "
                                            "VALUES(:PandaID, :prodSourceLabel, :jobStatus, :attemptNr, :data, :timeStamp) "
                                        ).format(panda_config.schemaPANDA)
                                        # insert
                                        varMap = {}
                                        varMap[":PandaID"] = pandaID
                                        varMap[":prodSourceLabel"] = zipJobSpec.prodSourceLabel
                                        varMap[":jobStatus"] = zipJobSpec.jobStatus
                                        varMap[":attemptNr"] = 0 if zipJobSpec.attemptNr in [None, "NULL", ""] else zipJobSpec.attemptNr
                                        varMap[":data"] = None
                                        varMap[":timeStamp"] = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
                                        try:
                                            self.cur.execute(sqlI + comment, varMap)
                                        except Exception:
                                            pass
                                        else:
                                            tmpLog.debug(f"successfully inserted job output report {pandaID}.{varMap[':attemptNr']}")
                        # update event
                        varMap = {}
                        varMap[":jediTaskID"] = jediTaskID
                        varMap[":pandaID"] = pandaID
                        varMap[":fileID"] = fileID
                        varMap[":job_processID"] = job_processID
                        varMap[":attemptNr"] = attemptNr
                        varMap[":eventStatus"] = intEventStatus
                        varMap[":objstoreID"] = objstoreID
                        varMap[":errorCode"] = errorCode
                        varMap[":pathConvention"] = pathConvention
                        varMap[":errorDiag"] = errorDiag
                        varMap[":esSent"] = EventServiceUtils.ST_sent
                        varMap[":esRunning"] = EventServiceUtils.ST_running
                        if version == 2:
                            varMap[":esReady"] = EventServiceUtils.ST_ready
                        if version != 0:
                            varMap[":zipRow_ID"] = zipRow_ID
                        varMapListU.append(varMap)
                        # fatal event
                        if isFatal:
                            varMap = {}
                            varMap[":jediTaskID"] = jediTaskID
                            varMap[":pandaID"] = pandaID
                            varMap[":fileID"] = fileID
                            varMap[":job_processID"] = job_processID
                            varMap[":oldAttemptNr"] = attemptNr
                            varMap[":newAttemptNr"] = 1
                            varMap[":eventStatus"] = EventServiceUtils.ST_failed
                            varMapListFA.append(varMap)
                        # nEvents of finished
                        if eventStatus in ["finished"]:
                            # get nEvents
                            if pandaID not in nEventsMap:
                                nEventsDef = 1
                                varMap = {}
                                varMap[":jediTaskID"] = jediTaskID
                                self.cur.execute(sqlC + comment, varMap)
                                resC = self.cur.fetchone()
                                if resC is not None:
                                    (splitRule,) = resC
                                    tmpM = re.search("ES=(\d+)", splitRule)
                                    if tmpM is not None:
                                        nEventsDef = int(tmpM.group(1))
                                nEventsMap[pandaID] = {
                                    "jediTaskID": jediTaskID,
                                    "nEvents": nEventsDef,
                                }
                    # soft kill
                    if commandToPilot not in [None, ""] and supErrorCode in [ErrorCode.EC_EventServicePreemption]:
                        commandToPilot = "softkill"
                if isOK:
                    retList.append(True)
                if pandaID not in commandMap:
                    commandMap[pandaID] = commandToPilot
            tmpLog.debug(f"update {len(varMapListU)} events")
            if len(varMapListU) > 0:
                self.cur.executemany(sqlU + comment, varMapListU)
            tmpLog.debug(f"fatal {len(varMapListFA)} events")
            if len(varMapListFA) > 0:
                self.cur.executemany(sqlFA + comment, varMapListFA)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # update nevents
            for pandaID in nEventsMap:
                data = nEventsMap[pandaID]
                self.conn.begin()
                varMap = {}
                varMap[":pandaID"] = pandaID
                varMap[":jediTaskID"] = data["jediTaskID"]
                varMap[":nEvents"] = data["nEvents"]
                varMap[":esFinished"] = EventServiceUtils.ST_finished
                varMap[":esDone"] = EventServiceUtils.ST_done
                varMap[":esMerged"] = EventServiceUtils.ST_merged
                self.cur.execute(sqlS + comment, varMap)
                if not self._commit():
                    raise RuntimeError("Commit error")
            regTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - regStart
            tmpLog.debug(f"done. {iSkipped} events out of {len(eventDictList)} events skipped. took {regTime.seconds} sec")
            return retList, commandMap
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            retList.append(False)
            return retList, commandMap

    # post-process for event service job
    def ppEventServiceJob(self, job, currentJobStatus, useCommit=True):
        comment = " /* DBProxy.ppEventServiceJob */"
        pandaID = job.PandaID
        attemptNr = job.attemptNr
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" < PandaID={pandaID} >"
        _logger.debug(f"{methodName} : start attemptNr={attemptNr}")
        try:
            # return values
            # 0 : generated a retry job
            # 1 : not retried due to a harmless reason
            # 2 : generated a merge job
            # 3 : max attempts reached
            # 4 : not generated a merge job since other consumers are still running
            # 5 : didn't process any events on WN
            # 6 : didn't process any events on WN and fail since the last one
            # 7 : all event ranges failed
            # 8 : generated a retry job but no events were processed
            # 9 : closed in bad job status
            # 10 : generated a merge job but didn't process any events by itself
            # None : fatal error
            retValue = 1, None
            # begin transaction
            if useCommit:
                self.conn.begin()
            self.cur.arraysize = 10
            # make job spec to not change the original
            jobSpec = copy.copy(job)
            jobSpec.Files = []
            # check if event service job
            if not EventServiceUtils.isEventServiceJob(jobSpec):
                _logger.debug(f"{methodName} : no event service job")
                # commit
                if useCommit:
                    if not self._commit():
                        raise RuntimeError("Commit error")
                return retValue
            # check if already retried or not good for retry
            if jobSpec.taskBufferErrorCode in [
                ErrorCode.EC_EventServiceRetried,
                ErrorCode.EC_EventServiceMerge,
                ErrorCode.EC_EventServiceInconsistentIn,
                ErrorCode.EC_EventServiceBadStatus,
            ]:
                _logger.debug(f"{methodName} : already post-processed for event service with EC={jobSpec.taskBufferErrorCode}")
                # commit
                if useCommit:
                    if not self._commit():
                        raise RuntimeError("Commit error")
                return retValue
            # check if JEDI is used
            if (
                hasattr(panda_config, "useJEDI")
                and panda_config.useJEDI is True
                and jobSpec.lockedby == "jedi"
                and self.checkTaskStatusJEDI(jobSpec.jediTaskID, self.cur)
            ):
                pass
            else:
                _logger.debug(f"{methodName} : JEDI is not used")
                # commit
                if useCommit:
                    if not self._commit():
                        raise RuntimeError("Commit error")
                return retValue
            # use an input file as lock since FOR UPDATE doesn't work on JEDI_Events
            lockFileSpec = None
            for fileSpec in job.Files:
                if fileSpec.type in ["input", "pseudo_input"]:
                    if lockFileSpec is None or lockFileSpec.fileID > fileSpec.fileID:
                        lockFileSpec = fileSpec
            if lockFileSpec is not None:
                # sql to lock the file
                sqlLIF = f"SELECT status FROM {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
                sqlLIF += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlLIF += "FOR UPDATE NOWAIT "
                varMap = dict()
                varMap[":jediTaskID"] = lockFileSpec.jediTaskID
                varMap[":datasetID"] = lockFileSpec.datasetID
                varMap[":fileID"] = lockFileSpec.fileID
                _logger.debug(f"{methodName} : locking {str(varMap)}")
                self.cur.execute(sqlLIF + comment, varMap)
                _logger.debug(f"{methodName} : locked")
            # change event status processed by jumbo jobs
            nRowDoneJumbo = 0
            nRowFailedJumbo = 0
            if EventServiceUtils.isCoJumboJob(jobSpec):
                # sql to change event status
                sqlJE = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                sqlJE += f"{panda_config.schemaJEDI}.JEDI_Events tab "
                sqlJE += "SET status=:newStatus "
                sqlJE += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlJE += "AND status=:oldStatus AND is_jumbo=:isJumbo "
                # sql to lock failed events
                sqlJFL = sqlJE + "AND processed_upto_eventID IS NOT NULL "
                # sql to copy failed events
                sqlJFC = f"INSERT INTO {panda_config.schemaJEDI}.JEDI_Events "
                sqlJFC += "(jediTaskID,datasetID,PandaID,fileID,attemptNr,status,"
                sqlJFC += "job_processID,def_min_eventID,def_max_eventID,processed_upto_eventID,"
                sqlJFC += "event_offset,is_jumbo) "
                sqlJFC += "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                sqlJFC += "jediTaskID,datasetID,event_offset,fileID,attemptNr-1,:newStatus,"
                sqlJFC += "job_processID,def_min_eventID,def_max_eventID,processed_upto_eventID,"
                sqlJFC += "event_offset,NULL "
                sqlJFC += f"FROM {panda_config.schemaJEDI}.JEDI_Events tab "
                sqlJFC += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlJFC += "AND status=:oldStatus AND processed_upto_eventID IS NOT NULL AND is_jumbo=:isJumbo "
                # sql to release failed events
                sqlJFR = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                sqlJFR += f"{panda_config.schemaJEDI}.JEDI_Events tab "
                sqlJFR += "SET PandaID=:pandaID,status=:newStatus,processed_upto_eventID=NULL "
                sqlJFR += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlJFR += "AND status=:oldStatus AND is_jumbo=:isJumbo "
                sqlJFR += "AND processed_upto_eventID IS NOT NULL "
                for fileSpec in job.Files:
                    if fileSpec.type != "input":
                        continue
                    # done events in jumbo jobs
                    varMap = {}
                    varMap[":jediTaskID"] = fileSpec.jediTaskID
                    varMap[":datasetID"] = fileSpec.datasetID
                    varMap[":fileID"] = fileSpec.fileID
                    varMap[":oldStatus"] = EventServiceUtils.ST_finished
                    varMap[":newStatus"] = EventServiceUtils.ST_done
                    varMap[":isJumbo"] = EventServiceUtils.eventTableIsJumbo
                    self.cur.execute(sqlJE + comment, varMap)
                    nRowDoneJumbo += self.cur.rowcount
                    # lock failed events
                    varMap = {}
                    varMap[":jediTaskID"] = fileSpec.jediTaskID
                    varMap[":datasetID"] = fileSpec.datasetID
                    varMap[":fileID"] = fileSpec.fileID
                    varMap[":oldStatus"] = EventServiceUtils.ST_failed
                    varMap[":newStatus"] = EventServiceUtils.ST_reserved_fail
                    varMap[":isJumbo"] = EventServiceUtils.eventTableIsJumbo
                    self.cur.execute(sqlJFL + comment, varMap)
                    tmpNumRow = self.cur.rowcount
                    if tmpNumRow > 0:
                        # copy failed events
                        varMap = {}
                        varMap[":jediTaskID"] = fileSpec.jediTaskID
                        varMap[":datasetID"] = fileSpec.datasetID
                        varMap[":fileID"] = fileSpec.fileID
                        varMap[":oldStatus"] = EventServiceUtils.ST_reserved_fail
                        varMap[":newStatus"] = EventServiceUtils.ST_ready
                        varMap[":isJumbo"] = EventServiceUtils.eventTableIsJumbo
                        self.cur.execute(sqlJFC + comment, varMap)
                        # release failed events. Change PandaID to avoid unique constraint of PK
                        varMap = {}
                        varMap[":jediTaskID"] = fileSpec.jediTaskID
                        varMap[":datasetID"] = fileSpec.datasetID
                        varMap[":fileID"] = fileSpec.fileID
                        varMap[":pandaID"] = pandaID
                        varMap[":oldStatus"] = EventServiceUtils.ST_reserved_fail
                        varMap[":newStatus"] = EventServiceUtils.ST_failed
                        varMap[":isJumbo"] = EventServiceUtils.eventTableIsJumbo
                        self.cur.execute(sqlJFR + comment, varMap)
                        nRowFailedJumbo += tmpNumRow
                _logger.debug(f"{methodName} : set done for jumbo to {nRowDoneJumbo} event ranges")
                _logger.debug(f"{methodName} : copied {nRowFailedJumbo} failed event ranges in jumbo")
            # change status to done
            sqlED = f"UPDATE {panda_config.schemaJEDI}.JEDI_Events SET status=:newStatus "
            sqlED += "WHERE jediTaskID=:jediTaskID AND pandaID=:pandaID AND status=:oldStatus "
            varMap = {}
            varMap[":jediTaskID"] = jobSpec.jediTaskID
            varMap[":pandaID"] = pandaID
            varMap[":oldStatus"] = EventServiceUtils.ST_finished
            varMap[":newStatus"] = EventServiceUtils.ST_done
            self.cur.execute(sqlED + comment, varMap)
            nRowDone = self.cur.rowcount
            _logger.info(f"{methodName} : set done to n_er_done={nRowDone} event ranges")
            # release unprocessed event ranges
            sqlEC = f"UPDATE {panda_config.schemaJEDI}.JEDI_Events "
            if jobSpec.decAttOnFailedES():
                sqlEC += "SET status=:newStatus,pandaID=:jobsetID "
            else:
                sqlEC += "SET status=:newStatus,attemptNr=attemptNr-1,pandaID=:jobsetID "
            sqlEC += "WHERE jediTaskID=:jediTaskID AND pandaID=:pandaID AND NOT status IN (:esDone,:esFailed,:esDiscarded,:esCancelled) "
            varMap = {}
            varMap[":jediTaskID"] = jobSpec.jediTaskID
            varMap[":pandaID"] = pandaID
            varMap[":jobsetID"] = jobSpec.jobsetID
            varMap[":esDone"] = EventServiceUtils.ST_done
            varMap[":esFailed"] = EventServiceUtils.ST_failed
            varMap[":esDiscarded"] = EventServiceUtils.ST_discarded
            varMap[":esCancelled"] = EventServiceUtils.ST_cancelled
            varMap[":newStatus"] = EventServiceUtils.ST_ready
            self.cur.execute(sqlEC + comment, varMap)
            nRowReleased = self.cur.rowcount
            _logger.info(f"{methodName} : released n_er_released={nRowReleased} event ranges")
            # copy failed event ranges
            varMap = {}
            varMap[":jediTaskID"] = jobSpec.jediTaskID
            varMap[":pandaID"] = pandaID
            varMap[":jobsetID"] = jobSpec.jobsetID
            varMap[":esFailed"] = EventServiceUtils.ST_failed
            varMap[":newStatus"] = EventServiceUtils.ST_ready
            sqlEF = f"INSERT INTO {panda_config.schemaJEDI}.JEDI_Events "
            sqlEF += "(jediTaskID,datasetID,PandaID,fileID,attemptNr,status,"
            sqlEF += "job_processID,def_min_eventID,def_max_eventID,processed_upto_eventID,event_offset) "
            sqlEF += "SELECT jediTaskID,datasetID,:jobsetID,fileID,attemptNr-1,:newStatus,"
            sqlEF += "job_processID,def_min_eventID,def_max_eventID,processed_upto_eventID,event_offset "
            sqlEF += f"FROM {panda_config.schemaJEDI}.JEDI_Events "
            sqlEF += "WHERE jediTaskID=:jediTaskID AND pandaID=:pandaID AND status=:esFailed "
            sqlEF += "AND processed_upto_eventID IS NOT NULL "
            self.cur.execute(sqlEF + comment, varMap)
            nRowCopied = self.cur.rowcount
            _logger.debug(f"{methodName} : copied {nRowCopied} failed event ranges")
            # unset processed_upto for failed events
            sqlUP = f"UPDATE {panda_config.schemaJEDI}.JEDI_Events SET processed_upto_eventID=NULL "
            sqlUP += "WHERE jediTaskID=:jediTaskID AND pandaID=:pandaID AND status=:esFailed "
            varMap = {}
            varMap[":jediTaskID"] = jobSpec.jediTaskID
            varMap[":pandaID"] = pandaID
            varMap[":esFailed"] = EventServiceUtils.ST_failed
            self.cur.execute(sqlUP + comment, varMap)
            nRowFailed = self.cur.rowcount
            _logger.info(f"{methodName} : failed n_er_failed={nRowFailed} event ranges")
            sqlEU = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlEU += f"COUNT(*) FROM {panda_config.schemaJEDI}.JEDI_Events "
            sqlEU += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID AND attemptNr=:minAttempt "
            sqlEU += "AND NOT status IN (:esDiscarded,:esCancelled) "
            # look for hopeless event ranges
            nRowFatal = 0
            for fileSpec in job.Files:
                if fileSpec.type != "input":
                    continue
                varMap = {}
                varMap[":jediTaskID"] = fileSpec.jediTaskID
                varMap[":datasetID"] = fileSpec.datasetID
                varMap[":fileID"] = fileSpec.fileID
                varMap[":minAttempt"] = 0
                varMap[":esDiscarded"] = EventServiceUtils.ST_discarded
                varMap[":esCancelled"] = EventServiceUtils.ST_cancelled
                self.cur.execute(sqlEU + comment, varMap)
                resEU = self.cur.fetchone()
                if resEU is not None:
                    nRowFatal += resEU[0]
            # there is hopeless event ranges
            _logger.info(f"{methodName} : has n_hopeless={nRowFatal} hopeless event ranges")
            if nRowFatal != 0:
                if jobSpec.acceptPartialFinish():
                    # set fatal to hopeless event ranges
                    sqlFH = f"UPDATE {panda_config.schemaJEDI}.JEDI_Events SET status=:esFatal "
                    sqlFH += "WHERE jediTaskID=:jediTaskID AND pandaID=:jobsetID AND attemptNr=:minAttempt AND status<>:esFatal "
                    varMap = {}
                    varMap[":jediTaskID"] = jobSpec.jediTaskID
                    varMap[":jobsetID"] = jobSpec.jobsetID
                    varMap[":esFatal"] = EventServiceUtils.ST_fatal
                    varMap[":minAttempt"] = 0
                    self.cur.execute(sqlFH + comment, varMap)
            # look for event ranges to process
            sqlERP = f"SELECT job_processID FROM {panda_config.schemaJEDI}.JEDI_Events "
            sqlERP += "WHERE jediTaskID=:jediTaskID AND pandaID=:jobsetID AND status=:esReady "
            sqlERP += "AND attemptNr>:minAttempt "
            varMap = {}
            varMap[":jediTaskID"] = jobSpec.jediTaskID
            varMap[":jobsetID"] = jobSpec.jobsetID
            varMap[":esReady"] = EventServiceUtils.ST_ready
            varMap[":minAttempt"] = 0
            self.cur.execute(sqlERP + comment, varMap)
            resERP = self.cur.fetchall()
            nRow = len(resERP)
            _logger.info(f"{methodName} : left n_er_unprocessed={nRow} unprocessed event ranges")
            otherRunning = False
            hasDoneRange = False
            if nRow == 0:
                # check if other consumers finished
                sqlEOC = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                sqlEOC += f"job_processID,attemptNr,status,processed_upto_eventID,PandaID FROM {panda_config.schemaJEDI}.JEDI_Events tab "
                sqlEOC += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlEOC += "AND ((NOT status IN (:esDone,:esDiscarded,:esCancelled,:esFatal,:esFailed,:esCorrupted) AND attemptNr>:minAttempt) "
                sqlEOC += "OR (status=:esFailed AND processed_upto_eventID IS NOT NULL)) "
                # count the number of done ranges
                sqlCDO = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                sqlCDO += f"COUNT(*) FROM {panda_config.schemaJEDI}.JEDI_Events tab "
                sqlCDO += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlCDO += "AND status=:esDone AND rownum=1 "
                for fileSpec in job.Files:
                    if fileSpec.type == "input":
                        varMap = {}
                        varMap[":jediTaskID"] = fileSpec.jediTaskID
                        varMap[":datasetID"] = fileSpec.datasetID
                        varMap[":fileID"] = fileSpec.fileID
                        varMap[":esDone"] = EventServiceUtils.ST_done
                        varMap[":esDiscarded"] = EventServiceUtils.ST_discarded
                        varMap[":esCancelled"] = EventServiceUtils.ST_cancelled
                        varMap[":esCorrupted"] = EventServiceUtils.ST_corrupted
                        varMap[":esFatal"] = EventServiceUtils.ST_fatal
                        varMap[":esFailed"] = EventServiceUtils.ST_failed
                        varMap[":minAttempt"] = 0
                        self.cur.execute(sqlEOC + comment, varMap)
                        resEOC = self.cur.fetchone()
                        if resEOC is not None:
                            # there are unprocessed ranges
                            otherRunning = True
                            eocDump = dict()
                            eocDump["jediTaskID"] = fileSpec.jediTaskID
                            eocDump["datasetID"] = fileSpec.datasetID
                            eocDump["fileID"] = fileSpec.fileID
                            eocDump["job_processID"] = resEOC[0]
                            eocDump["attemptNr"] = resEOC[1]
                            eocDump["status"] = resEOC[2]
                            eocDump["processed_upto_eventID"] = resEOC[3]
                            eocDump["PandaID"] = resEOC[4]
                            _logger.debug(f"{methodName} : some event ranges still running like {str(eocDump)}")
                            break
                        # check if there are done ranges
                        if not hasDoneRange:
                            varMap = {}
                            varMap[":jediTaskID"] = fileSpec.jediTaskID
                            varMap[":datasetID"] = fileSpec.datasetID
                            varMap[":fileID"] = fileSpec.fileID
                            varMap[":esDone"] = EventServiceUtils.ST_done
                            self.cur.execute(sqlCDO + comment, varMap)
                            resCDO = self.cur.fetchone()
                            (nCDORow,) = resCDO
                            if nCDORow != 0:
                                hasDoneRange = True
                # do merging since all ranges were done
                if not otherRunning:
                    doMerging = True
            else:
                doMerging = False
            # do nothing since other consumers are still running
            if otherRunning:
                _logger.debug(f"{methodName} : do nothing as other consumers are still running")
                # commit
                if useCommit:
                    if not self._commit():
                        raise RuntimeError("Commit error")
                if nRowDone == 0:
                    # didn't process any events
                    retValue = 5, None
                else:
                    # processed some events
                    retValue = 4, None
                return retValue
            # all failed
            if doMerging and not hasDoneRange:
                # fail immediately
                _logger.debug(f"{methodName} : all event ranges failed")
                # commit
                if useCommit:
                    if not self._commit():
                        raise RuntimeError("Commit error")
                retValue = 7, None
                return retValue
            # fail immediately if not all events were done in the largest attemptNr
            if (jobSpec.attemptNr >= jobSpec.maxAttempt and not (doMerging and hasDoneRange)) or (doMerging and nRowFatal > 0):
                _logger.debug(f"{methodName} : no more retry since not all events were done in the largest attemptNr")
                # check if there is active consumer
                sqlAC = "SELECT COUNT(*) FROM ("
                sqlAC += "SELECT PandaID FROM ATLAS_PANDA.jobsDefined4 "
                sqlAC += "WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID "
                sqlAC += "UNION "
                sqlAC += "SELECT PandaID FROM ATLAS_PANDA.jobsActive4 "
                sqlAC += "WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID "
                sqlAC += ") "
                varMap = {}
                varMap[":jediTaskID"] = jobSpec.jediTaskID
                varMap[":jobsetID"] = jobSpec.jobsetID
                self.cur.execute(sqlAC + comment, varMap)
                resAC = self.cur.fetchone()
                (numActiveEC,) = resAC
                _logger.debug(f"{methodName} : num of active consumers = {numActiveEC}")
                # commit
                if useCommit:
                    if not self._commit():
                        raise RuntimeError("Commit error")
                if numActiveEC <= 1:
                    # last one
                    retValue = 6, None
                else:
                    # there are active consumers
                    retValue = 5, None
                return retValue
            # no merging for inaction ES jobs
            if doMerging and nRowDoneJumbo == 0 and nRowDone == 0 and not job.allOkEvents():
                _logger.debug(f"{methodName} : skip merge generation since nDone=0")
                retValue = 5, None
                return retValue
            # change waiting file status
            if doMerging and EventServiceUtils.isCoJumboJob(jobSpec):
                # update file
                sqlUWF = f"UPDATE {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
                sqlUWF += "SET status=:newStatus,is_waiting=NULL "
                sqlUWF += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlUWF += "AND attemptNr=:attemptNr AND status=:oldStatus AND keepTrack=:keepTrack "
                # update dataset
                sqlUWD = f"UPDATE {panda_config.schemaJEDI}.JEDI_Datasets "
                sqlUWD += "SET nFilesUsed=nFilesUsed+:nDiff,nFilesWaiting=nFilesWaiting-:nDiff "
                sqlUWD += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                nFilesUsedMap = {}
                for fileSpec in job.Files:
                    if fileSpec.type not in ["input", "pseudo_input"]:
                        continue
                    varMap = {}
                    varMap[":jediTaskID"] = fileSpec.jediTaskID
                    varMap[":datasetID"] = fileSpec.datasetID
                    varMap[":fileID"] = fileSpec.fileID
                    varMap[":attemptNr"] = fileSpec.attemptNr
                    varMap[":newStatus"] = "running"
                    varMap[":oldStatus"] = "ready"
                    varMap[":keepTrack"] = 1
                    self.cur.execute(sqlUWF + comment, varMap)
                    nDiff = self.cur.rowcount
                    if nDiff > 0:
                        nFilesUsedMap.setdefault(fileSpec.datasetID, 0)
                        nFilesUsedMap[fileSpec.datasetID] += nDiff
                for datasetID in nFilesUsedMap:
                    nDiff = nFilesUsedMap[datasetID]
                    varMap = {}
                    varMap[":jediTaskID"] = jobSpec.jediTaskID
                    varMap[":datasetID"] = datasetID
                    varMap[":nDiff"] = nDiff
                    self.cur.execute(sqlUWD + comment, varMap)
            # check if there is fatal range
            hasFatalRange = False
            if doMerging:
                sqlCFE = f"SELECT COUNT(*) FROM {panda_config.schemaJEDI}.JEDI_Events "
                sqlCFE += "WHERE jediTaskID=:jediTaskID AND pandaID=:jobsetID AND "
                sqlCFE += "status=:esFatal AND rownum=1 "
                varMap = {}
                varMap[":jediTaskID"] = jobSpec.jediTaskID
                varMap[":jobsetID"] = jobSpec.jobsetID
                varMap[":esFatal"] = EventServiceUtils.ST_fatal
                self.cur.execute(sqlCFE + comment, varMap)
                resCFE = self.cur.fetchone()
                (nRowCEF,) = resCFE
                _logger.debug(f"{methodName} : {nRowCEF} fatal even ranges ")
                if nRowCEF > 0:
                    hasFatalRange = True
            # reset job attributes
            jobSpec.startTime = None
            jobSpec.creationTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            jobSpec.modificationTime = jobSpec.creationTime
            jobSpec.stateChangeTime = jobSpec.creationTime
            jobSpec.prodDBUpdateTime = jobSpec.creationTime
            jobSpec.attemptNr += 1
            jobSpec.batchID = None
            jobSpec.schedulerID = None
            jobSpec.pilotID = None
            if doMerging:
                jobSpec.maxAttempt = jobSpec.attemptNr
                jobSpec.currentPriority = 5000
            else:
                jobSpec.currentPriority += 1
            jobSpec.endTime = None
            jobSpec.transExitCode = None
            jobSpec.jobMetrics = None
            jobSpec.jobSubStatus = None
            jobSpec.actualCoreCount = None
            jobSpec.hs06sec = None
            jobSpec.nEvents = None
            jobSpec.cpuConsumptionTime = None
            # disable background flag
            jobSpec.jobExecutionID = 0
            if hasFatalRange:
                jobSpec.jobSubStatus = "partial"
            for attr in jobSpec._attributes:
                for patt in [
                    "ErrorCode",
                    "ErrorDiag",
                    "CHAR",
                    "BYTES",
                    "RSS",
                    "PSS",
                    "VMEM",
                    "SWAP",
                ]:
                    if attr.endswith(patt):
                        setattr(jobSpec, attr, None)
                        break
            # read files
            varMap = {}
            varMap[":PandaID"] = pandaID
            sqlFile = f"SELECT {FileSpec.columnNames()} FROM ATLAS_PANDA.filesTable4 "
            sqlFile += "WHERE PandaID=:PandaID"
            self.cur.arraysize = 100000
            self.cur.execute(sqlFile + comment, varMap)
            resFs = self.cur.fetchall()
            # loop over all files
            for resF in resFs:
                # add
                fileSpec = FileSpec()
                fileSpec.pack(resF)
                jobSpec.addFile(fileSpec)
                # reset file status
                if fileSpec.type in ["output", "log"]:
                    fileSpec.status = "unknown"
            # set current status if unspecified
            if currentJobStatus is None:
                currentJobStatus = "activated"
                for fileSpec in jobSpec.Files:
                    if fileSpec.type == "input" and fileSpec.status != "ready":
                        currentJobStatus = "assigned"
                        break
            if doMerging and currentJobStatus == "assigned":
                # send merge jobs to activated since input data don't have to move
                _logger.debug(f"{methodName} : sending to activated")
                jobSpec.jobStatus = "activated"
            elif currentJobStatus in ["defined", "assigned", "waiting", "pending"]:
                jobSpec.jobStatus = currentJobStatus
            else:
                jobSpec.jobStatus = "activated"
            # read job parameters
            sqlJobP = "SELECT jobParameters FROM ATLAS_PANDA.jobParamsTable WHERE PandaID=:PandaID"
            varMap = {}
            varMap[":PandaID"] = jobSpec.PandaID
            self.cur.execute(sqlJobP + comment, varMap)
            for (clobJobP,) in self.cur:
                try:
                    jobSpec.jobParameters = clobJobP.read()
                except AttributeError:
                    jobSpec.jobParameters = str(clobJobP)
                break
            # changes some attributes
            noNewJob = False
            closedInBadStatus = False
            if not doMerging:
                minUnprocessed = self.getConfigValue("dbproxy", "AES_MINEVENTSFORMCORE")

                sqlCore = (
                    "SELECT /* use_json_type */ scj.data.corecount, scj.data.status, scj.data.jobseed "
                    "FROM ATLAS_PANDA.schedconfig_json scj "
                    "WHERE scj.panda_queue=:siteid "
                )

                varMap = {}
                varMap[":siteid"] = jobSpec.computingSite
                self.cur.execute(sqlCore + comment, varMap)
                resCore = self.cur.fetchone()
                if resCore is not None:
                    coreCount, tmpState, tmpJobSeed = resCore
                    if coreCount is not None:
                        coreCount = int(coreCount)
                        if minUnprocessed is None:
                            minUnprocessed = coreCount
                        else:
                            minUnprocessed = max(minUnprocessed, coreCount)

                    if tmpState not in ["online", "brokeroff"] or tmpJobSeed == "std":
                        noNewJob = True
                if jobSpec.coreCount > 1 and minUnprocessed is not None and minUnprocessed > nRow:
                    self.setScoreSiteToEs(jobSpec, methodName, comment)
                # not to repeat useless consumers
                if currentJobStatus in ["defined", "pending"]:
                    noNewJob = True
                    closedInBadStatus = True
            else:
                # extract parameters for merge
                try:
                    tmpMatch = re.search(
                        "<PANDA_ESMERGE_TRF>(.*)</PANDA_ESMERGE_TRF>",
                        jobSpec.jobParameters,
                    )
                    jobSpec.transformation = tmpMatch.group(1)
                except Exception:
                    pass
                try:
                    tmpMatch = re.search("<PANDA_EVSMERGE>(.*)</PANDA_EVSMERGE>", jobSpec.jobParameters)
                    jobSpec.jobParameters = tmpMatch.group(1)
                except Exception:
                    pass
                # use siteid of jumbo jobs to generate merge jobs for fake co-jumbo
                isFakeCJ = False
                if jobSpec.computingSite == EventServiceUtils.siteIdForWaitingCoJumboJobs:
                    isFakeCJ = True
                    # sql to get PandaIDs of jumbo jobs
                    sqlJJ = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                    sqlJJ += f"DISTINCT PandaID FROM {panda_config.schemaJEDI}.JEDI_Events tab "
                    sqlJJ += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                    sqlJJ += "AND status=:esDone AND is_jumbo=:isJumbo "
                    # sql to get siteid of jumbo job
                    sqlJJS = "SELECT computingSite FROM ATLAS_PANDA.jobsActive4 "
                    sqlJJS += "WHERE PandaID=:PandaID "
                    sqlJJS += "UNION "
                    sqlJJS += "SELECT computingSite FROM ATLAS_PANDA.jobsArchived4 "
                    sqlJJS += "WHERE PandaID=:PandaID "
                    sqlJJS += "UNION "
                    sqlJJS += "SELECT computingSite FROM ATLAS_PANDAARCH.jobsArchived "
                    sqlJJS += "WHERE PandaID=:PandaID AND modificationTime>CURRENT_DATE-30 "
                    # look for jumbo jobs
                    toEscape = False
                    for fileSpec in job.Files:
                        if fileSpec.type != "input":
                            continue
                        # get PandaIDs of jumbo jobs
                        varMap = {}
                        varMap[":jediTaskID"] = fileSpec.jediTaskID
                        varMap[":datasetID"] = fileSpec.datasetID
                        varMap[":fileID"] = fileSpec.fileID
                        varMap[":esDone"] = EventServiceUtils.ST_done
                        varMap[":isJumbo"] = EventServiceUtils.eventTableIsJumbo
                        self.cur.execute(sqlJJ + comment, varMap)
                        resJJList = self.cur.fetchall()
                        for (jPandaID,) in resJJList:
                            # get siteid of jumbo job
                            varMap = {}
                            varMap[":PandaID"] = jPandaID
                            self.cur.execute(sqlJJS + comment, varMap)
                            resJJS = self.cur.fetchone()
                            if resJJS is not None:
                                tmpStr = f"{methodName} changed co-jumbo site {jobSpec.computingSite} "
                                jobSpec.computingSite = resJJS[0]
                                tmpStr += f"to {jobSpec.computingSite}"
                                toEscape = True
                                _logger.debug(tmpStr)
                                break
                        if toEscape:
                            break
                # change special handling and set the share to express for merge jobs
                EventServiceUtils.setEventServiceMerge(jobSpec)
                # set site
                self.setSiteForEsMerge(jobSpec, isFakeCJ, methodName, comment)
                jobSpec.coreCount = None
                jobSpec.minRamCount = 2000

            # reset resource type
            jobSpec.resource_type = self.get_resource_type_job(jobSpec)

            # no new job since ES is disabled
            if noNewJob:
                jobSpec.PandaID = None
                msgStr = f"{methodName} No new job since event service is disabled or queue is offline or old job status {currentJobStatus} is not active"
                _logger.debug(msgStr)
            else:
                # update input
                if doMerging:
                    self.updateInputStatusJedi(jobSpec.jediTaskID, jobSpec.PandaID, "merging")
                else:
                    self.updateInputStatusJedi(jobSpec.jediTaskID, jobSpec.PandaID, "queued", checkOthers=True)
                # insert job with new PandaID
                if jobSpec.jobStatus in ["defined", "assigned"]:
                    sql1 = f"INSERT INTO ATLAS_PANDA.jobsDefined4 ({JobSpec.columnNames()}) "
                elif jobSpec.jobStatus in ["waiting", "pending"]:
                    sql1 = f"INSERT INTO ATLAS_PANDA.jobsWaiting4 ({JobSpec.columnNames()}) "
                else:
                    sql1 = f"INSERT INTO ATLAS_PANDA.jobsActive4 ({JobSpec.columnNames()}) "
                sql1 += JobSpec.bindValuesExpression(useSeq=True)
                sql1 += " RETURNING PandaID INTO :newPandaID"
                # set parentID
                jobSpec.parentID = jobSpec.PandaID
                varMap = jobSpec.valuesMap(useSeq=True)
                varMap[":newPandaID"] = self.cur.var(varNUMBER)
                # insert
                if not noNewJob:
                    retI = self.cur.execute(sql1 + comment, varMap)
                    # set PandaID
                    val = self.getvalue_corrector(self.cur.getvalue(varMap[":newPandaID"]))
                    jobSpec.PandaID = int(val)
                else:
                    jobSpec.PandaID = None
                msgStr = f"{methodName} Generate new PandaID -> {jobSpec.PandaID}#{jobSpec.attemptNr} at {jobSpec.computingSite} "
                if doMerging:
                    msgStr += "for merge"
                else:
                    msgStr += "for retry"
                _logger.debug(msgStr)
                # insert files
                sqlFile = f"INSERT INTO ATLAS_PANDA.filesTable4 ({FileSpec.columnNames()}) "
                sqlFile += FileSpec.bindValuesExpression(useSeq=True)
                sqlFile += " RETURNING row_ID INTO :newRowID"
                sqlMaxFail = f"UPDATE {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
                sqlMaxFail += "SET maxFailure=(CASE "
                sqlMaxFail += "WHEN maxFailure IS NULL THEN failedAttempt+:increase "
                sqlMaxFail += "WHEN maxFailure>failedAttempt+:increase THEN failedAttempt+:increase "
                sqlMaxFail += "ELSE maxFailure "
                sqlMaxFail += "END) "
                sqlMaxFail += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlMaxFail += "AND keepTrack=:keepTrack "
                for fileSpec in jobSpec.Files:
                    # skip zip
                    if fileSpec.type.startswith("zip"):
                        continue
                    # reset rowID
                    fileSpec.row_ID = None
                    # change GUID and LFN for log
                    if fileSpec.type == "log":
                        fileSpec.GUID = str(uuid.uuid4())
                        if doMerging:
                            fileSpec.lfn = re.sub(
                                f"\\.{pandaID}$",
                                "".format(jobSpec.PandaID),
                                fileSpec.lfn,
                            )
                        else:
                            fileSpec.lfn = re.sub(
                                f"\\.{pandaID}$",
                                f".{jobSpec.PandaID}",
                                fileSpec.lfn,
                            )
                    # insert
                    varMap = fileSpec.valuesMap(useSeq=True)
                    varMap[":newRowID"] = self.cur.var(varNUMBER)
                    self.cur.execute(sqlFile + comment, varMap)
                    val = self.getvalue_corrector(self.cur.getvalue(varMap[":newRowID"]))
                    fileSpec.row_ID = int(val)
                    # change max failure for esmerge
                    if doMerging and fileSpec.type in ["input", "pseudo_input"]:
                        varMap = {}
                        varMap[":jediTaskID"] = fileSpec.jediTaskID
                        varMap[":datasetID"] = fileSpec.datasetID
                        varMap[":fileID"] = fileSpec.fileID
                        varMap[":increase"] = 5
                        varMap[":keepTrack"] = 1
                        self.cur.execute(sqlMaxFail + comment, varMap)
                # insert job parameters
                sqlJob = "INSERT INTO ATLAS_PANDA.jobParamsTable (PandaID,jobParameters) VALUES (:PandaID,:param)"
                varMap = {}
                varMap[":PandaID"] = jobSpec.PandaID
                varMap[":param"] = jobSpec.jobParameters
                self.cur.execute(sqlJob + comment, varMap)
                # propagate change to JEDI
                if doMerging:
                    relationType = "es_merge"
                else:
                    relationType = None
                self.updateForPilotRetryJEDI(jobSpec, self.cur, onlyHistory=True, relationType=relationType)
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError("Commit error")
            # set return
            if not doMerging:
                if closedInBadStatus:
                    # closed in bad status
                    retValue = 9, jobSpec.PandaID
                elif nRowDone == 0:
                    # didn't process any events
                    retValue = 8, jobSpec.PandaID
                else:
                    # processed some events
                    retValue = 0, jobSpec.PandaID
            else:
                if nRowDone == 0:
                    retValue = 10, jobSpec.PandaID
                else:
                    retValue = 2, jobSpec.PandaID
            # record status change
            try:
                if not noNewJob:
                    self.recordStatusChange(
                        jobSpec.PandaID,
                        jobSpec.jobStatus,
                        jobInfo=jobSpec,
                        useCommit=useCommit,
                    )
                    self.push_job_status_message(jobSpec, jobSpec.PandaID, jobSpec.jobStatus)
            except Exception:
                _logger.error("recordStatusChange in ppEventServiceJob")
            _logger.debug(f"{methodName} done for doMergeing={doMerging}")
            if retValue[-1] == "NULL":
                retValue = retValue[0], None
            return retValue
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None, None

    # set site for ES merge
    def setSiteForEsMerge(self, jobSpec, isFakeCJ, methodName, comment):
        _logger.debug(f"{methodName} looking for ES merge site")
        # merge on OS
        isMergeAtOS = EventServiceUtils.isMergeAtOS(jobSpec.specialHandling)
        # check where merge is done
        lookForMergeSite = True
        sqlWM = "SELECT /* use_json_type */ scj.data.catchall, scj.data.objectstores " "FROM ATLAS_PANDA.schedconfig_json scj " "WHERE scj.panda_queue=:siteid "

        varMap = {}
        varMap[":siteid"] = jobSpec.computingSite
        self.cur.execute(sqlWM + comment, varMap)
        resWM = self.cur.fetchone()
        resSN = []
        resSN_back = []
        catchAll, objectstores = None, None
        if resWM is not None:
            catchAll, objectstores = resWM
        if catchAll is None:
            catchAll = ""
        try:
            if isFakeCJ:
                objectstores = []
            else:
                objectstores = json.loads(objectstores)
        except Exception:
            objectstores = []
        # get objstoreIDs
        sqlZIP = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
        sqlZIP += f"DISTINCT zipRow_ID FROM {panda_config.schemaJEDI}.JEDI_Events "
        sqlZIP += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
        sqlZIP += "AND status=:esDone "
        sqlOST = f"SELECT fsize,destinationSE FROM {panda_config.schemaPANDA}.filesTable4 "
        sqlOST += "WHERE row_ID=:row_ID "
        sqlOST += "UNION "
        sqlOST += f"SELECT fsize,destinationSE FROM {panda_config.schemaPANDAARCH}.filesTable_ARCH "
        sqlOST += "WHERE row_ID=:row_ID "
        objStoreZipMap = dict()
        storageZipMap = dict()
        zipRowIDs = set()
        totalZipSize = 0
        for tmpFileSpec in jobSpec.Files:
            if tmpFileSpec.type in ["input", "pseudo_input"]:
                varMap = dict()
                varMap[":jediTaskID"] = tmpFileSpec.jediTaskID
                varMap[":datasetID"] = tmpFileSpec.datasetID
                varMap[":fileID"] = tmpFileSpec.fileID
                varMap[":esDone"] = EventServiceUtils.ST_done
                self.cur.execute(sqlZIP + comment, varMap)
                resZIP = self.cur.fetchall()
                for (zipRowID,) in resZIP:
                    if zipRowID is None:
                        continue
                    if zipRowID in zipRowIDs:
                        continue
                    zipRowIDs.add(zipRowID)
                    # get file info
                    varMap = dict()
                    varMap[":row_ID"] = zipRowID
                    self.cur.execute(sqlOST + comment, varMap)
                    resOST = self.cur.fetchone()
                    tmpFsize, tmpDestSE = resOST
                    totalZipSize += tmpFsize
                    tmpRSE = self.convertObjIDtoEndPoint(panda_config.endpoint_mapfile, int(tmpDestSE.split("/")[0]))
                    if tmpRSE is not None:
                        objStoreZipMap.setdefault(tmpRSE["name"], 0)
                        objStoreZipMap[tmpRSE["name"]] += tmpFsize
                        if tmpRSE["type"].endswith("DISK"):
                            storageZipMap.setdefault(tmpRSE["name"], 0)
                            storageZipMap[tmpRSE["name"]] += tmpFsize
        if len(storageZipMap) > 0:
            sortedOST = sorted(storageZipMap.items(), key=operator.itemgetter(1))
        else:
            sortedOST = sorted(objStoreZipMap.items(), key=operator.itemgetter(1))
        sortedOST.reverse()
        if len(sortedOST) > 0:
            _logger.debug(f"{methodName} old objectstores {str(objectstores)}")
            objectstores = [{"ddmendpoint": sortedOST[0][0]}]
            _logger.debug(f"{methodName} new objectstores {str(objectstores)}")
        if isFakeCJ:
            # use nucleus for fake co-jumbo since they don't have sub datasets
            pass
        elif "localEsMergeNC" in catchAll:
            # no site change
            lookForMergeSite = False
        else:
            # get sites in the nucleus associated to the site to run merge jobs in the same nucleus
            sqlSN = "SELECT /* use_json_type */ dr.panda_site_name, dr.ddm_endpoint_name "
            sqlSN += "FROM ATLAS_PANDA.panda_site ps1, ATLAS_PANDA.panda_site ps2, ATLAS_PANDA.schedconfig_json sc, ATLAS_PANDA.panda_ddm_relation dr "
            sqlSN += "WHERE ps1.panda_site_name=:site AND ps1.site_name=ps2.site_name AND sc.panda_queue=ps2.panda_site_name "
            sqlSN += "AND dr.panda_site_name=ps2.panda_site_name "
            sqlSN += "AND (sc.data.corecount IS NULL OR sc.data.corecount=1 OR sc.data.capability=:capability) "
            sqlSN += "AND (sc.data.maxtime=0 OR sc.data.maxtime>=86400) "
            sqlSN += "AND (sc.data.maxrss IS NULL OR sc.data.minrss=0) "
            sqlSN += "AND (sc.data.jobseed IS NULL OR sc.data.jobseed<>'es') "
            sqlSN += "AND sc.data.type != 'analysis' "

            if "localEsMerge" in catchAll and "useBrokerOff" in catchAll:
                sqlSN += "AND sc.data.status IN (:siteStatus1,:siteStatus2) "
            else:
                sqlSN += "AND sc.data.status=:siteStatus "

            sqlSN += "AND dr.default_write ='Y' "
            sqlSN += "AND (scope = 'default' OR scope IS NULL) "  # skip endpoints with analysis roles
            sqlSN += "AND (sc.data.wnconnectivity IS NULL OR sc.data.wnconnectivity LIKE :wc1) "

            varMap = {}
            varMap[":site"] = jobSpec.computingSite
            if "localEsMerge" in catchAll and "useBrokerOff" in catchAll:
                varMap[":siteStatus1"] = "online"
                varMap[":siteStatus2"] = "brokeroff"
            else:
                varMap[":siteStatus"] = "online"
            varMap[":wc1"] = "full%"
            varMap[":capability"] = "ucore"
            # get sites
            self.cur.execute(sqlSN + comment, varMap)
            if "localEsMerge" in catchAll:
                resSN = self.cur.fetchall()
            else:
                resSN_back = self.cur.fetchall()
        if len(resSN) == 0 and lookForMergeSite:
            # run merge jobs at destination
            if not jobSpec.destinationSE.startswith("nucleus:"):
                jobSpec.computingSite = jobSpec.destinationSE
                lookForMergeSite = False
            else:
                # use nucleus close to OS
                tmpNucleus = None
                if isMergeAtOS and len(objectstores) > 0:
                    osEndpoint = objectstores[0]["ddmendpoint"]
                    sqlCO = "SELECT site_name FROM ATLAS_PANDA.ddm_endpoint WHERE ddm_endpoint_name=:osEndpoint "
                    varMap = dict()
                    varMap[":osEndpoint"] = osEndpoint
                    self.cur.execute(sqlCO + comment, varMap)
                    resCO = self.cur.fetchone()
                    if resCO is not None:
                        (tmpNucleus,) = resCO
                        _logger.info(f"{methodName} look for merge sites in nucleus:{tmpNucleus} close to pre-merged files")
                # use nucleus
                if tmpNucleus is None:
                    tmpNucleus = jobSpec.destinationSE.split(":")[-1]
                    _logger.info(f"{methodName} look for merge sites in destination nucleus:{tmpNucleus}")
                # get sites in a nucleus
                sqlSN = "SELECT /* use_json_type */ dr.panda_site_name, dr.ddm_endpoint_name "
                sqlSN += "FROM ATLAS_PANDA.panda_site ps, ATLAS_PANDA.schedconfig_json sc, ATLAS_PANDA.panda_ddm_relation dr "
                sqlSN += "WHERE site_name=:nucleus AND sc.panda_queue=ps.panda_site_name "
                sqlSN += "AND dr.panda_site_name=ps.panda_site_name "
                sqlSN += "AND (sc.data.corecount IS NULL OR sc.data.corecount=1 OR sc.data.capability=:capability) "
                sqlSN += "AND (sc.maxtime=0 OR sc.maxtime>=86400) "
                sqlSN += "AND (sc.maxrss IS NULL OR sc.minrss=0) "
                sqlSN += "AND (sc.jobseed IS NULL OR sc.jobseed<>'es') "
                sqlSN += "AND sc.data.type != 'analysis' "
                sqlSN += "AND sc.data.status=:siteStatus "
                sqlSN += "AND dr.default_write='Y' "
                sqlSN += "AND (dr.scope = 'default' OR dr.scope IS NULL) "  # skip endpoints with analysis roles
                sqlSN += "AND (sc.data.wnconnectivity IS NULL OR sc.data.wnconnectivity LIKE :wc1) "

                varMap = {}
                varMap[":nucleus"] = tmpNucleus
                varMap[":siteStatus"] = "online"
                varMap[":wc1"] = "full%"
                varMap[":capability"] = "ucore"
                # get sites
                self.cur.execute(sqlSN + comment, varMap)
                resSN = self.cur.fetchall()

        # last resort for jumbo
        resSN_all = []
        if lookForMergeSite and (isFakeCJ or "useJumboJobs" in catchAll or len(resSN + resSN_back) == 0):
            sqlSN = "SELECT /* use_json_type */ dr.panda_site_name, dr.ddm_endpoint_name "
            sqlSN += "FROM ATLAS_PANDA.panda_site ps, ATLAS_PANDA.schedconfig_json sc, ATLAS_PANDA.panda_ddm_relation dr "
            sqlSN += "WHERE sc.panda_queue=ps.panda_site_name "
            sqlSN += "AND dr.panda_site_name=ps.panda_site_name "
            sqlSN += "AND (sc.data.corecount IS NULL OR sc.data.corecount=1 OR sc.data.capability=:capability) "
            sqlSN += "AND (sc.data.maxtime=0 OR sc.data.maxtime>=86400) "
            sqlSN += "AND (sc.data.maxrss IS NULL OR sc.data.minrss=0) "
            sqlSN += "AND (sc.data.jobseed IS NULL OR sc.data.jobseed<>'es') "
            sqlSN += "AND sc.data.type != 'analysis' "
            sqlSN += "AND sc.data.status=:siteStatus "
            sqlSN += "AND dr.default_write='Y' "
            sqlSN += "AND (dr.scope = 'default' OR dr.scope IS NULL) "  # skip endpoints with analysis roles
            sqlSN += "AND (sc.data.wnconnectivity IS NULL OR sc.data.wnconnectivity LIKE :wc1) "

            varMap = {}
            varMap[":siteStatus"] = "online"
            varMap[":wc1"] = "full%"
            varMap[":capability"] = "ucore"

            # get sites
            self.cur.execute(sqlSN + comment, varMap)
            resSN_all = self.cur.fetchall()

        # look for a site for merging
        if lookForMergeSite:
            # compare number of pilot requests
            maxNumPilot = 0
            sqlUG = "SELECT updateJob+getJob FROM ATLAS_PANDAMETA.sitedata "
            sqlUG += "WHERE site=:panda_site AND HOURS=:hours AND FLAG=:flag "

            sqlRJ = "SELECT SUM(num_of_jobs) FROM ATLAS_PANDA.MV_JOBSACTIVE4_STATS "
            sqlRJ += "WHERE computingSite=:panda_site AND jobStatus=:jobStatus "

            newSiteName = None
            for resItem in [resSN, resSN_back, resSN_all]:
                for tmp_panda_site_name, tmp_ddm_endpoint in resItem:
                    # get nPilot
                    varMap = {}
                    varMap[":panda_site"] = tmp_panda_site_name
                    varMap[":hours"] = 3
                    varMap[":flag"] = "production"
                    self.cur.execute(sqlUG + comment, varMap)
                    resUG = self.cur.fetchone()
                    if resUG is None:
                        nPilots = 0
                    else:
                        (nPilots,) = resUG
                    # get nRunning
                    varMap = {}
                    varMap[":panda_site"] = tmp_panda_site_name
                    varMap[":jobStatus"] = "running"
                    self.cur.execute(sqlRJ + comment, varMap)
                    resRJ = self.cur.fetchone()
                    if resRJ is None:
                        nRunning = 0
                    else:
                        (nRunning,) = resRJ
                    tmpStr = f"site={tmp_panda_site_name} nPilot={nPilots} nRunning={nRunning}"
                    _logger.info(f"{methodName} {tmpStr}")
                    # use larger
                    if maxNumPilot < nPilots:
                        maxNumPilot = nPilots
                        jobSpec.computingSite = tmp_panda_site_name
                        newSiteName = jobSpec.computingSite
                        for tmpFileSpec in jobSpec.Files:
                            if tmpFileSpec.destinationDBlockToken.startswith("ddd:"):
                                tmpFileSpec.destinationDBlockToken = f"ddd:{tmp_ddm_endpoint}"
                                tmpFileSpec.destinationSE = jobSpec.computingSite
                if newSiteName is not None:
                    _logger.info(f"{methodName} set merge site to {newSiteName}")
                    break
        # return
        return

    # set score site to ES job
    def setScoreSiteToEs(self, jobSpec, methodName, comment):
        _logger.debug(f"{methodName} looking for single-core site")

        # get score PQ in the nucleus associated to the site to run the small ES job
        sqlSN = "SELECT /* use_json_type */ ps2.panda_site_name "
        sqlSN += "FROM ATLAS_PANDA.panda_site ps1, ATLAS_PANDA.panda_site ps2, ATLAS_PANDA.schedconfig_json sc "
        sqlSN += "WHERE ps1.panda_site_name=:site AND ps1.site_name=ps2.site_name AND sc.panda_queue=ps2.panda_site_name "
        sqlSN += "AND (sc.data.corecount IS NULL OR sc.data.corecount=1 OR sc.data.capability=:capability) "
        sqlSN += "AND (sc.data.jobseed IS NULL OR sc.data.jobseed<>'std') "
        sqlSN += "AND sc.data.status=:siteStatus "

        varMap = {}
        varMap[":site"] = jobSpec.computingSite
        varMap[":siteStatus"] = "online"
        varMap[":capability"] = "ucore"

        # get sites
        self.cur.execute(sqlSN + comment, varMap)
        resSN = self.cur.fetchall()
        # compare number of pilot requests
        maxNumPilot = 0
        sqlUG = "SELECT updateJob+getJob FROM ATLAS_PANDAMETA.sitedata "
        sqlUG += "WHERE site=:panda_site AND HOURS=:hours AND FLAG=:flag "
        sqlRJ = "SELECT SUM(num_of_jobs) FROM ATLAS_PANDA.MV_JOBSACTIVE4_STATS "
        sqlRJ += "WHERE computingSite=:panda_site AND jobStatus=:jobStatus "
        newSiteName = None
        for (tmp_panda_site_name,) in resSN:
            # get nPilot
            varMap = {}
            varMap[":panda_site"] = tmp_panda_site_name
            varMap[":hours"] = 3
            varMap[":flag"] = "production"
            self.cur.execute(sqlUG + comment, varMap)
            resUG = self.cur.fetchone()
            if resUG is None:
                nPilots = 0
            else:
                (nPilots,) = resUG
            # get nRunning
            varMap = {}
            varMap[":panda_site"] = tmp_panda_site_name
            varMap[":jobStatus"] = "running"
            self.cur.execute(sqlRJ + comment, varMap)
            resRJ = self.cur.fetchone()
            if resRJ is None:
                nRunning = 0
            else:
                (nRunning,) = resRJ
            tmpStr = f"site={tmp_panda_site_name} nPilot={nPilots} nRunning={nRunning}"
            _logger.info(f"{methodName} {tmpStr}")
            # use larger
            if maxNumPilot < nPilots:
                maxNumPilot = nPilots
                jobSpec.computingSite = tmp_panda_site_name
                jobSpec.coreCount = 1
                jobSpec.minRamCount = 0
                jobSpec.resource_type = self.get_resource_type_job(jobSpec)
                newSiteName = jobSpec.computingSite
        if newSiteName is not None:
            _logger.info(f"{methodName} set single-core site to {newSiteName}")
        else:
            _logger.info(f"{methodName} no single-core site for {jobSpec.computingSite}")
        # return
        return

    # change task priority
    def changeTaskPriorityPanda(self, jediTaskID, newPriority):
        comment = " /* DBProxy.changeTaskPriorityPanda */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID}>"
        _logger.debug(f"{methodName} newPrio={newPriority}")
        try:
            # sql to update JEDI task table
            sqlT = f"UPDATE {panda_config.schemaJEDI}.JEDI_Tasks SET currentPriority=:newPriority WHERE jediTaskID=:jediTaskID "
            # sql to update DEFT task table
            schemaDEFT = self.getSchemaDEFT()
            sqlD = f"UPDATE {schemaDEFT}.T_TASK SET current_priority=:newPriority,timestamp=CURRENT_DATE WHERE taskid=:jediTaskID "
            # update job priorities
            sqlJ = "UPDATE ATLAS_PANDA.{0} SET currentPriority=:newPriority WHERE jediTaskID=:jediTaskID "
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            varMap[":newPriority"] = newPriority
            # update JEDI
            self.cur.execute(sqlT + comment, varMap)
            nRow = self.cur.rowcount
            if nRow == 1:
                # update jobs
                for tableName in ["jobsActive4", "jobsDefined4", "jobsWaiting4"]:
                    self.cur.execute(sqlJ.format(tableName) + comment, varMap)
            # update DEFT
            self.cur.execute(sqlD + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            _logger.debug(f"{methodName} done with {nRow}")
            return nRow
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None

    # throttle user jobs
    def throttleUserJobs(self, prodUserName, workingGroup, get_dict):
        comment = " /* DBProxy.throttleUserJobs */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <user={prodUserName} group={workingGroup}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            # sql to get tasks
            sqlT = "SELECT /*+ INDEX_RS_ASC(tab JOBSACTIVE4_PRODUSERNAMEST_IDX) */ "
            sqlT += "distinct jediTaskID "
            sqlT += "FROM ATLAS_PANDA.jobsActive4 tab "
            sqlT += "WHERE prodSourceLabel=:prodSourceLabel AND prodUserName=:prodUserName "
            sqlT += "AND jobStatus=:oldJobStatus AND relocationFlag=:oldRelFlag "
            sqlT += "AND maxCpuCount>:maxTime "
            if workingGroup is not None:
                sqlT += "AND workingGroup=:workingGroup "
            else:
                sqlT += "AND workingGroup IS NULL "
            # sql to get jobs
            sqlJ = (
                "SELECT "
                "PandaID, jediTaskID, cloud, computingSite, prodSourceLabel "
                "FROM ATLAS_PANDA.jobsActive4 "
                "WHERE jediTaskID=:jediTaskID "
                "AND jobStatus=:oldJobStatus AND relocationFlag=:oldRelFlag "
                "AND maxCpuCount>:maxTime "
            )
            # sql to update job
            sqlU = (
                "UPDATE {0}.jobsActive4 SET jobStatus=:newJobStatus,relocationFlag=:newRelFlag "
                "WHERE jediTaskID=:jediTaskID AND jobStatus=:oldJobStatus AND maxCpuCount>:maxTime"
            ).format(panda_config.schemaPANDA)
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            varMap = {}
            varMap[":prodSourceLabel"] = "user"
            varMap[":oldRelFlag"] = 1
            varMap[":prodUserName"] = prodUserName
            varMap[":oldJobStatus"] = "activated"
            varMap[":maxTime"] = 6 * 60 * 60
            if workingGroup is not None:
                varMap[":workingGroup"] = workingGroup
            # get tasks
            self.cur.execute(sqlT + comment, varMap)
            resT = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # loop over all tasks
            tasks = [jediTaskID for jediTaskID, in resT]
            random.shuffle(tasks)
            nRow = 0
            nRows = {}
            for jediTaskID in tasks:
                tmpLog.debug(f"reset jediTaskID={jediTaskID}")
                # start transaction
                self.conn.begin()
                # get jobs
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":oldRelFlag"] = 1
                varMap[":oldJobStatus"] = "activated"
                varMap[":maxTime"] = 6 * 60 * 60
                self.cur.execute(sqlJ + comment, varMap)
                resJ = self.cur.fetchall()
                infoMapDict = {
                    pandaID: {
                        "computingSite": computingSite,
                        "cloud": cloud,
                        "prodSourceLabel": prodSourceLabel,
                    }
                    for pandaID, taskID, cloud, computingSite, prodSourceLabel in resJ
                }
                # update jobs
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":newRelFlag"] = 3
                varMap[":newJobStatus"] = "throttled"
                varMap[":oldJobStatus"] = "activated"
                varMap[":maxTime"] = 6 * 60 * 60
                self.cur.execute(sqlU + comment, varMap)
                nTmp = self.cur.rowcount
                tmpLog.debug(f"reset {nTmp} jobs")
                if nTmp > 0:
                    nRow += nTmp
                    nRows[jediTaskID] = nTmp
                for pandaID in infoMapDict:
                    infoMap = infoMapDict[pandaID]
                    self.recordStatusChange(
                        pandaID,
                        varMap[":newJobStatus"],
                        infoMap=infoMap,
                        useCommit=False,
                    )
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
            if get_dict:
                tmpLog.debug(f"done with {nRows}")
                return nRows
            tmpLog.debug(f"done with {nRow}")
            return nRow
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None

    # unthrottle user jobs
    def unThrottleUserJobs(self, prodUserName, workingGroup, get_dict):
        comment = " /* DBProxy.unThrottleUserJobs */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <user={prodUserName} group={workingGroup}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            # sql to get tasks
            sqlT = "SELECT /*+ INDEX_RS_ASC(tab JOBSACTIVE4_PRODUSERNAMEST_IDX) */ "
            sqlT += "distinct jediTaskID "
            sqlT += "FROM ATLAS_PANDA.jobsActive4 tab "
            sqlT += "WHERE prodSourceLabel=:prodSourceLabel AND prodUserName=:prodUserName "
            sqlT += "AND jobStatus=:oldJobStatus AND relocationFlag=:oldRelFlag "
            if workingGroup is not None:
                sqlT += "AND workingGroup=:workingGroup "
            else:
                sqlT += "AND workingGroup IS NULL "
            # sql to get jobs
            sqlJ = (
                "SELECT "
                "PandaID, jediTaskID, cloud, computingSite, prodSourceLabel "
                "FROM ATLAS_PANDA.jobsActive4 "
                "WHERE jediTaskID=:jediTaskID "
                "AND jobStatus=:oldJobStatus AND relocationFlag=:oldRelFlag "
            )
            # sql to update job
            sqlU = (
                "UPDATE {0}.jobsActive4 SET jobStatus=:newJobStatus,relocationFlag=:newRelFlag " "WHERE jediTaskID=:jediTaskID AND jobStatus=:oldJobStatus "
            ).format(panda_config.schemaPANDA)
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            varMap = {}
            varMap[":prodSourceLabel"] = "user"
            varMap[":oldRelFlag"] = 3
            varMap[":prodUserName"] = prodUserName
            varMap[":oldJobStatus"] = "throttled"
            if workingGroup is not None:
                varMap[":workingGroup"] = workingGroup
            # get tasks
            self.cur.execute(sqlT + comment, varMap)
            resT = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # loop over all tasks
            tasks = [jediTaskID for jediTaskID, in resT]
            random.shuffle(tasks)
            nRow = 0
            nRows = {}
            for jediTaskID in tasks:
                tmpLog.debug(f"reset jediTaskID={jediTaskID}")
                # start transaction
                self.conn.begin()
                # get jobs
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":oldRelFlag"] = 3
                varMap[":oldJobStatus"] = "throttled"
                self.cur.execute(sqlJ + comment, varMap)
                resJ = self.cur.fetchall()
                infoMapDict = {
                    pandaID: {
                        "computingSite": computingSite,
                        "cloud": cloud,
                        "prodSourceLabel": prodSourceLabel,
                    }
                    for pandaID, taskID, cloud, computingSite, prodSourceLabel in resJ
                }
                # update jobs
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":newRelFlag"] = 1
                varMap[":newJobStatus"] = "activated"
                varMap[":oldJobStatus"] = "throttled"
                self.cur.execute(sqlU + comment, varMap)
                nTmp = self.cur.rowcount
                tmpLog.debug(f"reset {nTmp} jobs")
                if nTmp > 0:
                    nRow += nTmp
                    nRows[jediTaskID] = nTmp
                for pandaID in infoMapDict:
                    infoMap = infoMapDict[pandaID]
                    self.recordStatusChange(
                        pandaID,
                        varMap[":newJobStatus"],
                        infoMap=infoMap,
                        useCommit=False,
                    )
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
            if get_dict:
                tmpLog.debug(f"done with {nRows}")
                return nRows
            tmpLog.debug(f"done with {nRow}")
            return nRow
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None

    # get throttled users
    def getThrottledUsers(self):
        comment = " /* DBProxy.getThrottledUsers */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        retVal = set()
        try:
            # sql to get users
            sqlT = "SELECT distinct prodUserName,workingGroup FROM ATLAS_PANDA.jobsActive4 "
            sqlT += "WHERE prodSourceLabel=:prodSourceLabel AND jobStatus=:jobStatus AND relocationFlag=:relocationFlag "
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            varMap = {}
            varMap[":prodSourceLabel"] = "user"
            varMap[":relocationFlag"] = 3
            varMap[":jobStatus"] = "throttled"
            # get datasets
            self.cur.execute(sqlT + comment, varMap)
            resPs = self.cur.fetchall()
            for prodUserName, workingGroup in resPs:
                retVal.add((prodUserName, workingGroup))
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"done with {str(retVal)}")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return []

    # kill active consumers related to an ES job
    def killEventServiceConsumers(self, job, killedFlag, useCommit=True):
        comment = " /* DBProxy.killEventServiceConsumers */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <PandaID={job.PandaID}>"
        _logger.debug(f"{methodName} : start")
        try:
            # begin transaction
            if useCommit:
                self.conn.begin()
            # sql to get consumers
            sqlCP = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlCP += f"distinct PandaID FROM {panda_config.schemaJEDI}.JEDI_Events tab "
            sqlCP += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            sqlCP += "AND NOT status IN (:esDiscarded,:esCancelled) "
            # sql to discard or cancel event ranges
            sqlDE = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlDE += f"{panda_config.schemaJEDI}.JEDI_Events tab "
            sqlDE += "SET status=:status "
            sqlDE += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID AND PandaID=:PandaID "
            sqlDE += "AND status IN (:esFinished,:esDone) "
            sqlCE = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlCE += f"{panda_config.schemaJEDI}.JEDI_Events tab "
            sqlCE += "SET status=:status "
            sqlCE += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID AND PandaID=:PandaID "
            sqlCE += "AND NOT status IN (:esFinished,:esDone,:esDiscarded,:esCancelled,:esFailed,:esFatal,:esCorrupted) "
            # look for consumers for each input
            killPandaIDs = {}
            for fileSpec in job.Files:
                if fileSpec.type not in ["input", "pseudo_input"]:
                    continue
                if fileSpec.fileID in ["NULL", None]:
                    continue
                # get PandaIDs
                varMap = {}
                varMap[":jediTaskID"] = fileSpec.jediTaskID
                varMap[":datasetID"] = fileSpec.datasetID
                varMap[":fileID"] = fileSpec.fileID
                varMap[":esDiscarded"] = EventServiceUtils.ST_discarded
                varMap[":esCancelled"] = EventServiceUtils.ST_cancelled
                self.cur.arraysize = 100000
                self.cur.execute(sqlCP + comment, varMap)
                resPs = self.cur.fetchall()
                for (esPandaID,) in resPs:
                    if esPandaID not in killPandaIDs:
                        killPandaIDs[esPandaID] = set()
                    killPandaIDs[esPandaID].add((fileSpec.jediTaskID, fileSpec.datasetID, fileSpec.fileID))
            # kill consumers
            sqlDJS = f"SELECT {JobSpec.columnNames()} "
            sqlDJS += "FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
            sqlDJS += "FOR UPDATE NOWAIT "
            sqlDJD = "DELETE FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID"
            sqlDJI = f"INSERT INTO ATLAS_PANDA.jobsArchived4 ({JobSpec.columnNames()}) "
            sqlDJI += JobSpec.bindValuesExpression()
            sqlFSF = "UPDATE ATLAS_PANDA.filesTable4 SET status=:newStatus "
            sqlFSF += "WHERE PandaID=:PandaID AND type IN (:type1,:type2) "
            sqlFMod = "UPDATE ATLAS_PANDA.filesTable4 SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
            sqlMMod = "UPDATE ATLAS_PANDA.metaTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
            sqlPMod = "UPDATE ATLAS_PANDA.jobParamsTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
            nKilled = 0
            killPandaIDsList = sorted(killPandaIDs)
            for pandaID in killPandaIDsList:
                # ignore original PandaID since it will be killed by caller
                if pandaID == job.PandaID:
                    continue
                # skip jobsetID
                if pandaID == job.jobsetID:
                    continue
                # read job
                varMap = {}
                varMap[":PandaID"] = pandaID
                self.cur.arraysize = 10
                self.cur.execute(sqlDJS + comment, varMap)
                resJob = self.cur.fetchall()
                if len(resJob) == 0:
                    continue
                # instantiate JobSpec
                dJob = JobSpec()
                dJob.pack(resJob[0])
                # skip if jobset different
                if dJob.jobsetID != job.jobsetID:
                    _logger.debug(f"{methodName} : skip consumer {pandaID} since jobsetID is different")
                    continue
                # skip jumbo
                if EventServiceUtils.isJumboJob(dJob):
                    _logger.debug(f"{methodName} : skip jumbo {pandaID}")
                    continue
                _logger.debug(f"{methodName} : kill associated consumer {pandaID}")
                # delete
                varMap = {}
                varMap[":PandaID"] = pandaID
                self.cur.execute(sqlDJD + comment, varMap)
                retD = self.cur.rowcount
                if retD == 0:
                    continue
                # set error code
                dJob.endTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
                if EventServiceUtils.isJobCloningJob(dJob):
                    dJob.jobStatus = "closed"
                    dJob.jobSubStatus = "jc_unlock"
                    dJob.taskBufferErrorCode = ErrorCode.EC_JobCloningUnlock
                    dJob.taskBufferErrorDiag = f"closed since another clone PandaID={job.PandaID} got semaphore"
                elif killedFlag:
                    dJob.jobStatus = "cancelled"
                    dJob.jobSubStatus = "es_killed"
                    dJob.taskBufferErrorCode = ErrorCode.EC_EventServiceKillOK
                    dJob.taskBufferErrorDiag = f"killed since an associated consumer PandaID={job.PandaID} was killed"
                else:
                    dJob.jobStatus = "failed"
                    dJob.jobSubStatus = "es_aborted"
                    dJob.taskBufferErrorCode = ErrorCode.EC_EventServiceKillNG
                    dJob.taskBufferErrorDiag = f"killed since an associated consumer PandaID={job.PandaID} failed"
                dJob.modificationTime = dJob.endTime
                dJob.stateChangeTime = dJob.endTime
                # insert
                self.cur.execute(sqlDJI + comment, dJob.valuesMap())
                # set file status
                varMap = {}
                varMap[":PandaID"] = pandaID
                varMap[":type1"] = "output"
                varMap[":type2"] = "log"
                varMap[":newStatus"] = "failed"
                self.cur.execute(sqlFSF + comment, varMap)
                # update files,metadata,parametes
                varMap = {}
                varMap[":PandaID"] = pandaID
                varMap[":modificationTime"] = dJob.modificationTime
                self.cur.execute(sqlFMod + comment, varMap)
                self.cur.execute(sqlMMod + comment, varMap)
                self.cur.execute(sqlPMod + comment, varMap)
                nKilled += 1
                # discard event ranges
                nRowsDis = 0
                nRowsCan = 0
                for jediTaskID, datasetID, fileID in killPandaIDs[pandaID]:
                    varMap = {}
                    varMap[":jediTaskID"] = jediTaskID
                    varMap[":datasetID"] = datasetID
                    varMap[":fileID"] = fileID
                    varMap[":PandaID"] = pandaID
                    varMap[":status"] = EventServiceUtils.ST_discarded
                    varMap[":esFinished"] = EventServiceUtils.ST_finished
                    varMap[":esDone"] = EventServiceUtils.ST_done
                    if not job.notDiscardEvents():
                        self.cur.execute(sqlDE + comment, varMap)
                        nRowsDis += self.cur.rowcount
                    varMap[":status"] = EventServiceUtils.ST_cancelled
                    varMap[":esDiscarded"] = EventServiceUtils.ST_discarded
                    varMap[":esCancelled"] = EventServiceUtils.ST_cancelled
                    varMap[":esCorrupted"] = EventServiceUtils.ST_corrupted
                    varMap[":esFatal"] = EventServiceUtils.ST_fatal
                    varMap[":esFailed"] = EventServiceUtils.ST_failed
                    self.cur.execute(sqlCE + comment, varMap)
                    nRowsCan += self.cur.rowcount
                _logger.debug(f"{methodName} : {pandaID} discarded {nRowsDis} events")
                _logger.debug(f"{methodName} : {pandaID} cancelled {nRowsCan} events")
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError("Commit error")
            _logger.debug(f"{methodName} killed {nKilled} jobs")
            return True
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            if not useCommit:
                raise RuntimeError(f"{methodName} failed")
            return False

    # kill unused consumers related to an ES job
    def killUnusedEventServiceConsumers(self, job, useCommit=True, killAll=False, checkAttemptNr=False):
        comment = " /* DBProxy.killUnusedEventServiceConsumers */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <PandaID={job.PandaID}>"
        _logger.debug(f"{methodName} : start")
        try:
            # begin transaction
            if useCommit:
                self.conn.begin()
            self.cur.arraysize = 100000
            # get dataset
            sqlPD = "SELECT f.datasetID,f.fileID FROM ATLAS_PANDA.JEDI_Datasets d,ATLAS_PANDA.filesTable4 f "
            sqlPD += "WHERE d.jediTaskID=:jediTaskID AND d.type IN (:type1,:type2) AND d.masterID IS NULL "
            sqlPD += "AND f.PandaID=:PandaID AND f.jeditaskID=f.jediTaskID AND f.datasetID=d.datasetID "
            varMap = {}
            varMap[":jediTaskID"] = job.jediTaskID
            varMap[":PandaID"] = job.PandaID
            varMap[":type1"] = "input"
            varMap[":type2"] = "pseudo_input"
            self.cur.execute(sqlPD + comment, varMap)
            resPD = self.cur.fetchall()
            # get PandaIDs
            killPandaIDs = set()
            myAttemptNr = None
            sqlCP = "SELECT PandaID,attemptNr FROM ATLAS_PANDA.filesTable4 WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            for datasetID, fileID in resPD:
                if fileID is None:
                    continue
                varMap = {}
                varMap[":jediTaskID"] = job.jediTaskID
                varMap[":datasetID"] = datasetID
                varMap[":fileID"] = fileID
                self.cur.execute(sqlCP + comment, varMap)
                resCP = self.cur.fetchall()
                for esPandaID, esAttemptNr in resCP:
                    if esPandaID == job.PandaID:
                        myAttemptNr = esAttemptNr
                        continue
                    killPandaIDs.add((esPandaID, esAttemptNr))
            # kill consumers
            nKilled = 0
            sqlDJS = f"SELECT {JobSpec.columnNames()} "
            sqlDJS += "FROM ATLAS_PANDA.{0} WHERE PandaID=:PandaID"
            sqlDJD = "DELETE FROM ATLAS_PANDA.{0} WHERE PandaID=:PandaID"
            sqlDJI = f"INSERT INTO ATLAS_PANDA.jobsArchived4 ({JobSpec.columnNames()}) "
            sqlDJI += JobSpec.bindValuesExpression()
            sqlFSF = "UPDATE ATLAS_PANDA.filesTable4 SET status=:newStatus "
            sqlFSF += "WHERE PandaID=:PandaID AND type IN (:type1,:type2) "
            sqlFMod = "UPDATE ATLAS_PANDA.filesTable4 SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
            sqlMMod = "UPDATE ATLAS_PANDA.metaTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
            sqlPMod = "UPDATE ATLAS_PANDA.jobParamsTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
            for pandaID, attemptNr in killPandaIDs:
                # read job
                varMap = {}
                varMap[":PandaID"] = pandaID
                self.cur.arraysize = 10
                deletedFlag = False
                notToDelete = False
                for tableName in ["jobsActive4", "jobsDefined4", "jobsWaiting4"]:
                    # check attemptNr
                    if checkAttemptNr and attemptNr != myAttemptNr:
                        _logger.debug(f"{methodName} : skip to kill {pandaID} since attemptNr:{attemptNr} is different from mine={myAttemptNr}")
                        notToDelete = True
                        break
                    self.cur.execute(sqlDJS.format(tableName) + comment, varMap)
                    resJob = self.cur.fetchall()
                    if len(resJob) == 0:
                        continue
                    # instantiate JobSpec
                    dJob = JobSpec()
                    dJob.pack(resJob[0])
                    # not kill all status
                    if not killAll:
                        if dJob.jobStatus not in ["activated", "assigned", "throttled"]:
                            _logger.debug(f"{methodName} : skip to kill unused consumer {pandaID} since status={dJob.jobStatus}")
                            notToDelete = True
                            break
                    # skip merge
                    if EventServiceUtils.isEventServiceMerge(dJob):
                        _logger.debug(f"{methodName} : skip to kill merge {pandaID}")
                        notToDelete = True
                        break
                    # skip jumbo
                    if EventServiceUtils.isJumboJob(dJob):
                        _logger.debug(f"{methodName} : skip to kill jumbo {pandaID}")
                        notToDelete = True
                        break
                    # delete
                    varMap = {}
                    varMap[":PandaID"] = pandaID
                    self.cur.execute(sqlDJD.format(tableName) + comment, varMap)
                    retD = self.cur.rowcount
                    if retD != 0:
                        deletedFlag = True
                        break
                # not to be deleted
                if notToDelete:
                    continue
                # not found
                if not deletedFlag:
                    _logger.debug(f"{methodName} : skip to kill {pandaID} as already deleted")
                    continue
                _logger.debug(f"{methodName} : kill unused consumer {pandaID}")
                # set error code
                dJob.jobStatus = "closed"
                dJob.endTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
                if EventServiceUtils.isJobCloningJob(dJob):
                    dJob.jobSubStatus = "jc_unlock"
                    dJob.taskBufferErrorCode = ErrorCode.EC_JobCloningUnlock
                    dJob.taskBufferErrorDiag = f"closed since another clone PandaID={job.PandaID} got semaphore while waiting in the queue"
                else:
                    dJob.jobSubStatus = "es_unused"
                    dJob.taskBufferErrorCode = ErrorCode.EC_EventServiceUnused
                    dJob.taskBufferErrorDiag = "killed since all event ranges were processed by other consumers while waiting in the queue"
                dJob.modificationTime = dJob.endTime
                dJob.stateChangeTime = dJob.endTime
                # insert
                self.cur.execute(sqlDJI + comment, dJob.valuesMap())
                # set file status
                varMap = {}
                varMap[":PandaID"] = pandaID
                varMap[":type1"] = "output"
                varMap[":type2"] = "log"
                varMap[":newStatus"] = "failed"
                self.cur.execute(sqlFSF + comment, varMap)
                # update files,metadata,parametes
                varMap = {}
                varMap[":PandaID"] = pandaID
                varMap[":modificationTime"] = dJob.modificationTime
                self.cur.execute(sqlFMod + comment, varMap)
                self.cur.execute(sqlMMod + comment, varMap)
                self.cur.execute(sqlPMod + comment, varMap)
                nKilled += 1
                # record status change
                self.recordStatusChange(dJob.PandaID, dJob.jobStatus, jobInfo=dJob, useCommit=False)
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError("Commit error")
            _logger.debug(f"{methodName} killed {nKilled} jobs")
            return True
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            if not useCommit:
                raise RuntimeError(f"{methodName} failed")
            return False

    # kill unused event ranges
    def killUnusedEventRanges(self, jediTaskID, jobsetID):
        comment = " /* DBProxy.killUnusedEventRanges */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID} jobsetID={jobsetID}>"
        tmpLog = LogWrapper(_logger, methodName)
        # sql to kill event ranges
        varMap = {}
        varMap[":jediTaskID"] = jediTaskID
        varMap[":jobsetID"] = jobsetID
        varMap[":esReady"] = EventServiceUtils.ST_ready
        varMap[":esCancelled"] = EventServiceUtils.ST_cancelled
        sqlCE = f"UPDATE {panda_config.schemaJEDI}.JEDI_Events "
        sqlCE += "SET status=:esCancelled "
        sqlCE += "WHERE jediTaskID=:jediTaskID AND pandaID=:jobsetID "
        sqlCE += "AND status=:esReady "
        self.cur.execute(sqlCE, varMap)
        nRowsCan = self.cur.rowcount
        tmpLog.debug(f"cancelled {nRowsCan} events")

    # release unprocessed events
    def release_unprocessed_events(self, jedi_task_id, panda_id):
        comment = " /* DBProxy.release_unprocessed_events */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jedi_task_id} PandaID={panda_id}>"
        tmpLog = LogWrapper(_logger, methodName)
        # look for hopeless events
        varMap = {}
        varMap[":jediTaskID"] = jedi_task_id
        varMap[":PandaID"] = panda_id
        varMap[":esReady"] = EventServiceUtils.ST_ready
        varMap[":esFinished"] = EventServiceUtils.ST_finished
        varMap[":esFailed"] = EventServiceUtils.ST_failed
        sqlBE = (
            "SELECT job_processID FROM {0}.JEDI_Events "
            "WHERE jediTaskID=:jediTaskID AND pandaID=:PandaID "
            "AND status NOT IN (:esReady,:esFinished,:esFailed) "
            "AND attemptNr=1 "
        ).format(panda_config.schemaJEDI)
        self.cur.execute(sqlBE, varMap)
        resBD = self.cur.fetchall()
        if len(resBD) > 0:
            # report very large loss
            c = iDDS_Client(idds.common.utils.get_rest_host())
            for (sample_id,) in resBD:
                tmpLog.debug(f"reporting large loss for id={sample_id}")
                c.update_hyperparameter(workload_id=jedi_task_id, request_id=None, id=sample_id, loss=1e5)
        # release
        sqlCE = f"UPDATE {panda_config.schemaJEDI}.JEDI_Events "
        sqlCE += (
            "SET status=(CASE WHEN attemptNr>1 THEN :esReady ELSE :esFailed END),"
            "pandaID=(CASE WHEN attemptNr>1 THEN 0 ELSE pandaID END),"
            "attemptNr=attemptNr-1 "
            "WHERE jediTaskID=:jediTaskID AND pandaID=:PandaID "
            "AND status NOT IN (:esReady,:esFinished,:esFailed) "
        )
        self.cur.execute(sqlCE, varMap)
        nRowsCan = self.cur.rowcount
        tmpLog.debug(f"released {nRowsCan} events")

    # kill used event ranges
    def killUsedEventRanges(self, jediTaskID, pandaID, notDiscardEvents=False):
        comment = " /* DBProxy.killUsedEventRanges */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID} pandaID={pandaID}>"
        tmpLog = LogWrapper(_logger, methodName)
        # sql to discard or cancel event ranges
        sqlDE = "UPDATE "
        sqlDE += f"{panda_config.schemaJEDI}.JEDI_Events tab "
        sqlDE += "SET status=:status "
        sqlDE += "WHERE jediTaskID=:jediTaskID AND PandaID=:PandaID "
        sqlDE += "AND status IN (:esFinished,:esDone) "
        sqlCE = "UPDATE "
        sqlCE += f"{panda_config.schemaJEDI}.JEDI_Events tab "
        sqlCE += "SET status=:status "
        sqlCE += "WHERE jediTaskID=:jediTaskID AND PandaID=:PandaID "
        sqlCE += "AND NOT status IN (:esFinished,:esDone,:esDiscarded,:esCancelled,:esFailed,:esFatal,:esCorrupted) "
        varMap = {}
        varMap[":jediTaskID"] = jediTaskID
        varMap[":PandaID"] = pandaID
        varMap[":status"] = EventServiceUtils.ST_discarded
        varMap[":esFinished"] = EventServiceUtils.ST_finished
        varMap[":esDone"] = EventServiceUtils.ST_done
        if not notDiscardEvents:
            self.cur.execute(sqlDE + comment, varMap)
            nRowsDis = self.cur.rowcount
        else:
            nRowsDis = 0
        varMap[":status"] = EventServiceUtils.ST_cancelled
        varMap[":esDiscarded"] = EventServiceUtils.ST_discarded
        varMap[":esCancelled"] = EventServiceUtils.ST_cancelled
        varMap[":esCorrupted"] = EventServiceUtils.ST_corrupted
        varMap[":esFatal"] = EventServiceUtils.ST_fatal
        varMap[":esFailed"] = EventServiceUtils.ST_failed
        self.cur.execute(sqlCE + comment, varMap)
        nRowsCan = self.cur.rowcount
        tmpLog.debug(f"discarded {nRowsDis} events")
        tmpLog.debug(f"cancelled {nRowsCan} events")

    # set corrupted events
    def setCorruptedEventRanges(self, jediTaskID, pandaID):
        comment = " /* DBProxy.setCorruptedEventRanges */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID} pandaID={pandaID}>"
        tmpLog = LogWrapper(_logger, methodName)
        # sql to get bad files
        sqlBD = "SELECT lfn FROM ATLAS_PANDA.filesTable4 WHERE PandaID=:PandaID AND type=:type AND status=:status "
        # sql to get PandaID produced the bad file
        sqlPP = "SELECT row_ID,PandaID FROM ATLAS_PANDA.filesTable4 WHERE lfn=:lfn AND type=:type "
        # sql to get PandaIDs with jobMetrics
        sqlJJ = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
        sqlJJ += "DISTINCT e.PandaID FROM ATLAS_PANDA.filesTable4 f,ATLAS_PANDA.JEDI_Events e "
        sqlJJ += "WHERE f.PandaID=:PandaID AND f.type IN (:type1,:type2) "
        sqlJJ += "AND e.jediTaskID=f.jediTaskID AND e.datasetID=f.datasetID AND e.fileID=f.fileID "
        # sql to get jobMetrics
        sqlJM = "SELECT jobMetrics FROM ATLAS_PANDA.jobsArchived4 WHERE PandaID=:PandaID "
        sqlJM += "UNION "
        sqlJM += "SELECT jobMetrics FROM ATLAS_PANDAARCH.jobsArchived WHERE PandaID=:PandaID AND modificationTime=CURRENT_DATE-90 "
        # sql to get dataset and file IDs
        sqlGI = "SELECT datasetID,fileID FROM ATLAS_PANDA.filesTable4 "
        sqlGI += "WHERE PandaID=:PandaID AND type IN (:t1,:t2) "
        # sql to update event ranges
        sqlCE = "UPDATE "
        sqlCE += f"{panda_config.schemaJEDI}.JEDI_Events tab "
        sqlCE += "SET status=:esCorrupted "
        sqlCE += "WHERE jediTaskID=:jediTaskID AND PandaID=:PandaID AND zipRow_ID=:row_ID "
        sqlCE += "AND datasetID=:datasetID AND fileID=:fileID AND status=:esDone "
        # sql to update event ranges with jobMetrics
        sqlJE = "UPDATE "
        sqlJE += f"{panda_config.schemaJEDI}.JEDI_Events tab "
        sqlJE += "SET status=:esCorrupted "
        sqlJE += "WHERE jediTaskID=:jediTaskID AND PandaID=:PandaID "
        sqlJE += "AND datasetID=:datasetID AND fileID=:fileID AND status=:esDone "
        # get bad files
        varMap = {}
        varMap[":PandaID"] = pandaID
        varMap[":status"] = "corrupted"
        varMap[":type"] = "zipinput"
        self.cur.execute(sqlBD + comment, varMap)
        resBD = self.cur.fetchall()
        for (lfn,) in resBD:
            # get origon PandaID
            nCor = 0
            varMap = {}
            varMap[":lfn"] = lfn
            varMap[":type"] = "zipoutput"
            self.cur.execute(sqlPP + comment, varMap)
            resPP = self.cur.fetchall()
            if len(resPP) > 0:
                # with zipoutput
                for zipRow_ID, oPandaID in resPP:
                    # get dataset and file IDs
                    varMap = {}
                    varMap[":PandaID"] = oPandaID
                    varMap[":t1"] = "input"
                    varMap[":t2"] = "pseudo_input"
                    self.cur.execute(sqlGI + comment, varMap)
                    resGI = self.cur.fetchall()
                    # set corrupted
                    for datasetID, fileID in resGI:
                        varMap = {}
                        varMap[":PandaID"] = oPandaID
                        varMap[":row_ID"] = zipRow_ID
                        varMap[":jediTaskID"] = jediTaskID
                        varMap[":datasetID"] = datasetID
                        varMap[":fileID"] = fileID
                        varMap[":esDone"] = EventServiceUtils.ST_done
                        varMap[":esCorrupted"] = EventServiceUtils.ST_corrupted
                        self.cur.execute(sqlCE + comment, varMap)
                        nCor += self.cur.rowcount
            else:
                # check jobMetrics
                varMap = dict()
                varMap[":PandaID"] = pandaID
                varMap[":type1"] = "input"
                varMap[":type2"] = "pseudo_input"
                self.cur.execute(sqlJJ + comment, varMap)
                resJJ = self.cur.fetchall()
                # get jobMetrics
                for (oPandaID,) in resJJ:
                    varMap = dict()
                    varMap[":PandaID"] = oPandaID
                    self.cur.execute(sqlJM + comment, varMap)
                    resJM = self.cur.fetchone()
                    if resJM is not None:
                        (jobMetrics,) = resJM
                        if jobMetrics is not None and f"outputZipName={lfn}" in jobMetrics:
                            # get dataset and file IDs
                            varMap = {}
                            varMap[":PandaID"] = oPandaID
                            varMap[":t1"] = "input"
                            varMap[":t2"] = "pseudo_input"
                            self.cur.execute(sqlGI + comment, varMap)
                            resGI = self.cur.fetchall()
                            # set corrupted
                            for datasetID, fileID in resGI:
                                varMap = {}
                                varMap[":PandaID"] = oPandaID
                                varMap[":jediTaskID"] = jediTaskID
                                varMap[":datasetID"] = datasetID
                                varMap[":fileID"] = fileID
                                varMap[":esDone"] = EventServiceUtils.ST_done
                                varMap[":esCorrupted"] = EventServiceUtils.ST_corrupted
                                self.cur.execute(sqlJE + comment, varMap)
                                nCor += self.cur.rowcount
                            break
            tmpLog.debug(f"{nCor} corrupted events in {lfn}")

    # check failure count due to corrupted files
    def checkFailureCountWithCorruptedFiles(self, jediTaskID, pandaID):
        comment = " /* DBProxy.checkFailureCountWithCorruptedFiles */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID} pandaID={pandaID}>"
        tmpLog = LogWrapper(_logger, methodName)
        # sql to failure counts
        sqlBD = "SELECT f2.lfn,COUNT(*) FROM ATLAS_PANDA.filesTable4 f1, ATLAS_PANDA.filesTable4 f2 "
        sqlBD += "WHERE f1.PandaID=:PandaID AND f1.type=:type AND f1.status=:status "
        sqlBD += "AND f2.lfn=f1.lfn AND f2.type=:type AND f2.status=:status AND f2.jediTaskID=:jediTaskID "
        sqlBD += "GROUP BY f2.lfn "
        varMap = {}
        varMap[":jediTaskID"] = jediTaskID
        varMap[":PandaID"] = pandaID
        varMap[":status"] = "corrupted"
        varMap[":type"] = "zipinput"
        self.cur.execute(sqlBD + comment, varMap)
        resBD = self.cur.fetchall()
        tooMany = False
        for lfn, nFailed in resBD:
            tmpLog.debug(f"{nFailed} failures with {lfn}")
            if nFailed >= 3:
                tooMany = True
        tmpLog.debug(f"too many failures : {tooMany}")
        return tooMany

    # check attemptNr for more retry
    def checkMoreRetryJEDI(self, job):
        comment = " /* DBProxy.self.checkMoreRetryJEDI */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <PandaID={job.PandaID}>"
        _logger.debug(f"{methodName} : start")
        # sql to get files
        sqlGF = "SELECT datasetID,fileID,attemptNr FROM ATLAS_PANDA.filesTable4 "
        sqlGF += "WHERE PandaID=:PandaID AND type IN (:type1,:type2) "
        # sql to check file
        sqlFJ = f"SELECT attemptNr,maxAttempt,failedAttempt,maxFailure FROM {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
        sqlFJ += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
        sqlFJ += "AND attemptNr=:attemptNr AND keepTrack=:keepTrack AND PandaID=:PandaID "
        # get files
        varMap = {}
        varMap[":PandaID"] = job.PandaID
        varMap[":type1"] = "input"
        varMap[":type2"] = "pseudo_input"
        self.cur.execute(sqlGF + comment, varMap)
        resGF = self.cur.fetchall()
        for datasetID, fileID, attemptNr in resGF:
            # check JEDI contents
            varMap = {}
            varMap[":jediTaskID"] = job.jediTaskID
            varMap[":datasetID"] = datasetID
            varMap[":fileID"] = fileID
            varMap[":attemptNr"] = attemptNr
            varMap[":PandaID"] = job.PandaID
            varMap[":keepTrack"] = 1
            self.cur.execute(sqlFJ + comment, varMap)
            resFJ = self.cur.fetchone()
            if resFJ is None:
                continue
            attemptNr, maxAttempt, failedAttempt, maxFailure = resFJ
            if maxAttempt is None:
                continue
            if attemptNr + 1 >= maxAttempt:
                # hit the limit
                _logger.debug(f"{methodName} : NG - fileID={fileID} no more attempt attemptNr({attemptNr})+1>=maxAttempt({maxAttempt})")
                return False
            if maxFailure is not None and failedAttempt is not None and failedAttempt + 1 >= maxFailure:
                # hit the limit
                _logger.debug(f"{methodName} : NG - fileID={fileID} no more attempt failedAttempt({failedAttempt})+1>=maxFailure({maxFailure})")
                return False
        _logger.debug(f"{methodName} : OK")
        return True

    # get the list of jobdefIDs for failed jobs in a task
    def getJobdefIDsForFailedJob(self, jediTaskID):
        comment = " /* DBProxy.getJobdefIDsForFailedJob */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID}>"
        _logger.debug(f"{methodName} : start")
        try:
            # begin transaction
            self.conn.begin()
            # dql to get jobDefIDs
            sqlGF = "SELECT distinct jobDefinitionID FROM ATLAS_PANDA.jobsActive4 "
            sqlGF += "WHERE jediTaskID=:jediTaskID AND jobStatus=:jobStatus "
            sqlGF += "AND attemptNr<maxAttempt "
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            varMap[":jobStatus"] = "failed"
            self.cur.execute(sqlGF + comment, varMap)
            resGF = self.cur.fetchall()
            retList = []
            for (jobDefinitionID,) in resGF:
                retList.append(jobDefinitionID)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            _logger.debug(f"{methodName} : {str(retList)}")
            return retList
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return []

    # change task attribute
    def changeTaskAttributePanda(self, jediTaskID, attrName, attrValue):
        comment = " /* DBProxy.changeTaskAttributePanda */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID}>"
        _logger.debug(f"{methodName} name={attrName} value={attrValue}")
        try:
            # sql to update JEDI task table
            sqlT = f"UPDATE {panda_config.schemaJEDI}.JEDI_Tasks SET "
            sqlT += "{0}=:{0} WHERE jediTaskID=:jediTaskID ".format(attrName)
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            keyName = f":{attrName}"
            varMap[keyName] = attrValue
            # update JEDI
            self.cur.execute(sqlT + comment, varMap)
            nRow = self.cur.rowcount
            if nRow:
                self.reset_resource_type_task(jediTaskID, use_commit=False)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            _logger.debug(f"{methodName} done with {nRow}")
            return nRow
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None

    # change split rule for task
    def changeTaskSplitRulePanda(self, jediTaskID, attrName, attrValue, useCommit=True, sendLog=True):
        comment = " /* DBProxy.changeTaskSplitRulePanda */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID}>"
        tmpLog = LogWrapper(_logger, methodName, monToken=f"<jediTaskID={jediTaskID}>")
        tmpLog.debug(f"changing {attrName}={attrValue}")
        try:
            # sql to get split rule
            sqlS = f"SELECT splitRule FROM {panda_config.schemaJEDI}.JEDI_Tasks "
            sqlS += "WHERE jediTaskID=:jediTaskID "
            # sql to update JEDI task table
            sqlT = f"UPDATE {panda_config.schemaJEDI}.JEDI_Tasks SET "
            sqlT += "splitRule=:splitRule WHERE jediTaskID=:jediTaskID "
            # start transaction
            if useCommit:
                self.conn.begin()
            # select
            self.cur.arraysize = 10
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            # get split rule
            self.cur.execute(sqlS + comment, varMap)
            resS = self.cur.fetchone()
            if resS is None:
                retVal = 0
            else:
                splitRule = resS[0]
                if splitRule is None:
                    items = []
                else:
                    items = splitRule.split(",")
                # remove old
                newItems = []
                for tmpItem in items:
                    if tmpItem.startswith(f"{attrName}="):
                        continue
                    newItems.append(tmpItem)
                # add new
                if attrValue not in [None, "", "None"]:
                    newItems.append(f"{attrName}={attrValue}")
                # update
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":splitRule"] = ",".join(newItems)
                self.cur.execute(sqlT + comment, varMap)
                retVal = 1
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError("Commit error")
            tmpLog.debug(f"done with {retVal}")
            if sendLog:
                tmpLog.sendMsg(
                    f"set {attrName}={attrValue} to splitRule",
                    "jedi",
                    "pandasrv",
                )
            return retVal
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None

    # increase attempt number for unprocessed files
    def increaseAttemptNrPanda(self, jediTaskID, increasedNr):
        comment = " /* DBProxy.increaseAttemptNrPanda */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID}>"
        tmpLog = LogWrapper(_logger, methodName, monToken=f"<jediTaskID={jediTaskID}>")
        tmpLog.debug(f"increasedNr={increasedNr}")
        try:
            # sql to check task status
            sqlT = f"SELECT status,oldStatus FROM {panda_config.schemaJEDI}.JEDI_Tasks "
            sqlT += "WHERE jediTaskID=:jediTaskID FOR UPDATE "
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            # get task status
            self.cur.execute(sqlT + comment, varMap)
            resT = self.cur.fetchone()
            if resT is None:
                tmpMsg = f"jediTaskID={jediTaskID} not found"
                tmpLog.debug(tmpMsg)
                retVal = 1, tmpMsg
            else:
                taskStatus, oldStatus = resT
                # check task status
                okStatusList = ["running", "scouting", "ready"]
                if taskStatus not in okStatusList and oldStatus not in okStatusList:
                    tmpMsg = f"command rejected since status={taskStatus} or oldStatus={oldStatus} not in {str(okStatusList)}"
                    tmpLog.debug(tmpMsg)
                    retVal = 2, tmpMsg
                else:
                    # sql to get datasetIDs for master
                    sqlM = f"SELECT datasetID FROM {panda_config.schemaJEDI}.JEDI_Datasets "
                    sqlM += "WHERE jediTaskID=:jediTaskID AND type IN (:type1,:type2) "
                    # sql to increase attempt numbers
                    sqlAB = f"UPDATE {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
                    sqlAB += "SET maxAttempt=CASE WHEN maxAttempt > attemptNr THEN maxAttempt+:increasedNr ELSE attemptNr+:increasedNr END "
                    sqlAB += ",proc_status=CASE WHEN maxAttempt > attemptNr AND maxFailure > failedAttempt THEN proc_status ELSE :proc_status END "
                    sqlAB += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND status=:status AND keepTrack=:keepTrack "
                    # sql to increase attempt numbers and failure counts
                    sqlAF = f"UPDATE {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
                    sqlAF += "SET maxAttempt=CASE WHEN maxAttempt > attemptNr THEN maxAttempt+:increasedNr ELSE attemptNr+:increasedNr END "
                    sqlAF += ",maxFailure=maxFailure+:increasedNr "
                    sqlAF += ",proc_status=CASE WHEN maxAttempt > attemptNr AND maxFailure > failedAttempt THEN proc_status ELSE :proc_status END "
                    sqlAF += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND status=:status AND keepTrack=:keepTrack "
                    # sql to update datasets
                    sqlD = f"UPDATE {panda_config.schemaJEDI}.JEDI_Datasets "
                    sqlD += "SET nFilesUsed=nFilesUsed-:nFilesReset,nFilesFailed=nFilesFailed-:nFilesReset "
                    sqlD += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                    # get datasetIDs for master
                    varMap = {}
                    varMap[":jediTaskID"] = jediTaskID
                    varMap[":type1"] = "input"
                    varMap[":type2"] = "pseudo_input"
                    self.cur.execute(sqlM + comment, varMap)
                    resM = self.cur.fetchall()
                    total_nFilesIncreased = 0
                    total_nFilesReset = 0
                    for (datasetID,) in resM:
                        # increase attempt numbers
                        varMap = {}
                        varMap[":jediTaskID"] = jediTaskID
                        varMap[":datasetID"] = datasetID
                        varMap[":status"] = "ready"
                        varMap[":proc_status"] = "ready"
                        varMap[":keepTrack"] = 1
                        varMap[":increasedNr"] = increasedNr
                        nFilesIncreased = 0
                        nFilesReset = 0
                        # still active and maxFailure is undefined
                        sqlA = sqlAB + "AND maxAttempt>attemptNr AND maxFailure IS NULL "
                        self.cur.execute(sqlA + comment, varMap)
                        nRow = self.cur.rowcount
                        nFilesIncreased += nRow
                        # still active and maxFailure is defined
                        sqlA = sqlAF + "AND maxAttempt>attemptNr AND (maxFailure IS NOT NULL AND maxFailure>failedAttempt) "
                        self.cur.execute(sqlA + comment, varMap)
                        nRow = self.cur.rowcount
                        nFilesIncreased += nRow
                        # already done and maxFailure is undefined
                        sqlA = sqlAB + "AND maxAttempt<=attemptNr AND maxFailure IS NULL "
                        self.cur.execute(sqlA + comment, varMap)
                        nRow = self.cur.rowcount
                        nFilesReset += nRow
                        nFilesIncreased += nRow
                        # already done and maxFailure is defined
                        sqlA = sqlAF + "AND (maxAttempt<=attemptNr OR (maxFailure IS NOT NULL AND maxFailure=failedAttempt)) "
                        self.cur.execute(sqlA + comment, varMap)
                        nRow = self.cur.rowcount
                        nFilesReset += nRow
                        nFilesIncreased += nRow
                        # update dataset
                        if nFilesReset > 0:
                            varMap = {}
                            varMap[":jediTaskID"] = jediTaskID
                            varMap[":datasetID"] = datasetID
                            varMap[":nFilesReset"] = nFilesReset
                            tmpLog.debug(sqlD + comment + str(varMap))
                            self.cur.execute(sqlD + comment, varMap)
                        total_nFilesIncreased += nFilesIncreased
                        total_nFilesReset += nFilesReset
                    tmpMsg = f"increased attemptNr for {total_nFilesIncreased} inputs ({total_nFilesReset} reactivated)"
                    tmpLog.debug(tmpMsg)
                    tmpLog.sendMsg(tmpMsg, "jedi", "pandasrv")
                    retVal = 0, tmpMsg
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug("done")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmpLog, methodName)
            return None, "DB error"

    # get jediTaskID from taskName
    def getTaskIDwithTaskNameJEDI(self, userName, taskName):
        comment = " /* DBProxy.getTaskIDwithTaskNameJEDI */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <userName={userName} taskName={taskName}>"
        _logger.debug(f"{methodName} : start")
        try:
            # begin transaction
            self.conn.begin()
            # sql to get jediTaskID
            sqlGF = f"SELECT MAX(jediTaskID) FROM {panda_config.schemaJEDI}.JEDI_Tasks "
            sqlGF += "WHERE userName=:userName AND taskName=:taskName "
            varMap = {}
            varMap[":userName"] = userName
            varMap[":taskName"] = taskName
            self.cur.execute(sqlGF + comment, varMap)
            resFJ = self.cur.fetchone()
            if resFJ is not None:
                (jediTaskID,) = resFJ
            else:
                jediTaskID = None
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            _logger.debug(f"{methodName} : jediTaskID={jediTaskID}")
            return jediTaskID
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None

    # update error dialog for a jediTaskID
    def updateTaskErrorDialogJEDI(self, jediTaskID, msg):
        comment = " /* DBProxy.updateTaskErrorDialogJEDI */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID}>"
        _logger.debug(f"{methodName} : start")
        try:
            # begin transaction
            self.conn.begin()
            # get existing dialog
            sqlGF = f"SELECT errorDialog FROM {panda_config.schemaJEDI}.JEDI_Tasks "
            sqlGF += "WHERE jediTaskID=:jediTaskID "
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            self.cur.execute(sqlGF + comment, varMap)
            resFJ = self.cur.fetchone()
            if resFJ is not None:
                # update existing dialog
                (errorDialog,) = resFJ
                errorDialog = msg
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":errorDialog"] = errorDialog
                sqlUE = f"UPDATE {panda_config.schemaJEDI}.JEDI_Tasks SET errorDialog=:errorDialog,modificationTime=CURRENT_DATE "
                sqlUE += "WHERE jediTaskID=:jediTaskID "
                self.cur.execute(sqlUE + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            _logger.debug(f"{methodName} : done")
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False

    # update modificationtime for a jediTaskID to trigger subsequent process
    def updateTaskModTimeJEDI(self, jediTaskID, newStatus):
        comment = " /* DBProxy.updateTaskErrorDialogJEDI */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID}>"
        _logger.debug(f"{methodName} : start")
        try:
            # begin transaction
            self.conn.begin()
            # update mod time
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            if newStatus is not None:
                varMap[":newStatus"] = newStatus
            sqlUE = f"UPDATE {panda_config.schemaJEDI}.JEDI_Tasks SET "
            sqlUE += "modificationTime=CURRENT_DATE-1,"
            if newStatus is not None:
                sqlUE += "status=:newStatus,oldStatus=NULL,"
            sqlUE = sqlUE[:-1]
            sqlUE += " WHERE jediTaskID=:jediTaskID "
            self.cur.execute(sqlUE + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            _logger.debug(f"{methodName} : done")
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False

    # check input file status
    def checkInputFileStatusInJEDI(self, jobSpec, useCommit=True, withLock=False):
        comment = " /* DBProxy.checkInputFileStatusInJEDI */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <PandaID={jobSpec.PandaID}>"
        tmpLog = LogWrapper(_logger, methodName, monToken=f"<PandaID={jobSpec.PandaID}>")
        tmpLog.debug("start")
        try:
            # only JEDI
            if jobSpec.lockedby != "jedi":
                return True
            # sql to check file status
            sqlFileStat = "SELECT status,attemptNr,keepTrack,is_waiting FROM ATLAS_PANDA.JEDI_Dataset_Contents "
            sqlFileStat += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            if withLock:
                sqlFileStat += "FOR UPDATE NOWAIT "
            # begin transaction
            if useCommit:
                self.conn.begin()
            # get dataset
            sqlPD = "SELECT datasetID FROM ATLAS_PANDA.JEDI_Datasets "
            sqlPD += "WHERE jediTaskID=:jediTaskID AND type IN (:type1,:type2) AND masterID IS NULL "
            varMap = {}
            varMap[":jediTaskID"] = jobSpec.jediTaskID
            varMap[":type1"] = "input"
            varMap[":type2"] = "pseudo_input"
            self.cur.execute(sqlPD + comment, varMap)
            resPD = self.cur.fetchone()
            if resPD is not None:
                (datasetID,) = resPD
            else:
                datasetID = None
            # make pseudo files for dynamic number of events
            if EventServiceUtils.isDynNumEventsSH(jobSpec.specialHandling):
                pseudoFiles = self.create_pseudo_files_for_dyn_num_events(jobSpec, tmpLog)
            else:
                pseudoFiles = []
            # loop over all input files
            allOK = True
            for fileSpec in jobSpec.Files + pseudoFiles:
                if datasetID is None:
                    continue
                # only input file
                if jobSpec.processingType != "pmerge":
                    if fileSpec.datasetID != datasetID:
                        continue
                else:
                    if fileSpec.type != "input":
                        continue
                # skip if not normal JEDI files
                if fileSpec.fileID == "NULL":
                    continue
                varMap = {}
                varMap[":jediTaskID"] = fileSpec.jediTaskID
                varMap[":datasetID"] = fileSpec.datasetID
                varMap[":fileID"] = fileSpec.fileID
                self.cur.execute(sqlFileStat + comment, varMap)
                resFileStat = self.cur.fetchone()
                if resFileStat is None:
                    tmpLog.debug(f"jediTaskID={fileSpec.jediTaskID} datasetID={fileSpec.datasetID} fileID={fileSpec.fileID} is not found")
                    allOK = False
                    break
                else:
                    fileStatus, attemptNr, keepTrack, is_waiting = resFileStat
                    if attemptNr is None:
                        continue
                    if keepTrack != 1:
                        continue
                    if attemptNr != fileSpec.attemptNr:
                        tmpLog.debug(
                            "jediTaskID={0} datasetID={1} fileID={2} attemptNr={3} is inconsitent with attemptNr={4} in JEDI".format(
                                fileSpec.jediTaskID,
                                fileSpec.datasetID,
                                fileSpec.fileID,
                                fileSpec.attemptNr,
                                attemptNr,
                            )
                        )
                        allOK = False
                        break
                    if fileStatus in ["finished"] or (
                        fileStatus not in ["running"] and jobSpec.computingSite != EventServiceUtils.siteIdForWaitingCoJumboJobs and is_waiting is None
                    ):
                        tmpLog.debug(
                            "jediTaskID={0} datasetID={1} fileID={2} attemptNr={3} is in wrong status ({4}) in JEDI".format(
                                fileSpec.jediTaskID,
                                fileSpec.datasetID,
                                fileSpec.fileID,
                                fileSpec.attemptNr,
                                fileStatus,
                            )
                        )
                        allOK = False
                        break
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError("Commit error")
            tmpLog.debug(f"done with {allOK} for processingType={jobSpec.processingType}")
            return allOK
        except Exception:
            if useCommit:
                # roll back
                self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None

    # increase memory limit
    def increaseRamLimitJEDI(self, jediTaskID, jobRamCount, noLimits=False):
        comment = " /* DBProxy.increaseRamLimitJEDI */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID}>"
        _logger.debug(f"{methodName} : start")
        try:
            # RAM limit
            limitList = [1000, 2000, 3000, 4000, 6000, 8000]
            # begin transaction
            self.conn.begin()
            # get current limit
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            sqlUE = f"SELECT ramCount FROM {panda_config.schemaJEDI}.JEDI_Tasks "
            sqlUE += "WHERE jediTaskID=:jediTaskID "
            self.cur.execute(sqlUE + comment, varMap)
            (taskRamCount,) = self.cur.fetchone()
            _logger.debug(f"{methodName} : RAM limit task={taskRamCount} job={jobRamCount}")

            increased = False

            # skip if already increased or largest limit
            if taskRamCount > jobRamCount:
                dbgStr = f"no change since task RAM limit ({taskRamCount}) is larger than job limit ({jobRamCount})"
                _logger.debug(f"{methodName} : {dbgStr}")
            elif taskRamCount >= limitList[-1] and not noLimits:
                dbgStr = "no change "
                dbgStr += f"since task RAM limit ({taskRamCount}) is larger than or equal to the highest limit ({limitList[-1]})"
                _logger.debug(f"{methodName} : {dbgStr}")
            else:
                increased = True
                limit = max(taskRamCount, jobRamCount)
                for nextLimit in limitList:
                    if limit < nextLimit:
                        break
                # if there are no limits
                if limit > nextLimit and noLimits:
                    nextLimit = limit

                # update RAM limit
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":ramCount"] = nextLimit
                sqlRL = f"UPDATE {panda_config.schemaJEDI}.JEDI_Tasks "
                sqlRL += "SET ramCount=:ramCount "
                sqlRL += "WHERE jediTaskID=:jediTaskID "
                self.cur.execute(sqlRL + comment, varMap)
                _logger.debug(f"{methodName} : increased RAM limit to {nextLimit} from {taskRamCount}")
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")

            # reset the tasks resource type, since it could have jumped to HIMEM
            if increased:
                try:
                    self.reset_resource_type_task(jediTaskID)
                except Exception:
                    _logger.error(f"reset_resource_type excepted with {traceback.format_exc()}")

            _logger.debug(f"{methodName} : done")
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False

    # increase memory limit
    def increaseRamLimitJobJEDI(self, job, job_ram_count, jedi_task_id):
        """Note that this function only increases the min RAM count for the job,
        not for the entire task (for the latter use increaseRamLimitJEDI)
        """
        comment = " /* DBProxy.increaseRamLimitJobJEDI */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        method_name += f" <PanDAID={job.PandaID}>"
        tmp_logger = LogWrapper(_logger, method_name)

        tmp_logger.debug("start")

        # RAM limit
        limit_list = [1000, 2000, 3000, 4000, 6000, 8000]
        # Files defined as input types
        input_types = ("input", "pseudo_input", "pp_input", "trn_log", "trn_output")

        try:
            # if there is no task associated to the job, don't take any action
            if job.jediTaskID in [None, 0, "NULL"]:
                tmp_logger.debug(f"Done. No task({job.jediTaskID}) associated to job({job.PandaID}). Skipping")
                return True

            # get current task ram info
            var_map = {":jediTaskID": jedi_task_id}
            sql_get_ram_task = f"SELECT ramCount, ramUnit, baseRamCount FROM {panda_config.schemaJEDI}.JEDI_Tasks "
            sql_get_ram_task += "WHERE jediTaskID=:jediTaskID "
            self.cur.execute(sql_get_ram_task + comment, var_map)
            task_ram_count, task_ram_unit, task_base_ram_count = self.cur.fetchone()

            if task_base_ram_count in [0, None, "NULL"]:
                task_base_ram_count = 0

            core_count = job.coreCount

            if core_count in [0, None, "NULL"]:
                core_count = 1

            # roll back the memory compensation of the job
            job_ram_count = JobUtils.decompensate_ram_count(job_ram_count)

            tmp_logger.debug(
                f"RAM limit task={task_ram_count}{task_ram_unit} cores={core_count} baseRamCount={task_base_ram_count} job={job_ram_count}{job.minRamUnit}"
            )

            # If more than x% of the task's jobs needed a memory increase, increase the task's memory instead
            var_map = {":jediTaskID": jedi_task_id}
            i = 0
            for input_type in input_types:
                var_map[f":type{i}"] = input_type
                i += 1
            input_type_bindings = ",".join(f":type{i}" for i in range(len(input_types)))

            sql_get_memory_stats = (
                f"SELECT ramCount, count(*) "
                f"FROM {panda_config.schemaJEDI}.JEDI_Datasets tabD, {panda_config.schemaJEDI}.JEDI_Dataset_Contents tabC "
                f"WHERE tabD.jediTaskID=tabC.jediTaskID AND tabD.datasetID=tabC.datasetID AND tabD.jediTaskID=:jediTaskID "
                f"AND tabD.type IN ({input_type_bindings}) AND tabD.masterID IS NULL GROUP BY ramCount"
            )

            self.cur.execute(sql_get_memory_stats + comment, var_map)
            memory_stats = self.cur.fetchall()
            total = sum([entry[1] for entry in memory_stats])
            above_task = sum(tuple[1] for tuple in filter(lambda entry: entry[0] > task_ram_count, memory_stats))
            max_task = max([entry[0] for entry in memory_stats])
            tmp_logger.debug(f"Current task statistics: #increased_files: {above_task}; #total_files: {total}")

            # normalize the job ram-count by base ram count and number of cores
            try:
                normalized_job_ram_count = (job_ram_count - task_base_ram_count) * 1.0
                if task_ram_unit in [
                    "MBPerCore",
                    "MBPerCoreFixed",
                ] and job.minRamUnit in ("MB", None, "NULL"):
                    normalized_job_ram_count = normalized_job_ram_count / core_count
            except TypeError:
                normalized_job_ram_count = 0

            # increase task limit in case >30% of the jobs were increased and the task is not fixed
            if task_ram_unit != "MBPerCoreFixed" and (1.0 * above_task) / total > 0.3:
                minimum_ram = 0
                if normalized_job_ram_count and normalized_job_ram_count > minimum_ram:
                    minimum_ram = normalized_job_ram_count
                if max_task > minimum_ram:
                    minimum_ram = max_task - 1  # otherwise we go over the max_task step

                if minimum_ram:
                    tmp_logger.debug(f"calling increaseRamLimitJEDI with minimum_ram {minimum_ram}")
                    return self.increaseRamLimitJEDI(jedi_task_id, minimum_ram)

            # skip if already at largest limit
            if normalized_job_ram_count >= limit_list[-1]:
                tmp_logger.debug(
                    f"Done. No change since job RAM limit ({normalized_job_ram_count}) " f"is larger than or equal to the highest limit ({limit_list[-1]})"
                )
                return True

            # look for the next limit in the list above the current RAM count
            for next_limit in limit_list:
                if normalized_job_ram_count < next_limit:
                    break

            # task ram-count could already have been increased higher than the next limit. In this case don't do anything
            if task_ram_count > next_limit:
                tmp_logger.debug(f"Done. Task ram count ({task_ram_count}) has been increased and is larger than the next limit ({next_limit})")
                return True

            # update RAM limit
            var_map = {":jediTaskID": job.jediTaskID, ":ramCount": next_limit}
            input_files = filter(lambda panda_file: panda_file.type in input_types, job.Files)
            input_tuples = [(input_file.datasetID, input_file.fileID, input_file.attemptNr) for input_file in input_files]

            for entry in input_tuples:
                dataset_id, file_id, attempt_nr = entry
                var_map[":datasetID"] = dataset_id
                var_map[":fileID"] = file_id

                sql_get_update_ram_job = (
                    f"UPDATE {panda_config.schemaJEDI}.JEDI_Dataset_Contents SET ramCount=:ramCount "
                    f"WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID AND ramCount<:ramCount "
                )

                self.cur.execute(sql_get_update_ram_job + comment, var_map)
                tmp_logger.debug(
                    f"increased RAM limit to {next_limit} from {normalized_job_ram_count} for PandaID {job.PandaID} "
                    f"fileID {file_id} attemptNr {attempt_nr} jediTaskID {job.jediTaskID} datasetID {dataset_id}"
                )

            if not self._commit():
                raise RuntimeError("Commit error")

            tmp_logger.debug("Done")
            return True
        except Exception:
            self._rollback()
            self.dumpErrorMessage(_logger, method_name)
            return False

    # increase memory limit xtimes
    def increaseRamLimitJobJEDI_xtimes(self, job, jobRamCount, jediTaskID, attemptNr):
        """Note that this function only increases the min RAM count for the job,
        not for the entire task (for the latter use increaseRamLimitJEDI)
        """
        comment = " /* DBProxy.increaseRamLimitJobJEDI_xtimes */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <PanDAID={job.PandaID}>"
        _logger.debug(f"{methodName} : start")

        # Files defined as input types
        input_types = ("input", "pseudo_input", "pp_input", "trn_log", "trn_output")

        try:
            # If no task associated to job don't take any action
            if job.jediTaskID in [None, 0, "NULL"]:
                _logger.debug(f"No task({job.jediTaskID}) associated to job({job.PandaID}). Skipping increase of RAM limit xtimes")
            else:
                # get current task Ram info
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                sqlUE = f"SELECT ramCount, ramUnit, baseRamCount, splitRule FROM {panda_config.schemaJEDI}.JEDI_Tasks "
                sqlUE += "WHERE jediTaskID=:jediTaskID "
                self.cur.execute(sqlUE + comment, varMap)
                taskRamCount, taskRamUnit, taskBaseRamCount, splitRule = self.cur.fetchone()

                if taskBaseRamCount in [0, None, "NULL"]:
                    taskBaseRamCount = 0

                coreCount = job.coreCount

                if coreCount in [0, None, "NULL"]:
                    coreCount = 1

                if splitRule is None:
                    items = []
                else:
                    items = splitRule.split(",")

                # set default value
                retryRamOffset = 0
                retryRamStep = 1.0
                # set values from task
                for tmpItem in items:
                    if tmpItem.startswith("RX="):
                        retryRamOffset = int(tmpItem.replace("RX=", ""))
                    if tmpItem.startswith("RY="):
                        retryRamStep = float(tmpItem.replace("RY=", ""))

                _logger.debug(
                    "{0} : RAM limit task={1}{2} cores={3} baseRamCount={4} job={5}{6} jobPSS={7}kB retryRamOffset={8} retryRamStep={9} attemptNr={10}".format(
                        methodName,
                        taskRamCount,
                        taskRamUnit,
                        coreCount,
                        taskBaseRamCount,
                        jobRamCount,
                        job.minRamUnit,
                        job.maxPSS,
                        retryRamOffset,
                        retryRamStep,
                        attemptNr,
                    )
                )

                multiplier = retryRamStep * 1.0 / taskRamCount
                minimumRam = retryRamOffset + taskRamCount * (multiplier**attemptNr)

                if taskRamUnit != "MBPerCoreFixed":
                    # If more than x% of the task's jobs needed a memory increase, increase the task's memory instead
                    varMap = {}
                    varMap[":jediTaskID"] = jediTaskID
                    i = 0
                    for input_type in input_types:
                        varMap[f":type{i}"] = input_type
                        i += 1
                    input_type_bindings = ",".join(f":type{i}" for i in range(len(input_types)))

                    sqlMS = """
                             SELECT ramCount, count(*)
                             FROM {0}.JEDI_Datasets tabD,{0}.JEDI_Dataset_Contents tabC
                             WHERE tabD.jediTaskID=tabC.jediTaskID
                             AND tabD.datasetID=tabC.datasetID
                             AND tabD.jediTaskID=:jediTaskID
                             AND tabD.type IN ({1})
                             AND tabD.masterID IS NULL
                             GROUP BY ramCount
                             """.format(
                        panda_config.schemaJEDI, input_type_bindings
                    )

                    self.cur.execute(sqlMS + comment, varMap)
                    memory_stats = self.cur.fetchall()
                    total = sum([entry[1] for entry in memory_stats])
                    above_task = sum(tuple[1] for tuple in filter(lambda entry: entry[0] > taskRamCount, memory_stats))
                    # max_task = max([entry[0] for entry in memory_stats])
                    _logger.debug(f"{methodName} : #increased_files: {above_task}; #total_files: {total}")

                    # increase task limit in case >30% of the jobs were increased and the task is not fixed
                    if taskRamUnit != "MBPerCoreFixed" and (1.0 * above_task) / total > 0.3:
                        if minimumRam and minimumRam > taskRamCount:
                            _logger.debug(f"{methodName} : calling increaseRamLimitJEDI with minimumRam {minimumRam}")
                            return self.increaseRamLimitJEDI(jediTaskID, minimumRam, noLimits=True)

                # Ops could have increased task RamCount through direct DB access. In this case don't do anything
                if taskRamCount > minimumRam:
                    _logger.debug(f"{methodName} : task ramcount has already been increased and is higher than minimumRam. Skipping")
                    return True

                # skip if already at largest limit
                if jobRamCount >= minimumRam:
                    _logger.debug(f"{methodName} : job ramcount is larger than minimumRam. Skipping")
                    return True
                else:
                    nextLimit = minimumRam

                    # update RAM limit
                    varMap = {}
                    varMap[":jediTaskID"] = job.jediTaskID
                    varMap[":ramCount"] = nextLimit
                    input_files = filter(lambda pandafile: pandafile.type in input_types, job.Files)
                    input_tuples = [(input_file.datasetID, input_file.fileID, input_file.attemptNr) for input_file in input_files]

                    for entry in input_tuples:
                        datasetID, fileId, attemptNr = entry
                        varMap[":datasetID"] = datasetID
                        varMap[":fileID"] = fileId

                        sqlRL = f"UPDATE {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
                        sqlRL += "SET ramCount=:ramCount "
                        sqlRL += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                        sqlRL += "AND ramCount<:ramCount "

                        self.cur.execute(sqlRL + comment, varMap)
                        _logger.debug(
                            "{0} : increased RAM limit to {1} from {2} for PandaID {3} fileID {4} attemptNr {5} jediTaskID {6} datasetID {7}".format(
                                methodName,
                                nextLimit,
                                jobRamCount,
                                job.PandaID,
                                fileId,
                                attemptNr,
                                job.jediTaskID,
                                datasetID,
                            )
                        )
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")

            _logger.debug(f"{methodName} : done")
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False

    # reduce input per job
    def reduce_input_per_job(self, panda_id, jedi_task_id, attempt_nr, excluded_rules, steps, dry_mode):
        comment = " /* DBProxy.reduce_input_per_job */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name + f" < PandaID={panda_id} jediTaskID={jedi_task_id} attemptNr={attempt_nr} >")
        tmp_log.debug("start")
        try:
            # rules to skip action when they are set
            if not excluded_rules:
                excluded_rules = ["nEventsPerJob", "nFilesPerJob"]
            else:
                excluded_rules = excluded_rules.split(",")

            # thresholds with attempt numbers to trigger actions
            if not steps:
                threshold_low = 2
                threshold_middle = 4
                threshold_high = 7
            else:
                threshold_low, threshold_middle, threshold_high = [int(s) for s in steps.split(",")]

            # if no task associated to job don't take any action
            if jedi_task_id in [None, 0, "NULL"]:
                msg_str = "skipping since no task associated to job"
                tmp_log.debug(msg_str)
                return False, msg_str

            # check attempt number
            if attempt_nr < threshold_low:
                msg_str = f"skipping since not enough attempts ({attempt_nr} < {threshold_low}) have been made"
                tmp_log.debug(msg_str)
                return False, msg_str

            # get current split rules
            var_map = {":jediTaskID": jedi_task_id}
            sql_gr = f"SELECT splitRule FROM {panda_config.schemaJEDI}.JEDI_Tasks "
            sql_gr += "WHERE jediTaskID=:jediTaskID "
            self.cur.execute(sql_gr + comment, var_map)
            (split_rule,) = self.cur.fetchone()

            # extract split rule values
            rule_values = task_split_rules.extract_rule_values(
                split_rule, ["nEventsPerJob", "nFilesPerJob", "nGBPerJob", "nMaxFilesPerJob", "retryModuleRules"]
            )

            # no action if num events or files per job is specified
            for rule_name in excluded_rules:
                if rule_values[rule_name]:
                    msg_str = f"skipping since task uses {rule_name}"
                    tmp_log.debug(msg_str)
                    return False, msg_str

            # current max number of files or gigabytes per job
            current_max_files_per_job = rule_values["nMaxFilesPerJob"]
            if current_max_files_per_job:
                current_max_files_per_job = int(current_max_files_per_job)
            current_gigabytes_per_job = rule_values["nGBPerJob"]
            if current_gigabytes_per_job:
                current_gigabytes_per_job = int(current_gigabytes_per_job)

            # initial max number of files or gigabytes per job for retry module
            rules_for_retry_module = rule_values["retryModuleRules"]
            rule_values_for_retry_module = task_split_rules.extract_rule_values(rules_for_retry_module, ["nGBPerJob", "nMaxFilesPerJob"], is_sub_rule=True)
            init_gigabytes_per_job = rule_values_for_retry_module["nGBPerJob"]
            init_max_files_per_job = rule_values_for_retry_module["nMaxFilesPerJob"]

            # set initial values for the first action
            set_init_rules = False
            if not init_gigabytes_per_job:
                set_init_rules = True
                if current_gigabytes_per_job:
                    init_gigabytes_per_job = current_gigabytes_per_job
                else:
                    # use current job size as initial gigabytes per job for retry module
                    var_map = {":PandaID": panda_id}
                    sql_fz = f"SELECT SUM(fsize) FROM {panda_config.schemaPANDA}.filesTable4 "
                    sql_fz += "WHERE PandaID=:PandaID "
                    self.cur.execute(sql_fz + comment, var_map)
                    (init_gigabytes_per_job,) = self.cur.fetchone()
                    init_gigabytes_per_job = math.ceil(init_gigabytes_per_job / 1024 / 1024 / 1024)
            if not init_max_files_per_job:
                set_init_rules = True
                if current_max_files_per_job:
                    init_max_files_per_job = current_max_files_per_job
                else:
                    # use current job size as initial max number of files per job for retry module
                    var_map = {":PandaID": panda_id, ":jediTaskID": jedi_task_id, ":type1": "input", ":type2": "pseudo_input"}
                    sql_fc = f"SELECT COUNT(*) FROM {panda_config.schemaPANDA}.filesTable4 tabF, {panda_config.schemaJEDI}.JEDI_Datasets tabD "
                    sql_fc += (
                        "WHERE tabD.jediTaskID=:jediTaskID AND tabD.type IN (:type1, :type2) AND tabD.masterID IS NULL "
                        "AND tabF.PandaID=:PandaID AND tabF.datasetID=tabD.datasetID "
                    )
                    self.cur.execute(sql_fc + comment, var_map)
                    (init_max_files_per_job,) = self.cur.fetchone()

            # set target based on attempt number
            if attempt_nr < threshold_middle:
                target_gigabytes_per_job = math.floor(init_gigabytes_per_job / 2)
                target_max_files_per_job = math.floor(init_max_files_per_job / 2)
            elif attempt_nr < threshold_high:
                target_gigabytes_per_job = math.floor(init_gigabytes_per_job / 4)
                target_max_files_per_job = math.floor(init_max_files_per_job / 4)
            else:
                target_gigabytes_per_job = 1
                target_max_files_per_job = 1
            target_gigabytes_per_job = max(1, target_gigabytes_per_job)
            target_max_files_per_job = max(1, target_max_files_per_job)

            # update rules when initial values were unset or new values need to be set
            if set_init_rules or current_gigabytes_per_job != target_gigabytes_per_job or current_max_files_per_job != target_max_files_per_job:
                msg_str = "update splitRule: "
                if set_init_rules:
                    msg_str += f"initial nGBPerJob={init_gigabytes_per_job} nMaxFilesPerJob={init_max_files_per_job}. "
                    rules_for_retry_module = task_split_rules.replace_rule(rules_for_retry_module, "nGBPerJob", init_gigabytes_per_job, is_sub_rule=True)
                    rules_for_retry_module = task_split_rules.replace_rule(rules_for_retry_module, "nMaxFilesPerJob", init_max_files_per_job, is_sub_rule=True)
                    if not dry_mode:
                        self.changeTaskSplitRulePanda(
                            jedi_task_id, task_split_rules.split_rule_dict["retryModuleRules"], rules_for_retry_module, useCommit=False, sendLog=True
                        )
                if current_gigabytes_per_job != target_gigabytes_per_job:
                    msg_str += f"new nGBPerJob {current_gigabytes_per_job} -> {target_gigabytes_per_job}. "
                    if not dry_mode:
                        self.changeTaskSplitRulePanda(
                            jedi_task_id, task_split_rules.split_rule_dict["nGBPerJob"], target_gigabytes_per_job, useCommit=False, sendLog=True
                        )
                if current_max_files_per_job != target_max_files_per_job:
                    msg_str += f"new nMaxFilesPerJob {current_max_files_per_job} -> {target_max_files_per_job}. "
                    if not dry_mode:
                        self.changeTaskSplitRulePanda(
                            jedi_task_id, task_split_rules.split_rule_dict["nMaxFilesPerJob"], target_max_files_per_job, useCommit=False, sendLog=True
                        )
                tmp_log.debug(msg_str)
                # commit
                if not dry_mode and not self._commit():
                    raise RuntimeError("Commit error")
                return True, msg_str

            msg_str = "not applicable"
            _logger.debug(msg_str)
            return False, msg_str
        except Exception:
            # roll back
            if not dry_mode:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger, method_name)
            return None, "failed"

    # reset files in JEDI
    def resetFileStatusInJEDI(self, dn, prodManager, datasetName, lostFiles, recoverParent, simul):
        comment = " /* DBProxy.resetFileStatusInJEDI */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <datasetName={datasetName}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            # list of lost input files
            lostInputFiles = {}
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ["", "NULL", None]:
                compactDN = dn
            tmpLog.debug(f"userName={compactDN}")
            toSkip = False
            # begin transaction
            self.conn.begin()
            # get jediTaskID
            varMap = {}
            varMap[":type1"] = "log"
            varMap[":type2"] = "output"
            varMap[":name1"] = datasetName
            varMap[":name2"] = datasetName.split(":")[-1]
            sqlGI = f"SELECT jediTaskID,datasetID FROM {panda_config.schemaJEDI}.JEDI_Datasets "
            sqlGI += "WHERE type IN (:type1,:type2) AND datasetName IN (:name1,:name2) "
            self.cur.execute(sqlGI + comment, varMap)
            resGI = self.cur.fetchall()
            # use the largest datasetID since broken tasks might have been retried
            jediTaskID = None
            datasetID = None
            for tmpJediTaskID, tmpDatasetID in resGI:
                if jediTaskID is None or jediTaskID < tmpJediTaskID:
                    jediTaskID = tmpJediTaskID
                    datasetID = tmpDatasetID
                elif datasetID < tmpDatasetID:
                    datasetID = tmpDatasetID
            if jediTaskID is None:
                tmpLog.debug("jediTaskID not found")
                toSkip = True
            if not toSkip:
                # get task status and owner
                tmpLog.debug(f"jediTaskID={jediTaskID} datasetID={datasetID}")
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                sqlOW = f"SELECT status,userName,useJumbo FROM {panda_config.schemaJEDI}.JEDI_Tasks "
                sqlOW += "WHERE jediTaskID=:jediTaskID "
                self.cur.execute(sqlOW + comment, varMap)
                resOW = self.cur.fetchone()
                taskStatus, ownerName, useJumbo = resOW
                # check ownership
                if not prodManager and ownerName != compactDN:
                    tmpLog.debug(f"not the owner = {ownerName}")
                    toSkip = True
            if not toSkip:
                # get affected PandaIDs
                sqlLP = f"SELECT pandaID FROM {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
                sqlLP += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND lfn=:lfn "
                # sql to update file status
                sqlUFO = f"UPDATE {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
                sqlUFO += "SET status=:newStatus "
                sqlUFO += "WHERE jediTaskID=:jediTaskID AND type=:type AND status=:oldStatus AND PandaID=:PandaID "
                # sql to cancel events
                sqlCE = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                sqlCE += f"{panda_config.schemaJEDI}.JEDI_Events tab "
                sqlCE += "SET status=:status "
                sqlCE += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlCE += "AND status IN (:esFinished,:esDone,:esMerged) "
                # get affected PandaIDs
                lostPandaIDs = set([])
                nDiff = 0
                for lostFile in lostFiles:
                    varMap = {}
                    varMap[":jediTaskID"] = jediTaskID
                    varMap[":datasetID"] = datasetID
                    varMap[":lfn"] = lostFile
                    self.cur.execute(sqlLP + comment, varMap)
                    resLP = self.cur.fetchone()
                    if resLP is not None:
                        (pandaID,) = resLP
                        lostPandaIDs.add(pandaID)
                        # update the file and coproduced files to lost
                        varMap = {}
                        varMap[":jediTaskID"] = jediTaskID
                        varMap[":PandaID"] = pandaID
                        varMap[":type"] = "output"
                        varMap[":newStatus"] = "lost"
                        varMap[":oldStatus"] = "finished"
                        if not simul:
                            self.cur.execute(sqlUFO + comment, varMap)
                            nRow = self.cur.rowcount
                            if nRow > 0:
                                nDiff += 1
                        else:
                            tmpLog.debug(sqlUFO + comment + str(varMap))
                            nDiff += 1
                # update output dataset statistics
                sqlUDO = f"UPDATE {panda_config.schemaJEDI}.JEDI_Datasets "
                sqlUDO += "SET nFilesFinished=nFilesFinished-:nDiff "
                sqlUDO += "WHERE jediTaskID=:jediTaskID AND type=:type "
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":type"] = "output"
                varMap[":nDiff"] = nDiff
                tmpLog.debug(sqlUDO + comment + str(varMap))
                if not simul:
                    self.cur.execute(sqlUDO + comment, varMap)
                # get nEvents
                sqlGNE = "SELECT SUM(c.nEvents),c.datasetID "
                sqlGNE += "FROM {0}.JEDI_Datasets d,{0}.JEDI_Dataset_Contents c ".format(panda_config.schemaJEDI)
                sqlGNE += "WHERE c.jediTaskID=d.jediTaskID AND c.datasetID=d.datasetID "
                sqlGNE += "AND d.jediTaskID=:jediTaskID AND d.type=:type AND c.status=:status "
                sqlGNE += "GROUP BY c.datasetID "
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":type"] = "output"
                varMap[":status"] = "finished"
                self.cur.execute(sqlGNE + comment, varMap)
                resGNE = self.cur.fetchall()
                # update nEvents
                sqlUNE = f"UPDATE {panda_config.schemaJEDI}.JEDI_Datasets "
                sqlUNE += "SET nEvents=:nEvents "
                sqlUNE += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                for tmpCount, tmpDatasetID in resGNE:
                    varMap = {}
                    varMap[":jediTaskID"] = jediTaskID
                    varMap[":datasetID"] = tmpDatasetID
                    varMap[":nEvents"] = tmpCount
                    if not simul:
                        self.cur.execute(sqlUNE + comment, varMap)
                        tmpLog.debug(sqlUNE + comment + str(varMap))
                # get input datasets
                sqlID = f"SELECT datasetID,datasetName,masterID FROM {panda_config.schemaJEDI}.JEDI_Datasets "
                sqlID += "WHERE jediTaskID=:jediTaskID AND type=:type "
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":type"] = "input"
                self.cur.execute(sqlID + comment, varMap)
                resID = self.cur.fetchall()
                inputDatasets = {}
                masterID = None
                for tmpDatasetID, tmpDatasetName, tmpMasterID in resID:
                    inputDatasets[tmpDatasetID] = tmpDatasetName
                    if tmpMasterID is None:
                        masterID = tmpDatasetID
                # sql to get affected inputs
                if useJumbo is None:
                    sqlAI = f"SELECT fileID,datasetID,lfn,outPandaID FROM {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
                    sqlAI += "WHERE jediTaskID=:jediTaskID AND type IN (:type1,:type2,:type3) AND PandaID=:PandaID "
                else:
                    sqlAI = f"SELECT fileID,datasetID,lfn,NULL FROM {panda_config.schemaPANDA}.filesTable4 "
                    sqlAI += "WHERE PandaID=:PandaID AND type IN (:type1,:type2) "
                    sqlAI += "UNION "
                    sqlAI = f"SELECT fileID,datasetID,lfn,NULL FROM {panda_config.schemaPANDA}.filesTable4 "
                    sqlAI += "WHERE PandaID=:PandaID AND type IN (:type1,:type2) AND modificationTime>CURRENT_TIMESTAMP-365 "
                # sql to update input file status
                sqlUFI = f"UPDATE {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
                sqlUFI += "SET status=:newStatus,attemptNr=attemptNr+1 "
                sqlUFI += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID AND status=:oldStatus "
                # get affected inputs
                datasetCountMap = {}
                for lostPandaID in lostPandaIDs:
                    varMap = {}
                    if useJumbo is None:
                        varMap[":jediTaskID"] = jediTaskID
                        varMap[":PandaID"] = lostPandaID
                        varMap[":type1"] = "input"
                        varMap[":type2"] = "pseudo_input"
                        varMap[":type3"] = "output"
                    else:
                        varMap[":PandaID"] = lostPandaID
                        varMap[":type1"] = "input"
                        varMap[":type2"] = "pseudo_input"
                    self.cur.execute(sqlAI + comment, varMap)
                    resAI = self.cur.fetchall()
                    newResAI = []
                    for tmpItem in resAI:
                        tmpFileID, tmpDatasetID, tmpLFN, tmpOutPandaID = tmpItem
                        # skip output file
                        if lostPandaID == tmpOutPandaID:
                            continue
                        # input for merged files
                        if tmpOutPandaID is not None:
                            varMap = {}
                            varMap[":jediTaskID"] = jediTaskID
                            varMap[":PandaID"] = tmpOutPandaID
                            varMap[":type1"] = "input"
                            varMap[":type2"] = "pseudo_input"
                            varMap[":type3"] = "dummy"
                            self.cur.execute(sqlAI + comment, varMap)
                            resAI2 = self.cur.fetchall()
                            for tmpItem in resAI2:
                                newResAI.append(tmpItem)
                        else:
                            newResAI.append(tmpItem)
                    for tmpFileID, tmpDatasetID, tmpLFN, tmpOutPandaID in newResAI:
                        # collect if dataset was already deleted
                        is_lost = False
                        if recoverParent and tmpDatasetID == masterID:
                            lostInputFiles.setdefault(inputDatasets[tmpDatasetID], [])
                            lostInputFiles[inputDatasets[tmpDatasetID]].append(tmpLFN)
                            is_lost = True
                        # reset file status
                        if tmpDatasetID not in datasetCountMap:
                            datasetCountMap[tmpDatasetID] = 0
                        varMap = {}
                        varMap[":jediTaskID"] = jediTaskID
                        varMap[":datasetID"] = tmpDatasetID
                        varMap[":fileID"] = tmpFileID
                        if is_lost:
                            varMap[":newStatus"] = "lost"
                        else:
                            varMap[":newStatus"] = "ready"
                        varMap[":oldStatus"] = "finished"
                        if not simul:
                            self.cur.execute(sqlUFI + comment, varMap)
                            nRow = self.cur.rowcount
                        else:
                            tmpLog.debug(sqlUFI + comment + str(varMap))
                            nRow = 1
                        if nRow > 0:
                            datasetCountMap[tmpDatasetID] += 1
                            if useJumbo is not None:
                                # cancel events
                                varMap = {}
                                varMap[":jediTaskID"] = jediTaskID
                                varMap[":datasetID"] = tmpDatasetID
                                varMap[":fileID"] = tmpFileID
                                varMap[":status"] = EventServiceUtils.ST_cancelled
                                varMap[":esFinished"] = EventServiceUtils.ST_finished
                                varMap[":esDone"] = EventServiceUtils.ST_done
                                varMap[":esMerged"] = EventServiceUtils.ST_merged
                                if not simul:
                                    self.cur.execute(sqlCE + comment, varMap)
                                else:
                                    tmpLog.debug(sqlCE + comment + str(varMap))
                # update dataset statistics
                sqlUDI = f"UPDATE {panda_config.schemaJEDI}.JEDI_Datasets "
                sqlUDI += "SET nFilesUsed=nFilesUsed-:nDiff,nFilesFinished=nFilesFinished-:nDiff,"
                sqlUDI += "nEventsUsed=(SELECT SUM(CASE WHEN startEvent IS NULL THEN nEvents ELSE endEvent-startEvent+1 END) "
                sqlUDI += f"FROM {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
                sqlUDI += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND status=:status) "
                sqlUDI += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                for tmpDatasetID in datasetCountMap:
                    nDiff = datasetCountMap[tmpDatasetID]
                    varMap = {}
                    varMap[":jediTaskID"] = jediTaskID
                    varMap[":datasetID"] = tmpDatasetID
                    varMap[":nDiff"] = nDiff
                    varMap[":status"] = "finished"
                    tmpLog.debug(sqlUDI + comment + str(varMap))
                    if not simul:
                        self.cur.execute(sqlUDI + comment, varMap)
                # update task status
                if taskStatus == "done":
                    sqlUT = f"UPDATE {panda_config.schemaJEDI}.JEDI_Tasks SET status=:newStatus WHERE jediTaskID=:jediTaskID "
                    varMap = {}
                    varMap[":jediTaskID"] = jediTaskID
                    varMap[":newStatus"] = "finished"
                    if not simul:
                        self.cur.execute(sqlUT + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug("done")
            return True, jediTaskID, lostInputFiles
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False, None, None

    # record retry history
    def recordRetryHistoryJEDI(self, jediTaskID, newPandaID, oldPandaIDs, relationType, no_late_bulk_exec=True, extracted_sqls=None):
        comment = " /* DBProxy.recordRetryHistoryJEDI */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <PandaID={newPandaID}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        # sql to check record
        sqlCK = f"SELECT jediTaskID FROM {panda_config.schemaJEDI}.JEDI_Job_Retry_History "
        sqlCK += "WHERE jediTaskID=:jediTaskID AND oldPandaID=:oldPandaID AND newPandaID=:newPandaID AND originPandaID=:originPandaID "
        # sql to insert record
        sqlIN = f"INSERT INTO {panda_config.schemaJEDI}.JEDI_Job_Retry_History "
        if relationType is None:
            sqlIN += "(jediTaskID,oldPandaID,newPandaID,originPandaID) "
            sqlIN += "VALUES(:jediTaskID,:oldPandaID,:newPandaID,:originPandaID) "
        else:
            sqlIN += "(jediTaskID,oldPandaID,newPandaID,originPandaID,relationType) "
            sqlIN += "VALUES(:jediTaskID,:oldPandaID,:newPandaID,:originPandaID,:relationType) "
        for oldPandaID in oldPandaIDs:
            # get origin
            originIDs = self.getOriginPandaIDsJEDI(oldPandaID, jediTaskID, self.cur)
            for originID in originIDs:
                # check
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":oldPandaID"] = oldPandaID
                varMap[":newPandaID"] = newPandaID
                varMap[":originPandaID"] = originID
                self.cur.execute(sqlCK + comment, varMap)
                resCK = self.cur.fetchone()
                # insert
                if resCK is None:
                    varMap = {}
                    varMap[":jediTaskID"] = jediTaskID
                    varMap[":oldPandaID"] = oldPandaID
                    varMap[":newPandaID"] = newPandaID
                    varMap[":originPandaID"] = originID
                    if relationType is not None:
                        varMap[":relationType"] = relationType
                    if no_late_bulk_exec:
                        self.cur.execute(sqlIN + comment, varMap)
                    else:
                        extracted_sqls.setdefault("retry_history", {"sql": sqlIN + comment, "vars": []})
                        extracted_sqls["retry_history"]["vars"].append(varMap)
        # return
        tmpLog.debug("done")

    # copy file records
    def copy_file_records(self, new_lfns, file_spec):
        comment = " /* DBProxy.copy_file_records */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <PandaID={file_spec.PandaID} oldLFN={file_spec.lfn}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug(f"start with {len(new_lfns)} files")
        try:
            # begin transaction
            self.conn.begin()
            for idx_lfn, new_lfn in enumerate(new_lfns):
                # reset rowID
                tmpFileSpec = copy.copy(file_spec)
                tmpFileSpec.lfn = new_lfn
                if idx_lfn > 0:
                    tmpFileSpec.row_ID = None
                # insert file in JEDI
                if idx_lfn > 0 and tmpFileSpec.jediTaskID not in [None, "NULL"] and tmpFileSpec.fileID not in ["", "NULL", None]:
                    # get fileID
                    sqlFileID = "SELECT ATLAS_PANDA.JEDI_DATASET_CONT_FILEID_SEQ.nextval FROM dual "
                    self.cur.execute(sqlFileID + comment)
                    (newFileID,) = self.cur.fetchone()
                    # read file in JEDI
                    varMap = {}
                    varMap[":jediTaskID"] = tmpFileSpec.jediTaskID
                    varMap[":datasetID"] = tmpFileSpec.datasetID
                    varMap[":fileID"] = tmpFileSpec.fileID
                    sqlGI = f"SELECT * FROM {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
                    sqlGI += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                    self.cur.execute(sqlGI + comment, varMap)
                    resGI = self.cur.fetchone()
                    tmpFileSpec.fileID = newFileID
                    if resGI is not None:
                        # make sql and map
                        sqlJI = f"INSERT INTO {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
                        sqlJI += "VALUES ("
                        varMap = {}
                        for columDesc, columVal in zip(self.cur.description, resGI):
                            columName = columDesc[0]
                            # overwrite fileID
                            if columName.upper() == "FILEID":
                                columVal = tmpFileSpec.fileID
                            keyName = f":{columName}"
                            varMap[keyName] = columVal
                            sqlJI += f"{keyName},"
                        sqlJI = sqlJI[:-1]
                        sqlJI += ") "
                        # insert file in JEDI
                        self.cur.execute(sqlJI + comment, varMap)
                if idx_lfn > 0:
                    # insert file in Panda
                    sqlFile = f"INSERT INTO ATLAS_PANDA.filesTable4 ({FileSpec.columnNames()}) "
                    sqlFile += FileSpec.bindValuesExpression(useSeq=True)
                    varMap = tmpFileSpec.valuesMap(useSeq=True)
                    self.cur.execute(sqlFile + comment, varMap)
                else:
                    # update LFN
                    sqlFSF = "UPDATE ATLAS_PANDA.filesTable4 SET lfn=:lfn "
                    sqlFSF += "WHERE row_ID=:row_ID "
                    varMap = {}
                    varMap[":lfn"] = tmpFileSpec.lfn
                    varMap[":row_ID"] = tmpFileSpec.row_ID
                    self.cur.execute(sqlFSF + comment, varMap)
                # update LFN in JEDI
                if tmpFileSpec.fileID not in ["", "NULL", None]:
                    sqlJF = f"UPDATE {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
                    sqlJF += "SET lfn=:lfn "
                    sqlJF += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                    varMap = {}
                    varMap[":lfn"] = tmpFileSpec.lfn
                    varMap[":jediTaskID"] = tmpFileSpec.jediTaskID
                    varMap[":datasetID"] = tmpFileSpec.datasetID
                    varMap[":fileID"] = tmpFileSpec.fileID
                    self.cur.execute(sqlJF + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug("done")
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False

    # get error definitions from DB (values cached for 1 hour)
    @memoize
    def getRetrialRules(self):
        # Logging
        comment = " /* DBProxy.getRetrialRules */"
        method_name = comment.split(" ")[-2].split(".")[-1]

        tmp_logger = LogWrapper(_logger, method_name)
        tmp_logger.debug("start")

        # SQL to extract the error definitions
        sql = """
        SELECT re.retryerror_id, re.errorsource, re.errorcode, re.errorDiag, re.parameters, re.architecture, re.release, re.workqueue_id, ra.retry_action, re.active, ra.active
        FROM ATLAS_PANDA.RETRYERRORS re, ATLAS_PANDA.RETRYACTIONS ra
        WHERE re.retryaction=ra.retryaction_id
        AND (CURRENT_TIMESTAMP < re.expiration_date or re.expiration_date IS NULL)
        """
        self.cur.execute(sql + comment, {})
        definitions = self.cur.fetchall()  # example of output: [('pilotErrorCode', 1, None, None, None, None, 'no_retry', 'Y', 'Y'),...]

        # commit
        if not self._commit():
            raise RuntimeError("Commit error")

        # tmp_logger.debug("definitions %s"%(definitions))

        retrial_rules = {}
        for definition in definitions:
            (
                retryerror_id,
                error_source,
                error_code,
                error_diag,
                parameters,
                architecture,
                release,
                wqid,
                action,
                e_active,
                a_active,
            ) = definition

            # Convert the parameter string into a dictionary
            try:
                # 1. Convert a string like "key1=value1&key2=value2" into [[key1, value1],[key2,value2]]
                params_list = map(
                    lambda key_value_pair: key_value_pair.split("="),
                    parameters.split("&"),
                )
                # 2. Convert a list [[key1, value1],[key2,value2]] into {key1: value1, key2: value2}
                params_dict = dict((key, value) for (key, value) in params_list)
            except AttributeError:
                params_dict = {}
            except ValueError:
                params_dict = {}

            # Calculate if action and error combination should be active
            if e_active == "Y" and a_active == "Y":
                active = True  # Apply the action for this error
            else:
                active = False  # Do not apply the action for this error, only log

            retrial_rules.setdefault(error_source, {})
            retrial_rules[error_source].setdefault(error_code, [])
            retrial_rules[error_source][error_code].append(
                {
                    "error_id": retryerror_id,
                    "error_diag": error_diag,
                    "action": action,
                    "params": params_dict,
                    "architecture": architecture,
                    "release": release,
                    "wqid": wqid,
                    "active": active,
                }
            )
        # tmp_logger.debug("Loaded retrial rules from DB: %s" %retrial_rules)
        return retrial_rules

    def setMaxAttempt(self, jobID, taskID, files, maxAttempt):
        # Logging
        comment = " /* DBProxy.setMaxAttempt */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")

        # Update the file entries to avoid JEDI generating new jobs
        input_types = ("input", "pseudo_input", "pp_input", "trn_log", "trn_output")
        input_files = list(
            filter(
                lambda pandafile: pandafile.type in input_types and re.search("DBRelease", pandafile.lfn) is None,
                files,
            )
        )
        input_fileIDs = [input_file.fileID for input_file in input_files]
        input_datasetIDs = [input_file.datasetID for input_file in input_files]

        if input_fileIDs:
            try:
                # Start transaction
                self.conn.begin()

                varMap = {}
                varMap[":taskID"] = taskID
                varMap[":pandaID"] = jobID

                # Bind the files
                f = 0
                for fileID in input_fileIDs:
                    varMap[f":file{f}"] = fileID
                    f += 1
                file_bindings = ",".join(f":file{i}" for i in range(len(input_fileIDs)))

                # Bind the datasets
                d = 0
                for datasetID in input_datasetIDs:
                    varMap[f":dataset{d}"] = datasetID
                    d += 1
                dataset_bindings = ",".join(f":dataset{i}" for i in range(len(input_fileIDs)))

                # Get the minimum maxAttempt value of the files
                sql_select = f"""
                select min(maxattempt) from ATLAS_PANDA.JEDI_Dataset_Contents
                WHERE JEDITaskID = :taskID
                AND datasetID IN ({dataset_bindings})
                AND fileID IN ({file_bindings})
                AND pandaID = :pandaID
                """
                self.cur.execute(sql_select + comment, varMap)
                try:
                    maxAttempt_select = self.cur.fetchone()[0]
                except (TypeError, IndexError):
                    maxAttempt_select = None

                # Don't update the maxAttempt if the new value is higher than the old value
                if maxAttempt_select and maxAttempt_select > maxAttempt:
                    varMap[":maxAttempt"] = min(maxAttempt, maxAttempt_select)

                    sql_update = f"""
                    UPDATE ATLAS_PANDA.JEDI_Dataset_Contents
                    SET maxAttempt=:maxAttempt
                    WHERE JEDITaskID = :taskID
                    AND datasetID IN ({dataset_bindings})
                    AND fileID IN ({file_bindings})
                    AND pandaID = :pandaID
                    """

                    self.cur.execute(sql_update + comment, varMap)

                # Commit updates
                if not self._commit():
                    raise RuntimeError("Commit error")
            except Exception:
                # roll back
                self._rollback()
                # error
                self.dumpErrorMessage(_logger, methodName)
                return False

        tmpLog.debug("done")
        return True

    def increase_max_attempt(self, job_id, task_id, files):
        """Increase the max attempt number by one for specific files."""
        comment = " /* DBProxy.increase_max_attempt */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")

        # Update the file entries to increase the max attempt number by one
        input_types = ("input", "pseudo_input", "pp_input", "trn_log", "trn_output")
        input_files = [
            pandafile for pandafile in files
            if pandafile.type in input_types and re.search("DBRelease", pandafile.lfn) is None
        ]
        input_file_ids = [input_file.fileID for input_file in input_files]
        input_dataset_ids = [input_file.datasetID for input_file in input_files]

        if input_file_ids:
            try:
                # Start transaction
                self.conn.begin()

                var_map = {
                    ":taskID": task_id,
                    ":pandaID": job_id,
                }

                # Bind the files
                file_bindings = []
                for index, file_id in enumerate(input_file_ids):
                    var_map[f":file{index}"] = file_id
                    file_bindings.append(f":file{index}")
                file_bindings_str = ",".join(file_bindings)

                # Bind the datasets
                dataset_bindings = []
                for index, dataset_id in enumerate(input_dataset_ids):
                    var_map[f":dataset{index}"] = dataset_id
                    dataset_bindings.append(f":dataset{index}")
                dataset_bindings_str = ",".join(dataset_bindings)

                sql_update = f"""
                UPDATE ATLAS_PANDA.JEDI_Dataset_Contents
                SET maxAttempt = maxAttempt + 1
                WHERE JEDITaskID = :taskID
                AND datasetID IN ({dataset_bindings_str})
                AND fileID IN ({file_bindings_str})
                AND pandaID = :pandaID
                """

                self.cur.execute(sql_update + comment, var_map)

                # Commit updates
                if not self._commit():
                    raise RuntimeError("Commit error")

            except Exception:
                # Roll back
                self._rollback()
                # Log error
                self.dumpErrorMessage(_logger, method_name)
                return False

        tmp_log.debug("done")
        return True

    def setNoRetry(self, jobID, taskID, files):
        # Logging
        comment = " /* DBProxy.setNoRetry */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <PandaID={jobID} jediTaskID={taskID}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")

        # Update the file entries to avoid JEDI generating new jobs
        input_types = ("input", "pseudo_input", "pp_input", "trn_log", "trn_output")
        input_files = list(
            filter(
                lambda pandafile: pandafile.type in input_types and re.search("DBRelease", pandafile.lfn) is None,
                files,
            )
        )
        input_fileIDs = [input_file.fileID for input_file in input_files]
        input_datasetIDs = [input_file.datasetID for input_file in input_files]

        if input_fileIDs:
            try:
                # Start transaction
                self.conn.begin()

                # loop over all datasets
                for datasetID in input_datasetIDs:
                    varMap = {}
                    varMap[":taskID"] = taskID
                    varMap[":datasetID"] = datasetID
                    varMap[":keepTrack"] = 1

                    # Bind the files
                    f = 0
                    for fileID in input_fileIDs:
                        varMap[f":file{f}"] = fileID
                        f += 1
                    file_bindings = ",".join(f":file{i}" for i in range(len(input_fileIDs)))

                    sql_update = f"""
                    UPDATE ATLAS_PANDA.JEDI_Dataset_Contents
                    SET maxAttempt=attemptNr
                    WHERE JEDITaskID = :taskID
                    AND datasetID=:datasetID
                    AND fileID IN ({file_bindings})
                    AND maxAttempt IS NOT NULL AND attemptNr IS NOT NULL
                    AND maxAttempt > attemptNr
                    AND (maxFailure IS NULL OR failedAttempt IS NULL OR maxFailure > failedAttempt)
                    AND keepTrack=:keepTrack
                    AND status=:status
                    """

                    # update files in 'running' status. These files do NOT need to be counted for the nFiles*
                    varMap[":status"] = "running"
                    self.cur.execute(sql_update + comment, varMap)

                    # update files in 'ready' status. These files need to be counted for the nFiles*
                    varMap[":status"] = "ready"
                    self.cur.execute(sql_update + comment, varMap)
                    rowcount = self.cur.rowcount

                    # update datasets
                    if rowcount > 0:
                        sql_dataset = "UPDATE ATLAS_PANDA.JEDI_Datasets "
                        sql_dataset += "SET nFilesUsed=nFilesUsed+:nDiff,nFilesFailed=nFilesFailed+:nDiff "
                        sql_dataset += "WHERE jediTaskID=:taskID AND datasetID=:datasetID "
                        varMap = dict()
                        varMap[":taskID"] = taskID
                        varMap[":datasetID"] = datasetID
                        varMap[":nDiff"] = rowcount
                        self.cur.execute(sql_dataset + comment, varMap)

                # Commit updates
                if not self._commit():
                    raise RuntimeError("Commit error")
            except Exception:
                # roll back
                self._rollback()
                # error
                self.dumpErrorMessage(_logger, methodName)
                return False

        tmpLog.debug("done")
        return True

    def increaseCpuTimeTask(self, jobID, taskID, siteid, files, active):
        """
        Increases the CPU time of a task
        walltime = basewalltime + cpuefficiency*CPUTime*nEvents/Corepower/Corecount

        CPU time: execution time per event
        Walltime: time for a job
        Corepower: HS06 score
        Basewalltime: Setup time, time to download, etc. taken by the pilot
        """
        comment = " /* DBProxy.increaseCpuTimeTask */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <PandaID={jobID}; TaskID={taskID}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")

        # 1. Get the site information from schedconfig
        sql = """
        SELECT /* use_json_type */ sc.data.maxtime, sc.data.corepower,
            CASE
                WHEN sc.data.corecount IS NULL THEN 1
                ELSE sc.data.corecount
            END as corecount
        FROM ATLAS_PANDA.schedconfig_json sc
        WHERE sc.panda_queue=:siteid
        """
        varMap = {"siteid": siteid}
        self.cur.execute(sql + comment, varMap)
        siteParameters = self.cur.fetchone()  # example of output: [('pilotErrorCode', 1, None, None, None, None, 'no_retry', 'Y', 'Y'),...]

        if not siteParameters:
            tmpLog.debug(f"No site parameters retrieved for {siteid}")

        (maxtime, corepower, corecount) = siteParameters
        tmpLog.debug(f"siteid {siteid} has parameters: maxtime {maxtime}, corepower {corepower}, corecount {corecount}")
        if (not maxtime) or (not corepower) or (not corecount):
            tmpLog.debug(f"One or more site parameters are not defined for {siteid}... nothing to do")
            return None
        else:
            (maxtime, corepower, corecount) = (
                int(maxtime),
                float(corepower),
                int(corecount),
            )

        # 2. Get the task information
        sql = """
        SELECT jt.cputime, jt.walltime, jt.basewalltime, jt.cpuefficiency, jt.cputimeunit
        FROM ATLAS_PANDA.jedi_tasks jt
        WHERE jt.jeditaskid=:jeditaskid
        """
        varMap = {"jeditaskid": taskID}
        self.cur.execute(sql + comment, varMap)
        taskParameters = self.cur.fetchone()

        if not taskParameters:
            tmpLog.debug(f"No task parameters retrieved for jeditaskid {taskID}... nothing to do")
            return None

        (cputime, walltime, basewalltime, cpuefficiency, cputimeunit) = taskParameters
        if not cpuefficiency or not basewalltime:
            tmpLog.debug(f"CPU efficiency and/or basewalltime are not defined for task {taskID}... nothing to do")
            return None

        tmpLog.debug(
            "task {0} has parameters: cputime {1}, walltime {2}, basewalltime {3}, cpuefficiency {4}, cputimeunit {5}".format(
                taskID, cputime, walltime, basewalltime, cpuefficiency, cputimeunit
            )
        )

        # 2. Get the file information
        input_types = ("input", "pseudo_input", "pp_input", "trn_log", "trn_output")
        input_files = list(
            filter(
                lambda pandafile: pandafile.type in input_types and re.search("DBRelease", pandafile.lfn) is None,
                files,
            )
        )
        input_fileIDs = [input_file.fileID for input_file in input_files]
        input_datasetIDs = [input_file.datasetID for input_file in input_files]

        if input_fileIDs:
            varMap = {}
            varMap[":taskID"] = taskID
            varMap[":pandaID"] = jobID

            # Bind the files
            f = 0
            for fileID in input_fileIDs:
                varMap[f":file{f}"] = fileID
                f += 1
            file_bindings = ",".join(f":file{i}" for i in range(len(input_fileIDs)))

            # Bind the datasets
            d = 0
            for datasetID in input_datasetIDs:
                varMap[f":dataset{d}"] = datasetID
                d += 1
            dataset_bindings = ",".join(f":dataset{i}" for i in range(len(input_fileIDs)))

            sql_select = f"""
            SELECT jdc.fileid, jdc.nevents, jdc.startevent, jdc.endevent
            FROM ATLAS_PANDA.JEDI_Dataset_Contents jdc, ATLAS_PANDA.JEDI_Datasets jd
            WHERE jdc.JEDITaskID = :taskID
            AND jdc.datasetID IN ({dataset_bindings})
            AND jdc.fileID IN ({file_bindings})
            AND jd.datasetID = jdc.datasetID
            AND jd.masterID IS NULL
            AND jdc.pandaID = :pandaID
            """
            self.cur.execute(sql_select + comment, varMap)

            resList = self.cur.fetchall()
            nevents_total = 0
            for fileid, nevents, startevent, endevent in resList:
                tmpLog.debug(f"event information: fileid {fileid}, nevents {nevents}, startevent {startevent}, endevent {endevent}")

                if endevent is not None and startevent is not None:
                    nevents_total += endevent - startevent
                elif nevents:
                    nevents_total += nevents

            if not nevents_total:
                tmpLog.debug(f"nevents could not be calculated for job {jobID}... nothing to do")
                return None
        else:
            tmpLog.debug(f"No input files for job {jobID}, so could not update CPU time for task {taskID}")
            return None

        try:
            new_cputime = (maxtime - basewalltime) * corepower * corecount * 1.1 / (cpuefficiency / 100.0) / nevents_total

            if cputime > new_cputime:
                tmpLog.debug(f"Skipping CPU time increase since old CPU time {cputime} > new CPU time {new_cputime}")
                return None

            if active:  # only run the update if active mode. Otherwise return what would have been done
                sql_update_cputime = """
                UPDATE ATLAS_PANDA.jedi_tasks SET cputime=:cputime
                WHERE jeditaskid=:jeditaskid
                """
                varMap = {}
                varMap[":cputime"] = new_cputime
                varMap[":jeditaskid"] = taskID
                self.conn.begin()
                self.cur.execute(sql_update_cputime + comment, varMap)
                if not self._commit():
                    raise RuntimeError("Commit error")

                tmpLog.debug(f"Successfully updated the task CPU time from {cputime} to {new_cputime}")
            return new_cputime

        except (ZeroDivisionError, TypeError):
            return None

    def requestTaskParameterRecalculation(self, taskID):
        """
        Requests the recalculation of the CPU time of a task:
         1. set the walltimeUnit to NULL and the modificationTime to Now
         2. AtlasProdWatchDog > JediDBProxy.setScoutJobDataToTasks will pick up tasks with walltimeUnit == NULL
            and modificationTime > Now - 24h. This will trigger a recalculation of the task parameters (outDiskCount,
            outDiskUnit, outDiskCount, walltime, walltimeUnit, cpuTime, ioIntensity, ioIntensityUnit, ramCount, ramUnit,
            workDiskCount, workDiskUnit, workDiskCount)
        """
        comment = " /* DBProxy.requestTaskParameterRecalculation */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" TaskID={taskID}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")

        timeNow = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
        timeLimit = timeNow - datetime.timedelta(minutes=30)

        # update the task if it was not already updated in the last 30 minutes (avoid continuous recalculation)
        sql = """
               UPDATE ATLAS_PANDA.jedi_tasks
               SET walltimeUnit=NULL, modificationTime=:timeNow
               WHERE jediTaskId=:taskID AND modificationTime < :timeLimit
               """
        varMap = {"taskID": taskID, "timeNow": timeNow, "timeLimit": timeLimit}
        self.conn.begin()
        self.cur.execute(sql, varMap)

        rowcount = self.cur.rowcount

        if not self._commit():
            raise RuntimeError("Commit error")

        tmpLog.debug("Forced recalculation of CPUTime")
        return rowcount

    # add associate sub datasets for single consumer job
    def getDestDBlocksWithSingleConsumer(self, jediTaskID, PandaID, ngDatasets):
        comment = " /* DBProxy.getDestDBlocksWithSingleConsumer */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID},PandaID={PandaID}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            retMap = {}
            checkedDS = set()
            # sql to get files
            sqlF = "SELECT datasetID,fileID FROM ATLAS_PANDA.JEDI_Events "
            sqlF += "WHERE jediTaskID=:jediTaskID AND PandaID=:PandaID "
            # sql to get PandaIDs
            sqlP = "SELECT distinct PandaID FROM ATLAS_PANDA.filesTable4 "
            sqlP += "WHERE jediTaskID=:jediTaskID ANd datasetID=:datasetID AND fileID=:fileID "
            # sql to get sub datasets
            sqlD = "SELECT destinationDBlock,datasetID FROM ATLAS_PANDA.filesTable4 "
            sqlD += "WHERE PandaID=:PandaID AND type IN (:type1,:type2) "
            # sql to get PandaIDs in merging
            sqlM = "SELECT distinct PandaID FROM ATLAS_PANDA.filesTable4 "
            sqlM += "WHERE jediTaskID=:jediTaskID ANd datasetID=:datasetID AND status=:status "
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            varMap[":PandaID"] = PandaID
            # begin transaction
            self.conn.begin()
            # get files
            self.cur.execute(sqlF + comment, varMap)
            resF = self.cur.fetchall()
            for datasetID, fileID in resF:
                # get parallel jobs
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":datasetID"] = datasetID
                varMap[":fileID"] = fileID
                self.cur.execute(sqlP + comment, varMap)
                resP = self.cur.fetchall()
                for (sPandaID,) in resP:
                    if sPandaID == PandaID:
                        continue
                    # get sub datasets of parallel jobs
                    varMap = {}
                    varMap[":PandaID"] = sPandaID
                    varMap[":type1"] = "output"
                    varMap[":type2"] = "log"
                    self.cur.execute(sqlD + comment, varMap)
                    resD = self.cur.fetchall()
                    subDatasets = []
                    subDatasetID = None
                    for destinationDBlock, datasetID in resD:
                        if destinationDBlock in ngDatasets:
                            continue
                        if destinationDBlock in checkedDS:
                            continue
                        checkedDS.add(destinationDBlock)
                        subDatasets.append(destinationDBlock)
                        subDatasetID = datasetID
                    if subDatasets == []:
                        continue
                    # get merging PandaID which uses sub dataset
                    varMap = {}
                    varMap[":jediTaskID"] = jediTaskID
                    varMap[":datasetID"] = datasetID
                    varMap[":status"] = "merging"
                    self.cur.execute(sqlM + comment, varMap)
                    resM = self.cur.fetchone()
                    if resM is not None:
                        (mPandaID,) = resM
                        retMap[mPandaID] = subDatasets
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"got {len(retMap)} jobs")
            return retMap
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return {}

    # check validity of merge job
    def isValidMergeJob(self, pandaID, jediTaskID):
        comment = " /* DBProxy.isValidMergeJob */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <Panda={pandaID}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            retVal = True
            retMsg = ""
            # sql to check if merge job is active
            sqlJ = "SELECT jobStatus FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID "
            sqlJ += "UNION "
            sqlJ += "SELECT jobStatus FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
            # sql to get input files
            sqlF = "SELECT datasetID,fileID FROM ATLAS_PANDA.filesTable4 "
            sqlF += "WHERE PandaID=:PandaID AND type IN (:type1,:type2) "
            # sql to get PandaIDs for pre-merged jobs
            sqlP = "SELECT outPandaID FROM ATLAS_PANDA.JEDI_Dataset_Contents "
            sqlP += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID AND type<>:type1"
            # sql to check if pre-merge job is active
            sqlC = "SELECT jobStatus FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
            # begin transaction
            self.conn.begin()
            # check if merge job is active
            varMap = {}
            varMap[":PandaID"] = pandaID
            self.cur.execute(sqlJ + comment, varMap)
            resJ = self.cur.fetchone()
            if resJ is None:
                tmpLog.debug("merge job not found")
            else:
                # get input files
                varMap = {}
                varMap[":PandaID"] = pandaID
                varMap[":type1"] = "input"
                varMap[":type2"] = "pseudo_input"
                self.cur.execute(sqlF + comment, varMap)
                resF = self.cur.fetchall()
                firstDatasetID = None
                fileIDsMap = {}
                for datasetID, fileID in resF:
                    if datasetID not in fileIDsMap:
                        fileIDsMap[datasetID] = set()
                    fileIDsMap[datasetID].add(fileID)
                # get PandaIDs for pre-merged jobs
                pandaIDs = set()
                for datasetID in fileIDsMap:
                    fileIDs = fileIDsMap[datasetID]
                    for fileID in fileIDs:
                        varMap = {}
                        varMap[":jediTaskID"] = jediTaskID
                        varMap[":datasetID"] = datasetID
                        varMap[":fileID"] = fileID
                        varMap[":type1"] = "lib"
                        self.cur.execute(sqlP + comment, varMap)
                        resP = self.cur.fetchone()
                        if resP is not None and resP[0] is not None:
                            pandaIDs.add(resP[0])
                    # only files in the first dataset are enough
                    if len(pandaIDs) > 0:
                        break
                # check pre-merge job
                for tmpPandaID in pandaIDs:
                    varMap = {}
                    varMap[":PandaID"] = tmpPandaID
                    self.cur.execute(sqlC + comment, varMap)
                    resC = self.cur.fetchone()
                    if resC is None:
                        # not found
                        tmpLog.debug(f"pre-merge job {tmpPandaID} not found")
                        retVal = False
                        retMsg = tmpPandaID
                        break
                    elif resC[0] != "merging":
                        # not in merging
                        tmpLog.debug("pre-merge job in {0} != merging".format(tmpPandaID, resC[0]))
                        retVal = False
                        retMsg = tmpPandaID
                        break
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"ret={retVal}")
            return retVal, retMsg
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None, ""

    # update related ES jobs when ES-merge job is done
    def updateRelatedEventServiceJobs(self, job, killEvents=False, forceFailed=False):
        comment = " /* DBProxy.updateRelatedEventServiceJobs */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <PandaID={job.PandaID}>"
        tmpLog = LogWrapper(_logger, methodName)
        if forceFailed:
            jobStatus = "failed"
        else:
            jobStatus = job.jobStatus
        if not forceFailed and jobStatus not in ["finished"] and not (killEvents and not job.notDiscardEvents()):
            tmpLog.debug(f"skip jobStatus={jobStatus} killEvents={killEvents} discard={job.notDiscardEvents()}")
            return True
        tmpLog.debug(f"start jobStatus={jobStatus} killEvents={killEvents} discard={job.notDiscardEvents()}")
        try:
            # sql to read range
            sqlRR = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlRR += "distinct PandaID "
            sqlRR += f"FROM {panda_config.schemaJEDI}.JEDI_Events tab "
            sqlRR += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID AND status IN (:es_done,:es_finished,:es_merged) "
            # loop over all files
            esPandaIDs = set()
            for tmpFile in job.Files:
                # only for input
                if tmpFile.type in ["input", "pseudo_input"]:
                    # get ranges
                    if tmpFile.fileID is [None, "NULL"]:
                        continue
                    varMap = {}
                    varMap[":jediTaskID"] = tmpFile.jediTaskID
                    varMap[":datasetID"] = tmpFile.datasetID
                    varMap[":fileID"] = tmpFile.fileID
                    varMap[":es_done"] = EventServiceUtils.ST_done
                    varMap[":es_finished"] = EventServiceUtils.ST_finished
                    varMap[":es_merged"] = EventServiceUtils.ST_merged
                    self.cur.execute(sqlRR + comment, varMap)
                    resRR = self.cur.fetchall()
                    for (tmpPandaID,) in resRR:
                        esPandaIDs.add(tmpPandaID)
            # sql to update ES job
            sqlUE = "UPDATE {0} SET jobStatus=:newStatus,stateChangeTime=CURRENT_DATE,taskBufferErrorDiag=:errDiag "
            if jobStatus in ["failed"]:
                updateSubStatus = True
                sqlUE += ",jobSubStatus=:jobSubStatus "
            else:
                updateSubStatus = False
            sqlUE += "WHERE PandaID=:PandaID AND jobStatus in (:oldStatus1,:oldStatus2,:oldStatus3) AND modificationTime>(CURRENT_DATE-90) "
            sqlUE += "AND NOT eventService IN (:esJumbo) "
            for tmpPandaID in esPandaIDs:
                varMap = {}
                varMap[":PandaID"] = tmpPandaID
                varMap[":newStatus"] = jobStatus
                varMap[":oldStatus1"] = "closed"
                varMap[":oldStatus2"] = "merging"
                varMap[":oldStatus3"] = "failed"
                varMap[":esJumbo"] = EventServiceUtils.jumboJobFlagNumber
                if updateSubStatus is True:
                    if forceFailed:
                        varMap[":jobSubStatus"] = "es_discard"
                    elif EventServiceUtils.isEventServiceMerge(job):
                        varMap[":jobSubStatus"] = f"es_merge_{jobStatus}"
                    else:
                        varMap[":jobSubStatus"] = f"es_ass_{jobStatus}"
                if forceFailed:
                    varMap[":errDiag"] = f"{jobStatus} to discard old events to rety in PandaID={job.PandaID}"
                else:
                    varMap[":errDiag"] = f"{jobStatus} since an associated ES or merge job PandaID={job.PandaID} {jobStatus}"
                isUpdated = False
                for tableName in [
                    "ATLAS_PANDA.jobsArchived4",
                    "ATLAS_PANDAARCH.jobsArchived",
                ]:
                    self.cur.execute(sqlUE.format(tableName) + comment, varMap)
                    nRow = self.cur.rowcount
                    if nRow > 0:
                        tmpLog.debug(f"change PandaID={tmpPandaID} to {jobStatus}")
                        isUpdated = True
                # kill processed events if necessary
                if killEvents and isUpdated:
                    self.killUsedEventRanges(job.jediTaskID, tmpPandaID, job.notDiscardEvents())
            tmpLog.debug("done")
            return True
        except Exception:
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False

    # Configurator function: inserts data into the network matrix
    def insertNetworkMatrixData(self, data):
        comment = " /* DBProxy.insertNetworkMatrixData */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")

        # For performance reasons we will insert the data into a temporary table
        # and then merge this data into the permanent table.

        sql_insert = """
        INSERT INTO ATLAS_PANDA.network_matrix_kv_temp (src, dst, key, value, ts)
        VALUES (:src, :dst, :key, :value, :ts)
        """

        if self.backend == "postgres":
            sql_merge = (
                "INSERT INTO ATLAS_PANDA.network_matrix_kv "
                "(src, dst, key, value, ts) "
                "SELECT  src, dst, key, value, ts FROM ATLAS_PANDA.NETWORK_MATRIX_KV_TEMP "
                "ON CONFLICT (src, dst, key) "
                "DO UPDATE SET value=EXCLUDED.value, ts=EXCLUDED.ts "
            )
        else:
            sql_merge = """
            MERGE /*+ FULL(nm_kv) */ INTO ATLAS_PANDA.network_matrix_kv nm_kv USING
                (SELECT  src, dst, key, value, ts FROM ATLAS_PANDA.NETWORK_MATRIX_KV_TEMP) input
                ON (nm_kv.src = input.src AND nm_kv.dst= input.dst AND nm_kv.key = input.key)
            WHEN NOT MATCHED THEN
                INSERT (src, dst, key, value, ts)
                VALUES (input.src, input.dst, input.key, input.value, input.ts)
            WHEN MATCHED THEN
                UPDATE SET nm_kv.value = input.value, nm_kv.ts = input.ts
            """
        try:
            self.conn.begin()
            for shard in create_shards(data, 100):
                time1 = time.time()
                var_maps = []
                for entry in shard:
                    var_map = {
                        ":src": entry[0],
                        ":dst": entry[1],
                        ":key": entry[2],
                        ":value": entry[3],
                        ":ts": entry[4],
                    }
                    var_maps.append(var_map)

                time2 = time.time()
                self.cur.executemany(sql_insert + comment, var_maps)
                time3 = time.time()
                tmpLog.debug(f"Processing a shard took: {time2 - time1}s of data preparation and {time3 - time2}s of insertion = {time3 - time1}")

            time4 = time.time()
            self.cur.execute(sql_merge + comment)
            time5 = time.time()
            tmpLog.debug(f"Final merge took: {time5 - time4}s")
            if self.backend == "postgres":
                # cleanup since ON CONFLICT DO UPDATE doesn't work with duplicated entries
                self.cur.execute("DELETE FROM ATLAS_PANDA.NETWORK_MATRIX_KV_TEMP " + comment)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")

        except Exception:
            # roll back
            self._rollback()
            # error
            exc_type, exc_value, exc_traceback = sys.exc_info()
            tmpLog.debug(f"Failed to commit bulk with exception {repr(traceback.format_exception(exc_type, exc_value, exc_traceback))}")
            self.dumpErrorMessage(_logger, methodName)
            return None, ""

    # Configurator function: delete old network data
    def deleteOldNetworkData(self):
        comment = " /* DBProxy.deleteOldNetworkData */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")

        # delete any data older than a week
        sql_delete = """
        DELETE FROM ATLAS_PANDA.network_matrix_kv
        WHERE ts < (current_date - 7)
        """
        try:
            self.conn.begin()
            time1 = time.time()
            self.cur.execute(sql_delete + comment)
            time2 = time.time()
            tmpLog.debug(f"Deletion of old network data took: {time2 - time1}s")

            # commit
            if not self._commit():
                raise RuntimeError("Commit error")

        except Exception:
            # roll back
            self._rollback()
            # error
            exc_type, exc_value, exc_traceback = sys.exc_info()
            tmpLog.debug(f"Failed to commit: {repr(traceback.format_exception(exc_type, exc_value, exc_traceback))}")
            self.dumpErrorMessage(_logger, methodName)
            return None, ""

    # get dispatch datasets per user
    def getDispatchDatasetsPerUser(self, vo, prodSourceLabel, onlyActive, withSize):
        comment = " /* DBProxy.getDispatchDatasetsPerUser */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <vo={vo} label={prodSourceLabel}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        # mapping for table and job status
        tableStatMap = {"jobsDefined4": ["defined", "assigned"]}
        if not onlyActive:
            tableStatMap["jobsActive4"] = None
            tableStatMap["jobsArchived4"] = None
        try:
            userDispMap = {}
            for tableName in tableStatMap:
                statusList = tableStatMap[tableName]
                # make sql to get dispatch datasets
                varMap = {}
                varMap[":vo"] = vo
                varMap[":label"] = prodSourceLabel
                varMap[":dType"] = "dispatch"
                sqlJ = "SELECT distinct prodUserName,dispatchDBlock,jediTaskID,currentFiles "
                sqlJ += "FROM {0}.{1} j, {0}.Datasets d ".format(panda_config.schemaPANDA, tableName)
                sqlJ += "WHERE vo=:vo AND prodSourceLabel=:label "
                if statusList is not None:
                    sqlJ += "AND jobStatus IN ("
                    for tmpStat in statusList:
                        tmpKey = f":jobStat_{tmpStat}"
                        sqlJ += f"{tmpKey},"
                        varMap[tmpKey] = tmpStat
                    sqlJ = sqlJ[:-1]
                    sqlJ += ") "
                sqlJ += "AND dispatchDBlock IS NOT NULL "
                sqlJ += "AND d.name=j.dispatchDBlock AND d.modificationDate>CURRENT_DATE-14 "
                sqlJ += "AND d.type=:dType "
                # begin transaction
                self.conn.begin()
                # get dispatch datasets
                self.cur.execute(sqlJ + comment, varMap)
                resJ = self.cur.fetchall()
                if not self._commit():
                    raise RuntimeError("Commit error")
                # make map
                for prodUserName, dispatchDBlock, jediTaskID, dsSize in resJ:
                    transferType = "transfer"
                    try:
                        if dispatchDBlock.split(".")[4] == "prestaging":
                            transferType = "prestaging"
                    except Exception:
                        pass
                    userDispMap.setdefault(prodUserName, {})
                    userDispMap[prodUserName].setdefault(transferType, {"datasets": set(), "size": 0, "tasks": set()})
                    if dispatchDBlock not in userDispMap[prodUserName][transferType]["datasets"]:
                        userDispMap[prodUserName][transferType]["datasets"].add(dispatchDBlock)
                        userDispMap[prodUserName][transferType]["tasks"].add(jediTaskID)
                        userDispMap[prodUserName][transferType]["size"] += dsSize
            tmpLog.debug("done")
            return userDispMap
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return {}

    # get task parameters
    def getTaskParamsPanda(self, jediTaskID):
        comment = " /* DBProxy.getTaskParamsPanda */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            # sql to get task parameters
            sqlRR = f"SELECT jedi_task_parameters FROM {panda_config.schemaDEFT}.T_TASK "
            sqlRR += "WHERE taskid=:jediTaskID "
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            # start transaction
            self.conn.begin()
            self.cur.execute(sqlRR + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # read clob
            taskParams = ""
            for (clobJobP,) in self.cur:
                if clobJobP is not None:
                    try:
                        taskParams = clobJobP.read()
                    except AttributeError:
                        taskParams = str(clobJobP)
                break
            tmpLog.debug("done")
            return taskParams
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return ""

    # get task attributes
    def getTaskAttributesPanda(self, jediTaskID, attrs):
        comment = " /* DBProxy.getTaskAttributesPanda */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            # sql to get task attributes
            sqlRR = "SELECT "
            for attr in attrs:
                sqlRR += f"{attr},"
            sqlRR = sqlRR[:-1]
            sqlRR += f" FROM {panda_config.schemaJEDI}.JEDI_Tasks "
            sqlRR += "WHERE jediTaskID=:jediTaskID "
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            # start transaction
            self.conn.begin()
            self.cur.execute(sqlRR + comment, varMap)
            resRR = self.cur.fetchone()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            retVal = {}
            if resRR is not None:
                for idx, attr in enumerate(attrs):
                    retVal[attr] = resRR[idx]
            tmpLog.debug(f"done {str(retVal)}")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return {}

    # check for cloned jobs
    def checkClonedJob(self, jobSpec, useCommit=True):
        comment = " /* DBProxy.checkClonedJob */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" <PandaID={jobSpec.PandaID}>")
        tmpLog.debug("start")
        try:
            # return value {'lock': True if the job locked the semaphore,
            #               'last': True if the job is the last clone
            #               'win': PandaID of winner
            # None : fatal error
            retValue = {"lock": False, "last": False, "win": None}
            # begin transaction
            if useCommit:
                self.conn.begin()
            self.cur.arraysize = 10000
            # check if semaphore is locked
            sqlED = f"SELECT COUNT(*) FROM {panda_config.schemaJEDI}.JEDI_Events "
            sqlED += "WHERE jediTaskID=:jediTaskID AND pandaID=:pandaID "
            varMap = {}
            varMap[":jediTaskID"] = jobSpec.jediTaskID
            varMap[":pandaID"] = jobSpec.PandaID
            self.cur.execute(sqlED + comment, varMap)
            resEU = self.cur.fetchone()
            (nRowEU,) = resEU
            if nRowEU > 0:
                retValue["lock"] = True
            else:
                # get PandaID of the winner
                sqlWP = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                sqlWP += "distinct PandaID "
                sqlWP += f"FROM {panda_config.schemaJEDI}.JEDI_Events tab "
                sqlWP += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                for tmpFileSpec in jobSpec.Files:
                    if tmpFileSpec.type == "input":
                        varMap = {}
                        varMap[":jediTaskID"] = tmpFileSpec.jediTaskID
                        varMap[":datasetID"] = tmpFileSpec.datasetID
                        varMap[":fileID"] = tmpFileSpec.fileID
                        self.cur.execute(sqlWP + comment, varMap)
                        resWP = self.cur.fetchone()
                        if resWP is not None:
                            retValue["win"] = resWP[0]
                            break
            # get PandaIDs of clones
            sqlCP = "SELECT PandaID FROM ATLAS_PANDA.jobsActive4 "
            sqlCP += "WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID "
            sqlCP += "UNION "
            sqlCP += "SELECT PandaID FROM ATLAS_PANDA.jobsDefined4 "
            sqlCP += "WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID "
            sqlCP += "UNION "
            sqlCP += "SELECT PandaID FROM ATLAS_PANDA.jobsWaiting4 "
            sqlCP += "WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID "
            varMap = {}
            varMap[":jediTaskID"] = jobSpec.jediTaskID
            varMap[":jobsetID"] = jobSpec.jobsetID
            self.cur.execute(sqlCP + comment, varMap)
            resCP = self.cur.fetchall()
            pandaIDsList = set()
            for (pandaID,) in resCP:
                if pandaID != jobSpec.PandaID:
                    pandaIDsList.add(pandaID)
            if len(pandaIDsList) == 0:
                retValue["last"] = True
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError("Commit error")
            tmpLog.debug(retValue)
            return retValue
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None

    # set HS06sec
    def setHS06sec(self, pandaID, inActive=False):
        comment = " /* DBProxy.setHS06sec */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" <PandaID={pandaID}>")
        tmpLog.debug("start")
        hs06sec = None

        # sql to get job attributes
        sqlJ = "SELECT jediTaskID,startTime,endTime,actualCoreCount,coreCount,jobMetrics,computingSite "
        if inActive:
            sqlJ += "FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
        else:
            sqlJ += "FROM ATLAS_PANDA.jobsArchived4 WHERE PandaID=:PandaID "

        # sql to update HS06sec
        if inActive:
            sqlU = "UPDATE ATLAS_PANDA.jobsActive4 "
        else:
            sqlU = "UPDATE ATLAS_PANDA.jobsArchived4 "
        sqlU += "SET hs06sec=:hs06sec WHERE PandaID=:PandaID "

        # get job attributes
        varMap = {}
        varMap[":PandaID"] = pandaID
        self.cur.execute(sqlJ + comment, varMap)
        resJ = self.cur.fetchone()
        if resJ is None:
            tmpLog.debug("skip since job not found")
        else:
            (
                jediTaskID,
                startTime,
                endTime,
                actualCoreCount,
                defCoreCount,
                jobMetrics,
                computingSite,
            ) = resJ
            # get corePower
            corePower, tmpMsg = self.get_core_power(computingSite)
            if corePower is None:
                tmpLog.debug(f"skip since corePower is undefined for site={computingSite}")
            else:
                # get core count
                coreCount = JobUtils.getCoreCount(actualCoreCount, defCoreCount, jobMetrics)
                # get HS06sec
                hs06sec = JobUtils.getHS06sec(startTime, endTime, corePower, coreCount)
                if hs06sec is None:
                    tmpLog.debug("skip since HS06sec is None")
                else:
                    # cap
                    hs06sec = int(hs06sec)
                    maxHS06sec = 999999999
                    if hs06sec > maxHS06sec:
                        hs06sec = maxHS06sec
                    # update HS06sec
                    varMap = {}
                    varMap[":PandaID"] = pandaID
                    varMap[":hs06sec"] = hs06sec
                    self.cur.execute(sqlU + comment, varMap)
                    tmpLog.debug(f"set HS06sec={hs06sec}")
        # return
        return hs06sec

    def convert_computingsite_to_region(self, computing_site):
        comment = " /* DBProxy.convert_computingsite_to_region */"

        var_map = {":panda_queue": computing_site}
        sql = "SELECT /* use_json_type */ scj.data.region FROM ATLAS_PANDA.schedconfig_json scj WHERE scj.panda_queue=:panda_queue"
        self.cur.arraysize = 100
        self.cur.execute(sql + comment, var_map)
        res_region = self.cur.fetchone()
        region = "GRID"  # when region is not defined, take average values
        if res_region:
            region = res_region[0]

        return region

    def get_co2_emissions_site(self, computing_site):
        comment = " /* DBProxy.get_co2_emissions_site */"
        region = self.convert_computingsite_to_region(computing_site)
        if not region:
            return None

        var_map = {":region": region}
        sql = "SELECT timestamp, region, value FROM ATLAS_PANDA.CARBON_REGION_EMISSIONS WHERE region=:region"
        self.cur.execute(sql + comment, var_map)
        results = self.cur.fetchall()
        return results

    def get_co2_emissions_grid(self):
        comment = " /* DBProxy.get_co2_emissions_grid */"

        sql = "SELECT timestamp, region, value FROM ATLAS_PANDA.CARBON_REGION_EMISSIONS WHERE region='GRID'"
        self.cur.execute(sql + comment)
        results = self.cur.fetchall()
        return results

    # set CO2 emissions
    def set_co2_emissions(self, panda_id, in_active=False):
        comment = " /* DBProxy.set_co2_emissions */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name + f" <PandaID={panda_id}>")
        tmp_log.debug("start")
        gco2_regional, gco2_global = None, None

        # sql to get job attributes
        sql_read = "SELECT jediTaskID, startTime, endTime, actualCoreCount, coreCount, jobMetrics, computingSite "
        if in_active:
            sql_read += "FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
        else:
            sql_read += "FROM ATLAS_PANDA.jobsArchived4 WHERE PandaID=:PandaID "

        # sql to update CO2 emissions
        if in_active:
            sql_update = "UPDATE ATLAS_PANDA.jobsActive4 "
        else:
            sql_update = "UPDATE ATLAS_PANDA.jobsArchived4 "
        sql_update += "SET gCO2_global=:gco2_global, gCO2_regional=:gco2_regional WHERE PandaID=:PandaID "

        # get job attributes
        var_map = {":PandaID": panda_id}
        self.cur.execute(sql_read + comment, var_map)
        res_read = self.cur.fetchone()
        if res_read is None:
            tmp_log.debug("skip since job not found")
        else:
            (
                task_id,
                start_time,
                end_time,
                actual_cores,
                defined_cores,
                job_metrics,
                computing_site,
            ) = res_read

            # get core count
            core_count = JobUtils.getCoreCount(actual_cores, defined_cores, job_metrics)

            # get the queues watts per core value
            var_map = {":panda_queue": computing_site}
            sql_wpc = "SELECT /* use_json_type */ scj.data.coreenergy FROM ATLAS_PANDA.schedconfig_json scj WHERE scj.panda_queue=:panda_queue"
            self.cur.arraysize = 100
            self.cur.execute(sql_wpc + comment, var_map)
            res_wpc = self.cur.fetchone()
            try:
                watts_per_core = float(res_wpc[0])
            except Exception:
                watts_per_core = 10
            tmp_log.debug(f"using watts_per_core={watts_per_core} for computing_site={computing_site}")

            # get regional CO2 emissions
            co2_emissions = self.get_co2_emissions_site(computing_site)
            if not co2_emissions:
                tmp_log.debug(f"skip since co2_emissions are undefined for site={computing_site}")
            else:
                # get emitted CO2 for the job
                gco2_regional = JobUtils.get_job_co2(start_time, end_time, core_count, co2_emissions, watts_per_core)
                if gco2_regional is None:
                    tmp_log.debug("skip since the co2 emissions could not be calculated")
                else:
                    max_gco2 = 999999999
                    gco2_regional = min(gco2_regional, max_gco2)
                    tmp_log.debug(f"set gco2_regional={gco2_regional}")

            # get globally averaged CO2 emissions
            co2_emissions = self.get_co2_emissions_grid()
            if not co2_emissions:
                tmp_log.debug("skip since co2_emissions are undefined for the grid")
            else:
                # get emitted CO2 for the job
                gco2_global = JobUtils.get_job_co2(start_time, end_time, core_count, co2_emissions, watts_per_core)
                if gco2_global is None:
                    tmp_log.debug("skip since the co2 emissions could not be calculated")
                else:
                    max_gco2 = 999999999
                    gco2_global = min(gco2_global, max_gco2)

                    tmp_log.debug(f"set gco2_global={gco2_global}")

            var_map = {
                ":PandaID": panda_id,
                ":gco2_regional": gco2_regional,
                ":gco2_global": gco2_global,
            }
            self.cur.execute(sql_update + comment, var_map)

        tmp_log.debug("done")
        # return
        return gco2_regional, gco2_global

    # check if all events are done
    def checkAllEventsDone(self, job, pandaID, useCommit=False, dumpLog=True, getProcStatus=False):
        comment = " /* DBProxy.checkAllEventsDone */"
        if job is not None:
            pandaID = job.PandaID
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" <PandaID={pandaID}>")
        if dumpLog:
            tmpLog.debug("start")
        try:
            # get files
            sqlF = f"SELECT type,jediTaskID,datasetID,fileID FROM {panda_config.schemaPANDA}.filesTable4 "
            sqlF += "WHERE PandaID=:PandaID AND type=:type "
            # check if all events are done
            sqlEOC = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlEOC += f"distinct PandaID,status FROM {panda_config.schemaJEDI}.JEDI_Events tab "
            sqlEOC += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            sqlEOC += "AND NOT status IN (:esDone,:esDiscarded,:esCancelled,:esFatal,:esCorrupted,:esFailed,:esFinished) "
            sqlEOC += "AND NOT (status=:esReady AND attemptNr=0) "
            # get jumbo jobs
            sqlGJ = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlGJ += f"distinct PandaID FROM {panda_config.schemaJEDI}.JEDI_Events tab "
            sqlGJ += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            sqlGJ += "AND status IN (:esRunning,:esSent,:esFinished,:esDone) "
            # check if job is still alive
            sqlJAL = f"SELECT jobStatus,eventService FROM {panda_config.schemaPANDA}.jobsActive4 "
            sqlJAL += "WHERE PandaID=:PandaID "
            # begin transaction
            if useCommit:
                self.conn.begin()
            self.cur.arraysize = 1000000
            # get files if needed
            if job is not None:
                fileList = job.Files
            else:
                varMap = {}
                varMap[":PandaID"] = pandaID
                varMap[":type"] = "input"
                self.cur.execute(sqlF + comment, varMap)
                resF = self.cur.fetchall()
                fileList = []
                for tmpType, tmpJediTaskID, tmpDatasetID, tmpFileID in resF:
                    fileSpec = FileSpec()
                    fileSpec.type = tmpType
                    fileSpec.jediTaskID = tmpJediTaskID
                    fileSpec.datasetID = tmpDatasetID
                    fileSpec.fileID = tmpFileID
                    fileList.append(fileSpec)
            # check all inputs
            allDone = True
            proc_status = None
            checkedPandaIDs = set()
            jobStatusMap = dict()
            for fileSpec in fileList:
                if fileSpec.type == "input":
                    varMap = {}
                    varMap[":jediTaskID"] = fileSpec.jediTaskID
                    varMap[":datasetID"] = fileSpec.datasetID
                    varMap[":fileID"] = fileSpec.fileID
                    varMap[":esDone"] = EventServiceUtils.ST_done
                    varMap[":esFinished"] = EventServiceUtils.ST_finished
                    varMap[":esDiscarded"] = EventServiceUtils.ST_discarded
                    varMap[":esCancelled"] = EventServiceUtils.ST_cancelled
                    varMap[":esCorrupted"] = EventServiceUtils.ST_corrupted
                    varMap[":esFatal"] = EventServiceUtils.ST_fatal
                    varMap[":esFailed"] = EventServiceUtils.ST_failed
                    varMap[":esReady"] = EventServiceUtils.ST_ready
                    self.cur.execute(sqlEOC + comment, varMap)
                    resEOC = self.cur.fetchall()
                    for pandaID, esStatus in resEOC:
                        # skip redundant lookup
                        if pandaID in checkedPandaIDs:
                            continue
                        checkedPandaIDs.add(pandaID)
                        # not yet dispatched
                        if esStatus == EventServiceUtils.ST_ready:
                            tmpStr = "some events are not yet dispatched "
                            tmpStr += f"for jediTaskID={fileSpec.jediTaskID} datasetID={fileSpec.datasetID} fileID={fileSpec.fileID}"
                            if dumpLog:
                                tmpLog.debug(tmpStr)
                            allDone = False
                            break
                        # check job
                        varMap = {}
                        varMap[":PandaID"] = pandaID
                        self.cur.execute(sqlJAL + comment, varMap)
                        resJAL = self.cur.fetchone()
                        if resJAL is None:
                            # no active job
                            tmpStr = "no assiciated job is in active "
                            tmpStr += f"for jediTaskID={fileSpec.jediTaskID} datasetID={fileSpec.datasetID} fileID={fileSpec.fileID}"
                            if dumpLog:
                                tmpLog.debug(tmpStr)
                            jobStatusMap[pandaID] = None
                        else:
                            # still active
                            tmpStr = f"PandaID={pandaID} is associated in {resJAL[0]} "
                            tmpStr += f"for jediTaskID={fileSpec.jediTaskID} datasetID={fileSpec.datasetID} fileID={fileSpec.fileID}"
                            if dumpLog:
                                tmpLog.debug(tmpStr)
                            allDone = False
                            if resJAL[1] == EventServiceUtils.jumboJobFlagNumber:
                                jobStatusMap[pandaID] = resJAL[0]
                            else:
                                jobStatusMap[pandaID] = None
                            break
                        # escape
                        if not allDone:
                            break
                # escape
                if not allDone:
                    break
            # get proc_status
            if not allDone and getProcStatus:
                proc_status = "queued"
                to_escape = False
                is_starting = False
                for fileSpec in fileList:
                    if fileSpec.type == "input":
                        varMap = {}
                        varMap[":jediTaskID"] = fileSpec.jediTaskID
                        varMap[":datasetID"] = fileSpec.datasetID
                        varMap[":fileID"] = fileSpec.fileID
                        varMap[":esDone"] = EventServiceUtils.ST_done
                        varMap[":esFinished"] = EventServiceUtils.ST_finished
                        varMap[":esRunning"] = EventServiceUtils.ST_running
                        varMap[":esSent"] = EventServiceUtils.ST_sent
                        self.cur.execute(sqlGJ + comment, varMap)
                        resGJ = self.cur.fetchall()
                        for (pandaID,) in resGJ:
                            if pandaID not in jobStatusMap:
                                # get job
                                varMap = {}
                                varMap[":PandaID"] = pandaID
                                self.cur.execute(sqlJAL + comment, varMap)
                                resJAL = self.cur.fetchone()
                                if resJAL is None:
                                    jobStatusMap[pandaID] = None
                                else:
                                    if resJAL[1] == EventServiceUtils.jumboJobFlagNumber:
                                        jobStatusMap[pandaID] = resJAL[0]
                                    else:
                                        jobStatusMap[pandaID] = None
                            # check status
                            if jobStatusMap[pandaID] == "running":
                                proc_status = "running"
                                to_escape = True
                                break
                            elif jobStatusMap[pandaID] == "starting":
                                is_starting = True
                        if to_escape:
                            break
                if proc_status == "queued" and is_starting:
                    proc_status = "starting"
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError("Commit error")
            if dumpLog:
                tmpLog.debug(f"done with {allDone} {proc_status}")
            if getProcStatus:
                return (allDone, proc_status)
            return allDone
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            if getProcStatus:
                return (None, None)
            return None

    # get co-jumbo jobs to be finished
    def getCoJumboJobsToBeFinished(self, timeLimit, minPriority, maxJobs):
        comment = " /* DBProxy.getCoJumboJobsToBeFinished */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug(f"start for minPriority={minPriority} timeLimit={timeLimit}")
        try:
            # get co-jumbo jobs
            sqlEOD = "SELECT PandaID,jediTaskID,jobStatus,computingSite,creationTime FROM ATLAS_PANDA.{0} "
            sqlEOD += "WHERE eventService=:eventService "
            sqlEOD += "AND (prodDBUpdateTime IS NULL OR prodDBUpdateTime<:timeLimit) "
            sqlEOD += "AND currentPriority>=:minPriority "
            # lock job
            sqlPL = "SELECT 1 FROM ATLAS_PANDA.{0} "
            sqlPL += "WHERE PandaID=:PandaID "
            sqlPL += "AND (prodDBUpdateTime IS NULL OR prodDBUpdateTime<:timeLimit) "
            sqlPL += "FOR UPDATE NOWAIT "
            sqlLK = "UPDATE ATLAS_PANDA.{0} "
            sqlLK += "SET prodDBUpdateTime=CURRENT_DATE "
            sqlLK += "WHERE PandaID=:PandaID "
            sqlLK += "AND (prodDBUpdateTime IS NULL OR prodDBUpdateTime<:timeLimit) "
            # get useJumbo
            sqlJM = f"SELECT useJumbo FROM {panda_config.schemaJEDI}.JEDI_Tasks "
            sqlJM += "WHERE jediTaskID=:jediTaskID "
            # get datasetID and fileID of the primary input
            sqlID = "SELECT f.datasetID,f.fileID,c.status,c.proc_status FROM {0}.JEDI_Datasets d,{0}.JEDI_Dataset_Contents c,{1}.filesTable4 f ".format(
                panda_config.schemaJEDI, panda_config.schemaPANDA
            )
            sqlID += "WHERE d.jediTaskID=:jediTaskID AND d.type IN (:t1,:t2) AND d.masterID IS NULL "
            sqlID += "AND f.jediTaskID=d.jediTaskID AND f.datasetID=d.datasetID AND f.PandaID=:PandaID "
            sqlID += "AND c.jediTaskID=d.jediTaskID AND c.datasetID=d.datasetID AND c.fileID=f.fileID "
            # get PandaIDs
            sqlCP = "SELECT PandaID FROM ATLAS_PANDA.filesTable4 "
            sqlCP += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            # check jobs
            sqlWP = "SELECT 1 FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID "
            sqlWP += "UNION "
            sqlWP += "SELECT 1 FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
            self.cur.arraysize = 1000000
            timeLimit = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(minutes=timeLimit)
            timeLimitWaiting = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(hours=6)
            retList = []
            # get jobs
            coJumboTobeKilled = set()
            useJumbos = dict()
            for tableName in ["jobsActive4", "jobsDefined4", "jobsWaiting4"]:
                self.conn.begin()
                varMap = {}
                varMap[":eventService"] = EventServiceUtils.coJumboJobFlagNumber
                varMap[":timeLimit"] = timeLimit
                varMap[":minPriority"] = minPriority
                self.cur.execute(sqlEOD.format(tableName) + comment, varMap)
                tmpRes = self.cur.fetchall()
                if not self._commit():
                    raise RuntimeError("Commit error")
                tmpLog.debug(f"checking {len(tmpRes)} co-jumbo jobs in {tableName}")
                checkedPandaIDs = set()
                iJobs = 0
                # scan all jobs
                for (
                    pandaID,
                    jediTaskID,
                    jobStatus,
                    computingSite,
                    creationTime,
                ) in tmpRes:
                    # lock job
                    self.conn.begin()
                    varMap = {}
                    varMap[":PandaID"] = pandaID
                    varMap[":timeLimit"] = timeLimit
                    toSkip = False
                    resPL = None
                    try:
                        # lock with NOWAIT
                        self.cur.execute(sqlPL.format(tableName) + comment, varMap)
                        resPL = self.cur.fetchone()
                    except Exception:
                        toSkip = True
                    if resPL is None:
                        toSkip = True
                    if toSkip:
                        tmpLog.debug(f"skipped PandaID={pandaID} jediTaskID={jediTaskID} in {tableName} since locked by another")
                    else:
                        # lock
                        self.cur.execute(sqlLK.format(tableName) + comment, varMap)
                        nRow = self.cur.rowcount
                        if nRow > 0:
                            iJobs += 1
                            # check if all events are done
                            allDone, proc_status = self.checkAllEventsDone(None, pandaID, False, True, True)
                            if allDone is True:
                                tmpLog.debug(f"locked co-jumbo PandaID={pandaID} jediTaskID={jediTaskID} to finish in {tableName}")
                                checkedPandaIDs.add(pandaID)
                            elif jobStatus == "waiting" and computingSite == EventServiceUtils.siteIdForWaitingCoJumboJobs and proc_status == "queued":
                                # check if jumbo is disabled
                                if jediTaskID not in useJumbos:
                                    varMap = {}
                                    varMap[":jediTaskID"] = jediTaskID
                                    self.cur.execute(sqlJM + comment, varMap)
                                    resJM = self.cur.fetchone()
                                    (useJumbos[jediTaskID],) = resJM
                                if useJumbos[jediTaskID] == "D" or creationTime < timeLimitWaiting:
                                    # get info of the primary input
                                    varMap = {}
                                    varMap[":jediTaskID"] = jediTaskID
                                    varMap[":PandaID"] = pandaID
                                    varMap[":t1"] = "input"
                                    varMap[":t2"] = "pseudo_input"
                                    self.cur.execute(sqlID + comment, varMap)
                                    resID = self.cur.fetchone()
                                    (
                                        datasetID,
                                        fileID,
                                        fileStatus,
                                        fileProcStatus,
                                    ) = resID
                                    if fileStatus == "running" and fileProcStatus == "queued":
                                        # count # of active consumers
                                        nAct = 0
                                        varMap = {}
                                        varMap[":jediTaskID"] = jediTaskID
                                        varMap[":datasetID"] = datasetID
                                        varMap[":fileID"] = fileID
                                        self.cur.execute(sqlCP + comment, varMap)
                                        resCP = self.cur.fetchall()
                                        for (tmpPandaID,) in resCP:
                                            varMap = {}
                                            varMap[":PandaID"] = tmpPandaID
                                            self.cur.execute(sqlWP + comment, varMap)
                                            resWP = self.cur.fetchone()
                                            if resWP is not None:
                                                nAct += 1
                                        if nAct > 0:
                                            tmpLog.debug(f"skip to kill PandaID={pandaID} jediTaskID={jediTaskID} due to {nAct} active consumers")
                                        else:
                                            tmpLog.debug(f"locked co-jumbo PandaID={pandaID} jediTaskID={jediTaskID} to kill")
                                            coJumboTobeKilled.add(pandaID)
                            if proc_status is not None:
                                self.updateInputStatusJedi(jediTaskID, pandaID, "queued", checkOthers=True)
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    if iJobs >= maxJobs:
                        break
                retList.append(checkedPandaIDs)
            totJobs = 0
            for tmpList in retList:
                totJobs += len(tmpList)
            tmpLog.debug(f"got {totJobs} jobs to finish and {len(coJumboTobeKilled)} co-jumbo jobs to kill")
            retList.append(coJumboTobeKilled)
            return retList
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None

    # check if there are done events
    def hasDoneEvents(self, jediTaskID, pandaID, jobSpec, useCommit=True):
        comment = " /* DBProxy.hasDoneEvents */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(
            _logger,
            methodName + f" <PandaID={pandaID} jediTaskID={jediTaskID}>",
        )
        tmpLog.debug("start")
        retVal = False
        try:
            # sql to release events
            sqlR = f"UPDATE {panda_config.schemaJEDI}.JEDI_Events "
            if jobSpec.decAttOnFailedES():
                sqlR += "SET status=:newStatus,pandaID=event_offset,is_jumbo=NULL "
            else:
                sqlR += "SET status=:newStatus,attemptNr=attemptNr-1,pandaID=event_offset,is_jumbo=NULL "
            sqlR += "WHERE jediTaskID=:jediTaskID AND pandaID=:pandaID AND status IN (:esSent,:esRunning) "
            # sql to check event
            sqlF = f"SELECT COUNT(*) FROM {panda_config.schemaJEDI}.JEDI_Events "
            sqlF += "WHERE jediTaskID=:jediTaskID AND PandaID=:pandaID AND status IN (:esDone,:esFinished) "
            # begin transaction
            if useCommit:
                self.conn.begin()
            # release events
            varMap = {}
            varMap[":pandaID"] = pandaID
            varMap[":jediTaskID"] = jediTaskID
            varMap[":esSent"] = EventServiceUtils.ST_sent
            varMap[":esRunning"] = EventServiceUtils.ST_running
            varMap[":newStatus"] = EventServiceUtils.ST_ready
            self.cur.execute(sqlR + comment, varMap)
            resR = self.cur.rowcount
            tmpLog.debug(f"released {resR} event ranges")
            # check event
            varMap = {}
            varMap[":pandaID"] = pandaID
            varMap[":jediTaskID"] = jediTaskID
            varMap[":esDone"] = EventServiceUtils.ST_done
            varMap[":esFinished"] = EventServiceUtils.ST_finished
            self.cur.execute(sqlF + comment, varMap)
            resF = self.cur.fetchone()
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError("Commit error")
            nFinished = 0
            if resF is not None:
                (nFinished,) = resF
            if nFinished > 0:
                retVal = True
            else:
                retVal = False
            tmpLog.debug(f"finished {nFinished} event ranges. ret={retVal}")
            return retVal
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(tmpLog, methodName)
            return retVal

    # check if there are events to be processed
    def hasReadyEvents(self, jediTaskID):
        comment = " /* DBProxy.hasReadyEvents */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" <jediTaskID={jediTaskID}>")
        tmpLog.debug("start")
        retVal = None
        try:
            # sql to check event
            sqlF = f"SELECT COUNT(*) FROM {panda_config.schemaJEDI}.JEDI_Events "
            sqlF += "WHERE jediTaskID=:jediTaskID AND status=:esReady AND attemptNr>:minAttemptNr "
            # check event
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            varMap[":esReady"] = EventServiceUtils.ST_ready
            varMap[":minAttemptNr"] = 0
            # begin transaction
            self.conn.begin()
            self.cur.execute(sqlF + comment, varMap)
            resF = self.cur.fetchone()
            nReady = None
            if resF is not None:
                (nReady,) = resF
                retVal = nReady > 0
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"{nReady} ready events. ret={retVal}")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmpLog, methodName)
            return None

    # get number of events to be processed
    def getNumReadyEvents(self, jediTaskID):
        comment = " /* DBProxy.getNumReadyEvents */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" <jediTaskID={jediTaskID}>")
        tmpLog.debug("start")
        nReady = None
        try:
            # sql to count event
            sqlF = f"SELECT COUNT(*) FROM {panda_config.schemaJEDI}.JEDI_Events "
            sqlF += "WHERE jediTaskID=:jediTaskID AND status=:esReady AND attemptNr>:minAttemptNr "
            # count event
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            varMap[":esReady"] = EventServiceUtils.ST_ready
            varMap[":minAttemptNr"] = 0
            # begin transaction
            self.conn.begin()
            self.cur.execute(sqlF + comment, varMap)
            resF = self.cur.fetchone()
            nReady = None
            if resF is not None:
                (nReady,) = resF
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"{nReady} ready events")
            return nReady
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmpLog, methodName)
            return None

    # check if task is applicable for jumbo jobs
    def isApplicableTaskForJumbo(self, jediTaskID):
        comment = " /* DBProxy.isApplicableTaskForJumbo */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" <jediTaskID={jediTaskID}>")
        tmpLog.debug("start")
        retVal = True
        try:
            # sql to check event
            sqlF = "SELECT SUM(nFiles),SUM(nFilesFinished),SUM(nFilesFailed) "
            sqlF += f"FROM {panda_config.schemaJEDI}.JEDI_Datasets "
            sqlF += "WHERE jediTaskID=:jediTaskID AND type IN (:type1,:type2) "
            sqlF += "AND masterID IS NULL "
            # begin transaction
            self.conn.begin()
            # check task status
            if not self.checkTaskStatusJEDI(jediTaskID, self.cur):
                # task is in a final status
                retVal = False
            else:
                # threshold in % to stop jumbo jobs
                threshold = 100
                # check percentage
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":type1"] = "input"
                varMap[":type2"] = "pseudo_input"
                self.cur.execute(sqlF + comment, varMap)
                resF = self.cur.fetchone()
                nFiles, nFilesFinished, nFilesFailed = resF
                if (nFilesFinished + nFilesFailed) * 100 >= nFiles * threshold:
                    retVal = False
                    tmpLog.debug(f"nFilesFinished({nFilesFinished}) + nFilesFailed({nFilesFailed}) >= nFiles({nFiles})*{threshold}%")
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"done with {retVal}")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return retVal

    # cleanup jumbo jobs
    def cleanupJumboJobs(self, jediTaskID=None):
        comment = " /* DBProxy.cleanupJumboJobs */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" <jediTaskID={jediTaskID}>")
        tmpLog.debug("start")
        try:
            # sql to get jumbo jobs
            sql = "SELECT PandaID,jediTaskID,jobStatus FROM ATLAS_PANDA.jobsDefined4 WHERE eventService=:eventService "
            if jediTaskID is not None:
                sql += "AND jediTaskID=:jediTaskID "
            sql += "UNION "
            sql += "SELECT PandaID,jediTaskID,jobStatus FROM ATLAS_PANDA.jobsActive4 WHERE eventService=:eventService "
            if jediTaskID is not None:
                sql += "AND jediTaskID=:jediTaskID "
            sql += "UNION "
            sql += "SELECT PandaID,jediTaskID,jobStatus FROM ATLAS_PANDA.jobsWaiting4 WHERE eventService=:eventService "
            if jediTaskID is not None:
                sql += "AND jediTaskID=:jediTaskID "
            # begin transaction
            self.conn.begin()
            # get jobs
            varMap = {}
            varMap[":eventService"] = EventServiceUtils.jumboJobFlagNumber
            self.cur.execute(sql + comment, varMap)
            resF = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # get ID mapping
            idMap = {}
            for pandaID, tmpJediTaskID, jobStatus in resF:
                if jobStatus in ["transferring", "running", "holding"]:
                    continue
                if tmpJediTaskID not in idMap:
                    idMap[tmpJediTaskID] = set()
                idMap[tmpJediTaskID].add(pandaID)
            tmpLog.debug(f"got {len(idMap)} taks")
            # sql to check useJumbo
            sqlJ = "SELECT useJumbo FROM ATLAS_PANDA.JEDI_Tasks WHERE jediTaskID=:jediTaskID "
            # loop over all tasks
            for tmpJediTaskID in idMap:
                pandaIDs = idMap[tmpJediTaskID]
                # check useJumbo
                self.conn.begin()
                varMap = {}
                varMap[":jediTaskID"] = tmpJediTaskID
                self.cur.execute(sqlJ + comment, varMap)
                resJ = self.cur.fetchone()
                if resJ is not None and resJ[0] == "D":
                    disabledFlag = True
                    tmpLog.debug(f"kill disabled jumbo jobs for jediTaskID={tmpJediTaskID}")
                else:
                    disabledFlag = False
                if not self._commit():
                    raise RuntimeError("Commit error")
                if jediTaskID is not None or not self.isApplicableTaskForJumbo(tmpJediTaskID) or disabledFlag:
                    for pandaID in pandaIDs:
                        self.killJob(pandaID, "", "55", True)
                    tmpLog.debug(f"killed {len(pandaIDs)} jobs for jediTaskID={tmpJediTaskID}")
            tmpLog.debug("done")
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False

    # get core power
    @memoize
    def get_core_power(self, site_id):
        comment = " /* DBProxy.get_core_power */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" <siteid={site_id}>")
        tmpLog.debug("start")

        sqlS = "SELECT /* use_json_type */ scj.data.corepower FROM ATLAS_PANDA.schedconfig_json scj "
        sqlS += "WHERE panda_queue=:siteid "

        varMap = {":siteid": site_id}

        try:
            self.cur.arraysize = 100
            self.cur.execute(sqlS + comment, varMap)
            resS = self.cur.fetchone()
            core_power = None
            if resS is not None:
                (core_power,) = resS
                core_power = float(core_power)
            tmpLog.debug(f"got {core_power}")
            return core_power, None

        except Exception:
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None, "failed to get corePower"

    # convert ObjID to endpoint
    @memoize
    def convertObjIDtoEndPoint(self, srcFileName, objID):
        comment = " /* DBProxy.convertObjIDtoEndPoint */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" <ID={objID}>")
        tmpLog.debug("start")
        try:
            for srcFile in srcFileName.split(","):
                if not os.path.exists(srcFile):
                    continue
                with open(srcFile) as f:
                    data = json.load(f)
                    for rseName in data:
                        rseData = data[rseName]
                        if objID in [rseData["id"], rseName]:
                            retMap = {
                                "name": rseName,
                                "is_deterministic": rseData["is_deterministic"],
                                "type": rseData["type"],
                            }
                            tmpLog.debug(f"got {str(retMap)}")
                            return retMap
            tmpLog.debug("not found")
        except Exception:
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None

    # bulk fetch PandaIDs
    def bulk_fetch_panda_ids(self, num_ids):
        comment = " /* JediDBProxy.bulk_fetch_panda_ids */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name + f" <num={num_ids}>")
        tmp_log.debug("start")
        try:
            new_ids = []
            var_map = {}
            var_map[":nIDs"] = num_ids
            # sql to get fileID
            sqlFID = "SELECT ATLAS_PANDA.JOBSDEFINED4_PANDAID_SEQ.nextval FROM "
            sqlFID += "(SELECT level FROM dual CONNECT BY level<=:nIDs) "
            # start transaction
            self.conn.begin()
            self.cur.arraysize = 10000
            self.cur.execute(sqlFID + comment, var_map)
            resFID = self.cur.fetchall()
            for (id,) in resFID:
                new_ids.append(id)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmp_log.debug(f"got {len(new_ids)} IDs")
            return sorted(new_ids)
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmp_log, method_name)
            return []

    # bulk fetch fileIDs
    def bulkFetchFileIDsPanda(self, nIDs):
        comment = " /* JediDBProxy.bulkFetchFileIDsPanda */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" <nIDs={nIDs}>")
        tmpLog.debug("start")
        try:
            newFileIDs = []
            varMap = {}
            varMap[":nIDs"] = nIDs
            # sql to get fileID
            sqlFID = "SELECT ATLAS_PANDA.FILESTABLE4_ROW_ID_SEQ.nextval FROM "
            sqlFID += "(SELECT level FROM dual CONNECT BY level<=:nIDs) "
            # start transaction
            self.conn.begin()
            self.cur.arraysize = 10000
            self.cur.execute(sqlFID + comment, varMap)
            resFID = self.cur.fetchall()
            for (fileID,) in resFID:
                newFileIDs.append(fileID)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"got {len(newFileIDs)} IDs")
            return newFileIDs
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmpLog, methodName)
            return []

    # get task status
    def getTaskStatus(self, jediTaskID):
        comment = " /* DBProxy.getTaskStatus */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" <jediTaskID={jediTaskID} >")
        tmpLog.debug("start")
        try:
            # sql to update input file status
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            sql = f"SELECT status FROM {panda_config.schemaJEDI}.JEDI_Tasks "
            sql += "WHERE jediTaskID=:jediTaskID "

            # start transaction
            self.conn.begin()
            self.cur.arraysize = 1000
            self.cur.execute(sql + comment, varMap)
            res = self.cur.fetchone()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            if res:
                tmpLog.debug(f"task {jediTaskID} has status: {res[0]} ")
            else:
                res = []
                tmpLog.debug(f"task {jediTaskID} not found")
            return res
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmpLog, methodName)
            return []

    # reactivate task
    def reactivateTask(self, jediTaskID, keep_attempt_nr=False, trigger_job_generation=False):
        comment = " /* DBProxy.reactivateTask */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID}>"
        tmpLog = LogWrapper(_logger, methodName, monToken=f"<jediTaskID={jediTaskID}>")
        tmpLog.debug("start")
        try:
            # sql to update task status
            sql = f"UPDATE {panda_config.schemaJEDI}.JEDI_Tasks "
            sql += "SET status=:status "
            sql += "WHERE jediTaskID=:jediTaskID "
            # sql to get datasetIDs for master
            sqlM = f"SELECT datasetID FROM {panda_config.schemaJEDI}.JEDI_Datasets "
            sqlM += "WHERE jediTaskID=:jediTaskID AND type IN (:type1,:type2,:type3) "
            # sql to increase attempt numbers and update status
            sqlAB = f"UPDATE {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
            if keep_attempt_nr:
                sqlAB += "SET status=:status "
            else:
                sqlAB += "SET status=:status,attemptNr=0 "
            sqlAB += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
            # sql to update datasets
            sqlD = f"UPDATE {panda_config.schemaJEDI}.JEDI_Datasets "
            sqlD += "SET status=:status,nFilesUsed=0,nFilesTobeUsed=nFiles,nFilesFinished=0,nFilesFailed=0 "
            sqlD += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
            # update task status
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            varMap[":status"] = "ready"
            self.cur.execute(sql + comment, varMap)
            res = self.cur.rowcount
            # get datasetIDs for master
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            varMap[":type1"] = "input"
            varMap[":type2"] = "pseudo_input"
            varMap[":type3"] = "random_seed"
            self.cur.execute(sqlM + comment, varMap)
            resM = self.cur.fetchall()
            total_nFiles = 0

            for (datasetID,) in resM:
                # increase attempt numbers
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":datasetID"] = datasetID
                varMap[":status"] = "ready"

                # update status and attempt number for datasets
                self.cur.execute(sqlAB + comment, varMap)
                nFiles = self.cur.rowcount

                # update dataset
                if nFiles > 0:
                    varMap = {}
                    varMap[":jediTaskID"] = jediTaskID
                    varMap[":datasetID"] = datasetID
                    varMap[":status"] = "ready"
                    tmpLog.debug(sqlD + comment + str(varMap))
                    self.cur.execute(sqlD + comment, varMap)
                    total_nFiles += nFiles

            tmpMsg = f"updated {total_nFiles} inputs and task {jediTaskID} was reactivated "
            tmpLog.debug(tmpMsg)
            tmpLog.sendMsg(tmpMsg, "jedi", "pandasrv")
            retVal = 0, tmpMsg
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # send message
            if trigger_job_generation:
                # message
                msg = srv_msg_utils.make_message("generate_job", taskid=jediTaskID)
                mb_proxy = self.get_mb_proxy("panda_jedi")
                if mb_proxy:
                    mb_proxy.send(msg)
                    tmpLog.debug(f"sent generate_job message: {msg}")
                else:
                    tmpLog.debug("message queue is not configured")
            tmpLog.debug("done")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmpLog, methodName)
            return None, "DB error"

    # get event statistics
    def getEventStat(self, jediTaskID, PandaID):
        comment = " /* DBProxy.getEventStat */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" <jediTaskID={jediTaskID} PandaID={PandaID}>"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            # sql to get event stats
            sql = f"SELECT status,COUNT(*) FROM {panda_config.schemaJEDI}.JEDI_Events "
            sql += "WHERE jediTaskID=:jediTaskID AND PandaID=:PandaID "
            sql += "GROUP BY status "
            # start transaction
            self.conn.begin()
            self.cur.arraysize = 10000
            # get stats
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            varMap[":PandaID"] = PandaID
            self.cur.execute(sql + comment, varMap)
            resM = self.cur.fetchall()
            retMap = {}
            for eventStatus, cnt in resM:
                retMap[eventStatus] = cnt
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"done with {str(retMap)}")
            return retMap
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmpLog, methodName)
            return {}

    def __get_hs_leave_distribution(self, leave_shares):
        """
        Get the current HS06 distribution for running and queued jobs
        """
        comment = " /* DBProxy.get_hs_leave_distribution */"

        sql_hs_distribution = """
            SELECT gshare, jobstatus_grouped, SUM(HS)
            FROM
                (SELECT gshare, HS,
                     CASE
                         WHEN jobstatus IN('activated') THEN 'queued'
                         WHEN jobstatus IN('sent', 'running') THEN 'executing'
                         ELSE 'ignore'
                     END jobstatus_grouped
                 FROM ATLAS_PANDA.JOBS_SHARE_STATS JSS) a
            GROUP BY gshare, jobstatus_grouped
            """

        self.cur.execute(sql_hs_distribution + comment)
        hs_distribution_raw = self.cur.fetchall()

        # get the hs distribution data into a dictionary structure
        hs_distribution_dict = {}
        hs_queued_total = 0
        hs_executing_total = 0
        hs_ignore_total = 0
        for hs_entry in hs_distribution_raw:
            gshare, status_group, hs = hs_entry
            if hs is None:
                continue
            hs = float(hs)
            hs_distribution_dict.setdefault(
                gshare,
                {
                    GlobalShares.PLEDGED: 0,
                    GlobalShares.QUEUED: 0,
                    GlobalShares.EXECUTING: 0,
                },
            )
            hs_distribution_dict[gshare][status_group] = hs
            # calculate totals
            if status_group == GlobalShares.QUEUED:
                hs_queued_total += hs
            elif status_group == GlobalShares.EXECUTING:
                hs_executing_total += hs
            else:
                hs_ignore_total += hs

        # Calculate the ideal HS06 distribution based on shares.
        for share_node in leave_shares:
            share_name, share_value = share_node.name, share_node.value
            hs_pledged_share = hs_executing_total * share_value / 100.0

            hs_distribution_dict.setdefault(
                share_name,
                {
                    GlobalShares.PLEDGED: 0,
                    GlobalShares.QUEUED: 0,
                    GlobalShares.EXECUTING: 0,
                },
            )
            # Pledged HS according to global share definitions
            hs_distribution_dict[share_name]["pledged"] = hs_pledged_share
        return hs_distribution_dict

    # retrieve global shares
    def get_shares(self, parents=""):
        comment = " /* DBProxy.get_shares */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")

        sql = """
               SELECT NAME, VALUE, PARENT, PRODSOURCELABEL, WORKINGGROUP, CAMPAIGN, PROCESSINGTYPE, TRANSPATH, RTYPE,
               VO, QUEUE_ID, THROTTLED
               FROM ATLAS_PANDA.GLOBAL_SHARES
               """
        var_map = None

        if parents == "":
            # Get all shares
            pass
        elif parents is None:
            # Get top level shares
            sql += "WHERE parent IS NULL"

        elif isinstance(parents, str):
            # Get the children of a specific share
            var_map = {":parent": parents}
            sql += "WHERE parent = :parent"

        elif type(parents) in (list, tuple):
            # Get the children of a list of shares
            i = 0
            var_map = {}
            for parent in parents:
                key = f":parent{i}"
                var_map[key] = parent
                i += 1

            parentBindings = ",".join(f":parent{i}" for i in range(len(parents)))
            sql += f"WHERE parent IN ({parentBindings})"

        self.cur.execute(sql + comment, var_map)
        resList = self.cur.fetchall()

        tmpLog.debug("done")
        return resList

    def __reload_shares(self, force=False):
        """
        Reloads the shares from the DB and recalculates distributions
        """

        # Don't reload shares every time
        if (self.__t_update_shares is not None and self.__t_update_shares > datetime.datetime.now() - datetime.timedelta(hours=1)) or force:
            return

        # Root dummy node
        t_before = time.time()
        tree = GlobalShares.Share("root", 100, None, None, None, None, None, None, None, None, None, None)
        t_after = time.time()
        total = t_after - t_before
        _logger.debug(f"Root dummy tree took {total}s")

        # Get top level shares from DB
        t_before = time.time()
        shares_top_level = self.get_shares(parents=None)
        t_after = time.time()
        total = t_after - t_before
        _logger.debug(f"Getting shares took {total}s")

        # Load branches
        t_before = time.time()
        for (
            name,
            value,
            parent,
            prodsourcelabel,
            workinggroup,
            campaign,
            processingtype,
            transpath,
            rtype,
            vo,
            queue_id,
            throttled,
        ) in shares_top_level:
            share = GlobalShares.Share(
                name,
                value,
                parent,
                prodsourcelabel,
                workinggroup,
                campaign,
                processingtype,
                transpath,
                rtype,
                vo,
                queue_id,
                throttled,
            )
            tree.children.append(self.__load_branch(share))
        t_after = time.time()
        total = t_after - t_before
        _logger.debug(f"Loading the branches took {total}s")

        # Normalize the values in the database
        t_before = time.time()
        tree.normalize()
        t_after = time.time()
        total = t_after - t_before
        _logger.debug(f"Normalizing the values took {total}s")

        # get the leave shares (the ones not having more children)
        t_before = time.time()
        leave_shares = tree.get_leaves()
        t_after = time.time()
        total = t_after - t_before
        _logger.debug(f"Getting the leaves took {total}s")

        self.leave_shares = leave_shares
        self.__t_update_shares = datetime.datetime.now()

        # get the distribution of shares
        t_before = time.time()
        # Retrieve the current HS06 distribution of jobs from the database and then aggregate recursively up to the root
        hs_distribution = self.__get_hs_leave_distribution(leave_shares)
        tree.aggregate_hs_distribution(hs_distribution)
        t_after = time.time()
        total = t_after - t_before
        _logger.debug(f"Aggregating the hs distribution took {total}s")

        self.tree = tree
        self.__hs_distribution = hs_distribution
        self.__t_update_distribution = datetime.datetime.now()
        return

    def __reload_hs_distribution(self):
        """
        Reloads the HS distribution
        """

        _logger.debug(self.__t_update_distribution)
        _logger.debug(self.__hs_distribution)
        # Reload HS06s distribution every 10 seconds
        if self.__t_update_distribution is not None and self.__t_update_distribution > datetime.datetime.now() - datetime.timedelta(seconds=10):
            _logger.debug("release")
            return

        # Retrieve the current HS06 distribution of jobs from the database and then aggregate recursively up to the root
        _logger.debug("get dist")
        t_before = time.time()
        hs_distribution = self.__get_hs_leave_distribution(self.leave_shares)
        _logger.debug("aggr dist")
        self.tree.aggregate_hs_distribution(hs_distribution)
        t_after = time.time()
        total = t_after - t_before
        _logger.debug(f"Reloading the hs distribution took {total}s")

        self.__hs_distribution = hs_distribution
        self.__t_update_distribution = datetime.datetime.now()

        # log the distribution for debugging purposes
        _logger.debug(f"Current HS06 distribution is {hs_distribution}")

        return

    def get_sorted_leaves(self):
        """
        Re-loads the shares, then returns the leaves sorted by under usage
        """
        self.__reload_shares()
        self.__reload_hs_distribution()
        return self.tree.sort_branch_by_current_hs_distribution(self.__hs_distribution)

    def get_tree_of_gshare_names(self):
        """
        get nested dict of gshare names implying the tree structure
        """

        def get_nested_gshare(share):
            val = None
            if not share.children:
                # leaf
                pass
            else:
                # branch
                val = {}
                for child in share.children:
                    val[child.name] = get_nested_gshare(child)
            return val

        ret_dict = get_nested_gshare(self.tree)
        return ret_dict

    def __load_branch(self, share):
        """
        Recursively load a branch
        """
        node = GlobalShares.Share(
            share.name,
            share.value,
            share.parent,
            share.prodsourcelabel,
            share.workinggroup,
            share.campaign,
            share.processingtype,
            share.transpath,
            share.rtype,
            share.vo,
            share.queue_id,
            share.throttled,
        )

        children = self.get_shares(parents=share.name)
        if not children:
            return node

        for (
            name,
            value,
            parent,
            prodsourcelabel,
            workinggroup,
            campaign,
            processingtype,
            transpath,
            rtype,
            vo,
            queue_id,
            throttled,
        ) in children:
            child = GlobalShares.Share(
                name,
                value,
                parent,
                prodsourcelabel,
                workinggroup,
                campaign,
                processingtype,
                transpath,
                rtype,
                vo,
                queue_id,
                throttled,
            )
            node.children.append(self.__load_branch(child))

        return node

    # get the resource type of a site
    def get_rtype_site(self, site):
        comment = " /* DBProxy.get_rtype_site */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" <site={site} >")
        tmpLog.debug("start")
        try:
            var_map = {":site": site}
            sql = "SELECT /* use_json_type */ scj.data.resource_type FROM ATLAS_PANDA.schedconfig_json scj WHERE panda_queue=:site "

            # start transaction
            self.conn.begin()
            self.cur.arraysize = 1
            self.cur.execute(sql + comment, var_map)
            rtype = self.cur.fetchone()[0]
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"site {site} has rtype: {rtype} ")
            return rtype
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmpLog, methodName)
            return None

    def compare_share_task(self, share, task):
        """
        Logic to compare the relevant fields of share and task.
        Return: False if some condition does NOT match. True if all conditions match.
        """
        if share.prodsourcelabel is not None and task.prodSourceLabel is not None and re.match(share.prodsourcelabel, task.prodSourceLabel) is None:
            return False

        working_group = task.workingGroup
        if working_group is None:
            working_group = " "

        if share.workinggroup is not None and working_group is not None and re.match(share.workinggroup, working_group) is None:
            return False

        if share.campaign is not None and task.campaign and re.match(share.campaign, task.campaign) is None:
            return False

        if share.processingtype is not None and task.processingType is not None and re.match(share.processingtype, task.processingType) is None:
            return False

        if share.transpath is not None and task.transPath is not None and re.match(share.transpath, task.transPath) is None:
            return False

        if share.rtype is not None and task.site is not None:
            try:
                site = task.site.split(",")[0]  # if task assigned to more than one site, take the first one
                rtype_site = self.get_rtype_site(site)
                if rtype_site and re.match(share.rtype, rtype_site) is None:
                    return False
            except Exception:
                return False

        return True

    def compare_share_job(self, share, job):
        """
        Logic to compare the relevant fields of share and job. It's basically the same as compare_share_task, but
        does not check for the campaign field, which is not part of the job
        """

        if share.prodsourcelabel is not None and re.match(share.prodsourcelabel, job.prodSourceLabel) is None:
            return False

        if share.workinggroup is not None and re.match(share.workinggroup, job.workingGroup) is None:
            return False

        if share.processingtype is not None and re.match(share.processingtype, job.processingType) is None:
            return False

        if share.rtype is not None and job.computingSite is not None:
            try:
                site = job.computingSite.split(",")[0]
                rtype_site = self.get_rtype_site(site)
                if re.match(share.rtype, rtype_site) is None:
                    return False
            except Exception:
                return False

        return True

    def get_share_for_task(self, task):
        """
        Return the share based on a task specification
        """
        self.__reload_shares()
        selected_share_name = "Undefined"

        for share in self.leave_shares:
            if self.compare_share_task(share, task):
                selected_share_name = share.name
                break

        if selected_share_name == "Undefined":
            _logger.warning(
                "No share matching jediTaskId={0} (prodSourceLabel={1} workingGroup={2} campaign={3} transpath={4} site={5})".format(
                    task.jediTaskID,
                    task.prodSourceLabel,
                    task.workingGroup,
                    task.campaign,
                    task.transPath,
                    task.site,
                )
            )

        return selected_share_name

    def get_share_for_job(self, job):
        """
        Return the share based on a job specification
        """
        # special case: esmerge jobs go to Express share
        if job.eventService == EventServiceUtils.esMergeJobFlagNumber:
            return "Express"

        self.__reload_shares()
        selected_share_name = "Undefined"

        for share in self.leave_shares:
            if self.compare_share_job(share, job):
                selected_share_name = share.name
                break

        if selected_share_name == "Undefined":
            _logger.warning(f"No share matching PandaID={job.PandaID} (prodSourceLabel={job.prodSourceLabel} workingGroup={job.workingGroup})")

        return selected_share_name

    def is_valid_share(self, share_name):
        """
        Checks whether the share is a valid leave share
        """
        self.__reload_shares()
        for share in self.leave_shares:
            if share_name == share.name:
                # Share found
                return True

        # Share not found
        return False

    def reassignShare(self, jedi_task_ids, gshare, reassign_running):
        """
        Will reassign all tasks and their jobs that have not yet completed to specified share
        @param jedi_task_ids: task ids
        @param share_dest: dest share
        """

        comment = " /* DBProxy.reassignShare */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")

        try:
            if not self.is_valid_share(gshare):
                error_msg = f"Share {gshare} is not a leave share "
                tmp_log.debug(error_msg)
                ret_val = 1, error_msg
                tmp_log.debug(error_msg)
                return ret_val

            # begin transaction
            self.conn.begin()

            # update in shards of 100 task ids
            for shard in create_shards(jedi_task_ids, 100):
                # Prepare the bindings
                var_map = {}
                shard_taskid_set = set()
                i = 0
                for _task_id in shard:
                    jedi_task_id = int(_task_id)
                    var_map[f":jtid{i}"] = jedi_task_id
                    i += 1
                    shard_taskid_set.add(jedi_task_id)
                jtid_bindings = ",".join(f":jtid{i}" for i in range(len(shard_taskid_set)))

                # select only tasks without lock
                sql_tasks_not_locked = f"""
                       SELECT jediTaskID FROM ATLAS_PANDA.JEDI_Tasks
                       WHERE jediTaskID IN ({jtid_bindings}) AND lockedBy IS NULL
                       """

                self.cur.execute(sql_tasks_not_locked + comment, var_map)
                res = self.cur.fetchall()

                # Update the bindings and prepare var map
                var_map = {":gshare": gshare}
                good_taskid_set = set()
                i = 0
                for (_task_id,) in res:
                    jedi_task_id = int(_task_id)
                    var_map[f":jtid{i}"] = jedi_task_id
                    i += 1
                    good_taskid_set.add(jedi_task_id)
                jtid_bindings = ",".join(f":jtid{i}" for i in range(len(good_taskid_set)))
                locked_taskid_set = shard_taskid_set - good_taskid_set
                if locked_taskid_set:
                    tmp_log.debug(f"skip locked tasks: {','.join([str(i) for i in locked_taskid_set])}")

                # update the task
                sql_task = f"""
                       UPDATE ATLAS_PANDA.jedi_tasks set gshare=:gshare
                       WHERE jeditaskid IN ({jtid_bindings})
                       """

                self.cur.execute(sql_task + comment, var_map)
                tmp_log.debug(f"""set tasks {",".join([str(i) for i in good_taskid_set])} to gshare={gshare}""")

                var_map[":pending"] = "pending"
                var_map[":defined"] = "defined"
                var_map[":assigned"] = "assigned"
                var_map[":waiting"] = "waiting"
                var_map[":activated"] = "activated"
                jobstatus = ":pending, :defined, :assigned, :waiting, :activated"

                # add running status in case these jobs also need to be reassigned
                if reassign_running:
                    jobstatus = f"{jobstatus}, :running, :starting"
                    var_map[":running"] = "running"
                    var_map[":starting"] = "starting"

                # update the jobs
                sql_jobs = """
                       UPDATE ATLAS_PANDA.{0} set gshare=:gshare
                       WHERE jeditaskid IN ({1})
                       AND jobstatus IN ({2})
                       """

                for table in ["jobsactive4", "jobswaiting4", "jobsdefined4"]:
                    self.cur.execute(
                        sql_jobs.format(table, jtid_bindings, jobstatus) + comment,
                        var_map,
                    )

            # commit
            if not self._commit():
                raise RuntimeError("Commit error")

            tmp_log.debug("done")
            return 0, None

        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            tmp_log.error(f"reassignShare : {type} {value}")
            return -1, None

    def getCommands(self, harvester_id, n_commands):
        """
        Gets n commands in status 'new' for a particular harvester instance and updates their status to 'retrieved'
        """

        comment = " /* DBProxy.getCommands */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name + f" < harvesterID={harvester_id} >")
        tmp_log.debug("start")

        try:
            self.conn.begin()
            # Prepare the bindings and var map to get the oldest n commands in 'new' status
            var_map = {
                ":harvester_id": harvester_id,
                ":n_commands": n_commands,
                ":status": "new",
            }

            sql = None
            if self.backend in ["oracle", "postgres"]:
                sql = """
                      SELECT command_id, command, params, ack_requested, creation_date FROM
                          (SELECT command_id, command, params, ack_requested, creation_date FROM ATLAS_PANDA.HARVESTER_COMMANDS
                              WHERE harvester_id=:harvester_id AND status=:status
                              ORDER BY creation_date) a
                      WHERE ROWNUM <= :n_commands
                      """
            else:
                sql = """
                      SELECT command_id, command, params, ack_requested, creation_date FROM (SELECT (@rownum:=@rownum+1) AS ROWNUM, command_id, command, params, ack_requested, creation_date FROM
                          (SELECT command_id, command, params, ack_requested, creation_date FROM ATLAS_PANDA.HARVESTER_COMMANDS
                              WHERE harvester_id=:harvester_id AND status=:status
                              ORDER BY creation_date) a, (SELECT @rownum:=0) r ) comm
                      WHERE ROWNUM <= :n_commands
                      """
            self.cur.execute(sql + comment, var_map)
            entries = self.cur.fetchall()
            tmp_log.debug(f"entries {entries}")

            # pack the commands into a dictionary for transmission to harvester
            commands = []
            command_ids = []
            for entry in entries:
                command_dict = {}
                command_dict["command_id"] = entry[0]
                command_ids.append(entry[0])  # we need to update the commands as dispatched
                command_dict["command"] = entry[1]
                command_dict["params"] = entry[2]
                if command_dict["params"] is not None:
                    try:
                        theParams = command_dict["params"].read()
                    except AttributeError:
                        theParams = command_dict["params"]
                    command_dict["params"] = json.loads(theParams)
                command_dict["ack_requested"] = entry[3]
                command_dict["creation_date"] = entry[4].isoformat()
                commands.append(command_dict)

            tmp_log.debug(f"commands {commands}")
            tmp_log.debug(f"command_ids {command_ids}")

            if command_ids:
                # update the commands and set them as retrieved
                # Prepare the bindings and var map
                var_map = {":retrieved": "retrieved"}
                for i, command_id in enumerate(command_ids):
                    var_map[f":command_id{i}"] = command_id
                command_id_bindings = ",".join(f":command_id{i}" for i in range(len(command_ids)))

                sql = f"""
                      UPDATE ATLAS_PANDA.HARVESTER_COMMANDS
                      SET status=:retrieved,status_date=CURRENT_DATE
                      WHERE command_id IN({command_id_bindings})
                      """

                # run the update
                self.cur.execute(sql + comment, var_map)

            if not self._commit():
                raise RuntimeError("Commit error")

            tmp_log.debug("done")
            return 0, commands

        except Exception:
            self._rollback()
            self.dumpErrorMessage(tmp_log, method_name)
            return -1, []

    def ackCommands(self, command_ids):
        """
        Sets the commands to acknowledged
        """
        comment = " /* DBProxy.ackCommands */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")

        try:
            # Prepare the bindings and var map
            var_map = {":acknowledged": "acknowledged"}
            for i, command_id in enumerate(command_ids):
                var_map[f":command_id{i}"] = command_id
            command_id_bindings = ",".join(f":command_id{i}" for i in range(len(command_ids)))

            sql = f"""
                  UPDATE ATLAS_PANDA.HARVESTER_COMMANDS
                  SET status=:acknowledged,status_date=CURRENT_DATE
                  WHERE command_id IN({command_id_bindings})
                  """

            # run the update
            self.conn.begin()
            self.cur.execute(sql + comment, var_map)
            if not self._commit():
                raise RuntimeError("Commit error")

            return 0

        except Exception:
            # roll back
            self._rollback()
            type, value, traceback = sys.exc_info()
            _logger.error(f"{comment}: {sql} {var_map}")
            _logger.error(f"{comment}: {type} {value}")
            return -1

    # update workers
    def updateWorkers(self, harvesterID, data, useCommit=True):
        """
        Update workers
        """
        comment = " /* DBProxy.updateWorkers */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(
            _logger,
            methodName + f" < HarvesterID={harvesterID} pid={os.getpid()} >",
        )
        try:
            tmpLog.debug(f"start {len(data)} workers")
            regStart = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            sqlC = f"SELECT {WorkerSpec.columnNames()} FROM ATLAS_PANDA.Harvester_Workers "
            sqlC += "WHERE harvesterID=:harvesterID AND workerID=:workerID "
            # loop over all workers
            retList = []
            for workerData in data:
                timeNow = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
                if useCommit:
                    self.conn.begin()
                workerSpec = WorkerSpec()
                workerSpec.harvesterID = harvesterID
                workerSpec.workerID = workerData["workerID"]
                tmpLog.debug(f"workerID={workerSpec.workerID} start")
                # check if already exists
                varMap = dict()
                varMap[":harvesterID"] = workerSpec.harvesterID
                varMap[":workerID"] = workerSpec.workerID
                self.cur.execute(sqlC + comment, varMap)
                resC = self.cur.fetchone()
                if resC is None:
                    # not exist
                    toInsert = True
                    oldLastUpdate = None
                else:
                    # already exists
                    toInsert = False
                    workerSpec.pack(resC)
                    oldLastUpdate = workerSpec.lastUpdate
                # set new values
                oldStatus = workerSpec.status
                for key in workerData:
                    val = workerData[key]
                    if hasattr(workerSpec, key):
                        setattr(workerSpec, key, val)
                workerSpec.lastUpdate = timeNow
                if oldStatus in ["finished", "failed", "cancelled", "missed"] and (
                    oldLastUpdate is not None and oldLastUpdate > timeNow - datetime.timedelta(hours=3)
                ):
                    tmpLog.debug(f"workerID={workerSpec.workerID} keep old status={oldStatus} instead of new {workerSpec.status}")
                    workerSpec.status = oldStatus
                # insert or update
                if toInsert:
                    # insert
                    tmpLog.debug(f"workerID={workerSpec.workerID} insert for status={workerSpec.status}")
                    sqlI = f"INSERT INTO ATLAS_PANDA.Harvester_Workers ({WorkerSpec.columnNames()}) "
                    sqlI += WorkerSpec.bindValuesExpression()
                    varMap = workerSpec.valuesMap()
                    self.cur.execute(sqlI + comment, varMap)
                else:
                    # update
                    tmpLog.debug(f"workerID={workerSpec.workerID} update for status={workerSpec.status}")
                    sqlU = f"UPDATE ATLAS_PANDA.Harvester_Workers SET {workerSpec.bindUpdateChangesExpression()} "
                    sqlU += "WHERE harvesterID=:harvesterID AND workerID=:workerID "
                    varMap = workerSpec.valuesMap(onlyChanged=True)
                    self.cur.execute(sqlU + comment, varMap)
                # job relation
                if "pandaid_list" in workerData and len(workerData["pandaid_list"]) > 0:
                    tmpLog.debug(f"workerID={workerSpec.workerID} update/insert job relation")
                    sqlJC = "SELECT PandaID FROM ATLAS_PANDA.Harvester_Rel_Jobs_Workers "
                    sqlJC += "WHERE harvesterID=:harvesterID AND workerID=:workerID "
                    sqlJI = "INSERT INTO ATLAS_PANDA.Harvester_Rel_Jobs_Workers (harvesterID,workerID,PandaID,lastUpdate) "
                    sqlJI += "VALUES (:harvesterID,:workerID,:PandaID,:lastUpdate) "
                    sqlJU = "UPDATE ATLAS_PANDA.Harvester_Rel_Jobs_Workers SET lastUpdate=:lastUpdate "
                    sqlJU += "WHERE harvesterID=:harvesterID AND workerID=:workerID AND PandaID=:PandaID "
                    # get jobs
                    varMap = dict()
                    varMap[":harvesterID"] = harvesterID
                    varMap[":workerID"] = workerData["workerID"]
                    self.cur.execute(sqlJC + comment, varMap)
                    resJC = self.cur.fetchall()
                    exPandaIDs = set()
                    for (pandaID,) in resJC:
                        exPandaIDs.add(pandaID)
                    for pandaID in workerData["pandaid_list"]:
                        # update or insert
                        varMap = dict()
                        varMap[":harvesterID"] = harvesterID
                        varMap[":workerID"] = workerData["workerID"]
                        varMap[":PandaID"] = pandaID
                        varMap[":lastUpdate"] = timeNow
                        if pandaID not in exPandaIDs:
                            # insert
                            self.cur.execute(sqlJI + comment, varMap)
                        else:
                            # update
                            self.cur.execute(sqlJU + comment, varMap)
                            exPandaIDs.discard(pandaID)
                    # delete redundant list
                    sqlJD = "DELETE FROM ATLAS_PANDA.Harvester_Rel_Jobs_Workers "
                    sqlJD += "WHERE harvesterID=:harvesterID AND workerID=:workerID AND PandaID=:PandaID "
                    for pandaID in exPandaIDs:
                        varMap = dict()
                        varMap[":PandaID"] = pandaID
                        varMap[":harvesterID"] = harvesterID
                        varMap[":workerID"] = workerData["workerID"]
                        self.cur.execute(sqlJD + comment, varMap)
                    tmpLog.debug(f"workerID={workerSpec.workerID} deleted {len(exPandaIDs)} jobs")
                # comprehensive heartbeat
                tmpLog.debug(f"workerID={workerSpec.workerID} get jobs")
                sqlCJ = "SELECT r.PandaID FROM "
                sqlCJ += "ATLAS_PANDA.Harvester_Rel_Jobs_Workers r,ATLAS_PANDA.jobsActive4 j  "
                sqlCJ += "WHERE r.harvesterID=:harvesterID AND r.workerID=:workerID "
                sqlCJ += "AND j.PandaID=r.PandaID AND NOT j.jobStatus IN (:holding) "
                sqlJAC = "SELECT jobStatus,prodSourceLabel,attemptNr FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID  "
                sqlJAA = "UPDATE ATLAS_PANDA.jobsActive4 SET modificationTime=CURRENT_DATE WHERE PandaID=:PandaID AND jobStatus IN (:js1,:js2) "
                sqlJAE = "UPDATE ATLAS_PANDA.jobsActive4 SET taskBufferErrorCode=:code,taskBufferErrorDiag=:diag,"
                sqlJAE += "startTime=(CASE WHEN jobStatus=:starting THEN NULL ELSE startTime END) "
                sqlJAE += "WHERE PandaID=:PandaID "
                sqlJSE = "UPDATE {0} SET supErrorCode=:code,supErrorDiag=:diag,stateChangeTime=CURRENT_DATE "
                sqlJSE += "WHERE PandaID=:PandaID AND NOT jobStatus IN (:finished) AND modificationTime>CURRENT_DATE-30"
                varMap = dict()
                varMap[":harvesterID"] = harvesterID
                varMap[":workerID"] = workerData["workerID"]
                varMap[":holding"] = "holding"
                self.cur.execute(sqlCJ + comment, varMap)
                resCJ = self.cur.fetchall()
                tmpLog.debug(f"workerID={workerSpec.workerID} update {len(resCJ)} jobs")
                for (pandaID,) in resCJ:
                    # check job status when worker is in a final state
                    if workerSpec.status in [
                        "finished",
                        "failed",
                        "cancelled",
                        "missed",
                    ]:
                        varMap = dict()
                        varMap[":PandaID"] = pandaID
                        self.cur.execute(sqlJAC + comment, varMap)
                        resJAC = self.cur.fetchone()
                        if resJAC is not None:
                            jobStatus, prodSourceLabel, attemptNr = resJAC
                            tmpLog.debug(f"workerID={workerSpec.workerID} {workerSpec.status} while PandaID={pandaID} {jobStatus}")
                            # set failed if out of sync
                            if "syncLevel" in workerData and workerData["syncLevel"] == 1 and jobStatus in ["running", "starting"]:
                                tmpLog.debug(f"workerID={workerSpec.workerID} set failed to PandaID={pandaID} due to sync error")
                                varMap = dict()
                                varMap[":PandaID"] = pandaID
                                varMap[":code"] = ErrorCode.EC_WorkerDone
                                varMap[":starting"] = "starting"
                                varMap[":diag"] = f"The worker was {workerSpec.status} while the job was {jobStatus} : {workerSpec.diagMessage}"
                                varMap[":diag"] = JobSpec.truncateStringAttr("taskBufferErrorDiag", varMap[":diag"])
                                self.cur.execute(sqlJAE + comment, varMap)
                                # make an empty file to triggre registration for zip files in Adder
                                # tmpFileName = '{0}_{1}_{2}'.format(pandaID, 'failed',
                                #                                    uuid.uuid3(uuid.NAMESPACE_DNS,''))
                                # tmpFileName = os.path.join(panda_config.logdir, tmpFileName)
                                # try:
                                #     open(tmpFileName, 'w').close()
                                # except Exception:
                                #     pass
                                # sql to insert empty job output report for adder
                                sqlI = (
                                    "INSERT INTO {0}.Job_Output_Report "
                                    "(PandaID, prodSourceLabel, jobStatus, attemptNr, data, timeStamp) "
                                    "VALUES(:PandaID, :prodSourceLabel, :jobStatus, :attemptNr, :data, :timeStamp) "
                                ).format(panda_config.schemaPANDA)
                                # insert
                                varMap = {}
                                varMap[":PandaID"] = pandaID
                                varMap[":prodSourceLabel"] = prodSourceLabel
                                varMap[":jobStatus"] = "failed"
                                varMap[":attemptNr"] = attemptNr
                                varMap[":data"] = None
                                varMap[":timeStamp"] = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
                                try:
                                    self.cur.execute(sqlI + comment, varMap)
                                except Exception:
                                    pass
                                else:
                                    tmpLog.debug(f"successfully inserted job output report {pandaID}.{varMap[':attemptNr']}")
                        if workerSpec.errorCode not in [None, 0]:
                            varMap = dict()
                            varMap[":PandaID"] = pandaID
                            varMap[":code"] = workerSpec.errorCode
                            varMap[":diag"] = f"Diag from worker : {workerSpec.diagMessage}"
                            varMap[":diag"] = JobSpec.truncateStringAttr("supErrorDiag", varMap[":diag"])
                            varMap[":finished"] = "finished"
                            for tableName in [
                                "ATLAS_PANDA.jobsActive4",
                                "ATLAS_PANDA.jobsArchived4",
                                "ATLAS_PANDAARCH.jobsArchived",
                            ]:
                                self.cur.execute(sqlJSE.format(tableName) + comment, varMap)
                    """
                    varMap = dict()
                    varMap[':PandaID'] = pandaID
                    varMap[':js1'] = 'running'
                    varMap[':js2'] = 'starting'
                    self.cur.execute(sqlJAA+comment, varMap)
                    nRowJA = self.cur.rowcount
                    if nRowJA > 0:
                        tmpLog.debug('workerID={0} PandaID={1} updated modificationTime'.format(workerSpec.workerID, pandaID))
                    """
                tmpLog.debug(f"workerID={workerSpec.workerID} end")
                # commit
                if useCommit:
                    if not self._commit():
                        raise RuntimeError("Commit error")
                retList.append(True)
            regTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - regStart
            tmpLog.debug("done. exec_time=%s.%03d sec" % (regTime.seconds, regTime.microseconds / 1000))
            return retList
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            self.dumpErrorMessage(tmpLog, methodName)
            return None

    # update workers
    def updateServiceMetrics(self, harvesterID, data):
        """
        Update service metrics
        """
        comment = " /* DBProxy.updateServiceMetrics */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" < HarvesterID={harvesterID} >")
        try:
            # generate the SQL to insert metrics into the DB
            sql = f"INSERT INTO ATLAS_PANDA.harvester_Metrics ({HarvesterMetricsSpec.columnNames()}) "
            sql += HarvesterMetricsSpec.bindValuesExpression()

            # generate the entries for the DB
            var_maps = []
            for entry in data:
                tmpLog.debug(f"entry {entry}")
                metrics_spec = HarvesterMetricsSpec()
                metrics_spec.harvester_ID = harvesterID
                metrics_spec.creation_time = datetime.datetime.strptime(entry[0], "%Y-%m-%d %H:%M:%S.%f")
                metrics_spec.harvester_host = entry[1]
                metrics_spec.metrics = entry[2]

                var_maps.append(metrics_spec.valuesMap())

            # run the SQL
            self.cur.executemany(sql + comment, var_maps)
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug("done")
            return [True]
        except Exception:
            # roll back
            self._rollback()
            self.dumpErrorMessage(tmpLog, methodName)
            return None

    # heartbeat for harvester
    def harvesterIsAlive(self, user, host, harvesterID, data):
        """
        update harvester instance information
        """
        comment = " /* DBProxy.harvesterIsAlive */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" < harvesterID={harvesterID} >")
        tmpLog.debug("start")
        try:
            # update
            varMap = dict()
            varMap[":harvesterID"] = harvesterID
            owner = self.cleanUserID(user)
            varMap[":owner"] = owner
            varMap[":hostName"] = host
            sqlC = "UPDATE ATLAS_PANDA.Harvester_Instances SET owner=:owner,hostName=:hostName,lastUpdate=CURRENT_DATE"
            for tmpKey in data:
                tmpVal = data[tmpKey]
                if tmpKey == "commands":
                    continue
                sqlC += ",{0}=:{0}".format(tmpKey)
                if isinstance(tmpVal, str) and tmpVal.startswith("datetime/"):
                    tmpVal = datetime.datetime.strptime(tmpVal.split("/")[-1], "%Y-%m-%d %H:%M:%S.%f")
                varMap[f":{tmpKey}"] = tmpVal
            sqlC += " WHERE harvester_ID=:harvesterID "
            # exec
            self.conn.begin()
            self.cur.execute(sqlC + comment, varMap)
            nRow = self.cur.rowcount
            if nRow == 0:
                # insert instance info
                varMap = dict()
                varMap[":harvesterID"] = harvesterID
                varMap[":owner"] = owner
                varMap[":hostName"] = host
                varMap[":descr"] = "automatic"
                sqlI = (
                    "INSERT INTO ATLAS_PANDA.Harvester_Instances "
                    "(harvester_ID,owner,hostName,lastUpdate,description) "
                    "VALUES(:harvesterID,:owner,:hostName,CURRENT_DATE,:descr) "
                )
                self.cur.execute(sqlI + comment, varMap)
            # insert command locks
            if "commands" in data:
                for item in data["commands"]:
                    self.addCommandLockHarvester(
                        harvesterID,
                        item["command"],
                        item["computingSite"],
                        item["resourceType"],
                        False,
                    )
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug("done")
            if nRow == 0:
                retStr = "no instance record"
            else:
                retStr = "succeeded"
            return retStr
        except Exception:
            # roll back
            self._rollback()
            self.dumpErrorMessage(tmpLog, methodName)
            return None

    def storePilotLog(self, panda_id, pilot_log):
        """
        Stores the pilotlog in the pandalog table
        """
        comment = " /* DBProxy.storePilotLog */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug(f"start {panda_id} {pilot_log}")

        try:
            # Prepare the bindings and var map
            var_map = {
                ":panda_id": panda_id,
                ":message": pilot_log[:4000],  # clip if longer than 4k characters
                ":now": datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None),
                ":name": "panda.mon.prod",
                ":module": "JobDispatcher",
                ":type": "pilotLog",
                ":file_name": "JobDispatcher.py",
                ":log_level": 20,
                ":level_name": "INFO",
            }

            sql = """
                  INSERT INTO ATLAS_PANDA.PANDALOG (BINTIME, NAME, MODULE, TYPE, PID, LOGLEVEL, LEVELNAME,
                                                    TIME, FILENAME, MESSAGE)
                  VALUES (:now, :name, :module, :type, :panda_id, :log_level, :level_name,
                          :now, :file_name, :message)
                  """

            # run the insert
            self.conn.begin()
            self.cur.execute(sql + comment, var_map)
            if not self._commit():
                raise RuntimeError("Commit error")

            return 0

        except Exception:
            # roll back
            self._rollback()
            type, value, traceback = sys.exc_info()
            _logger.error(f"{comment}: {sql} {var_map}")
            _logger.error(f"{comment}: {type} {value}")
            return -1

    def __reload_resource_spec_mapper(self):
        # update once per hour only
        if self.__t_update_resource_type_mapper and self.__t_update_resource_type_mapper > datetime.datetime.now() - datetime.timedelta(hours=1):
            return

        # get the resource types from the DB and make the ResourceSpecMapper object
        resource_types = self.load_resource_types()
        if resource_types:
            self.__resource_spec_mapper = ResourceSpecMapper(resource_types)
            self.__t_update_resource_type_mapper = datetime.datetime.now()
        return

    def load_resource_types(self, formatting="spec"):
        """
        Load the resource type table to memory
        """
        comment = " /* JediDBProxy.load_resource_types */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        try:
            sql = f"SELECT {ResourceSpec.column_names()} FROM {panda_config.schemaJEDI}.resource_types "
            self.cur.execute(sql + comment)
            resource_list = self.cur.fetchall()
            resource_spec_list = []
            for row in resource_list:
                resource_name, mincore, maxcore, minrampercore, maxrampercore = row
                if formatting == "dict":
                    resource_dict = {
                        "resource_name": resource_name,
                        "mincore": mincore,
                        "maxcore": maxcore,
                        "minrampercore": minrampercore,
                        "maxrampercore": maxrampercore,
                    }
                    resource_spec_list.append(resource_dict)
                else:
                    resource_spec_list.append(
                        ResourceSpec(
                            resource_name,
                            mincore,
                            maxcore,
                            minrampercore,
                            maxrampercore,
                        )
                    )

            tmp_log.debug("done")
            return resource_spec_list
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error(f"{comment}: {sql}")
            return []

    def get_resource_type_task(self, task_spec):
        """
        Identify the resource type of the task based on the resource type map.
        Return the name of the resource type
        """
        comment = " /* JediDBProxy.get_resource_type_task */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")

        resource_map = self.load_resource_types()

        for resource_spec in resource_map:
            if resource_spec.match_task(task_spec):
                tmp_log.debug(f"done. resource_type is {resource_spec.resource_name}")
                return resource_spec.resource_name

        tmp_log.debug("done. resource_type is Undefined")
        return "Undefined"

    def reset_resource_type_task(self, jedi_task_id, use_commit=True):
        """
        Retrieve the relevant task parameters and reset the resource type
        """
        comment = " /* JediDBProxy.reset_resource_type */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")

        # 1. Get the task parameters
        var_map = {":jedi_task_id": jedi_task_id}
        sql = f"SELECT corecount, ramcount, baseramcount, ramunit FROM {panda_config.schemaJEDI}.jedi_tasks WHERE jeditaskid = :jedi_task_id "
        self.cur.execute(sql + comment, var_map)
        corecount, ramcount, baseramcount, ramunit = self.cur.fetchone()
        tmp_log.debug(
            "retrieved following values for jediTaskid {0}: corecount {1}, ramcount {2}, baseramcount {3}, ramunit {4}".format(
                jedi_task_id, corecount, ramcount, baseramcount, ramunit
            )
        )

        # 2. Load the resource types and figure out the matching one
        resource_map = self.load_resource_types()
        resource_name = "Undefined"
        for resource_spec in resource_map:
            if resource_spec.match_task_basic(corecount, ramcount, baseramcount, ramunit):
                resource_name = resource_spec.resource_name
                break

        tmp_log.debug(f"decided resource_type {resource_name} jediTaskid {jedi_task_id}")

        # 3. Update the task
        try:
            var_map = {":jedi_task_id": jedi_task_id, ":resource_type": resource_name}
            sql = f"UPDATE {panda_config.schemaJEDI}.JEDI_Tasks SET resource_type = :resource_type WHERE jeditaskid = :jedi_task_id "
            tmp_log.debug("conn begin...")
            if use_commit:
                self.conn.begin()
            tmp_log.debug("execute...")
            self.cur.execute(sql + comment, var_map)
            tmp_log.debug("commit...")
            if use_commit:
                if not self._commit():
                    raise RuntimeError("Commit error")
                tmp_log.debug("commited...")
        except Exception:
            # roll back
            if use_commit:
                tmp_log.debug("rolling back...")
                self._rollback()
            type, value, traceback = sys.exc_info()
            _logger.error(f"{comment}: {sql} {var_map}")
            _logger.error(f"{comment}: {type} {value}")
            return False

        tmp_log.debug("done")
        return True

    def get_resource_type_job(self, job_spec):
        """
        Identify the resource type of the job based on the resource type map.
        Return the name of the resource type
        """
        comment = " /* JediDBProxy.get_resource_type_job */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")

        resource_map = self.load_resource_types()
        tmp_log.debug(
            "going to call match_job for pandaid {0} with minRamCount {1} (type{2}) and coreCount {3} (type{4})".format(
                job_spec.PandaID,
                job_spec.minRamCount,
                type(job_spec.minRamCount),
                job_spec.coreCount,
                type(job_spec.coreCount),
            )
        )
        resource_type = JobUtils.get_resource_type_job(resource_map, job_spec)
        tmp_log.debug(f"done. resource_type is {resource_type}")
        return resource_type

    # update stat of workers with jobtype breakdown
    def reportWorkerStats_jobtype(self, harvesterID, siteName, parameter_list):
        comment = " /* DBProxy.reportWorkerStats_jobtype */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(
            _logger,
            f"{method_name} < harvesterID={harvesterID} siteName={siteName} >",
        )
        tmp_log.debug("start")
        tmp_log.debug(f"params={str(parameter_list)}")
        try:
            # load new site data
            parameter_list = json.loads(parameter_list)
            # set autocommit on
            self.conn.begin()

            # lock the site data rows
            var_map = dict()
            var_map[":harvesterID"] = harvesterID
            var_map[":siteName"] = siteName
            sql_lock = "SELECT harvester_ID, computingSite FROM ATLAS_PANDA.Harvester_Worker_Stats "
            sql_lock += "WHERE harvester_ID=:harvesterID AND computingSite=:siteName FOR UPDATE NOWAIT "
            try:
                self.cur.execute(sql_lock + comment, var_map)
            except Exception:
                self._rollback()
                message = "rows locked by another update"
                tmp_log.debug(message)
                tmp_log.debug("done")
                return False, message

            # delete them
            sql_delete = "DELETE FROM ATLAS_PANDA.Harvester_Worker_Stats "
            sql_delete += "WHERE harvester_ID=:harvesterID AND computingSite=:siteName "
            self.cur.execute(sql_delete + comment, var_map)

            # insert new site data
            sql_insert = "INSERT INTO ATLAS_PANDA.Harvester_Worker_Stats (harvester_ID, computingSite, jobType, resourceType, status, n_workers, lastUpdate) "
            sql_insert += "VALUES (:harvester_ID, :siteName, :jobType, :resourceType, :status, :n_workers, CURRENT_DATE) "

            var_map_list = []
            for jobType in parameter_list:
                jt_params = parameter_list[jobType]
                for resourceType in jt_params:
                    params = jt_params[resourceType]
                    if resourceType == "Undefined":
                        continue
                    for status in params:
                        n_workers = params[status]
                        var_map = {
                            ":harvester_ID": harvesterID,
                            ":siteName": siteName,
                            ":status": status,
                            ":jobType": jobType,
                            ":resourceType": resourceType,
                            ":n_workers": n_workers,
                        }
                        var_map_list.append(var_map)

            self.cur.executemany(sql_insert + comment, var_map_list)

            if not self._commit():
                raise RuntimeError("Commit error")

            tmp_log.debug("done")
            return True, "OK"
        except Exception as e:
            self._rollback()
            self.dumpErrorMessage(tmp_log, method_name)
            return False, "database error"

    # get stat of workers
    def getWorkerStats(self):
        comment = " /* DBProxy.getWorkerStats */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger)
        tmpLog.debug("start")
        try:
            # set autocommit on
            self.conn.begin()
            # sql to get nPilot
            # sqlP = ("SELECT getJob+updateJob FROM ATLAS_PANDAMETA.SiteData "
            #         "WHERE HOURS=:hours AND FLAG IN (:flag1,:flag2) ")
            # varMap = dict()
            # varMap[":hours"] = 1
            # varMap[":flag1"] = "production"
            # varMap[":flag2"] = "analysis"
            # self.cur.execute(sqlP + comment, varMap)
            # res = self.cur.fetchone()
            # if res is not None:
            #     (nPilot,) = res
            # else:
            #     nPilot = 0
            # sql to get stat of workers
            sqlGA = (
                "SELECT SUM(n_workers), computingSite, harvester_ID, jobType, resourceType, status "
                "FROM ATLAS_PANDA.Harvester_Worker_Stats "
                "WHERE lastUpdate>=:time_limit "
                "GROUP BY computingSite,harvester_ID,jobType,resourceType,status "
            )
            varMap = dict()
            varMap[":time_limit"] = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(hours=4)
            self.cur.execute(sqlGA + comment, varMap)
            res_active = self.cur.fetchall()
            retMap = {}
            for cnt, computingSite, harvesterID, jobType, resourceType, status in res_active:
                retMap.setdefault(computingSite, {})
                retMap[computingSite].setdefault(harvesterID, {})
                retMap[computingSite][harvesterID].setdefault(jobType, {})
                if resourceType not in retMap[computingSite][harvesterID][jobType]:
                    retMap[computingSite][harvesterID][jobType][resourceType] = dict()
                retMap[computingSite][harvesterID][jobType][resourceType][status] = cnt
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # return
            tmpLog.debug(f"done with {str(retMap)}")
            return retMap
        except Exception:
            # roll back
            self._rollback()
            self.dumpErrorMessage(tmpLog, methodName)
            return {}

    # send command to harvester or lock command
    def commandToHarvester(
        self,
        harvester_ID,
        command,
        ack_requested,
        status,
        lockInterval,
        comInterval,
        params,
        useCommit=True,
    ):
        comment = " /* DBProxy.commandToHarvester */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(
            _logger,
            methodName + f" < harvesterID={harvester_ID} command={command} >",
        )
        tmpLog.debug("start")
        tmpLog.debug(f"params={str(params)}")
        try:
            if useCommit:
                self.conn.begin()
            # check if command exists
            sqlC = "SELECT status,status_date FROM ATLAS_PANDA.HARVESTER_COMMANDS "
            sqlC += "WHERE harvester_ID=:harvester_ID AND command=:command "
            varMap = dict()
            varMap[":harvester_ID"] = harvester_ID
            varMap[":command"] = command
            self.cur.execute(sqlC + comment, varMap)
            resC = self.cur.fetchone()
            # check existing command
            toSkip = False
            if resC is not None:
                commandStatus, statusDate = resC
                # not overwrite existing command
                if (
                    commandStatus in ["new", "lock", "retrieved"]
                    and lockInterval is not None
                    and statusDate > datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(minutes=lockInterval)
                ):
                    toSkip = True
                elif (
                    commandStatus in ["retrieved", "acknowledged"]
                    and comInterval is not None
                    and statusDate > datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(minutes=comInterval)
                ):
                    toSkip = True
                else:
                    # delete existing command
                    sqlD = "DELETE FROM ATLAS_PANDA.HARVESTER_COMMANDS "
                    sqlD += "WHERE harvester_ID=:harvester_ID AND command=:command "
                    varMap = dict()
                    varMap[":harvester_ID"] = harvester_ID
                    varMap[":command"] = command
                    self.cur.execute(sqlD + comment, varMap)
            # insert
            if not toSkip:
                varMap = dict()
                varMap[":harvester_ID"] = harvester_ID
                varMap[":command"] = command
                varMap[":ack_requested"] = ack_requested
                varMap[":status"] = status
                sqlI = "INSERT INTO ATLAS_PANDA.HARVESTER_COMMANDS "
                sqlI += "(command_id,creation_date,status_date,command,harvester_id,ack_requested,status"
                if params is not None:
                    varMap[":params"] = json.dumps(params)
                    sqlI += ",params"
                sqlI += ") "
                sqlI += "VALUES (ATLAS_PANDA.HARVESTER_COMMAND_ID_SEQ.nextval,CURRENT_DATE,CURRENT_DATE,:command,:harvester_id,:ack_requested,:status"
                if params is not None:
                    sqlI += ",:params"
                sqlI += ") "
                self.cur.execute(sqlI + comment, varMap)
            if useCommit:
                if not self._commit():
                    raise RuntimeError("Commit error")
            tmpLog.debug("done")
            if toSkip:
                return False
            return True
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            self.dumpErrorMessage(tmpLog, methodName)
            return False

    # check Job status
    def checkJobStatus(self, pandaID):
        comment = " /* DBProxy.checkJobStatus */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName + f" < PandaID={pandaID} >")
        tmpLog.debug("start")
        retVal = {"command": None, "status": None}
        try:
            sqlC = (
                "SELECT jobStatus,commandToPilot FROM ATLAS_PANDA.jobsActive4 "
                "WHERE PandaID=:pandaID "
                "UNION "
                "SELECT /*+ INDEX_RS_ASC(JOBSARCHIVED4 PART_JOBSARCHIVED4_PK) */ "
                "jobStatus,commandToPilot FROM ATLAS_PANDA.jobsArchived4 "
                "WHERE PandaID=:pandaID AND modificationTime>:timeLimit "
            )
            varMap = dict()
            varMap[":pandaID"] = int(pandaID)
            varMap[":timeLimit"] = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(hours=1)
            # begin transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            self.cur.execute(sqlC + comment, varMap)
            res = self.cur.fetchone()
            if res is not None:
                retVal["status"], retVal["command"] = res
            else:
                retVal["status"], retVal["command"] = "unknown", "tobekilled"
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"done with {str(retVal)}")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            self.dumpErrorMessage(tmpLog, methodName)
            return retVal

    # add command lock
    def addCommandLockHarvester(self, harvester_ID, command, computingSite, resourceType, useCommit=True):
        comment = " /* DBProxy.addCommandLockHarvester */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(
            _logger,
            methodName + f" < harvesterID={harvester_ID} command={command} site={computingSite}> resource={resourceType} >",
        )
        tmpLog.debug("start")
        try:
            # check if lock is available
            sqlC = "SELECT 1 FROM ATLAS_PANDA.Harvester_Command_Lock "
            sqlC += "WHERE harvester_ID=:harvester_ID AND computingSite=:siteName AND resourceType=:resourceType AND command=:command "
            # sql to add lock
            sqlA = "INSERT INTO ATLAS_PANDA.Harvester_Command_Lock "
            sqlA += "(harvester_ID,computingSite,resourceType,command,lockedTime) "
            sqlA += "VALUES (:harvester_ID,:siteName,:resourceType,:command,CURRENT_DATE-1) "
            if useCommit:
                self.conn.begin()
            # check
            varMap = dict()
            varMap[":harvester_ID"] = harvester_ID
            varMap[":siteName"] = computingSite
            varMap[":resourceType"] = resourceType
            varMap[":command"] = command
            self.cur.execute(sqlC + comment, varMap)
            res = self.cur.fetchone()
            if res is None:
                # add lock
                self.cur.execute(sqlA + comment, varMap)
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError("Commit error")
            tmpLog.debug("done")
            return True
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            self.dumpErrorMessage(tmpLog, methodName)
            return False

    # get command locks
    def getCommandLocksHarvester(self, harvester_ID, command, lockedBy, lockInterval, commandInterval):
        comment = " /* DBProxy.getCommandLocksHarvester */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(
            _logger,
            methodName + f" < harvesterID={harvester_ID} command={command} >",
        )
        tmpLog.debug("start")
        try:
            timeNow = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            # sql to get commands
            sqlC = "SELECT computingSite,resourceType FROM ATLAS_PANDA.Harvester_Command_Lock "
            sqlC += "WHERE harvester_ID=:harvester_ID AND command=:command "
            sqlC += "AND ((lockedBy IS NULL AND lockedTime<:limitComm) OR (lockedBy IS NOT NULL AND lockedTime<:limitLock)) "
            sqlC += "FOR UPDATE "
            # sql to lock command
            sqlL = "UPDATE ATLAS_PANDA.Harvester_Command_Lock SET lockedBy=:lockedBy,lockedTime=CURRENT_DATE "
            sqlL += "WHERE harvester_ID=:harvester_ID AND command=:command AND computingSite=:siteName AND resourceType=:resourceType "
            sqlL += "AND ((lockedBy IS NULL AND lockedTime<:limitComm) OR (lockedBy IS NOT NULL AND lockedTime<:limitLock)) "
            # get commands
            self.conn.begin()
            self.cur.arraysize = 10000
            varMap = dict()
            varMap[":harvester_ID"] = harvester_ID
            varMap[":command"] = command
            varMap[":limitComm"] = timeNow - datetime.timedelta(minutes=commandInterval)
            varMap[":limitLock"] = timeNow - datetime.timedelta(minutes=lockInterval)
            self.cur.execute(sqlC + comment, varMap)
            res = self.cur.fetchall()
            # lock commands
            retMap = dict()
            for computingSite, resourceType in res:
                varMap = dict()
                varMap[":harvester_ID"] = harvester_ID
                varMap[":command"] = command
                varMap[":siteName"] = computingSite
                varMap[":resourceType"] = resourceType
                varMap[":limitComm"] = timeNow - datetime.timedelta(minutes=commandInterval)
                varMap[":limitLock"] = timeNow - datetime.timedelta(minutes=lockInterval)
                varMap[":lockedBy"] = lockedBy
                self.cur.execute(sqlL + comment, varMap)
                nRow = self.cur.rowcount
                if nRow > 0:
                    if computingSite not in retMap:
                        retMap[computingSite] = []
                    retMap[computingSite].append(resourceType)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"done with {str(retMap)}")
            return retMap
        except Exception:
            # roll back
            self._rollback()
            self.dumpErrorMessage(tmpLog, methodName)
            return {}

    # release command lock
    def releaseCommandLockHarvester(self, harvester_ID, command, computingSite, resourceType, lockedBy):
        comment = " /* DBProxy.releaseCommandLockHarvester */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(
            _logger,
            methodName + f" < harvesterID={harvester_ID} com={command} site={computingSite} resource={resourceType} lockedBy={lockedBy} >",
        )
        tmpLog.debug("start")
        try:
            # sql to release lock
            sqlL = "UPDATE ATLAS_PANDA.Harvester_Command_Lock SET lockedBy=NULL "
            sqlL += "WHERE harvester_ID=:harvester_ID AND command=:command "
            sqlL += "AND computingSite=:computingSite AND resourceType=:resourceType AND lockedBy=:lockedBy "
            varMap = dict()
            varMap[":harvester_ID"] = harvester_ID
            varMap[":command"] = command
            varMap[":computingSite"] = computingSite
            varMap[":resourceType"] = resourceType
            varMap[":lockedBy"] = lockedBy
            # release lock
            self.conn.begin()
            self.cur.execute(sqlL + comment, varMap)
            nRow = self.cur.rowcount
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"done with {nRow}")
            return True
        except Exception:
            # roll back
            self._rollback()
            self.dumpErrorMessage(tmpLog, methodName)
            return False

    def ups_get_queues(self):
        """
        Identify unified pilot streaming (ups) queues: served in pull (late binding) model
        :return: list of panda queues
        """
        comment = " /* DBProxy.ups_get_queues */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, method_name)
        tmpLog.debug("start")

        ups_queues = []
        sql = f"""
              SELECT /* use_json_type */ scj.panda_queue FROM {panda_config.schemaPANDA}.schedconfig_json scj
              WHERE scj.data.capability='ucore' AND scj.data.workflow = 'pull_ups'
              """

        self.cur.execute(sql + comment)
        res = self.cur.fetchall()
        for (ups_queue,) in res:
            ups_queues.append(ups_queue)

        tmpLog.debug("done")
        return ups_queues

    def ups_load_worker_stats(self):
        """
        Load the harvester worker stats
        :return: dictionary with worker statistics
        """
        comment = " /* DBProxy.ups_load_worker_stats */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, method_name)
        tmpLog.debug("start")

        # get current pilot distribution in harvester for the queue
        sql = f"""
              SELECT computingsite, harvester_id, jobType, resourceType, status, n_workers
              FROM {panda_config.schemaPANDA}.harvester_worker_stats
              WHERE lastupdate > :time_limit
              """
        var_map = {}
        var_map[":time_limit"] = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(minutes=60)

        self.cur.execute(sql + comment, var_map)
        worker_stats_rows = self.cur.fetchall()
        worker_stats_dict = {}
        for (
            computing_site,
            harvester_id,
            job_type,
            resource_type,
            status,
            n_workers,
        ) in worker_stats_rows:
            worker_stats_dict.setdefault(computing_site, {})
            worker_stats_dict[computing_site].setdefault(harvester_id, {})
            worker_stats_dict[computing_site][harvester_id].setdefault(job_type, {})
            worker_stats_dict[computing_site][harvester_id][job_type].setdefault(resource_type, {})
            worker_stats_dict[computing_site][harvester_id][job_type][resource_type][status] = n_workers

        tmpLog.debug("done")
        return worker_stats_dict

    def get_average_memory_jobs(self, computingsite, target):
        """
        Calculates the average memory for running and queued (starting) jobs at a particular panda queue.
        This function is equivalent to the get_average_memory_workers (for PULL), but is meant for PUSH queues.

        :param computingsite: name of the PanDA queue
        :param target: memory target for the queue in MB. This value is only used in the logging

        :return: average_memory_running_submitted, average_memory_running
        """

        comment = " /* DBProxy.get_average_memory_jobs */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_logger = LogWrapper(_logger, method_name)
        tmp_logger.debug("start")
        try:
            sql_running_and_submitted = (
                f"SELECT /*+ RESULT_CACHE */ COMPUTINGSITE, SUM(NJOBS * PRORATED_MEM_AVG) / SUM(NJOBS) AS avg_memory "
                f"FROM {panda_config.schemaPANDA}.JOBS_SHARE_STATS "
                f"WHERE COMPUTINGSITE = :computingsite "
                f"AND jobstatus IN ('running', 'starting') "
                f"GROUP BY COMPUTINGSITE"
            )

            sql_running = (
                f"SELECT /*+ RESULT_CACHE */ COMPUTINGSITE, SUM(NJOBS * PRORATED_MEM_AVG) / SUM(NJOBS) AS avg_memory "
                f"FROM {panda_config.schemaPANDA}.JOBS_SHARE_STATS "
                f"WHERE COMPUTINGSITE = :computingsite "
                f"AND jobstatus = 'running' "
                f"GROUP BY COMPUTINGSITE"
            )

            var_map = {":computingsite": computingsite}

            self.cur.execute(sql_running_and_submitted + comment, var_map)
            results = self.cur.fetchone()
            try:
                average_memory_running_submitted = results[1] if results[1] is not None else 0
            except TypeError:
                average_memory_running_submitted = 0

            self.cur.execute(sql_running + comment, var_map)
            results = self.cur.fetchone()
            try:
                average_memory_running = results[1] if results[1] is not None else 0
            except TypeError:
                average_memory_running = 0

            tmp_logger.info(
                f"computingsite={computingsite} currently has "
                f"meanrss_running_submitted={average_memory_running_submitted} "
                f"meanrss_running={average_memory_running} "
                f"meanrss_target={target} MB"
            )
            return average_memory_running_submitted, average_memory_running

        except Exception:
            self.dumpErrorMessage(tmp_logger, method_name)
            return 0, 0

    def get_average_memory_workers(self, queue, harvester_id, target):
        """
        Calculates the average memory for running and queued workers at a particular panda queue

        :param queue: name of the PanDA queue
        :param worker_stats_harvester: worker statistics for the particular harvester instance
        :param harvester_id: string with the harvester ID serving the queue
        :param target: memory target for the queue in MB. This value is only used in the logging

        :return: average_memory_running_submitted, average_memory_running
        """

        comment = " /* DBProxy.get_average_memory_workers */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_logger = LogWrapper(_logger, method_name)
        tmp_logger.debug("start")
        try:
            # sql to calculate the average memory for the queue - harvester_id combination
            sql_running_and_submitted = (
                "SELECT /*+ RESULT_CACHE */ sum(total_memory) / NULLIF(sum(n_workers * corecount), 0) "
                "FROM ( "
                "    SELECT hws.computingsite, "
                "           hws.harvester_id, "
                "           hws.n_workers, "
                "           hws.n_workers * NVL(rt.maxcore, NVL(sj.data.corecount, 1)) * NVL(rt.maxrampercore, sj.data.maxrss / NVL(sj.data.corecount, 1)) as total_memory, "
                "           NVL(rt.maxcore, NVL(sj.data.corecount, 1)) as corecount "
                "    FROM ATLAS_PANDA.harvester_worker_stats hws "
                "    JOIN ATLAS_PANDA.resource_types rt ON hws.resourcetype = rt.resource_name "
                "    JOIN ATLAS_PANDA.schedconfig_json sj ON hws.computingsite = sj.panda_queue "
                "    WHERE lastupdate > CAST(SYSTIMESTAMP AT TIME ZONE 'UTC' - INTERVAL '1' HOUR AS DATE) "
                "      AND status IN ('running', 'submitted', 'to_submit') "
                "      AND computingsite=:queue AND harvester_id=:harvester_id"
                ")GROUP BY computingsite, harvester_id "
            )

            sql_running = (
                "SELECT /*+ RESULT_CACHE */ sum(total_memory) / NULLIF(sum(n_workers * corecount), 0) "
                "FROM ( "
                "    SELECT hws.computingsite, "
                "           hws.harvester_id, "
                "           hws.n_workers, "
                "           hws.n_workers * NVL(rt.maxcore, NVL(sj.data.corecount, 1)) * NVL(rt.maxrampercore, sj.data.maxrss / NVL(sj.data.corecount, 1)) as total_memory, "
                "           NVL(rt.maxcore, NVL(sj.data.corecount, 1)) as corecount "
                "    FROM ATLAS_PANDA.harvester_worker_stats hws "
                "    JOIN ATLAS_PANDA.resource_types rt ON hws.resourcetype = rt.resource_name "
                "    JOIN ATLAS_PANDA.schedconfig_json sj ON hws.computingsite = sj.panda_queue "
                "    WHERE lastupdate > CAST(SYSTIMESTAMP AT TIME ZONE 'UTC' - INTERVAL '1' HOUR AS DATE) "
                "      AND status = 'running' "
                "      AND computingsite=:queue AND harvester_id=:harvester_id"
                ")GROUP BY computingsite, harvester_id "
            )

            var_map = {":queue": queue, ":harvester_id": harvester_id}

            self.cur.execute(sql_running_and_submitted + comment, var_map)
            results = self.cur.fetchone()
            try:
                average_memory_running_submitted = results[0] if results[0] is not None else 0
            except TypeError:
                average_memory_running_submitted = 0

            self.cur.execute(sql_running + comment, var_map)
            results = self.cur.fetchone()
            try:
                average_memory_running = results[0] if results[0] is not None else 0
            except TypeError:
                average_memory_running = 0

            tmp_logger.info(
                f"computingsite={queue} and harvester_id={harvester_id} currently has "
                f"meanrss_running_submitted={average_memory_running_submitted} "
                f"meanrss_running={average_memory_running} "
                f"meanrss_target={target} MB"
            )
            return average_memory_running_submitted, average_memory_running

        except Exception:
            self.dumpErrorMessage(tmp_logger, method_name)
            return 0, 0

    def ups_new_worker_distribution(self, queue, worker_stats):
        """
        Assuming we want to have n_cores_queued >= n_cores_running * .5, calculate how many pilots need to be submitted
        and choose the number

        :param queue: name of the PanDA queue
        :param worker_stats: queue worker stats
        :return:
        """

        comment = " /* DBProxy.ups_new_worker_distribution */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, f"{method_name}-{queue}")
        tmp_log.debug("start")
        n_cores_running = 0
        workers_queued = {}
        n_cores_queued = 0
        harvester_ids_temp = list(worker_stats)

        HIMEM = "HIMEM"
        self.__reload_resource_spec_mapper()

        # get the configuration for maximum workers of each type
        pq_data_des = self.get_config_for_pq(queue)
        resource_type_limits = {}
        queue_type = "production"
        average_memory_target = None

        if not pq_data_des:
            tmp_log.debug("Error retrieving queue configuration from DB, limits can not be applied")
        else:
            try:
                resource_type_limits = pq_data_des["uconfig"]["resource_type_limits"]
            except KeyError:
                tmp_log.debug("No resource type limits")
                pass
            try:
                if pq_data_des["meanrss"] != 0:
                    average_memory_target = pq_data_des["meanrss"]
                else:
                    tmp_log.debug("meanrss is 0, not using it as average_memory_target")
            except KeyError:
                tmp_log.debug("No average memory defined")
                pass
            try:
                queue_type = pq_data_des["type"]
            except KeyError:
                tmp_log.error("No queue type")
                pass
            try:
                cores_queue = pq_data_des["corecount"]
                if not cores_queue:
                    cores_queue = 1
            except KeyError:
                tmp_log.error("No corecount")
                pass

        # Retrieve the assigned harvester instance and submit UPS commands only to this instance. We have had multiple
        # cases of test instances submitting to large queues in classic pull mode and not following commands.
        try:
            assigned_harvester_id = pq_data_des["harvester"]
        except KeyErrorException:
            assigned_harvester_id = None

        # If there is no harvester instance assigned to the queue or there are no statistics, we exit without any action
        if assigned_harvester_id and assigned_harvester_id in harvester_ids_temp:
            harvester_id = assigned_harvester_id
        else:
            tmp_log.error("No harvester instance assigned or not in statistics")
            return {}

        # If the site defined a memory target, calculate the memory requested by running and queued workers
        resource_types_under_target = []
        if average_memory_target:
            average_memory_workers_running_submitted, average_memory_workers_running = self.get_average_memory_workers(
                queue, harvester_id, average_memory_target
            )
            # if the queue is over memory, we will only submit lower workers in the next cycle
            if average_memory_target < max(average_memory_workers_running_submitted, average_memory_workers_running):
                resource_types_under_target = self.__resource_spec_mapper.filter_out_high_memory_resourcetypes(memory_threshold=average_memory_target)
                tmp_log.debug(f"Accepting {resource_types_under_target} resource types to respect mean memory target")
            else:
                tmp_log.debug(f"Accepting all resource types as under memory target")

        for job_type in worker_stats[harvester_id]:
            workers_queued.setdefault(job_type, {})
            for resource_type in worker_stats[harvester_id][job_type]:
                core_factor = self.__resource_spec_mapper.translate_resourcetype_to_cores(resource_type, cores_queue)
                try:
                    n_cores_running = n_cores_running + worker_stats[harvester_id][job_type][resource_type]["running"] * core_factor

                    # This limit is in #JOBS or #WORKERS, not in #CORES
                    if resource_type in resource_type_limits:
                        resource_type_limits[resource_type] = (
                            resource_type_limits[resource_type] - worker_stats[harvester_id][job_type][resource_type]["running"]
                        )
                        tmp_log.debug(f"Limit for rt {resource_type} down to {resource_type_limits[resource_type]}")

                    # This limit is in #CORES, since it mixes single and multi core jobs
                    if self.__resource_spec_mapper.is_high_memory(resource_type) and HIMEM in resource_type_limits:
                        resource_type_limits[HIMEM] = resource_type_limits[HIMEM] - worker_stats[harvester_id][job_type][resource_type]["running"] * core_factor
                        tmp_log.debug(f"Limit for rt group {HIMEM} down to {resource_type_limits[HIMEM]}")

                except KeyError:
                    pass

                try:  # submitted
                    workers_queued[job_type].setdefault(resource_type, 0)
                    workers_queued[job_type][resource_type] = (
                        workers_queued[job_type][resource_type] + worker_stats[harvester_id][job_type][resource_type]["submitted"]
                    )
                    n_cores_queued = n_cores_queued + worker_stats[harvester_id][job_type][resource_type]["submitted"] * core_factor
                except KeyError:
                    pass

                try:  # ready
                    workers_queued[job_type].setdefault(resource_type, 0)
                    workers_queued[job_type][resource_type] = (
                        workers_queued[job_type][resource_type] + worker_stats[harvester_id][job_type][resource_type]["ready"]
                    )
                    n_cores_queued = n_cores_queued + worker_stats[harvester_id][job_type][resource_type]["ready"] * core_factor
                except KeyError:
                    pass

        tmp_log.debug(f"Queue {queue} queued worker overview: {workers_queued}")

        # For queues that need more pressure towards reaching a target
        n_cores_running_fake = 0
        try:
            if pq_data_des["status"] in [
                "online",
                "brokeroff",
            ]:  # don't flood test sites with workers
                n_cores_running_fake = pq_data_des["params"]["ups_core_target"]
                tmp_log.debug(f"Using ups_core_target {n_cores_running_fake} for queue {queue}")
        except KeyError:  # no value defined in CRIC
            pass

        n_cores_running = max(n_cores_running, n_cores_running_fake)

        n_cores_target = max(int(n_cores_running * 0.4), 75 * cores_queue)
        n_cores_to_submit = max(n_cores_target - n_cores_queued, 5 * cores_queue)
        tmp_log.debug(f"IN CORES: nrunning {n_cores_running}, ntarget {n_cores_target}, nqueued {n_cores_queued}. We need to process {n_cores_to_submit} cores")

        # Get the sorted global shares
        sorted_shares = self.get_sorted_leaves()

        # Run over the activated jobs by gshare & priority, and subtract them from the queued
        # A negative value for queued will mean more pilots of that resource type are missing
        for share in sorted_shares:
            var_map = {":queue": queue, ":gshare": share.name}
            sql = (
                f"SELECT gshare, prodsourcelabel, resource_type FROM {panda_config.schemaPANDA}.jobsactive4 "
                "WHERE jobstatus = 'activated' "
                "AND computingsite=:queue "
                "AND gshare=:gshare "
            )

            # if we need to filter on resource types
            if resource_types_under_target:
                resource_type_string = ", ".join([f":{item}" for item in resource_types_under_target])
                sql += f"   AND resource_type IN ({resource_type_string}) "
                var_map.update({f":{item}": item for item in resource_types_under_target})

            sql += "ORDER BY currentpriority DESC"
            self.cur.execute(sql + comment, var_map)
            activated_jobs = self.cur.fetchall()

            tmp_log.debug(f"Processing share: {share.name}. Got {len(activated_jobs)} activated jobs")
            for gshare, prodsourcelabel, resource_type in activated_jobs:
                core_factor = self.__resource_spec_mapper.translate_resourcetype_to_cores(resource_type, cores_queue)

                # translate prodsourcelabel to a subset of job types, typically 'user' and 'managed'
                job_type = JobUtils.translate_prodsourcelabel_to_jobtype(queue_type, prodsourcelabel)
                # if we reached the limit for the resource type, skip the job
                if resource_type in resource_type_limits and resource_type_limits[resource_type] <= 0:
                    # tmp_log.debug('Reached resource type limit for {0}'.format(resource_type))
                    continue

                # if we reached the limit for the HIMEM resource type group, skip the job
                if self.__resource_spec_mapper.is_high_memory(resource_type) and HIMEM in resource_type_limits and resource_type_limits[HIMEM] <= 0:
                    # tmp_log.debug('Reached resource type limit for {0}'.format(resource_type))
                    continue

                workers_queued.setdefault(job_type, {})
                workers_queued[job_type].setdefault(resource_type, 0)
                workers_queued[job_type][resource_type] = workers_queued[job_type][resource_type] - 1
                if workers_queued[job_type][resource_type] <= 0:
                    # we've gone over the jobs that already have a queued worker, now we go for new workers
                    n_cores_to_submit = n_cores_to_submit - core_factor

                # We reached the number of workers needed
                if n_cores_to_submit <= 0:
                    tmp_log.debug("Reached cores needed (inner)")
                    break

            # We reached the number of workers needed
            if n_cores_to_submit <= 0:
                tmp_log.debug("Reached cores needed (outer)")
                break

        tmp_log.debug(f"workers_queued: {workers_queued}")

        new_workers = {}
        for job_type in workers_queued:
            new_workers.setdefault(job_type, {})
            for resource_type in workers_queued[job_type]:
                if workers_queued[job_type][resource_type] >= 0:
                    # we have too many workers queued already, don't submit more
                    new_workers[job_type][resource_type] = 0
                elif workers_queued[job_type][resource_type] < 0:
                    # we don't have enough workers for this resource type
                    new_workers[job_type][resource_type] = -workers_queued[job_type][resource_type] + 1

        tmp_log.debug(f"preliminary new workers: {new_workers}")

        # We should still submit a basic worker, even if there are no activated jobs to avoid queue deactivation
        workers = False
        for job_type in new_workers:
            for resource_type in new_workers[job_type]:
                if new_workers[job_type][resource_type] > 0:
                    workers = True
                    break
        if not workers:
            new_workers["managed"] = {BASIC_RESOURCE_TYPE: 1}

        tmp_log.debug(f"new workers: {new_workers}")

        new_workers_per_harvester = {harvester_id: new_workers}

        tmp_log.debug(f"Workers to submit: {new_workers_per_harvester}")
        tmp_log.debug("done")
        return new_workers_per_harvester

    # get active consumers
    def getActiveConsumers(self, jediTaskID, jobsetID, myPandaID):
        comment = " /* DBProxy.getActiveConsumers */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" < jediTaskID={jediTaskID} jobsetID={jobsetID} PandaID={myPandaID} >"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            # sql to get sites where consumers are active
            sqlA = "SELECT PandaID FROM ATLAS_PANDA.jobsActive4 WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID "
            sqlA += "UNION "
            sqlA += "SELECT PandaID FROM ATLAS_PANDA.jobsDefined4 WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID "
            # get IDs
            ids = set()
            varMap = dict()
            varMap[":jediTaskID"] = jediTaskID
            varMap[":jobsetID"] = jobsetID
            self.cur.execute(sqlA + comment, varMap)
            resA = self.cur.fetchall()
            for (pandaID,) in resA:
                if pandaID != myPandaID:
                    ids.add(pandaID)
            nIDs = len(ids)
            if nIDs == 0:
                # get dataset
                sqlPD = "SELECT f.datasetID,f.fileID FROM ATLAS_PANDA.JEDI_Datasets d,ATLAS_PANDA.filesTable4 f "
                sqlPD += "WHERE d.jediTaskID=:jediTaskID AND d.type IN (:type1,:type2) AND d.masterID IS NULL "
                sqlPD += "AND f.PandaID=:PandaID AND f.jeditaskID=f.jediTaskID AND f.datasetID=d.datasetID "
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":PandaID"] = myPandaID
                varMap[":type1"] = "input"
                varMap[":type2"] = "pseudo_input"
                self.cur.execute(sqlPD + comment, varMap)
                resPD = self.cur.fetchall()
                # get PandaIDs
                idAttrMap = dict()
                sqlCP = "SELECT PandaID,attemptNr FROM ATLAS_PANDA.filesTable4 "
                sqlCP += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlWP = "SELECT 1 FROM ATLAS_PANDA.jobsWaiting4 WHERE PandaID=:PandaID AND computingSite=:computingSite "
                for datasetID, fileID in resPD:
                    if fileID is None:
                        continue
                    varMap = {}
                    varMap[":jediTaskID"] = jediTaskID
                    varMap[":datasetID"] = datasetID
                    varMap[":fileID"] = fileID
                    self.cur.execute(sqlCP + comment, varMap)
                    resCP = self.cur.fetchall()
                    for pandaID, attemptNr in resCP:
                        idAttrMap[pandaID] = attemptNr
                # look for my attemptNr
                if myPandaID in idAttrMap:
                    myAttemptNr = idAttrMap[myPandaID]
                    for pandaID in idAttrMap:
                        attemptNr = idAttrMap[pandaID]
                        if attemptNr == myAttemptNr and pandaID != myPandaID and pandaID not in ids:
                            varMap = {}
                            varMap[":PandaID"] = pandaID
                            varMap[":computingSite"] = EventServiceUtils.siteIdForWaitingCoJumboJobs
                            self.cur.execute(sqlWP + comment, varMap)
                            resWP = self.cur.fetchone()
                            if resWP is not None:
                                nIDs += 1
            tmpLog.debug(f"got {nIDs} ids")
            return nIDs
        except Exception:
            # error
            self.dumpErrorMessage(_logger, methodName)
            return 0

    # check event availability
    def checkEventsAvailability(self, pandaID, jobsetID, jediTaskID):
        comment = " /* DBProxy.checkEventsAvailability */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" < PandaID={pandaID} jobsetID={jobsetID} jediTaskID={jediTaskID} >"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            sqlJ = f"SELECT eventService FROM {panda_config.schemaJEDI}.jobsActive4 WHERE PandaID=:PandaID "
            # start transaction
            self.conn.begin()
            # get job to check if a jumbo job
            isJumbo = False
            varMap = {}
            varMap[":PandaID"] = pandaID
            self.cur.execute(sqlJ + comment, varMap)
            res = self.cur.fetchone()
            if res is not None:
                (eventService,) = res
                if eventService == EventServiceUtils.jumboJobFlagNumber:
                    isJumbo = True
            # get number of event ranges
            sqlE = "SELECT COUNT(*) "
            sqlE += f"FROM {panda_config.schemaJEDI}.JEDI_Events "
            sqlE += "WHERE jediTaskID=:jediTaskID AND status=:eventStatus AND attemptNr>:minAttemptNr "
            varMap = {}
            varMap[":eventStatus"] = EventServiceUtils.ST_ready
            varMap[":minAttemptNr"] = 0
            varMap[":jediTaskID"] = jediTaskID
            if not isJumbo:
                varMap[":jobsetID"] = jobsetID
                sqlE += "AND PandaID=:jobsetID "
            self.cur.execute(sqlE + comment, varMap)
            res = self.cur.fetchone()
            if res is not None:
                (nEvents,) = res
            else:
                nEvents = 0
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"has {nEvents} event ranges")
            return nEvents
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None

    # get LNFs for jumbo job
    def getLFNsForJumbo(self, jediTaskID):
        comment = " /* DBProxy.getLFNsForJumbo */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        method_name += f"< jediTaskID={jediTaskID} >"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        try:
            sqlS = "SELECT lfn,scope FROM {0}.JEDI_Datasets d, {0}.JEDI_Dataset_Contents c ".format(panda_config.schemaJEDI)
            sqlS += "WHERE d.jediTaskID=c.jediTaskID AND d.datasetID=c.datasetID AND d.jediTaskID=:jediTaskID "
            sqlS += "AND d.type IN (:type1,:type2) AND d.masterID IS NULL "
            retSet = set()
            # start transaction
            self.conn.begin()
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            varMap[":type1"] = "input"
            varMap[":type2"] = "pseudo_input"
            self.cur.execute(sqlS + comment, varMap)
            res = self.cur.fetchall()
            for tmpLFN, tmpScope in res:
                name = f"{tmpScope}:{tmpLFN}"
                retSet.add(name)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmp_log.debug(f"has {len(retSet)} LFNs")
            return retSet
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, method_name)
            return []

    # get active job attribute
    def getActiveJobAttributes(self, pandaID, attrs):
        comment = " /* DBProxy.getActiveJobAttributes */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        method_name += f"< PandaID={pandaID} >"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        try:
            sqlS = f"SELECT {','.join(attrs)} FROM ATLAS_PANDA.jobsActive4 "
            sqlS += "WHERE PandaID=:PandaID "
            # start transaction
            self.conn.begin()
            varMap = {}
            varMap[":PandaID"] = pandaID
            self.cur.execute(sqlS + comment, varMap)
            res = self.cur.fetchone()
            if res is not None:
                retMap = dict()
                for idx, attr in enumerate(attrs):
                    retMap[attr] = res[idx]
            else:
                retMap = None
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmp_log.debug(f"got {str(retMap)}")
            return retMap
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, method_name)
            return None

    # get number of started events
    def getNumStartedEvents(self, jobSpec):
        comment = " /* DBProxy.getNumStartedEvents */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        method_name += f" < PandaID={jobSpec.PandaID} >"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        try:
            # count the number of started ranges
            sqlCDO = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlCDO += f"COUNT(*) FROM {panda_config.schemaJEDI}.JEDI_Events tab "
            sqlCDO += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            sqlCDO += "AND status IN (:esSent,:esRunning,:esFinished,:esDone) "
            # start transaction
            self.conn.begin()
            nEvt = 0
            for fileSpec in jobSpec.Files:
                if fileSpec.type != "input":
                    continue
                varMap = {}
                varMap[":jediTaskID"] = fileSpec.jediTaskID
                varMap[":datasetID"] = fileSpec.datasetID
                varMap[":fileID"] = fileSpec.fileID
                varMap[":esSent"] = EventServiceUtils.ST_sent
                varMap[":esRunning"] = EventServiceUtils.ST_running
                varMap[":esFinished"] = EventServiceUtils.ST_finished
                varMap[":esDone"] = EventServiceUtils.ST_done
                self.cur.execute(sqlCDO + comment, varMap)
                res = self.cur.fetchone()
                if res is not None:
                    nEvt += res[0]
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmp_log.debug(f"{nEvt} events started")
            return nEvt
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, method_name)
            return None

    # add harvester dialog messages
    def addHarvesterDialogs(self, harvesterID, dialogs):
        comment = " /* DBProxy.addHarvesterDialogs */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        method_name += f" < harvesterID={harvesterID} >"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        try:
            # sql to delete message
            sqlC = f"DELETE FROM {panda_config.schemaPANDA}.Harvester_Dialogs "
            sqlC += "WHERE harvester_id=:harvester_id AND diagID=:diagID "
            # sql to insert message
            sqlI = f"INSERT INTO {panda_config.schemaPANDA}.Harvester_Dialogs "
            sqlI += "(harvester_id,diagID,moduleName,identifier,creationTime,messageLevel,diagMessage) "
            sqlI += "VALUES(:harvester_id,:diagID,:moduleName,:identifier,:creationTime,:messageLevel,:diagMessage) "
            for diagDict in dialogs:
                # start transaction
                self.conn.begin()
                # delete
                varMap = dict()
                varMap[":diagID"] = diagDict["diagID"]
                varMap[":harvester_id"] = harvesterID
                self.cur.execute(sqlC + comment, varMap)
                # insert
                varMap = dict()
                varMap[":diagID"] = diagDict["diagID"]
                varMap[":identifier"] = diagDict["identifier"]
                varMap[":moduleName"] = diagDict["moduleName"]
                varMap[":creationTime"] = datetime.datetime.strptime(diagDict["creationTime"], "%Y-%m-%d %H:%M:%S.%f")
                varMap[":messageLevel"] = diagDict["messageLevel"]
                varMap[":diagMessage"] = diagDict["diagMessage"]
                varMap[":harvester_id"] = harvesterID
                self.cur.execute(sqlI + comment, varMap)
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
            tmp_log.debug(f"added {len(dialogs)} messages")
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, method_name)
            return False

    # get job statistics per site and resource
    def getJobStatisticsPerSiteResource(self, timeWindow):
        comment = " /* DBProxy.getJobStatisticsPerSiteResource */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        sql0 = "SELECT computingSite,jobStatus,resource_type,COUNT(*) FROM %s "
        sql0 += "GROUP BY computingSite,jobStatus,resource_type "
        sqlA = "SELECT /*+ INDEX_RS_ASC(tab (MODIFICATIONTIME PRODSOURCELABEL)) */ computingSite,jobStatus,resource_type,COUNT(*) "
        sqlA += "FROM ATLAS_PANDA.jobsArchived4 tab WHERE modificationTime>:modificationTime "
        sqlA += "GROUP BY computingSite,jobStatus,resource_type "
        tables = [
            "ATLAS_PANDA.jobsActive4",
            "ATLAS_PANDA.jobsDefined4",
            "ATLAS_PANDA.jobsArchived4",
        ]
        # sql for materialized view
        sqlMV = re.sub("COUNT\(\*\)", "SUM(njobs)", sql0)
        sqlMV = re.sub("SELECT ", "SELECT /*+ RESULT_CACHE */ ", sqlMV)
        ret = dict()
        try:
            if timeWindow is None:
                timeLimit = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(hours=12)
            else:
                timeLimit = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(minutes=int(timeWindow))
            for table in tables:
                # start transaction
                self.conn.begin()
                self.cur.arraysize = 10000
                # select
                varMap = {}
                if table == "ATLAS_PANDA.jobsArchived4":
                    varMap[":modificationTime"] = timeLimit
                    sqlExe = sqlA + comment
                elif table == "ATLAS_PANDA.jobsActive4":
                    sqlExe = (sqlMV + comment) % "ATLAS_PANDA.JOBS_SHARE_STATS"
                else:
                    sqlExe = (sql0 + comment) % table
                self.cur.execute(sqlExe, varMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                # create map
                for computingSite, jobStatus, resource_type, nJobs in res:
                    ret.setdefault(computingSite, dict())
                    ret[computingSite].setdefault(resource_type, dict())
                    ret[computingSite][resource_type].setdefault(jobStatus, 0)
                    ret[computingSite][resource_type][jobStatus] += nJobs
            # for zero
            stateList = ["assigned", "activated", "running", "finished", "failed"]
            for computingSite in ret:
                for resource_type in ret[computingSite]:
                    for jobStatus in stateList:
                        ret[computingSite][resource_type].setdefault(jobStatus, 0)

            return ret
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, method_name)
            return dict()

    # get job statistics per site, source label, and resource type
    def get_job_statistics_per_site_label_resource(self, time_window):
        comment = " /* DBProxy.get_job_statistics_per_site_label_resource */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        sqlD = (
            "SELECT computingSite,jobStatus,gshare,resource_type,COUNT(*) FROM ATLAS_PANDA.jobsDefined4 "
            "GROUP BY computingSite,jobStatus,gshare,resource_type "
        )
        sqlF = (
            "SELECT computingSite,jobStatus,gshare,resource_type,COUNT(*) FROM ATLAS_PANDA.jobsActive4 "
            "WHERE jobStatus=:jobStatus AND modificationTime>:modificationTime "
            "GROUP BY computingSite,jobStatus,gshare,resource_type "
        )
        sqlM = (
            "SELECT /*+ RESULT_CACHE */ computingSite,jobStatus,gshare,resource_type,SUM(njobs) "
            "FROM ATLAS_PANDA.JOBS_SHARE_STATS "
            "WHERE jobStatus<>:jobStatus "
            "GROUP BY computingSite,jobStatus,gshare,resource_type "
        )
        sqlA = "SELECT /*+ INDEX_RS_ASC(tab (MODIFICATIONTIME PRODSOURCELABEL)) */ "
        sqlA += "computingSite,jobStatus,gshare,resource_type,COUNT(*) "
        sqlA += "FROM ATLAS_PANDA.jobsArchived4 tab WHERE modificationTime>:modificationTime "
        sqlA += "GROUP BY computingSite,jobStatus,gshare,resource_type "
        ret = dict()
        try:
            if time_window is None:
                timeLimit = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(hours=12)
            else:
                timeLimit = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(minutes=int(time_window))
            sqlVarList = [
                (sqlD, {}),
                (sqlF, {":jobStatus": "failed", ":modificationTime": timeLimit}),
                (sqlM, {":jobStatus": "failed"}),
                (sqlA, {":modificationTime": timeLimit}),
            ]
            for sqlExe, varMap in sqlVarList:
                # start transaction
                self.conn.begin()
                self.cur.arraysize = 10000
                # select
                sqlExe = sqlExe + comment
                self.cur.execute(sqlExe, varMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
                self.__reload_shares()
                # create map
                shareLabelMap = dict()
                for computingSite, jobStatus, gshare, resource_type, nJobs in res:
                    if gshare not in shareLabelMap:
                        for share in self.leave_shares:
                            if gshare == share.name:
                                prodSourceLabel = share.prodsourcelabel
                                if "|" in prodSourceLabel:
                                    prodSourceLabel = prodSourceLabel.split("|")[0]
                                    prodSourceLabel = prodSourceLabel.replace(".*", "")
                                shareLabelMap[gshare] = prodSourceLabel
                                break
                        if gshare not in shareLabelMap:
                            shareLabelMap[gshare] = "unknown"
                    prodSourceLabel = shareLabelMap[gshare]
                    ret.setdefault(computingSite, dict())
                    ret[computingSite].setdefault(prodSourceLabel, dict())
                    ret[computingSite][prodSourceLabel].setdefault(resource_type, dict())
                    ret[computingSite][prodSourceLabel][resource_type].setdefault(jobStatus, 0)
                    ret[computingSite][prodSourceLabel][resource_type][jobStatus] += nJobs
            # for zero
            stateList = ["assigned", "activated", "running", "finished", "failed"]
            for computingSite in ret:
                for prodSourceLabel in ret[computingSite]:
                    for resource_type in ret[computingSite][prodSourceLabel]:
                        for jobStatus in stateList:
                            ret[computingSite][prodSourceLabel][resource_type].setdefault(jobStatus, 0)
            # return
            tmp_log.debug(f"{str(ret)}")
            return ret
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, method_name)
            return dict()

    # set num slots for workload provisioning
    def setNumSlotsForWP(self, pandaQueueName, numSlots, gshare, resourceType, validPeriod):
        comment = " /* DBProxy.setNumSlotsForWP */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        method_name += f" < pq={pandaQueueName} >"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        # sql to check
        sqlC = "SELECT 1 FROM ATLAS_PANDA.Harvester_Slots "
        sqlC += "WHERE pandaQueueName=:pandaQueueName "
        if gshare is None:
            sqlC += "AND gshare IS NULL "
        else:
            sqlC += "AND gshare=:gshare "
        if resourceType is None:
            sqlC += "AND resourceType IS NULL "
        else:
            sqlC += "AND resourceType=:resourceType "
        # sql to insert
        sqlI = "INSERT INTO ATLAS_PANDA.Harvester_Slots (pandaQueueName,gshare,resourceType,numSlots,modificationTime,expirationTime) "
        sqlI += "VALUES (:pandaQueueName,:gshare,:resourceType,:numSlots,:modificationTime,:expirationTime) "
        # sql to update
        if numSlots == -1:
            sqlU = "DELETE FROM ATLAS_PANDA.Harvester_Slots "
        else:
            sqlU = "UPDATE ATLAS_PANDA.Harvester_Slots SET "
            sqlU += "numSlots=:numSlots,modificationTime=:modificationTime,expirationTime=:expirationTime "
        sqlU += "WHERE pandaQueueName=:pandaQueueName "
        if gshare is None:
            sqlU += "AND gshare IS NULL "
        else:
            sqlU += "AND gshare=:gshare "
        if resourceType is None:
            sqlU += "AND resourceType IS NULL "
        else:
            sqlU += "AND resourceType=:resourceType "
        try:
            timeNow = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            # start transaction
            self.conn.begin()
            # check
            varMap = dict()
            varMap[":pandaQueueName"] = pandaQueueName
            if gshare is not None:
                varMap[":gshare"] = gshare
            if resourceType is not None:
                varMap[":resourceType"] = resourceType
            self.cur.execute(sqlC, varMap)
            resC = self.cur.fetchone()
            # insert or update
            varMap = dict()
            varMap[":pandaQueueName"] = pandaQueueName
            if resC is None or gshare is not None:
                varMap[":gshare"] = gshare
            if resC is None or resourceType is not None:
                varMap[":resourceType"] = resourceType
            if numSlots != -1:
                varMap[":numSlots"] = numSlots
                varMap[":modificationTime"] = timeNow
                if validPeriod is None:
                    varMap[":expirationTime"] = None
                else:
                    varMap[":expirationTime"] = timeNow + datetime.timedelta(days=int(validPeriod))
                if resC is None:
                    # insert
                    self.cur.execute(sqlI, varMap)
                else:
                    # update
                    self.cur.execute(sqlU, varMap)
            else:
                # delete
                if resC is not None:
                    self.cur.execute(sqlU, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # return
            tmp_log.debug(f"set nSlots={numSlots}")
            return (
                0,
                f"set numSlots={numSlots} for PQ={pandaQueueName} gshare={gshare} resource={resourceType}",
            )
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, method_name)
            return (1, "database error in the panda server")

    # enable jumbo jobs
    def enableJumboJobs(self, jediTaskID, nJumboJobs, nJumboPerSite, useCommit=True, sendLog=True):
        comment = " /* DBProxy.enableJumboJobs */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        method_name += f" < jediTaskID={jediTaskID} >"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        try:
            # sql to set flag
            sqlJumboF = f"UPDATE {panda_config.schemaJEDI}.JEDI_Tasks "
            sqlJumboF += "SET useJumbo=:newJumbo WHERE jediTaskID=:jediTaskID "
            # start transaction
            if useCommit:
                self.conn.begin()
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            if nJumboJobs == 0:
                varMap[":newJumbo"] = "D"
            else:
                varMap[":newJumbo"] = "W"
            self.cur.execute(sqlJumboF, varMap)
            nRow = self.cur.rowcount
            if nRow > 0:
                self.changeTaskSplitRulePanda(jediTaskID, "NJ", nJumboJobs, useCommit=False, sendLog=sendLog)
                self.changeTaskSplitRulePanda(jediTaskID, "MJ", nJumboPerSite, useCommit=False, sendLog=sendLog)
                retVal = (0, "done")
                tmp_log.debug(f"set nJumboJobs={nJumboJobs} nJumboPerSite={nJumboPerSite} useJumbo={varMap[':newJumbo']}")
            else:
                retVal = (2, "task not found")
                tmp_log.debug("task not found")
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError("Commit error")
            # return
            return retVal
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger, method_name)
            return (1, "database error in the panda server")

    # update input status in JEDI
    def updateInputStatusJedi(self, jediTaskID, pandaID, newStatus, checkOthers=False, no_late_bulk_exec=True, extracted_sqls=None):
        comment = " /* DBProxy.updateInputStatusJedi */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        method_name += f" < jediTaskID={jediTaskID} PandaID={pandaID} >"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug(f"start newStatus={newStatus}")
        statusMap = {
            "ready": ["queued", "starting", "running", "merging", "transferring"],
            "queued": ["ready", "starting", "running"],
            "starting": ["queued", "running", "ready"],
            "running": ["starting", "queued", "ready"],
            "merging": ["queued", "running"],
            "transferring": ["running", "merging"],
            "finished": ["running", "transferring", "merging"],
            "failed": ["running", "transferring", "merging", "queued", "starting"],
        }
        try:
            # change cancelled/closed to failed
            if newStatus in ["cancelled", "closed"]:
                newStatus = "failed"
            # check status
            if newStatus not in statusMap:
                tmp_log.error(f"unknown status : {newStatus}")
                return False
            # get datasetID and fileID
            sqlF = f"SELECT f.datasetID,f.fileID,f.attemptNr FROM {panda_config.schemaJEDI}.JEDI_Datasets d,{panda_config.schemaPANDA}.filesTable4 f "
            sqlF += "WHERE d.jediTaskID=:jediTaskID AND d.type IN (:type1,:type2) AND d.masterID IS NULL "
            sqlF += "AND f.datasetID=d.datasetID AND f.PandaID=:PandaID "
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            varMap[":PandaID"] = pandaID
            varMap[":type1"] = "input"
            varMap[":type2"] = "pseudo_input"
            self.cur.execute(sqlF + comment, varMap)
            resF = self.cur.fetchall()
            # get status and attemptNr in JEDI
            sqlJ = "SELECT status,proc_status,attemptNr,maxAttempt,failedAttempt,maxFailure "
            sqlJ += f"FROM {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
            sqlJ += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            sqlU = f"UPDATE {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
            sqlU += "SET proc_status=:newStatus "
            sqlU += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            sqlU += "AND attemptNr=:attemptNr "
            sqlC = "SELECT j.PandaID FROM {0}.jobsActive4 j,{0}.filesTable4 f ".format(panda_config.schemaPANDA)
            sqlC += "WHERE j.PandaID=f.PandaID AND j.jobStatus=:jobStatus "
            sqlC += "AND f.jediTaskID=:jediTaskID AND f.datasetID=:datasetID AND f.fileID=:fileID "
            sqlC += "AND f.attemptNr=:attemptNr "
            for datasetID, fileID, f_attemptNr in resF:
                # increment attemptNr for final status
                if newStatus in ["finished", "failed"]:
                    f_attemptNr += 1
                # check others
                if checkOthers and newStatus == "queued":
                    otherStatus = "running"
                    varMap = {}
                    varMap[":jediTaskID"] = jediTaskID
                    varMap[":datasetID"] = datasetID
                    varMap[":fileID"] = fileID
                    varMap[":attemptNr"] = f_attemptNr
                    varMap[":jobStatus"] = otherStatus
                    self.cur.execute(sqlC + comment, varMap)
                    resC = self.cur.fetchall()
                    if len(resC) > 0:
                        tmp_log.debug(f"skip to update fileID={fileID} to {newStatus} since others like PandaID={resC[0][0]} is {otherStatus}")
                        continue
                # get data in JEDI
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":datasetID"] = datasetID
                varMap[":fileID"] = fileID
                self.cur.execute(sqlJ + comment, varMap)
                (
                    fileStatus,
                    oldStatus,
                    j_attemptNr,
                    maxAttempt,
                    failedAttempt,
                    maxFailure,
                ) = self.cur.fetchone()
                # check attemptNr
                if j_attemptNr != f_attemptNr:
                    tmp_log.error(f"inconsistent attempt number : JEDI:{j_attemptNr} Panda:{f_attemptNr} for fileID={fileID} newStatus={newStatus}")
                    continue
                # check status
                if oldStatus is not None and oldStatus not in statusMap[newStatus] and oldStatus != newStatus:
                    tmp_log.error(f"{oldStatus} -> {newStatus} is forbidden for fileID={fileID}")
                    continue
                # conversion for failed
                tmpNewStatus = newStatus
                if newStatus == "failed" and j_attemptNr < maxAttempt and (maxFailure is None or failedAttempt < maxFailure):
                    tmpNewStatus = "ready"
                # no change
                if tmpNewStatus == oldStatus:
                    tmp_log.debug(f"skip to update fileID={fileID} due to no status change already in {tmpNewStatus}")
                    continue
                # ready
                if tmpNewStatus in ["ready", "failed"] and fileStatus != "ready":
                    tmp_log.debug(f"skip to update fileID={fileID} to {tmpNewStatus} since the file status is {fileStatus}")
                    continue
                # update
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":datasetID"] = datasetID
                varMap[":fileID"] = fileID
                varMap[":attemptNr"] = f_attemptNr
                varMap[":newStatus"] = tmpNewStatus
                if no_late_bulk_exec:
                    self.cur.execute(sqlU + comment, varMap)
                    nRow = self.cur.rowcount
                    tmp_log.debug(f"{oldStatus} -> {tmpNewStatus} for fileID={fileID} with {nRow}")
                else:
                    extracted_sqls.setdefault("jedi_input", {"sql": sqlU + comment, "vars": []})
                    extracted_sqls["jedi_input"]["vars"].append(varMap)
            # return
            tmp_log.debug("done")
            return True
        except Exception:
            # error
            self.dumpErrorMessage(_logger, method_name)
            return False

    # make fake co-jumbo
    def makeFakeCoJumbo(self, oldJobSpec):
        comment = " /* DBProxy.self.makeFakeCoJumbo */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        method_name += f" < PandaID={oldJobSpec.PandaID} >"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        try:
            # make a new job
            jobSpec = copy.copy(oldJobSpec)
            jobSpec.Files = []
            # reset job attributes
            jobSpec.startTime = None
            jobSpec.creationTime = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            jobSpec.modificationTime = jobSpec.creationTime
            jobSpec.stateChangeTime = jobSpec.creationTime
            jobSpec.batchID = None
            jobSpec.schedulerID = None
            jobSpec.pilotID = None
            jobSpec.endTime = None
            jobSpec.transExitCode = None
            jobSpec.jobMetrics = None
            jobSpec.jobSubStatus = None
            jobSpec.actualCoreCount = None
            jobSpec.hs06sec = None
            jobSpec.nEvents = None
            jobSpec.cpuConsumptionTime = None
            jobSpec.computingSite = EventServiceUtils.siteIdForWaitingCoJumboJobs
            jobSpec.jobExecutionID = 0
            jobSpec.jobStatus = "waiting"
            jobSpec.jobSubStatus = None
            for attr in jobSpec._attributes:
                for patt in [
                    "ErrorCode",
                    "ErrorDiag",
                    "CHAR",
                    "BYTES",
                    "RSS",
                    "PSS",
                    "VMEM",
                    "SWAP",
                ]:
                    if attr.endswith(patt):
                        setattr(jobSpec, attr, None)
                        break
            # read files
            varMap = {}
            varMap[":PandaID"] = oldJobSpec.PandaID
            sqlFile = f"SELECT {FileSpec.columnNames()} FROM ATLAS_PANDA.filesTable4 "
            sqlFile += "WHERE PandaID=:PandaID "
            self.cur.arraysize = 100000
            self.cur.execute(sqlFile + comment, varMap)
            resFs = self.cur.fetchall()
            # loop over all files
            for resF in resFs:
                # add
                fileSpec = FileSpec()
                fileSpec.pack(resF)
                # skip zip
                if fileSpec.type.startswith("zip"):
                    continue
                jobSpec.addFile(fileSpec)
                # reset file status
                if fileSpec.type in ["output", "log"]:
                    fileSpec.status = "unknown"
            # read job parameters
            sqlJobP = "SELECT jobParameters FROM ATLAS_PANDA.jobParamsTable WHERE PandaID=:PandaID "
            varMap = {}
            varMap[":PandaID"] = oldJobSpec.PandaID
            self.cur.execute(sqlJobP + comment, varMap)
            for (clobJobP,) in self.cur:
                try:
                    jobSpec.jobParameters = clobJobP.read()
                except AttributeError:
                    jobSpec.jobParameters = str(clobJobP)
                break
            # insert job with new PandaID
            sql1 = f"INSERT INTO ATLAS_PANDA.jobsWaiting4 ({JobSpec.columnNames()}) "
            sql1 += JobSpec.bindValuesExpression(useSeq=True)
            sql1 += " RETURNING PandaID INTO :newPandaID"
            varMap = jobSpec.valuesMap(useSeq=True)
            varMap[":newPandaID"] = self.cur.var(varNUMBER)
            # insert
            retI = self.cur.execute(sql1 + comment, varMap)
            # set PandaID
            val = self.getvalue_corrector(self.cur.getvalue(varMap[":newPandaID"]))
            jobSpec.PandaID = int(val)
            msgStr = f"Generate a fake co-jumbo new PandaID={jobSpec.PandaID} at {jobSpec.computingSite} "
            tmp_log.debug(msgStr)
            # insert files
            sqlFile = f"INSERT INTO ATLAS_PANDA.filesTable4 ({FileSpec.columnNames()}) "
            sqlFile += FileSpec.bindValuesExpression(useSeq=True)
            sqlFile += " RETURNING row_ID INTO :newRowID"
            for fileSpec in jobSpec.Files:
                # reset rowID
                fileSpec.row_ID = None
                # change GUID and LFN for log
                if fileSpec.type == "log":
                    fileSpec.GUID = str(uuid.uuid4())
                    fileSpec.lfn = re.sub(f"\\.{oldJobSpec.PandaID}$", "", fileSpec.lfn)
                # insert
                varMap = fileSpec.valuesMap(useSeq=True)
                varMap[":newRowID"] = self.cur.var(varNUMBER)
                self.cur.execute(sqlFile + comment, varMap)
                val = self.getvalue_corrector(self.cur.getvalue(varMap[":newRowID"]))
                fileSpec.row_ID = int(val)
            # insert job parameters
            sqlJob = "INSERT INTO ATLAS_PANDA.jobParamsTable (PandaID,jobParameters) VALUES (:PandaID,:param) "
            varMap = {}
            varMap[":PandaID"] = jobSpec.PandaID
            varMap[":param"] = jobSpec.jobParameters
            self.cur.execute(sqlJob + comment, varMap)
            self.recordStatusChange(jobSpec.PandaID, jobSpec.jobStatus, jobInfo=jobSpec, useCommit=False)
            self.push_job_status_message(jobSpec, jobSpec.PandaID, jobSpec.jobStatus)
            # return
            tmp_log.debug("done")
            return 1
        except Exception:
            # error
            self.dumpErrorMessage(_logger, method_name)
            return 0

    # enable event service
    def enableEventService(self, jediTaskID):
        comment = " /* DBProxy.enableEventService */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        method_name += f" < jediTaskID={jediTaskID} >"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        try:
            # get default values
            nEsConsumers = self.getConfigValue("taskrefiner", "AES_NESCONSUMERS", "jedi", "atlas")
            if nEsConsumers is None:
                nEsConsumers = 1
            nSitesPerJob = self.getConfigValue("taskrefiner", "AES_NSITESPERJOB", "jedi", "atlas")
            # get task params
            sqlTP = f"SELECT taskParams FROM {panda_config.schemaJEDI}.JEDI_TaskParams WHERE jediTaskID=:jediTaskID "
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            tmpV, taskParams = self.getClobObj(sqlTP, varMap)
            if taskParams is None:
                errStr = "task parameter is not found"
                tmp_log.error(errStr)
                return (3, errStr)
            try:
                taskParamMap = json.loads(taskParams[0][0])
            except Exception:
                errStr = "cannot load task parameter"
                tmp_log.error(errStr)
                return (4, errStr)
            # extract parameters
            transPath = "UnDefined"
            jobParameters = "UnDefined"
            if "esmergeSpec" in taskParamMap:
                if "transPath" in taskParamMap["esmergeSpec"]:
                    transPath = taskParamMap["esmergeSpec"]["transPath"]
                if "jobParameters" in taskParamMap["esmergeSpec"]:
                    jobParameters = taskParamMap["esmergeSpec"]["jobParameters"]
            esJobParameters = "<PANDA_ESMERGE_TRF>" + transPath + "</PANDA_ESMERGE_TRF>" + "<PANDA_ESMERGE_JOBP>" + jobParameters + "</PANDA_ESMERGE_JOBP>"
            esJobParameters = str(esJobParameters)
            # get job params template
            sqlJT = f"SELECT jobParamsTemplate FROM {panda_config.schemaJEDI}.JEDI_JobParams_Template WHERE jediTaskID=:jediTaskID "
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            tmpV, jobParamsTemplate = self.getClobObj(sqlJT, varMap)
            if jobParamsTemplate is None:
                errStr = "job params template is not found"
                tmp_log.error(errStr)
                return (5, errStr)
            jobParamsTemplate = jobParamsTemplate[0][0]
            # sql to set flag
            sqlES = f"UPDATE {panda_config.schemaJEDI}.JEDI_Tasks "
            sqlES += "SET eventService=:newEventService,coreCount=0,"
            sqlES += f"workqueue_id=(SELECT queue_id FROM {panda_config.schemaJEDI}.JEDI_Work_Queue WHERE queue_name=:queueName) "
            sqlES += "WHERE jediTaskID=:jediTaskID AND lockedBy IS NULL "
            # start transaction
            self.conn.begin()
            # update ES flag
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            varMap[":newEventService"] = 1
            varMap[":queueName"] = "eventservice"
            self.cur.execute(sqlES, varMap)
            nRow = self.cur.rowcount
            if nRow > 0:
                # update splitrule
                self.changeTaskSplitRulePanda(jediTaskID, "EC", nEsConsumers, useCommit=False, sendLog=True)
                if nSitesPerJob is not None:
                    self.changeTaskSplitRulePanda(jediTaskID, "NS", nSitesPerJob, useCommit=False, sendLog=True)
                self.changeTaskSplitRulePanda(jediTaskID, "ES", 1, useCommit=False, sendLog=True)
                self.changeTaskSplitRulePanda(jediTaskID, "RE", 1, useCommit=False, sendLog=True)
                self.changeTaskSplitRulePanda(jediTaskID, "ME", 1, useCommit=False, sendLog=True)
                self.changeTaskSplitRulePanda(jediTaskID, "XA", 1, useCommit=False, sendLog=True)
                self.changeTaskSplitRulePanda(jediTaskID, "XJ", 0, useCommit=False, sendLog=True)
                self.changeTaskSplitRulePanda(jediTaskID, "ND", 1, useCommit=False, sendLog=True)
                self.changeTaskSplitRulePanda(jediTaskID, "XF", 1, useCommit=False, sendLog=True)
                self.changeTaskSplitRulePanda(jediTaskID, "SC", None, useCommit=False, sendLog=True)
                if esJobParameters not in jobParamsTemplate:
                    # update job params template
                    sqlUJ = f"UPDATE {panda_config.schemaJEDI}.JEDI_JobParams_Template SET jobParamsTemplate=:new WHERE jediTaskID=:jediTaskID "
                    varMap = {}
                    varMap[":jediTaskID"] = jediTaskID
                    varMap[":new"] = jobParamsTemplate + esJobParameters
                    self.cur.execute(sqlUJ, varMap)
                retVal = (0, "done")
                tmp_log.debug("done")
            else:
                retVal = (2, "task not found or locked")
                tmp_log.debug("failed to update the flag")
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # return
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, method_name)
            return (1, "database error in the panda server")

    # get JEDI file attributes
    def getJediFileAttributes(self, PandaID, jediTaskID, datasetID, fileID, attrs):
        comment = " /* DBProxy.getJediFileAttributes */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" < PandaID={PandaID} >"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug(f"start for jediTaskID={jediTaskID} datasetId={datasetID} fileID={fileID}")
        try:
            # sql to get task attributes
            sqlRR = "SELECT "
            for attr in attrs:
                sqlRR += f"{attr},"
            sqlRR = sqlRR[:-1]
            sqlRR += f" FROM {panda_config.schemaJEDI}.JEDI_Dataset_Contents "
            sqlRR += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            varMap = {}
            varMap[":jediTaskID"] = jediTaskID
            varMap[":datasetID"] = datasetID
            varMap[":fileID"] = fileID
            # start transaction
            self.conn.begin()
            self.cur.execute(sqlRR + comment, varMap)
            resRR = self.cur.fetchone()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            retVal = {}
            if resRR is not None:
                for idx, attr in enumerate(attrs):
                    retVal[attr] = resRR[idx]
            tmpLog.debug(f"done {str(retVal)}")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return {}

    # get workers for a job
    def getWorkersForJob(self, PandaID):
        comment = " /* DBProxy.getWorkersForJob */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" < PandaID={PandaID} >"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            # sql to get workers
            sqlC = f"SELECT {WorkerSpec.columnNames(prefix='w')} FROM ATLAS_PANDA.Harvester_Workers w, ATLAS_PANDA.Harvester_Rel_Jobs_Workers r "
            sqlC += "WHERE w.harvesterID=r.harvesterID AND w.workerID=r.workerID AND r.PandaID=:PandaID "
            varMap = {}
            varMap[":PandaID"] = PandaID
            # start transaction
            self.conn.begin()
            self.cur.execute(sqlC + comment, varMap)
            resCs = self.cur.fetchall()
            retList = []
            for resC in resCs:
                workerSpec = WorkerSpec()
                workerSpec.pack(resC)
                retList.append(workerSpec)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"got {len(retList)} workers")
            return retList
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return []

    # get user job metadata
    def getUserJobMetadata(self, jediTaskID):
        comment = " /* DBProxy.getUserJobMetadata */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" < jediTaskID={jediTaskID} >"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            # sql to get workers
            sqlC = "SELECT j.PandaID,m.metaData FROM {0} j,{1} m "
            sqlC += "WHERE j.jediTaskID=:jediTaskID AND j.jobStatus=:jobStatus AND m.PandaID=j.PandaID AND j.prodSourceLabel=:label "
            retMap = dict()
            for a, m in [
                ("ATLAS_PANDA.jobsArchived4", "ATLAS_PANDA.metaTable"),
                ("ATLAS_PANDAARCH.jobsArchived", "ATLAS_PANDAARCH.metaTable_ARCH"),
            ]:
                sql = sqlC.format(a, m)
                varMap = {}
                varMap[":jediTaskID"] = jediTaskID
                varMap[":label"] = "user"
                varMap[":jobStatus"] = "finished"
                # start transaction
                self.conn.begin()
                self.cur.execute(sql + comment, varMap)
                resCs = self.cur.fetchall()
                for pandaID, clobMeta in resCs:
                    try:
                        metadata = clobMeta.read()
                    except AttributeError:
                        metadata = str(clobMeta)
                    try:
                        retMap[pandaID] = json.loads(metadata)
                    except Exception:
                        pass
                # commit
                if not self._commit():
                    raise RuntimeError("Commit error")
            tmpLog.debug(f"got {len(retMap)} data blocks")
            return retMap
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return {}

    # get jumbo job datasets
    def getJumboJobDatasets(self, n_days, grace_period):
        comment = " /* DBProxy.getUserJobMetadata */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" < nDays={n_days} >"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            # sql to get workers
            sqlC = "SELECT t.jediTaskID,d.datasetName,t.status FROM ATLAS_PANDA.JEDI_Tasks t,ATLAS_PANDA.JEDI_Datasets d "
            sqlC += "WHERE t.prodSourceLabel='managed' AND t.useJumbo IS NOT NULL "
            sqlC += "AND t.modificationTime>CURRENT_DATE-:days AND t.modificationTime<CURRENT_DATE-:grace_period "
            sqlC += "AND t.status IN ('finished','done') "
            sqlC += "AND d.jediTaskID=t.jediTaskID AND d.type='output' "
            varMap = {}
            varMap[":days"] = n_days
            varMap[":grace_period"] = grace_period
            # start transaction
            self.conn.begin()
            self.cur.execute(sqlC + comment, varMap)
            resCs = self.cur.fetchall()
            retMap = dict()
            nDS = 0
            for jediTaskID, datasetName, status in resCs:
                retMap.setdefault(jediTaskID, {"status": status, "datasets": []})
                retMap[jediTaskID]["datasets"].append(datasetName)
                nDS += 1
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"got {nDS} datasets")
            return retMap
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return {}

    def getGShareStatus(self):
        """
        Generates a list with sorted leave branches
        """

        comment = " /* DBProxy.getGShareStatus */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")

        self.__reload_shares()
        self.__reload_hs_distribution()
        sorted_shares = self.tree.sort_branch_by_current_hs_distribution(self.__hs_distribution)

        sorted_shares_export = []
        for share in sorted_shares:
            sorted_shares_export.append(
                {
                    "name": share.name,
                    "running": self.__hs_distribution[share.name]["executing"],
                    "target": self.__hs_distribution[share.name]["pledged"],
                    "queuing": self.__hs_distribution[share.name]["queued"],
                }
            )
        return sorted_shares_export

    # get output datasets
    def getOutputDatasetsJEDI(self, panda_id):
        comment = " /* DBProxy.getOutputDatasetsJEDI */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" < PandaID={panda_id} >"
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            # sql to get workers
            sqlC = "SELECT d.datasetID,d.datasetName FROM ATLAS_PANDA.filesTable4 f,ATLAS_PANDA.JEDI_Datasets d "
            sqlC += "WHERE f.PandaID=:PandaID AND f.type IN (:type1,:type2) AND d.jediTaskID=f.jediTaskID AND d.datasetID=f.datasetID "
            varMap = {}
            varMap[":PandaID"] = panda_id
            varMap[":type1"] = "output"
            varMap[":type2"] = "log"
            # start transaction
            self.conn.begin()
            self.cur.execute(sqlC + comment, varMap)
            retMap = dict()
            resCs = self.cur.fetchall()
            for datasetID, datasetName in resCs:
                retMap[datasetID] = datasetName
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"got {len(retMap)}")
            return retMap
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return {}

    def getQueuesInJSONSchedconfig(self):
        comment = " /* DBProxy.getQueuesInJSONSchedconfig */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")
        try:
            # sql to get workers
            sqlC = "SELECT /* use_json_type */ panda_queue FROM ATLAS_PANDA.schedconfig_json"
            # start transaction
            self.conn.begin()
            self.cur.execute(sqlC + comment)
            panda_queues = [row[0] for row in self.cur.fetchall()]
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"got {len(panda_queues)} queues")
            return panda_queues
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None

    # update queues
    def upsertQueuesInJSONSchedconfig(self, schedconfig_dump):
        comment = " /* DBProxy.upsertQueuesInJSONSchedconfig */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")

        if not schedconfig_dump:
            tmp_log.error("empty schedconfig dump")
            return "ERROR"

        try:
            existing_queues = self.getQueuesInJSONSchedconfig()
            if existing_queues is None:
                tmp_log.error("Could not retrieve already existing queues")
                return None

            # separate the queues to the ones we have to update (existing) and the ones we have to insert (new)
            var_map_insert = []
            var_map_update = []
            utc_now = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            for pq in schedconfig_dump:
                data = json.dumps(schedconfig_dump[pq])
                if not data:
                    tmp_log.error(f"no data for {pq}")
                    continue

                if pq in existing_queues:
                    tmp_log.debug(f"pq {pq} present")
                    var_map_update.append({":pq": pq, ":data": data, ":last_update": utc_now})
                else:
                    tmp_log.debug(f"pq {pq} is new")
                    var_map_insert.append({":pq": pq, ":data": data, ":last_update": utc_now})

            # start transaction
            self.conn.begin()

            # run the updates
            if var_map_update:
                sql_update = """
                             UPDATE ATLAS_PANDA.SCHEDCONFIG_JSON SET data = :data, last_update = :last_update
                             WHERE panda_queue = :pq
                             """
                tmp_log.debug("start updates")
                self.cur.executemany(sql_update + comment, var_map_update)
                tmp_log.debug("finished updates")

            # run the inserts
            if var_map_insert:
                sql_insert = """
                             INSERT INTO ATLAS_PANDA.SCHEDCONFIG_JSON (panda_queue, data, last_update)
                             VALUES (:pq, :data, :last_update)
                             """
                tmp_log.debug("start inserts")
                self.cur.executemany(sql_insert + comment, var_map_insert)
                tmp_log.debug("finished inserts")

            # delete inactive queues
            tmp_log.debug("Going to delete obsoleted queues")
            sql_delete = """
                         DELETE FROM ATLAS_PANDA.SCHEDCONFIG_JSON WHERE last_update < current_date - INTERVAL '7' DAY
                         """
            self.cur.execute(sql_delete + comment)
            tmp_log.debug("deleted old queues")

            if not self._commit():
                raise RuntimeError("Commit error")

            tmp_log.debug("done")
            return "OK"

        except Exception:
            # roll back
            self._rollback()
            self.dumpErrorMessage(_logger, method_name)
            return "ERROR"

    # update queues
    def loadSWTags(self, sw_tags):
        comment = " /* DBProxy.loadSWTags */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")

        if not sw_tags:
            tmp_log.error("empty sw tag dump")
            return "ERROR"

        try:
            var_map_tags = []

            utc_now = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            for pq in sw_tags:
                data = sw_tags[pq]
                var_map_tags.append({":pq": pq, ":data": json.dumps(data), ":last_update": utc_now})

            # start transaction on SW_TAGS table
            # delete everything in the table to start every time from a clean table
            # cleaning and filling needs to be done within the same transaction
            self.conn.begin()

            sql_delete = "DELETE FROM ATLAS_PANDA.SW_TAGS"
            tmp_log.debug("start cleaning up SW_TAGS table")
            self.cur.execute(sql_delete + comment)
            tmp_log.debug("done cleaning up SW_TAGS table")

            sql_insert = "INSERT INTO ATLAS_PANDA.SW_TAGS (panda_queue, data, last_update) VALUES (:pq, :data, :last_update)"
            tmp_log.debug("start filling up SW_TAGS table")
            for shard in create_shards(var_map_tags, 100):  # insert in batches of 100 rows
                self.cur.executemany(sql_insert + comment, shard)
            tmp_log.debug("done filling up table")
            if not self._commit():
                raise RuntimeError("Commit error")

            tmp_log.debug("done")
            return "OK"

        except Exception:
            # roll back
            self._rollback()
            self.dumpErrorMessage(_logger, method_name)
            return "ERROR"

    # update queues
    def sweepPQ(self, panda_queue_des, status_list_des, ce_list_des, submission_host_list_des):
        comment = " /* DBProxy.sweepPQ */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")

        try:
            # Figure out the harvester instance serving the queues and check the CEs match
            pq_data_des = self.get_config_for_pq(panda_queue_des)
            if not pq_data_des:
                return "Error retrieving queue configuration from DB"

            harvester_id = pq_data_des["harvester"]
            if not harvester_id:
                return "Queue not served by any harvester ID"

            # check CEs
            if ce_list_des == "ALL":
                ce_list_des_sanitized = "ALL"
            else:
                computing_elements = pq_data_des["queues"]
                ce_names = [str(ce["ce_endpoint"]) for ce in computing_elements]
                ce_list_des_sanitized = [ce for ce in ce_list_des if ce in ce_names]

            # we can't correct submission hosts or the status list

            command = "KILL_WORKERS"
            ack_requested = False
            status = "new"
            lock_interval = None
            com_interval = None
            params = {
                "status": status_list_des,
                "computingSite": [panda_queue_des],
                "computingElement": ce_list_des_sanitized,
                "submissionHost": submission_host_list_des,
            }

            self.commandToHarvester(
                harvester_id,
                command,
                ack_requested,
                status,
                lock_interval,
                com_interval,
                params,
            )

            tmp_log.debug("done")
            return "OK"

        except Exception:
            self.dumpErrorMessage(_logger, method_name)
            return "Problem generating command. Check PanDA server logs"

    def get_config_for_pq(self, pq_name):
        """
        Get the CRIC json configuration for a particular queue
        """

        comment = " /* DBProxy.get_config_for_pq */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, f"<{method_name} {pq_name}>")
        tmp_log.debug("start")

        var_map = {":pq": pq_name}
        sql_get_queue_config = """
        SELECT data FROM ATLAS_PANDA.SCHEDCONFIG_JSON
        WHERE panda_queue = :pq
        """
        tmp_v, pq_data = self.getClobObj(sql_get_queue_config + comment, var_map)
        if pq_data is None:
            tmp_log.error("Could not find queue configuration")
            return None

        try:
            pq_data_des = pq_data[0][0]
            if not isinstance(pq_data_des, dict):
                pq_data_des = json.loads(pq_data_des)
        except Exception:
            tmp_log.error("Could not find queue configuration")
            return None

        tmp_log.debug("done")
        return pq_data_des

    # lock process
    def lockProcess_PANDA(self, component, pid, time_limit, force=False):
        comment = " /* DBProxy.lockProcess_PANDA */"
        method_name = "lockProcess_PANDA"
        # defaults
        vo = "default"
        prodSourceLabel = "default"
        cloud = "default"
        workqueue_id = 0
        resource_name = "default"
        method_name += f" <component={component} pid={pid}>"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        try:
            retVal = False
            # sql to check
            sqlCT = (
                "SELECT lockedBy "
                "FROM {0}.JEDI_Process_Lock "
                "WHERE vo=:vo AND prodSourceLabel=:prodSourceLabel "
                "AND cloud=:cloud AND workqueue_id=:workqueue_id "
                "AND resource_type=:resource_name AND component=:component "
                "AND lockedTime>:lockedTime "
                "FOR UPDATE"
            ).format(panda_config.schemaJEDI)
            # sql to delete
            sqlCD = (
                "DELETE FROM {0}.JEDI_Process_Lock "
                "WHERE vo=:vo AND prodSourceLabel=:prodSourceLabel "
                "AND cloud=:cloud AND workqueue_id=:workqueue_id "
                "AND resource_type=:resource_name AND component=:component "
            ).format(panda_config.schemaJEDI)
            # sql to insert
            sqlFR = (
                "INSERT INTO {0}.JEDI_Process_Lock "
                "(vo, prodSourceLabel, cloud, workqueue_id, resource_type, component, lockedBy, lockedTime) "
                "VALUES(:vo, :prodSourceLabel, :cloud, :workqueue_id, :resource_name, :component, :lockedBy, CURRENT_DATE) "
            ).format(panda_config.schemaJEDI)
            # start transaction
            self.conn.begin()
            # check
            if not force:
                varMap = {}
                varMap[":vo"] = vo
                varMap[":prodSourceLabel"] = prodSourceLabel
                varMap[":cloud"] = cloud
                varMap[":workqueue_id"] = workqueue_id
                varMap[":resource_name"] = resource_name
                varMap[":component"] = component
                varMap[":lockedTime"] = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(minutes=time_limit)
                self.cur.execute(sqlCT + comment, varMap)
                resCT = self.cur.fetchone()
            else:
                resCT = None
            if resCT is not None:
                tmp_log.debug(f"skipped, locked by {resCT[0]}")
            else:
                # delete
                varMap = {}
                varMap[":vo"] = vo
                varMap[":prodSourceLabel"] = prodSourceLabel
                varMap[":cloud"] = cloud
                varMap[":workqueue_id"] = workqueue_id
                varMap[":resource_name"] = resource_name
                varMap[":component"] = component
                self.cur.execute(sqlCD + comment, varMap)
                # insert
                varMap = {}
                varMap[":vo"] = vo
                varMap[":prodSourceLabel"] = prodSourceLabel
                varMap[":cloud"] = cloud
                varMap[":workqueue_id"] = workqueue_id
                varMap[":resource_name"] = resource_name
                varMap[":component"] = component
                varMap[":lockedBy"] = pid
                self.cur.execute(sqlFR + comment, varMap)
                tmp_log.debug("successfully locked")
                retVal = True
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmp_log, method_name)
            return retVal

    # unlock process
    def unlockProcess_PANDA(self, component, pid):
        comment = " /* DBProxy.unlockProcess_PANDA */"
        method_name = "unlockProcess_PANDA"
        # defaults
        vo = "default"
        prodSourceLabel = "default"
        cloud = "default"
        workqueue_id = 0
        resource_name = "default"
        method_name += f" <component={component} pid={pid}>"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        try:
            retVal = False
            # sql to delete
            sqlCD = (
                "DELETE FROM {0}.JEDI_Process_Lock "
                "WHERE vo=:vo AND prodSourceLabel=:prodSourceLabel AND cloud=:cloud "
                "AND workqueue_id=:workqueue_id AND lockedBy=:lockedBy "
                "AND resource_type=:resource_name AND component=:component "
            ).format(panda_config.schemaJEDI)
            # start transaction
            self.conn.begin()
            # check
            varMap = {}
            varMap[":vo"] = vo
            varMap[":prodSourceLabel"] = prodSourceLabel
            varMap[":cloud"] = cloud
            varMap[":workqueue_id"] = workqueue_id
            varMap[":resource_name"] = resource_name
            varMap[":component"] = component
            varMap[":lockedBy"] = pid
            self.cur.execute(sqlCD + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmp_log.debug("done")
            retVal = True
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmp_log, method_name)
            return retVal

    # check process lock
    def checkProcessLock_PANDA(self, component, pid, time_limit, check_base=False):
        comment = " /* DBProxy.checkProcessLock_PANDA */"
        method_name = "checkProcessLock_PANDA"
        # defaults
        vo = "default"
        prodSourceLabel = "default"
        cloud = "default"
        workqueue_id = 0
        resource_name = "default"
        method_name += f" <component={component} pid={pid}>"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        try:
            retVal = False, None
            # sql to check
            sqlCT = (
                "SELECT lockedBy, lockedTime "
                "FROM {0}.JEDI_Process_Lock "
                "WHERE vo=:vo AND prodSourceLabel=:prodSourceLabel "
                "AND cloud=:cloud AND workqueue_id=:workqueue_id "
                "AND resource_type=:resource_name AND component=:component "
                "AND lockedTime>:lockedTime "
            ).format(panda_config.schemaJEDI)
            # start transaction
            self.conn.begin()
            # check
            varMap = {}
            varMap[":vo"] = vo
            varMap[":prodSourceLabel"] = prodSourceLabel
            varMap[":cloud"] = cloud
            varMap[":workqueue_id"] = workqueue_id
            varMap[":resource_name"] = resource_name
            varMap[":component"] = component
            varMap[":lockedTime"] = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(minutes=time_limit)
            self.cur.execute(sqlCT + comment, varMap)
            resCT = self.cur.fetchone()
            if resCT is not None:
                lockedBy, lockedTime = resCT
                if check_base:
                    # check only base part
                    if not lockedBy.startswith(pid):
                        retVal = True, lockedTime
                else:
                    # check whole string
                    if lockedBy != pid:
                        retVal = True, lockedTime
                if retVal[0]:
                    tmp_log.debug(f"found locked by {lockedBy} at {lockedTime.strftime('%Y-%m-%d_%H:%M:%S')}")
                else:
                    tmp_log.debug("found unlocked")
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmp_log.debug("done")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmp_log, method_name)
            return retVal

    # insert job output report
    def insertJobOutputReport(self, panda_id, prod_source_label, job_status, attempt_nr, data):
        comment = " /* DBProxy.insertJobOutputReport */"
        method_name = "insertJobOutputReport"
        # defaults
        method_name += f" <PandaID={panda_id} attemptNr={attempt_nr}>"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        # sql to insert
        sqlI = (
            "INSERT INTO {0}.Job_Output_Report "
            "(PandaID, prodSourceLabel, jobStatus, attemptNr, data, timeStamp) "
            "VALUES(:PandaID, :prodSourceLabel, :jobStatus, :attemptNr, :data, :timeStamp) "
        ).format(panda_config.schemaPANDA)
        try:
            retVal = False
            # start transaction
            self.conn.begin()
            # insert
            varMap = {}
            varMap[":PandaID"] = panda_id
            varMap[":prodSourceLabel"] = prod_source_label
            varMap[":jobStatus"] = job_status
            varMap[":attemptNr"] = attempt_nr
            varMap[":data"] = data
            varMap[":timeStamp"] = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            self.cur.execute(sqlI + comment, varMap)
            tmp_log.debug("successfully inserted")
            retVal = True
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmp_log.debug("done")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmp_log, method_name)
            return retVal

    # update data of job output report
    def updateJobOutputReport(self, panda_id, attempt_nr, data):
        comment = " /* DBProxy.updateJobOutputReport */"
        method_name = "updateJobOutputReport"
        # defaults
        method_name += f" <PandaID={panda_id} attemptNr={attempt_nr}>"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        # try to lock
        try:
            retVal = False
            # sql to update
            sqlU = f"UPDATE {panda_config.schemaPANDA}.Job_Output_Report SET data=:data, timeStamp=:timeStamp WHERE PandaID=:PandaID AND attemptNr=:attemptNr "
            # start transaction
            self.conn.begin()
            # update
            varMap = {}
            varMap[":PandaID"] = panda_id
            varMap[":attemptNr"] = attempt_nr
            varMap[":data"] = data
            varMap[":timeStamp"] = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            self.cur.execute(sqlU + comment, varMap)
            nRow = self.cur.rowcount
            if nRow == 1:
                tmp_log.debug("successfully updated")
                retVal = True
            elif nRow == 0:
                tmp_log.debug("entry not found, not updated")
            else:
                tmp_log.warning(f"updated unspecific number of rows: {nRow}")
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmp_log.debug("done")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmp_log, method_name)
            return retVal

    # deleted job output report
    def deleteJobOutputReport(self, panda_id, attempt_nr):
        comment = " /* DBProxy.deleteJobOutputReport */"
        method_name = "deleteJobOutputReport"
        # defaults
        method_name += f" <PandaID={panda_id} attemptNr={attempt_nr}>"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        # sql to delete
        sqlD = f"DELETE FROM {panda_config.schemaPANDA}.Job_Output_Report WHERE PandaID=:PandaID AND attemptNr=:attemptNr "
        try:
            retVal = False
            # start transaction
            self.conn.begin()
            # delete
            varMap = {}
            varMap[":PandaID"] = panda_id
            varMap[":attemptNr"] = attempt_nr
            self.cur.execute(sqlD + comment, varMap)
            tmp_log.debug("successfully deleted")
            retVal = True
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmp_log.debug("done")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmp_log, method_name)
            return retVal

    # get record of a job output report
    def getJobOutputReport(self, panda_id, attempt_nr):
        comment = " /* DBProxy.getJobOutputReport */"
        method_name = "getJobOutputReport"
        method_name += f" <PandaID={panda_id} attemptNr={attempt_nr}>"
        # defaults
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        # try to lock
        try:
            retVal = {}
            # sql to get records
            sqlGR = (
                "SELECT PandaID,prodSourceLabel,jobStatus,attemptNr,data,timeStamp,lockedBy,lockedTime "
                "FROM {0}.Job_Output_Report "
                "WHERE PandaID=:PandaID AND attemptNr=:attemptNr "
            ).format(panda_config.schemaPANDA)
            # start transaction
            self.conn.begin()
            # check
            varMap = {}
            varMap[":PandaID"] = panda_id
            varMap[":attemptNr"] = attempt_nr
            self.cur.execute(sqlGR + comment, varMap)
            resGR = self.cur.fetchall()
            if not resGR:
                tmp_log.debug("record does not exist, skipped")
            for (
                PandaID,
                prodSourceLabel,
                jobStatus,
                attemptNr,
                data,
                timeStamp,
                lockedBy,
                lockedTime,
            ) in resGR:
                # fill result
                retVal = {
                    "PandaID": PandaID,
                    "attemptNr": prodSourceLabel,
                    "jobStatus": jobStatus,
                    "attemptNr": attemptNr,
                    "timeStamp": timeStamp,
                    "data": data,
                    "lockedBy": lockedBy,
                    "lockedTime": lockedTime,
                }
                tmp_log.debug("got record")
                break
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmp_log.debug("done")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmp_log, method_name)
            return retVal

    # lock job output report
    def lockJobOutputReport(self, panda_id, attempt_nr, pid, time_limit, take_over_from=None):
        comment = " /* DBProxy.lockJobOutputReport */"
        method_name = "lockJobOutputReport"
        method_name += f" <PandaID={panda_id} attemptNr={attempt_nr}>"
        # defaults
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        # try to lock
        try:
            retVal = []
            # sql to get lock
            sqlGL = (
                "SELECT PandaID,attemptNr "
                "FROM {0}.Job_Output_Report "
                "WHERE PandaID=:PandaID AND attemptNr=:attemptNr "
                "AND (lockedBy IS NULL OR lockedBy=:lockedBy OR lockedTime<:lockedTime) "
                "FOR UPDATE NOWAIT "
            ).format(panda_config.schemaPANDA)
            # sql to update lock
            sqlUL = (
                "UPDATE {0}.Job_Output_Report " "SET lockedBy=:lockedBy, lockedTime=:lockedTime " "WHERE PandaID=:PandaID AND attemptNr=:attemptNr "
            ).format(panda_config.schemaPANDA)
            # start transaction
            self.conn.begin()
            # check
            varMap = {}
            varMap[":PandaID"] = panda_id
            varMap[":attemptNr"] = attempt_nr
            if take_over_from is None:
                varMap[":lockedBy"] = pid
            else:
                varMap[":lockedBy"] = take_over_from
            varMap[":lockedTime"] = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(minutes=time_limit)
            utc_now = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            try:
                self.cur.execute(sqlGL + comment, varMap)
                resGL = self.cur.fetchall()
                if not resGL:
                    tmp_log.debug("record already locked by other thread, skipped")
            except Exception:
                resGL = None
                tmp_log.debug("record skipped due to NOWAIT")
            if resGL:
                for panda_id, attempt_nr in resGL:
                    # lock
                    varMap = {}
                    varMap[":PandaID"] = panda_id
                    varMap[":attemptNr"] = attempt_nr
                    varMap[":lockedBy"] = pid
                    varMap[":lockedTime"] = utc_now
                    self.cur.execute(sqlUL + comment, varMap)
                    if take_over_from is None:
                        tmp_log.debug(f"successfully locked record by {pid}")
                    else:
                        tmp_log.debug(f"successfully took over locked record from {take_over_from} by {pid}")
                    retVal = True
                    break
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmp_log.debug("done")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmp_log, method_name)
            return retVal

    # unlock job output report
    def unlockJobOutputReport(self, panda_id, attempt_nr, pid, lock_offset):
        comment = " /* DBProxy.unlockJobOutputReport */"
        method_name = "unlockJobOutputReport"
        method_name += f" <PandaID={panda_id} attemptNr={attempt_nr}>"
        # defaults
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        # try to lock
        try:
            retVal = []
            # sql to get lock
            sqlGL = (
                "SELECT PandaID,attemptNr "
                "FROM {0}.Job_Output_Report "
                "WHERE PandaID=:PandaID AND attemptNr=:attemptNr "
                "AND lockedBy=:lockedBy "
                "FOR UPDATE"
            ).format(panda_config.schemaPANDA)
            # sql to update lock
            sqlUL = f"UPDATE {panda_config.schemaPANDA}.Job_Output_Report SET lockedTime=:lockedTime WHERE PandaID=:PandaID AND attemptNr=:attemptNr "
            # start transaction
            self.conn.begin()
            # check
            varMap = {}
            varMap[":PandaID"] = panda_id
            varMap[":attemptNr"] = attempt_nr
            varMap[":lockedBy"] = pid
            self.cur.execute(sqlGL + comment, varMap)
            resGL = self.cur.fetchall()
            if not resGL:
                tmp_log.debug("record not locked by this thread, skipped")
            else:
                for panda_id, attempt_nr in resGL:
                    # lock
                    varMap = {}
                    varMap[":PandaID"] = panda_id
                    varMap[":attemptNr"] = attempt_nr
                    varMap[":lockedTime"] = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(minutes=lock_offset)
                    self.cur.execute(sqlUL + comment, varMap)
                    tmp_log.debug("successfully unlocked record")
                    retVal = True
                    break
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmp_log.debug("done")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmp_log, method_name)
            return retVal

    # list pandaID, jobStatus, attemptNr, timeStamp of job output report
    def listJobOutputReport(self, only_unlocked, time_limit, limit, grace_period, labels, anti_labels):
        comment = " /* DBProxy.listJobOutputReport */"
        method_name = "listJobOutputReport"
        # defaults
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug(f"start label={str(labels)} anti_label={str(anti_labels)}")
        try:
            retVal = None
            if only_unlocked:
                # try to get only records unlocked or with expired lock
                varMap = {}
                varMap[":limit"] = limit
                varMap[":lockedTime"] = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(minutes=time_limit)
                varMap[":timeStamp"] = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(seconds=grace_period)
                # sql to get record
                sqlGR = (
                    "SELECT * "
                    "FROM ( "
                    "SELECT PandaID,jobStatus,attemptNr,timeStamp "
                    "FROM {0}.Job_Output_Report "
                    "WHERE (lockedBy IS NULL OR lockedTime<:lockedTime) "
                    "AND timeStamp<:timeStamp ".format(panda_config.schemaPANDA)
                )
                if labels is not None:
                    sqlGR += "AND prodSourceLabel IN ("
                    for l in labels:
                        k = f":l_{l}"
                        varMap[k] = l
                        sqlGR += f"{k},"
                    sqlGR = sqlGR[:-1]
                    sqlGR += ") "
                if anti_labels is not None:
                    sqlGR += "AND prodSourceLabel NOT IN ("
                    for l in anti_labels:
                        k = f":al_{l}"
                        varMap[k] = l
                        sqlGR += f"{k},"
                    sqlGR = sqlGR[:-1]
                    sqlGR += ") "
                sqlGR += "ORDER BY timeStamp " ") " "WHERE rownum<=:limit "
                # start transaction
                self.conn.begin()
                # check
                self.cur.execute(sqlGR + comment, varMap)
                retVal = self.cur.fetchall()
                tmp_log.debug(f"listed {len(retVal)} unlocked records")
            else:
                # sql to select
                sqlS = (
                    "SELECT * "
                    "FROM ( "
                    "SELECT PandaID,jobStatus,attemptNr,timeStamp "
                    "FROM {0}.Job_Output_Report "
                    "ORDER BY timeStamp "
                    ") "
                    "WHERE rownum<=:limit "
                ).format(panda_config.schemaPANDA)
                # start transaction
                self.conn.begin()
                varMap = {}
                varMap[":limit"] = limit
                # check
                self.cur.execute(sqlS + comment, varMap)
                retVal = self.cur.fetchall()
                tmp_log.debug(f"listed {len(retVal)} records")
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmp_log.debug("done")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmp_log, method_name)
            return retVal

    # update problematic resource info for user
    def update_problematic_resource_info(self, user_name, jedi_task_id, resource, problem_type):
        comment = " /* DBProxy.update_problematic_resource_info */"
        method_name = comment.split()[1].split(".")[-1]
        method_name += f" < user={user_name} jediTaskID={jedi_task_id} >"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        retVal = False
        try:
            if problem_type not in ["dest", None]:
                tmp_log.debug(f"unknown problem type: {problem_type}")
                return None
            sqlR = "SELECT pagecache FROM ATLAS_PANDAMETA.users " "WHERE name=:name "
            sqlW = "UPDATE ATLAS_PANDAMETA.users SET pagecache=:data " "WHERE name=:name "
            # string to use a dict key
            jedi_task_id = str(jedi_task_id)
            # start transaction
            self.conn.begin()
            # read
            varMap = {}
            varMap[":name"] = user_name
            self.cur.execute(sqlR + comment, varMap)
            data = self.cur.fetchone()
            if data is None:
                tmp_log.debug("user not found")
            else:
                try:
                    data = json.loads(data[0])
                except Exception:
                    data = {}
                if problem_type is not None:
                    data.setdefault(problem_type, {})
                    data[problem_type].setdefault(jedi_task_id, {})
                    data[problem_type][jedi_task_id].setdefault(resource, None)
                    old = data[problem_type][jedi_task_id][resource]
                    if old is None or datetime.datetime.now(datetime.timezone.utc) - datetime.datetime.fromtimestamp(
                        old, datetime.timezone.utc
                    ) > datetime.timedelta(days=1):
                        retVal = True
                        data[problem_type][jedi_task_id][resource] = time.time()
                # delete old data
                for p in list(data):
                    for t in list(data[p]):
                        for r in list(data[p][t]):
                            ts = data[p][t][r]
                            if datetime.datetime.now(datetime.timezone.utc) - datetime.datetime.fromtimestamp(ts, datetime.timezone.utc) > datetime.timedelta(
                                days=7
                            ):
                                del data[p][t][r]
                        if not data[p][t]:
                            del data[p][t]
                    if not data[p]:
                        del data[p]
                # update
                varMap = {}
                varMap[":name"] = user_name
                varMap[":data"] = json.dumps(data)
                self.cur.execute(sqlW + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmp_log.debug(f"done with {retVal} : {str(data)}")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, method_name)
            return None

    # send command to a job
    def send_command_to_job(self, panda_id, com):
        comment = " /* DBProxy.send_command_to_job */"
        method_name = comment.split()[1].split(".")[-1]
        method_name += f" < PandaID={panda_id} >"
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")
        retVal = None
        try:
            # check length
            new_com = JobSpec.truncateStringAttr("commandToPilot", com)
            if len(new_com) != len(com):
                retVal = (
                    False,
                    f"command string too long. must be less than {len(new_com)} chars",
                )
            else:
                sqlR = "SELECT commandToPilot FROM ATLAS_PANDA.{} WHERE PandaID=:PandaID FOR UPDATE "
                sqlU = "UPDATE ATLAS_PANDA.{} SET commandToPilot=:commandToPilot " "WHERE PandaID=:PandaID "
                for table in ["jobsDefined4", "jobsActive4"]:
                    # start transaction
                    self.conn.begin()
                    # read
                    varMap = {}
                    varMap[":PandaID"] = panda_id
                    self.cur.execute(sqlR.format(table) + comment, varMap)
                    data = self.cur.fetchone()
                    if data is not None:
                        (commandToPilot,) = data
                        if commandToPilot == "tobekilled":
                            retVal = (False, "job is being killed")
                        else:
                            varMap = {}
                            varMap[":PandaID"] = panda_id
                            varMap[":commandToPilot"] = com
                            self.cur.execute(sqlU.format(table) + comment, varMap)
                            nRow = self.cur.rowcount
                            if nRow:
                                retVal = (True, "command received")
                    # commit
                    if not self._commit():
                        raise RuntimeError("Commit error")
                    if retVal is not None:
                        break
            if retVal is None:
                retVal = (False, f"no active job with PandaID={panda_id}")
            tmp_log.debug(f"done with {str(retVal)}")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, method_name)
            return False, "database error"

    # get parent task id
    def get_parent_task_id_with_name(self, user_name, parent_name):
        comment = " /* DBProxy.get_task_id_with_dataset */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        try:
            methodName += f" <{user_name}>"
            tmpLog = LogWrapper(_logger, methodName)
            tmpLog.debug(f"try to find parent={parent_name}")
            # sql to get workers
            sqlC = "SELECT jediTaskID FROM ATLAS_PANDA.JEDI_Tasks " "WHERE userName=:userName AND taskName=:taskName " "ORDER BY jediTaskID DESC "
            # start transaction
            self.conn.begin()
            varMap = {}
            varMap[":userName"] = user_name
            varMap[":taskName"] = parent_name
            self.cur.execute(sqlC + comment, varMap)
            tid = self.cur.fetchone()
            if tid:
                (tid,) = tid
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug(f"got {tid}")
            return tid
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None

    # get workers with stale harvester states and newer pilot state
    def get_workers_to_synchronize(self):
        comment = " /* DBProxy.get_workers_to_synchronize */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        try:
            tmp_log.debug("Starting")

            # give harvester a chance to discover the status change itself
            discovery_period = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(minutes=60)
            # don't repeat the same workers in each cycle
            retry_period = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None) - datetime.timedelta(minutes=30)

            # Select workers where the status is more advanced according to the pilot than to harvester
            sql_select = """
            SELECT /*+ INDEX_RS_ASC(harvester_workers HARVESTER_WORKERS_STATUS_IDX) */ harvesterID, workerID, pilotStatus
            FROM ATLAS_PANDA.harvester_workers
            WHERE (status in ('submitted', 'ready') AND pilotStatus='running' AND pilotStartTime < :discovery_period)
            OR (status in ('submitted', 'ready', 'running', 'idle') AND pilotStatus='finished' AND pilotEndTime < :discovery_period)
            AND lastupdate > sysdate - interval '7' day
            AND submittime > sysdate - interval '14' day
            AND (pilotStatusSyncTime > :retry_period OR pilotStatusSyncTime IS NULL)
            FOR UPDATE
            """
            var_map = {
                ":discovery_period": discovery_period,
                ":retry_period": retry_period,
            }

            now_ts = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            sql_update = """
            UPDATE ATLAS_PANDA.harvester_workers
            SET pilotStatusSyncTime = :lastSync
            WHERE harvesterID= :harvesterID
            AND workerID= :workerID
            """

            # run query to select workers
            self.conn.begin()
            self.cur.arraysize = 10000
            self.cur.execute(sql_select + comment, var_map)
            db_workers = self.cur.fetchall()

            # prepare workers and separate by harvester instance and site
            workers_to_sync = {}
            var_maps = []
            for harvester_id, worker_id, pilot_status in db_workers:
                workers_to_sync.setdefault(harvester_id, {})
                workers_to_sync[harvester_id].setdefault(pilot_status, [])

                # organization for harvester commands
                workers_to_sync[harvester_id][pilot_status].append(worker_id)
                # organization to set lastSync
                var_maps.append(
                    {
                        ":workerID": worker_id,
                        ":harvesterID": harvester_id,
                        ":lastSync": now_ts,
                    }
                )

            self.cur.executemany(sql_update + comment, var_maps)

            # commit
            if not self._commit():
                raise RuntimeError("Commit error")

            tmp_log.debug("Done")
            return workers_to_sync
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmp_log, method_name)
            return []

    # get users and groups to boost job priorities
    def get_dict_to_boost_job_prio(self, vo):
        comment = " /* DBProxy.get_dict_to_boost_job_prio */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        if self.job_prio_boost_dict_update_time and datetime.datetime.now(datetime.timezone.utc).replace(
            tzinfo=None
        ) - self.job_prio_boost_dict_update_time < datetime.timedelta(minutes=15):
            return self.job_prio_boost_dict
        try:
            self.job_prio_boost_dict_update_time = datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None)
            self.job_prio_boost_dict = {}
            # get configs
            tmpLog = LogWrapper(_logger, methodName)
            # sql to get configs
            sqlC = "SELECT value FROM ATLAS_PANDA.Config " "WHERE app=:app AND component=:component AND vo=:vo AND key LIKE :key "
            # start transaction
            self.conn.begin()
            varMap = {}
            varMap[":app"] = "pandaserver"
            varMap[":component"] = "dbproxy"
            varMap[":vo"] = vo
            varMap[":key"] = "USER_JOB_PRIO_BOOST_LIST_%"
            self.cur.execute(sqlC + comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            # parse list
            for (tmp_data,) in res:
                if tmp_data:
                    for tmp_item in tmp_data.split(","):
                        try:
                            tmp_name, tmp_type, tmp_prio, tmp_expire = tmp_item.split(":")
                            # check expiration
                            if tmp_expire:
                                tmp_expire = datetime.datetime.strptime(tmp_expire, "%Y%m%d")
                                if tmp_expire < datetime.datetime.now(datetime.timezone.utc).replace(tzinfo=None):
                                    continue
                            self.job_prio_boost_dict.setdefault(tmp_type, {})
                            self.job_prio_boost_dict[tmp_type][tmp_name] = int(tmp_prio)
                        except Exception as e:
                            tmpLog.error(str(e))
            tmpLog.debug(f"got {self.job_prio_boost_dict}")
            return self.job_prio_boost_dict
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return {}

    # set user secret
    def set_user_secret(self, owner, key, value):
        comment = " /* DBProxy.set_user_secret */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" < owner={owner} key={key} >"
        try:
            # get configs
            tmpLog = LogWrapper(_logger, methodName)
            # sql to check data
            sqlC = "SELECT data FROM ATLAS_PANDA.Secrets WHERE owner=:owner "
            # sql to insert dummy
            sqlI = "INSERT INTO ATLAS_PANDA.Secrets (owner, updated_at) " "VALUES(:owner,CURRENT_TIMESTAMP) "
            # sql to update data
            sqlU = "UPDATE ATLAS_PANDA.Secrets SET updated_at=CURRENT_TIMESTAMP,data=:data " "WHERE owner=:owner "
            # start transaction
            self.conn.begin()
            # check
            varMap = {}
            varMap[":owner"] = owner
            tmpS, tmpR = self.getClobObj(sqlC, varMap, use_commit=False)
            if not tmpR:
                # insert dummy for new entry
                self.cur.execute(sqlI + comment, varMap)
                data = {}
            else:
                data = json.loads(tmpR[0][0])
            # update
            if key is None:
                # delete all
                data = {}
            elif value is None:
                # delete key
                if key in data:
                    del data[key]
                else:
                    file_key = f"___file___:{key}"
                    if file_key in data:
                        del data[file_key]
            else:
                data[key] = value
            varMap = {}
            varMap[":owner"] = owner
            varMap[":data"] = json.dumps(data)
            self.cur.execute(sqlU + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError("Commit error")
            tmpLog.debug("done")
            return True, "OK"
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False, "database error"

    # get user secrets
    def get_user_secrets(self, owner, keys=None, get_json=False, use_commit=True):
        comment = " /* DBProxy.get_user_secrets */"
        methodName = comment.split(" ")[-2].split(".")[-1]
        methodName += f" < owner={owner} keys={keys} >"
        try:
            # get configs
            tmpLog = LogWrapper(_logger, methodName)
            # sql to get data
            sqlC = "SELECT data FROM ATLAS_PANDA.Secrets WHERE owner=:owner "
            # check
            varMap = {}
            varMap[":owner"] = owner
            tmpS, tmpR = self.getClobObj(sqlC, varMap, use_commit=use_commit)
            if not tmpR:
                data = {}
                if not get_json:
                    data = json.dumps({})
            else:
                data = tmpR[0][0]
                # return only interesting keys
                if keys:
                    keys = set(keys.split(","))
                    data = json.loads(data)
                    for k in list(data):
                        if k not in keys:
                            data.pop(k)
                    if not get_json:
                        data = json.dumps(data)
                else:
                    if get_json:
                        data = json.loads(data)
            tmpLog.debug(f"got data with length={len(data)}")
            return True, data
        except Exception:
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False, "database error"

    # get mb proxy
    def get_mb_proxy(self, channel):
        if self.mb_proxy_dict is None:
            self.mb_proxy_dict = get_mb_proxy_dict()
        if not self.mb_proxy_dict or channel not in self.mb_proxy_dict["out"]:
            return None
        return self.mb_proxy_dict["out"][channel]

    def configurator_write_sites(self, site_list):
        """
        Cache the CRIC site information in the PanDA database
        """
        comment = " /* DBProxy.configurator_write_sites */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")

        try:
            # begin transaction
            self.conn.begin()

            # get existing sites
            tmp_log.debug("getting existing sites")
            sql_get = "SELECT site_name FROM ATLAS_PANDA.site"
            self.cur.execute(sql_get + comment)
            results = self.cur.fetchall()
            site_name_list = list(map(lambda result: result[0], results))
            tmp_log.debug("finished getting existing sites")

            # see which sites need an update and which need to be inserted new
            var_map_insert = []
            var_map_update = []
            for site in site_list:
                if site["site_name"] in site_name_list:
                    var_map_update.append(convert_dict_to_bind_vars(site))
                else:
                    var_map_insert.append(convert_dict_to_bind_vars(site))

            tmp_log.debug("Updating sites")
            sql_update = "UPDATE ATLAS_PANDA.site set role=:role, tier_level=:tier_level WHERE site_name=:site_name"
            for shard in create_shards(var_map_update, 100):
                self.cur.executemany(sql_update + comment, shard)

            tmp_log.debug("Inserting sites")
            sql_insert = "INSERT INTO ATLAS_PANDA.site (site_name, role, tier_level) " "VALUES(:site_name, :role, :tier_level)"
            for shard in create_shards(var_map_insert, 100):
                self.cur.executemany(sql_insert + comment, shard)

            # commit
            if not self._commit():
                raise RuntimeError("Commit error")

            tmp_log.debug("Done")
            return 0, None

        except Exception:
            self._rollback()
            type, value, tb = sys.exc_info()
            tmp_log.error(f"{type} {value}")
            return -1, None

    def configurator_write_panda_sites(self, panda_site_list):
        comment = " /* DBProxy.configurator_write_panda_sites */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")

        try:
            # begin transaction
            self.conn.begin()

            # get existing panda sites
            tmp_log.debug("getting existing panda sites")
            sql_get = "SELECT panda_site_name FROM ATLAS_PANDA.panda_site"
            self.cur.execute(sql_get + comment)
            results = self.cur.fetchall()
            panda_site_name_list = list(map(lambda result: result[0], results))
            tmp_log.debug("finished getting existing panda sites")

            # see which sites need an update and which need to be inserted new
            var_map_insert = []
            var_map_update = []
            for panda_site in panda_site_list:
                if panda_site["panda_site_name"] in panda_site_name_list:
                    var_map_update.append(convert_dict_to_bind_vars(panda_site))
                else:
                    var_map_insert.append(convert_dict_to_bind_vars(panda_site))

            tmp_log.debug("Updating panda sites")
            sql_update = "UPDATE ATLAS_PANDA.panda_site set site_name=:site_name WHERE panda_site_name=:panda_site_name "
            for shard in create_shards(var_map_update, 100):
                self.cur.executemany(sql_update + comment, shard)

            tmp_log.debug("Inserting panda sites")
            sql_insert = "INSERT INTO ATLAS_PANDA.panda_site (panda_site_name, site_name) " "VALUES(:panda_site_name, :site_name)"
            for shard in create_shards(var_map_insert, 100):
                self.cur.executemany(sql_insert + comment, shard)

            # commit
            if not self._commit():
                raise RuntimeError("Commit error")

            tmp_log.debug("Done")
            return 0, None

        except Exception:
            self._rollback()
            type, value, tb = sys.exc_info()
            tmp_log.error(f"{type} {value}")
            tmp_log.error(format(traceback.format_exc()))
            return -1, None

    def configurator_write_ddm_endpoints(self, ddm_endpoint_list):
        comment = " /* DBProxy.configurator_write_ddm_endpoints */"
        method_name = comment.split(" ")[-2].split(".")[-1]
        tmp_log = LogWrapper(_logger, method_name)
        tmp_log.debug("start")

        try:
            # begin transaction
            self.conn.begin()

            # get existing ddm endpoints
            tmp_log.debug("getting existing ddm endpoints")
            sql_get = "SELECT ddm_endpoint_name FROM ATLAS_PANDA.ddm_endpoint"
            self.cur.execute(sql_get + comment)
            results = self.cur.fetchall()
            ddm_endpoint_name_list = list(map(lambda result: result[0], results))
            tmp_log.debug("finished getting existing ddm endpoints")

            # see which sites need an update and which need to be inserted new
            var_map_insert = []
            var_map_update = []
            for ddm_endpoint in ddm_endpoint_list:
                if ddm_endpoint["ddm_endpoint_name"] in ddm_endpoint_name_list:
                    var_map_update.append(convert_dict_to_bind_vars(ddm_endpoint))
                else:
                    var_map_insert.append(convert_dict_to_bind_vars(ddm_endpoint))

            tmp_log.debug("Updating ddm endpoints")
            sql_update = (
                "UPDATE ATLAS_PANDA.ddm_endpoint set "
                "site_name=:site_name, ddm_spacetoken_name=:ddm_spacetoken_name, type=:type, is_tape=:is_tape, "
                "blacklisted=:blacklisted, blacklisted_write=:blacklisted_write, blacklisted_read=:blacklisted_read, "
                "space_used=:space_used, space_free=:space_free, space_total=:space_total, space_expired=:space_expired, space_timestamp=:space_timestamp "
                "WHERE ddm_endpoint_name=:ddm_endpoint_name"
            )
            for shard in create_shards(var_map_update, 100):
                self.cur.executemany(sql_update + comment, shard)

            tmp_log.debug("Inserting ddm endpoints")
            sql_insert = (
                "INSERT INTO ATLAS_PANDA.ddm_endpoint (ddm_endpoint_name, site_na