import datetime
import itertools
import re
import string
import warnings
from fractions import Fraction
from pathlib import Path
from ... import output
from ... import terms
from .. import plain
from .. import LaTeX
from ...fields import Field, DiracFermionField
from ...models import STANDARD_MODEL


GREEK_TO_OUTPUT_NAME = {
	output.GREEK[i]: LaTeX.GREEK_LATEX[i].replace('\\', '')
	for i in range(len(output.GREEK))
}
# add “C” to capital letters to avoid name clashes in case-insensitive
# environments
GREEK_TO_OUTPUT_NAME = {k: 'C' + v if v[0].isupper() else v
	for k, v in GREEK_TO_OUTPUT_NAME.items()
}
OUTPUT_DIR = './SARAH_model_{}/'
SARAH_FILE_COMMENT = (
	'(* This file has been automatically generated by minimal-lagrangians *)'
)
MODEL_AUTHORS = 'minimal-lagrangians (automatically generated)'
FERMION_OFFSET = 5
SCALAR_OFFSET = 1
LAG_NO_HC = 'LagBSMNoHC'
LAG_HC = 'LagBSMHC'
VEV_NAME = 'v'
MIXING_MATRIX_O_PREFIX = 'ZZ'
MIXING_MATRIX_H_PREFIX = 'UU'
MIXING_MATRIX_NH_PREFIX_1 = 'VV'
MIXING_MATRIX_NH_PREFIX_2 = 'UU'
MIXING_MATRIX_MAX_LEN = 4
DIRAC_SPINOR_PREFIX = 'F'
PDG_OFFSET = 900
FEYNARTS_NR_OFFSET = 900
LES_HOUCHES_OFFSET = 10
SPHENO_PARAMETER_SUFFIX = 'Input'
SM_VEV_COMMENT = '(* Standard Model Higgs VEV *)'
BSM_VEV_COMMENT = (
	'(* BSM VEVs and splitting neutral scalars into real and imaginary parts *)'
)
BSM_MIXING_COMMENT = '(* mixing of new fields *)'
BSM_DIRAC_SPINORS_COMMENT = '(* new Dirac spinors *)'
PARAMETERS_LAGRANGIAN_COMMENT = '(* BSM parameters in the Lagrangian *)'
PARAMETERS_VEV_COMMENT = '(* BSM VEVs *)'
PARAMETERS_MIXING_COMMENT = '(* BSM mixing matrices *)'

FORBIDDEN = list('.,_ +-−*/^=⁺⁻\'"()!?\\')
REPLACEMENTS = [
	'', '', '', '', 'p', 'm', 'm', 't', 'd', '', 'eq', 'p', 'm', 'p', 'q', '',
	'', '', '', '',
]
assert len(FORBIDDEN) == len(REPLACEMENTS), \
	'{} ≠ {}'.format(len(FORBIDDEN), len(REPLACEMENTS))
REPLACEMENT_MAP = {
	FORBIDDEN[i]: REPLACEMENTS[i] for i in range(len(FORBIDDEN))
}


def sarah_string_escape(string):
	return string.replace('\\', r'\\').replace('"', r'\"')

# formats a model name so that it can be used as a “safe” file name for SARAH
def format_model_filename(name):
	name = re.sub(r'−(\d)', r'm\1', name)
	name = re.sub(
		'[' + ''.join('\\' + c if c in r'\-' else c for c in FORBIDDEN) + ']+',
		r'_', name
	)
	result = ''
	for letter in name:
		if letter in output.GREEK:
			result += GREEK_TO_OUTPUT_NAME[letter]
		else:
			result += letter
	return result.rstrip('_')


# TODO: use instance state and methods instead of class methods
class SARAHFormatter(output.Formatter):
	FORMAT_OPT = {
		'linebreak': ' \\\n\t',
		'param_symbol_mass': 'M',
		'param_symbol_yukawa': 'y',
		'param_symbol_coupling': 'λ',
		'join_str': ' - ',
		'join_str_hc': ' + ',
	}

	@classmethod
	def _format_product_default(self, product):
		return '.'.join(self.format_field(f) for f in product)

	@classmethod
	def _format_term_default(self, term):
		return '.'.join(self.format_product(p) for p in term)

	@classmethod
	def format_field(self, field):
		if isinstance(field, DiracFermionField):
			return self.format_dirac_spinor(field)
		if field.sarah_symbol is not None:
			symbol = field.sarah_symbol
		else:
			symbol = self.format_symbol(field.symbol)
		if field.is_adjoint:
			return 'conj[{}]'.format(symbol)
		else:
			return symbol

	@classmethod
	def format_field_components(self, field):
		symbols = [self.format_field(c) for c in field.components]
		if field.su2_triplet:
			# see
			# https://gitlab.in2p3.fr/goodsell/sarah/-/wikis/Particle_Content_non-SUSY
			# or other model files (e.g. SM+Triplet/Real) for the normalization
			result = '{%s/Sqrt[2], %s}, {%s, -%s/Sqrt[2]}' % (
				symbols[1], symbols[0], symbols[2], symbols[1]
			)
		else:
			result = ', '.join(symbols)
		if field.su2_multiplicity > 1:
			result = '{%s}' % result
		return result

	@classmethod
	def format_prefix(self, prefix):
		if prefix:
			return str(prefix) + ' '
		else:
			return ''

	@classmethod
	def format_subscript(self, subscript):
		return self._format_script(subscript)

	@classmethod
	def format_superscript(self, superscript):
		return self._format_script(superscript)

	# formats (field) symbols for use within SARAH (replacing characters which
	# cannot be used inside Mathematica variable names)
	@classmethod
	def format_symbol(self, symbol):
		# For SPheno/micrOMEGAs: OutputName does not seem to work for
		# particles, so just use safe names everywhere
		return self.format_symbol_output_name(symbol)

	# formats (field) symbols for use as OutputName (replacing characters so
	# that only “safe” ASCII characters are used)
	@classmethod
	def format_symbol_output_name(self, symbol, max_len=None):
		result = ''
		for letter in symbol:
			if letter in FORBIDDEN:
				replacement = REPLACEMENT_MAP[letter]
				if '\\' not in replacement:
					result += REPLACEMENT_MAP[letter]
			elif letter in output.SUPERSCRIPTS or letter in output.SUBSCRIPTS:
				result += output.SCRIPT_TO_DIGIT[letter]
			elif letter in output.GREEK:
				result += GREEK_TO_OUTPUT_NAME[letter]
			else:
				result += letter
		return result[:max_len]

	@classmethod
	def format_field_output_name(self, field, max_len=None):
		if isinstance(field, DiracFermionField):
			symbol = self.format_field(field)
		elif field.sarah_symbol:
			symbol = field.sarah_symbol
		else:
			symbol = field.symbol
		return self.format_symbol_output_name(symbol)

	@classmethod
	def format_parameter_output_name(self, param, max_len=None):
		return (
			self.format_symbol_output_name(param.name, max_len) +
			self.format_symbol_output_name(
				self.format_subscript(param.subscript)
			) +
			self.format_symbol_output_name(
				self.format_superscript(param.superscript)
			)
		)

	@classmethod
	def format_field_latex(self, field):
		return sarah_string_escape(LaTeX.Formatter.format_symbol(
			self.format_dirac_spinor(field, symbol=True)
			if isinstance(field, DiracFermionField) else
			field.symbol
		))

	@classmethod
	def format_vev(self, field):
		return self.format_parameter_symbol(
			field.sarah_vev_symbols[0] if field.sarah_vev_symbols
				else VEV_NAME + self.format_field(field)
		) if field.gets_vev else '0'

	@classmethod
	def format_mixing_matrix(self, mixed_field, idx, prefix_kind=None):
		prefixes = {
			'O': MIXING_MATRIX_O_PREFIX,
			'H': MIXING_MATRIX_H_PREFIX,
			'NH1': MIXING_MATRIX_NH_PREFIX_1,
			'NH2': MIXING_MATRIX_NH_PREFIX_2,
		}
		if prefix_kind is None:
			if mixed_field.scalar:
				prefix_kind = 'O'
			elif mixed_field.neutral:
				prefix_kind = 'H'
		symbol_full = self.format_parameter_symbol(
			mixed_field.sarah_mixing_symbols[1]
			if mixed_field.sarah_mixing_symbols else
			prefixes[prefix_kind] + self.format_field(mixed_field)
		)
		if len(symbol_full) <= MIXING_MATRIX_MAX_LEN:
			return symbol_full, False
		else:
			str_idx = str(idx)
			return (
				symbol_full[:MIXING_MATRIX_MAX_LEN - len(str_idx)] + str_idx,
				True
			)

	@classmethod
	def format_dirac_spinor(self, dirac_field, symbol=False):
		left, right = dirac_field
		assert (
			left.sarah_dirac_spinor_symbols == right.sarah_dirac_spinor_symbols
		)
		return (left.sarah_dirac_spinor_symbols
			if left.sarah_dirac_spinor_symbols else
			DIRAC_SPINOR_PREFIX	+ (
				left.symbol if symbol else self.format_field(left)
			)
		)

	@classmethod
	def format_slha_block_name(self, name):
		name = self.format_symbol_output_name(format_model_filename(name))
		return name.upper()

	@classmethod
	def check_duplicate_names(self, model, params_no_hc, params_hc, cpv=True):
		multiplets = list(model.fields)
		components = list({c.original_field for c in model.components})
		neutral_scalars = [
			c for c in components if c.scalar and c.neutral and
				# skip real scalars without VEV
				(c.gets_vev or not c.real)
		]
		neutral_scalar_parts = [
			p
			for s in neutral_scalars for p in s.complex_parts
			if isinstance(s.complex_parts.scalar, Field) and
				isinstance(s.complex_parts.pseudoscalar, Field)
		]
		mixed_fields_grouped = [
			mix_group
			for mix_group in model.ewsb_mixed_fields(cpv=cpv)
			if sum(g.generations for g in mix_group[0].gauge_es) > 1 and
				(len(mix_group) == 1 or
					sum(g.generations for g in mix_group[1].gauge_es) > 1)
		]
		mixed_fields = [
			mf for mix_group in mixed_fields_grouped for mf in mix_group
		]
		dirac_spinors = model.ewsb_dirac_spinors()
		# field symbols
		field_symbols = []
		field_symbols += [
			self.format_field(f.original_field)
			for f in
				multiplets + components + neutral_scalar_parts + mixed_fields
		]
		field_symbols += [self.format_dirac_spinor(df) for df in dirac_spinors]
		# field OutputNames
		field_output_names = [
			self.format_field_output_name(f)
			for f in components + mixed_fields + dirac_spinors
		]
		# parameter symbols
		param_symbols = []
		param_symbols += [
			self.format_parameter(param, prefix=False) + suffix
			for param in params_no_hc + params_hc
			for suffix in ['', SPHENO_PARAMETER_SUFFIX]
		]
		vev_symbols = [
			self.format_vev(s) for s in neutral_scalars if s.gets_vev
		]
		param_symbols += vev_symbols
		mixing_matrix_symbols = []
		mix_idx = 1
		for mix_group in mixed_fields_grouped:
			mix0 = mix_group[0]
			if len(mix_group) == 1:
				mixing_matrix_symbol, truncated = (
					self.format_mixing_matrix(mix0, mix_idx)
				)
				if truncated:
					mix_idx += 1
				mixing_matrix_symbols.append(mixing_matrix_symbol)
			elif len(mix_group) == 2:
				mix1 = mix_group[1]
				assert mix0.fermion and mix1.fermion
				assert (
					sum(f.generations for f in mix0.gauge_es) ==
					sum(f.generations for f in mix0.gauge_es)
				)
				if mix0.sarah_charge_preference == '-':
					mix0, mix1 = mix1, mix0
				mixing_matrix_symbol_0, truncated = (
					self.format_mixing_matrix(mix0, mix_idx, 'NH1')
				)
				if truncated: mix_idx += 1
				mixing_matrix_symbol_1, truncated = (
					self.format_mixing_matrix(mix1, mix_idx, 'NH2')
				)
				if truncated: mix_idx += 1
				mixing_matrix_symbols.append(mixing_matrix_symbol_0)
				mixing_matrix_symbols.append(mixing_matrix_symbol_1)
			else: assert False
		param_symbols += mixing_matrix_symbols
		# parameter OutputNames
		param_output_names = []
		param_output_names += [
			self.format_parameter_output_name(
				param, max_len=None if param.term_type.mass_term else 3
			) for param in params_no_hc + params_hc
		]
		param_output_names += [
			self.format_symbol_output_name(vev_symbol)
			for vev_symbol in vev_symbols
		]
		param_output_names += [
			self.format_symbol_output_name(mixing_matrix_symbol)
			for mixing_matrix_symbol in mixing_matrix_symbols
		]
		# check for duplicates and print potential warning
		if (
			len(field_symbols + param_symbols) >
			len(set(field_symbols + param_symbols))
		):
			warnings.warn((
					'Duplicate SARAH symbols for particles/parameters '
						'detected!\n'
					'Particle symbols: {}\n'
					'Parameter symbols: {}'
				).format(', '.join(field_symbols), ', '.join(param_symbols))
			)
		if (
			len(field_output_names + param_output_names) >
			len(set(field_output_names + param_output_names))
		):
			warnings.warn((
					'Duplicate SARAH OutputName values for '
						'particles/parameters detected!\n'
					'Particle OutputNames: {}\n'
					'Parameter OutputNames: {}'
				).format(
					', '.join(field_output_names),
					', '.join(param_output_names)
				)
			)

	@classmethod
	def sarah_model_fields_definition(self, model):
		def append_u1(result):
			if model.u1_len:
				result += ', ' + ', '.join(
					'%d' % (f.u1[i] if i < len(f.u1) else 0)
					for i in range(model.u1_len)
				)
			result += '};\n'
			return result
		bsm_fields = [f for f in model.fields if f.bsm]
		fermions = [f for f in bsm_fields if f.fermion]
		scalars = [f for f in bsm_fields if f.scalar]
		real_scalars = (
			[f for f in scalars if f.real and f.su2_singlet] +
			[c for f in scalars for c in f.components if c.real and
				not f.su2_singlet]
		)
		result = ''
		for i, f in enumerate(fermions):
			f = f.adjoint if f.is_adjoint else f
			# {name, gens, components, Y/2, SU(2), SU(3), global}
			result += 'FermionFields[[%d]] = {%s, %d, %s, %s, %d, %d, %d' % (
				FERMION_OFFSET + 1 + i, self.format_field(f), 1,
				self.format_field_components(f), Fraction(f.Y, 2),
				f.su2_multiplicity, 1, f.z2
			)
			result = append_u1(result)
		for i, f in enumerate(scalars):
			f = f.adjoint if f.is_adjoint else f
			# {name, gens, components, Y/2, SU(2), SU(3), global}
			result += 'ScalarFields[[%d]]  = {%s, %d, %s, %s, %d, %d, %d' % (
				SCALAR_OFFSET + 1 + i, self.format_field(f), 1,
				self.format_field_components(f), Fraction(f.Y, 2),
				f.su2_multiplicity, 1, f.z2
			)
			result = append_u1(result)
		if real_scalars:
			result += 'RealScalars = {%s};\n' % ', '.join(
				self.format_field(f) for f in real_scalars
			)
		return result

	@classmethod
	def sarah_model_ewsb(self, model, cpv=True):
		### DEFINITION[EWSB][VEVs] (VEV)
		neutral_scalars = [
			c for c in model.components if c.scalar and c.neutral and
				# skip real scalars without VEV (no split necessary)
				(c.gets_vev or not c.real)
		]
		VEVs_items = []
		for s in neutral_scalars:
			c_parts = s.complex_parts
			if not s.bsm:
				# insert comment before the only SM VEV
				VEVs_items.append('\t' + SM_VEV_COMMENT)
			VEVs_items.append((
				'\t{%s,\n'
				'\t\t{%s, 1/Sqrt[2]},\n'
				'\t\t{%s, \\[ImaginaryI]/Sqrt[2]},\n'
				'\t\t{%s, 1/Sqrt[2]}\n\t},'
			) % (
				self.format_field(s),
				self.format_vev(s),
				self.format_field(c_parts.pseudoscalar)
					if isinstance(c_parts.pseudoscalar, Field) else
					str(c_parts.pseudoscalar),
				self.format_field(c_parts.scalar)
					if isinstance(c_parts.scalar, Field) else
					str(c_parts.scalar),
			))
			if not s.bsm and len(neutral_scalars) > 1:
				# insert comment after the only SM VEV
				VEVs_items.append('\t' + BSM_VEV_COMMENT)
		VEVs_items[-1] = VEVs_items[-1].rstrip(',')
		VEVs = '\n'.join(VEVs_items)
		### DEFINITION[EWSB][MatterSector] (mixing)
		MatterSector_items = []
		comment_inserted = False
		mix_idx = 1
		for mix_group in model.ewsb_mixed_fields(cpv=cpv):
			mix0 = mix_group[0]
			# mixing should only be included if there is more than one field
			if len(mix0.gauge_es) == 1 and mix0.gauge_es[0].generations == 1:
				continue
			# insert comment when starting BSM-only mixings
			if all(f.bsm for f in mix0.gauge_es) and not comment_inserted:
				MatterSector_items.append('\t' + BSM_MIXING_COMMENT)
				comment_inserted = True
			# get first mixing matrix name
			# scalar/neutral (Majorana) fermion case
			if len(mix_group) == 1:
				mixing_matrix_symbol, truncated = (
					self.format_mixing_matrix(mix0, mix_idx)
				)
				if truncated:
					mix_idx += 1
				MatterSector_items.append(
					'\t{{%s}, {%s, %s}},' % (
						', '.join(self.format_field(c) for c in mix0.gauge_es),
						self.format_field(mix0),
						mixing_matrix_symbol,
					)
				)
			# charged (Dirac) fermion case
			elif len(mix_group) == 2:
				mix1 = mix_group[1]
				assert mix0.fermion and mix1.fermion
				assert (
					sum(f.generations for f in mix0.gauge_es) ==
					sum(f.generations for f in mix0.gauge_es)
				)
				if mix0.sarah_charge_preference == '-':
					mix0, mix1 = mix1, mix0
				mixing_matrix_symbol_0, truncated = (
					self.format_mixing_matrix(mix0, mix_idx, 'NH1')
				)
				if truncated: mix_idx += 1
				mixing_matrix_symbol_1, truncated = (
					self.format_mixing_matrix(mix1, mix_idx, 'NH2')
				)
				if truncated: mix_idx += 1
				MatterSector_items.append(
					'\t{{{%s}, {%s}}, {{%s, %s}, {%s, %s}}},' % (
						', '.join(self.format_field(c) for c in mix0.gauge_es),
						', '.join(self.format_field(c) for c in mix1.gauge_es),
						self.format_field(mix0),
						mixing_matrix_symbol_0,
						self.format_field(mix1),
						mixing_matrix_symbol_1,
					)
				)
			else: assert False
		MatterSector_items[-1] = MatterSector_items[-1].rstrip(',')
		MatterSector = '\n'.join(MatterSector_items)
		### DEFINITION[EWSB][DiracSpinors] (Dirac spinors)
		dirac_fields = model.ewsb_dirac_spinors()
		DiracSpinors_items = []
		comment_inserted = False
		for dirac_field in dirac_fields:
			left, right = dirac_field
			if left.sarah_charge_preference == '-':
				left, right = right.adjoint, left.adjoint
			# insert comment when starting BSM-only Dirac spinors
			if all(f.bsm for f in left.gauge_es) and not comment_inserted:
				DiracSpinors_items.append('\t' + BSM_DIRAC_SPINORS_COMMENT)
				comment_inserted = True
			DiracSpinors_items.append(
				'\t%s -> {%s, %s},' % (
					self.format_dirac_spinor(dirac_field),
					self.format_field(left),
					self.format_field(right) if isinstance(right, Field) else
					str(right),
			)
		)
		DiracSpinors_items[-1] = DiracSpinors_items[-1].rstrip(',')
		DiracSpinors = '\n'.join(DiracSpinors_items)
		return VEVs, MatterSector, DiracSpinors

	@classmethod
	def sarah_particles(self, model, cpv=True):
		bsm_fields = [
			f.adjoint if f.is_adjoint else f
			for f in model.fields if f.bsm
		]
		components = [
			comp for f in bsm_fields for comp in f.components
			if not comp.is_adjoint
		]
		mixed_fields = [
			f
			for mix_group in model.ewsb_mixed_fields(cpv=cpv)
			for f in mix_group
			if any(g.bsm for g in f.gauge_es)
		]
		### ParticleDefinitions[GaugeES]
		GaugeES = '\n'.join((
				'\t{%s, {Description -> "BSM field %s",\n'
				'\t\t\tOutputName -> "%s",\n'
				'\t\t\tElectricCharge -> %s,\n'
				'\t\t\tLaTeX -> "%s"\n\t\t}\n\t},'
			) % (
				self.format_field(component),
				sarah_string_escape(component.symbol),
				sarah_string_escape(self.format_field_output_name(component)),
				component.charge,
				self.format_field_latex(component),
			) for component in components
		)
		### ParticleDefinitions[EWSB]
		mixed_scalars = [f for f in mixed_fields if f.scalar]
		dirac_fields = [
			f for f in model.ewsb_dirac_spinors()
			if any(g.bsm for g in f.left.gauge_es) or (
				isinstance(f.right, Field) and
				any(g.bsm for g in f.right.gauge_es)
			)
		]
		template = (
			'\t{%s, {Description -> "BSM field %s (EWSB)",\n'
			'\t\t\tOutputName -> "%s",\n'
			'\t\t\tPDG -> {%s},\n'
			'\t\t\tFeynArtsNr -> %s,\n'
			'\t\t\tElectricCharge -> %s,\n'
			'\t\t\tLaTeX -> "%s"\n\t\t}\n\t},'
		)
		EWSB_items = []
		pdg = PDG_OFFSET
		feynarts_nr = FEYNARTS_NR_OFFSET
		for f in mixed_scalars + dirac_fields:
			dirac = isinstance(f, DiracFermionField)
			if dirac:
				left, right = f
				if left.sarah_charge_preference == '-':
					left, right = right.adjoint, left.adjoint
				num_fields = sum(g.generations for g in left.gauge_es)
				charge = left.charge
			else:
				num_fields = sum(g.generations for g in f.gauge_es)
				charge = f.charge
			EWSB_items.append(template % (
				self.format_field(f),
				sarah_string_escape(
					self.format_dirac_spinor(f, symbol=True) if dirac else
					f.symbol
				),
				sarah_string_escape(self.format_field_output_name(f)),
				', '.join(str(pdg + i) for i in range(num_fields)),
				feynarts_nr,
				charge,
				self.format_field_latex(f),
			))
			pdg += num_fields
			feynarts_nr += num_fields
		EWSB = '\n'.join(EWSB_items)
		# Goldstone bosons for Z and W⁺⁻
		H = STANDARD_MODEL.get_field('H')
		Ah = H.components[1].complex_parts.pseudoscalar
		Hp = H.components[0].components[0]
		goldstone = {'Z': Ah.sarah_symbol, 'W': Hp.sarah_symbol}
		for s in mixed_scalars:
			if Ah in s.gauge_es:
				goldstone['Z'] = self.format_field(s) + '[{1}]'
			if Hp in s.gauge_es:
				goldstone['W'] = self.format_field(s) + '[{1}]'
		### WeylFermionAndIndermediate
		gauge_es = [g for f in mixed_fields for g in f.gauge_es]
		weyl_spinors = [
			w.adjoint if w.is_adjoint else w for df in dirac_fields for w in df
		]
		WeylFermionAndIndermediate = '\n'.join(sorted({
			'\t{%s, {LaTeX -> "%s"}},' % (
				self.format_field(f),
				self.format_field_latex(f),
			) for f in bsm_fields + gauge_es + weyl_spinors
		}))
		return GaugeES, EWSB, goldstone, WeylFermionAndIndermediate

	@classmethod
	def sarah_parameters(self, model, params_no_hc, params_hc, cpv=True):
		no_hc = [True] * len(params_no_hc)
		hc = [False] * len(params_hc)
		ParameterDefinitions_items = []
		les_houches_name = self.format_slha_block_name(model.name)
		les_houches = LES_HOUCHES_OFFSET
		# Lagrangian parameters
		ParameterDefinitions_items.append('\t' + PARAMETERS_LAGRANGIAN_COMMENT)
		for real, param in zip(no_hc + hc, params_no_hc + params_hc):
			ParameterDefinitions_items.append((
					'\t{%s, {Description -> "BSM parameter %s",\n'
					'\t\t\tOutputName -> "%s",\n'
					'%s'
					'\t\t\tLesHouches -> {"%s", %s},\n'
					'\t\t\tLaTeX -> "%s"\n\t\t}\n\t},'
				) % (
					self.format_parameter(param, prefix=False),
					sarah_string_escape(plain.Formatter.format_parameter(
						param, prefix=False
					)),
					self.format_parameter_output_name(
						param, max_len=None if param.term_type.mass_term else 3
					),
					'\t\t\tReal -> True,\n' if real else '',
					les_houches_name,
					les_houches,
					sarah_string_escape(
						LaTeX.Formatter.format_parameter(param, prefix=False)
					),
				)
			)
			les_houches += 1
		# VEVs
		vev_scalars = [s for s in model.components if s.gets_vev and s.bsm]
		if vev_scalars:
			ParameterDefinitions_items.append('\n')
			ParameterDefinitions_items.append('\t' + PARAMETERS_VEV_COMMENT)
			for s in vev_scalars:
				format_vev = self.format_vev(s)
				ParameterDefinitions_items.append((
						'\t{%s, {Description -> "BSM VEV %s",\n'
						'\t\t\tOutputName -> "%s",\n'
						'\t\t\tReal -> True,\n'
						'\t\t\tLesHouches -> {"%s", %s},\n'
						'\t\t\tLaTeX -> "%s"\n\t\t}\n\t},'
					) % (
						format_vev,
						sarah_string_escape(format_vev),
						self.format_symbol_output_name(format_vev),
						les_houches_name,
						les_houches,
						sarah_string_escape(
							LaTeX.Formatter.format_parameter_symbol(
								format_vev
						)),
					)
				)
				les_houches += 1
		# mixing matrices
		mixed_fields = [
			mix_group
			for mix_group in model.ewsb_mixed_fields(cpv=cpv)
			# SM mixing matrices are already present in template parameters.m
			if any(g.bsm for f in mix_group for g in f.gauge_es) and
			# mixing should only be included if there is more than one field
			sum(f.generations for f in mix_group[0].gauge_es) > 1 or
			# include neutrinos
				STANDARD_MODEL.get_field('L').components[0].components[0] in
					mix_group[0].gauge_es
		]
		if mixed_fields:
			ParameterDefinitions_items.append('\n')
			ParameterDefinitions_items.append('\t' + PARAMETERS_MIXING_COMMENT)
			mix_idx = 1
			for mix_group in mixed_fields:
				if mix_group[0].sarah_charge_preference == '-':
					mg = reversed(mix_group)
				else:
					mg = mix_group
				for i, f_mix in enumerate(mg, start=1):
					# format entry
					format_mixing_matrix, truncated = (
						self.format_mixing_matrix(
							f_mix, mix_idx,
							'NH' + str(i) if len(mix_group) > 1 else None
						)
					)
					if truncated:
						mix_idx += 1
					ParameterDefinitions_items.append((
							'\t{%s, {Description -> "BSM mixing matrix %s",\n'
							'\t\t\tOutputName -> "%s",\n'
							'\t\t\tLesHouches -> "%s",\n'
							'\t\t\tLaTeX -> "%s"\n\t\t}\n\t},'
						) % (
							format_mixing_matrix,
							sarah_string_escape(format_mixing_matrix),
							self.format_symbol_output_name(
								format_mixing_matrix
							),
							self.format_slha_block_name(format_mixing_matrix),
							sarah_string_escape(
								LaTeX.Formatter.format_parameter_symbol(
									format_mixing_matrix
							)),
						)
					)
		# combine
		ParameterDefinitions = '\n'.join(ParameterDefinitions_items)
		return ParameterDefinitions

	@classmethod
	def sarah_spheno(self, model, params_no_hc, params_hc, cpv=True):
		### MINPAR
		vevs = [
			output.Parameter('', self.format_vev(s), '', '', None)
			for s in model.components if s.bsm and s.gets_vev
		]
		MINPAR = ',\n'.join(
			'\t{%s, %s}' % (
				les_houches,
				self.format_parameter(param, prefix=False) +
					SPHENO_PARAMETER_SUFFIX,
			)
			for les_houches, param in enumerate(
				params_no_hc + params_hc + vevs, start=LES_HOUCHES_OFFSET
			)
		)
		### BoundaryLowScaleInput
		BoundaryLowScaleInput = ',\n'.join(
			'\t{%s, %s}' % (
				self.format_parameter(param, prefix=False),
				self.format_parameter(param, prefix=False) +
					SPHENO_PARAMETER_SUFFIX,
			)
			for param in params_no_hc + params_hc + vevs
		)
		### ParametersToSolveTadpoles
		# TODO: Simply listing all mass parameters is not correct? See SARAH
		#       model THDM-CPV
		ParametersToSolveTadpoles = ''.join(
			', ' + self.format_parameter(param, prefix=False)
			for param in params_no_hc
			if param.term_type.mass_term and
				isinstance(param.subscript, Field) and
				any(c.gets_vev for c in param.subscript.components)
		)
		### ListDecayParticles
		H = STANDARD_MODEL.get_field('H')
		Ah = H.components[1].complex_parts.pseudoscalar
		L = STANDARD_MODEL.get_field('L')
		vL = L.components[0].components[0]
		mixed_scalars = [
			f
			for mix_group in model.ewsb_mixed_fields(cpv=cpv)
			for f in mix_group
			# exclude non-mixing Ah
			if f.scalar and {Ah} != set(f.gauge_es)
		]
		dirac_fields = [
			f for f in model.ewsb_dirac_spinors()
			# exclude neutrinos
			if vL not in f.left.gauge_es
		]
		fields = sorted(
			mixed_scalars + dirac_fields,
			key=lambda f: (
				bool(f.left.bsm) if isinstance(f, DiracFermionField) else
					bool(f.bsm),
				not isinstance(f, DiracFermionField),
				self.format_field(f),
			)
		)
		ListDecayParticles = ', '.join(self.format_field(f) for f in fields)
		return (
			MINPAR, BoundaryLowScaleInput, ParametersToSolveTadpoles,
			ListDecayParticles
		)

	# Writes the SARAH model files <model>.m, particles.m, parameters.m, and
	# SPheno.m
	@classmethod
	def write_sarah_files(
		self, model, params_no_hc, params_hc, lag_no_hc, lag_hc, cpv=True
	):
		self.check_duplicate_names(model, params_no_hc, params_hc, cpv=cpv)
		# get paths
		pkg_path = Path(__path__[0]) / 'data'
		model_filename = format_model_filename(model.name)
		output_path = Path(OUTPUT_DIR.format(model_filename))
		# format output
		VEVs, MatterSector, DiracSpinors = self.sarah_model_ewsb(
			model, cpv=cpv
		)
		GaugeES, EWSB, goldstone, WeylFermionAndIndermediate = (
			self.sarah_particles(model, cpv=cpv)
		)
		ParameterDefinitions = self.sarah_parameters(
			model, params_no_hc, params_hc, cpv=cpv
		)
		MINPAR, BoundaryLowScaleInput, ParametersToSolveTadpoles, \
			ListDecayParticles = self.sarah_spheno(
				model, params_no_hc, params_hc, cpv=cpv
			)
		# substitute values into template
		model_lagrangian_input_list = ''
		if lag_no_hc:
			model_lagrangian_input_list += (
				'\t{%s, {AddHC -> False}}' % LAG_NO_HC
			)
		if lag_hc:
			if model_lagrangian_input_list:
				model_lagrangian_input_list += ',\n'
			model_lagrangian_input_list += (
				'\t{%s,   {AddHC -> True }}' % LAG_HC
			)
		template_subs = {
			'sarah_file_comment': SARAH_FILE_COMMENT,
			'model_name': sarah_string_escape(
				# Unicode or reserved shell characters in Model`Name cause
				# issues with SPheno/micrOMEGAs compilation
				format_model_filename(model.name)
			),
			'model_name_latex': sarah_string_escape(
				LaTeX.Formatter.format_symbol(model.name)
			),
			'model_authors': MODEL_AUTHORS,
			'model_date': datetime.date.today().isoformat(),
			'model_u1_definition': '\n'.join(
				'Global[[%d]] = {U[1], U1%d};' % (i, i)
				# offset by 2 to account for 1-based indexing and the entry
				# Global[[1]] = {Z[2], Z2};
				for i in range(2, model.u1_len + 2)
			),
			'model_u1_sm': (', ' + ', '.join(['0'] * model.u1_len)
				if model.u1_len else ''),
			'model_fields_definition':
				 self.sarah_model_fields_definition(model),
			'model_lagrangian_input_comma': ',' if lag_no_hc or lag_hc else '',
			'model_lagrangian_input_list': model_lagrangian_input_list,
			'model_lagrangian_hc':
				'{} ={};'.format(LAG_HC, lag_hc) if lag_hc else '',
			'model_lagrangian_no_hc':
				'{} ={};'.format(LAG_NO_HC, lag_no_hc) if lag_no_hc else '',
			'model_vev': VEVs,
			'model_mixing': MatterSector,
			'model_dirac_spinors': DiracSpinors,
			'parameters_definition': ParameterDefinitions,
			'particles_GaugeES': GaugeES,
			'particles_EWSB': EWSB,
			'particles_goldstone_W': goldstone['W'],
			'particles_goldstone_Z': goldstone['Z'],
			'particles_WeylFermionAndIndermediate': WeylFermionAndIndermediate,
			'spheno_minpar': MINPAR,
			'spheno_boundary_comma': ',' if BoundaryLowScaleInput else '',
			'spheno_boundary': BoundaryLowScaleInput,
			'spheno_parameters_tadpoles': ParametersToSolveTadpoles,
			'spheno_decay_particles': ListDecayParticles,
		}
		# create output directory
		try:
			output_path.mkdir()
		except FileExistsError:
			pass
		# substitute values in template files and write output files
		template_names = []
		for template_path in (p for p in pkg_path.iterdir() if p.is_file()):
			with template_path.open() as template_f:
				template = template_f.read()
			result = string.Template(template).substitute(template_subs)
			if template_path.name == 'model.m':
				output_name = model_filename + '.m'
			else:
				output_name = template_path.name
				template_names.append(output_name)
			with (output_path / output_name).open('w') as output_f:
				output_f.write(result)
		return [model_filename + '.m'] + sorted(template_names, reverse=True)

	@classmethod
	def _format_lagrangian(self, model, lagrangian, **kwargs):
		# split Lagrangians into two parts: with terms with (+ H.c.) and without
		lag_no_hc = terms.Lagrangian(
			term for term in lagrangian
			if term not in lagrangian.add_hc and
				term.adjoint not in lagrangian.add_hc
		)
		lag_hc = terms.Lagrangian(
			term for term in lagrangian
			if term in lagrangian.add_hc
		)
		# build the Lagrangians in SARAH format
		# use same parameter numbering as if lagrangian were formatted as a
		# whole
		lagrangian_list = [
			term for term in lagrangian
			# remove “+ H.c.” terms because these do not have their separate
			# parameters
			if term.adjoint not in lagrangian.add_hc
		]
		_, params_all = super()._format_lagrangian(model, lagrangian)
		indices_no_hc = [
			param.subscript for idx, param in enumerate(params_all)
			if lagrangian_list[idx] in lag_no_hc
		]
		indices_hc = [
			param.subscript for idx, param in enumerate(params_all)
			if lagrangian_list[idx] in lag_hc or
				lagrangian_list[idx].adjoint in lag_hc
		]
		assert len(indices_no_hc) == len(lag_no_hc)
		assert len(indices_hc) == len(lag_hc)
		assert len(indices_no_hc) + len(indices_hc) == len(params_all), \
			'{} + {} != {}'.format(
				len(indices_no_hc), len(indices_hc), len(params_all)
			)
		# format using order of parameter indices obtained above
		res_no_hc, params_no_hc = super()._format_lagrangian(
			model, lag_no_hc, param_idx_list=indices_no_hc
		)
		res_hc, params_hc = super()._format_lagrangian(
			model, lag_hc, param_idx_list=indices_hc
		)
		result = ''
		if res_no_hc:
			result += '{} ={};'.format(LAG_NO_HC, res_no_hc)
		if res_hc:
			if result: result += '\n'
			result += '{} ={};'.format(LAG_HC, res_hc)
		# write the SARAH model files
		cpv = not kwargs.get('sarah_no_scalar_cpv', False)
		written_file_names = self.write_sarah_files(
			model, params_no_hc, params_hc, res_no_hc, res_hc, cpv=cpv
		)
		# also return the Lagrangian from the model file as a string
		model_name = format_model_filename(model.name)
		result += (
			'\n\nThe SARAH model files ({})\n' +
			'have been written to the directory {}'
		).format(', '.join(written_file_names), OUTPUT_DIR.format(model_name))
		return result, params_no_hc + params_hc

Formatter = SARAHFormatter

