"""SPICE datetime module.

The full metakernel specifications are available on NAIF website:

https://naif.jpl.nasa.gov/pub/naif/toolkit_docs/C/req/time.html

"""

import numpy as np

import spiceypy as sp


def datetime(string, *others):
    """Parse datetime with SPICE convention.

    Parameters
    ----------
    string: str
        Input datetime string. Many format are supported, see the NAIF docs:

        https://naif.jpl.nasa.gov/pub/naif/toolkit_docs/C/req/time.html#Input%20String%20Conversion

        If the string starts with a `@`, this character will be discarded
        before being parsed.

    *others: str, optional
        Addition input string(s) to parse.

    Returns
    -------
    numpy.datetime64 or [numpy.datetime64, …]
        Parsed string(s) as numpy datetime64 object(s).

    Raises
    ------
    TypeError
        If the input time a not a string.
    ValueError
        If the provided string is not recognized by SPICE.

    Note
    ----
    This routine is the simple parser and does not require
    any kernel to be loaded.

    Warning
    -------
    This routine does not implement time convertion from UTC
    to TDB or TDT. You need to load at least a leapsecond kernel
    to perform these conversions.

    """
    if others:
        return [datetime(s) for s in [string, *others]]

    if isinstance(string, (tuple, list, np.ndarray)):
        return [datetime(s) for s in string]

    if not isinstance(string, str):
        raise TypeError(f'Input time must be a string not `{type(string)}`')

    if string.startswith('@'):
        return datetime(string[1:])

    # Extract date and time parts and reformat the string
    jdn, h, m, s, ms = jdn_hms(string)
    yy, mm, dd = ymd(jdn)

    time = f'{yy:02d}-{mm:02d}-{dd:02d}T{h:02d}:{m:02d}:{s:02d}.{ms:03d}'

    return np.datetime64(clean_time(time))


def jdn_hms(string):
    """Extract the julian day and the time from a string.

    Parameters
    ----------
    string: str
        Input string to parse.

    Returns
    -------
    int, int, int, int, int
        Julian date number, hours, minutes, seconds and milliseconds values.

    Raises
    ------
    ValueError
        If the string can not be parsed as a SPICE time.

    Note
    ----
    The precision output time is rounded at 1 millisecond.
    The Julian Day Number is always defined with respect to the Gregorian calendar.

    See Also
    --------
    :py:func:`ymd`

    """
    # Parse the string with SPICE to get the number of seconds after 2000.
    sp2000, err = sp.tparse(string)

    if err:
        raise ValueError(err)

    # Extract the time part
    s = int(np.floor(sp2000))
    msec = int(np.round((sp2000 - s) * 1_000))
    m, second = s // 60, s % 60
    h, minute = m // 60, m % 60
    hour = (h + 12) % 24

    # Extract the Days past 2000
    dp2000 = (s - (3_600 * hour + 60 * minute + second)) / sp.spd() + .5
    jdn = int(sp.j2000() + dp2000)

    return jdn, hour, minute, second, msec


def ymd(jdn):
    """SPICE convertion from Julian Day to the Gregorian Calendar.

    Parameters
    ----------
    jdn: int
        Julian Date Number (no decimal value).

    Returns
    -------
    int, int, int
        Parsed year, month and day in the Gregorian Calendar.

    Warning
    -------
    The dates before October 15th, 1582 are still represented in the Gregorian
    calendar and not in the Julian calendar. This is not strictly correct but
    it does correspond to the default behavior of SPICE and Numpy:

    >>> spiceypy.tparse('1582-10-14')
    >>> numpy.datetime64('1582-10-14')

    Don't throw errors even if theses date don't exists,
    although the day before 1582-10-15 should be 1582-10-04.

    >>> numpy.datetime64('1582-10-15') - numpy.datetime64('1582-10-04') == \
        numpy.timedelta64(1,'D')
    False

    """
    alpha = np.floor((jdn - 1_867_216.25) / 36_524.25)
    s = jdn + 1 + alpha - np.floor(alpha / 4)

    b = s + 1_524
    c = np.floor((b - 122.1) / 365.25)
    d = np.floor(365.25 * c)
    e = np.floor((b - d) / 30.6001)

    day = b - d - np.floor(30.6001 * e)
    month = e - 1 if e < 14 else e - 13
    year = c - 4_716 if month > 2 else c - 4_715

    return int(year), int(month), int(day)


def clean_time(time):
    """Clean time string (discard null leading values)."""
    if '.' in time:
        time = time.rstrip('0')
        time = time.rstrip('.')

    for _ in range(2):  # :MM:SS
        if time.endswith(':00'):
            time = time[:-3]

    if time.endswith('T00'):  # THH
        time = time[:-3]

    for _ in range(2):  # -MM-DD
        if time.endswith('-01'):
            time = time[:-3]

    return time
