"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrCreateCluster = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
const cluster_utils_1 = require("./private/cluster-utils");
/**
 * A Step Functions Task to create an EMR Cluster.
 *
 * The ClusterConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the ClusterId.
 *
 * @stability stable
 */
class EmrCreateCluster extends sfn.TaskStateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id, props);
        this.props = props;
        this.visibleToAllUsers = (_b = this.props.visibleToAllUsers) !== null && _b !== void 0 ? _b : true;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, EmrCreateCluster.SUPPORTED_INTEGRATION_PATTERNS);
        this._autoScalingRole = this.props.autoScalingRole;
        // If the Roles are undefined then they weren't provided, so create them
        this._serviceRole = (_c = this.props.serviceRole) !== null && _c !== void 0 ? _c : this.createServiceRole();
        this._clusterRole = (_d = this.props.clusterRole) !== null && _d !== void 0 ? _d : this.createClusterRole();
        // AutoScaling roles are not valid with InstanceFleet clusters.
        // Attempt to create only if .instances.instanceFleets is undefined or empty
        if (this.props.instances.instanceFleets === undefined || this.props.instances.instanceFleets.length === 0) {
            this._autoScalingRole = this._autoScalingRole || this.createAutoScalingRole();
            // If InstanceFleets are used and an AutoScaling Role is specified, throw an error
        }
        else if (this._autoScalingRole !== undefined) {
            throw new Error('Auto Scaling roles can not be specified with instance fleets.');
        }
        this.taskPolicies = this.createPolicyStatements(this._serviceRole, this._clusterRole, this._autoScalingRole);
    }
    /**
     * The service role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     *
     * @stability stable
     */
    get serviceRole() {
        if (this._serviceRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._serviceRole;
    }
    /**
     * The instance role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     *
     * @stability stable
     */
    get clusterRole() {
        if (this._clusterRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._clusterRole;
    }
    /**
     * The autoscaling role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     *
     * @stability stable
     */
    get autoScalingRole() {
        if (this._autoScalingRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._autoScalingRole;
    }
    /**
     * @internal
     */
    _renderTask() {
        var _b, _c, _d;
        return {
            Resource: task_utils_1.integrationResourceArn('elasticmapreduce', 'createCluster', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                Instances: cluster_utils_1.InstancesConfigPropertyToJson(this.props.instances),
                JobFlowRole: cdk.stringToCloudFormation(this._clusterRole.roleName),
                Name: cdk.stringToCloudFormation(this.props.name),
                ServiceRole: cdk.stringToCloudFormation(this._serviceRole.roleName),
                AdditionalInfo: cdk.stringToCloudFormation(this.props.additionalInfo),
                Applications: cdk.listMapper(cluster_utils_1.ApplicationConfigPropertyToJson)(this.props.applications),
                AutoScalingRole: cdk.stringToCloudFormation((_b = this._autoScalingRole) === null || _b === void 0 ? void 0 : _b.roleName),
                BootstrapActions: cdk.listMapper(cluster_utils_1.BootstrapActionConfigToJson)(this.props.bootstrapActions),
                Configurations: cdk.listMapper(cluster_utils_1.ConfigurationPropertyToJson)(this.props.configurations),
                CustomAmiId: cdk.stringToCloudFormation(this.props.customAmiId),
                EbsRootVolumeSize: (_c = this.props.ebsRootVolumeSize) === null || _c === void 0 ? void 0 : _c.toGibibytes(),
                KerberosAttributes: this.props.kerberosAttributes ? cluster_utils_1.KerberosAttributesPropertyToJson(this.props.kerberosAttributes) : undefined,
                LogUri: cdk.stringToCloudFormation(this.props.logUri),
                ReleaseLabel: cdk.stringToCloudFormation(this.props.releaseLabel),
                ScaleDownBehavior: cdk.stringToCloudFormation((_d = this.props.scaleDownBehavior) === null || _d === void 0 ? void 0 : _d.valueOf()),
                SecurityConfiguration: cdk.stringToCloudFormation(this.props.securityConfiguration),
                ...(this.props.tags ? this.renderTags(this.props.tags) : undefined),
                VisibleToAllUsers: cdk.booleanToCloudFormation(this.visibleToAllUsers),
            }),
        };
    }
    renderTags(tags) {
        return tags ? { Tags: Object.keys(tags).map((key) => ({ Key: key, Value: tags[key] })) } : {};
    }
    /**
     * This generates the PolicyStatements required by the Task to call CreateCluster.
     */
    createPolicyStatements(serviceRole, clusterRole, autoScalingRole) {
        const stack = cdk.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['elasticmapreduce:RunJobFlow', 'elasticmapreduce:DescribeCluster', 'elasticmapreduce:TerminateJobFlows'],
                resources: ['*'],
            }),
        ];
        // Allow the StateMachine to PassRole to Cluster roles
        policyStatements.push(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [serviceRole.roleArn, clusterRole.roleArn],
        }));
        if (autoScalingRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [autoScalingRole.roleArn],
            }));
        }
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRRunJobFlowRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    /**
     * Generate the Role used by the EMR Service
     */
    createServiceRole() {
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceRole')],
        });
    }
    /**
     * Generate the Role and Instance Profile used by the EC2 instances
     *
     * Data access permissions will need to be updated by the user
     */
    createClusterRole() {
        const role = new iam.Role(this, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [role.roleName],
            instanceProfileName: role.roleName,
        });
        return role;
    }
    /**
     * Generate the Role used to AutoScale the Cluster
     */
    createAutoScalingRole() {
        var _b;
        const role = new iam.Role(this, 'AutoScalingRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceforAutoScalingRole')],
        });
        (_b = role.assumeRolePolicy) === null || _b === void 0 ? void 0 : _b.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            principals: [new iam.ServicePrincipal('application-autoscaling.amazonaws.com')],
            actions: ['sts:AssumeRole'],
        }));
        return role;
    }
}
exports.EmrCreateCluster = EmrCreateCluster;
_a = JSII_RTTI_SYMBOL_1;
EmrCreateCluster[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EmrCreateCluster", version: "1.115.0" };
EmrCreateCluster.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
/**
 * A Step Functions Task to create an EMR Cluster.
 *
 * The ClusterConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the ClusterId.
 *
 * @stability stable
 */
(function (EmrCreateCluster) {
    /**
     * The Cluster ScaleDownBehavior specifies the way that individual Amazon EC2 instances terminate when an automatic scale-in activity occurs or an instance group is resized.
     *
     * @see https://docs.aws.amazon.com/emr/latest/APIReference/API_RunJobFlow.html#EMR-RunJobFlow-request-ScaleDownBehavior
     * @stability stable
     */
    let EmrClusterScaleDownBehavior;
    (function (EmrClusterScaleDownBehavior) {
        EmrClusterScaleDownBehavior["TERMINATE_AT_INSTANCE_HOUR"] = "TERMINATE_AT_INSTANCE_HOUR";
        EmrClusterScaleDownBehavior["TERMINATE_AT_TASK_COMPLETION"] = "TERMINATE_AT_TASK_COMPLETION";
    })(EmrClusterScaleDownBehavior = EmrCreateCluster.EmrClusterScaleDownBehavior || (EmrCreateCluster.EmrClusterScaleDownBehavior = {}));
    /**
     * Instance Role Types.
     *
     * @stability stable
     */
    let InstanceRoleType;
    (function (InstanceRoleType) {
        InstanceRoleType["MASTER"] = "MASTER";
        InstanceRoleType["CORE"] = "CORE";
        InstanceRoleType["TASK"] = "TASK";
    })(InstanceRoleType = EmrCreateCluster.InstanceRoleType || (EmrCreateCluster.InstanceRoleType = {}));
    /**
     * EBS Volume Types.
     *
     * @stability stable
     */
    let EbsBlockDeviceVolumeType;
    (function (EbsBlockDeviceVolumeType) {
        EbsBlockDeviceVolumeType["GP2"] = "gp2";
        EbsBlockDeviceVolumeType["IO1"] = "io1";
        EbsBlockDeviceVolumeType["STANDARD"] = "standard";
    })(EbsBlockDeviceVolumeType = EmrCreateCluster.EbsBlockDeviceVolumeType || (EmrCreateCluster.EbsBlockDeviceVolumeType = {}));
    /**
     * Spot Timeout Actions.
     *
     * @stability stable
     */
    let SpotTimeoutAction;
    (function (SpotTimeoutAction) {
        SpotTimeoutAction["SWITCH_TO_ON_DEMAND"] = "SWITCH_TO_ON_DEMAND";
        SpotTimeoutAction["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    })(SpotTimeoutAction = EmrCreateCluster.SpotTimeoutAction || (EmrCreateCluster.SpotTimeoutAction = {}));
    /**
     * CloudWatch Alarm Comparison Operators.
     *
     * @stability stable
     */
    let CloudWatchAlarmComparisonOperator;
    (function (CloudWatchAlarmComparisonOperator) {
        CloudWatchAlarmComparisonOperator["GREATER_THAN_OR_EQUAL"] = "GREATER_THAN_OR_EQUAL";
        CloudWatchAlarmComparisonOperator["GREATER_THAN"] = "GREATER_THAN";
        CloudWatchAlarmComparisonOperator["LESS_THAN"] = "LESS_THAN";
        CloudWatchAlarmComparisonOperator["LESS_THAN_OR_EQUAL"] = "LESS_THAN_OR_EQUAL";
    })(CloudWatchAlarmComparisonOperator = EmrCreateCluster.CloudWatchAlarmComparisonOperator || (EmrCreateCluster.CloudWatchAlarmComparisonOperator = {}));
    /**
     * CloudWatch Alarm Statistics.
     *
     * @stability stable
     */
    let CloudWatchAlarmStatistic;
    (function (CloudWatchAlarmStatistic) {
        CloudWatchAlarmStatistic["SAMPLE_COUNT"] = "SAMPLE_COUNT";
        CloudWatchAlarmStatistic["AVERAGE"] = "AVERAGE";
        CloudWatchAlarmStatistic["SUM"] = "SUM";
        CloudWatchAlarmStatistic["MINIMUM"] = "MINIMUM";
        CloudWatchAlarmStatistic["MAXIMUM"] = "MAXIMUM";
    })(CloudWatchAlarmStatistic = EmrCreateCluster.CloudWatchAlarmStatistic || (EmrCreateCluster.CloudWatchAlarmStatistic = {}));
    /**
     * CloudWatch Alarm Units.
     *
     * @stability stable
     */
    let CloudWatchAlarmUnit;
    (function (CloudWatchAlarmUnit) {
        CloudWatchAlarmUnit["NONE"] = "NONE";
        CloudWatchAlarmUnit["SECONDS"] = "SECONDS";
        CloudWatchAlarmUnit["MICRO_SECONDS"] = "MICRO_SECONDS";
        CloudWatchAlarmUnit["MILLI_SECONDS"] = "MILLI_SECONDS";
        CloudWatchAlarmUnit["BYTES"] = "BYTES";
        CloudWatchAlarmUnit["KILO_BYTES"] = "KILO_BYTES";
        CloudWatchAlarmUnit["MEGA_BYTES"] = "MEGA_BYTES";
        CloudWatchAlarmUnit["GIGA_BYTES"] = "GIGA_BYTES";
        CloudWatchAlarmUnit["TERA_BYTES"] = "TERA_BYTES";
        CloudWatchAlarmUnit["BITS"] = "BITS";
        CloudWatchAlarmUnit["KILO_BITS"] = "KILO_BITS";
        CloudWatchAlarmUnit["MEGA_BITS"] = "MEGA_BITS";
        CloudWatchAlarmUnit["GIGA_BITS"] = "GIGA_BITS";
        CloudWatchAlarmUnit["TERA_BITS"] = "TERA_BITS";
        CloudWatchAlarmUnit["PERCENT"] = "PERCENT";
        CloudWatchAlarmUnit["COUNT"] = "COUNT";
        CloudWatchAlarmUnit["BYTES_PER_SECOND"] = "BYTES_PER_SECOND";
        CloudWatchAlarmUnit["KILO_BYTES_PER_SECOND"] = "KILO_BYTES_PER_SECOND";
        CloudWatchAlarmUnit["MEGA_BYTES_PER_SECOND"] = "MEGA_BYTES_PER_SECOND";
        CloudWatchAlarmUnit["GIGA_BYTES_PER_SECOND"] = "GIGA_BYTES_PER_SECOND";
        CloudWatchAlarmUnit["TERA_BYTES_PER_SECOND"] = "TERA_BYTES_PER_SECOND";
        CloudWatchAlarmUnit["BITS_PER_SECOND"] = "BITS_PER_SECOND";
        CloudWatchAlarmUnit["KILO_BITS_PER_SECOND"] = "KILO_BITS_PER_SECOND";
        CloudWatchAlarmUnit["MEGA_BITS_PER_SECOND"] = "MEGA_BITS_PER_SECOND";
        CloudWatchAlarmUnit["GIGA_BITS_PER_SECOND"] = "GIGA_BITS_PER_SECOND";
        CloudWatchAlarmUnit["TERA_BITS_PER_SECOND"] = "TERA_BITS_PER_SECOND";
        CloudWatchAlarmUnit["COUNT_PER_SECOND"] = "COUNT_PER_SECOND";
    })(CloudWatchAlarmUnit = EmrCreateCluster.CloudWatchAlarmUnit || (EmrCreateCluster.CloudWatchAlarmUnit = {}));
    /**
     * EC2 Instance Market.
     *
     * @stability stable
     */
    let InstanceMarket;
    (function (InstanceMarket) {
        InstanceMarket["ON_DEMAND"] = "ON_DEMAND";
        InstanceMarket["SPOT"] = "SPOT";
    })(InstanceMarket = EmrCreateCluster.InstanceMarket || (EmrCreateCluster.InstanceMarket = {}));
    /**
     * AutoScaling Adjustment Type.
     *
     * @stability stable
     */
    let ScalingAdjustmentType;
    (function (ScalingAdjustmentType) {
        ScalingAdjustmentType["CHANGE_IN_CAPACITY"] = "CHANGE_IN_CAPACITY";
        ScalingAdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PERCENT_CHANGE_IN_CAPACITY";
        ScalingAdjustmentType["EXACT_CAPACITY"] = "EXACT_CAPACITY";
    })(ScalingAdjustmentType = EmrCreateCluster.ScalingAdjustmentType || (EmrCreateCluster.ScalingAdjustmentType = {}));
})(EmrCreateCluster = exports.EmrCreateCluster || (exports.EmrCreateCluster = {}));
//# sourceMappingURL=data:application/json;base64,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