"""
Module to parse index.xml generated by Doxygen
"""

from xml.dom import minidom
from .class_obj import Class
import os


class IndexParser:
    """
    Class parsing index.xml and storing the classes and namespaces descriptions

    Attributes:
        _index: str
            path to the index.xml to parse
        _classes: list
            list of class object describing the classes
        _namespaces: list
            list of namespaces
    """

    def __init__(self, in_index):
        """

        :param in_index: index.xml file path
        :type in_index: str
        """
        self._index = in_index
        self._classes = []
        self._namespaces = []

        self._parse_index()
        self._parse_classes()

    @property
    def classes(self):
        """Get _classes"""
        return self._classes

    @property
    def namespaces(self):
        """Get _namespaces"""
        return self._namespaces

    def _parse_index(self):
        """
        Parse index file to get classes and namespace ids
        """
        xml_doc = minidom.parse(self._index)

        compounds = xml_doc.getElementsByTagName('compound')

        classes = []
        namespaces = []

        for s in compounds:
            kind = s.attributes["kind"].value
            if kind in ["class","struct","interface","protocol","exception"]:
                # print(kind)
                # print(s.getElementsByTagName('name')[0].firstChild.nodeValue)
                classes.append((s.getElementsByTagName('name')[0].firstChild.nodeValue, s.attributes["refid"].value))
            if kind in ["namespace"]:
                # print(s.getElementsByTagName('name')[0].firstChild.nodeValue)
                namespaces.append((s.getElementsByTagName('name')[0].firstChild.nodeValue, s.attributes["refid"].value))

        self._classes = classes
        self._namespaces = namespaces

    def _parse_classes(self):
        """
        Parse the classes xml
        """
        classes = []
        doxy_root = os.path.dirname(self._index)

        for clss in self._classes:
            obj = Class.from_xml(os.path.join(doxy_root, clss[1] + ".xml"))
            classes.append(obj)

        self._classes = classes


if __name__ == "__main__":
    index = "example_res/Doxygen/xml/index.xml"
    parser = IndexParser(index)

    for c in parser.classes:
        print(c)
