"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const cdk = require("aws-cdk-lib");
const lib_1 = require("../lib");
const api = require("aws-cdk-lib/aws-apigateway");
const iam = require("aws-cdk-lib/aws-iam");
const core_1 = require("@aws-solutions-constructs/core");
// App setup
const app = new cdk.App();
const stack = new cdk.Stack(app, core_1.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-apigateway-iot with overriden params';
const apiGatewayProps = {
    restApiName: 'RestApi-Regional',
    description: 'Description for the Regional Rest Api',
    endpointConfiguration: { types: [api.EndpointType.REGIONAL] },
    apiKeySourceType: api.ApiKeySourceType.HEADER,
    defaultMethodOptions: {
        authorizationType: api.AuthorizationType.NONE,
    }
};
const policyJSON = {
    Version: "2012-10-17",
    Statement: [
        {
            Action: [
                "iot:UpdateThingShadow"
            ],
            Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/*`,
            Effect: "Allow"
        },
        {
            Action: [
                "iot:Publish"
            ],
            Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/*`,
            Effect: "Allow"
        }
    ]
};
const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
const iamRoleProps = {
    assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
    path: '/',
    inlinePolicies: { testPolicy: policyDocument }
};
// Create a policy that overrides the default policy that gets created with the construct
const apiGatewayExecutionRole = new iam.Role(stack, 'apigateway-iot-role', iamRoleProps);
// Api gateway setup
const props = {
    iotEndpoint: 'a1234567890123-ats',
    apiGatewayCreateApiKey: true,
    apiGatewayExecutionRole,
    apiGatewayProps
};
// Instantiate construct
new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot', props);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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