"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaArnCaptorCustomResource = exports.ComprehendS3olab = exports.IamRoleName = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const lambda = require("aws-cdk-lib/aws-lambda-nodejs");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const s3delpoy = require("aws-cdk-lib/aws-s3-deployment");
const s3objectlambda = require("aws-cdk-lib/aws-s3objectlambda");
const cr = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
const comprehend_lambdas_1 = require("./comprehend-lambdas");
const iam_roles_1 = require("./iam-roles");
var IamRoleName;
(function (IamRoleName) {
    IamRoleName["GENERAL"] = "General";
    IamRoleName["ADMIN"] = "Admin";
    IamRoleName["BILLING"] = "Billing";
    IamRoleName["CUST_SUPPORT"] = "CustSupport";
})(IamRoleName = exports.IamRoleName || (exports.IamRoleName = {}));
/**
 * Creates the foundation necessary to deploy the S3 Object Lambda Acceess Control Use Case.
 */
class ComprehendS3olab extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Creates a redaction Lambda function for a speicfic IAM role.
         *
         * @param roleName the name of the corresponding IAM role
         * @param serverlessLambdaConfig the corresponding parameter setting for the Amazon Comprehend.
         */
        this.initializeRedactLambda = (roleName, serverlessLambdaConfig) => {
            const semanticVersion = serverlessLambdaConfig?.semanticVersion ?? '1.0.2';
            const logLevel = serverlessLambdaConfig?.logLevel ?? 'INFO';
            const unsupportedFileHandling = serverlessLambdaConfig?.unsupportedFileHandling ?? 'FAIL';
            const isPartialObjectSupported = serverlessLambdaConfig?.isPartialObjectSupported ?? 'false';
            const documentMaxSizeContainsPiiEntities = serverlessLambdaConfig?.documentMaxSizeContainsPiiEntities ?? '50000';
            const documentMaxSizeDetectPiiEntities = serverlessLambdaConfig?.documentMaxSizeDetectPiiEntities ?? '5000';
            const piiEntityTypes = serverlessLambdaConfig?.piiEntityTypes ?? 'ALL';
            const maskCharacter = serverlessLambdaConfig?.maskCharacter ?? '*';
            const maskMode = serverlessLambdaConfig?.maskMode ?? 'MASK';
            const subsegmentOverlappingTokens = serverlessLambdaConfig?.subsegmentOverlappingTokens ?? '20';
            const documentMaxSize = serverlessLambdaConfig?.documentMaxSize ?? '102400';
            const confidenceThreshold = serverlessLambdaConfig?.confidenceThreshold ?? '0.5';
            const maxCharsOverlap = serverlessLambdaConfig?.maxCharsOverlap ?? '200';
            const defaultLanguageCode = serverlessLambdaConfig?.defaultLanguageCode ?? 'en';
            const detectPiiEntitiesThreadCount = serverlessLambdaConfig?.detectPiiEntitiesThreadCount ?? '8';
            const containsPiiEntitiesThreadCount = serverlessLambdaConfig?.containsPiiEntitiesThreadCount ?? '20';
            const publishCloudWatchMetrics = serverlessLambdaConfig?.publishCloudWatchMetrics ?? 'true';
            return new comprehend_lambdas_1.RedactionLambda(this, `ServerlessRedactLambda-${roleName}`, {
                semanticVersion: semanticVersion,
                logLevel: logLevel,
                unsupportedFileHandling: unsupportedFileHandling,
                isPartialObjectSupported: isPartialObjectSupported,
                documentMaxSizeContainsPiiEntities: documentMaxSizeContainsPiiEntities,
                documentMaxSizeDetectPiiEntities: documentMaxSizeDetectPiiEntities,
                piiEntityTypes: piiEntityTypes,
                maskCharacter: maskCharacter,
                maskMode: maskMode,
                subsegmentOverlappingTokens: subsegmentOverlappingTokens,
                documentMaxSize: documentMaxSize,
                confidenceThreshold: confidenceThreshold,
                maxCharsOverlap: maxCharsOverlap,
                defaultLanguageCode: defaultLanguageCode,
                detectPiiEntitiesThreadCount: detectPiiEntitiesThreadCount,
                containsPiiEntitiesThreadCount: containsPiiEntitiesThreadCount,
                publishCloudWatchMetrics: publishCloudWatchMetrics,
            });
        };
        /**
         * Gets properties related to the IAM roles.
         *
         * @param roleName the name of the corresponding IAM role.
         * @param roleConfig the corresponding parameter setting for one of the IAM role.
         * @returns the property group.
         */
        this.getIamRoleConfig = (roleName, roleConfig) => {
            return {
                policyName: roleConfig?.policyName ?? this._getDefaultPolicyName(roleName),
                objectLambdaAccessPointName: roleConfig?.objectLambdaAccessPointName ?? this._getObjectLambdaAccessPointName(roleName),
                iamRoleName: roleConfig?.policyName ?? this._getRoleName(roleName),
            };
        };
        const exampleFileDir = props?.exampleFileDir ?? __dirname;
        const generateRandomCharacters = props?.generateRandomCharacters ?? true;
        // prerequisites for IAM roles
        const generalRoleConfig = this.getIamRoleConfig(IamRoleName.GENERAL, props?.generalRoleConfig);
        const adminRoleConfig = this.getIamRoleConfig(IamRoleName.ADMIN, props?.adminRoleConfig);
        const billingRoleConfig = this.getIamRoleConfig(IamRoleName.BILLING, props?.billingRoleConfig);
        const custSupportRoleConfig = this.getIamRoleConfig(IamRoleName.CUST_SUPPORT, props?.custSupportRoleConfig);
        // s3 bucekt and its access points
        const surveyBucketPrefix = props.surveyBucketPrefix ?? this.generateS3Prefix(6);
        const transcriptsBucketPrefix = props.transcriptsBucketPrefix ?? this.generateS3Prefix(6);
        const s3AccessPointNames = {
            general: props?.s3AccessPointNames?.general ?? ((generateRandomCharacters) ? `accessctl-s3-ap-survey-results-unknown-pii-${this.generateS3Prefix(6)}` : 'accessctl-s3-ap-survey-results-unknown-pii'),
            admin: props?.s3AccessPointNames?.admin ?? ((generateRandomCharacters) ? `admin-s3-access-point-call-transcripts-known-pii-${this.generateS3Prefix(6)}` : 'admin-s3-access-point-call-transcripts-known-pii'),
            billing: props?.s3AccessPointNames?.billing ?? ((generateRandomCharacters) ? `bill-s3-access-point-call-transcripts-known-pii-${this.generateS3Prefix(6)}` : 'bill-s3-access-point-call-transcripts-known-pii'),
            customerSupport: props?.s3AccessPointNames?.customerSupport ?? ((generateRandomCharacters) ? `cs-s3-access-point-call-transcripts-known-pii-${this.generateS3Prefix(6)}` : 'cs-s3-access-point-call-transcripts-known-pii'),
        };
        // object Lambda
        const acsemanticVersion = props?.accessControlLambdaConfig?.semanticVersion ?? '1.0.2';
        const acconfidenceThreshold = props?.accessControlLambdaConfig?.confidenceThreshold ?? '0.5';
        const accontainsPiiEntitiesThreadCount = props?.accessControlLambdaConfig?.containsPiiEntitiesThreadCount ?? '20';
        const acdefaultLanguageCode = props?.accessControlLambdaConfig?.defaultLanguageCode ?? 'en';
        const acdocumentMaxSize = props?.accessControlLambdaConfig?.documentMaxSize ?? '102400';
        const acdocumentMaxSizeContainsPiiEntities = props?.accessControlLambdaConfig?.documentMaxSizeContainsPiiEntities ?? '50000';
        const acisPartialObjectSupported = props?.accessControlLambdaConfig?.isPartialObjectSupported ?? 'false';
        const aclogLevel = props?.accessControlLambdaConfig?.logLevel ?? 'INFO';
        const acmaxCharsOverlap = props?.accessControlLambdaConfig?.maxCharsOverlap ?? '200';
        const acpiiEntityTypes = props?.accessControlLambdaConfig?.piiEntityTypes ?? 'ALL';
        const acpublishCloudWatchMetrics = props?.accessControlLambdaConfig?.publishCloudWatchMetrics ?? 'true';
        const acsubsegmentOverlappingTokens = props?.accessControlLambdaConfig?.subsegmentOverlappingTokens ?? '20';
        const acunsupportedFileHandling = props?.accessControlLambdaConfig?.unsupportedFileHandling ?? 'FAIL';
        const accessControlLambda = new comprehend_lambdas_1.AccessConrtolLambda(this, 'ServerlessLambda', {
            semanticVersion: acsemanticVersion,
            confidenceThreshold: acconfidenceThreshold,
            containsPiiEntitiesThreadCount: accontainsPiiEntitiesThreadCount,
            defaultLanguageCode: acdefaultLanguageCode,
            documentMaxSize: acdocumentMaxSize,
            documentMaxSizeContainsPiiEntities: acdocumentMaxSizeContainsPiiEntities,
            isPartialObjectSupported: acisPartialObjectSupported,
            logLevel: aclogLevel,
            maxCharsOverlap: acmaxCharsOverlap,
            piiEntityTypes: acpiiEntityTypes,
            publishCloudWatchMetrics: acpublishCloudWatchMetrics,
            subsegmentOverlappingTokens: acsubsegmentOverlappingTokens,
            unsupportedFileHandling: acunsupportedFileHandling,
        });
        // ComprehendPiiRedactionS3ObjectLambda
        const adminRedactLambda = this.initializeRedactLambda(IamRoleName.ADMIN, props?.adminRedactionLambdaConfig);
        const billingRedactLambda = this.initializeRedactLambda(IamRoleName.BILLING, props?.billingRedactionLambdaConfig);
        const custSupportRedactLambda = this.initializeRedactLambda(IamRoleName.CUST_SUPPORT, props?.cusrtSupportRedactionLambdaConfig);
        billingRedactLambda.node.addDependency(adminRedactLambda);
        custSupportRedactLambda.node.addDependency(billingRedactLambda);
        // IAM roles
        const generalRole = new iam_roles_1.GeneralRole(this, 'General', {
            objectLambdaAccessPointName: generalRoleConfig.objectLambdaAccessPointName,
            policyName: generalRoleConfig.policyName,
            iamRoleName: generalRoleConfig.iamRoleName,
        });
        const adminRole = new iam_roles_1.AdminRole(this, 'Admin', {
            objectLambdaAccessPointName: adminRoleConfig.objectLambdaAccessPointName,
            policyName: adminRoleConfig.policyName,
            iamRoleName: adminRoleConfig.iamRoleName,
        });
        const billingRole = new iam_roles_1.BillingRole(this, 'Billing', {
            objectLambdaAccessPointName: billingRoleConfig.objectLambdaAccessPointName,
            policyName: billingRoleConfig.policyName,
            iamRoleName: billingRoleConfig.iamRoleName,
        });
        const custSupportRole = new iam_roles_1.CustSupportRole(this, 'CustSupport', {
            objectLambdaAccessPointName: custSupportRoleConfig.objectLambdaAccessPointName,
            policyName: custSupportRoleConfig.policyName,
            iamRoleName: custSupportRoleConfig.iamRoleName,
        });
        // S3 buckets
        const surveyBucket = new s3.Bucket(this, 'SurveyResultBucket', {
            bucketName: (generateRandomCharacters) ? `survey-results-unknown-pii-${surveyBucketPrefix}` : 'survey-results-unknown-pii-123456',
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
        });
        surveyBucket.addToResourcePolicy(new iam.PolicyStatement({
            sid: 'AWSBucketGetPolicy',
            principals: [new iam.AccountRootPrincipal()],
            actions: ['s3:GetObject'],
            effect: iam.Effect.ALLOW,
            resources: [`${surveyBucket.bucketArn}/*`],
            conditions: { ['StringEquals']: { 's3:DataAccessPointAccount': cdk.Aws.ACCOUNT_ID } },
        }));
        surveyBucket.addToResourcePolicy(new iam.PolicyStatement({
            sid: 'AWSBucketPutPolicy',
            principals: [new iam.AccountRootPrincipal()],
            actions: ['s3:PutObject'],
            effect: iam.Effect.ALLOW,
            resources: [`${surveyBucket.bucketArn}/*`],
        }));
        const transcriptBucket = new s3.Bucket(this, 'TranscriptBucket', {
            bucketName: (generateRandomCharacters) ? `call-transcripts-known-pii-${transcriptsBucketPrefix}` : 'call-transcripts-known-pii-1234456',
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
        });
        transcriptBucket.addToResourcePolicy(new iam.PolicyStatement({
            sid: 'AWSBucketGetPolicy',
            principals: [new iam.AccountRootPrincipal()],
            actions: ['s3:GetObject'],
            effect: iam.Effect.ALLOW,
            resources: [`${transcriptBucket.bucketArn}/*`],
            conditions: { ['StringEquals']: { 's3:DataAccessPointAccount': cdk.Aws.ACCOUNT_ID } },
        }));
        transcriptBucket.addToResourcePolicy(new iam.PolicyStatement({
            sid: 'AWSBucketPutPolicy',
            principals: [new iam.AccountRootPrincipal()],
            actions: ['s3:PutObject'],
            effect: iam.Effect.ALLOW,
            resources: [`${transcriptBucket.bucketArn}/*`],
        }));
        surveyBucket.node.addDependency(accessControlLambda);
        surveyBucket.node.addDependency(generalRole);
        transcriptBucket.node.addDependency(adminRedactLambda);
        transcriptBucket.node.addDependency(billingRedactLambda);
        transcriptBucket.node.addDependency(custSupportRedactLambda);
        transcriptBucket.node.addDependency(adminRole);
        transcriptBucket.node.addDependency(billingRole);
        transcriptBucket.node.addDependency(custSupportRole);
        // custom resources
        const generalLambdaArnCaptor = new LambdaArnCaptorCustomResource(this, 'GeneralCaptor', {
            roleName: IamRoleName.GENERAL,
            partialLambdaName: 'PiiAccessControlFunction',
        });
        const adminLambdaArnCaptor = new LambdaArnCaptorCustomResource(this, 'AdminLambdaCaptor', {
            roleName: IamRoleName.ADMIN,
            partialLambdaName: 'PiiRedactionFunction',
        });
        const billingLambdaArnCaptor = new LambdaArnCaptorCustomResource(this, 'BillingCaptor', {
            roleName: IamRoleName.BILLING,
            partialLambdaName: 'PiiRedactionFunction',
        });
        const customSupportLambdaArnCaptor = new LambdaArnCaptorCustomResource(this, 'CustomSupportCaptor', {
            roleName: IamRoleName.CUST_SUPPORT,
            partialLambdaName: 'PiiRedactionFunction',
        });
        generalLambdaArnCaptor.node.addDependency(accessControlLambda);
        adminLambdaArnCaptor.node.addDependency(adminRedactLambda);
        billingLambdaArnCaptor.node.addDependency(billingRedactLambda);
        customSupportLambdaArnCaptor.node.addDependency(custSupportRedactLambda);
        this.piiAccessConrtolLambdaArn = generalLambdaArnCaptor.lambdaArn;
        this.adminLambdaArn = adminLambdaArnCaptor.lambdaArn;
        this.billingLambdaArn = billingLambdaArnCaptor.lambdaArn;
        this.customerSupportLambdaArn = customSupportLambdaArnCaptor.lambdaArn;
        // S3 access points
        const generalAccessPoint = new s3.CfnAccessPoint(this, 'AccessControlS3AccessPoint', {
            bucket: surveyBucket.bucketName,
        });
        generalAccessPoint.addPropertyOverride('Name', s3AccessPointNames.general.substring(0, 50));
        generalAccessPoint.addOverride;
        const adminAccessPoint = new s3.CfnAccessPoint(this, 'AdminS3AccessPoint', {
            bucket: transcriptBucket.bucketName,
        });
        adminAccessPoint.addPropertyOverride('Name', s3AccessPointNames.admin.substring(0, 50));
        const billingAccessPoint = new s3.CfnAccessPoint(this, 'BillingS3AccessPoint', {
            bucket: transcriptBucket.bucketName,
        });
        adminAccessPoint.addPropertyOverride('Name', s3AccessPointNames.billing.substring(0, 50));
        const customerSupportAccessPoint = new s3.CfnAccessPoint(this, 'CustomerSupportS3AccessPoint', {
            bucket: transcriptBucket.bucketName,
        });
        customerSupportAccessPoint.addPropertyOverride('Name', s3AccessPointNames.customerSupport.substring(0, 50));
        // S3ObjectLambda resources
        const accessControlObjectLambda = new s3objectlambda.CfnAccessPoint(this, 'GeneralLambdaAccessPoint', {
            name: generalRoleConfig.objectLambdaAccessPointName,
            objectLambdaConfiguration: {
                supportingAccessPoint: `arn:${cdk.Aws.PARTITION}:s3:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:accesspoint/${generalAccessPoint.attrName}`,
                transformationConfigurations: [
                    {
                        actions: ['GetObject'],
                        contentTransformation: new cdk.CfnJson(this, 'GeneralObjectLambdaConfig', {
                            value: {
                                AwsLambda: {
                                    FunctionArn: this.piiAccessConrtolLambdaArn,
                                },
                            },
                        }),
                    },
                ],
            },
        });
        const adminObjectLambda = new s3objectlambda.CfnAccessPoint(this, 'AdminLambdaAccessPoint', {
            name: adminRoleConfig.objectLambdaAccessPointName,
            objectLambdaConfiguration: {
                supportingAccessPoint: `arn:${cdk.Aws.PARTITION}:s3:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:accesspoint/${adminAccessPoint.attrName}`,
                transformationConfigurations: [
                    {
                        actions: ['GetObject'],
                        contentTransformation: new cdk.CfnJson(this, 'AdminObjectLambdaConfig', {
                            value: {
                                AwsLambda: {
                                    FunctionArn: this.adminLambdaArn,
                                },
                            },
                        }),
                    },
                ],
            },
        });
        const billingObjectLambda = new s3objectlambda.CfnAccessPoint(this, 'BillingLambdaAccessPoint', {
            name: billingRoleConfig.objectLambdaAccessPointName,
            objectLambdaConfiguration: {
                supportingAccessPoint: `arn:${cdk.Aws.PARTITION}:s3:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:accesspoint/${billingAccessPoint.attrName}`,
                transformationConfigurations: [
                    {
                        actions: ['GetObject'],
                        contentTransformation: new cdk.CfnJson(this, 'BillingObjectLambdaConfig', {
                            value: {
                                AwsLambda: {
                                    FunctionArn: this.billingLambdaArn,
                                },
                            },
                        }),
                    },
                ],
            },
        });
        const customerSupportObjectLambda = new s3objectlambda.CfnAccessPoint(this, 'CustomerSupportLambdaAccessPoint', {
            name: custSupportRoleConfig.objectLambdaAccessPointName,
            objectLambdaConfiguration: {
                supportingAccessPoint: `arn:${cdk.Aws.PARTITION}:s3:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:accesspoint/${customerSupportAccessPoint.attrName}`,
                transformationConfigurations: [
                    {
                        actions: ['GetObject'],
                        contentTransformation: new cdk.CfnJson(this, 'CustomerSupportObjectLambdaConfig', {
                            value: {
                                AwsLambda: {
                                    FunctionArn: this.customerSupportLambdaArn,
                                },
                            },
                        }),
                    },
                ],
            },
        });
        // Automatic uploading
        const surveyFilePath = path.join(exampleFileDir, 'files/access_control');
        const redactionFilePath = path.join(exampleFileDir, 'files/redaction');
        console.log(`surveyFilePath: ${surveyFilePath}`);
        console.log(`redactionFilePath: ${redactionFilePath}`);
        if (fs.existsSync(surveyFilePath)) {
            const deployFiles = new s3delpoy.BucketDeployment(this, 'DeploySurveyResultFiles', {
                sources: [s3delpoy.Source.asset(surveyFilePath)],
                destinationBucket: surveyBucket,
            });
            deployFiles.node.addDependency(accessControlObjectLambda);
        }
        ;
        if (fs.existsSync(redactionFilePath)) {
            const deployTranscriptFiles = new s3delpoy.BucketDeployment(this, 'DeployTranscriptFiles', {
                sources: [s3delpoy.Source.asset(redactionFilePath)],
                destinationBucket: transcriptBucket,
            });
            deployTranscriptFiles.node.addDependency(adminObjectLambda);
            deployTranscriptFiles.node.addDependency(billingObjectLambda);
            deployTranscriptFiles.node.addDependency(customerSupportObjectLambda);
        }
        ;
        this.s3objectLambdaAccessControlArn = cdk.Token.asString(cdk.Fn.getAtt(accessControlObjectLambda.logicalId, 'Arn'));
        this.s3objectLambdaAdminArn = cdk.Token.asString(cdk.Fn.getAtt(adminObjectLambda.logicalId, 'Arn'));
        this.s3objectLambdaBillingArn = cdk.Token.asString(cdk.Fn.getAtt(billingObjectLambda.logicalId, 'Arn'));
        this.s3objectLambdaCustomerSupportArn = cdk.Token.asString(cdk.Fn.getAtt(customerSupportObjectLambda.logicalId, 'Arn'));
        accessControlObjectLambda.node.addDependency(generalLambdaArnCaptor);
        adminObjectLambda.node.addDependency(adminLambdaArnCaptor);
        billingObjectLambda.node.addDependency(billingLambdaArnCaptor);
        customerSupportObjectLambda.node.addDependency(customSupportLambdaArnCaptor);
    }
    generateS3Prefix(length) {
        var result = '';
        const characters = 'abcdefghijklmnopqrstuvwxyz0123456789';
        const charactersLength = characters.length;
        for (var i = 0; i < length; i++) {
            result += characters.charAt(Math.floor(Math.random() * charactersLength));
        }
        return result;
    }
    /**
     * Returns the policy name according to the IAM role.
     *
     * @param roleName the name of the corresponding IAM role.
     */
    _getDefaultPolicyName(roleName) {
        switch (roleName) {
            case IamRoleName.GENERAL: return 'general-role-s3olap-policy';
            case IamRoleName.ADMIN: return 'admin-role-s3olap-policy';
            case IamRoleName.BILLING: return 'billing-role-s3olap-policy';
            case IamRoleName.CUST_SUPPORT: return 'customersupport-role-s3olap-policy';
            default:
                console.log('Check the name of the IAM role, there might be a problematci spelling.');
                return 'undefined';
        }
    }
    /**
     * Returns the corresponding name for the access point of the S3 Object Lambda according to the IAM role.
     *
     * @param roleName the name of the corresponding IAM role.
     */
    _getObjectLambdaAccessPointName(roleName) {
        switch (roleName) {
            case IamRoleName.GENERAL: return 'accessctl-s3olap-survey-results-unknown-pii';
            case IamRoleName.ADMIN: return 'admin-s3olap-call-transcripts-known-pii';
            case IamRoleName.BILLING: return 'billing-s3olap-call-transcripts-known-pii';
            case IamRoleName.CUST_SUPPORT: return 'custsupport-s3olap-call-transcripts-known-pii';
            default:
                console.log('Check the name of the IAM role, there might be a problematci spelling.');
                return 'undefined';
        }
    }
    /**
     * Returns the corresponding name for the IAM role.
     *
     * @param roleName the name of the corresponding IAM role.
     */
    _getRoleName(roleName) {
        switch (roleName) {
            case IamRoleName.GENERAL: return 'GeneralRole';
            case IamRoleName.ADMIN: return 'RedactionAdminRole';
            case IamRoleName.BILLING: return 'RedactionBillingRole';
            case IamRoleName.CUST_SUPPORT: return 'RedactionCustSupportRole';
            default:
                console.log('Check the name of the IAM role, there might be a problematci spelling.');
                return 'undefined';
        }
    }
}
exports.ComprehendS3olab = ComprehendS3olab;
_a = JSII_RTTI_SYMBOL_1;
ComprehendS3olab[_a] = { fqn: "cdk-comprehend-s3olap.ComprehendS3olab", version: "2.0.311" };
class LambdaArnCaptorCustomResource extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const customResourceRole = new iam.Role(this, 'CRRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            description: 'An execution role to find out the ARN of the access control Lambda.',
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('AWSXRayDaemonWriteAccess'),
            ],
            inlinePolicies: {
                ComprehendS3ObjectLambdaCustomResourcePolicy: new iam.PolicyDocument({
                    assignSids: true,
                    statements: [
                        new iam.PolicyStatement({
                            sid: 'ListLambdaPermissions',
                            effect: iam.Effect.ALLOW,
                            actions: ['lambda:ListFunctions'],
                            resources: ['*'],
                        }),
                        new iam.PolicyStatement({
                            sid: 'GetLambdaFunctionsPermissions',
                            effect: iam.Effect.ALLOW,
                            actions: ['lambda:GetFunction', 'lambda:InvokeFunction'],
                            resources: [`arn:${cdk.Aws.PARTITION}:lambda:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:function:*`],
                        }),
                    ],
                }),
            },
        });
        const onEvent = new lambda.NodejsFunction(this, 'ArnExpert', {
            description: `A Lambda function that gets the ARN of \`${props.partialLambdaName}\``,
            entry: fs.existsSync(path.join(__dirname, 'resources/lambda-arn-helper.ts')) ? path.join(__dirname, 'resources/lambda-arn-helper.ts') : path.join(__dirname, 'resources/lambda-arn-helper.js'),
            handler: 'lambdaHandler',
            timeout: cdk.Duration.minutes(2),
            runtime: aws_lambda_1.Runtime.NODEJS_14_X,
            bundling: {
                minify: false,
                sourceMap: true,
                externalModules: ['aws-sdk'],
            },
            role: customResourceRole,
        });
        const provider = new cr.Provider(this, 'Provider', {
            onEventHandler: onEvent,
            logRetention: logs.RetentionDays.ONE_MONTH,
            role: customResourceRole.withoutPolicyUpdates(),
        });
        const lambdaArnSearchUnit = new cdk.CustomResource(this, `${props.roleName}GeneralSearchUnit`, {
            serviceToken: provider.serviceToken,
            properties: {
                LambdaFixedName: props.partialLambdaName,
                Genre: props.roleName,
            },
        });
        this.lambdaArn = cdk.Token.asString(lambdaArnSearchUnit.getAtt('LambdaArn'));
    }
}
exports.LambdaArnCaptorCustomResource = LambdaArnCaptorCustomResource;
_b = JSII_RTTI_SYMBOL_1;
LambdaArnCaptorCustomResource[_b] = { fqn: "cdk-comprehend-s3olap.LambdaArnCaptorCustomResource", version: "2.0.311" };
//# sourceMappingURL=data:application/json;base64,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