# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['eoplatform',
 'eoplatform.composites',
 'eoplatform.metadata',
 'eoplatform.platforms',
 'eoplatform.platforms.download',
 'eoplatform.platforms.info']

package_data = \
{'': ['*'], 'eoplatform': ['data/composites/*', 'data/platforms/*']}

install_requires = \
['numpy>=1.21.3,<2.0.0', 'rich>=10.12.0,<11.0.0', 'typer[all]>=0.4.0,<0.5.0']

entry_points = \
{'console_scripts': ['eoplatform = eoplatform.cli:app']}

setup_kwargs = {
    'name': 'eoplatform',
    'version': '0.2.4',
    'description': 'Earth Observation made easy.',
    'long_description': '<br/>\n<p align="center">\n  <a href="https://github.com/mtralka/EOPlatform">\n    <img src="images/logo.jpg" alt="EOP Logo" width="300" height="300">\n  </a>\n\n  <h3 align="center">An Earth Observation Platform</h3>\n\n  <p align="center">\n    Earth Observation made easy. \n    <br/>\n    <br/>\n    <a href="https://github.com/mtralka/EOPlatform/issues">Report Bug</a>\n    |\n    <a href="https://github.com/mtralka/EOPlatform/issues">Request Feature</a>\n  </p>\n</p>\n\n![Downloads](https://img.shields.io/github/downloads/mtralka/EOPlatform/total) ![Forks](https://img.shields.io/github/forks/mtralka/EOPlatform?style=social) ![Stargazers](https://img.shields.io/github/stars/mtralka/EOPlatform?style=social) <br/> ![Issues](https://img.shields.io/github/issues/mtralka/EOPlatform) ![License](https://img.shields.io/github/license/mtralka/EOPlatform) [![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black) ![mypy](https://img.shields.io/badge/mypy-checked-brightgreen)\n\n## About\n\n*eoplatform* is a Python package that aims to simplify Remote Sensing Earth Observation by providing actionable information on a wide swath of RS platforms and provide a simple API for downloading and visualizing RS imagery. Made for scientists, educators, and hobbiests alike.\n\n* Easy access to information on RS platforms\n* `metadata` module for extracting platform metadata\n  * supports `.txt` and `.xml` files\n* `composites` modules for creating and learning about 91 RS band composites\n\nComing soon:\n\n* Data downloading\n  * Landsat 8\n  * Sentinel-2\n* Raster tools\n  * Raster IO functions\n\n### Installation\n\n*eoplatform* can be installed by running `pip install eoplatform`. It requires Python 3.8 or above to run.\n\n#### *eoplatform* development is ongoing -\n\nupgrade to the latest version by running:\n`pip install eoplatform -U`\n\nInstall the latest dev version with:\n`pip install git+https://github.com/mtralka/EOPlatform/eoplatform`\n\n### Usage\n\n*eoplatform* is accessible through the command line (CLI) and as a module import.\n\n### Querying information\n\n#### ...through CLI\n\n`NAME` is any supported composite or platform (case-insensitive)\n\n```sh\nUsage: eoplatform info [OPTIONS] NAME\n\nArguments:\n  NAME  [required]\n\nOptions:\n  -b, --only-bands\n   / -nd, --no-description  [default: True]\n  --help                    Show this message and exit.\n```\n\nEX:\n\n**Show all information on `landsat8`**\n\n```sh\neoplatform info landsat8\n```\n\n![EOPlatform Landsat8 info example](images/eoplatform-info-landsat8.PNG)\n\n**Show only `landsat8` bands**\n\n```sh\neoplatform info landsat8 -b\n```\n\n**Show information on composite `NDVI`**\n\n```sh\neoplatform info ndvi\n```\n\n![EOPlatform NDVI info example](images/eoplatform-composite-info-ndvi.PNG)\n\n#### ...through imports\n\nYou can search through the `eoplatform` module\n\n```python\nimport eoplatform as eop\n\neop.info("Landsat8")  # case insensitive\neop.info("NDVI)\n```\n\nOr import your exact platform/composite\n\n```python\nfrom eoplatform.platforms import landsat8\nfrom eoplatform.composites import NDVI\n\nlandsat8.info()  # OR print(landsat8)\nlandsat8.bands.info()\nlandsat8.bands.RED.info()\nNDVI.info()  # or print(NDVI)\n```\n\n### Band composites\n\nImportable through `eoplatforms.composites` or searchable (shown above) from `eoplatform.info()`\n\nSee [implemented composite data dir](/eoplatform/data/composites) for the exact implemented composites (inspired by [awesome spectral indices](https://github.com/davemlz/awesome-spectral-indices))\n\n#### Composite information\n\n```python\nfrom eoplatform.composites import NDVI  # DVI, etc\n\nNDVI.info()\n```\n\n#### Creating composite\n\nComposites bands must be passed in as keyword arguments assigned to NumPy arrays.\n\n```python\nfrom eoplatform.composites import NDVI\n\nred_array: np.ndarray = ...\nnir_array: np.ndarray = ...\n\nndvi: np.ndarray = NDVI.create(NIR=nir_array, RED=red_array)\n```\n\n### Metadata extraction\n\nSupports `.txt` and `.xml` files through `extract_XML_metadata` and `extract_TXT_metadata`.\n\n```python\nfrom eoplatform.metadata import extract_XML_metadata\n\nfile_path: str = ...\ntarget_attributes: List[str] = ...\n\nvalues: Dict[str, str] = extract_XML_metadata(file_path, target_attributes)\n```\n\n## Adding platforms / composites\n\nPlatforms and composites are auto-generated from `/data/(composites,platforms)`. To add platforms or composites, simply create a new json file in the desired directory.\n\n### Platform\n\nPlatforms must have:\n\n```python\nabbreviation: str\nname: str\n```\n\n#### Example platform JSON\n\nStandard key with value:\n\n```json\n"abbreviation": "L8",\n//snip\n```\n\nKey with value and metadata:\n\n```json\n"altitude": {\n    "meta": {\n      "unit": "km"\n    },\n    "value": 705\n  }\n//snip\n```\n\n#### Bands standard for Platform key `bands`\n\nPlatform key for `bands` must be a list of following the `band` standard. EX:\n\n```json\n"bands": [\n    {\n      "abbreviation": "CAER",\n      "description":"",\n      "name": "Coastal Aerosol",\n      "number": 1,\n      "resolution": "30",\n      "sensor": "OLI",\n      "wavelength": "0.433-0.453"\n    },\n  //snip\n]\n```\n\n#### Band standard for el in `bands`\n\nEach element to `bands` is a `band`. Each band must have:\n\n```python\nnumber: int\nname: str\nabbreviation: str\n```\n\n### Composites\n\nComposites must have:\n\n```python\nabbreviation: str\nformula: str\nname: str\nreference: str\ntype: str  # matching CompositeType\nbands: List[str]\n```\n\n#### Example JSON Composite\n\n```json\n{\n  "abbreviation": "ARI",\n  "bands": [\n    "GREEN",\n    "VRE1"\n  ],\n  "description": "",\n  "formula": "(1 / GREEN) - (1 / VRE1)",\n  "name": "Anthocyanin Reflectance Index",\n  "reference": "https://doi.org/10.1562/0031-8655(2001)074%3C0038:OPANEO%3E2.0.CO;2",\n  "type": "vegetation"\n}\n\n```\n\n#### Composite Types\n\nPossible values for composite `type` key\n\n* VEGETATION\n* BURN\n* WATER\n* SNOW\n* DROUGHT\n* URBAN\n* KERNEL\n* NONE\n\n**For both platforms and composites, all other attributes will be dynamically rendered and shown in `info`**\n\n## Roadmap\n\nSee the [open issues](https://github.com/mtralka/EOPlatform/issues) for a list of proposed features (and known issues).\n\n* download support\n\n## Contributing\n\nContributions are welcome. Currently, *eoplatform* is undergoing rapid development and contribution opportunities may be scarce.\n\n* If you have suggestions for adding or removing features, feel free to [open an issue](https://github.com/mtralka/EOPlatform/issues/new) to discuss it, or directly create a pull request with the proposed changes.\n* Create individual PR for each suggestion.\n* Use pre-commit hooks - `pre-commit install`\n* Code style is `black`, `mypy --strict`\n\n## License\n\nDistributed under the GNU GPL-3.0 License. See [LICENSE](https://github.com/mtralka/EOPlatform/blob/main/LICENSE.md) for more information.\n\n## Built With\n\n* [Rich](https://github.com/willmcgugan/rich)\n* [Typer](https://github.com/tiangolo/typer)\n* [Numpy](https://numpy.org/)\n* [awesome spectral indices](https://github.com/davemlz/awesome-spectral-indices)\n\n## Authors\n\n* [**Matthew Tralka**](https://github.com/mtralka/)\n',
    'author': 'Matthew Tralka',
    'author_email': 'matthew@tralka.xyz',
    'maintainer': 'Matthew Tralka',
    'maintainer_email': 'matthew@tralka.xyz',
    'url': 'https://github.com/mtralka/EOPlatform',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<3.11',
}


setup(**setup_kwargs)
