"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
describe('Version', () => {
    describe('.isGreaterThan', () => {
        test.each([
            [
                'equal version',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.1',
                    expectedValue: false,
                },
            ], [
                'less than',
                {
                    firstVersion: '10.0.9.2',
                    secondVersion: '10.1.9.2',
                    expectedValue: false,
                },
            ], [
                'less than',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.2',
                    expectedValue: false,
                },
            ], [
                'greater than',
                {
                    firstVersion: '2.0.0.1',
                    secondVersion: '2.0.0.0',
                    expectedValue: true,
                },
            ],
        ])('%s', (_name, testcase) => {
            const { firstVersion, secondVersion, expectedValue } = testcase;
            // WHEN
            const lhs = lib_1.Version.parse(firstVersion);
            const result = lhs.isGreaterThan(lib_1.Version.parse(secondVersion));
            expect(result).toEqual(expectedValue);
        });
    });
    test('.isGreaterThan constructor', () => {
        // WHEN
        const lhs = new lib_1.Version([10, 1, 9, 2]);
        const result = lhs.isGreaterThan(lib_1.Version.parse('10.0.9.2'));
        expect(result).toEqual(true);
    });
    describe('.isLessThan', () => {
        test.each([
            [
                'equal version',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.1',
                    expectedValue: false,
                },
            ], [
                'greater minor version',
                {
                    firstVersion: '10.1.9.2',
                    secondVersion: '10.0.9.2',
                    expectedValue: false,
                },
            ], [
                'greater patch version',
                {
                    firstVersion: '1.1.1.2',
                    secondVersion: '1.1.1.1',
                    expectedValue: false,
                },
            ], [
                'less than',
                {
                    firstVersion: '2.0.0.0',
                    secondVersion: '2.0.0.1',
                    expectedValue: true,
                },
            ],
        ])('%s', (_name, testcase) => {
            const { firstVersion, secondVersion, expectedValue } = testcase;
            // WHEN
            const lhs = lib_1.Version.parse(firstVersion);
            const result = lhs.isLessThan(lib_1.Version.parse(secondVersion));
            expect(result).toEqual(expectedValue);
        });
    });
    describe('.isEqual', () => {
        test.each([
            [
                'equal version',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.1',
                    expectedValue: true,
                },
            ], [
                'unequal',
                {
                    firstVersion: '2.1.1.1',
                    secondVersion: '1.1.1.1',
                    expectedValue: false,
                },
            ], [
                'less than',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.2',
                    expectedValue: false,
                },
            ],
        ])('%s', (_name, testcase) => {
            const { firstVersion, secondVersion, expectedValue } = testcase;
            // WHEN
            const lhs = lib_1.Version.parse(firstVersion);
            const result = lhs.isEqual(lib_1.Version.parse(secondVersion));
            expect(result).toEqual(expectedValue);
        });
    });
    test('.isLessThan using constructor', () => {
        // WHEN
        const lhs = new lib_1.Version([10, 0, 9, 2]);
        const result = lhs.isLessThan(lib_1.Version.parse('10.1.9.2'));
        expect(result).toEqual(true);
    });
    describe('constructor validation', () => {
        test.each([
            [
                'incorrect component count',
                {
                    version: [10, 1, 9],
                    expectedException: /Invalid version format. Version should contain exactly 4 components./,
                },
            ], [
                'negative value',
                {
                    version: [10, -1, 9, 2],
                    expectedException: /Invalid version format. None of the version components can be negative./,
                },
            ], [
                'decimal value',
                {
                    version: [10, 1, 9.2, 2],
                    expectedException: /Invalid version format. None of the version components can contain decimal values./,
                },
            ], [
                'correct value',
                {
                    version: [10, 1, 9, 2],
                },
            ],
        ])('%s', (_name, testcase) => {
            const { version, expectedException } = testcase;
            // WHEN
            if (expectedException) {
                expect(() => new lib_1.Version(version)).toThrow(expectedException);
            }
            else {
                const versionObj = new lib_1.Version(version);
                expect(versionObj.majorVersion).toEqual(version[0]);
                expect(versionObj.minorVersion).toEqual(version[1]);
                expect(versionObj.releaseVersion).toEqual(version[2]);
                expect(versionObj.patchVersion).toEqual(version[3]);
            }
        });
    });
    describe('.parse throws exception', () => {
        test.each([
            [
                'ending with .',
                {
                    version: '10.1.9.',
                    expectedException: /Invalid version format/,
                },
            ], [
                'empty string',
                {
                    version: '',
                    expectedException: /Invalid version format/,
                },
            ], [
                'negative value',
                {
                    version: '10.-1.9.2',
                    expectedException: /Invalid version format/,
                },
            ], [
                'correct version',
                {
                    version: '10.1.9.2',
                },
            ],
        ])('%s', (_name, testcase) => {
            const { version, expectedException } = testcase;
            // WHEN
            if (expectedException) {
                expect(() => lib_1.Version.parse(version)).toThrow(expectedException);
            }
            else {
                expect(() => lib_1.Version.parse(version)).not.toThrow();
            }
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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