"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('ThinkboxDockerRecipes', () => {
    let app;
    let stack;
    let stage;
    // GIVEN
    const STAGE_PATH = path.join(__dirname, 'assets');
    const RCS_RECIPE_NAME = 'rcs';
    const RCS_RECIPE = {
        description: 'rcs',
        title: 'rcs',
        buildArgs: {
            a: 'a',
            b: 'b',
        },
        target: undefined,
    };
    const LICENSE_FORWARDER_RECIPE_NAME = 'license-forwarder';
    const LICENSE_FORWARDER_RECIPE = {
        title: 'license-forwarder',
        description: 'license-forwarder',
        buildArgs: {
            c: 'c',
            d: 'd',
        },
        target: 'lf',
    };
    const MAJOR_VERSION = 10;
    const MINOR_VERSION = 1;
    const RELEASE_VERSION = 9;
    const PATCH_VERSION = 2;
    const RELEASE_VERSION_STRING = `${MAJOR_VERSION}.${MINOR_VERSION}.${RELEASE_VERSION}`;
    const FULL_VERSION_STRING = `${RELEASE_VERSION_STRING}.${PATCH_VERSION}`;
    beforeEach(() => {
        app = new core_1.App();
        class TestStage extends lib_1.Stage {
            constructor(props) {
                super(props);
            }
        }
        stage = new TestStage({
            path: STAGE_PATH,
            manifest: {
                schema: 1,
                version: FULL_VERSION_STRING,
                recipes: {
                    [RCS_RECIPE_NAME]: RCS_RECIPE,
                    [LICENSE_FORWARDER_RECIPE_NAME]: LICENSE_FORWARDER_RECIPE,
                },
            },
        });
        stack = new core_1.Stack(app, 'Stack');
    });
    /**
     * Tests that the remote connection server and license forwarder Docker recipes
     * create the correct {@link DockerImageAsset} instances using the build arguments,
     * and target from the supplied manifest.
     *
     * This is done by comparing the `sourceHash` property of the expected vs actual
     * DockerImageAsset instances.
     */
    test.each([
        [
            'remoteConnnectionServer',
            () => {
                return new aws_ecr_assets_1.DockerImageAsset(stack, 'SomeID', {
                    directory: STAGE_PATH,
                    buildArgs: RCS_RECIPE.buildArgs,
                    target: RCS_RECIPE.target,
                });
            },
            (recipes) => {
                return recipes.remoteConnectionServer;
            },
        ],
        [
            'licenseForwarder',
            () => {
                return new aws_ecr_assets_1.DockerImageAsset(stack, 'SomeID', {
                    directory: STAGE_PATH,
                    buildArgs: LICENSE_FORWARDER_RECIPE.buildArgs,
                    target: LICENSE_FORWARDER_RECIPE.target,
                });
            },
            (recipes) => {
                return recipes.licenseForwarder;
            },
        ],
    ])('has proper %p container image', (_imageName, getExpected, getActual) => {
        // GIVEN
        const expectedImage = getExpected();
        // WHEN
        const recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
            stage,
        });
        const actualImage = getActual(recipes);
        // THEN
        expect(actualImage.sourceHash).toEqual(expectedImage.sourceHash);
    });
    test('provides the Deadline version', () => {
        // WHEN
        new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
            stage,
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_DEADLINE_INSTALLERS', {
            forceRun: assert_1.stringLike('*'),
            versionString: RELEASE_VERSION_STRING,
        }));
    });
    test.each([
        ['rcs', {
                [LICENSE_FORWARDER_RECIPE_NAME]: LICENSE_FORWARDER_RECIPE,
            }],
        ['license-forwarder', {
                [RCS_RECIPE_NAME]: RCS_RECIPE,
            }],
    ])('manifest missing required recipe %s throws error', (_recipeName, recipes) => {
        // GIVEN
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        class StageWithPublicConstructor extends lib_1.Stage {
            constructor(props) {
                super(props);
            }
        }
        const manifest = {
            recipes,
            schema: 1,
            version: '1.2.3.4',
        };
        // WHEN
        expect(() => {
            new lib_1.ThinkboxDockerRecipes(isolatedStack, 'Recipes', {
                stage: new StageWithPublicConstructor({
                    manifest,
                    path: '/some/path',
                }),
            });
        })
            // THEN
            .toThrow();
    });
});
//# sourceMappingURL=data:application/json;base64,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