"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
let app;
let stack;
let vpc;
let version;
function escapeTokenRegex(s) {
    // A CDK Token looks like: ${Token[TOKEN.12]}
    // This contains the regex special characters: ., $, {, }, [, and ]
    // Escape those for use in a regex.
    return s.replace(/[.${}[\]]/g, '\\$&');
}
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack');
    vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    version = {
        majorVersion: 10,
        minorVersion: 1,
        releaseVersion: 9,
        linuxInstallers: {
            patchVersion: 2,
            repository: {
                objectKey: 'testInstaller',
                s3Bucket: new aws_s3_1.Bucket(stack, 'InstallerBucket'),
            },
        },
        linuxFullVersionString: () => '10.1.9.2',
    };
});
test('can create two repositories', () => {
    // GIVEN
    new lib_1.Repository(stack, 'Repo1', {
        vpc,
        version,
    });
    // THEN
    expect(() => {
        new lib_1.Repository(stack, 'Repo2', {
            vpc,
            version,
        });
    }).not.toThrow();
});
test('repository installer instance is created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        Properties: {
            MaxSize: '1',
            MinSize: '1',
            VPCZoneIdentifier: [
                {
                    Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                },
                {
                    Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                },
            ],
        },
        CreationPolicy: {
            AutoScalingCreationPolicy: {
                MinSuccessfulInstancesPercent: 100,
            },
            ResourceSignal: {
                Timeout: 'PT15M',
            },
        },
        UpdatePolicy: {
            AutoScalingReplacingUpdate: {
                WillReplace: true,
            },
            AutoScalingScheduledAction: {
                IgnoreUnmodifiedGroupSizeProperties: true,
            },
        },
        DependsOn: [
            'repositoryInstallerDocumentDatabaseInstance11A6F8C8E',
        ],
    }, assert_1.ResourcePart.CompleteDefinition));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't3.large',
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: 2049,
        ToPort: 2049,
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerFileSystemEfsSecurityGroup289D043C',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    }));
});
test('repository installer honors vpcSubnet', () => {
    // Note: Default is private subnets, so it's sufficient to test something other than
    // private subnets.
    // WHEN
    const publicSubnetIds = ['PublicSubnet1', 'PublicSubnet2'];
    const attrVpc = aws_ec2_1.Vpc.fromVpcAttributes(stack, 'TestVpc', {
        availabilityZones: ['us-east-1a', 'us-east-1b'],
        vpcId: 'vpcid',
        publicSubnetIds,
    });
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc: attrVpc,
        version,
        vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PUBLIC },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        VPCZoneIdentifier: publicSubnetIds,
    }));
});
test('repository installer security groups created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: 2049,
        ToPort: 2049,
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerFileSystemEfsSecurityGroup289D043C',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseA36CE7FE',
                'Port',
            ],
        },
        ToPort: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseA36CE7FE',
                'Port',
            ],
        },
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseSecurityGroupBEFDC58F',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    }));
});
/*
IAM Policy document tests. The policy for the installer instance is:
  expectCDK(stack).to(haveResourceLike('AWS::IAM::Policy', {
    PolicyDocument: {
      Statement: [
        {}, // CloudWatch log group put
        {}, // cloudwatch agent install script
        {}, // cloudwatch agent string parameters
        {}, // cloudwatch agent get installer permissions
        {}, // DocDB secret get
        {}, // filesystem mount script get
        {}, // installer get
        {}, // repository installation script asset get
        {}, // update autoscaling policy
        {}, // describe tags
      ],
    },
  }));

  We only explicitly test for the permissions we explicitly add:
    - docDB secret get
    - installer get
    - autoscaling policy (handled by: 'alidate instance self-termination' test)
    - describe tags (handled by: 'alidate instance self-termination' test)
  The others are side-effects of adding the cloudwatch agent, and mounting the filesystem.
  We make sure that we do those things in other ways, and trust to their constructs to be
  verifying proper permissions.
 */
test('repository installer iam permissions: db secret access', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {},
                {},
                {},
                {},
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'repositoryInstallerDocumentDatabaseSecretAttachment29753B7C',
                    },
                },
                {},
                {},
                {},
                {},
                {},
            ],
        },
    }));
});
test('repository installer iam permissions: installer get', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Action: [
                    's3:GetObject*',
                    's3:GetBucket*',
                    's3:List*',
                ],
                Effect: 'Allow',
                Resource: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':s3:::',
                            {
                                Ref: assert_1.stringLike('AssetParameters*S3Bucket352E624B'),
                            },
                        ],
                    ],
                }, {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':s3:::',
                            {
                                Ref: assert_1.stringLike('AssetParameters*S3Bucket352E624B'),
                            },
                            '/*',
                        ],
                    ],
                }),
            }),
        },
    }));
});
test('default repository installer log group created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: '/renderfarm/repositoryInstaller',
    }));
});
test('repository installer logs all required files', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    // The CloudWatchAgent stores the agent configuration in an SSM Parameter. Check it for the required setup.
    // Note: This would be better implemented using the tools in: https://github.com/aws/aws-cdk/pull/8444
    // but that is not yet available.
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SSM::Parameter', {
        Type: 'String',
        Value: {
            'Fn::Join': [
                '',
                [
                    '{\"logs\":{\"logs_collected\":{\"files\":{\"collect_list\":[{\"log_group_name\":\"',
                    {},
                    '\",\"log_stream_name\":\"cloud-init-output-{instance_id}\",\"file_path\":\"/var/log/cloud-init-output.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                    {},
                    '\",\"log_stream_name\":\"deadlineRepositoryInstallationLogs-{instance_id}\",\"file_path\":\"/tmp/bitrock_installer.log\",\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}',
                ],
            ],
        },
    }));
});
test('repository mounts repository filesystem', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // WHEN
    const userData = repo.node.defaultChild.userData.render();
    // THEN
    expect(userData).toMatch(new RegExp(escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 rw')));
});
test.each([
    [core_1.RemovalPolicy.DESTROY, 'Delete'],
    [core_1.RemovalPolicy.RETAIN, 'Retain'],
    [core_1.RemovalPolicy.SNAPSHOT, 'Snapshot'],
])('repository honors database removal policy: %p', (policy, expected) => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            database: policy,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        DeletionPolicy: expected,
    }, assert_1.ResourcePart.CompleteDefinition));
});
test.each([
    [core_1.RemovalPolicy.DESTROY, 'Delete'],
    [core_1.RemovalPolicy.RETAIN, 'Retain'],
    [core_1.RemovalPolicy.SNAPSHOT, 'Snapshot'],
])('repository honors filesystem removal policy: %p', (policy, expected) => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            filesystem: policy,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EFS::FileSystem', {
        DeletionPolicy: expected,
    }, assert_1.ResourcePart.CompleteDefinition));
});
test('repository warns if removal policy for filesystem when filesystem provided', () => {
    // GIVEN
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testFS = new core_2.MountableEfs(stack, {
        filesystem: testEFS,
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
        removalPolicy: {
            filesystem: core_1.RemovalPolicy.DESTROY,
        },
    });
    // THEN
    expect(repo.node.metadata).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct',
        }),
    ]));
});
test('repository warns if removal policy for database when database provided', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceProps: {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
        removalPolicy: {
            database: core_1.RemovalPolicy.DESTROY,
        },
    });
    // THEN
    expect(repo.node.metadata).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'RemovalPolicy for database will not be applied since a database is not being created by this construct',
        }),
    ]));
});
test('repository creates deadlineDatabase if none provided', () => {
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testFS = new core_2.MountableEfs(stack, {
        filesystem: testEFS,
    });
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DocDB::DBCluster'));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DocDB::DBInstance'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        EnableCloudwatchLogsExports: ['audit'],
    }, assert_1.ResourcePart.Properties));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBClusterParameterGroup', {
        Parameters: {
            audit_logs: 'enabled',
        },
    }, assert_1.ResourcePart.Properties));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBInstance', {
        AutoMinorVersionUpgrade: true,
    }));
});
test('disabling Audit logging does not enable Cloudwatch audit logs', () => {
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testFS = new core_2.MountableEfs(stack, {
        filesystem: testEFS,
    });
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
        databaseAuditLogging: false,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DocDB::DBCluster'));
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        EnableCloudwatchLogsExports: ['audit'],
    }, assert_1.ResourcePart.Properties));
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::DocDB::DBClusterParameterGroup', {
        Parameters: {
            audit_logs: 'enabled',
        },
    }, assert_1.ResourcePart.Properties));
});
test('repository warns if databaseAuditLogging defined and database is specified', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceProps: {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            filesystem: core_1.RemovalPolicy.DESTROY,
        },
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        databaseAuditLogging: true,
    });
    const warningMsg = 'The parameter databaseAuditLogging only has an effect when the Repository is creating its own database.\n' +
        'Please ensure that the Database provided is configured correctly.';
    // THEN
    expect(repo.node.metadata).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: warningMsg,
        }),
    ]));
});
test('honors subnet specification', () => {
    // GIVEN
    const dependencyStack = new core_1.Stack(app, 'DepStack');
    const dependencyVpc = new aws_ec2_1.Vpc(dependencyStack, 'DepVpc');
    const subnets = [
        aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet1', {
            subnetId: 'SubnetID1',
            availabilityZone: 'us-west-2a',
        }),
        aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet2', {
            subnetId: 'SubnetID2',
            availabilityZone: 'us-west-2b',
        }),
    ];
    const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
    // WHEN
    new lib_1.Repository(isolatedStack, 'repositoryInstaller', {
        vpc: dependencyVpc,
        version,
        vpcSubnets: {
            subnets,
        },
    });
    // THEN
    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::DocDB::DBSubnetGroup', {
        SubnetIds: [
            'SubnetID1',
            'SubnetID2',
        ],
    }));
    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EFS::MountTarget', { SubnetId: 'SubnetID1' }));
    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EFS::MountTarget', { SubnetId: 'SubnetID2' }));
});
test('repository honors database instance count', () => {
    // GIVEN
    const instanceCount = 2;
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        documentDbInstanceCount: instanceCount,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::DocDB::DBInstance', instanceCount, {
        AutoMinorVersionUpgrade: true,
    }));
});
test('repository honors database retention period', () => {
    // GIVEN
    const period = 20;
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        backupOptions: {
            databaseRetention: core_1.Duration.days(period),
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        BackupRetentionPeriod: period,
    }));
});
test('warns if both retention period and database provided', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceProps: {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
        backupOptions: {
            databaseRetention: core_1.Duration.days(20),
        },
    });
    // THEN
    expect(repo.node.metadata).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'Backup retention for database will not be applied since a database is not being created by this construct',
        }),
    ]));
});
test('repository creates filesystem if none provided', () => {
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceProps: {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
        },
        backup: {
            retention: core_1.Duration.days(15),
        },
    });
    if (!fsDatabase.secret) {
        throw new Error('secret cannot be null');
    }
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EFS::FileSystem'));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EFS::MountTarget'));
});
test('default repository instance is created using user defined installation path prefix', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        repositoryInstallationPrefix: 'xyz',
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/\/mnt\/efs\/fs1\/xyz/);
});
test('default repository instance is created using user defined installation path prefix with extra slashes in path', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        repositoryInstallationPrefix: '/xyz//',
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/\/mnt\/efs\/fs1\/xyz/);
});
test('repository instance is created with user defined timeout', () => {
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
        CreationPolicy: {
            AutoScalingCreationPolicy: {
                MinSuccessfulInstancesPercent: 100,
            },
            ResourceSignal: {
                Timeout: 'PT30M',
            },
        },
    }, assert_1.ResourcePart.CompleteDefinition));
});
test('repository instance is created with correct installer path version', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/10\.1\.9\.2/);
});
test.each([
    'test-prefix/',
    '',
])('repository instance is created with correct LogGroup prefix %s', (testPrefix) => {
    // GIVEN
    const id = 'repositoryInstaller';
    // WHEN
    new lib_1.Repository(stack, id, {
        vpc,
        version,
        logGroupProps: {
            logGroupPrefix: testPrefix,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        LogGroupName: testPrefix + id,
    }));
});
test('validate instance self-termination', () => {
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const asgLogicalId = stack.getLogicalId(repo.node.defaultChild.node.defaultChild);
    // THEN
    const expectedString = `function exitTrap(){\nexitCode=$?\nsleep 1m\nINSTANCE=\"$(curl http://169.254.169.254/latest/meta-data/instance-id)\"\nASG=\"$(aws --region \${Token[AWS.Region.4]} ec2 describe-tags --filters \"Name=resource-id,Values=\${INSTANCE}\" \"Name=key,Values=aws:autoscaling:groupName\" --query \"Tags[0].Value\" --output text)\"\naws --region \${Token[AWS.Region.4]} autoscaling update-auto-scaling-group --auto-scaling-group-name \${ASG} --min-size 0 --max-size 0 --desired-capacity 0\n/opt/aws/bin/cfn-signal --stack ${stack.stackName} --resource ${asgLogicalId} --region \${Token[AWS.Region.4]} -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n}`;
    expect(repo.node.defaultChild.userData.render()).toMatch(expectedString);
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Action: 'autoscaling:UpdateAutoScalingGroup',
                Condition: {
                    StringEquals: {
                        'autoscaling:ResourceTag/resourceLogicalId': repo.node.uniqueId,
                    },
                },
                Effect: 'Allow',
                Resource: '*',
            }, {
                Action: 'ec2:DescribeTags',
                Effect: 'Allow',
                Resource: '*',
            }),
        },
    }));
});
test('repository configure client instance', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const instance = new aws_ec2_1.Instance(stack, 'Instance', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    // WHEN
    repo.configureClientInstance({
        host: instance,
        mountPoint: '/mnt/repository',
    });
    const userData = instance.userData.render();
    // THEN
    // white-box testing. If we mount the filesystem, then we've called: setupDirectConnect()
    expect(userData).toMatch(new RegExp(escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/repository rw')));
    // Make sure we added the DB connection args
    expect(userData).toMatch(/.*export -f configure_deadline_database.*/);
    // Make sure we call the configureRepositoryDirectConnect script with appropriate argument.
    const regex = new RegExp(escapeTokenRegex('\'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \\"/mnt/repository/DeadlineRepository\\"'));
    expect(userData).toMatch(regex);
});
test('configureClientInstance uses singleton for repo config script', () => {
    // Note: If this test fails, then check the asset hash for REPO_DC_ASSET -- it may have changed.
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const instance1 = new aws_ec2_1.Instance(stack, 'Instance1', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    const instance2 = new aws_ec2_1.Instance(stack, 'Instance2', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    // WHEN
    repo.configureClientInstance({
        host: instance1,
        mountPoint: '/mnt/repository',
    });
    repo.configureClientInstance({
        host: instance2,
        mountPoint: '/mnt/repository',
    });
    // THEN
    // Make sure that both instances have access to the same Asset for the configureRepositoryDirectConnect script
    assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Policy', 2, {
        PolicyDocument: {
            Statement: [
                {},
                {},
                {
                    Effect: 'Allow',
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Resource: [
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        Ref: asset_constants_1.REPO_DC_ASSET.Bucket,
                                    },
                                ],
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        Ref: asset_constants_1.REPO_DC_ASSET.Bucket,
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                },
            ],
        },
    }));
});
test('windows client cannot direct connect to repository', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const instance = new aws_ec2_1.Instance(stack, 'Instance', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
    });
    // THEN
    expect(() => {
        repo.configureClientInstance({
            host: instance,
            mountPoint: 'd:\\',
        });
    }).toThrowError('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
});
describe('tagging', () => {
    tag_helpers_1.testConstructTags({
        constructName: 'Repository',
        createConstruct: () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            new lib_1.Repository(isolatedStack, 'Repository', {
                vpc,
                version,
            });
            return isolatedStack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 3,
            'AWS::DocDB::DBClusterParameterGroup': 1,
            'AWS::DocDB::DBSubnetGroup': 1,
            'AWS::SecretsManager::Secret': 1,
            'AWS::DocDB::DBCluster': 1,
            'AWS::DocDB::DBInstance': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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