"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionQuery = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
/**
 * The abstract class for new or imported(custom) Deadline Version.
 */
class VersionQueryBase extends core_1.Construct {
}
/**
 * This class encapsulates information about a particular version of Thinkbox's Deadline software.
 *
 * Information such as the version number, and where to get installers for that version from Amazon S3.
 *
 * The version of an official release of Deadline is always four numeric version components separated by dots.
 * ex: 10.1.8.5. We refer to the components in this version, in order from left-to-right, as the
 * major, minor, release, and patch versions. For example, Deadline version 10.1.8.5 is majorVersion 10, minorVersion 1,
 * releaseVersion 8, and patchVersion 5.
 *
 * All of the installers provided by an instance of this class must be for the same Deadline release (ex: 10.1.8),
 * but the patch versions may differ between operating systems depending on the particulars of that release of Deadline.
 * This class provides a simple way to query a version of Deadline prior to or during deployment of a
 * CDK app.
 *
 * You pass an instance of this class to various Deadline constructs in this library to tell those
 * constructs which version of Deadline you want them to use, and be configured for.
 *
 * @stability stable
 */
class VersionQuery extends VersionQueryBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const lambdaCode = aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs'));
        const lambdaFunc = new aws_lambda_1.SingletonFunction(this, 'VersionProviderFunction', {
            uuid: '2e19e243-16ee-4d1a-a3c9-18d35eddd446',
            description: 'Used by the Version construct to get installer locations for a specific Deadline version.',
            code: lambdaCode,
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'version-provider.handler',
            timeout: core_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        const deadlineProperties = {
            versionString: props === null || props === void 0 ? void 0 : props.version,
            // If we don't have a full static version string, create a random string that will force the Lambda to always
            // run on redeploys, effectively checking for version updates.
            forceRun: this.forceRun(props === null || props === void 0 ? void 0 : props.version),
        };
        const deadlineResource = new core_1.CustomResource(this, 'DeadlineResource', {
            serviceToken: lambdaFunc.functionArn,
            properties: deadlineProperties,
            resourceType: 'Custom::RFDK_DEADLINE_INSTALLERS',
        });
        this.majorVersion = core_1.Token.asNumber(deadlineResource.getAtt('MajorVersion'));
        this.minorVersion = core_1.Token.asNumber(deadlineResource.getAtt('MinorVersion'));
        this.releaseVersion = core_1.Token.asNumber(deadlineResource.getAtt('ReleaseVersion'));
        this.linuxInstallers = {
            patchVersion: core_1.Token.asNumber(deadlineResource.getAtt('LinuxPatchVersion')),
            repository: {
                objectKey: core_1.Token.asString(deadlineResource.getAtt('LinuxRepositoryInstaller')),
                s3Bucket: aws_s3_1.Bucket.fromBucketName(scope, 'InstallerBucket', core_1.Token.asString(deadlineResource.getAtt('S3Bucket'))),
            },
        };
    }
    /**
     * Construct the full version string for the linux patch release referenced in this version object.
     *
     * This is constructed by joining the major, minor,
     * release, and patch versions by dots.
     *
     * @stability stable
     */
    linuxFullVersionString() {
        const major = core_1.Token.isUnresolved(this.majorVersion) ? core_1.Token.asString(this.majorVersion) : this.majorVersion.toString();
        const minor = core_1.Token.isUnresolved(this.minorVersion) ? core_1.Token.asString(this.minorVersion) : this.minorVersion.toString();
        const release = core_1.Token.isUnresolved(this.releaseVersion) ? core_1.Token.asString(this.releaseVersion) : this.releaseVersion.toString();
        const patch = core_1.Token.isUnresolved(this.linuxInstallers.patchVersion)
            ? core_1.Token.asString(this.linuxInstallers.patchVersion)
            : this.linuxInstallers.patchVersion.toString();
        return `${major}.${minor}.${release}.${patch}`;
    }
    /**
     * Check if we have a full version in the supplied version string. If we don't, we want to make sure the Lambda
     * that fetches the full version number and the installers for it is always run. This allows for Deadline updates
     * to be discovered.
     */
    forceRun(version) {
        return !this.isFullVersion(version) ? crypto_1.randomBytes(32).toString('base64').slice(0, 32) : undefined;
    }
    /**
     * Checks if the supplied version contains the major, minor, release, and patch version numbers,
     * and returns true only if all 4 are supplied.
     */
    isFullVersion(version) {
        const components = version === null || version === void 0 ? void 0 : version.split('.').map(x => parseInt(x));
        if (!components || (components === null || components === void 0 ? void 0 : components.length) != 4) {
            return false;
        }
        return true;
    }
}
exports.VersionQuery = VersionQuery;
//# sourceMappingURL=data:application/json;base64,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