"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const fs = require("fs");
const path = require("path");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const _1 = require("./");
/**
 * Class for interacting with the Deadline stage directory.
 *
 * The stage is a directory that conforms to a conventional structure that RFDK
 * requires to deploy Deadline. It should contain a manifest file, the Deadline
 * installers, and any supporting files required for building the Deadline
 * container.
 *
 * Note: Current version of RFDK supports Deadline v10.1.9.2 and later.
 *
 * @stability stable
 */
class Stage {
    /**
     * Constructs a Stage instance.
     *
     * @stability stable
     */
    constructor(props) {
        Stage.validateManifest(props.manifest);
        this.dirPath = props.path;
        this.manifest = props.manifest;
    }
    /**
     * Loads and parses the manifest file from a given path.
     *
     * @param manifestPath The path to the manifest JSON file.
     * @stability stable
     */
    static loadManifest(manifestPath) {
        const content = fs.readFileSync(manifestPath, { encoding: 'utf8' });
        return JSON.parse(content);
    }
    /**
     * Returns a {@link Stage} loaded using the specified directory as the Docker build context and loads and uses the manifest named `manifest.json` in the directory.
     *
     * @param stagePath The path to the Deadline stage directory.
     * @stability stable
     */
    static fromDirectory(stagePath) {
        const manifest = Stage.loadManifest(path.join(stagePath, Stage.MANIFEST_REL_PATH));
        return new Stage({
            manifest,
            path: stagePath,
        });
    }
    /**
     * Ensures that the manifest is a valid manifest object schema.
     *
     * If the object does not fit the schema, then an Error is thrown.
     *
     * @param rawManifest The parsed manifest object
     */
    static validateManifest(rawManifest) {
        if (rawManifest !== undefined && typeof rawManifest !== 'object') {
            throw new TypeError(`Expected object but got ${typeof rawManifest}`);
        }
        const schema = rawManifest.schema;
        if (schema === undefined) {
            throw new Error('Manifest contains no "schema" key');
        }
        else if (typeof schema !== 'number') {
            throw new TypeError(`Expected a numeric "schema" but got: ${typeof schema}`);
        }
        const version = rawManifest.version;
        if (version === undefined) {
            throw new Error('Manifest contains no "version" key');
        }
        else if (typeof version !== 'string') {
            throw new TypeError(`Expected a string "version" but got: ${typeof version}`);
        }
        // Do minimum supported deadline version check
        const stagedVersion = _1.Version.parse(version);
        if (stagedVersion.isLessThan(_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION)) {
            throw new TypeError(`Staged Deadline Version (${version}) is less than the minimum supported version (${_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()})`);
        }
        return true;
    }
    /**
     * Creates a {@link Version} based on the manifest version.
     *
     * @param scope The parent scope.
     * @param id The construct ID.
     * @stability stable
     */
    getVersion(scope, id) {
        const releaseVersion = this.getReleaseVersion(this.manifest.version);
        return new _1.VersionQuery(scope, id, { version: releaseVersion });
    }
    /**
     * Construct a {@link DockerImageAsset} instance from a recipe in the Stage.
     *
     * @param scope The scope for the {@link DockerImageAsset}.
     * @param id The construct ID of the {@link DockerImageAsset}.
     * @param recipeName The name of the recipe.
     * @stability stable
     */
    imageFromRecipe(scope, id, recipeName) {
        const recipe = this.manifest.recipes[recipeName];
        if (!recipe) {
            throw new Error(`No such recipe: ${recipeName}`);
        }
        return new aws_ecr_assets_1.DockerImageAsset(scope, id, {
            directory: this.dirPath,
            ...recipe,
        });
    }
    /**
     * This removes the patch version from a full version string. No validation is done as that is handled
     * in the constructor with the version check.
     */
    getReleaseVersion(fullVersion) {
        const versionComponents = fullVersion.split('.');
        return `${versionComponents[0]}.${versionComponents[1]}.${versionComponents[2]}`;
    }
}
exports.Stage = Stage;
/**
 * The path to the manifest JSON file relative to the {@link Stage}'s directory'.
 */
Stage.MANIFEST_REL_PATH = 'manifest.json';
//# sourceMappingURL=data:application/json;base64,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