"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountableEfs = void 0;
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const mount_permissions_helper_1 = require("./mount-permissions-helper");
/**
 * This class encapsulates scripting that can be used to mount an Amazon Elastic File System onto an instance.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 * @remark The default access point is writeable only by the root user.
 * @todo Add support for specifying an AccessPoint for the EFS filesystem to  enforce user and group information for all file system requests.
 */
class MountableEfs {
    /**
     * @stability stable
     */
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
    }
    /**
     * Mount the filesystem to the given instance at instance startup.
     *
     * This is accomplished by
     * adding scripting to the UserData of the instance to mount the filesystem on startup.
     * If required, the instance's security group is granted ingress to the filesystem's security
     * group on the required ports.
     *
     * @stability stable
     * @inheritdoc true
     */
    mountToLinuxInstance(target, mount) {
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        target.connections.allowTo(this.props.filesystem, this.props.filesystem.connections.defaultPort);
        const mountScriptAsset = this.mountAssetSingleton();
        mountScriptAsset.grantRead(target.grantPrincipal);
        const mountScript = target.userData.addS3DownloadCommand({
            bucket: mountScriptAsset.bucket,
            bucketKey: mountScriptAsset.s3ObjectKey,
        });
        const mountDir = path.posix.normalize(mount.location);
        const mountOptions = [mount_permissions_helper_1.MountPermissionsHelper.toLinuxMountOption(mount.permissions)];
        if (this.props.extraMountOptions) {
            mountOptions.push(...this.props.extraMountOptions);
        }
        const mountOptionsStr = mountOptions.join(',');
        target.userData.addCommands('TMPDIR=$(mktemp -d)', 'pushd "$TMPDIR"', `unzip ${mountScript}`, `bash ./mountEfs.sh ${this.props.filesystem.fileSystemId} ${mountDir} ${mountOptionsStr}`, 'popd', `rm -f ${mountScript}`);
    }
    /**
     * Fetch the Asset singleton for the EFS mounting scripts, or generate it if needed.
     *
     * @stability stable
     */
    mountAssetSingleton() {
        var _a;
        const stack = core_1.Stack.of(this.scope);
        const uuid = '2b31c419-5b0b-4bb8-99ad-5b2575b2c06b';
        const uniqueId = 'MountableEfsAsset' + uuid.replace(/[-]/g, '');
        return (_a = stack.node.tryFindChild(uniqueId)) !== null && _a !== void 0 ? _a : new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'bash'),
            exclude: ['**/*', '!mountEfs.sh', '!metadataUtilities.sh', '!ec2-certificates.crt'],
        });
    }
}
exports.MountableEfs = MountableEfs;
//# sourceMappingURL=data:application/json;base64,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