"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogGroupFactory = void 0;
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const exporting_log_group_1 = require("./exporting-log-group");
/**
 * This factory will return an ILogGroup based on the configuration provided to it.
 *
 * The LogGroup will either be
 * wrapped in a LogRetention from the aws-lambda package that has the ability to look up and reuse an existing LogGroup
 * or an ExportingLogGroup that uses a LogRetention and adds additional functionality to export the logs to S3.
 *
 * @stability stable
 */
class LogGroupFactory {
    /**
     * Either create a new LogGroup given the LogGroup name, or return the existing LogGroup.
     *
     * @stability stable
     */
    static createOrFetch(scope, logWrapperId, logGroupName, props) {
        const fullLogGroupName = (props === null || props === void 0 ? void 0 : props.logGroupPrefix) ? `${props.logGroupPrefix}${logGroupName}` : logGroupName;
        const retention = (props === null || props === void 0 ? void 0 : props.retention) ? props.retention : LogGroupFactory.DEFAULT_LOG_RETENTION;
        return (props === null || props === void 0 ? void 0 : props.bucketName) ? new exporting_log_group_1.ExportingLogGroup(scope, logWrapperId, {
            bucketName: props.bucketName,
            logGroupName: fullLogGroupName,
            retention: props.retention,
        }).logGroup
            : aws_logs_1.LogGroup.fromLogGroupArn(scope, `${logGroupName}LogGroup`, new aws_lambda_1.LogRetention(scope, logWrapperId, {
                logGroupName: fullLogGroupName,
                retention,
            }).logGroupArn);
    }
}
exports.LogGroupFactory = LogGroupFactory;
/**
 * Default retention period to hold logs in CloudWatch for.
 */
LogGroupFactory.DEFAULT_LOG_RETENTION = aws_logs_1.RetentionDays.THREE_DAYS;
//# sourceMappingURL=data:application/json;base64,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