"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExportingLogGroup = void 0;
const path = require("path");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
/**
 * This construct takes the name of a CloudWatch LogGroup and will either create it if it doesn't already exist, or reuse the existing one.
 *
 * It also creates a regularly scheduled lambda that will export LogEvents to S3
 * before they expire in CloudWatch.
 *
 * It's used for cost-reduction, as it is more economical to archive logs in S3 than CloudWatch when
 * retaining them for more than a week.
 * Note, that it isn't economical to export logs to S3 if you plan on storing them for less than
 * 7 days total (CloudWatch and S3 combined).
 *
 * Resources Deployed
 * ------------------------
 * - The Lambda SingletonFunction that checks for the existence of the LogGroup.
 * - The CloudWatch LogGroup (if it didn't exist already).
 * - The CloudWatch Alarm watching log exportation failures.
 * - The CloudWatch Event Rule to schedule log exportation.
 * - The Lambda SingletonFunction, with role, to export log groups to S3 by schedule.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 *
 * @stability stable
 */
class ExportingLogGroup extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * UUID needed to identify the SingletonFunction for the log exporter.
         */
        this.LOG_EXPORTER_UUID = '6382448c-e4b2-42e9-b14f-a0a9ccdb198e';
        /**
         * Duration of time between export task Lambda runs.
         */
        this.EXPORT_TASK_FREQUENCY = core_1.Duration.hours(1);
        /**
         * Default value for the number of days to retain logs in CloudWatch for.
         */
        this.RETENTION_DEFAULT = aws_logs_1.RetentionDays.THREE_DAYS;
        const retentionInDays = props.retention ? props.retention : this.RETENTION_DEFAULT;
        const exportLogsFunction = this.setupLogsExporter();
        this.exportErrorAlarm = exportLogsFunction.metricErrors().createAlarm(this, 'LogExporterFailureAlarm', {
            evaluationPeriods: 1,
            threshold: 1,
        });
        const logRetention = new aws_lambda_1.LogRetention(this, 'LogRetention', {
            logGroupName: props.logGroupName,
            retention: retentionInDays,
        });
        this.logGroup = aws_logs_1.LogGroup.fromLogGroupArn(scope, `${props.logGroupName}LogGroup`, logRetention.logGroupArn);
        this.logGroup.grant(exportLogsFunction, 'logs:CreateExportTask');
        const scheduledLogExportation = new aws_events_1.Rule(this, 'LogExporterRule', {
            schedule: aws_events_1.Schedule.rate(this.EXPORT_TASK_FREQUENCY),
        });
        scheduledLogExportation.addTarget(new aws_events_targets_1.LambdaFunction(exportLogsFunction, {
            event: aws_events_1.RuleTargetInput.fromObject({
                BucketName: props.bucketName,
                ExportFrequencyInHours: this.EXPORT_TASK_FREQUENCY.toHours(),
                LogGroupName: props.logGroupName,
                RetentionInHours: retentionInDays.valueOf() * 24,
            }),
        }));
    }
    setupLogsExporter() {
        const exportLogsFunction = new aws_lambda_1.SingletonFunction(this, 'LogExporterFunction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs', 'export-logs')),
            handler: 'index.handler',
            lambdaPurpose: 'LogGroupExporter',
            logRetention: aws_logs_1.RetentionDays.ONE_DAY,
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            uuid: this.LOG_EXPORTER_UUID,
        });
        exportLogsFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['logs:DescribeExportTasks'],
            effect: aws_iam_1.Effect.ALLOW,
            resources: ['*'],
        }));
        return exportLogsFunction;
    }
}
exports.ExportingLogGroup = ExportingLogGroup;
//# sourceMappingURL=data:application/json;base64,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