"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Instance = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const aspects_1 = require("./aspects");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const ebs_util_1 = require("./private/ebs-util");
const security_group_1 = require("./security-group");
const vpc_1 = require("./vpc");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * This represents a single EC2 instance.
 *
 * @stability stable
 */
class Instance extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        this.securityGroups = [];
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_InstanceProps(props);
        if (props.initOptions && !props.init) {
            throw new Error('Setting \'initOptions\' requires that \'init\' is also set');
        }
        if (props.securityGroup) {
            this.securityGroup = props.securityGroup;
        }
        else {
            this.securityGroup = new security_group_1.SecurityGroup(this, 'InstanceSecurityGroup', {
                vpc: props.vpc,
                allowAllOutbound: props.allowAllOutbound !== false,
            });
        }
        this.connections = new connections_1.Connections({ securityGroups: [this.securityGroup] });
        this.securityGroups.push(this.securityGroup);
        core_1.Tags.of(this).add(NAME_TAG, props.instanceName || this.node.path);
        this.role = props.role || new iam.Role(this, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.role.roleName],
        });
        // use delayed evaluation
        const imageConfig = props.machineImage.getImage(this);
        this.userData = (_b = props.userData) !== null && _b !== void 0 ? _b : imageConfig.userData;
        const userDataToken = core_1.Lazy.string({ produce: () => core_1.Fn.base64(this.userData.render()) });
        const securityGroupsToken = core_1.Lazy.list({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        let subnet;
        if (props.availabilityZone) {
            const selected = subnets.filter(sn => sn.availabilityZone === props.availabilityZone);
            if (selected.length === 1) {
                subnet = selected[0];
            }
            else {
                core_1.Annotations.of(this).addError(`Need exactly 1 subnet to match AZ '${props.availabilityZone}', found ${selected.length}. Use a different availabilityZone.`);
            }
        }
        else {
            if (subnets.length > 0) {
                subnet = subnets[0];
            }
            else {
                core_1.Annotations.of(this).addError(`Did not find any subnets matching '${JSON.stringify(props.vpcSubnets)}', please use a different selection.`);
            }
        }
        if (!subnet) {
            // We got here and we don't have a subnet because of validation errors.
            // Invent one on the spot so the code below doesn't fail.
            subnet = vpc_1.Subnet.fromSubnetAttributes(this, 'DummySubnet', {
                subnetId: 's-notfound',
                availabilityZone: 'az-notfound',
            });
        }
        this.instance = new ec2_generated_1.CfnInstance(this, 'Resource', {
            imageId: imageConfig.imageId,
            keyName: props.keyName,
            instanceType: props.instanceType.toString(),
            securityGroupIds: securityGroupsToken,
            iamInstanceProfile: iamProfile.ref,
            userData: userDataToken,
            subnetId: subnet.subnetId,
            availabilityZone: subnet.availabilityZone,
            sourceDestCheck: props.sourceDestCheck,
            blockDeviceMappings: props.blockDevices !== undefined ? ebs_util_1.instanceBlockDeviceMappings(this, props.blockDevices) : undefined,
            privateIpAddress: props.privateIpAddress,
            propagateTagsToVolumeOnCreation: props.propagateTagsToVolumeOnCreation,
        });
        this.instance.node.addDependency(this.role);
        this.osType = imageConfig.osType;
        this.node.defaultChild = this.instance;
        this.instanceId = this.instance.ref;
        this.instanceAvailabilityZone = this.instance.attrAvailabilityZone;
        this.instancePrivateDnsName = this.instance.attrPrivateDnsName;
        this.instancePrivateIp = this.instance.attrPrivateIp;
        this.instancePublicDnsName = this.instance.attrPublicDnsName;
        this.instancePublicIp = this.instance.attrPublicIp;
        if (props.init) {
            this.applyCloudFormationInit(props.init, props.initOptions);
        }
        this.applyUpdatePolicies(props);
        // Trigger replacement (via new logical ID) on user data change, if specified or cfn-init is being used.
        //
        // This is slightly tricky -- we need to resolve the UserData string (in order to get at actual Asset hashes,
        // instead of the Token stringifications of them ('${Token[1234]}'). However, in the case of CFN Init usage,
        // a UserData is going to contain the logicalID of the resource itself, which means infinite recursion if we
        // try to naively resolve. We need a recursion breaker in this.
        const originalLogicalId = core_1.Stack.of(this).getLogicalId(this.instance);
        let recursing = false;
        this.instance.overrideLogicalId(core_1.Lazy.uncachedString({
            produce: (context) => {
                var _b;
                if (recursing) {
                    return originalLogicalId;
                }
                if (!((_b = props.userDataCausesReplacement) !== null && _b !== void 0 ? _b : props.initOptions)) {
                    return originalLogicalId;
                }
                const md5 = crypto.createHash('md5');
                recursing = true;
                try {
                    md5.update(JSON.stringify(context.resolve(this.userData.render())));
                }
                finally {
                    recursing = false;
                }
                const digest = md5.digest('hex').substr(0, 16);
                return `${originalLogicalId}${digest}`;
            },
        }));
        if (props.requireImdsv2) {
            core_1.Aspects.of(this).add(new aspects_1.InstanceRequireImdsv2Aspect());
        }
    }
    /**
     * Add the security group to the instance.
     *
     * @param securityGroup : The security group to add.
     * @stability stable
     */
    addSecurityGroup(securityGroup) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_ISecurityGroup(securityGroup);
        this.securityGroups.push(securityGroup);
    }
    /**
     * Add command to the startup script of the instance.
     *
     * The command must be in the scripting language supported by the instance's OS (i.e. Linux/Windows).
     *
     * @stability stable
     */
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     *
     * @stability stable
     */
    addToRolePolicy(statement) {
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Use a CloudFormation Init configuration at instance startup
     *
     * This does the following:
     *
     * - Attaches the CloudFormation Init metadata to the Instance resource.
     * - Add commands to the instance UserData to run `cfn-init` and `cfn-signal`.
     * - Update the instance's CreationPolicy to wait for the `cfn-signal` commands.
     */
    applyCloudFormationInit(init, options = {}) {
        var _b;
        init.attach(this.instance, {
            platform: this.osType,
            instanceRole: this.role,
            userData: this.userData,
            configSets: options.configSets,
            embedFingerprint: options.embedFingerprint,
            printLog: options.printLog,
            ignoreFailures: options.ignoreFailures,
            includeRole: options.includeRole,
            includeUrl: options.includeUrl,
        });
        this.waitForResourceSignal((_b = options.timeout) !== null && _b !== void 0 ? _b : core_1.Duration.minutes(5));
    }
    /**
     * Wait for a single additional resource signal
     *
     * Add 1 to the current ResourceSignal Count and add the given timeout to the current timeout.
     *
     * Use this to pause the CloudFormation deployment to wait for the instances
     * in the AutoScalingGroup to report successful startup during
     * creation and updates. The UserData script needs to invoke `cfn-signal`
     * with a success or failure code after it is done setting up the instance.
     */
    waitForResourceSignal(timeout) {
        var _b, _c;
        const oldResourceSignal = (_b = this.instance.cfnOptions.creationPolicy) === null || _b === void 0 ? void 0 : _b.resourceSignal;
        this.instance.cfnOptions.creationPolicy = {
            ...this.instance.cfnOptions.creationPolicy,
            resourceSignal: {
                count: ((_c = oldResourceSignal === null || oldResourceSignal === void 0 ? void 0 : oldResourceSignal.count) !== null && _c !== void 0 ? _c : 0) + 1,
                timeout: ((oldResourceSignal === null || oldResourceSignal === void 0 ? void 0 : oldResourceSignal.timeout) ? core_1.Duration.parse(oldResourceSignal === null || oldResourceSignal === void 0 ? void 0 : oldResourceSignal.timeout).plus(timeout) : timeout).toIsoString(),
            },
        };
    }
    /**
     * Apply CloudFormation update policies for the instance
     */
    applyUpdatePolicies(props) {
        if (props.resourceSignalTimeout !== undefined) {
            this.instance.cfnOptions.creationPolicy = {
                ...this.instance.cfnOptions.creationPolicy,
                resourceSignal: {
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toIsoString(),
                },
            };
        }
    }
}
exports.Instance = Instance;
_a = JSII_RTTI_SYMBOL_1;
Instance[_a] = { fqn: "@aws-cdk/aws-ec2.Instance", version: "1.135.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5zdGFuY2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbnN0YW5jZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSxpQ0FBaUM7QUFDakMsd0NBQXdDO0FBRXhDLHdDQUEyRztBQUUzRyx1Q0FBd0Q7QUFFeEQsK0NBQTBEO0FBQzFELG1EQUE4QztBQUc5QyxpREFBaUU7QUFDakUscURBQWlFO0FBR2pFLCtCQUFzRDtBQUV0RDs7R0FFRztBQUNILE1BQU0sUUFBUSxHQUFXLE1BQU0sQ0FBQzs7Ozs7O0FBdUZoQyxNQUFhLFFBQVMsU0FBUSxlQUFROzs7O0lBbUNwQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQW9COztRQUM1RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBSEYsbUJBQWMsR0FBcUIsRUFBRSxDQUFDOztRQUtyRCxJQUFJLEtBQUssQ0FBQyxXQUFXLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFO1lBQ3BDLE1BQU0sSUFBSSxLQUFLLENBQUMsNERBQTRELENBQUMsQ0FBQztTQUMvRTtRQUVELElBQUksS0FBSyxDQUFDLGFBQWEsRUFBRTtZQUN2QixJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUM7U0FDMUM7YUFBTTtZQUNMLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSw4QkFBYSxDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRTtnQkFDcEUsR0FBRyxFQUFFLEtBQUssQ0FBQyxHQUFHO2dCQUNkLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsS0FBSyxLQUFLO2FBQ25ELENBQUMsQ0FBQztTQUNKO1FBQ0QsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLHlCQUFXLENBQUMsRUFBRSxjQUFjLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQzdFLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUM3QyxXQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLFlBQVksSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRWxFLElBQUksQ0FBQyxJQUFJLEdBQUcsS0FBSyxDQUFDLElBQUksSUFBSSxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUMzRCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUM7U0FDekQsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDO1FBRWhDLE1BQU0sVUFBVSxHQUFHLElBQUksR0FBRyxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSxpQkFBaUIsRUFBRTtZQUNyRSxLQUFLLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztTQUM1QixDQUFDLENBQUM7UUFFSCx5QkFBeUI7UUFDekIsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDdEQsSUFBSSxDQUFDLFFBQVEsU0FBRyxLQUFLLENBQUMsUUFBUSxtQ0FBSSxXQUFXLENBQUMsUUFBUSxDQUFDO1FBQ3ZELE1BQU0sYUFBYSxHQUFHLFdBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsU0FBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQ3hGLE1BQU0sbUJBQW1CLEdBQUcsV0FBSSxDQUFDLElBQUksQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsRUFBRSxDQUFDLENBQUM7UUFFNUcsTUFBTSxFQUFFLE9BQU8sRUFBRSxHQUFHLEtBQUssQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUM5RCxJQUFJLE1BQU0sQ0FBQztRQUNYLElBQUksS0FBSyxDQUFDLGdCQUFnQixFQUFFO1lBQzFCLE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLEtBQUssS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUM7WUFDdEYsSUFBSSxRQUFRLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtnQkFDekIsTUFBTSxHQUFHLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQzthQUN0QjtpQkFBTTtnQkFDTCxrQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQUMsc0NBQXNDLEtBQUssQ0FBQyxnQkFBZ0IsWUFBWSxRQUFRLENBQUMsTUFBTSxxQ0FBcUMsQ0FBQyxDQUFDO2FBQzdKO1NBQ0Y7YUFBTTtZQUNMLElBQUksT0FBTyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7Z0JBQ3RCLE1BQU0sR0FBRyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7YUFDckI7aUJBQU07Z0JBQ0wsa0JBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUFDLHNDQUFzQyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsc0NBQXNDLENBQUMsQ0FBQzthQUM3STtTQUNGO1FBQ0QsSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUNYLHVFQUF1RTtZQUN2RSx5REFBeUQ7WUFDekQsTUFBTSxHQUFHLFlBQU0sQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO2dCQUN4RCxRQUFRLEVBQUUsWUFBWTtnQkFDdEIsZ0JBQWdCLEVBQUUsYUFBYTthQUNoQyxDQUFDLENBQUM7U0FDSjtRQUVELElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSwyQkFBVyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDaEQsT0FBTyxFQUFFLFdBQVcsQ0FBQyxPQUFPO1lBQzVCLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztZQUN0QixZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUU7WUFDM0MsZ0JBQWdCLEVBQUUsbUJBQW1CO1lBQ3JDLGtCQUFrQixFQUFFLFVBQVUsQ0FBQyxHQUFHO1lBQ2xDLFFBQVEsRUFBRSxhQUFhO1lBQ3ZCLFFBQVEsRUFBRSxNQUFNLENBQUMsUUFBUTtZQUN6QixnQkFBZ0IsRUFBRSxNQUFNLENBQUMsZ0JBQWdCO1lBQ3pDLGVBQWUsRUFBRSxLQUFLLENBQUMsZUFBZTtZQUN0QyxtQkFBbUIsRUFBRSxLQUFLLENBQUMsWUFBWSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsc0NBQTJCLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUN6SCxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCO1lBQ3hDLCtCQUErQixFQUFFLEtBQUssQ0FBQywrQkFBK0I7U0FDdkUsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUU1QyxJQUFJLENBQUMsTUFBTSxHQUFHLFdBQVcsQ0FBQyxNQUFNLENBQUM7UUFDakMsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQztRQUV2QyxJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDO1FBQ3BDLElBQUksQ0FBQyx3QkFBd0IsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLG9CQUFvQixDQUFDO1FBQ25FLElBQUksQ0FBQyxzQkFBc0IsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDO1FBQy9ELElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLGFBQWEsQ0FBQztRQUNyRCxJQUFJLENBQUMscUJBQXFCLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQztRQUM3RCxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUM7UUFFbkQsSUFBSSxLQUFLLENBQUMsSUFBSSxFQUFFO1lBQ2QsSUFBSSxDQUFDLHVCQUF1QixDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1NBQzdEO1FBRUQsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRWhDLHdHQUF3RztRQUN4RyxFQUFFO1FBQ0YsNkdBQTZHO1FBQzdHLDRHQUE0RztRQUM1Ryw0R0FBNEc7UUFDNUcsK0RBQStEO1FBQy9ELE1BQU0saUJBQWlCLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ3JFLElBQUksU0FBUyxHQUFHLEtBQUssQ0FBQztRQUN0QixJQUFJLENBQUMsUUFBUSxDQUFDLGlCQUFpQixDQUFDLFdBQUksQ0FBQyxjQUFjLENBQUM7WUFDbEQsT0FBTyxFQUFFLENBQUMsT0FBTyxFQUFFLEVBQUU7O2dCQUNuQixJQUFJLFNBQVMsRUFBRTtvQkFBRSxPQUFPLGlCQUFpQixDQUFDO2lCQUFFO2dCQUM1QyxJQUFJLENBQUMsT0FBQyxLQUFLLENBQUMseUJBQXlCLG1DQUFJLEtBQUssQ0FBQyxXQUFXLENBQUMsRUFBRTtvQkFBRSxPQUFPLGlCQUFpQixDQUFDO2lCQUFFO2dCQUUxRixNQUFNLEdBQUcsR0FBRyxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUNyQyxTQUFTLEdBQUcsSUFBSSxDQUFDO2dCQUNqQixJQUFJO29CQUNGLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7aUJBQ3JFO3dCQUFTO29CQUNSLFNBQVMsR0FBRyxLQUFLLENBQUM7aUJBQ25CO2dCQUNELE1BQU0sTUFBTSxHQUFHLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztnQkFDL0MsT0FBTyxHQUFHLGlCQUFpQixHQUFHLE1BQU0sRUFBRSxDQUFDO1lBQ3pDLENBQUM7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksS0FBSyxDQUFDLGFBQWEsRUFBRTtZQUN2QixjQUFPLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLHFDQUEyQixFQUFFLENBQUMsQ0FBQztTQUN6RDtLQUNGOzs7Ozs7O0lBR00sZ0JBQWdCLENBQUMsYUFBNkI7O1FBQ25ELElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0tBQ3pDOzs7Ozs7OztJQUdNLFdBQVcsQ0FBQyxHQUFHLFFBQWtCO1FBQ3RDLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLEdBQUcsUUFBUSxDQUFDLENBQUM7S0FDeEM7Ozs7OztJQUdNLGVBQWUsQ0FBQyxTQUE4QjtRQUNuRCxJQUFJLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFNBQVMsQ0FBQyxDQUFDO0tBQzNDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSyx1QkFBdUIsQ0FBQyxJQUF3QixFQUFFLFVBQTBDLEVBQUU7O1FBQ3BHLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUN6QixRQUFRLEVBQUUsSUFBSSxDQUFDLE1BQU07WUFDckIsWUFBWSxFQUFFLElBQUksQ0FBQyxJQUFJO1lBQ3ZCLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUTtZQUN2QixVQUFVLEVBQUUsT0FBTyxDQUFDLFVBQVU7WUFDOUIsZ0JBQWdCLEVBQUUsT0FBTyxDQUFDLGdCQUFnQjtZQUMxQyxRQUFRLEVBQUUsT0FBTyxDQUFDLFFBQVE7WUFDMUIsY0FBYyxFQUFFLE9BQU8sQ0FBQyxjQUFjO1lBQ3RDLFdBQVcsRUFBRSxPQUFPLENBQUMsV0FBVztZQUNoQyxVQUFVLEVBQUUsT0FBTyxDQUFDLFVBQVU7U0FDL0IsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLHFCQUFxQixPQUFDLE9BQU8sQ0FBQyxPQUFPLG1DQUFJLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztLQUNwRTtJQUVEOzs7Ozs7Ozs7T0FTRztJQUNLLHFCQUFxQixDQUFDLE9BQWlCOztRQUM3QyxNQUFNLGlCQUFpQixTQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLGNBQWMsMENBQUUsY0FBYyxDQUFDO1FBQ2xGLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLGNBQWMsR0FBRztZQUN4QyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLGNBQWM7WUFDMUMsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSxPQUFDLGlCQUFpQixhQUFqQixpQkFBaUIsdUJBQWpCLGlCQUFpQixDQUFFLEtBQUssbUNBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQztnQkFDMUMsT0FBTyxFQUFFLENBQUMsQ0FBQSxpQkFBaUIsYUFBakIsaUJBQWlCLHVCQUFqQixpQkFBaUIsQ0FBRSxPQUFPLEVBQUMsQ0FBQyxDQUFDLGVBQVEsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLGFBQWpCLGlCQUFpQix1QkFBakIsaUJBQWlCLENBQUUsT0FBTyxDQUFDLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxXQUFXLEVBQUU7YUFDekg7U0FDRixDQUFDO0tBQ0g7SUFFRDs7T0FFRztJQUNLLG1CQUFtQixDQUFDLEtBQW9CO1FBQzlDLElBQUksS0FBSyxDQUFDLHFCQUFxQixLQUFLLFNBQVMsRUFBRTtZQUM3QyxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxjQUFjLEdBQUc7Z0JBQ3hDLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsY0FBYztnQkFDMUMsY0FBYyxFQUFFO29CQUNkLE9BQU8sRUFBRSxLQUFLLENBQUMscUJBQXFCLElBQUksS0FBSyxDQUFDLHFCQUFxQixDQUFDLFdBQVcsRUFBRTtpQkFDbEY7YUFDRixDQUFDO1NBQ0g7S0FDRjs7QUFwT0gsNEJBcU9DIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY3J5cHRvIGZyb20gJ2NyeXB0byc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5cbmltcG9ydCB7IEFubm90YXRpb25zLCBBc3BlY3RzLCBEdXJhdGlvbiwgRm4sIElSZXNvdXJjZSwgTGF6eSwgUmVzb3VyY2UsIFN0YWNrLCBUYWdzIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IEluc3RhbmNlUmVxdWlyZUltZHN2MkFzcGVjdCB9IGZyb20gJy4vYXNwZWN0cyc7XG5pbXBvcnQgeyBDbG91ZEZvcm1hdGlvbkluaXQgfSBmcm9tICcuL2Nmbi1pbml0JztcbmltcG9ydCB7IENvbm5lY3Rpb25zLCBJQ29ubmVjdGFibGUgfSBmcm9tICcuL2Nvbm5lY3Rpb25zJztcbmltcG9ydCB7IENmbkluc3RhbmNlIH0gZnJvbSAnLi9lYzIuZ2VuZXJhdGVkJztcbmltcG9ydCB7IEluc3RhbmNlVHlwZSB9IGZyb20gJy4vaW5zdGFuY2UtdHlwZXMnO1xuaW1wb3J0IHsgSU1hY2hpbmVJbWFnZSwgT3BlcmF0aW5nU3lzdGVtVHlwZSB9IGZyb20gJy4vbWFjaGluZS1pbWFnZSc7XG5pbXBvcnQgeyBpbnN0YW5jZUJsb2NrRGV2aWNlTWFwcGluZ3MgfSBmcm9tICcuL3ByaXZhdGUvZWJzLXV0aWwnO1xuaW1wb3J0IHsgSVNlY3VyaXR5R3JvdXAsIFNlY3VyaXR5R3JvdXAgfSBmcm9tICcuL3NlY3VyaXR5LWdyb3VwJztcbmltcG9ydCB7IFVzZXJEYXRhIH0gZnJvbSAnLi91c2VyLWRhdGEnO1xuaW1wb3J0IHsgQmxvY2tEZXZpY2UgfSBmcm9tICcuL3ZvbHVtZSc7XG5pbXBvcnQgeyBJVnBjLCBTdWJuZXQsIFN1Ym5ldFNlbGVjdGlvbiB9IGZyb20gJy4vdnBjJztcblxuLyoqXG4gKiBOYW1lIHRhZyBjb25zdGFudFxuICovXG5jb25zdCBOQU1FX1RBRzogc3RyaW5nID0gJ05hbWUnO1xuXG5leHBvcnQgaW50ZXJmYWNlIElJbnN0YW5jZSBleHRlbmRzIElSZXNvdXJjZSwgSUNvbm5lY3RhYmxlLCBpYW0uSUdyYW50YWJsZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGluc3RhbmNlSWQ6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpbnN0YW5jZUF2YWlsYWJpbGl0eVpvbmU6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpbnN0YW5jZVByaXZhdGVEbnNOYW1lOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpbnN0YW5jZVByaXZhdGVJcDogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGluc3RhbmNlUHVibGljRG5zTmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpbnN0YW5jZVB1YmxpY0lwOiBzdHJpbmc7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSW5zdGFuY2VQcm9wcyB7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBrZXlOYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZwY1N1Ym5ldHM/OiBTdWJuZXRTZWxlY3Rpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhdmFpbGFiaWxpdHlab25lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYWxsb3dBbGxPdXRib3VuZD86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByZXNvdXJjZVNpZ25hbFRpbWVvdXQ/OiBEdXJhdGlvbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdnBjOiBJVnBjO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cD86IElTZWN1cml0eUdyb3VwO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGluc3RhbmNlVHlwZTogSW5zdGFuY2VUeXBlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbWFjaGluZUltYWdlOiBJTWFjaGluZUltYWdlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB1c2VyRGF0YT86IFVzZXJEYXRhO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB1c2VyRGF0YUNhdXNlc1JlcGxhY2VtZW50PzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaW5zdGFuY2VOYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNvdXJjZURlc3RDaGVjaz86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYmxvY2tEZXZpY2VzPzogQmxvY2tEZXZpY2VbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcHJpdmF0ZUlwQWRkcmVzcz86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcHJvcGFnYXRlVGFnc1RvVm9sdW1lT25DcmVhdGlvbj86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGluaXQ/OiBDbG91ZEZvcm1hdGlvbkluaXQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpbml0T3B0aW9ucz86IEFwcGx5Q2xvdWRGb3JtYXRpb25Jbml0T3B0aW9ucztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVxdWlyZUltZHN2Mj86IGJvb2xlYW47XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIEluc3RhbmNlIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJSW5zdGFuY2Uge1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgb3NUeXBlOiBPcGVyYXRpbmdTeXN0ZW1UeXBlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBjb25uZWN0aW9uczogQ29ubmVjdGlvbnM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgcm9sZTogaWFtLklSb2xlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBpYW0uSVByaW5jaXBhbDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSB1c2VyRGF0YTogVXNlckRhdGE7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGluc3RhbmNlOiBDZm5JbnN0YW5jZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgaW5zdGFuY2VJZDogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBpbnN0YW5jZUF2YWlsYWJpbGl0eVpvbmU6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgaW5zdGFuY2VQcml2YXRlRG5zTmFtZTogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBpbnN0YW5jZVByaXZhdGVJcDogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBpbnN0YW5jZVB1YmxpY0Ruc05hbWU6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgaW5zdGFuY2VQdWJsaWNJcDogc3RyaW5nO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgc2VjdXJpdHlHcm91cDogSVNlY3VyaXR5R3JvdXA7XG4gIHByaXZhdGUgcmVhZG9ubHkgc2VjdXJpdHlHcm91cHM6IElTZWN1cml0eUdyb3VwW10gPSBbXTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogSW5zdGFuY2VQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBpZiAocHJvcHMuaW5pdE9wdGlvbnMgJiYgIXByb3BzLmluaXQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignU2V0dGluZyBcXCdpbml0T3B0aW9uc1xcJyByZXF1aXJlcyB0aGF0IFxcJ2luaXRcXCcgaXMgYWxzbyBzZXQnKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMuc2VjdXJpdHlHcm91cCkge1xuICAgICAgdGhpcy5zZWN1cml0eUdyb3VwID0gcHJvcHMuc2VjdXJpdHlHcm91cDtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5zZWN1cml0eUdyb3VwID0gbmV3IFNlY3VyaXR5R3JvdXAodGhpcywgJ0luc3RhbmNlU2VjdXJpdHlHcm91cCcsIHtcbiAgICAgICAgdnBjOiBwcm9wcy52cGMsXG4gICAgICAgIGFsbG93QWxsT3V0Ym91bmQ6IHByb3BzLmFsbG93QWxsT3V0Ym91bmQgIT09IGZhbHNlLFxuICAgICAgfSk7XG4gICAgfVxuICAgIHRoaXMuY29ubmVjdGlvbnMgPSBuZXcgQ29ubmVjdGlvbnMoeyBzZWN1cml0eUdyb3VwczogW3RoaXMuc2VjdXJpdHlHcm91cF0gfSk7XG4gICAgdGhpcy5zZWN1cml0eUdyb3Vwcy5wdXNoKHRoaXMuc2VjdXJpdHlHcm91cCk7XG4gICAgVGFncy5vZih0aGlzKS5hZGQoTkFNRV9UQUcsIHByb3BzLmluc3RhbmNlTmFtZSB8fCB0aGlzLm5vZGUucGF0aCk7XG5cbiAgICB0aGlzLnJvbGUgPSBwcm9wcy5yb2xlIHx8IG5ldyBpYW0uUm9sZSh0aGlzLCAnSW5zdGFuY2VSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2VjMi5hbWF6b25hd3MuY29tJyksXG4gICAgfSk7XG4gICAgdGhpcy5ncmFudFByaW5jaXBhbCA9IHRoaXMucm9sZTtcblxuICAgIGNvbnN0IGlhbVByb2ZpbGUgPSBuZXcgaWFtLkNmbkluc3RhbmNlUHJvZmlsZSh0aGlzLCAnSW5zdGFuY2VQcm9maWxlJywge1xuICAgICAgcm9sZXM6IFt0aGlzLnJvbGUucm9sZU5hbWVdLFxuICAgIH0pO1xuXG4gICAgLy8gdXNlIGRlbGF5ZWQgZXZhbHVhdGlvblxuICAgIGNvbnN0IGltYWdlQ29uZmlnID0gcHJvcHMubWFjaGluZUltYWdlLmdldEltYWdlKHRoaXMpO1xuICAgIHRoaXMudXNlckRhdGEgPSBwcm9wcy51c2VyRGF0YSA/PyBpbWFnZUNvbmZpZy51c2VyRGF0YTtcbiAgICBjb25zdCB1c2VyRGF0YVRva2VuID0gTGF6eS5zdHJpbmcoeyBwcm9kdWNlOiAoKSA9PiBGbi5iYXNlNjQodGhpcy51c2VyRGF0YS5yZW5kZXIoKSkgfSk7XG4gICAgY29uc3Qgc2VjdXJpdHlHcm91cHNUb2tlbiA9IExhenkubGlzdCh7IHByb2R1Y2U6ICgpID0+IHRoaXMuc2VjdXJpdHlHcm91cHMubWFwKHNnID0+IHNnLnNlY3VyaXR5R3JvdXBJZCkgfSk7XG5cbiAgICBjb25zdCB7IHN1Ym5ldHMgfSA9IHByb3BzLnZwYy5zZWxlY3RTdWJuZXRzKHByb3BzLnZwY1N1Ym5ldHMpO1xuICAgIGxldCBzdWJuZXQ7XG4gICAgaWYgKHByb3BzLmF2YWlsYWJpbGl0eVpvbmUpIHtcbiAgICAgIGNvbnN0IHNlbGVjdGVkID0gc3VibmV0cy5maWx0ZXIoc24gPT4gc24uYXZhaWxhYmlsaXR5Wm9uZSA9PT0gcHJvcHMuYXZhaWxhYmlsaXR5Wm9uZSk7XG4gICAgICBpZiAoc2VsZWN0ZWQubGVuZ3RoID09PSAxKSB7XG4gICAgICAgIHN1Ym5ldCA9IHNlbGVjdGVkWzBdO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgQW5ub3RhdGlvbnMub2YodGhpcykuYWRkRXJyb3IoYE5lZWQgZXhhY3RseSAxIHN1Ym5ldCB0byBtYXRjaCBBWiAnJHtwcm9wcy5hdmFpbGFiaWxpdHlab25lfScsIGZvdW5kICR7c2VsZWN0ZWQubGVuZ3RofS4gVXNlIGEgZGlmZmVyZW50IGF2YWlsYWJpbGl0eVpvbmUuYCk7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIGlmIChzdWJuZXRzLmxlbmd0aCA+IDApIHtcbiAgICAgICAgc3VibmV0ID0gc3VibmV0c1swXTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIEFubm90YXRpb25zLm9mKHRoaXMpLmFkZEVycm9yKGBEaWQgbm90IGZpbmQgYW55IHN1Ym5ldHMgbWF0Y2hpbmcgJyR7SlNPTi5zdHJpbmdpZnkocHJvcHMudnBjU3VibmV0cyl9JywgcGxlYXNlIHVzZSBhIGRpZmZlcmVudCBzZWxlY3Rpb24uYCk7XG4gICAgICB9XG4gICAgfVxuICAgIGlmICghc3VibmV0KSB7XG4gICAgICAvLyBXZSBnb3QgaGVyZSBhbmQgd2UgZG9uJ3QgaGF2ZSBhIHN1Ym5ldCBiZWNhdXNlIG9mIHZhbGlkYXRpb24gZXJyb3JzLlxuICAgICAgLy8gSW52ZW50IG9uZSBvbiB0aGUgc3BvdCBzbyB0aGUgY29kZSBiZWxvdyBkb2Vzbid0IGZhaWwuXG4gICAgICBzdWJuZXQgPSBTdWJuZXQuZnJvbVN1Ym5ldEF0dHJpYnV0ZXModGhpcywgJ0R1bW15U3VibmV0Jywge1xuICAgICAgICBzdWJuZXRJZDogJ3Mtbm90Zm91bmQnLFxuICAgICAgICBhdmFpbGFiaWxpdHlab25lOiAnYXotbm90Zm91bmQnLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgdGhpcy5pbnN0YW5jZSA9IG5ldyBDZm5JbnN0YW5jZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBpbWFnZUlkOiBpbWFnZUNvbmZpZy5pbWFnZUlkLFxuICAgICAga2V5TmFtZTogcHJvcHMua2V5TmFtZSxcbiAgICAgIGluc3RhbmNlVHlwZTogcHJvcHMuaW5zdGFuY2VUeXBlLnRvU3RyaW5nKCksXG4gICAgICBzZWN1cml0eUdyb3VwSWRzOiBzZWN1cml0eUdyb3Vwc1Rva2VuLFxuICAgICAgaWFtSW5zdGFuY2VQcm9maWxlOiBpYW1Qcm9maWxlLnJlZixcbiAgICAgIHVzZXJEYXRhOiB1c2VyRGF0YVRva2VuLFxuICAgICAgc3VibmV0SWQ6IHN1Ym5ldC5zdWJuZXRJZCxcbiAgICAgIGF2YWlsYWJpbGl0eVpvbmU6IHN1Ym5ldC5hdmFpbGFiaWxpdHlab25lLFxuICAgICAgc291cmNlRGVzdENoZWNrOiBwcm9wcy5zb3VyY2VEZXN0Q2hlY2ssXG4gICAgICBibG9ja0RldmljZU1hcHBpbmdzOiBwcm9wcy5ibG9ja0RldmljZXMgIT09IHVuZGVmaW5lZCA/IGluc3RhbmNlQmxvY2tEZXZpY2VNYXBwaW5ncyh0aGlzLCBwcm9wcy5ibG9ja0RldmljZXMpIDogdW5kZWZpbmVkLFxuICAgICAgcHJpdmF0ZUlwQWRkcmVzczogcHJvcHMucHJpdmF0ZUlwQWRkcmVzcyxcbiAgICAgIHByb3BhZ2F0ZVRhZ3NUb1ZvbHVtZU9uQ3JlYXRpb246IHByb3BzLnByb3BhZ2F0ZVRhZ3NUb1ZvbHVtZU9uQ3JlYXRpb24sXG4gICAgfSk7XG4gICAgdGhpcy5pbnN0YW5jZS5ub2RlLmFkZERlcGVuZGVuY3kodGhpcy5yb2xlKTtcblxuICAgIHRoaXMub3NUeXBlID0gaW1hZ2VDb25maWcub3NUeXBlO1xuICAgIHRoaXMubm9kZS5kZWZhdWx0Q2hpbGQgPSB0aGlzLmluc3RhbmNlO1xuXG4gICAgdGhpcy5pbnN0YW5jZUlkID0gdGhpcy5pbnN0YW5jZS5yZWY7XG4gICAgdGhpcy5pbnN0YW5jZUF2YWlsYWJpbGl0eVpvbmUgPSB0aGlzLmluc3RhbmNlLmF0dHJBdmFpbGFiaWxpdHlab25lO1xuICAgIHRoaXMuaW5zdGFuY2VQcml2YXRlRG5zTmFtZSA9IHRoaXMuaW5zdGFuY2UuYXR0clByaXZhdGVEbnNOYW1lO1xuICAgIHRoaXMuaW5zdGFuY2VQcml2YXRlSXAgPSB0aGlzLmluc3RhbmNlLmF0dHJQcml2YXRlSXA7XG4gICAgdGhpcy5pbnN0YW5jZVB1YmxpY0Ruc05hbWUgPSB0aGlzLmluc3RhbmNlLmF0dHJQdWJsaWNEbnNOYW1lO1xuICAgIHRoaXMuaW5zdGFuY2VQdWJsaWNJcCA9IHRoaXMuaW5zdGFuY2UuYXR0clB1YmxpY0lwO1xuXG4gICAgaWYgKHByb3BzLmluaXQpIHtcbiAgICAgIHRoaXMuYXBwbHlDbG91ZEZvcm1hdGlvbkluaXQocHJvcHMuaW5pdCwgcHJvcHMuaW5pdE9wdGlvbnMpO1xuICAgIH1cblxuICAgIHRoaXMuYXBwbHlVcGRhdGVQb2xpY2llcyhwcm9wcyk7XG5cbiAgICAvLyBUcmlnZ2VyIHJlcGxhY2VtZW50ICh2aWEgbmV3IGxvZ2ljYWwgSUQpIG9uIHVzZXIgZGF0YSBjaGFuZ2UsIGlmIHNwZWNpZmllZCBvciBjZm4taW5pdCBpcyBiZWluZyB1c2VkLlxuICAgIC8vXG4gICAgLy8gVGhpcyBpcyBzbGlnaHRseSB0cmlja3kgLS0gd2UgbmVlZCB0byByZXNvbHZlIHRoZSBVc2VyRGF0YSBzdHJpbmcgKGluIG9yZGVyIHRvIGdldCBhdCBhY3R1YWwgQXNzZXQgaGFzaGVzLFxuICAgIC8vIGluc3RlYWQgb2YgdGhlIFRva2VuIHN0cmluZ2lmaWNhdGlvbnMgb2YgdGhlbSAoJyR7VG9rZW5bMTIzNF19JykuIEhvd2V2ZXIsIGluIHRoZSBjYXNlIG9mIENGTiBJbml0IHVzYWdlLFxuICAgIC8vIGEgVXNlckRhdGEgaXMgZ29pbmcgdG8gY29udGFpbiB0aGUgbG9naWNhbElEIG9mIHRoZSByZXNvdXJjZSBpdHNlbGYsIHdoaWNoIG1lYW5zIGluZmluaXRlIHJlY3Vyc2lvbiBpZiB3ZVxuICAgIC8vIHRyeSB0byBuYWl2ZWx5IHJlc29sdmUuIFdlIG5lZWQgYSByZWN1cnNpb24gYnJlYWtlciBpbiB0aGlzLlxuICAgIGNvbnN0IG9yaWdpbmFsTG9naWNhbElkID0gU3RhY2sub2YodGhpcykuZ2V0TG9naWNhbElkKHRoaXMuaW5zdGFuY2UpO1xuICAgIGxldCByZWN1cnNpbmcgPSBmYWxzZTtcbiAgICB0aGlzLmluc3RhbmNlLm92ZXJyaWRlTG9naWNhbElkKExhenkudW5jYWNoZWRTdHJpbmcoe1xuICAgICAgcHJvZHVjZTogKGNvbnRleHQpID0+IHtcbiAgICAgICAgaWYgKHJlY3Vyc2luZykgeyByZXR1cm4gb3JpZ2luYWxMb2dpY2FsSWQ7IH1cbiAgICAgICAgaWYgKCEocHJvcHMudXNlckRhdGFDYXVzZXNSZXBsYWNlbWVudCA/PyBwcm9wcy5pbml0T3B0aW9ucykpIHsgcmV0dXJuIG9yaWdpbmFsTG9naWNhbElkOyB9XG5cbiAgICAgICAgY29uc3QgbWQ1ID0gY3J5cHRvLmNyZWF0ZUhhc2goJ21kNScpO1xuICAgICAgICByZWN1cnNpbmcgPSB0cnVlO1xuICAgICAgICB0cnkge1xuICAgICAgICAgIG1kNS51cGRhdGUoSlNPTi5zdHJpbmdpZnkoY29udGV4dC5yZXNvbHZlKHRoaXMudXNlckRhdGEucmVuZGVyKCkpKSk7XG4gICAgICAgIH0gZmluYWxseSB7XG4gICAgICAgICAgcmVjdXJzaW5nID0gZmFsc2U7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgZGlnZXN0ID0gbWQ1LmRpZ2VzdCgnaGV4Jykuc3Vic3RyKDAsIDE2KTtcbiAgICAgICAgcmV0dXJuIGAke29yaWdpbmFsTG9naWNhbElkfSR7ZGlnZXN0fWA7XG4gICAgICB9LFxuICAgIH0pKTtcblxuICAgIGlmIChwcm9wcy5yZXF1aXJlSW1kc3YyKSB7XG4gICAgICBBc3BlY3RzLm9mKHRoaXMpLmFkZChuZXcgSW5zdGFuY2VSZXF1aXJlSW1kc3YyQXNwZWN0KCkpO1xuICAgIH1cbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZFNlY3VyaXR5R3JvdXAoc2VjdXJpdHlHcm91cDogSVNlY3VyaXR5R3JvdXApOiB2b2lkIHtcbiAgICB0aGlzLnNlY3VyaXR5R3JvdXBzLnB1c2goc2VjdXJpdHlHcm91cCk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhZGRVc2VyRGF0YSguLi5jb21tYW5kczogc3RyaW5nW10pIHtcbiAgICB0aGlzLnVzZXJEYXRhLmFkZENvbW1hbmRzKC4uLmNvbW1hbmRzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkVG9Sb2xlUG9saWN5KHN0YXRlbWVudDogaWFtLlBvbGljeVN0YXRlbWVudCkge1xuICAgIHRoaXMucm9sZS5hZGRUb1ByaW5jaXBhbFBvbGljeShzdGF0ZW1lbnQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFVzZSBhIENsb3VkRm9ybWF0aW9uIEluaXQgY29uZmlndXJhdGlvbiBhdCBpbnN0YW5jZSBzdGFydHVwXG4gICAqXG4gICAqIFRoaXMgZG9lcyB0aGUgZm9sbG93aW5nOlxuICAgKlxuICAgKiAtIEF0dGFjaGVzIHRoZSBDbG91ZEZvcm1hdGlvbiBJbml0IG1ldGFkYXRhIHRvIHRoZSBJbnN0YW5jZSByZXNvdXJjZS5cbiAgICogLSBBZGQgY29tbWFuZHMgdG8gdGhlIGluc3RhbmNlIFVzZXJEYXRhIHRvIHJ1biBgY2ZuLWluaXRgIGFuZCBgY2ZuLXNpZ25hbGAuXG4gICAqIC0gVXBkYXRlIHRoZSBpbnN0YW5jZSdzIENyZWF0aW9uUG9saWN5IHRvIHdhaXQgZm9yIHRoZSBgY2ZuLXNpZ25hbGAgY29tbWFuZHMuXG4gICAqL1xuICBwcml2YXRlIGFwcGx5Q2xvdWRGb3JtYXRpb25Jbml0KGluaXQ6IENsb3VkRm9ybWF0aW9uSW5pdCwgb3B0aW9uczogQXBwbHlDbG91ZEZvcm1hdGlvbkluaXRPcHRpb25zID0ge30pIHtcbiAgICBpbml0LmF0dGFjaCh0aGlzLmluc3RhbmNlLCB7XG4gICAgICBwbGF0Zm9ybTogdGhpcy5vc1R5cGUsXG4gICAgICBpbnN0YW5jZVJvbGU6IHRoaXMucm9sZSxcbiAgICAgIHVzZXJEYXRhOiB0aGlzLnVzZXJEYXRhLFxuICAgICAgY29uZmlnU2V0czogb3B0aW9ucy5jb25maWdTZXRzLFxuICAgICAgZW1iZWRGaW5nZXJwcmludDogb3B0aW9ucy5lbWJlZEZpbmdlcnByaW50LFxuICAgICAgcHJpbnRMb2c6IG9wdGlvbnMucHJpbnRMb2csXG4gICAgICBpZ25vcmVGYWlsdXJlczogb3B0aW9ucy5pZ25vcmVGYWlsdXJlcyxcbiAgICAgIGluY2x1ZGVSb2xlOiBvcHRpb25zLmluY2x1ZGVSb2xlLFxuICAgICAgaW5jbHVkZVVybDogb3B0aW9ucy5pbmNsdWRlVXJsLFxuICAgIH0pO1xuICAgIHRoaXMud2FpdEZvclJlc291cmNlU2lnbmFsKG9wdGlvbnMudGltZW91dCA/PyBEdXJhdGlvbi5taW51dGVzKDUpKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBXYWl0IGZvciBhIHNpbmdsZSBhZGRpdGlvbmFsIHJlc291cmNlIHNpZ25hbFxuICAgKlxuICAgKiBBZGQgMSB0byB0aGUgY3VycmVudCBSZXNvdXJjZVNpZ25hbCBDb3VudCBhbmQgYWRkIHRoZSBnaXZlbiB0aW1lb3V0IHRvIHRoZSBjdXJyZW50IHRpbWVvdXQuXG4gICAqXG4gICAqIFVzZSB0aGlzIHRvIHBhdXNlIHRoZSBDbG91ZEZvcm1hdGlvbiBkZXBsb3ltZW50IHRvIHdhaXQgZm9yIHRoZSBpbnN0YW5jZXNcbiAgICogaW4gdGhlIEF1dG9TY2FsaW5nR3JvdXAgdG8gcmVwb3J0IHN1Y2Nlc3NmdWwgc3RhcnR1cCBkdXJpbmdcbiAgICogY3JlYXRpb24gYW5kIHVwZGF0ZXMuIFRoZSBVc2VyRGF0YSBzY3JpcHQgbmVlZHMgdG8gaW52b2tlIGBjZm4tc2lnbmFsYFxuICAgKiB3aXRoIGEgc3VjY2VzcyBvciBmYWlsdXJlIGNvZGUgYWZ0ZXIgaXQgaXMgZG9uZSBzZXR0aW5nIHVwIHRoZSBpbnN0YW5jZS5cbiAgICovXG4gIHByaXZhdGUgd2FpdEZvclJlc291cmNlU2lnbmFsKHRpbWVvdXQ6IER1cmF0aW9uKSB7XG4gICAgY29uc3Qgb2xkUmVzb3VyY2VTaWduYWwgPSB0aGlzLmluc3RhbmNlLmNmbk9wdGlvbnMuY3JlYXRpb25Qb2xpY3k/LnJlc291cmNlU2lnbmFsO1xuICAgIHRoaXMuaW5zdGFuY2UuY2ZuT3B0aW9ucy5jcmVhdGlvblBvbGljeSA9IHtcbiAgICAgIC4uLnRoaXMuaW5zdGFuY2UuY2ZuT3B0aW9ucy5jcmVhdGlvblBvbGljeSxcbiAgICAgIHJlc291cmNlU2lnbmFsOiB7XG4gICAgICAgIGNvdW50OiAob2xkUmVzb3VyY2VTaWduYWw/LmNvdW50ID8/IDApICsgMSxcbiAgICAgICAgdGltZW91dDogKG9sZFJlc291cmNlU2lnbmFsPy50aW1lb3V0ID8gRHVyYXRpb24ucGFyc2Uob2xkUmVzb3VyY2VTaWduYWw/LnRpbWVvdXQpLnBsdXModGltZW91dCkgOiB0aW1lb3V0KS50b0lzb1N0cmluZygpLFxuICAgICAgfSxcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIEFwcGx5IENsb3VkRm9ybWF0aW9uIHVwZGF0ZSBwb2xpY2llcyBmb3IgdGhlIGluc3RhbmNlXG4gICAqL1xuICBwcml2YXRlIGFwcGx5VXBkYXRlUG9saWNpZXMocHJvcHM6IEluc3RhbmNlUHJvcHMpIHtcbiAgICBpZiAocHJvcHMucmVzb3VyY2VTaWduYWxUaW1lb3V0ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHRoaXMuaW5zdGFuY2UuY2ZuT3B0aW9ucy5jcmVhdGlvblBvbGljeSA9IHtcbiAgICAgICAgLi4udGhpcy5pbnN0YW5jZS5jZm5PcHRpb25zLmNyZWF0aW9uUG9saWN5LFxuICAgICAgICByZXNvdXJjZVNpZ25hbDoge1xuICAgICAgICAgIHRpbWVvdXQ6IHByb3BzLnJlc291cmNlU2lnbmFsVGltZW91dCAmJiBwcm9wcy5yZXNvdXJjZVNpZ25hbFRpbWVvdXQudG9Jc29TdHJpbmcoKSxcbiAgICAgICAgfSxcbiAgICAgIH07XG4gICAgfVxuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBBcHBseUNsb3VkRm9ybWF0aW9uSW5pdE9wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNvbmZpZ1NldHM/OiBzdHJpbmdbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdGltZW91dD86IER1cmF0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW1iZWRGaW5nZXJwcmludD86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwcmludExvZz86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpZ25vcmVGYWlsdXJlcz86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaW5jbHVkZVVybD86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGluY2x1ZGVSb2xlPzogYm9vbGVhbjtcbn1cbiJdfQ==