"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RESERVED_TUNNEL_INSIDE_CIDR = exports.VpnConnection = exports.VpnGateway = exports.VpnConnectionType = void 0;
const net = require("net");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * The VPN connection type.
 */
var VpnConnectionType;
(function (VpnConnectionType) {
    VpnConnectionType["IPSEC_1"] = "ipsec.1";
    VpnConnectionType["DUMMY"] = "dummy";
})(VpnConnectionType = exports.VpnConnectionType || (exports.VpnConnectionType = {}));
/**
 * The VPN Gateway that shall be added to the VPC.
 *
 * @resource AWS::EC2::VPNGateway
 */
class VpnGateway extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        // This is 'Default' instead of 'Resource', because using 'Default' will generate
        // a logical ID for a VpnGateway which is exactly the same as the logical ID that used
        // to be created for the CfnVPNGateway (and 'Resource' would not do that).
        const vpnGW = new ec2_generated_1.CfnVPNGateway(this, 'Default', props);
        this.gatewayId = vpnGW.ref;
    }
}
exports.VpnGateway = VpnGateway;
/**
 * Define a VPN Connection.
 *
 * @resource AWS::EC2::VPNConnection
 */
class VpnConnection extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.vpc.vpnGatewayId) {
            props.vpc.enableVpnGateway({
                type: 'ipsec.1',
                amazonSideAsn: props.asn,
            });
        }
        if (!net.isIPv4(props.ip)) {
            throw new Error(`The \`ip\` ${props.ip} is not a valid IPv4 address.`);
        }
        const type = VpnConnectionType.IPSEC_1;
        const bgpAsn = props.asn || 65000;
        const customerGateway = new ec2_generated_1.CfnCustomerGateway(this, 'CustomerGateway', {
            bgpAsn,
            ipAddress: props.ip,
            type,
        });
        this.customerGatewayId = customerGateway.ref;
        this.customerGatewayAsn = bgpAsn;
        this.customerGatewayIp = props.ip;
        // Validate tunnel options
        if (props.tunnelOptions) {
            if (props.tunnelOptions.length > 2) {
                throw new Error('Cannot specify more than two `tunnelOptions`');
            }
            if (props.tunnelOptions.length === 2 && props.tunnelOptions[0].tunnelInsideCidr === props.tunnelOptions[1].tunnelInsideCidr) {
                throw new Error(`Same ${props.tunnelOptions[0].tunnelInsideCidr} \`tunnelInsideCidr\` cannot be used for both tunnels.`);
            }
            props.tunnelOptions.forEach((options, index) => {
                if (options.preSharedKey && !core_1.Token.isUnresolved(options.preSharedKey) && !/^[a-zA-Z1-9._][a-zA-Z\d._]{7,63}$/.test(options.preSharedKey)) {
                    /* eslint-disable max-len */
                    throw new Error(`The \`preSharedKey\` ${options.preSharedKey} for tunnel ${index + 1} is invalid. Allowed characters are alphanumeric characters and ._. Must be between 8 and 64 characters in length and cannot start with zero (0).`);
                    /* eslint-enable max-len */
                }
                if (options.tunnelInsideCidr) {
                    if (exports.RESERVED_TUNNEL_INSIDE_CIDR.includes(options.tunnelInsideCidr)) {
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is a reserved inside CIDR.`);
                    }
                    if (!/^169\.254\.\d{1,3}\.\d{1,3}\/30$/.test(options.tunnelInsideCidr)) {
                        /* eslint-disable-next-line max-len */
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is not a size /30 CIDR block from the 169.254.0.0/16 range.`);
                    }
                }
            });
        }
        const vpnConnection = new ec2_generated_1.CfnVPNConnection(this, 'Resource', {
            type,
            customerGatewayId: customerGateway.ref,
            staticRoutesOnly: props.staticRoutes ? true : false,
            vpnGatewayId: props.vpc.vpnGatewayId,
            vpnTunnelOptionsSpecifications: props.tunnelOptions,
        });
        this.vpnId = vpnConnection.ref;
        if (props.staticRoutes) {
            props.staticRoutes.forEach(route => {
                new ec2_generated_1.CfnVPNConnectionRoute(this, `Route${route.replace(/[^\d]/g, '')}`, {
                    destinationCidrBlock: route,
                    vpnConnectionId: this.vpnId,
                });
            });
        }
    }
    /**
     * Return the given named metric for all VPN connections in the account/region.
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/VPN',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the tunnel state of all VPN connections in the account/region.
     *
     * @default average over 5 minutes
     */
    static metricAllTunnelState(props) {
        return this.metricAll('TunnelState', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the tunnel data in of all VPN connections in the account/region.
     *
     * @default sum over 5 minutes
     */
    static metricAllTunnelDataIn(props) {
        return this.metricAll('TunnelDataIn', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the tunnel data out of all VPN connections.
     *
     * @default sum over 5 minutes
     */
    static metricAllTunnelDataOut(props) {
        return this.metricAll('TunnelDataOut', { statistic: 'sum', ...props });
    }
}
exports.VpnConnection = VpnConnection;
exports.RESERVED_TUNNEL_INSIDE_CIDR = [
    '169.254.0.0/30',
    '169.254.1.0/30',
    '169.254.2.0/30',
    '169.254.3.0/30',
    '169.254.4.0/30',
    '169.254.5.0/30',
    '169.254.169.252/30',
];
//# sourceMappingURL=data:application/json;base64,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