"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubnetFilter = void 0;
const network_util_1 = require("./network-util");
/**
 * Contains logic which chooses a set of subnets from a larger list, in conjunction with SubnetSelection, to determine where to place AWS resources such as VPC endpoints, EC2 instances, etc.
 */
class SubnetFilter {
    /**
     * Chooses subnets which are in one of the given availability zones.
     */
    static availabilityZones(availabilityZones) {
        return new AvailabilityZoneSubnetFilter(availabilityZones);
    }
    /**
     * Chooses subnets such that there is at most one per availability zone.
     */
    static onePerAz() {
        return new OnePerAZSubnetFilter();
    }
    /**
     * Chooses subnets which contain any of the specified IP addresses.
     */
    static containsIpAddresses(ipv4addrs) {
        return new ContainsIpAddressesSubnetFilter(ipv4addrs);
    }
    /**
     * Executes the subnet filtering logic, returning a filtered set of subnets.
     */
    selectSubnets(_subnets) {
        throw new Error('Cannot select subnets with an abstract SubnetFilter. `selectSubnets` needs to be implmemented.');
    }
}
exports.SubnetFilter = SubnetFilter;
/**
 * Chooses subnets which are in one of the given availability zones.
 */
class AvailabilityZoneSubnetFilter extends SubnetFilter {
    constructor(availabilityZones) {
        super();
        this.availabilityZones = availabilityZones;
    }
    /**
     * Executes the subnet filtering logic.
     */
    selectSubnets(subnets) {
        return subnets.filter(s => this.availabilityZones.includes(s.availabilityZone));
    }
}
/**
 * Chooses subnets such that there is at most one per availability zone.
 */
class OnePerAZSubnetFilter extends SubnetFilter {
    constructor() {
        super();
    }
    /**
     * Executes the subnet filtering logic.
     */
    selectSubnets(subnets) {
        return this.retainOnePerAz(subnets);
    }
    retainOnePerAz(subnets) {
        const azsSeen = new Set();
        return subnets.filter(subnet => {
            if (azsSeen.has(subnet.availabilityZone)) {
                return false;
            }
            azsSeen.add(subnet.availabilityZone);
            return true;
        });
    }
}
/**
 * Chooses subnets which contain any of the specified IP addresses.
 */
class ContainsIpAddressesSubnetFilter extends SubnetFilter {
    constructor(ipAddresses) {
        super();
        this.ipAddresses = ipAddresses;
    }
    /**
     * Executes the subnet filtering logic.
     */
    selectSubnets(subnets) {
        return this.retainByIp(subnets, this.ipAddresses);
    }
    retainByIp(subnets, ips) {
        const cidrBlockObjs = ips.map(ip => {
            const ipNum = network_util_1.NetworkUtils.ipToNum(ip);
            return new network_util_1.CidrBlock(ipNum, 32);
        });
        return subnets.filter(s => {
            const subnetCidrBlock = new network_util_1.CidrBlock(s.ipv4CidrBlock);
            return cidrBlockObjs.some(cidr => subnetCidrBlock.containsCidr(cidr));
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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