import { custom_resources, aws_iam as iam, CustomResource } from 'aws-cdk-lib';
import { Construct } from 'constructs';
/**
 * The properties of an OU
 */
export interface OUProps {
    /**
     * The name of the OU
     */
    readonly name: string;
    /**
     * Whether or not to import an existing OU if the new OU is a duplicate.
     * If this is false and the OU already exists an error will be thrown.
     *
     * @default false
     */
    readonly importOnDuplicate?: boolean;
    /**
     * Whether or not a missing OU should be recreated during an update.
     * If this is false and the OU does not exist an error will be thrown when you try to update it.
     *
     * @default false
     */
    readonly allowRecreateOnUpdate?: boolean;
}
/**
 * The properties of an Account
 */
export interface AccountProps {
    /**
     * The name of the account
     */
    readonly name: string;
    /**
     * The email address of the account. Most be unique.
     */
    readonly email: string;
    /**
     * Whether or not to import an existing account if the new account is a duplicate.
     * If this is false and the account already exists an error will be thrown.
     *
     * @default false
     */
    readonly importOnDuplicate?: boolean;
    /**
     * Whether or not to allow this account to be moved between OUs. If importing is enabled
     * this will also allow imported accounts to be moved.
     *
     * @default false
     */
    readonly allowMove?: boolean;
    /**
     * Whether or not attempting to delete an account should raise an error.
     *
     * Accounts cannot be deleted programmatically, but they can be removed as a managed resource.
     * This property will allow you to control whether or not an error is thrown
     * when the stack wants to delete an account (orphan it) or if it should continue
     * silently.
     *
     * @see https://aws.amazon.com/premiumsupport/knowledge-center/close-aws-account/
     *
     * @default false
     */
    readonly disableDelete?: boolean;
}
/**
 * The structure of an OrgObject
 */
export interface OUObject {
    /**
     * The unique id of the OUObject. This is used as the unique identifier when instantiating a construct object.
     * This is important for the CDK to be able to maintain a reference for the object when utilizing
     * the processOUObj function rather then using the name property of an object which could change.
     * If the id changes the CDK treats this as a new construct and will create a new construct resource and
     * destroy the old one.
     *
     * Not strictly required but useful when using the processOUObj function. If the id is not provided
     * the name property will be used as the id in processOUObj.
     *
     * You can create a unique id however you like. A bash example is provided.
     * @example
     * echo "ou-$( echo $RANDOM | md5sum | head -c 8 )"
     */
    readonly id?: string;
    /**
     * The OU object properties.
     */
    readonly properties: OUProps;
    /**
     * Accounts that belong to this OU
     */
    readonly accounts?: AccountProps[];
    /**
     * OUs that are children of this OU
     */
    readonly children: OUObject[];
}
/**
 * @function processOUObj
 * Function to process an OrgObject and create the corresponding AWS resources
 *
 * @param {Construct} this The construct resources will be added to.
 * @param {custom_resources.Provider} ouProvider The custom resource provider for managing OUs
 * @param {custom_resources.Provider} accountProvider The custom resource provider for managing accounts
 * @param {OUObject} obj The OrgObject to process.
 * @param {string | OrganizationOU} parent The parent of the OrgObject. This is either a string, like for the org root, or an OrganizationOU object from the same stack.
 */
export declare function processOUObj(this: Construct, ouProvider: custom_resources.Provider, accountProvider: custom_resources.Provider, obj: OUObject, parent: string | OrganizationOU): void;
/**
 * The properties for the OU custom resource provider.
 */
export interface OrganizationOUProviderProps {
    /**
     * The role the custom resource should use for taking actions on OUs if one is not provided one will be created automatically
     */
    readonly role?: iam.IRole;
}
/**
 * The provider for OU custom resources
 *
 * This creates a lambda function that handles custom resource requests for creating/updating/deleting OUs.
 */
export declare class OrganizationOUProvider extends Construct {
    readonly provider: custom_resources.Provider;
    constructor(scope: Construct, id: string, props: OrganizationOUProviderProps);
}
/**
 * The properties of an OrganizationOU custom resource.
 */
export interface OUResourceProps extends OUProps {
    /**
     * The parent OU id
     */
    readonly parent: string | OrganizationOU;
    /**
     * The provider to use for the custom resource that will create the OU. You can create a provider with the OrganizationOuProvider class
     */
    readonly provider: custom_resources.Provider;
}
/**
 * The construct to create or update an Organization OU
 *
 * This relies on the custom resource provider OrganizationOUProvider.
*/
export declare class OrganizationOU extends Construct {
    readonly resource: CustomResource;
    constructor(scope: Construct, id: string, props: OUResourceProps);
}
/**
 * The properties for the account custom resource provider.
 */
export interface OrganizationAccountProviderProps {
    /**
     * The role the custom resource should use for taking actions on OUs if one is not provided one will be created automatically
     */
    readonly role?: iam.IRole;
}
/**
 * The provider for account custom resources
 *
 * This creates a lambda function that handles custom resource requests for creating/updating accounts.
 */
export declare class OrganizationAccountProvider extends Construct {
    readonly provider: custom_resources.Provider;
    constructor(scope: Construct, id: string, props: OrganizationOUProviderProps);
}
/**
 * The properties of an OrganizationAccount custom resource.
 */
export interface AccountResourceProps extends AccountProps {
    /**
     * The parent OU id
     */
    readonly parent: string | OrganizationOU;
    /**
     * The provider to use for the custom resource that will create the OU. You can create a provider with the OrganizationOuProvider class
     */
    readonly provider: custom_resources.Provider;
}
/**
 * The construct to create or update an Organization account
 *
 * This relies on the custom resource provider OrganizationAccountProvider.
*/
export declare class OrganizationAccount extends Construct {
    readonly resource: CustomResource;
    constructor(scope: Construct, id: string, props: AccountResourceProps);
}
/**
 * The properties of an IPAM administrator custom resource provider.
 */
export interface IPAMAdministratorProviderProps {
    /**
     * The role the custom resource should use for working with the IPAM administrator delegation if one is not provided one will be created automatically
     */
    readonly role?: iam.IRole;
}
export declare class IPAMAdministratorProvider extends Construct {
    readonly provider: custom_resources.Provider;
    constructor(scope: Construct, id: string, props: IPAMAdministratorProviderProps);
}
/**
 * The properties of an OrganizationAccount custom resource.
 */
export interface IPAMAdministratorProps {
    /**
     * The account id of the IPAM administrator
     */
    readonly delegatedAdminAccountId: string;
    /**
     * The provider to use for the custom resource that will handle IPAM admin delegation. You can create a provider with the IPAMAdministratorProvider class
     */
    readonly provider: custom_resources.Provider;
}
/**
 * The construct to create or update the delegated IPAM administrator for an organization
 *
 * This relies on the custom resource provider IPAMAdministratorProvider.
*/
export declare class IPAMdministrator extends Construct {
    readonly resource: CustomResource;
    constructor(scope: Construct, id: string, props: IPAMAdministratorProps);
}
