import unittest
from unittest.mock import MagicMock, call

from osgar.drivers.cortexpilot import Cortexpilot
from osgar.bus import BusHandler


SAMPLE_DATA = (b'\x00\x02,\x01\x0c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x16' +
    b'\xdd\x13>\x0f@>\xbcT\x17j=\x91\xeb|\xbf\xaaW#C\xef\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00' +
    b'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x19')


class CortextpilotTest(unittest.TestCase):

    def test_usage(self):
        q = MagicMock()
        logger = MagicMock()
        logger.write = MagicMock(return_value=135)
        bus = BusHandler(logger=logger,
                out={'raw': [(q, 'raw')], 'encoders': [], 'emergency_stop': [],
                     'pose2d': [], 'buttons': []})
        robot = Cortexpilot(config={}, bus=bus)
        robot.start()
        robot.request_stop()
        robot.join()
        # at first request firmware version
        q.put.assert_called_once_with((135, 'raw', 
            b'\x00\x00\x03\x01\x01\xfb'))

    def test_2nd_loop(self):
        q = MagicMock()
        logger = MagicMock()
        logger.write = MagicMock(return_value=135)
        bus = BusHandler(logger=logger,
                out={'raw': [(q, 'raw')], 'encoders': [], 'emergency_stop': [],
                     'pose2d': [], 'buttons': []})
        robot = Cortexpilot(config={}, bus=bus)
        bus.queue.put((123, 'raw', b'\x00\x00\x10\x01\x01Robik V4.0.2\x00\x8f'))
        robot.start()
        robot.request_stop()
        robot.join()

        self.assertEqual(q.put.call_args_list, [
            call((135, 'raw', b'\x00\x00\x03\x01\x01\xfb')),  # request version
            call((135, 'raw', b'\x00\x00\x0f\x01\x0c\x00\x00\x00\x00\x00\x00\x00\x00A\x00\x00\x00\xa3')) # cmd
            ])

    def test_create_packet(self):
        robot = Cortexpilot(config={}, bus=None)
        packet = robot.create_packet()
        self.assertEqual(len(packet), 3 + 15)
        self.assertEqual(sum(packet) % 256, 0)
        self.assertEqual(packet[-1], 0xa3)

        robot.desired_speed = 0.5
        packet = robot.create_packet()
        self.assertEqual(len(packet), 3 + 15)
        self.assertEqual(sum(packet) % 256, 0)
        self.assertEqual(packet[-1], 0x64)

        # test packet checksum
        robot.desired_speed = -0.12314114151
        packet = robot.create_packet()

    def test_get_packet(self):
        robot = Cortexpilot(config={}, bus=None)
        packet = robot.get_packet()
        self.assertIsNone(packet)

        robot._buf = SAMPLE_DATA
        self.assertEqual(len(robot._buf), 559)

        packet = robot.get_packet()
        self.assertIsNotNone(packet)
        self.assertEqual(len(packet), 512 + 44 + 3)
        self.assertEqual(len(robot._buf), 559 - 512 - 44 - 3)

        packet = robot.get_packet()
        self.assertIsNone(packet)

    def test_parse_packet(self):
        robot = Cortexpilot(config={}, bus=MagicMock())
        robot.parse_packet(SAMPLE_DATA)
        self.assertEqual(robot.flags, 0x0)
        self.assertAlmostEqual(robot.voltage, 0.0)

        # trigger pose update
        robot.parse_packet(SAMPLE_DATA)

# vim: expandtab sw=4 ts=4
