# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fhir_kindling',
 'fhir_kindling.fhir_query',
 'fhir_kindling.fhir_server',
 'fhir_kindling.figures',
 'fhir_kindling.generators',
 'fhir_kindling.privacy',
 'fhir_kindling.serde',
 'fhir_kindling.tests',
 'fhir_kindling.util']

package_data = \
{'': ['*']}

install_requires = \
['authlib',
 'fhir.resources',
 'httpx',
 'networkx>=2.8.8,<3.0.0',
 'orjson',
 'pendulum',
 'pydantic',
 'tqdm',
 'xmltodict']

extras_require = \
{'ds': ['pandas>=1.5.2,<2.0.0', 'plotly']}

setup_kwargs = {
    'name': 'fhir-kindling',
    'version': '1.0.0a1',
    'description': 'Python library for working with FHIR servers and resources.',
    'long_description': '![Header](./docs/logo/kindling_header.png)\n\n[![CI](https://github.com/migraf/fhir-kindling/actions/workflows/main_ci.yml/badge.svg?branch=main)](https://github.com/migraf/fhir-kindling/actions/workflows/main_ci.yml)\n[![codecov](https://codecov.io/gh/migraf/fhir-kindling/branch/main/graph/badge.svg?token=FKQENFXACB)](https://codecov.io/gh/migraf/fhir-kindling)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![Maintainability](https://api.codeclimate.com/v1/badges/3b83aa52724b6e75fc22/maintainability)](https://codeclimate.com/github/migraf/fhir-kindling/maintainability)\n![PyPI - Downloads](https://img.shields.io/pypi/dm/fhir_kindling)\n![PyPI](https://img.shields.io/pypi/v/fhir_kindling)\n\n\n\nPython client library for interacting with HL7 FHIR servers, with resource validation and parsing powered by\nthe [pydantic](https://github.com/samuelcolvin/pydantic)\nmodels created by [fhir.resources](https://github.com/nazrulworld/fhir.resources). More details in\nthe [Documentation](https://migraf.github.io/fhir-kindling/).\n\n## Features\n\n- Create, Read, Update, Delete resources using a FHIR server\'s REST API\n- Transfer resources between servers while maintaining referential integrity using server-given IDs\n- Bundle creation, validation and data management on a FHIR server via the REST API\n- Supports Hapi, Blaze and IBM FHIR servers\n- CSV serialization of query results\n- Synthetic data generation and\n\nTable of Contents\n=================\n\n* [FHIR Kindling](#fire-fhir-kindling)\n   * [Features](#features)\n   * [Installation](#installation)\n      * [Extras (optional)](#extras-optional)\n   * [Performance](#performance)\n   * [Usage](#usage)\n      * [Connecting to a FHIR server](#connecting-to-a-fhir-server)\n      * [Query resources from the server](#query-resources-from-the-server)\n         * [Basic resource query](#basic-resource-query)\n         * [Query with filters](#query-with-filters)\n         * [Including related resources in the query](#including-related-resources-in-the-query)\n         * [Query resources by reference](#query-resources-by-reference)\n      * [Add resources to the server](#add-resources-to-the-server)\n      * [Deleting/Updating resources](#deletingupdating-resources)\n      * [Transfer resources between servers](#transfer-resources-between-servers)\n   * [Credits](#credits)\n\n<!-- Created by https://github.com/ekalinin/github-markdown-toc -->\n\n\n## Installation\n\nInstall the package using pip:\n\n```shell\npip install fhir-kindling --user\n```\n\n### Extras (optional)\nFhir kindling can be used with the following extras:\n- `ds` for data science related features, such as flattening of resources into a tabular format\n- `app` installs a web app for building queries in a GUI\n\n```\npip install fhir-kindling[ds,app] --user\n```\n\n\n## Performance\n\nThis library performs request at least 1.5 times faster than other popular python FHIR libraries.\nSee [Benchmarks](benchmarks/README.md) for a more detailed description of the benchmarks.\n![Query Results](benchmarks/results/query_plot.png)\n\n\n## Usage\n\n### Connecting to a FHIR server\n\n```python\nfrom fhir_kindling import FhirServer\n\n# Connect with basic auth \nbasic_auth_server = FhirServer("https://fhir.server/fhir", username="admin", password="admin")\n# Connect with static token\ntoken_server = FhirServer("https://fhir.server/fhir", token="your_token")\n\n# Connect using oauth2/oidc\noidc_server = FhirServer("https://fhir.server/fhir", client_id="client_id", client_secret="secret",\n                         oidc_provider_url="url")\n\n# Print the server\'s capability statement\nprint(basic_auth_server.capabilities)\n\n```\n\n### Query resources from the server\n\n#### Basic resource query\n\n```python\nfrom fhir_kindling import FhirServer\nfrom fhir.resources.patient import Patient\n\n# Connect using oauth2/oidc\noidc_server = FhirServer("https://fhir.server/fhir", client_id="client_id", client_secret="secret",\n                         oidc_provider_url="url")\n\n# query all patients on the server\nquery = oidc_server.query(Patient, output_format="json").all()\nprint(query.response)\n\n# Query resources based on name of resource\nquery = oidc_server.query("Patient", output_format="json").all()\nprint(query.response)\n```\n\n#### Query with filters\n\nFiltering the targeted resource is done using the `where` method on the query object. The filter is created by defining\nthe target field, the comparison operator and the value to compare.\n\n```python\nfrom fhir_kindling import FhirServer\n\nserver = FhirServer(api_address="https://fhir.server/fhir")\n\nquery = server.query("Patient").where(field="birthDate", operator="gt", value="1980").all()\n```\n\n#### Including related resources in the query\n\nResources that reference or are referenced by resources targeted by the query can be included in the response using\nthe `include` method on the query object.\n\n```python\n# server initialization omitted\n# get the patients along with the queried conditions\nquery_patient_condition = server.query("Condition").include(resource="Condition", reference_param="subject").all()\n\n# get the conditions for a patient\nquery_patient_condition = server.query("Patient")\nquery_patient_condition = query_patient_condition.include(resource="Condition", reference_param="subject", reverse=True)\nresponse = query_patient_condition.all()\n```\n\n#### Query resources by reference\n\nIf you know the id and resource type of the resource you want to query, you can use the `get` method for a single\nreference\nfor a list of references use `get_many`. The passed references should follow the format of `<resource_type>/<id>`.\n\n```python\n# server initialization omitted\npatient = server.get("Patient/123")\n\npatients = server.get_many(["Patient/123", "Patient/456"])\n\n```\n\n### Add resources to the server\n\nResources can be added to the server using the `add` method on the server object. Lists of resources can be added using\n\'add_all\'.\n\n```python\nfrom fhir_kindling import FhirServer\nfrom fhir.resources.patient import Patient\n\n# Connect to the server\nserver = FhirServer(api_address="https://fhir.server/fhir")\n\n# add a single resource\npatient = Patient(name=[{"family": "Smith", "given": ["John"]}])\nresponse = server.add(patient)\n\n# add multiple resources\npatients = [Patient(name=[{"family": f"Smith_{i}", "given": ["John"]}]) for i in range(10)]\nresponse = server.add_all(patients)\n```\n\n### Deleting/Updating resources\n\nResources can be deleted from the server using the `delete` method on the server object, it takes as input either\nreferences to the resources or the resources itself.  \nSimilarly the `update` method can be used to update the resources on the server, by passing a list of updated resources.\n\n```python\nfrom fhir_kindling import FhirServer\nfrom fhir.resources.patient import Patient\n\n# Connect to the server\nserver = FhirServer(api_address="https://fhir.server/fhir")\n\n# add some patients\npatients = [Patient(name=[{"family": f"Smith_{i}", "given": ["John"]}]) for i in range(10)]\nresponse = server.add_all(patients)\n\n# change the name of the patients\nfor patient in response.resources:\n    patient.name[0].given[0] = "Jane"\n\n# update the patients on the server\nupdated_patients = server.update(resources=response.resources)\n\n# delete based on reference\nserver.delete(references=response.references[:5])\n# delete based on resources\nserver.delete(resources=response.resources[5:])\n```\n\n### Transfer resources between servers\n\nTransferring resources between servers is done using the `transfer` method on the server object. Using this method\nserver assigned ids are used for transfer and referential integrity is maintained.  \nThis method will also attempt to get all the resources that are referenced by the resources being transferred from the\norigin\nserver and transfer them to the destination server as well.\n\n```python\nfrom fhir_kindling import FhirServer\n\n# initialize the two servers\nserver_1 = FhirServer(api_address="https://fhir.server/fhir")\nserver_2 = FhirServer(api_address="https://fhir.server/fhir")\n\n# query some resources from server 1\nconditions = server_1.query("Condition").limit(10)\n# transfer the resources to server 2\nresponse = server_1.transfer(server_2, conditions)\n\n```\n\n\n## Credits\n\nThis package was created with Cookiecutter and\nthe [audreyr/cookiecutter-pypackage](https://github.com/audreyr/cookiecutter) project template.\n\n\n\n\n\n',
    'author': 'Michael Graf',
    'author_email': 'michael.graf3110@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://migraf.github.io/fhir-kindling/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
