from typing import Callable

from webviz_config import WebvizConfigTheme
import dash_html_components as html
import webviz_core_components as wcc

from .selector_view import (
    ensemble_selector,
    filter_vector_selector,
    vector_selector,
    parameter_selector,
    date_selector,
    plot_options,
    html_details,
    color_selector,
    color_opacity_selector,
)
from ..models import ParametersModel, SimulationTimeSeriesModel


def timeseries_view(get_uuid: Callable) -> html.Div:
    return html.Div(
        children=[
            wcc.Graph(
                id=get_uuid("vector-vs-time-graph"),
                config={"displayModeBar": False},
                style={"height": "38vh"},
            ),
        ],
    )


def selector_view(
    get_uuid: Callable,
    vectormodel: SimulationTimeSeriesModel,
    parametermodel: ParametersModel,
    theme: WebvizConfigTheme,
) -> html.Div:

    theme_colors = theme.plotly_theme.get("layout", {}).get("colorway", [])
    theme_colors = (
        theme_colors[1:12] if theme_colors and len(theme_colors) >= 12 else theme_colors
    )

    return html.Div(
        style={
            "height": "80vh",
            "overflowY": "auto",
            "font-size": "15px",
        },
        className="framed",
        children=[
            html_details(
                summary="Selections",
                children=[
                    ensemble_selector(
                        get_uuid=get_uuid,
                        parametermodel=parametermodel,
                        tab="response",
                        id_string="ensemble-selector",
                        heading="Ensemble:",
                        value=parametermodel.ensembles[0],
                    ),
                    vector_selector(get_uuid=get_uuid, vectormodel=vectormodel),
                    date_selector(get_uuid=get_uuid, vectormodel=vectormodel),
                    parameter_selector(
                        get_uuid=get_uuid, parametermodel=parametermodel, tab="response"
                    ),
                ],
                open_details=True,
            ),
            html_details(
                summary=[
                    "Filters ",
                    html.Span(
                        "(for correlations)",
                        style={
                            "float": "right",
                            "margin-top": "10px",
                            "fontSize": "13px",
                            "font-weight": "normal",
                        },
                    ),
                ],
                children=[
                    filter_vector_selector(
                        get_uuid=get_uuid, vectormodel=vectormodel, tab="response"
                    )
                ],
                open_details=False,
            ),
            html_details(
                summary="Options",
                children=[
                    plot_options(get_uuid=get_uuid, tab="response"),
                    color_selector(
                        get_uuid=get_uuid,
                        tab="response",
                        colors=[theme_colors, "Greys", "BrBG"],
                        bargap=0.2,
                        height=50,
                    ),
                    color_opacity_selector(
                        get_uuid=get_uuid, tab="response", value=0.5
                    ),
                ],
                open_details=False,
            ),
        ],
    )


def parameter_response_view(
    get_uuid: Callable,
    parametermodel: ParametersModel,
    vectormodel: SimulationTimeSeriesModel,
    theme: WebvizConfigTheme,
) -> wcc.FlexBox:
    return wcc.FlexBox(
        style={"margin": "20px"},
        children=[
            html.Div(
                style={"flex": 1, "width": "90%"},
                children=selector_view(
                    get_uuid=get_uuid,
                    parametermodel=parametermodel,
                    vectormodel=vectormodel,
                    theme=theme,
                ),
            ),
            html.Div(
                style={"flex": 2, "height": "80vh"},
                children=[
                    html.Div(
                        className="framed",
                        style={"height": "37.5vh"},
                        children=timeseries_view(get_uuid=get_uuid),
                    ),
                    html.Div(
                        className="framed",
                        style={"height": "37.5vh"},
                        children=[
                            wcc.Graph(
                                id=get_uuid("vector-vs-param-scatter"),
                                config={"displayModeBar": False},
                                style={"height": "38vh"},
                            )
                        ],
                    ),
                ],
            ),
            html.Div(
                style={"flex": 2, "height": "80vh"},
                children=[
                    html.Div(
                        className="framed",
                        style={"height": "37.5vh"},
                        children=[
                            wcc.Graph(
                                config={"displayModeBar": False},
                                style={"height": "38vh"},
                                id=get_uuid("vector-corr-graph"),
                            ),
                        ],
                    ),
                    html.Div(
                        className="framed",
                        style={"height": "37.5vh"},
                        children=[
                            wcc.Graph(
                                config={"displayModeBar": False},
                                style={"height": "38vh"},
                                id=get_uuid("param-corr-graph"),
                            ),
                        ],
                    ),
                ],
            ),
        ],
    )
