"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../lib");
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
test('create basic training job', () => {
    // WHEN
    const task = new sfn.Task(stack, 'TrainSagemaker', { task: new tasks.SagemakerTrainTask({
            trainingJobName: "MyTrainJob",
            algorithmSpecification: {
                algorithmName: "BlazingText",
            },
            inputDataConfig: [
                {
                    channelName: 'train',
                    dataSource: {
                        s3DataSource: {
                            s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucket', 'mybucket'), 'mytrainpath')
                        }
                    }
                }
            ],
            outputDataConfig: {
                s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'OutputBucket', 'mybucket'), 'myoutputpath')
            },
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::sagemaker:createTrainingJob",
                ],
            ],
        },
        End: true,
        Parameters: {
            AlgorithmSpecification: {
                AlgorithmName: 'BlazingText',
                TrainingInputMode: 'File',
            },
            InputDataConfig: [
                {
                    ChannelName: 'train',
                    DataSource: {
                        S3DataSource: {
                            S3DataType: 'S3Prefix',
                            S3Uri: {
                                'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/mytrainpath']]
                            }
                        }
                    }
                }
            ],
            OutputDataConfig: {
                S3OutputPath: {
                    'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/myoutputpath']]
                }
            },
            ResourceConfig: {
                InstanceCount: 1,
                InstanceType: 'ml.m4.xlarge',
                VolumeSizeInGB: 10
            },
            RoleArn: { "Fn::GetAtt": ["TrainSagemakerSagemakerRole89E8C593", "Arn"] },
            StoppingCondition: {
                MaxRuntimeInSeconds: 3600
            },
            TrainingJobName: 'MyTrainJob',
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'TrainSagemaker', { task: new tasks.SagemakerTrainTask({
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
                trainingJobName: "MyTrainJob",
                algorithmSpecification: {
                    algorithmName: "BlazingText",
                },
                inputDataConfig: [
                    {
                        channelName: 'train',
                        dataSource: {
                            s3DataSource: {
                                s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucket', 'mybucket'), 'mytrainpath')
                            }
                        }
                    }
                ],
                outputDataConfig: {
                    s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'OutputBucket', 'mybucket'), 'myoutputpath')
                },
            }) });
    }).toThrow(/Invalid Service Integration Pattern: WAIT_FOR_TASK_TOKEN is not supported to call SageMaker./i);
});
test('create complex training job', () => {
    // WHEN
    const kmsKey = new kms.Key(stack, 'Key');
    const vpc = new ec2.Vpc(stack, "VPC");
    const securityGroup = new ec2.SecurityGroup(stack, 'SecurityGroup', { vpc, description: 'My SG' });
    securityGroup.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'allow ssh access from the world');
    const role = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
        managedPolicies: [
            iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess')
        ],
    });
    const trainTask = new tasks.SagemakerTrainTask({
        trainingJobName: "MyTrainJob",
        integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
        role,
        algorithmSpecification: {
            algorithmName: "BlazingText",
            trainingInputMode: tasks.InputMode.FILE,
            metricDefinitions: [
                {
                    name: 'mymetric', regex: 'regex_pattern'
                }
            ]
        },
        hyperparameters: {
            lr: "0.1"
        },
        inputDataConfig: [
            {
                channelName: "train",
                contentType: "image/jpeg",
                compressionType: tasks.CompressionType.NONE,
                recordWrapperType: tasks.RecordWrapperType.RECORD_IO,
                dataSource: {
                    s3DataSource: {
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucketA', 'mybucket'), 'mytrainpath'),
                    }
                }
            },
            {
                channelName: "test",
                contentType: "image/jpeg",
                compressionType: tasks.CompressionType.GZIP,
                recordWrapperType: tasks.RecordWrapperType.RECORD_IO,
                dataSource: {
                    s3DataSource: {
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucketB', 'mybucket'), 'mytestpath'),
                    }
                }
            }
        ],
        outputDataConfig: {
            s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'OutputBucket', 'mybucket'), 'myoutputpath'),
            encryptionKey: kmsKey
        },
        resourceConfig: {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.P3, ec2.InstanceSize.XLARGE2),
            volumeSizeInGB: 50,
            volumeEncryptionKey: kmsKey,
        },
        stoppingCondition: {
            maxRuntime: cdk.Duration.hours(1)
        },
        tags: {
            Project: "MyProject"
        },
        vpcConfig: {
            vpc,
        }
    });
    trainTask.addSecurityGroup(securityGroup);
    const task = new sfn.Task(stack, 'TrainSagemaker', { task: trainTask });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::sagemaker:createTrainingJob.sync",
                ],
            ],
        },
        End: true,
        Parameters: {
            TrainingJobName: 'MyTrainJob',
            RoleArn: { "Fn::GetAtt": ["Role1ABCC5F0", "Arn"] },
            AlgorithmSpecification: {
                TrainingInputMode: 'File',
                AlgorithmName: 'BlazingText',
                MetricDefinitions: [
                    { Name: "mymetric", Regex: "regex_pattern" }
                ]
            },
            HyperParameters: {
                lr: "0.1"
            },
            InputDataConfig: [
                {
                    ChannelName: 'train',
                    CompressionType: 'None',
                    RecordWrapperType: 'RecordIO',
                    ContentType: 'image/jpeg',
                    DataSource: {
                        S3DataSource: {
                            S3DataType: 'S3Prefix',
                            S3Uri: {
                                'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/mytrainpath']]
                            }
                        }
                    }
                },
                {
                    ChannelName: 'test',
                    CompressionType: 'Gzip',
                    RecordWrapperType: 'RecordIO',
                    ContentType: 'image/jpeg',
                    DataSource: {
                        S3DataSource: {
                            S3DataType: 'S3Prefix',
                            S3Uri: {
                                'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/mytestpath']]
                            }
                        }
                    }
                }
            ],
            OutputDataConfig: {
                S3OutputPath: {
                    'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/myoutputpath']]
                },
                KmsKeyId: { "Fn::GetAtt": ["Key961B73FD", "Arn"] },
            },
            ResourceConfig: {
                InstanceCount: 1,
                InstanceType: 'ml.p3.2xlarge',
                VolumeSizeInGB: 50,
                VolumeKmsKeyId: { "Fn::GetAtt": ["Key961B73FD", "Arn"] },
            },
            StoppingCondition: {
                MaxRuntimeInSeconds: 3600
            },
            Tags: [
                { Key: "Project", Value: "MyProject" }
            ],
            VpcConfig: {
                SecurityGroupIds: [
                    { "Fn::GetAtt": ["SecurityGroupDD263621", "GroupId"] },
                    { "Fn::GetAtt": ["TrainSagemakerTrainJobSecurityGroup7C858EB9", "GroupId"] },
                ],
                Subnets: [
                    { Ref: "VPCPrivateSubnet1Subnet8BCA10E0" },
                    { Ref: "VPCPrivateSubnet2SubnetCFCDAA7A" },
                ]
            }
        },
    });
});
test('pass param to training job', () => {
    // WHEN
    const role = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
        managedPolicies: [
            iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'),
        ],
    });
    const task = new sfn.Task(stack, 'TrainSagemaker', { task: new tasks.SagemakerTrainTask({
            trainingJobName: sfn.Data.stringAt('$.JobName'),
            role,
            algorithmSpecification: {
                algorithmName: "BlazingText",
                trainingInputMode: tasks.InputMode.FILE
            },
            inputDataConfig: [
                {
                    channelName: 'train',
                    dataSource: {
                        s3DataSource: {
                            s3DataType: tasks.S3DataType.S3_PREFIX,
                            s3Location: tasks.S3Location.fromJsonExpression('$.S3Bucket')
                        }
                    }
                }
            ],
            outputDataConfig: {
                s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'Bucket', 'mybucket'), 'myoutputpath'),
            },
            resourceConfig: {
                instanceCount: 1,
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.P3, ec2.InstanceSize.XLARGE2),
                volumeSizeInGB: 50
            },
            stoppingCondition: {
                maxRuntime: cdk.Duration.hours(1)
            }
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::sagemaker:createTrainingJob",
                ],
            ],
        },
        End: true,
        Parameters: {
            'TrainingJobName.$': '$.JobName',
            'RoleArn': { "Fn::GetAtt": ["Role1ABCC5F0", "Arn"] },
            'AlgorithmSpecification': {
                TrainingInputMode: 'File',
                AlgorithmName: 'BlazingText',
            },
            'InputDataConfig': [
                {
                    ChannelName: 'train',
                    DataSource: {
                        S3DataSource: {
                            'S3DataType': 'S3Prefix',
                            'S3Uri.$': '$.S3Bucket'
                        }
                    }
                }
            ],
            'OutputDataConfig': {
                S3OutputPath: {
                    'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/myoutputpath']]
                }
            },
            'ResourceConfig': {
                InstanceCount: 1,
                InstanceType: 'ml.p3.2xlarge',
                VolumeSizeInGB: 50
            },
            'StoppingCondition': {
                MaxRuntimeInSeconds: 3600
            }
        },
    });
});
test('Cannot create a SageMaker train task with both algorithm name and image name missing', () => {
    expect(() => new tasks.SagemakerTrainTask({
        trainingJobName: 'myTrainJob',
        algorithmSpecification: {},
        inputDataConfig: [
            {
                channelName: 'train',
                dataSource: {
                    s3DataSource: {
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromJsonExpression('$.S3Bucket')
                    }
                }
            }
        ],
        outputDataConfig: {
            s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'Bucket', 'mybucket'), 'myoutputpath/')
        },
    }))
        .toThrowError(/Must define either an algorithm name or training image URI in the algorithm specification/);
});
//# sourceMappingURL=data:application/json;base64,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