"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const batch = require("@aws-cdk/aws-batch");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const path = require("path");
const tasks = require("../lib");
let stack;
let batchJobDefinition;
let batchJobQueue;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    batchJobDefinition = new batch.JobDefinition(stack, 'JobDefinition', {
        container: {
            image: ecs.ContainerImage.fromAsset(path.join(__dirname, 'batchjob-image'))
        }
    });
    batchJobQueue = new batch.JobQueue(stack, 'JobQueue', {
        computeEnvironments: [
            {
                order: 1,
                computeEnvironment: new batch.ComputeEnvironment(stack, 'ComputeEnv', {
                    computeResources: { vpc: new ec2.Vpc(stack, 'vpc') }
                })
            }
        ]
    });
});
test('Task with only the required parameters', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunBatchJob({
            jobDefinition: batchJobDefinition,
            jobName: 'JobName',
            jobQueue: batchJobQueue
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition'
                    },
                    ':states:::batch:submitJob.sync'
                ]
            ]
        },
        End: true,
        Parameters: {
            JobDefinition: { Ref: 'JobDefinition24FFE3ED' },
            JobName: 'JobName',
            JobQueue: { Ref: 'JobQueueEE3AD499' }
        }
    });
});
test('Task with all the parameters', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunBatchJob({
            jobDefinition: batchJobDefinition,
            jobName: 'JobName',
            jobQueue: batchJobQueue,
            arraySize: 15,
            containerOverrides: {
                command: ['sudo', 'rm'],
                environment: { key: 'value' },
                instanceType: new ec2.InstanceType('MULTI'),
                memory: 1024,
                gpuCount: 1,
                vcpus: 10
            },
            dependsOn: [{ jobId: '1234', type: 'some_type' }],
            payload: {
                foo: sfn.Data.stringAt('$.bar')
            },
            attempts: 3,
            timeout: cdk.Duration.seconds(60),
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition'
                    },
                    ':states:::batch:submitJob'
                ]
            ]
        },
        End: true,
        Parameters: {
            JobDefinition: { Ref: 'JobDefinition24FFE3ED' },
            JobName: 'JobName',
            JobQueue: { Ref: 'JobQueueEE3AD499' },
            ArrayProperties: { Size: 15 },
            ContainerOverrides: {
                Command: ['sudo', 'rm'],
                Environment: [{ Name: 'key', Value: 'value' }],
                InstanceType: 'MULTI',
                Memory: 1024,
                ResourceRequirements: [{ Type: 'GPU', Value: '1' }],
                Vcpus: 10
            },
            DependsOn: [{ JobId: '1234', Type: 'some_type' }],
            Parameters: { 'foo.$': '$.bar' },
            RetryStrategy: { Attempts: 3 },
            Timeout: { AttemptDurationSeconds: 60 }
        }
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'JobName',
                jobQueue: batchJobQueue,
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            })
        });
    }).toThrow(/Invalid Service Integration Pattern: WAIT_FOR_TASK_TOKEN is not supported to call RunBatchJob./i);
});
test('Task throws if environment in containerOverrides contain env with name starting with AWS_BATCH', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'JobName',
                jobQueue: batchJobQueue,
                containerOverrides: {
                    environment: { AWS_BATCH_MY_NAME: 'MY_VALUE' }
                }
            })
        });
    }).toThrow(/Invalid environment variable name: AWS_BATCH_MY_NAME. Environment variable names starting with 'AWS_BATCH' are reserved./i);
});
test('Task throws if arraySize is out of limits 2-10000', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'JobName',
                jobQueue: batchJobQueue,
                arraySize: 1
            })
        });
    }).toThrow(/Invalid value of arraySize. The array size can be between 2 and 10,000./i);
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'JobName',
                jobQueue: batchJobQueue,
                arraySize: 10001
            })
        });
    }).toThrow(/Invalid value of arraySize. The array size can be between 2 and 10,000./i);
});
test('Task throws if dependencies exceeds 20', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'JobName',
                jobQueue: batchJobQueue,
                dependsOn: [...Array(21).keys()].map(i => ({
                    jobId: `${i}`,
                    type: `some_type-${i}`
                }))
            })
        });
    }).toThrow(/Invalid number of dependencies. A job can depend upon a maximum of 20 jobs./i);
});
test('Task throws if attempts is out of limits 1-10', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'JobName',
                jobQueue: batchJobQueue,
                attempts: 0
            })
        });
    }).toThrow(/Invalid value of attempts. You may specify between 1 and 10 attempts./i);
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'JobName',
                jobQueue: batchJobQueue,
                attempts: 11
            })
        });
    }).toThrow(/Invalid value of attempts. You may specify between 1 and 10 attempts./i);
});
test('Task throws if timeout is less than 60 sec', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'JobName',
                jobQueue: batchJobQueue,
                timeout: cdk.Duration.seconds(59)
            })
        });
    }).toThrow(/Invalid value of timrout. The minimum value for the timeout is 60 seconds./i);
});
//# sourceMappingURL=data:application/json;base64,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