"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const path = require("path");
const lib_1 = require("../../aws-lambda/lib");
const tasks = require("../lib");
const app = new cdk.App();
const stack = new cdk.Stack(app, 'aws-stepfunctions-integ');
const handler = new lib_1.Function(stack, 'Handler', {
    code: lib_1.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
    handler: 'index.main',
    runtime: lib_1.Runtime.PYTHON_3_6
});
const submitJob = new sfn.Task(stack, 'Invoke Handler', {
    task: new tasks.InvokeFunction(handler),
});
const callbackHandler = new lib_1.Function(stack, 'CallbackHandler', {
    code: lib_1.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
    handler: 'index.main',
    runtime: lib_1.Runtime.PYTHON_3_6
});
const taskTokenHandler = new sfn.Task(stack, 'Invoke Handler with task token', {
    task: new tasks.RunLambdaTask(callbackHandler, {
        integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
        payload: {
            token: sfn.Context.taskToken
        }
    }),
    inputPath: '$.guid',
    resultPath: '$.status',
});
const isComplete = new sfn.Choice(stack, 'Job Complete?');
const jobFailed = new sfn.Fail(stack, 'Job Failed', {
    cause: 'AWS Batch Job Failed',
    error: 'DescribeJob returned FAILED',
});
const finalStatus = new sfn.Pass(stack, 'Final step');
const chain = sfn.Chain
    .start(submitJob)
    .next(taskTokenHandler)
    .next(isComplete
    .when(sfn.Condition.stringEquals('$.status', 'FAILED'), jobFailed)
    .when(sfn.Condition.stringEquals('$.status', 'SUCCEEDED'), finalStatus));
new sfn.StateMachine(stack, 'StateMachine', {
    definition: chain,
    timeout: cdk.Duration.seconds(30)
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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