"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
/**
 * A Step Functions Task to call StartExecution on another state machine.
 *
 * It supports three service integration patterns: FIRE_AND_FORGET, SYNC and WAIT_FOR_TASK_TOKEN.
 */
class StartExecution {
    constructor(stateMachine, props = {}) {
        this.stateMachine = stateMachine;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call Step Functions.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is missing in input (pass Context.taskToken somewhere in input)');
        }
    }
    bind(task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn("states", "startExecution", this.integrationPattern),
            policyStatements: this.createScopedAccessPolicy(task),
            parameters: {
                Input: this.props.input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name
            }
        };
    }
    /**
     * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
     *
     * the scoped access policy should be generated accordingly.
     *
     * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
     */
    createScopedAccessPolicy(task) {
        const stack = core_1.Stack.of(task);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn]
            })
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['states:DescribeExecution', 'states:StopExecution'],
                resources: ['*']
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule'
                    })]
            }));
        }
        return policyStatements;
    }
}
exports.StartExecution = StartExecution;
//# sourceMappingURL=data:application/json;base64,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