"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
const sagemaker_task_base_types_1 = require("./sagemaker-task-base-types");
/**
 * Class representing the SageMaker Create Training Job task.
 *
 * @experimental
 */
class SagemakerTrainTask {
    constructor(props) {
        this.props = props;
        /**
         * Allows specify security group connections for instances of this fleet.
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call SageMaker.`);
        }
        // set the default resource config if not defined.
        this.resourceConfig = props.resourceConfig || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
            volumeSizeInGB: 10
        };
        // set the stopping condition if not defined
        this.stoppingCondition = props.stoppingCondition || {
            maxRuntime: core_1.Duration.hours(1)
        };
        // check that either algorithm name or image is defined
        if ((!props.algorithmSpecification.algorithmName) && (!props.algorithmSpecification.trainingImage)) {
            throw new Error("Must define either an algorithm name or training image URI in the algorithm specification");
        }
        // set the input mode to 'File' if not defined
        this.algorithmSpecification = (props.algorithmSpecification.trainingInputMode) ?
            (props.algorithmSpecification) :
            ({ ...props.algorithmSpecification, trainingInputMode: sagemaker_task_base_types_1.InputMode.FILE });
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.inputDataConfig = props.inputDataConfig.map(config => {
            if (!config.dataSource.s3DataSource.s3DataType) {
                return Object.assign({}, config, { dataSource: { s3DataSource: { ...config.dataSource.s3DataSource, s3DataType: sagemaker_task_base_types_1.S3DataType.S3_PREFIX } } });
            }
            else {
                return config;
            }
        });
        // add the security groups to the connections object
        if (props.vpcConfig) {
            this.vpc = props.vpcConfig.vpc;
            this.subnets = (props.vpcConfig.subnets) ?
                (this.vpc.selectSubnets(props.vpcConfig.subnets).subnetIds) : this.vpc.selectSubnets().subnetIds;
        }
    }
    /**
     * The execution role for the Sagemaker training job.
     *
     * Only available after task has been added to a state machine.
     */
    get role() {
        if (this._role === undefined) {
            throw new Error(`role not available yet--use the object in a Task first`);
        }
        return this._role;
    }
    get grantPrincipal() {
        if (this._grantPrincipal === undefined) {
            throw new Error(`Principal not available yet--use the object in a Task first`);
        }
        return this._grantPrincipal;
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    bind(task) {
        // set the sagemaker role or create new one
        this._grantPrincipal = this._role = this.props.role || new iam.Role(task, 'SagemakerRole', {
            assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
            inlinePolicies: {
                CreateTrainingJob: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: [
                                'cloudwatch:PutMetricData',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                                'logs:CreateLogGroup',
                                'logs:DescribeLogStreams',
                                'ecr:GetAuthorizationToken',
                                ...this.props.vpcConfig
                                    ? [
                                        'ec2:CreateNetworkInterface',
                                        'ec2:CreateNetworkInterfacePermission',
                                        'ec2:DeleteNetworkInterface',
                                        'ec2:DeleteNetworkInterfacePermission',
                                        'ec2:DescribeNetworkInterfaces',
                                        'ec2:DescribeVpcs',
                                        'ec2:DescribeDhcpOptions',
                                        'ec2:DescribeSubnets',
                                        'ec2:DescribeSecurityGroups',
                                    ]
                                    : [],
                            ],
                            resources: ['*'],
                        })
                    ]
                }),
            }
        });
        if (this.props.outputDataConfig.encryptionKey) {
            this.props.outputDataConfig.encryptionKey.grantEncrypt(this._role);
        }
        if (this.props.resourceConfig && this.props.resourceConfig.volumeEncryptionKey) {
            this.props.resourceConfig.volumeEncryptionKey.grant(this._role, 'kms:CreateGrant');
        }
        // create a security group if not defined
        if (this.vpc && this.securityGroup === undefined) {
            this.securityGroup = new ec2.SecurityGroup(task, 'TrainJobSecurityGroup', {
                vpc: this.vpc
            });
            this.connections.addSecurityGroup(this.securityGroup);
            this.securityGroups.push(this.securityGroup);
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn("sagemaker", "createTrainingJob", this.integrationPattern),
            parameters: this.renderParameters(),
            policyStatements: this.makePolicyStatements(task),
        };
    }
    renderParameters() {
        return {
            TrainingJobName: this.props.trainingJobName,
            RoleArn: this._role.roleArn,
            ...(this.renderAlgorithmSpecification(this.algorithmSpecification)),
            ...(this.renderInputDataConfig(this.inputDataConfig)),
            ...(this.renderOutputDataConfig(this.props.outputDataConfig)),
            ...(this.renderResourceConfig(this.resourceConfig)),
            ...(this.renderStoppingCondition(this.stoppingCondition)),
            ...(this.renderHyperparameters(this.props.hyperparameters)),
            ...(this.renderTags(this.props.tags)),
            ...(this.renderVpcConfig(this.props.vpcConfig)),
        };
    }
    renderAlgorithmSpecification(spec) {
        return {
            AlgorithmSpecification: {
                TrainingInputMode: spec.trainingInputMode,
                ...(spec.trainingImage) ? { TrainingImage: spec.trainingImage.bind(this).imageUri } : {},
                ...(spec.algorithmName) ? { AlgorithmName: spec.algorithmName } : {},
                ...(spec.metricDefinitions) ?
                    { MetricDefinitions: spec.metricDefinitions
                            .map(metric => ({ Name: metric.name, Regex: metric.regex })) } : {}
            }
        };
    }
    renderInputDataConfig(config) {
        return {
            InputDataConfig: config.map(channel => ({
                ChannelName: channel.channelName,
                DataSource: {
                    S3DataSource: {
                        S3Uri: channel.dataSource.s3DataSource.s3Location.bind(this, { forReading: true }).uri,
                        S3DataType: channel.dataSource.s3DataSource.s3DataType,
                        ...(channel.dataSource.s3DataSource.s3DataDistributionType) ?
                            { S3DataDistributionType: channel.dataSource.s3DataSource.s3DataDistributionType } : {},
                        ...(channel.dataSource.s3DataSource.attributeNames) ?
                            { AtttributeNames: channel.dataSource.s3DataSource.attributeNames } : {},
                    }
                },
                ...(channel.compressionType) ? { CompressionType: channel.compressionType } : {},
                ...(channel.contentType) ? { ContentType: channel.contentType } : {},
                ...(channel.inputMode) ? { InputMode: channel.inputMode } : {},
                ...(channel.recordWrapperType) ? { RecordWrapperType: channel.recordWrapperType } : {},
            }))
        };
    }
    renderOutputDataConfig(config) {
        return {
            OutputDataConfig: {
                S3OutputPath: config.s3OutputLocation.bind(this, { forWriting: true }).uri,
                ...(config.encryptionKey) ? { KmsKeyId: config.encryptionKey.keyArn } : {},
            }
        };
    }
    renderResourceConfig(config) {
        return {
            ResourceConfig: {
                InstanceCount: config.instanceCount,
                InstanceType: 'ml.' + config.instanceType,
                VolumeSizeInGB: config.volumeSizeInGB,
                ...(config.volumeEncryptionKey) ? { VolumeKmsKeyId: config.volumeEncryptionKey.keyArn } : {},
            }
        };
    }
    renderStoppingCondition(config) {
        return {
            StoppingCondition: {
                MaxRuntimeInSeconds: config.maxRuntime && config.maxRuntime.toSeconds()
            }
        };
    }
    renderHyperparameters(params) {
        return (params) ? { HyperParameters: params } : {};
    }
    renderTags(tags) {
        return (tags) ? { Tags: Object.keys(tags).map(key => ({ Key: key, Value: tags[key] })) } : {};
    }
    renderVpcConfig(config) {
        return (config) ? { VpcConfig: {
                SecurityGroupIds: core_1.Lazy.listValue({ produce: () => (this.securityGroups.map(sg => (sg.securityGroupId))) }),
                Subnets: this.subnets,
            } } : {};
    }
    makePolicyStatements(task) {
        const stack = core_1.Stack.of(task);
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTrainingJob', 'sagemaker:DescribeTrainingJob', 'sagemaker:StopTrainingJob'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'training-job',
                        // If the job name comes from input, we cannot target the policy to a particular ARN prefix reliably...
                        resourceName: sfn.Data.isJsonPathString(this.props.trainingJobName) ? '*' : `${this.props.trainingJobName}*`
                    })
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*']
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this._role.roleArn],
                conditions: {
                    StringEquals: { "iam:PassedToService": "sagemaker.amazonaws.com" }
                }
            })
        ];
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ["events:PutTargets", "events:PutRule", "events:DescribeRule"],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTrainingJobsRule'
                    })]
            }));
        }
        return policyStatements;
    }
}
exports.SagemakerTrainTask = SagemakerTrainTask;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2FnZW1ha2VyLXRyYWluLXRhc2suanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzYWdlbWFrZXItdHJhaW4tdGFzay50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLHdDQUF3QztBQUN4Qyx3Q0FBd0M7QUFDeEMsa0RBQWtEO0FBQ2xELHdDQUFzRDtBQUN0RCwrREFBdUQ7QUFDdkQsMkVBQ3lGO0FBc0Z6Rjs7OztHQUlHO0FBQ0gsTUFBYSxrQkFBa0I7SUFtQzNCLFlBQTZCLEtBQThCO1FBQTlCLFVBQUssR0FBTCxLQUFLLENBQXlCO1FBakMzRDs7V0FFRztRQUNhLGdCQUFXLEdBQW9CLElBQUksR0FBRyxDQUFDLFdBQVcsRUFBRSxDQUFDO1FBd0JwRCxtQkFBYyxHQUF5QixFQUFFLENBQUM7UUFPdkQsSUFBSSxDQUFDLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxHQUFHLENBQUMseUJBQXlCLENBQUMsZUFBZSxDQUFDO1FBRXBHLE1BQU0saUJBQWlCLEdBQUc7WUFDdEIsR0FBRyxDQUFDLHlCQUF5QixDQUFDLGVBQWU7WUFDN0MsR0FBRyxDQUFDLHlCQUF5QixDQUFDLElBQUk7U0FDckMsQ0FBQztRQUVGLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLEVBQUU7WUFDdEQsTUFBTSxJQUFJLEtBQUssQ0FBQyx3Q0FBd0MsSUFBSSxDQUFDLGtCQUFrQixzQ0FBc0MsQ0FBQyxDQUFDO1NBQzFIO1FBRUQsa0RBQWtEO1FBQ2xELElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsSUFBSTtZQUMxQyxhQUFhLEVBQUUsQ0FBQztZQUNoQixZQUFZLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxFQUFFLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUM7WUFDaEYsY0FBYyxFQUFFLEVBQUU7U0FDckIsQ0FBQztRQUVGLDRDQUE0QztRQUM1QyxJQUFJLENBQUMsaUJBQWlCLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixJQUFJO1lBQ2hELFVBQVUsRUFBRSxlQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztTQUNoQyxDQUFDO1FBRUYsdURBQXVEO1FBQ3ZELElBQUksQ0FBQyxDQUFDLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUMsS0FBSyxDQUFDLHNCQUFzQixDQUFDLGFBQWEsQ0FBQyxFQUFFO1lBQ2hHLE1BQU0sSUFBSSxLQUFLLENBQUMsMkZBQTJGLENBQUMsQ0FBQztTQUNoSDtRQUVELDhDQUE4QztRQUM5QyxJQUFJLENBQUMsc0JBQXNCLEdBQUcsQ0FBRSxLQUFLLENBQUMsc0JBQXNCLENBQUMsaUJBQWlCLENBQUUsQ0FBQyxDQUFDO1lBQzlFLENBQUUsS0FBSyxDQUFDLHNCQUFzQixDQUFFLENBQUMsQ0FBQztZQUNsQyxDQUFFLEVBQUUsR0FBRyxLQUFLLENBQUMsc0JBQXNCLEVBQUUsaUJBQWlCLEVBQUUscUNBQVMsQ0FBQyxJQUFJLEVBQUUsQ0FBRSxDQUFDO1FBRS9FLHdGQUF3RjtRQUN4RixJQUFJLENBQUMsZUFBZSxHQUFHLEtBQUssQ0FBQyxlQUFlLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQ3RELElBQUksQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQyxVQUFVLEVBQUU7Z0JBQzVDLE9BQU8sTUFBTSxDQUFDLE1BQU0sQ0FBQyxFQUFFLEVBQUUsTUFBTSxFQUFFLEVBQUUsVUFBVSxFQUFFLEVBQUUsWUFBWSxFQUN6RCxFQUFFLEdBQUcsTUFBTSxDQUFDLFVBQVUsQ0FBQyxZQUFZLEVBQUUsVUFBVSxFQUFFLHNDQUFVLENBQUMsU0FBUyxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUM7YUFDcEY7aUJBQU07Z0JBQ0gsT0FBTyxNQUFNLENBQUM7YUFDakI7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILG9EQUFvRDtRQUNwRCxJQUFJLEtBQUssQ0FBQyxTQUFTLEVBQUU7WUFDakIsSUFBSSxDQUFDLEdBQUcsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQztZQUMvQixJQUFJLENBQUMsT0FBTyxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO2dCQUN0QyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsYUFBYSxFQUFFLENBQUMsU0FBUyxDQUFDO1NBQ3hHO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxJQUFXLElBQUk7UUFDWCxJQUFJLElBQUksQ0FBQyxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQzFCLE1BQU0sSUFBSSxLQUFLLENBQUMsd0RBQXdELENBQUMsQ0FBQztTQUM3RTtRQUNELE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQztJQUN0QixDQUFDO0lBRUQsSUFBVyxjQUFjO1FBQ3JCLElBQUksSUFBSSxDQUFDLGVBQWUsS0FBSyxTQUFTLEVBQUU7WUFDcEMsTUFBTSxJQUFJLEtBQUssQ0FBQyw2REFBNkQsQ0FBQyxDQUFDO1NBQ2xGO1FBQ0QsT0FBTyxJQUFJLENBQUMsZUFBZSxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLGdCQUFnQixDQUFDLGFBQWlDO1FBQ3JELElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFTSxJQUFJLENBQUMsSUFBYztRQUN0QiwyQ0FBMkM7UUFDM0MsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxJQUFJLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO1lBQ3ZGLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyx5QkFBeUIsQ0FBQztZQUM5RCxjQUFjLEVBQUU7Z0JBQ1osaUJBQWlCLEVBQUUsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDO29CQUN0QyxVQUFVLEVBQUU7d0JBQ1IsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDOzRCQUNwQixPQUFPLEVBQUU7Z0NBQ0wsMEJBQTBCO2dDQUMxQixzQkFBc0I7Z0NBQ3RCLG1CQUFtQjtnQ0FDbkIscUJBQXFCO2dDQUNyQix5QkFBeUI7Z0NBQ3pCLDJCQUEyQjtnQ0FDM0IsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVM7b0NBQ25CLENBQUMsQ0FBQzt3Q0FDRSw0QkFBNEI7d0NBQzVCLHNDQUFzQzt3Q0FDdEMsNEJBQTRCO3dDQUM1QixzQ0FBc0M7d0NBQ3RDLCtCQUErQjt3Q0FDL0Isa0JBQWtCO3dDQUNsQix5QkFBeUI7d0NBQ3pCLHFCQUFxQjt3Q0FDckIsNEJBQTRCO3FDQUMvQjtvQ0FDRCxDQUFDLENBQUMsRUFBRTs2QkFDWDs0QkFDRCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7eUJBQ25CLENBQUM7cUJBQ0w7aUJBQ0osQ0FBQzthQUNMO1NBQ0osQ0FBQyxDQUFDO1FBRUgsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsRUFBRTtZQUMzQyxJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQ3RFO1FBRUQsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxtQkFBbUIsRUFBRTtZQUM1RSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1NBQ3RGO1FBRUQseUNBQXlDO1FBQ3pDLElBQUksSUFBSSxDQUFDLEdBQUcsSUFBSSxJQUFJLENBQUMsYUFBYSxLQUFLLFNBQVMsRUFBRTtZQUM5QyxJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsdUJBQXVCLEVBQUU7Z0JBQ3RFLEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRzthQUNoQixDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsV0FBVyxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUN0RCxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7U0FDaEQ7UUFFRCxPQUFPO1lBQ0wsV0FBVyxFQUFFLG9DQUFjLENBQUMsV0FBVyxFQUFFLG1CQUFtQixFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQztZQUN0RixVQUFVLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixFQUFFO1lBQ25DLGdCQUFnQixFQUFFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUM7U0FDbEQsQ0FBQztJQUNOLENBQUM7SUFFTyxnQkFBZ0I7UUFDcEIsT0FBTztZQUNILGVBQWUsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWU7WUFDM0MsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFNLENBQUMsT0FBTztZQUM1QixHQUFHLENBQUMsSUFBSSxDQUFDLDRCQUE0QixDQUFDLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1lBQ25FLEdBQUcsQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1lBQ3JELEdBQUcsQ0FBQyxJQUFJLENBQUMsc0JBQXNCLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1lBQzdELEdBQUcsQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1lBQ25ELEdBQUcsQ0FBQyxJQUFJLENBQUMsdUJBQXVCLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLENBQUM7WUFDekQsR0FBRyxDQUFDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxDQUFDO1lBQzNELEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDckMsR0FBRyxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQztTQUNsRCxDQUFDO0lBQ04sQ0FBQztJQUVPLDRCQUE0QixDQUFDLElBQTRCO1FBQzdELE9BQU87WUFDSCxzQkFBc0IsRUFBRTtnQkFDcEIsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLGlCQUFpQjtnQkFDekMsR0FBRyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxhQUFhLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUU7Z0JBQ3hGLEdBQUcsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsYUFBYSxFQUFFLElBQUksQ0FBQyxhQUFhLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDcEUsR0FBRyxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUM7b0JBQzdCLEVBQUUsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLGlCQUFpQjs2QkFDdEMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUU7YUFDMUU7U0FDSixDQUFDO0lBQ04sQ0FBQztJQUVPLHFCQUFxQixDQUFDLE1BQWlCO1FBQzNDLE9BQU87WUFDSCxlQUFlLEVBQUUsTUFBTSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUM7Z0JBQ3BDLFdBQVcsRUFBRSxPQUFPLENBQUMsV0FBVztnQkFDaEMsVUFBVSxFQUFFO29CQUNSLFlBQVksRUFBRTt3QkFDVixLQUFLLEVBQUUsT0FBTyxDQUFDLFVBQVUsQ0FBQyxZQUFZLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsRUFBRSxVQUFVLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxHQUFHO3dCQUN0RixVQUFVLEVBQUUsT0FBTyxDQUFDLFVBQVUsQ0FBQyxZQUFZLENBQUMsVUFBVTt3QkFDdEQsR0FBRyxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsWUFBWSxDQUFDLHNCQUFzQixDQUFDLENBQUMsQ0FBQzs0QkFDekQsRUFBRSxzQkFBc0IsRUFBRSxPQUFPLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQyxzQkFBc0IsRUFBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO3dCQUMxRixHQUFHLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQzs0QkFDckQsRUFBRSxlQUFlLEVBQUUsT0FBTyxDQUFDLFVBQVUsQ0FBQyxZQUFZLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUU7cUJBQzNFO2lCQUNKO2dCQUNELEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsZUFBZSxFQUFFLE9BQU8sQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDaEYsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxXQUFXLEVBQUUsT0FBTyxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFO2dCQUNwRSxHQUFHLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLFNBQVMsRUFBRSxPQUFPLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUU7Z0JBQzlELEdBQUcsQ0FBQyxPQUFPLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxpQkFBaUIsRUFBRSxPQUFPLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRTthQUN6RixDQUFDLENBQUM7U0FDTixDQUFDO0lBQ04sQ0FBQztJQUVPLHNCQUFzQixDQUFDLE1BQXdCO1FBQ25ELE9BQU87WUFDSCxnQkFBZ0IsRUFBRTtnQkFDZCxZQUFZLEVBQUUsTUFBTSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsRUFBRSxVQUFVLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxHQUFHO2dCQUMxRSxHQUFHLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLFFBQVEsRUFBRSxNQUFNLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFO2FBQzdFO1NBQ0osQ0FBQztJQUNOLENBQUM7SUFFTyxvQkFBb0IsQ0FBQyxNQUFzQjtRQUMvQyxPQUFPO1lBQ0gsY0FBYyxFQUFFO2dCQUNaLGFBQWEsRUFBRSxNQUFNLENBQUMsYUFBYTtnQkFDbkMsWUFBWSxFQUFFLEtBQUssR0FBRyxNQUFNLENBQUMsWUFBWTtnQkFDekMsY0FBYyxFQUFFLE1BQU0sQ0FBQyxjQUFjO2dCQUNyQyxHQUFHLENBQUMsTUFBTSxDQUFDLG1CQUFtQixDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsY0FBYyxFQUFFLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRTthQUMvRjtTQUNKLENBQUM7SUFDTixDQUFDO0lBRU8sdUJBQXVCLENBQUMsTUFBeUI7UUFDckQsT0FBTztZQUNILGlCQUFpQixFQUFFO2dCQUNmLG1CQUFtQixFQUFFLE1BQU0sQ0FBQyxVQUFVLElBQUksTUFBTSxDQUFDLFVBQVUsQ0FBQyxTQUFTLEVBQUU7YUFDMUU7U0FDSixDQUFDO0lBQ04sQ0FBQztJQUVPLHFCQUFxQixDQUFDLE1BQXdDO1FBQ2xFLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxlQUFlLEVBQUUsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztJQUN2RCxDQUFDO0lBRU8sVUFBVSxDQUFDLElBQXNDO1FBQ3JELE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxDQUFDLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztJQUNsRyxDQUFDO0lBRU8sZUFBZSxDQUFDLE1BQTZCO1FBQ2pELE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxTQUFTLEVBQUU7Z0JBQzNCLGdCQUFnQixFQUFFLFdBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2dCQUMxRyxPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87YUFDeEIsRUFBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFDWixDQUFDO0lBRU8sb0JBQW9CLENBQUMsSUFBYztRQUN2QyxNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTdCLDBFQUEwRTtRQUMxRSxNQUFNLGdCQUFnQixHQUFHO1lBQ3JCLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDcEIsT0FBTyxFQUFFLENBQUMsNkJBQTZCLEVBQUUsK0JBQStCLEVBQUUsMkJBQTJCLENBQUM7Z0JBQ3RHLFNBQVMsRUFBRTtvQkFDUCxLQUFLLENBQUMsU0FBUyxDQUFDO3dCQUNaLE9BQU8sRUFBRSxXQUFXO3dCQUNwQixRQUFRLEVBQUUsY0FBYzt3QkFDeEIsdUdBQXVHO3dCQUN2RyxZQUFZLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxlQUFlLEdBQUc7cUJBQy9HLENBQUM7aUJBQ0w7YUFDSixDQUFDO1lBQ0YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUNwQixPQUFPLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQztnQkFDL0IsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO2FBQ25CLENBQUM7WUFDRixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQ3BCLE9BQU8sRUFBRSxDQUFDLGNBQWMsQ0FBQztnQkFDekIsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLEtBQU0sQ0FBQyxPQUFPLENBQUM7Z0JBQ2hDLFVBQVUsRUFBRTtvQkFDUixZQUFZLEVBQUUsRUFBRSxxQkFBcUIsRUFBRSx5QkFBeUIsRUFBRTtpQkFDckU7YUFDSixDQUFDO1NBQ0wsQ0FBQztRQUVGLElBQUksSUFBSSxDQUFDLGtCQUFrQixLQUFLLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJLEVBQUU7WUFDaEUsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDMUMsT0FBTyxFQUFFLENBQUMsbUJBQW1CLEVBQUUsZ0JBQWdCLEVBQUUscUJBQXFCLENBQUM7Z0JBQ3ZFLFNBQVMsRUFBRSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUM7d0JBQ3hCLE9BQU8sRUFBRSxRQUFRO3dCQUNqQixRQUFRLEVBQUUsTUFBTTt3QkFDaEIsWUFBWSxFQUFFLG9EQUFvRDtxQkFDckUsQ0FBQyxDQUFDO2FBQ04sQ0FBQyxDQUFDLENBQUM7U0FDUDtRQUVELE9BQU8sZ0JBQWdCLENBQUM7SUFDMUIsQ0FBQztDQUNOO0FBdlRELGdEQXVUQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGVjMiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIHNmbiBmcm9tICdAYXdzLWNkay9hd3Mtc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgeyBEdXJhdGlvbiwgTGF6eSwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IGdldFJlc291cmNlQXJuIH0gZnJvbSAnLi9yZXNvdXJjZS1hcm4tc3VmZml4JztcbmltcG9ydCB7IEFsZ29yaXRobVNwZWNpZmljYXRpb24sIENoYW5uZWwsIElucHV0TW9kZSwgT3V0cHV0RGF0YUNvbmZpZywgUmVzb3VyY2VDb25maWcsXG4gICAgICAgICBTM0RhdGFUeXBlLCBTdG9wcGluZ0NvbmRpdGlvbiwgVnBjQ29uZmlnLCAgfSBmcm9tICcuL3NhZ2VtYWtlci10YXNrLWJhc2UtdHlwZXMnO1xuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGNyZWF0aW5nIGFuIEFtYXpvbiBTYWdlTWFrZXIgdHJhaW5pbmcgam9iXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFNhZ2VtYWtlclRyYWluVGFza1Byb3BzIHtcblxuICAgIC8qKlxuICAgICAqIFRyYWluaW5nIEpvYiBOYW1lLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRyYWluaW5nSm9iTmFtZTogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogUm9sZSBmb3IgdGhlIFRyYWluaW5nIEpvYi4gVGhlIHJvbGUgbXVzdCBiZSBncmFudGVkIGFsbCBuZWNlc3NhcnkgcGVybWlzc2lvbnMgZm9yIHRoZSBTYWdlTWFrZXIgdHJhaW5pbmcgam9iIHRvXG4gICAgICogYmUgYWJsZSB0byBvcGVyYXRlLlxuICAgICAqXG4gICAgICogU2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcl9mci9zYWdlbWFrZXIvbGF0ZXN0L2RnL3NhZ2VtYWtlci1yb2xlcy5odG1sI3NhZ2VtYWtlci1yb2xlcy1jcmVhdGV0cmFpbmluZ2pvYi1wZXJtc1xuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBhIHJvbGUgd2l0aCBhcHByb3ByaWF0ZSBwZXJtaXNzaW9ucyB3aWxsIGJlIGNyZWF0ZWQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcm9sZT86IGlhbS5JUm9sZTtcblxuICAgIC8qKlxuICAgICAqIFRoZSBzZXJ2aWNlIGludGVncmF0aW9uIHBhdHRlcm4gaW5kaWNhdGVzIGRpZmZlcmVudCB3YXlzIHRvIGNhbGwgU2FnZU1ha2VyIEFQSXMuXG4gICAgICpcbiAgICAgKiBUaGUgdmFsaWQgdmFsdWUgaXMgZWl0aGVyIEZJUkVfQU5EX0ZPUkdFVCBvciBTWU5DLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgRklSRV9BTkRfRk9SR0VUXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW50ZWdyYXRpb25QYXR0ZXJuPzogc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm47XG5cbiAgICAvKipcbiAgICAgKiBJZGVudGlmaWVzIHRoZSB0cmFpbmluZyBhbGdvcml0aG0gdG8gdXNlLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFsZ29yaXRobVNwZWNpZmljYXRpb246IEFsZ29yaXRobVNwZWNpZmljYXRpb247XG5cbiAgICAvKipcbiAgICAgKiBBbGdvcml0aG0tc3BlY2lmaWMgcGFyYW1ldGVycyB0aGF0IGluZmx1ZW5jZSB0aGUgcXVhbGl0eSBvZiB0aGUgbW9kZWwuIFNldCBoeXBlcnBhcmFtZXRlcnMgYmVmb3JlIHlvdSBzdGFydCB0aGUgbGVhcm5pbmcgcHJvY2Vzcy5cbiAgICAgKiBGb3IgYSBsaXN0IG9mIGh5cGVycGFyYW1ldGVycyBwcm92aWRlZCBieSBBbWF6b24gU2FnZU1ha2VyXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2FnZW1ha2VyL2xhdGVzdC9kZy9hbGdvcy5odG1sXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGh5cGVycGFyYW1ldGVyc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGh5cGVycGFyYW1ldGVycz86IHtba2V5OiBzdHJpbmddOiBhbnl9O1xuXG4gICAgLyoqXG4gICAgICogIERlc2NyaWJlcyB0aGUgdmFyaW91cyBkYXRhc2V0cyAoZS5nLiB0cmFpbiwgdmFsaWRhdGlvbiwgdGVzdCkgYW5kIHRoZSBBbWF6b24gUzMgbG9jYXRpb24gd2hlcmUgc3RvcmVkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGlucHV0RGF0YUNvbmZpZzogQ2hhbm5lbFtdO1xuXG4gICAgLyoqXG4gICAgICogVGFncyB0byBiZSBhcHBsaWVkIHRvIHRoZSB0cmFpbiBqb2IuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIHRhZ3NcbiAgICAgKi9cbiAgICByZWFkb25seSB0YWdzPzoge1trZXk6IHN0cmluZ106IHN0cmluZ307XG5cbiAgICAvKipcbiAgICAgKiBJZGVudGlmaWVzIHRoZSBBbWF6b24gUzMgbG9jYXRpb24gd2hlcmUgeW91IHdhbnQgQW1hem9uIFNhZ2VNYWtlciB0byBzYXZlIHRoZSByZXN1bHRzIG9mIG1vZGVsIHRyYWluaW5nLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IG91dHB1dERhdGFDb25maWc6IE91dHB1dERhdGFDb25maWc7XG5cbiAgICAvKipcbiAgICAgKiBTcGVjaWZpZXMgdGhlIHJlc291cmNlcywgTUwgY29tcHV0ZSBpbnN0YW5jZXMsIGFuZCBNTCBzdG9yYWdlIHZvbHVtZXMgdG8gZGVwbG95IGZvciBtb2RlbCB0cmFpbmluZy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gMSBpbnN0YW5jZSBvZiBFQzIgYE00LlhMYXJnZWAgd2l0aCBgMTBHQmAgdm9sdW1lXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmVzb3VyY2VDb25maWc/OiBSZXNvdXJjZUNvbmZpZztcblxuICAgIC8qKlxuICAgICAqIFNldHMgYSB0aW1lIGxpbWl0IGZvciB0cmFpbmluZy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gbWF4IHJ1bnRpbWUgb2YgMSBob3VyXG4gICAgICovXG4gICAgcmVhZG9ubHkgc3RvcHBpbmdDb25kaXRpb24/OiBTdG9wcGluZ0NvbmRpdGlvbjtcblxuICAgIC8qKlxuICAgICAqIFNwZWNpZmllcyB0aGUgVlBDIHRoYXQgeW91IHdhbnQgeW91ciB0cmFpbmluZyBqb2IgdG8gY29ubmVjdCB0by5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gVlBDXG4gICAgICovXG4gICAgcmVhZG9ubHkgdnBjQ29uZmlnPzogVnBjQ29uZmlnO1xufVxuXG4vKipcbiAqIENsYXNzIHJlcHJlc2VudGluZyB0aGUgU2FnZU1ha2VyIENyZWF0ZSBUcmFpbmluZyBKb2IgdGFzay5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBjbGFzcyBTYWdlbWFrZXJUcmFpblRhc2sgaW1wbGVtZW50cyBpYW0uSUdyYW50YWJsZSwgZWMyLklDb25uZWN0YWJsZSwgc2ZuLklTdGVwRnVuY3Rpb25zVGFzayB7XG5cbiAgICAvKipcbiAgICAgKiBBbGxvd3Mgc3BlY2lmeSBzZWN1cml0eSBncm91cCBjb25uZWN0aW9ucyBmb3IgaW5zdGFuY2VzIG9mIHRoaXMgZmxlZXQuXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGNvbm5lY3Rpb25zOiBlYzIuQ29ubmVjdGlvbnMgPSBuZXcgZWMyLkNvbm5lY3Rpb25zKCk7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgQWxnb3JpdGhtIFNwZWNpZmljYXRpb25cbiAgICAgKi9cbiAgICBwcml2YXRlIHJlYWRvbmx5IGFsZ29yaXRobVNwZWNpZmljYXRpb246IEFsZ29yaXRobVNwZWNpZmljYXRpb247XG5cbiAgICAvKipcbiAgICAgKiBUaGUgSW5wdXQgRGF0YSBDb25maWcuXG4gICAgICovXG4gICAgcHJpdmF0ZSByZWFkb25seSBpbnB1dERhdGFDb25maWc6IENoYW5uZWxbXTtcblxuICAgIC8qKlxuICAgICAqIFRoZSByZXNvdXJjZSBjb25maWcgZm9yIHRoZSB0YXNrLlxuICAgICAqL1xuICAgIHByaXZhdGUgcmVhZG9ubHkgcmVzb3VyY2VDb25maWc6IFJlc291cmNlQ29uZmlnO1xuXG4gICAgLyoqXG4gICAgICogVGhlIHJlc291cmNlIGNvbmZpZyBmb3IgdGhlIHRhc2suXG4gICAgICovXG4gICAgcHJpdmF0ZSByZWFkb25seSBzdG9wcGluZ0NvbmRpdGlvbjogU3RvcHBpbmdDb25kaXRpb247XG5cbiAgICBwcml2YXRlIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuICAgIHByaXZhdGUgc2VjdXJpdHlHcm91cD86IGVjMi5JU2VjdXJpdHlHcm91cDtcbiAgICBwcml2YXRlIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBzOiBlYzIuSVNlY3VyaXR5R3JvdXBbXSA9IFtdO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgc3VibmV0cz86IHN0cmluZ1tdO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybjtcbiAgICBwcml2YXRlIF9yb2xlPzogaWFtLklSb2xlO1xuICAgIHByaXZhdGUgX2dyYW50UHJpbmNpcGFsPzogaWFtLklQcmluY2lwYWw7XG5cbiAgICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBTYWdlbWFrZXJUcmFpblRhc2tQcm9wcykge1xuICAgICAgICB0aGlzLmludGVncmF0aW9uUGF0dGVybiA9IHByb3BzLmludGVncmF0aW9uUGF0dGVybiB8fCBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5GSVJFX0FORF9GT1JHRVQ7XG5cbiAgICAgICAgY29uc3Qgc3VwcG9ydGVkUGF0dGVybnMgPSBbXG4gICAgICAgICAgICBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5GSVJFX0FORF9GT1JHRVQsXG4gICAgICAgICAgICBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5TWU5DXG4gICAgICAgIF07XG5cbiAgICAgICAgaWYgKCFzdXBwb3J0ZWRQYXR0ZXJucy5pbmNsdWRlcyh0aGlzLmludGVncmF0aW9uUGF0dGVybikpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBTZXJ2aWNlIEludGVncmF0aW9uIFBhdHRlcm46ICR7dGhpcy5pbnRlZ3JhdGlvblBhdHRlcm59IGlzIG5vdCBzdXBwb3J0ZWQgdG8gY2FsbCBTYWdlTWFrZXIuYCk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBzZXQgdGhlIGRlZmF1bHQgcmVzb3VyY2UgY29uZmlnIGlmIG5vdCBkZWZpbmVkLlxuICAgICAgICB0aGlzLnJlc291cmNlQ29uZmlnID0gcHJvcHMucmVzb3VyY2VDb25maWcgfHwge1xuICAgICAgICAgICAgaW5zdGFuY2VDb3VudDogMSxcbiAgICAgICAgICAgIGluc3RhbmNlVHlwZTogZWMyLkluc3RhbmNlVHlwZS5vZihlYzIuSW5zdGFuY2VDbGFzcy5NNCwgZWMyLkluc3RhbmNlU2l6ZS5YTEFSR0UpLFxuICAgICAgICAgICAgdm9sdW1lU2l6ZUluR0I6IDEwXG4gICAgICAgIH07XG5cbiAgICAgICAgLy8gc2V0IHRoZSBzdG9wcGluZyBjb25kaXRpb24gaWYgbm90IGRlZmluZWRcbiAgICAgICAgdGhpcy5zdG9wcGluZ0NvbmRpdGlvbiA9IHByb3BzLnN0b3BwaW5nQ29uZGl0aW9uIHx8IHtcbiAgICAgICAgICAgIG1heFJ1bnRpbWU6IER1cmF0aW9uLmhvdXJzKDEpXG4gICAgICAgIH07XG5cbiAgICAgICAgLy8gY2hlY2sgdGhhdCBlaXRoZXIgYWxnb3JpdGhtIG5hbWUgb3IgaW1hZ2UgaXMgZGVmaW5lZFxuICAgICAgICBpZiAoKCFwcm9wcy5hbGdvcml0aG1TcGVjaWZpY2F0aW9uLmFsZ29yaXRobU5hbWUpICYmICghcHJvcHMuYWxnb3JpdGhtU3BlY2lmaWNhdGlvbi50cmFpbmluZ0ltYWdlKSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKFwiTXVzdCBkZWZpbmUgZWl0aGVyIGFuIGFsZ29yaXRobSBuYW1lIG9yIHRyYWluaW5nIGltYWdlIFVSSSBpbiB0aGUgYWxnb3JpdGhtIHNwZWNpZmljYXRpb25cIik7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBzZXQgdGhlIGlucHV0IG1vZGUgdG8gJ0ZpbGUnIGlmIG5vdCBkZWZpbmVkXG4gICAgICAgIHRoaXMuYWxnb3JpdGhtU3BlY2lmaWNhdGlvbiA9ICggcHJvcHMuYWxnb3JpdGhtU3BlY2lmaWNhdGlvbi50cmFpbmluZ0lucHV0TW9kZSApID9cbiAgICAgICAgICAgICggcHJvcHMuYWxnb3JpdGhtU3BlY2lmaWNhdGlvbiApIDpcbiAgICAgICAgICAgICggeyAuLi5wcm9wcy5hbGdvcml0aG1TcGVjaWZpY2F0aW9uLCB0cmFpbmluZ0lucHV0TW9kZTogSW5wdXRNb2RlLkZJTEUgfSApO1xuXG4gICAgICAgIC8vIHNldCB0aGUgUzMgRGF0YSB0eXBlIG9mIHRoZSBpbnB1dCBkYXRhIGNvbmZpZyBvYmplY3RzIHRvIGJlICdTM1ByZWZpeCcgaWYgbm90IGRlZmluZWRcbiAgICAgICAgdGhpcy5pbnB1dERhdGFDb25maWcgPSBwcm9wcy5pbnB1dERhdGFDb25maWcubWFwKGNvbmZpZyA9PiB7XG4gICAgICAgICAgICBpZiAoIWNvbmZpZy5kYXRhU291cmNlLnMzRGF0YVNvdXJjZS5zM0RhdGFUeXBlKSB7XG4gICAgICAgICAgICAgICAgcmV0dXJuIE9iamVjdC5hc3NpZ24oe30sIGNvbmZpZywgeyBkYXRhU291cmNlOiB7IHMzRGF0YVNvdXJjZTpcbiAgICAgICAgICAgICAgICAgICAgeyAuLi5jb25maWcuZGF0YVNvdXJjZS5zM0RhdGFTb3VyY2UsIHMzRGF0YVR5cGU6IFMzRGF0YVR5cGUuUzNfUFJFRklYIH0gfSB9KTtcbiAgICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAgICAgcmV0dXJuIGNvbmZpZztcbiAgICAgICAgICAgIH1cbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy8gYWRkIHRoZSBzZWN1cml0eSBncm91cHMgdG8gdGhlIGNvbm5lY3Rpb25zIG9iamVjdFxuICAgICAgICBpZiAocHJvcHMudnBjQ29uZmlnKSB7XG4gICAgICAgICAgICB0aGlzLnZwYyA9IHByb3BzLnZwY0NvbmZpZy52cGM7XG4gICAgICAgICAgICB0aGlzLnN1Ym5ldHMgPSAocHJvcHMudnBjQ29uZmlnLnN1Ym5ldHMpID9cbiAgICAgICAgICAgICAgICAodGhpcy52cGMuc2VsZWN0U3VibmV0cyhwcm9wcy52cGNDb25maWcuc3VibmV0cykuc3VibmV0SWRzKSA6IHRoaXMudnBjLnNlbGVjdFN1Ym5ldHMoKS5zdWJuZXRJZHM7XG4gICAgICAgIH1cbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBUaGUgZXhlY3V0aW9uIHJvbGUgZm9yIHRoZSBTYWdlbWFrZXIgdHJhaW5pbmcgam9iLlxuICAgICAqXG4gICAgICogT25seSBhdmFpbGFibGUgYWZ0ZXIgdGFzayBoYXMgYmVlbiBhZGRlZCB0byBhIHN0YXRlIG1hY2hpbmUuXG4gICAgICovXG4gICAgcHVibGljIGdldCByb2xlKCk6IGlhbS5JUm9sZSB7XG4gICAgICAgIGlmICh0aGlzLl9yb2xlID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgcm9sZSBub3QgYXZhaWxhYmxlIHlldC0tdXNlIHRoZSBvYmplY3QgaW4gYSBUYXNrIGZpcnN0YCk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHRoaXMuX3JvbGU7XG4gICAgfVxuXG4gICAgcHVibGljIGdldCBncmFudFByaW5jaXBhbCgpOiBpYW0uSVByaW5jaXBhbCB7XG4gICAgICAgIGlmICh0aGlzLl9ncmFudFByaW5jaXBhbCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFByaW5jaXBhbCBub3QgYXZhaWxhYmxlIHlldC0tdXNlIHRoZSBvYmplY3QgaW4gYSBUYXNrIGZpcnN0YCk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHRoaXMuX2dyYW50UHJpbmNpcGFsO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIEFkZCB0aGUgc2VjdXJpdHkgZ3JvdXAgdG8gYWxsIGluc3RhbmNlcyB2aWEgdGhlIGxhdW5jaCBjb25maWd1cmF0aW9uXG4gICAgICogc2VjdXJpdHkgZ3JvdXBzIGFycmF5LlxuICAgICAqXG4gICAgICogQHBhcmFtIHNlY3VyaXR5R3JvdXA6IFRoZSBzZWN1cml0eSBncm91cCB0byBhZGRcbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkU2VjdXJpdHlHcm91cChzZWN1cml0eUdyb3VwOiBlYzIuSVNlY3VyaXR5R3JvdXApOiB2b2lkIHtcbiAgICAgICAgdGhpcy5zZWN1cml0eUdyb3Vwcy5wdXNoKHNlY3VyaXR5R3JvdXApO1xuICAgIH1cblxuICAgIHB1YmxpYyBiaW5kKHRhc2s6IHNmbi5UYXNrKTogc2ZuLlN0ZXBGdW5jdGlvbnNUYXNrQ29uZmlnICB7XG4gICAgICAgIC8vIHNldCB0aGUgc2FnZW1ha2VyIHJvbGUgb3IgY3JlYXRlIG5ldyBvbmVcbiAgICAgICAgdGhpcy5fZ3JhbnRQcmluY2lwYWwgPSB0aGlzLl9yb2xlID0gdGhpcy5wcm9wcy5yb2xlIHx8IG5ldyBpYW0uUm9sZSh0YXNrLCAnU2FnZW1ha2VyUm9sZScsIHtcbiAgICAgICAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzYWdlbWFrZXIuYW1hem9uYXdzLmNvbScpLFxuICAgICAgICAgICAgaW5saW5lUG9saWNpZXM6IHtcbiAgICAgICAgICAgICAgICBDcmVhdGVUcmFpbmluZ0pvYjogbmV3IGlhbS5Qb2xpY3lEb2N1bWVudCh7XG4gICAgICAgICAgICAgICAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgICAgICAgICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICdjbG91ZHdhdGNoOlB1dE1ldHJpY0RhdGEnLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAnbG9nczpDcmVhdGVMb2dTdHJlYW0nLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAnbG9nczpQdXRMb2dFdmVudHMnLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAnbG9nczpDcmVhdGVMb2dHcm91cCcsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICdsb2dzOkRlc2NyaWJlTG9nU3RyZWFtcycsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICdlY3I6R2V0QXV0aG9yaXphdGlvblRva2VuJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgLi4udGhpcy5wcm9wcy52cGNDb25maWdcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgID8gW1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICdlYzI6Q3JlYXRlTmV0d29ya0ludGVyZmFjZScsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgJ2VjMjpDcmVhdGVOZXR3b3JrSW50ZXJmYWNlUGVybWlzc2lvbicsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgJ2VjMjpEZWxldGVOZXR3b3JrSW50ZXJmYWNlJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAnZWMyOkRlbGV0ZU5ldHdvcmtJbnRlcmZhY2VQZXJtaXNzaW9uJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAnZWMyOkRlc2NyaWJlTmV0d29ya0ludGVyZmFjZXMnLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICdlYzI6RGVzY3JpYmVWcGNzJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAnZWMyOkRlc2NyaWJlRGhjcE9wdGlvbnMnLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICdlYzI6RGVzY3JpYmVTdWJuZXRzJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAnZWMyOkRlc2NyaWJlU2VjdXJpdHlHcm91cHMnLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgOiBbXSxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIHJlc291cmNlczogWycqJ10sIC8vIFRob3NlIHBlcm1pc3Npb25zIGNhbm5vdCBiZSByZXNvdXJjZS1zY29wZWRcbiAgICAgICAgICAgICAgICAgICAgICAgIH0pXG4gICAgICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICB9KSxcbiAgICAgICAgICAgIH1cbiAgICAgICAgfSk7XG5cbiAgICAgICAgaWYgKHRoaXMucHJvcHMub3V0cHV0RGF0YUNvbmZpZy5lbmNyeXB0aW9uS2V5KSB7XG4gICAgICAgICAgICB0aGlzLnByb3BzLm91dHB1dERhdGFDb25maWcuZW5jcnlwdGlvbktleS5ncmFudEVuY3J5cHQodGhpcy5fcm9sZSk7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAodGhpcy5wcm9wcy5yZXNvdXJjZUNvbmZpZyAmJiB0aGlzLnByb3BzLnJlc291cmNlQ29uZmlnLnZvbHVtZUVuY3J5cHRpb25LZXkpIHtcbiAgICAgICAgICAgIHRoaXMucHJvcHMucmVzb3VyY2VDb25maWcudm9sdW1lRW5jcnlwdGlvbktleS5ncmFudCh0aGlzLl9yb2xlLCAna21zOkNyZWF0ZUdyYW50Jyk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBjcmVhdGUgYSBzZWN1cml0eSBncm91cCBpZiBub3QgZGVmaW5lZFxuICAgICAgICBpZiAodGhpcy52cGMgJiYgdGhpcy5zZWN1cml0eUdyb3VwID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIHRoaXMuc2VjdXJpdHlHcm91cCA9IG5ldyBlYzIuU2VjdXJpdHlHcm91cCh0YXNrLCAnVHJhaW5Kb2JTZWN1cml0eUdyb3VwJywge1xuICAgICAgICAgICAgICAgIHZwYzogdGhpcy52cGNcbiAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgdGhpcy5jb25uZWN0aW9ucy5hZGRTZWN1cml0eUdyb3VwKHRoaXMuc2VjdXJpdHlHcm91cCk7XG4gICAgICAgICAgICB0aGlzLnNlY3VyaXR5R3JvdXBzLnB1c2godGhpcy5zZWN1cml0eUdyb3VwKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgcmVzb3VyY2VBcm46IGdldFJlc291cmNlQXJuKFwic2FnZW1ha2VyXCIsIFwiY3JlYXRlVHJhaW5pbmdKb2JcIiwgdGhpcy5pbnRlZ3JhdGlvblBhdHRlcm4pLFxuICAgICAgICAgIHBhcmFtZXRlcnM6IHRoaXMucmVuZGVyUGFyYW1ldGVycygpLFxuICAgICAgICAgIHBvbGljeVN0YXRlbWVudHM6IHRoaXMubWFrZVBvbGljeVN0YXRlbWVudHModGFzayksXG4gICAgICAgIH07XG4gICAgfVxuXG4gICAgcHJpdmF0ZSByZW5kZXJQYXJhbWV0ZXJzKCk6IHtba2V5OiBzdHJpbmddOiBhbnl9IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIFRyYWluaW5nSm9iTmFtZTogdGhpcy5wcm9wcy50cmFpbmluZ0pvYk5hbWUsXG4gICAgICAgICAgICBSb2xlQXJuOiB0aGlzLl9yb2xlIS5yb2xlQXJuLFxuICAgICAgICAgICAgLi4uKHRoaXMucmVuZGVyQWxnb3JpdGhtU3BlY2lmaWNhdGlvbih0aGlzLmFsZ29yaXRobVNwZWNpZmljYXRpb24pKSxcbiAgICAgICAgICAgIC4uLih0aGlzLnJlbmRlcklucHV0RGF0YUNvbmZpZyh0aGlzLmlucHV0RGF0YUNvbmZpZykpLFxuICAgICAgICAgICAgLi4uKHRoaXMucmVuZGVyT3V0cHV0RGF0YUNvbmZpZyh0aGlzLnByb3BzLm91dHB1dERhdGFDb25maWcpKSxcbiAgICAgICAgICAgIC4uLih0aGlzLnJlbmRlclJlc291cmNlQ29uZmlnKHRoaXMucmVzb3VyY2VDb25maWcpKSxcbiAgICAgICAgICAgIC4uLih0aGlzLnJlbmRlclN0b3BwaW5nQ29uZGl0aW9uKHRoaXMuc3RvcHBpbmdDb25kaXRpb24pKSxcbiAgICAgICAgICAgIC4uLih0aGlzLnJlbmRlckh5cGVycGFyYW1ldGVycyh0aGlzLnByb3BzLmh5cGVycGFyYW1ldGVycykpLFxuICAgICAgICAgICAgLi4uKHRoaXMucmVuZGVyVGFncyh0aGlzLnByb3BzLnRhZ3MpKSxcbiAgICAgICAgICAgIC4uLih0aGlzLnJlbmRlclZwY0NvbmZpZyh0aGlzLnByb3BzLnZwY0NvbmZpZykpLFxuICAgICAgICB9O1xuICAgIH1cblxuICAgIHByaXZhdGUgcmVuZGVyQWxnb3JpdGhtU3BlY2lmaWNhdGlvbihzcGVjOiBBbGdvcml0aG1TcGVjaWZpY2F0aW9uKToge1trZXk6IHN0cmluZ106IGFueX0ge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgQWxnb3JpdGhtU3BlY2lmaWNhdGlvbjoge1xuICAgICAgICAgICAgICAgIFRyYWluaW5nSW5wdXRNb2RlOiBzcGVjLnRyYWluaW5nSW5wdXRNb2RlLFxuICAgICAgICAgICAgICAgIC4uLihzcGVjLnRyYWluaW5nSW1hZ2UpID8geyBUcmFpbmluZ0ltYWdlOiBzcGVjLnRyYWluaW5nSW1hZ2UuYmluZCh0aGlzKS5pbWFnZVVyaSB9IDoge30sXG4gICAgICAgICAgICAgICAgLi4uKHNwZWMuYWxnb3JpdGhtTmFtZSkgPyB7IEFsZ29yaXRobU5hbWU6IHNwZWMuYWxnb3JpdGhtTmFtZSB9IDoge30sXG4gICAgICAgICAgICAgICAgLi4uKHNwZWMubWV0cmljRGVmaW5pdGlvbnMpID9cbiAgICAgICAgICAgICAgICB7IE1ldHJpY0RlZmluaXRpb25zOiBzcGVjLm1ldHJpY0RlZmluaXRpb25zXG4gICAgICAgICAgICAgICAgICAgIC5tYXAobWV0cmljID0+ICh7IE5hbWU6IG1ldHJpYy5uYW1lLCBSZWdleDogbWV0cmljLnJlZ2V4IH0pKSB9IDoge31cbiAgICAgICAgICAgIH1cbiAgICAgICAgfTtcbiAgICB9XG5cbiAgICBwcml2YXRlIHJlbmRlcklucHV0RGF0YUNvbmZpZyhjb25maWc6IENoYW5uZWxbXSk6IHtba2V5OiBzdHJpbmddOiBhbnl9IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIElucHV0RGF0YUNvbmZpZzogY29uZmlnLm1hcChjaGFubmVsID0+ICh7XG4gICAgICAgICAgICAgICAgQ2hhbm5lbE5hbWU6IGNoYW5uZWwuY2hhbm5lbE5hbWUsXG4gICAgICAgICAgICAgICAgRGF0YVNvdXJjZToge1xuICAgICAgICAgICAgICAgICAgICBTM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgICAgICAgICAgICAgIFMzVXJpOiBjaGFubmVsLmRhdGFTb3VyY2UuczNEYXRhU291cmNlLnMzTG9jYXRpb24uYmluZCh0aGlzLCB7IGZvclJlYWRpbmc6IHRydWUgfSkudXJpLFxuICAgICAgICAgICAgICAgICAgICAgICAgUzNEYXRhVHlwZTogY2hhbm5lbC5kYXRhU291cmNlLnMzRGF0YVNvdXJjZS5zM0RhdGFUeXBlLFxuICAgICAgICAgICAgICAgICAgICAgICAgLi4uKGNoYW5uZWwuZGF0YVNvdXJjZS5zM0RhdGFTb3VyY2UuczNEYXRhRGlzdHJpYnV0aW9uVHlwZSkgP1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIHsgUzNEYXRhRGlzdHJpYnV0aW9uVHlwZTogY2hhbm5lbC5kYXRhU291cmNlLnMzRGF0YVNvdXJjZS5zM0RhdGFEaXN0cmlidXRpb25UeXBlfSA6IHt9LFxuICAgICAgICAgICAgICAgICAgICAgICAgLi4uKGNoYW5uZWwuZGF0YVNvdXJjZS5zM0RhdGFTb3VyY2UuYXR0cmlidXRlTmFtZXMpID9cbiAgICAgICAgICAgICAgICAgICAgICAgIHsgQXR0dHJpYnV0ZU5hbWVzOiBjaGFubmVsLmRhdGFTb3VyY2UuczNEYXRhU291cmNlLmF0dHJpYnV0ZU5hbWVzIH0gOiB7fSxcbiAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgLi4uKGNoYW5uZWwuY29tcHJlc3Npb25UeXBlKSA/IHsgQ29tcHJlc3Npb25UeXBlOiBjaGFubmVsLmNvbXByZXNzaW9uVHlwZSB9IDoge30sXG4gICAgICAgICAgICAgICAgLi4uKGNoYW5uZWwuY29udGVudFR5cGUpID8geyBDb250ZW50VHlwZTogY2hhbm5lbC5jb250ZW50VHlwZSB9IDoge30sXG4gICAgICAgICAgICAgICAgLi4uKGNoYW5uZWwuaW5wdXRNb2RlKSA/IHsgSW5wdXRNb2RlOiBjaGFubmVsLmlucHV0TW9kZSB9IDoge30sXG4gICAgICAgICAgICAgICAgLi4uKGNoYW5uZWwucmVjb3JkV3JhcHBlclR5cGUpID8geyBSZWNvcmRXcmFwcGVyVHlwZTogY2hhbm5lbC5yZWNvcmRXcmFwcGVyVHlwZSB9IDoge30sXG4gICAgICAgICAgICB9KSlcbiAgICAgICAgfTtcbiAgICB9XG5cbiAgICBwcml2YXRlIHJlbmRlck91dHB1dERhdGFDb25maWcoY29uZmlnOiBPdXRwdXREYXRhQ29uZmlnKToge1trZXk6IHN0cmluZ106IGFueX0ge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgT3V0cHV0RGF0YUNvbmZpZzoge1xuICAgICAgICAgICAgICAgIFMzT3V0cHV0UGF0aDogY29uZmlnLnMzT3V0cHV0TG9jYXRpb24uYmluZCh0aGlzLCB7IGZvcldyaXRpbmc6IHRydWUgfSkudXJpLFxuICAgICAgICAgICAgICAgIC4uLihjb25maWcuZW5jcnlwdGlvbktleSkgPyB7IEttc0tleUlkOiBjb25maWcuZW5jcnlwdGlvbktleS5rZXlBcm4gfSA6IHt9LFxuICAgICAgICAgICAgfVxuICAgICAgICB9O1xuICAgIH1cblxuICAgIHByaXZhdGUgcmVuZGVyUmVzb3VyY2VDb25maWcoY29uZmlnOiBSZXNvdXJjZUNvbmZpZyk6IHtba2V5OiBzdHJpbmddOiBhbnl9IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIFJlc291cmNlQ29uZmlnOiB7XG4gICAgICAgICAgICAgICAgSW5zdGFuY2VDb3VudDogY29uZmlnLmluc3RhbmNlQ291bnQsXG4gICAgICAgICAgICAgICAgSW5zdGFuY2VUeXBlOiAnbWwuJyArIGNvbmZpZy5pbnN0YW5jZVR5cGUsXG4gICAgICAgICAgICAgICAgVm9sdW1lU2l6ZUluR0I6IGNvbmZpZy52b2x1bWVTaXplSW5HQixcbiAgICAgICAgICAgICAgICAuLi4oY29uZmlnLnZvbHVtZUVuY3J5cHRpb25LZXkpID8geyBWb2x1bWVLbXNLZXlJZDogY29uZmlnLnZvbHVtZUVuY3J5cHRpb25LZXkua2V5QXJuIH0gOiB7fSxcbiAgICAgICAgICAgIH1cbiAgICAgICAgfTtcbiAgICB9XG5cbiAgICBwcml2YXRlIHJlbmRlclN0b3BwaW5nQ29uZGl0aW9uKGNvbmZpZzogU3RvcHBpbmdDb25kaXRpb24pOiB7W2tleTogc3RyaW5nXTogYW55fSB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBTdG9wcGluZ0NvbmRpdGlvbjoge1xuICAgICAgICAgICAgICAgIE1heFJ1bnRpbWVJblNlY29uZHM6IGNvbmZpZy5tYXhSdW50aW1lICYmIGNvbmZpZy5tYXhSdW50aW1lLnRvU2Vjb25kcygpXG4gICAgICAgICAgICB9XG4gICAgICAgIH07XG4gICAgfVxuXG4gICAgcHJpdmF0ZSByZW5kZXJIeXBlcnBhcmFtZXRlcnMocGFyYW1zOiB7W2tleTogc3RyaW5nXTogYW55fSB8IHVuZGVmaW5lZCk6IHtba2V5OiBzdHJpbmddOiBhbnl9IHtcbiAgICAgICAgcmV0dXJuIChwYXJhbXMpID8geyBIeXBlclBhcmFtZXRlcnM6IHBhcmFtcyB9IDoge307XG4gICAgfVxuXG4gICAgcHJpdmF0ZSByZW5kZXJUYWdzKHRhZ3M6IHtba2V5OiBzdHJpbmddOiBhbnl9IHwgdW5kZWZpbmVkKToge1trZXk6IHN0cmluZ106IGFueX0ge1xuICAgICAgICByZXR1cm4gKHRhZ3MpID8geyBUYWdzOiBPYmplY3Qua2V5cyh0YWdzKS5tYXAoa2V5ID0+ICh7IEtleToga2V5LCBWYWx1ZTogdGFnc1trZXldIH0pKSB9IDoge307XG4gICAgfVxuXG4gICAgcHJpdmF0ZSByZW5kZXJWcGNDb25maWcoY29uZmlnOiBWcGNDb25maWcgfCB1bmRlZmluZWQpOiB7W2tleTogc3RyaW5nXTogYW55fSB7XG4gICAgICAgIHJldHVybiAoY29uZmlnKSA/IHsgVnBjQ29uZmlnOiB7XG4gICAgICAgICAgICBTZWN1cml0eUdyb3VwSWRzOiBMYXp5Lmxpc3RWYWx1ZSh7IHByb2R1Y2U6ICgpID0+ICh0aGlzLnNlY3VyaXR5R3JvdXBzLm1hcChzZyA9PiAoc2cuc2VjdXJpdHlHcm91cElkKSkpIH0pLFxuICAgICAgICAgICAgU3VibmV0czogdGhpcy5zdWJuZXRzLFxuICAgICAgICB9fSA6IHt9O1xuICAgIH1cblxuICAgIHByaXZhdGUgbWFrZVBvbGljeVN0YXRlbWVudHModGFzazogc2ZuLlRhc2spOiBpYW0uUG9saWN5U3RhdGVtZW50W10ge1xuICAgICAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHRhc2spO1xuXG4gICAgICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zdGVwLWZ1bmN0aW9ucy9sYXRlc3QvZGcvc2FnZW1ha2VyLWlhbS5odG1sXG4gICAgICAgIGNvbnN0IHBvbGljeVN0YXRlbWVudHMgPSBbXG4gICAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgICAgYWN0aW9uczogWydzYWdlbWFrZXI6Q3JlYXRlVHJhaW5pbmdKb2InLCAnc2FnZW1ha2VyOkRlc2NyaWJlVHJhaW5pbmdKb2InLCAnc2FnZW1ha2VyOlN0b3BUcmFpbmluZ0pvYiddLFxuICAgICAgICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgICAgICAgICBzdGFjay5mb3JtYXRBcm4oe1xuICAgICAgICAgICAgICAgICAgICAgICAgc2VydmljZTogJ3NhZ2VtYWtlcicsXG4gICAgICAgICAgICAgICAgICAgICAgICByZXNvdXJjZTogJ3RyYWluaW5nLWpvYicsXG4gICAgICAgICAgICAgICAgICAgICAgICAvLyBJZiB0aGUgam9iIG5hbWUgY29tZXMgZnJvbSBpbnB1dCwgd2UgY2Fubm90IHRhcmdldCB0aGUgcG9saWN5IHRvIGEgcGFydGljdWxhciBBUk4gcHJlZml4IHJlbGlhYmx5Li4uXG4gICAgICAgICAgICAgICAgICAgICAgICByZXNvdXJjZU5hbWU6IHNmbi5EYXRhLmlzSnNvblBhdGhTdHJpbmcodGhpcy5wcm9wcy50cmFpbmluZ0pvYk5hbWUpID8gJyonIDogYCR7dGhpcy5wcm9wcy50cmFpbmluZ0pvYk5hbWV9KmBcbiAgICAgICAgICAgICAgICAgICAgfSlcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgICAgYWN0aW9uczogWydzYWdlbWFrZXI6TGlzdFRhZ3MnXSxcbiAgICAgICAgICAgICAgICByZXNvdXJjZXM6IFsnKiddXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICBhY3Rpb25zOiBbJ2lhbTpQYXNzUm9sZSddLFxuICAgICAgICAgICAgICAgIHJlc291cmNlczogW3RoaXMuX3JvbGUhLnJvbGVBcm5dLFxuICAgICAgICAgICAgICAgIGNvbmRpdGlvbnM6IHtcbiAgICAgICAgICAgICAgICAgICAgU3RyaW5nRXF1YWxzOiB7IFwiaWFtOlBhc3NlZFRvU2VydmljZVwiOiBcInNhZ2VtYWtlci5hbWF6b25hd3MuY29tXCIgfVxuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH0pXG4gICAgICAgIF07XG5cbiAgICAgICAgaWYgKHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuID09PSBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5TWU5DKSB7XG4gICAgICAgICAgICBwb2xpY3lTdGF0ZW1lbnRzLnB1c2gobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICAgIGFjdGlvbnM6IFtcImV2ZW50czpQdXRUYXJnZXRzXCIsIFwiZXZlbnRzOlB1dFJ1bGVcIiwgXCJldmVudHM6RGVzY3JpYmVSdWxlXCJdLFxuICAgICAgICAgICAgICAgIHJlc291cmNlczogW3N0YWNrLmZvcm1hdEFybih7XG4gICAgICAgICAgICAgICAgICAgIHNlcnZpY2U6ICdldmVudHMnLFxuICAgICAgICAgICAgICAgICAgICByZXNvdXJjZTogJ3J1bGUnLFxuICAgICAgICAgICAgICAgICAgICByZXNvdXJjZU5hbWU6ICdTdGVwRnVuY3Rpb25zR2V0RXZlbnRzRm9yU2FnZU1ha2VyVHJhaW5pbmdKb2JzUnVsZSdcbiAgICAgICAgICAgICAgICB9KV1cbiAgICAgICAgICAgIH0pKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHJldHVybiBwb2xpY3lTdGF0ZW1lbnRzO1xuICAgICAgfVxufVxuIl19