"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
/**
 * Constructs `IS3Location` objects.
 *
 * @experimental
 */
class S3Location {
    /**
     * An `IS3Location` built with a determined bucket and key prefix.
     *
     * @param bucket    is the bucket where the objects are to be stored.
     * @param keyPrefix is the key prefix used by the location.
     */
    static fromBucket(bucket, keyPrefix) {
        return new StandardS3Location({ bucket, keyPrefix, uri: bucket.urlForObject(keyPrefix) });
    }
    /**
     * An `IS3Location` determined fully by a JSON Path from the task input.
     *
     * Due to the dynamic nature of those locations, the IAM grants that will be set by `grantRead` and `grantWrite`
     * apply to the `*` resource.
     *
     * @param expression the JSON expression resolving to an S3 location URI.
     */
    static fromJsonExpression(expression) {
        return new StandardS3Location({ uri: sfn.Data.stringAt(expression) });
    }
}
exports.S3Location = S3Location;
/**
 * Creates `IDockerImage` instances.
 *
 * @experimental
 */
class DockerImage {
    /**
     * Reference a Docker image stored in an ECR repository.
     *
     * @param repository the ECR repository where the image is hosted.
     * @param tag an optional `tag`
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new StandardDockerImage({ repository, imageUri: repository.repositoryUriForTag(tag) });
    }
    /**
     * Reference a Docker image which URI is obtained from the task's input.
     *
     * @param expression           the JSON path expression with the task input.
     * @param allowAnyEcrImagePull whether ECR access should be permitted (set to `false` if the image will never be in ECR).
     */
    static fromJsonExpression(expression, allowAnyEcrImagePull = true) {
        return new StandardDockerImage({ imageUri: expression, allowAnyEcrImagePull });
    }
    /**
     * Reference a Docker image by it's URI.
     *
     * When referencing ECR images, prefer using `inEcr`.
     *
     * @param imageUri the URI to the docker image.
     */
    static fromRegistry(imageUri) {
        return new StandardDockerImage({ imageUri });
    }
    /**
     * Reference a Docker image that is provided as an Asset in the current app.
     *
     * @param scope the scope in which to create the Asset.
     * @param id    the ID for the asset in the construct tree.
     * @param props the configuration props of the asset.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new StandardDockerImage({ repository: asset.repository, imageUri: asset.imageUri });
    }
}
exports.DockerImage = DockerImage;
/**
 * S3 Data Type.
 *
 * @experimental
 */
var S3DataType;
(function (S3DataType) {
    /**
     * Manifest File Data Type
     */
    S3DataType["MANIFEST_FILE"] = "ManifestFile";
    /**
     * S3 Prefix Data Type
     */
    S3DataType["S3_PREFIX"] = "S3Prefix";
    /**
     * Augmented Manifest File Data Type
     */
    S3DataType["AUGMENTED_MANIFEST_FILE"] = "AugmentedManifestFile";
})(S3DataType = exports.S3DataType || (exports.S3DataType = {}));
/**
 * S3 Data Distribution Type.
 *
 * @experimental
 */
var S3DataDistributionType;
(function (S3DataDistributionType) {
    /**
     * Fully replicated S3 Data Distribution Type
     */
    S3DataDistributionType["FULLY_REPLICATED"] = "FullyReplicated";
    /**
     * Sharded By S3 Key Data Distribution Type
     */
    S3DataDistributionType["SHARDED_BY_S3_KEY"] = "ShardedByS3Key";
})(S3DataDistributionType = exports.S3DataDistributionType || (exports.S3DataDistributionType = {}));
/**
 * Define the format of the input data.
 *
 * @experimental
 */
var RecordWrapperType;
(function (RecordWrapperType) {
    /**
     * None record wrapper type
     */
    RecordWrapperType["NONE"] = "None";
    /**
     * RecordIO record wrapper type
     */
    RecordWrapperType["RECORD_IO"] = "RecordIO";
})(RecordWrapperType = exports.RecordWrapperType || (exports.RecordWrapperType = {}));
/**
 *  Input mode that the algorithm supports.
 *
 * @experimental
 */
var InputMode;
(function (InputMode) {
    /**
     * Pipe mode
     */
    InputMode["PIPE"] = "Pipe";
    /**
     * File mode.
     */
    InputMode["FILE"] = "File";
})(InputMode = exports.InputMode || (exports.InputMode = {}));
/**
 * Compression type of the data.
 *
 * @experimental
 */
var CompressionType;
(function (CompressionType) {
    /**
     * None compression type
     */
    CompressionType["NONE"] = "None";
    /**
     * Gzip compression type
     */
    CompressionType["GZIP"] = "Gzip";
})(CompressionType = exports.CompressionType || (exports.CompressionType = {}));
/**
 * Specifies the number of records to include in a mini-batch for an HTTP inference request.
 *
 * @experimental
 */
var BatchStrategy;
(function (BatchStrategy) {
    /**
     * Fits multiple records in a mini-batch.
     */
    BatchStrategy["MULTI_RECORD"] = "MultiRecord";
    /**
     * Use a single record when making an invocation request.
     */
    BatchStrategy["SINGLE_RECORD"] = "SingleRecord";
})(BatchStrategy = exports.BatchStrategy || (exports.BatchStrategy = {}));
/**
 * Method to use to split the transform job's data files into smaller batches.
 *
 * @experimental
 */
var SplitType;
(function (SplitType) {
    /**
     * Input data files are not split,
     */
    SplitType["NONE"] = "None";
    /**
     * Split records on a newline character boundary.
     */
    SplitType["LINE"] = "Line";
    /**
     * Split using MXNet RecordIO format.
     */
    SplitType["RECORD_IO"] = "RecordIO";
    /**
     * Split using TensorFlow TFRecord format.
     */
    SplitType["TF_RECORD"] = "TFRecord";
})(SplitType = exports.SplitType || (exports.SplitType = {}));
/**
 * How to assemble the results of the transform job as a single S3 object.
 *
 * @experimental
 */
var AssembleWith;
(function (AssembleWith) {
    /**
     * Concatenate the results in binary format.
     */
    AssembleWith["NONE"] = "None";
    /**
     * Add a newline character at the end of every transformed record.
     */
    AssembleWith["LINE"] = "Line";
})(AssembleWith = exports.AssembleWith || (exports.AssembleWith = {}));
class StandardDockerImage extends DockerImage {
    constructor(opts) {
        super();
        this.allowAnyEcrImagePull = !!opts.allowAnyEcrImagePull;
        this.imageUri = opts.imageUri;
        this.repository = opts.repository;
    }
    bind(task) {
        if (this.repository) {
            this.repository.grantPull(task);
        }
        if (this.allowAnyEcrImagePull) {
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'ecr:BatchCheckLayerAvailability',
                    'ecr:GetDownloadUrlForLayer',
                    'ecr:BatchGetImage',
                ],
                resources: ['*']
            }));
        }
        return {
            imageUri: this.imageUri,
        };
    }
}
class StandardS3Location extends S3Location {
    constructor(opts) {
        super();
        this.bucket = opts.bucket;
        this.keyGlob = `${opts.keyPrefix || ''}*`;
        this.uri = opts.uri;
    }
    bind(task, opts) {
        if (this.bucket) {
            if (opts.forReading) {
                this.bucket.grantRead(task, this.keyGlob);
            }
            if (opts.forWriting) {
                this.bucket.grantWrite(task, this.keyGlob);
            }
        }
        else {
            const actions = new Array();
            if (opts.forReading) {
                actions.push('s3:GetObject', 's3:ListBucket');
            }
            if (opts.forWriting) {
                actions.push('s3:PutObject');
            }
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({ actions, resources: ['*'], }));
        }
        return { uri: this.uri };
    }
}
//# sourceMappingURL=data:application/json;base64,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