"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
/**
 * A StepFunctions Task to run a Task on ECS or Fargate
 */
class EcsRunTaskBase {
    constructor(props) {
        this.props = props;
        /**
         * Manage allowed network traffic for this service
         */
        this.connections = new ec2.Connections();
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            sfn.ServiceIntegrationPattern.SYNC
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call ECS.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.containerOverrides)) {
            throw new Error('Task Token is missing in containerOverrides (pass Context.taskToken somewhere in containerOverrides)');
        }
        for (const override of this.props.containerOverrides || []) {
            const name = override.containerName;
            if (!cdk.Token.isUnresolved(name)) {
                const cont = this.props.taskDefinition.node.tryFindChild(name);
                if (!cont) {
                    throw new Error(`Overrides mention container with name '${name}', but no such container in task definition`);
                }
            }
        }
    }
    bind(task) {
        if (this.networkConfiguration !== undefined) {
            // Make sure we have a security group if we're using AWSVPC networking
            if (this.securityGroup === undefined) {
                this.securityGroup = new ec2.SecurityGroup(task, 'SecurityGroup', { vpc: this.props.cluster.vpc });
            }
            this.connections.addSecurityGroup(this.securityGroup);
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn("ecs", "runTask", this.integrationPattern),
            parameters: {
                Cluster: this.props.cluster.clusterArn,
                TaskDefinition: this.props.taskDefinition.taskDefinitionArn,
                NetworkConfiguration: this.networkConfiguration,
                Overrides: renderOverrides(this.props.containerOverrides),
                ...this.props.parameters,
            },
            policyStatements: this.makePolicyStatements(task),
        };
    }
    configureAwsVpcNetworking(vpc, assignPublicIp, subnetSelection, securityGroup) {
        if (subnetSelection === undefined) {
            subnetSelection = { subnetType: assignPublicIp ? ec2.SubnetType.PUBLIC : ec2.SubnetType.PRIVATE };
        }
        // If none is given here, one will be created later on during bind()
        this.securityGroup = securityGroup;
        this.networkConfiguration = {
            AwsvpcConfiguration: {
                AssignPublicIp: assignPublicIp !== undefined ? (assignPublicIp ? 'ENABLED' : 'DISABLED') : undefined,
                Subnets: vpc.selectSubnets(subnetSelection).subnetIds,
                SecurityGroups: cdk.Lazy.listValue({ produce: () => [this.securityGroup.securityGroupId] }),
            }
        };
    }
    makePolicyStatements(task) {
        const stack = cdk.Stack.of(task);
        // https://docs.aws.amazon.com/step-functions/latest/dg/ecs-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.props.taskDefinition.taskDefinitionArn],
            }),
            new iam.PolicyStatement({
                actions: ['ecs:StopTask', 'ecs:DescribeTasks'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: cdk.Lazy.listValue({ produce: () => this.taskExecutionRoles().map(r => r.roleArn) })
            }),
        ];
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ["events:PutTargets", "events:PutRule", "events:DescribeRule"],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForECSTaskRule'
                    })]
            }));
        }
        return policyStatements;
    }
    taskExecutionRoles() {
        // Need to be able to pass both Task and Execution role, apparently
        const ret = new Array();
        ret.push(this.props.taskDefinition.taskRole);
        if (this.props.taskDefinition.executionRole) {
            ret.push(this.props.taskDefinition.executionRole);
        }
        return ret;
    }
}
exports.EcsRunTaskBase = EcsRunTaskBase;
function renderOverrides(containerOverrides) {
    if (!containerOverrides) {
        return undefined;
    }
    const ret = new Array();
    for (const override of containerOverrides) {
        ret.push({
            Name: override.containerName,
            Command: override.command,
            Cpu: override.cpu,
            Memory: override.memoryLimit,
            MemoryReservation: override.memoryReservation,
            Environment: override.environment && override.environment.map(e => ({
                Name: e.name,
                Value: e.value,
            }))
        });
    }
    return { ContainerOverrides: ret };
}
//# sourceMappingURL=data:application/json;base64,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