"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
/**
 * A Step Functions Task to create an EMR Cluster.
 *
 * The ClusterConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the ClusterId.
 *
 * @experimental
 */
class EmrCreateCluster {
    constructor(props) {
        this.props = props;
        this.visibleToAllUsers = (this.props.visibleToAllUsers !== undefined) ? this.props.visibleToAllUsers : true;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.SYNC;
        this._serviceRole = this.props.serviceRole;
        this._clusterRole = this.props.clusterRole;
        this._autoScalingRole = this.props.autoScalingRole;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call CreateCluster.`);
        }
    }
    /**
     * The service role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get serviceRole() {
        if (this._serviceRole === undefined) {
            throw new Error(`role not available yet--use the object in a Task first`);
        }
        return this._serviceRole;
    }
    /**
     * The instance role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get clusterRole() {
        if (this._clusterRole === undefined) {
            throw new Error(`role not available yet--use the object in a Task first`);
        }
        return this._clusterRole;
    }
    /**
     * The autoscaling role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get autoScalingRole() {
        if (this._autoScalingRole === undefined) {
            throw new Error(`role not available yet--use the object in a Task first`);
        }
        return this._autoScalingRole;
    }
    bind(task) {
        var _a, _b;
        // If the Roles are undefined then they weren't provided, so create them
        this._serviceRole = this._serviceRole || this.createServiceRole(task);
        this._clusterRole = this._clusterRole || this.createClusterRole(task);
        // AutoScaling roles are not valid with InstanceFleet clusters.
        // Attempt to create only if .instances.instanceFleets is undefined or empty
        if (this.props.instances.instanceFleets === undefined || this.props.instances.instanceFleets.length === 0) {
            this._autoScalingRole = this._autoScalingRole || this.createAutoScalingRole(task);
            // If InstanceFleets are used and an AutoScaling Role is specified, throw an error
        }
        else if (this._autoScalingRole !== undefined) {
            throw new Error('Auto Scaling roles can not be specified with instance fleets.');
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('elasticmapreduce', 'createCluster', this.integrationPattern),
            policyStatements: this.createPolicyStatements(task, this._serviceRole, this._clusterRole, this._autoScalingRole),
            parameters: {
                Instances: EmrCreateCluster.InstancesConfigPropertyToJson(this.props.instances),
                JobFlowRole: cdk.stringToCloudFormation(this._clusterRole.roleName),
                Name: cdk.stringToCloudFormation(this.props.name),
                ServiceRole: cdk.stringToCloudFormation(this._serviceRole.roleName),
                AdditionalInfo: cdk.stringToCloudFormation(this.props.additionalInfo),
                Applications: cdk.listMapper(EmrCreateCluster.ApplicationConfigPropertyToJson)(this.props.applications),
                AutoScalingRole: cdk.stringToCloudFormation((_a = this._autoScalingRole) === null || _a === void 0 ? void 0 : _a.roleName),
                BootstrapActions: cdk.listMapper(EmrCreateCluster.BootstrapActionConfigToJson)(this.props.bootstrapActions),
                Configurations: cdk.listMapper(EmrCreateCluster.ConfigurationPropertyToJson)(this.props.configurations),
                CustomAmiId: cdk.stringToCloudFormation(this.props.customAmiId),
                EbsRootVolumeSize: cdk.numberToCloudFormation(this.props.ebsRootVolumeSize),
                KerberosAttributes: (this.props.kerberosAttributes === undefined) ?
                    this.props.kerberosAttributes :
                    EmrCreateCluster.KerberosAttributesPropertyToJson(this.props.kerberosAttributes),
                LogUri: cdk.stringToCloudFormation(this.props.logUri),
                ReleaseLabel: cdk.stringToCloudFormation(this.props.releaseLabel),
                ScaleDownBehavior: cdk.stringToCloudFormation((_b = this.props.scaleDownBehavior) === null || _b === void 0 ? void 0 : _b.valueOf()),
                SecurityConfiguration: cdk.stringToCloudFormation(this.props.securityConfiguration),
                Tags: cdk.listMapper(cdk.cfnTagToCloudFormation)(this.props.tags),
                VisibleToAllUsers: cdk.booleanToCloudFormation(this.visibleToAllUsers)
            }
        };
    }
    /**
     * This generates the PolicyStatements required by the Task to call CreateCluster.
     */
    createPolicyStatements(task, serviceRole, clusterRole, autoScalingRole) {
        const stack = cdk.Stack.of(task);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: [
                    'elasticmapreduce:RunJobFlow',
                    'elasticmapreduce:DescribeCluster',
                    'elasticmapreduce:TerminateJobFlows'
                ],
                resources: ['*']
            })
        ];
        // Allow the StateMachine to PassRole to Cluster roles
        policyStatements.push(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [
                serviceRole.roleArn,
                clusterRole.roleArn
            ]
        }));
        if (autoScalingRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [autoScalingRole.roleArn]
            }));
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRRunJobFlowRule'
                    })]
            }));
        }
        return policyStatements;
    }
    /**
     * Generate the Role used by the EMR Service
     */
    createServiceRole(task) {
        return new iam.Role(task, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceRole')
            ]
        });
    }
    /**
     * Generate the Role and Instance Profile used by the EC2 instances
     *
     * Data access permissions will need to be updated by the user
     */
    createClusterRole(task) {
        const role = new iam.Role(task, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com')
        });
        new iam.CfnInstanceProfile(task, 'InstanceProfile', {
            roles: [role.roleName],
            instanceProfileName: role.roleName
        });
        return role;
    }
    /**
     * Generate the Role used to AutoScale the Cluster
     */
    createAutoScalingRole(task) {
        var _a;
        const role = new iam.Role(task, 'AutoScalingRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceforAutoScalingRole')
            ]
        });
        (_a = role.assumeRolePolicy) === null || _a === void 0 ? void 0 : _a.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            principals: [
                new iam.ServicePrincipal('application-autoscaling.amazonaws.com')
            ],
            actions: [
                'sts:AssumeRole'
            ]
        }));
        return role;
    }
}
exports.EmrCreateCluster = EmrCreateCluster;
(function (EmrCreateCluster) {
    /**
     * Valid valus for the Cluster ScaleDownBehavior
     *
     * @experimental
     */
    let EmrClusterScaleDownBehavior;
    (function (EmrClusterScaleDownBehavior) {
        /**
         * Indicates that Amazon EMR terminates nodes at the instance-hour boundary, regardless of when the request to terminate the instance was
         * submitted. This option is only available with Amazon EMR 5.1.0 and later and is the default for clusters created using that version
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_INSTANCE_HOUR"] = "TERMINATE_AT_INSTANCE_HOUR";
        /**
         * Indicates that Amazon EMR blacklists and drains tasks from nodes before terminating the Amazon EC2 instances, regardless of the
         * instance-hour boundary.
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_TASK_COMPLETION"] = "TERMINATE_AT_TASK_COMPLETION";
    })(EmrClusterScaleDownBehavior = EmrCreateCluster.EmrClusterScaleDownBehavior || (EmrCreateCluster.EmrClusterScaleDownBehavior = {}));
    /**
     * Instance Role Types
     *
     * @experimental
     */
    let InstanceRoleType;
    (function (InstanceRoleType) {
        /**
         * Master Node
         */
        InstanceRoleType["MASTER"] = "MASTER";
        /**
         * Core Node
         */
        InstanceRoleType["CORE"] = "CORE";
        /**
         * Task Node
         */
        InstanceRoleType["TASK"] = "TASK";
    })(InstanceRoleType = EmrCreateCluster.InstanceRoleType || (EmrCreateCluster.InstanceRoleType = {}));
    /**
     * EBS Volume Types
     *
     * @experimental
     */
    let EbsBlockDeviceVolumeType;
    (function (EbsBlockDeviceVolumeType) {
        /**
         * gp2 Volume Type
         */
        EbsBlockDeviceVolumeType["GP2"] = "gp2";
        /**
         * io1 Volume Type
         */
        EbsBlockDeviceVolumeType["IO1"] = "io1";
        /**
         * Standard Volume Type
         */
        EbsBlockDeviceVolumeType["STANDARD"] = "standard";
    })(EbsBlockDeviceVolumeType = EmrCreateCluster.EbsBlockDeviceVolumeType || (EmrCreateCluster.EbsBlockDeviceVolumeType = {}));
    /**
     * Render the EbsBlockDeviceConfigProperty as JSON
     *
     * @param property
     */
    function EbsBlockDeviceConfigPropertyToJson(property) {
        var _a;
        return {
            VolumeSpecification: {
                Iops: cdk.numberToCloudFormation(property.volumeSpecification.iops),
                SizeInGB: cdk.numberToCloudFormation(property.volumeSpecification.sizeInGB),
                VolumeType: cdk.stringToCloudFormation((_a = property.volumeSpecification.volumeType) === null || _a === void 0 ? void 0 : _a.valueOf())
            },
            VolumesPerInstance: cdk.numberToCloudFormation(property.volumesPerInstance)
        };
    }
    EmrCreateCluster.EbsBlockDeviceConfigPropertyToJson = EbsBlockDeviceConfigPropertyToJson;
    /**
     * Render the EbsConfigurationProperty to JSON
     *
     * @param property
     */
    function EbsConfigurationPropertyToJson(property) {
        return {
            EbsBlockDeviceConfigs: cdk.listMapper(EbsBlockDeviceConfigPropertyToJson)(property.ebsBlockDeviceConfigs),
            EbsOptimized: cdk.booleanToCloudFormation(property.ebsOptimized)
        };
    }
    EmrCreateCluster.EbsConfigurationPropertyToJson = EbsConfigurationPropertyToJson;
    /**
     * Render the InstanceTypeConfigProperty to JSON]
     *
     * @param property
     */
    function InstanceTypeConfigPropertyToJson(property) {
        var _a;
        return {
            BidPrice: cdk.stringToCloudFormation(property.bidPrice),
            BidPriceAsPercentageOfOnDemandPrice: cdk.numberToCloudFormation(property.bidPriceAsPercentageOfOnDemandPrice),
            Configurations: cdk.listMapper(ConfigurationPropertyToJson)(property.configurations),
            EbsConfiguration: (property.ebsConfiguration === undefined) ?
                property.ebsConfiguration :
                EbsConfigurationPropertyToJson(property.ebsConfiguration),
            InstanceType: cdk.stringToCloudFormation((_a = property.instanceType) === null || _a === void 0 ? void 0 : _a.valueOf()),
            WeightedCapacity: cdk.numberToCloudFormation(property.weightedCapacity)
        };
    }
    EmrCreateCluster.InstanceTypeConfigPropertyToJson = InstanceTypeConfigPropertyToJson;
    /**
     * Spot Timeout Actions
     *
     * @experimental
     */
    let SpotTimeoutAction;
    (function (SpotTimeoutAction) {
        /**\
         * SWITCH_TO_ON_DEMAND
         */
        SpotTimeoutAction["SWITCH_TO_ON_DEMAND"] = "SWITCH_TO_ON_DEMAND";
        /**
         * TERMINATE_CLUSTER
         */
        SpotTimeoutAction["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    })(SpotTimeoutAction = EmrCreateCluster.SpotTimeoutAction || (EmrCreateCluster.SpotTimeoutAction = {}));
    /**
     * Render the InstanceFleetProvisioningSpecificationsProperty to JSON
     *
     * @param property
     */
    function InstanceFleetProvisioningSpecificationsPropertyToJson(property) {
        var _a;
        return {
            SpotSpecification: {
                BlockDurationMinutes: cdk.numberToCloudFormation(property.spotSpecification.blockDurationMinutes),
                TimeoutAction: cdk.stringToCloudFormation((_a = property.spotSpecification.timeoutAction) === null || _a === void 0 ? void 0 : _a.valueOf()),
                TimeoutDurationMinutes: cdk.numberToCloudFormation(property.spotSpecification.timeoutDurationMinutes)
            }
        };
    }
    EmrCreateCluster.InstanceFleetProvisioningSpecificationsPropertyToJson = InstanceFleetProvisioningSpecificationsPropertyToJson;
    /**
     * Render the InstanceFleetConfigProperty as JSON
     *
     * @param property
     */
    function InstanceFleetConfigPropertyToJson(property) {
        var _a;
        return {
            InstanceFleetType: cdk.stringToCloudFormation((_a = property.instanceFleetType) === null || _a === void 0 ? void 0 : _a.valueOf()),
            InstanceTypeConfigs: cdk.listMapper(InstanceTypeConfigPropertyToJson)(property.instanceTypeConfigs),
            LaunchSpecifications: (property.launchSpecifications === undefined) ?
                property.launchSpecifications :
                InstanceFleetProvisioningSpecificationsPropertyToJson(property.launchSpecifications),
            Name: cdk.stringToCloudFormation(property.name),
            TargetOnDemandCapacity: cdk.numberToCloudFormation(property.targetOnDemandCapacity),
            TargetSpotCapacity: cdk.numberToCloudFormation(property.targetSpotCapacity)
        };
    }
    EmrCreateCluster.InstanceFleetConfigPropertyToJson = InstanceFleetConfigPropertyToJson;
    /**
     * CloudWatch Alarm Comparison Operators
     *
     * @experimental
     */
    let CloudWatchAlarmComparisonOperator;
    (function (CloudWatchAlarmComparisonOperator) {
        /**
         * GREATER_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN_OR_EQUAL"] = "GREATER_THAN_OR_EQUAL";
        /**
         * GREATER_THAN
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN"] = "GREATER_THAN";
        /**
         * LESS_THAN
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN"] = "LESS_THAN";
        /**
         * LESS_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN_OR_EQUAL"] = "LESS_THAN_OR_EQUAL";
    })(CloudWatchAlarmComparisonOperator = EmrCreateCluster.CloudWatchAlarmComparisonOperator || (EmrCreateCluster.CloudWatchAlarmComparisonOperator = {}));
    /**
     * CloudWatch Alarm Statistics
     *
     * @experimental
     */
    let CloudWatchAlarmStatistic;
    (function (CloudWatchAlarmStatistic) {
        /**
         * SAMPLE_COUNT
         */
        CloudWatchAlarmStatistic["SAMPLE_COUNT"] = "SAMPLE_COUNT";
        /**
         * AVERAGE
         */
        CloudWatchAlarmStatistic["AVERAGE"] = "AVERAGE";
        /**
         * SUM
         */
        CloudWatchAlarmStatistic["SUM"] = "SUM";
        /**
         * MINIMUM
         */
        CloudWatchAlarmStatistic["MINIMUM"] = "MINIMUM";
        /**
         * MAXIMUM
         */
        CloudWatchAlarmStatistic["MAXIMUM"] = "MAXIMUM";
    })(CloudWatchAlarmStatistic = EmrCreateCluster.CloudWatchAlarmStatistic || (EmrCreateCluster.CloudWatchAlarmStatistic = {}));
    /**
     * CloudWatch Alarm Units
     *
     * @experimental
     */
    let CloudWatchAlarmUnit;
    (function (CloudWatchAlarmUnit) {
        /**
         * NONE
         */
        CloudWatchAlarmUnit["NONE"] = "NONE";
        /**
         * SECONDS
         */
        CloudWatchAlarmUnit["SECONDS"] = "SECONDS";
        /**
         * MICRO_SECONDS
         */
        CloudWatchAlarmUnit["MICRO_SECONDS"] = "MICRO_SECONDS";
        /**
         * MILLI_SECONDS
         */
        CloudWatchAlarmUnit["MILLI_SECONDS"] = "MILLI_SECONDS";
        /**
         * BYTES
         */
        CloudWatchAlarmUnit["BYTES"] = "BYTES";
        /**
         * KILO_BYTES
         */
        CloudWatchAlarmUnit["KILO_BYTES"] = "KILO_BYTES";
        /**
         * MEGA_BYTES
         */
        CloudWatchAlarmUnit["MEGA_BYTES"] = "MEGA_BYTES";
        /**
         * GIGA_BYTES
         */
        CloudWatchAlarmUnit["GIGA_BYTES"] = "GIGA_BYTES";
        /**
         * TERA_BYTES
         */
        CloudWatchAlarmUnit["TERA_BYTES"] = "TERA_BYTES";
        /**
         * BITS
         */
        CloudWatchAlarmUnit["BITS"] = "BITS";
        /**
         * KILO_BITS
         */
        CloudWatchAlarmUnit["KILO_BITS"] = "KILO_BITS";
        /**
         * MEGA_BITS
         */
        CloudWatchAlarmUnit["MEGA_BITS"] = "MEGA_BITS";
        /**
         * GIGA_BITS
         */
        CloudWatchAlarmUnit["GIGA_BITS"] = "GIGA_BITS";
        /**
         * TERA_BITS
         */
        CloudWatchAlarmUnit["TERA_BITS"] = "TERA_BITS";
        /**
         * PERCENT
         */
        CloudWatchAlarmUnit["PERCENT"] = "PERCENT";
        /**
         * COUNT
         */
        CloudWatchAlarmUnit["COUNT"] = "COUNT";
        /**
         * BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["BYTES_PER_SECOND"] = "BYTES_PER_SECOND";
        /**
         * KILO_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BYTES_PER_SECOND"] = "KILO_BYTES_PER_SECOND";
        /**
         * MEGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BYTES_PER_SECOND"] = "MEGA_BYTES_PER_SECOND";
        /**
         * GIGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BYTES_PER_SECOND"] = "GIGA_BYTES_PER_SECOND";
        /**
         * TERA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BYTES_PER_SECOND"] = "TERA_BYTES_PER_SECOND";
        /**
         * BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["BITS_PER_SECOND"] = "BITS_PER_SECOND";
        /**
         * KILO_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BITS_PER_SECOND"] = "KILO_BITS_PER_SECOND";
        /**
         * MEGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BITS_PER_SECOND"] = "MEGA_BITS_PER_SECOND";
        /**
         * GIGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BITS_PER_SECOND"] = "GIGA_BITS_PER_SECOND";
        /**
         * TERA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BITS_PER_SECOND"] = "TERA_BITS_PER_SECOND";
        /**
         * COUNT_PER_SECOND
         */
        CloudWatchAlarmUnit["COUNT_PER_SECOND"] = "COUNT_PER_SECOND";
    })(CloudWatchAlarmUnit = EmrCreateCluster.CloudWatchAlarmUnit || (EmrCreateCluster.CloudWatchAlarmUnit = {}));
    /**
     * Render the MetricDimensionProperty as JSON
     *
     * @param property
     */
    function MetricDimensionPropertyToJson(property) {
        return {
            Key: cdk.stringToCloudFormation(property.key),
            Value: cdk.stringToCloudFormation(property.value)
        };
    }
    EmrCreateCluster.MetricDimensionPropertyToJson = MetricDimensionPropertyToJson;
    /**
     * Render the ScalingTriggerProperty to JSON
     *
     * @param property
     */
    function ScalingTriggerPropertyToJson(property) {
        var _a, _b, _c;
        return {
            CloudWatchAlarmDefinition: {
                ComparisonOperator: cdk.stringToCloudFormation((_a = property.cloudWatchAlarmDefinition.comparisonOperator) === null || _a === void 0 ? void 0 : _a.valueOf()),
                Dimensions: cdk.listMapper(MetricDimensionPropertyToJson)(property.cloudWatchAlarmDefinition.dimensions),
                EvaluationPeriods: cdk.numberToCloudFormation(property.cloudWatchAlarmDefinition.evalutionPeriods),
                MetricName: cdk.stringToCloudFormation(property.cloudWatchAlarmDefinition.metricName),
                Namespace: cdk.stringToCloudFormation(property.cloudWatchAlarmDefinition.namespace),
                Period: cdk.numberToCloudFormation(property.cloudWatchAlarmDefinition.period.toSeconds()),
                Statistic: cdk.stringToCloudFormation((_b = property.cloudWatchAlarmDefinition.statistic) === null || _b === void 0 ? void 0 : _b.valueOf()),
                Threshold: cdk.numberToCloudFormation(property.cloudWatchAlarmDefinition.threshold),
                Unit: cdk.stringToCloudFormation((_c = property.cloudWatchAlarmDefinition.unit) === null || _c === void 0 ? void 0 : _c.valueOf())
            }
        };
    }
    EmrCreateCluster.ScalingTriggerPropertyToJson = ScalingTriggerPropertyToJson;
    /**
     * EC2 Instance Market
     *
     * @experimental
     */
    let InstanceMarket;
    (function (InstanceMarket) {
        /**
         * On Demand Instance
         */
        InstanceMarket["ON_DEMAND"] = "ON_DEMAND";
        /**
         * Spot Instance
         */
        InstanceMarket["SPOT"] = "SPOT";
    })(InstanceMarket = EmrCreateCluster.InstanceMarket || (EmrCreateCluster.InstanceMarket = {}));
    /**
     * AutoScaling Adjustment Type
     *
     * @experimental
     */
    let ScalingAdjustmentType;
    (function (ScalingAdjustmentType) {
        /**
         * CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["CHANGE_IN_CAPACITY"] = "CHANGE_IN_CAPACITY";
        /**
         * PERCENT_CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PERCENT_CHANGE_IN_CAPACITY";
        /**
         * EXACT_CAPACITY
         */
        ScalingAdjustmentType["EXACT_CAPACITY"] = "EXACT_CAPACITY";
    })(ScalingAdjustmentType = EmrCreateCluster.ScalingAdjustmentType || (EmrCreateCluster.ScalingAdjustmentType = {}));
    /**
     * Render the ScalingActionPropety to JSON
     *
     * @param property
     */
    function ScalingActionPropertyToJson(property) {
        var _a;
        return {
            Market: cdk.stringToCloudFormation((_a = property.market) === null || _a === void 0 ? void 0 : _a.valueOf()),
            SimpleScalingPolicyConfiguration: {
                AdjustmentType: cdk.stringToCloudFormation(property.simpleScalingPolicyConfiguration.adjustmentType),
                CoolDown: cdk.numberToCloudFormation(property.simpleScalingPolicyConfiguration.coolDown),
                ScalingAdjustment: cdk.numberToCloudFormation(property.simpleScalingPolicyConfiguration.scalingAdjustment)
            }
        };
    }
    EmrCreateCluster.ScalingActionPropertyToJson = ScalingActionPropertyToJson;
    /**
     * Render the ScalingRuleProperty to JSON
     *
     * @param property
     */
    function ScalingRulePropertyToJson(property) {
        return {
            Action: ScalingActionPropertyToJson(property.action),
            Description: cdk.stringToCloudFormation(property.description),
            Name: cdk.stringToCloudFormation(property.name),
            Trigger: ScalingTriggerPropertyToJson(property.trigger)
        };
    }
    EmrCreateCluster.ScalingRulePropertyToJson = ScalingRulePropertyToJson;
    /**
     * Render the AutoScalingPolicyProperty to JSON
     *
     * @param property
     */
    function AutoScalingPolicyPropertyToJson(property) {
        return {
            Constraints: {
                MaxCapacity: cdk.numberToCloudFormation(property.constraints.maxCapacity),
                MinCapacity: cdk.numberToCloudFormation(property.constraints.minCapacity)
            },
            Rules: cdk.listMapper(ScalingRulePropertyToJson)(property.rules)
        };
    }
    EmrCreateCluster.AutoScalingPolicyPropertyToJson = AutoScalingPolicyPropertyToJson;
    /**
     * Render the InstanceGroupConfigProperty to JSON
     *
     * @param property
     */
    function InstanceGroupConfigPropertyToJson(property) {
        var _a, _b;
        return {
            AutoScalingPolicy: (property.autoScalingPolicy === undefined) ?
                property.autoScalingPolicy :
                AutoScalingPolicyPropertyToJson(property.autoScalingPolicy),
            BidPrice: cdk.numberToCloudFormation(property.bidPrice),
            Configurations: cdk.listMapper(ConfigurationPropertyToJson)(property.configurations),
            EbsConfiguration: (property.ebsConfiguration === undefined) ?
                property.ebsConfiguration :
                EbsConfigurationPropertyToJson(property.ebsConfiguration),
            InstanceCount: cdk.numberToCloudFormation(property.instanceCount),
            InstanceRole: cdk.stringToCloudFormation((_a = property.instanceRole) === null || _a === void 0 ? void 0 : _a.valueOf()),
            InstanceType: cdk.stringToCloudFormation(property.instanceType),
            Market: cdk.stringToCloudFormation((_b = property.market) === null || _b === void 0 ? void 0 : _b.valueOf()),
            Name: cdk.stringToCloudFormation(property.name)
        };
    }
    EmrCreateCluster.InstanceGroupConfigPropertyToJson = InstanceGroupConfigPropertyToJson;
    /**
     * Render the PlacementTypeProperty to JSON
     *
     * @param property
     */
    function PlacementTypePropertyToJson(property) {
        return {
            AvailabilityZone: cdk.stringToCloudFormation(property.availabilityZone),
            AvailabilityZones: cdk.listMapper(cdk.stringToCloudFormation)(property.availabilityZones)
        };
    }
    EmrCreateCluster.PlacementTypePropertyToJson = PlacementTypePropertyToJson;
    /**
     * Render the InstancesConfigProperty to JSON
     *
     * @param property
     */
    function InstancesConfigPropertyToJson(property) {
        return {
            AdditionalMasterSecurityGroups: cdk.listMapper(cdk.stringToCloudFormation)(property.additionalMasterSecurityGroups),
            AdditionalSlaveSecurityGroups: cdk.listMapper(cdk.stringToCloudFormation)(property.additionalSlaveSecurityGroups),
            Ec2KeyName: cdk.stringToCloudFormation(property.ec2KeyName),
            Ec2SubnetId: cdk.stringToCloudFormation(property.ec2SubnetId),
            Ec2SubnetIds: cdk.listMapper(cdk.stringToCloudFormation)(property.ec2SubnetIds),
            EmrManagedMasterSecurityGroup: cdk.stringToCloudFormation(property.emrManagedMasterSecurityGroup),
            EmrManagedSlaveSecurityGroup: cdk.stringToCloudFormation(property.emrManagedSlaveSecurityGroup),
            HadoopVersion: cdk.stringToCloudFormation(property.hadoopVersion),
            InstanceCount: cdk.numberToCloudFormation(property.instanceCount),
            InstanceFleets: cdk.listMapper(InstanceFleetConfigPropertyToJson)(property.instanceFleets),
            InstanceGroups: cdk.listMapper(InstanceGroupConfigPropertyToJson)(property.instanceGroups),
            KeepJobFlowAliveWhenNoSteps: true,
            MasterInstanceType: cdk.stringToCloudFormation(property.masterInstanceType),
            Placement: (property.placement === undefined) ?
                property.placement :
                PlacementTypePropertyToJson(property.placement),
            ServiceAccessSecurityGroup: cdk.stringToCloudFormation(property.serviceAccessSecurityGroup),
            SlaveInstanceType: cdk.stringToCloudFormation(property.slaveInstanceType),
            TerminationProtected: cdk.booleanToCloudFormation(property.terminationProtected)
        };
    }
    EmrCreateCluster.InstancesConfigPropertyToJson = InstancesConfigPropertyToJson;
    /**
     * Render the ApplicationConfigProperty as JSON
     *
     * @param property
     */
    function ApplicationConfigPropertyToJson(property) {
        return {
            Name: cdk.stringToCloudFormation(property.name),
            Args: cdk.listMapper(cdk.stringToCloudFormation)(property.args),
            Version: cdk.stringToCloudFormation(property.version),
            AdditionalInfo: cdk.objectToCloudFormation(property.additionalInfo)
        };
    }
    EmrCreateCluster.ApplicationConfigPropertyToJson = ApplicationConfigPropertyToJson;
    /**
     * Render the BootstrapActionProperty as JSON
     *
     * @param property
     */
    function BootstrapActionConfigToJson(property) {
        return {
            Name: cdk.stringToCloudFormation(property.name),
            ScriptBootstrapAction: {
                Path: cdk.stringToCloudFormation(property.scriptBootstrapAction.path),
                Args: cdk.listMapper(cdk.stringToCloudFormation)(property.scriptBootstrapAction.args)
            }
        };
    }
    EmrCreateCluster.BootstrapActionConfigToJson = BootstrapActionConfigToJson;
    /**
     * Render the ConfigurationProperty as JSON
     *
     * @param property
     */
    function ConfigurationPropertyToJson(property) {
        return {
            Classification: cdk.stringToCloudFormation(property.classification),
            Properties: cdk.objectToCloudFormation(property.properties),
            Configurations: cdk.listMapper(ConfigurationPropertyToJson)(property.configurations)
        };
    }
    EmrCreateCluster.ConfigurationPropertyToJson = ConfigurationPropertyToJson;
    /**
     * Render the KerberosAttributesProperty as JSON
     *
     * @param property
     */
    function KerberosAttributesPropertyToJson(property) {
        return {
            ADDomainJoinPassword: cdk.stringToCloudFormation(property.adDomainJoinPassword),
            ADDomainJoinUser: cdk.stringToCloudFormation(property.adDomainJoinUser),
            CrossRealmTrustPrincipalPassword: cdk.stringToCloudFormation(property.crossRealmTrustPrincipalPassword),
            KdcAdminPassword: cdk.stringToCloudFormation(property.kdcAdminPassword),
            Realm: cdk.stringToCloudFormation(property.realm)
        };
    }
    EmrCreateCluster.KerberosAttributesPropertyToJson = KerberosAttributesPropertyToJson;
})(EmrCreateCluster = exports.EmrCreateCluster || (exports.EmrCreateCluster = {}));
//# sourceMappingURL=data:application/json;base64,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