import re
from dataclasses import dataclass, field
from enum import Enum, auto
from inspect import getfile
from pathlib import Path
from typing import Iterable, List, Optional, Type, Union, Dict, Any

from amora.config import settings
from sqlalchemy import MetaData
from sqlmodel import SQLModel, Field, select
from sqlmodel.sql.expression import Select, SelectOfScalar
from sqlalchemy.orm import declared_attr


Compilable = Union[Select, SelectOfScalar]
select = select
Field = Field


@dataclass
class PartitionConfig:
    field: str
    data_type: str = "date"
    granularity: str = "day"
    range: Optional[Dict[str, Any]] = None


class AutoName(Enum):
    def _generate_next_value_(name, start, count, last_values):
        return name


class MaterializationTypes(AutoName):
    ephemeral = auto()
    view = auto()
    table = auto()


@dataclass
class ModelConfig:
    description: str = "Undocumented! Generated by Amora Data Build Tool 💚"
    materialized: MaterializationTypes = MaterializationTypes.view
    partition_by: Optional[PartitionConfig] = None
    cluster_by: List[str] = field(default_factory=list)
    labels: Dict[str, str] = field(default_factory=dict)


def list_files(path: Union[str, Path], suffix: str) -> Iterable[Path]:
    yield from Path(path).rglob(f"*{suffix}")


def list_model_files() -> Iterable[Path]:
    return list_files(settings.MODELS_PATH, suffix=".py")


def list_amora_model_files() -> Iterable[Path]:
    raise NotImplementedError


def list_target_files() -> Iterable[Path]:
    return list_files(settings.TARGET_PATH, suffix=".sql")


# todo: Como defino "Objeto que tenha 2 atributos: `source` e `output`" ?
def is_py_model(obj) -> bool:
    return hasattr(obj, "source") and hasattr(obj, "output")


metadata = MetaData(
    schema=f"{settings.TARGET_PROJECT}.{settings.TARGET_SCHEMA}"
)


class AmoraModel(SQLModel):
    __depends_on__: List["AmoraModel"] = []
    __model_config__ = ModelConfig(materialized=MaterializationTypes.view)
    __table_args__ = {"extend_existing": True}
    metadata = metadata

    @declared_attr  # type: ignore
    def __tablename__(cls) -> str:  # type: ignore
        """
        By default, `__tablename__` is the `snake_case` class name.

        ```python
        class MyModel(AmoraModel): ...

        assert MyModel.__tablename__ == 'my_model
        ```
        """
        return re.sub(r"(?<!^)(?=[A-Z])", "_", cls.__name__).lower()

    @classmethod
    def dependencies(cls) -> Iterable["AmoraModel"]:
        source = cls.source()
        if source is None:
            return []

        # todo: Remover necessidade de __depends_on__ inspecionando a query e chegando ao modelo de origem
        # tables: List[Table] = source.froms

        return cls.__depends_on__

    @classmethod
    def source(cls) -> Optional[Compilable]:
        """
        Called when `amora compile` is executed, Amora will build this model
        in your data warehouse by wrapping it in a `create view as` or `create table as` statement.

        Return `None` for defining models for tables/views that already exist on the data warehouse
        and shouldn't be managed by Amora.

        Returning a `Compilable`, which is a sqlalchemy select statement
        :return:
        """
        return None

    @classmethod
    def target_path(cls, model_file_path: Union[str, Path]) -> Path:
        # {settings.dbt_models_path}/a_model/a_model.py -> a_model/a_model.py
        strip_path = settings.MODELS_PATH
        relative_model_path = str(model_file_path).split(strip_path)[1][1:]
        # a_model/a_model.py -> ~/project/amora/target/a_model/a_model.sql
        target_file_path = Path(settings.TARGET_PATH).joinpath(
            relative_model_path.replace(".py", ".sql")
        )

        return target_file_path

    @classmethod
    def model_file_path(cls) -> Path:
        return Path(getfile(cls))

    @classmethod
    @property
    def unique_name(cls) -> str:
        return str(cls.__table__)
