"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeDeployEcsDeployAction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const action_1 = require("../action");
/**
 * @stability stable
 */
class CodeDeployEcsDeployAction extends action_1.Action {
    /**
     * @stability stable
     */
    constructor(props) {
        const inputs = [];
        inputs.push(determineTaskDefinitionArtifact(props));
        inputs.push(determineAppSpecArtifact(props));
        if (props.containerImageInputs) {
            if (props.containerImageInputs.length > 4) {
                throw new Error(`Action cannot have more than 4 container image inputs, got: ${props.containerImageInputs.length}`);
            }
            for (const imageInput of props.containerImageInputs) {
                inputs.push(imageInput.input);
            }
        }
        super({
            ...props,
            resource: props.deploymentGroup,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'CodeDeployToECS',
            artifactBounds: { minInputs: 1, maxInputs: 5, minOutputs: 0, maxOutputs: 0 },
            inputs,
        });
        this.actionProps = props;
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @stability stable
     */
    bound(_scope, _stage, options) {
        // permissions, based on:
        // https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.application.applicationArn],
            actions: ['codedeploy:GetApplication', 'codedeploy:GetApplicationRevision', 'codedeploy:RegisterApplicationRevision'],
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.deploymentGroupArn],
            actions: ['codedeploy:CreateDeployment', 'codedeploy:GetDeployment'],
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.deploymentConfig.deploymentConfigArn],
            actions: ['codedeploy:GetDeploymentConfig'],
        }));
        // Allow action to register the task definition template file with ECS
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['ecs:RegisterTaskDefinition'],
        }));
        // Allow passing any roles specified in the task definition template file to ECS
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: ['*'],
            conditions: {
                StringEqualsIfExists: {
                    'iam:PassedToService': [
                        'ecs-tasks.amazonaws.com',
                    ],
                },
            },
        }));
        // the Action's Role needs to read from the Bucket to get artifacts
        options.bucket.grantRead(options.role);
        const taskDefinitionTemplateArtifact = determineTaskDefinitionArtifact(this.actionProps);
        const appSpecTemplateArtifact = determineAppSpecArtifact(this.actionProps);
        const actionConfig = {
            configuration: {
                ApplicationName: this.actionProps.deploymentGroup.application.applicationName,
                DeploymentGroupName: this.actionProps.deploymentGroup.deploymentGroupName,
                TaskDefinitionTemplateArtifact: core_1.Lazy.string({ produce: () => taskDefinitionTemplateArtifact.artifactName }),
                TaskDefinitionTemplatePath: this.actionProps.taskDefinitionTemplateFile
                    ? this.actionProps.taskDefinitionTemplateFile.fileName
                    : 'taskdef.json',
                AppSpecTemplateArtifact: core_1.Lazy.string({ produce: () => appSpecTemplateArtifact.artifactName }),
                AppSpecTemplatePath: this.actionProps.appSpecTemplateFile
                    ? this.actionProps.appSpecTemplateFile.fileName
                    : 'appspec.yaml',
            },
        };
        if (this.actionProps.containerImageInputs) {
            for (let i = 1; i <= this.actionProps.containerImageInputs.length; i++) {
                const imageInput = this.actionProps.containerImageInputs[i - 1];
                actionConfig.configuration[`Image${i}ArtifactName`] = core_1.Lazy.string({ produce: () => imageInput.input.artifactName });
                actionConfig.configuration[`Image${i}ContainerName`] = imageInput.taskDefinitionPlaceholder
                    ? imageInput.taskDefinitionPlaceholder
                    : 'IMAGE';
            }
        }
        return actionConfig;
    }
}
exports.CodeDeployEcsDeployAction = CodeDeployEcsDeployAction;
_a = JSII_RTTI_SYMBOL_1;
CodeDeployEcsDeployAction[_a] = { fqn: "@aws-cdk/aws-codepipeline-actions.CodeDeployEcsDeployAction", version: "1.93.0" };
function determineTaskDefinitionArtifact(props) {
    if (props.taskDefinitionTemplateFile && props.taskDefinitionTemplateInput) {
        throw new Error("Exactly one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' can be provided in the ECS CodeDeploy Action");
    }
    if (props.taskDefinitionTemplateFile) {
        return props.taskDefinitionTemplateFile.artifact;
    }
    if (props.taskDefinitionTemplateInput) {
        return props.taskDefinitionTemplateInput;
    }
    throw new Error("Specifying one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' is required for the ECS CodeDeploy Action");
}
function determineAppSpecArtifact(props) {
    if (props.appSpecTemplateFile && props.appSpecTemplateInput) {
        throw new Error("Exactly one of 'appSpecTemplateInput' or 'appSpecTemplateFile' can be provided in the ECS CodeDeploy Action");
    }
    if (props.appSpecTemplateFile) {
        return props.appSpecTemplateFile.artifact;
    }
    if (props.appSpecTemplateInput) {
        return props.appSpecTemplateInput;
    }
    throw new Error("Specifying one of 'appSpecTemplateInput' or 'appSpecTemplateFile' is required for the ECS CodeDeploy Action");
}
//# sourceMappingURL=data:application/json;base64,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