import web3
from unittest import TestCase
from py_order_utils.facades import Erc20Facade, Erc1155Facade , LimitOrderProtocolFacade


class TestFacade(TestCase):

    def setUp(self):
        self.erc20_facade = Erc20Facade()
        self.erc1155_facade = Erc1155Facade()
        self.lop_facade = LimitOrderProtocolFacade()
    
    def test_erc20_balance_of(self):
        address = "0xAb5801a7D398351b8bE11C439e05C5B3259aeC9B"
        # Expected input data generated using ethers
        expected = "0x70a08231000000000000000000000000ab5801a7d398351b8be11c439e05c5b3259aec9b"

        self.assertEqual(expected, self.erc20_facade.balance_of(address))

    def test_erc20_transfer_from(self):
        from_address = "0xAb5801a7D398351b8bE11C439e05C5B3259aeC9B"
        to_address = "0x1Db3439a222C519ab44bb1144fC28167b4Fa6EE6"
        value = web3.Web3.toWei(100, 'ether')
        # Expected input data generated using ethers
        expected = "0x23b872dd000000000000000000000000ab5801a7d398351b8be11c439e05c5b3259aec9b0000000000000000000000001db3439a222c519ab44bb1144fc28167b4fa6ee60000000000000000000000000000000000000000000000056bc75e2d63100000"
        
        self.assertEqual(expected, self.erc20_facade.transfer_from(from_address, to_address, value))

    def test_erc1155_balance_of(self):
        address = "0xAb5801a7D398351b8bE11C439e05C5B3259aeC9B"
        token_id = 0
        # Expected input data generated using ethers
        expected = "0x00fdd58e000000000000000000000000ab5801a7d398351b8be11c439e05c5b3259aec9b0000000000000000000000000000000000000000000000000000000000000000"

        self.assertEqual(expected, self.erc1155_facade.balance_of(address, token_id))

    def test_erc1155_transfer_from(self):
        from_address = "0xAb5801a7D398351b8bE11C439e05C5B3259aeC9B"
        to_address = "0x1Db3439a222C519ab44bb1144fC28167b4Fa6EE6"
        value = web3.Web3.toWei(100, 'ether')
        token_address = "0xadbeD21409324e0fcB80AE8b5e662B0C857D85ed"
        token_id = 0
        
        # Expected input data generated using ethers
        expected = "0x23b872e1000000000000000000000000ab5801a7d398351b8be11c439e05c5b3259aec9b0000000000000000000000001db3439a222c519ab44bb1144fc28167b4fa6ee60000000000000000000000000000000000000000000000056bc75e2d63100000000000000000000000000000adbed21409324e0fcb80ae8b5e662b0c857d85ed000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000c000000000000000000000000000000000000000000000000000000000000000010000000000000000000000000000000000000000000000000000000000000000"
        self.assertEqual(expected, self.erc1155_facade.transfer_from(from_address, to_address, token_address, token_id, value))

    def test_lop_and(self):
        predicates=[
            "0x63592c2b0000000000000000000000000000000000000000000000000000000061f3f65e", 
            "0xcf6fc6e300000000000000000000000070997970c51812dc3a010c7d01b50e0d17dc79c80000000000000000000000000000000000000000000000000000000000000000"
        ]

        expected = "0x961d5b1e000000000000000000000000000000000000000000000000000000000000004000000000000000000000000000000000000000000000000000000000000000a00000000000000000000000000000000000000000000000000000000000000002000000000000000000000000cf7ed3acca5a467e9e704c703e8d87f634fb0fc9000000000000000000000000cf7ed3acca5a467e9e704c703e8d87f634fb0fc90000000000000000000000000000000000000000000000000000000000000002000000000000000000000000000000000000000000000000000000000000004000000000000000000000000000000000000000000000000000000000000000a0000000000000000000000000000000000000000000000000000000000000002463592c2b0000000000000000000000000000000000000000000000000000000061f3f65e000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000044cf6fc6e300000000000000000000000070997970c51812dc3a010c7d01b50e0d17dc79c8000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        self.assertEqual(expected, self.lop_facade.lop_and("0xCf7Ed3AccA5a467e9e704C703E8D87F634fB0Fc9", predicates))

    def test_lop_nonce_equals(self):
        maker_address = "0x70997970C51812dc3A010C7d01b50e0d17dc79C8"
        maker_nonce = 0
        expected = "0xcf6fc6e300000000000000000000000070997970c51812dc3a010c7d01b50e0d17dc79c80000000000000000000000000000000000000000000000000000000000000000"
        self.assertEqual(expected, self.lop_facade.nonce_equals(maker_address, maker_nonce))

    def test_lop_ts_below(self):
        timestamp = 1643379170
        expected = "0x63592c2b0000000000000000000000000000000000000000000000000000000061f3f9e2"
        self.assertEqual(expected, self.lop_facade.timestamp_below(timestamp))
    
    def test_lop_get_maker_amount(self):
        maker_amount = 1000000
        taker_amount = 500000

        expected = "0xf4a215c300000000000000000000000000000000000000000000000000000000000f4240000000000000000000000000000000000000000000000000000000000007a120"

        self.assertEqual(expected, self.lop_facade.get_maker_amount_data(maker_amount, taker_amount))


    def test_lop_get_taker_amount(self):
        maker_amount = 1000000
        taker_amount = 500000
        expected = "0x296637bf00000000000000000000000000000000000000000000000000000000000f4240000000000000000000000000000000000000000000000000000000000007a120"

        self.assertEqual(expected, self.lop_facade.get_taker_amount_data(maker_amount, taker_amount))
        