
import mimetypes
from pathlib import Path
from typing import Iterator, Dict, Any, List

from .config import Config
from .modes.full import read_full_content
from .modes.python_parser import extract_ast_info

def _is_excluded(file_path: Path, root_path: Path, exclude_patterns: List[str]) -> bool:
    """Checks if a file_path should be excluded based on patterns."""
    relative_path = file_path.relative_to(root_path)
    for pattern in exclude_patterns:
        match_file = file_path.match(pattern)
        match_relative = relative_path.match(pattern)
        if match_file or match_relative:
            return True
    return False

def determine_mode(file_path: Path, config: Config) -> str:
    """Determines the processing mode for a file based on configuration."""
    if file_path in config.full_exceptions:
        return "full"
    if file_path in config.ast_exceptions:
        return "ast"

    if config.default_mode == "ast" and file_path.suffix == ".py":
        return "ast"
    
    if file_path.suffix != ".py":
        return "full"

    return config.default_mode

def scan_directory(path: Path, config: Config) -> Iterator[Dict[str, Any]]:
    """Scans a directory and yields information about each file."""
    for file_path in path.rglob("*"):
        if not file_path.is_file():
            continue

        if _is_excluded(file_path, path, config.exclude_patterns):
            continue

        if file_path.suffix not in config.ext:
            continue

        mime_type, _ = mimetypes.guess_type(file_path)
        if mime_type and not (mime_type.startswith("text/") or mime_type == "application/json"):
            continue

        mode = determine_mode(file_path, config)
        content: Any
        if mode == "ast":
            content = extract_ast_info(file_path)
        else:
            content = read_full_content(file_path, config.max_file_size)

        yield {
            "path": file_path,
            "size": file_path.stat().st_size,
            "mime": mime_type or "unknown",
            "mode": mode,
            "content": content,
        }
