
import ast
from pathlib import Path
from typing import Dict, Any, List

def _format_annotation(node) -> str:
    if node is None:
        return ""
    if isinstance(node, ast.Name):
        return node.id
    if isinstance(node, ast.Constant) and node.value is None:
        return "None"
    if hasattr(ast, 'unparse'):
        return ast.unparse(node).strip().replace('\n', ' ')
    return "..."

def extract_ast_info(file_path: Path) -> Dict[str, Any]:
    """Extracts functions, classes, and imports from a Python file using AST."""
    try:
        content = file_path.read_text(encoding="utf-8")
        tree = ast.parse(content)
    except Exception as e:
        return {"error": f"Failed to parse AST: {e}"}

    functions = []
    classes = []
    imports: List[str] = []

    for node in tree.body:
        if isinstance(node, (ast.Import, ast.ImportFrom)):
            if isinstance(node, ast.Import):
                imports.extend(alias.name for alias in node.names)
            else: # ImportFrom
                module = node.module or "."
                imports.extend(f"{module}.{alias.name}" for alias in node.names)
        elif isinstance(node, ast.FunctionDef):
            functions.append({
                "name": node.name,
                "line": node.lineno,
                "args": [{"name": arg.arg, "annotation": _format_annotation(arg.annotation)} for arg in node.args.args],
                "returns": _format_annotation(node.returns),
                "docstring": ast.get_docstring(node),
            })
        elif isinstance(node, ast.ClassDef):
            methods = []
            for item in node.body:
                if isinstance(item, ast.FunctionDef):
                    methods.append({
                        "name": item.name,
                        "args": [{"name": arg.arg, "annotation": _format_annotation(arg.annotation)} for arg in item.args.args],
                        "returns": _format_annotation(item.returns),
                    })
            classes.append({
                "name": node.name, 
                "line": node.lineno, 
                "methods": methods
            })

    return {"functions": functions, "classes": classes, "imports": sorted(list(set(imports)))}
