from pathlib import Path
from typing import List, Dict

def to_tree(paths: List[Path], root_dir: Path) -> str:
    """Generates an ASCII art tree structure from a list of paths."""
    if not paths:
        return ""

    tree_lines = []
    
    # Build a dictionary representing the tree structure
    tree_dict: Dict[str, Dict] = {}

    # Sort paths to ensure consistent output and correct tree structure
    relative_paths = [p.relative_to(root_dir) for p in paths]
    sorted_paths = sorted(relative_paths, key=lambda p: str(p).lower())

    for path in sorted_paths:
        parts = path.parts
        current_level = tree_dict
        for i, part in enumerate(parts):
            if part not in current_level:
                current_level[part] = {}
            current_level = current_level[part]

    def _build_tree_lines(current_dict: Dict[str, Dict], prefix=""):
        # Sort keys for consistent output
        keys = sorted(current_dict.keys())
        
        for i, key in enumerate(keys):
            is_last_item_in_level = (i == len(keys) - 1)
            connector = "└── " if is_last_item_in_level else "├── "
            
            is_empty_dict = not current_dict[key]
            
            if is_empty_dict: # It's a file
                tree_lines.append(f"{prefix}{connector}{key}")
            else: # It's a directory
                tree_lines.append(f"{prefix}{connector}{key}/")
                extension_prefix = "    " if is_last_item_in_level else "│   "
                _build_tree_lines(current_dict[key], prefix + extension_prefix)
    
    # Add the root directory name as the first line of the tree
    if root_dir.name and root_dir.name != ".":
        tree_lines.append(f"{root_dir.name}/")
    else:
        tree_lines.append("./")
    
    _build_tree_lines(tree_dict)
    return "\n".join(tree_lines)
