
from pathlib import Path
from typing import Iterable, Dict, Any


def _format_ast_content(content: Dict[str, Any]) -> str:
    """Formats AST data into a markdown string."""
    if content.get("error"):
        return f"- **Error:** {content['error']}"

    lines = []
    if content.get("imports"):
        lines.append("- **Imports:** `" + ", ".join(content["imports"]) + "`")

    if content.get("classes"):
        lines.append("- **Classes:**")
        for c in content["classes"]:
            method_names = ", ".join([m["name"] for m in c["methods"]])
            lines.append(f"  - `{c['name']}` (Methods: {method_names})")

    if content.get("functions"):
        lines.append("- **Functions:**")
        for f in content["functions"]:
            args_str = ", ".join([f'{arg["name"]}: {arg["annotation"]}' if arg["annotation"] else arg["name"] for arg in f["args"]])
            returns_str = f" -> {f['returns']}" if f['returns'] else ""
            lines.append(f"  - `{f['name']}({args_str}){returns_str}")
            if f.get("docstring"):
                lines.append(f"    > {f['docstring'].splitlines()[0]}")

    return "\n".join(lines)

def to_markdown(documents: Iterable[Dict[str, Any]], root_dir: Path) -> str:
    """Converts a list of file information dicts to a Markdown string."""
    output = ["# 🧩 Project Tree (generated by projext v0.1.0)\n"]

    current_dir = None

    for doc in sorted(documents, key=lambda x: x["path"]):
        file_path = doc["path"]
        relative_path = file_path.relative_to(root_dir)

        if relative_path.parent != current_dir:
            current_dir = relative_path.parent
            output.append(f"## 📁 {current_dir}/\n")

        output.append(f"### 📄 {file_path.name} ({doc['size']} bytes) [mode: {doc['mode']}]\n")

        if doc["mode"] == "ast":
            output.append(_format_ast_content(doc["content"]))
        else:
            lang_map = {"py": "python", "js": "javascript", "ts": "typescript", "md": "markdown"}
            lang_ext = file_path.suffix.lstrip(".")
            lang = lang_map.get(lang_ext, lang_ext)
            output.append(f"```{lang}\n{doc['content']}\n```")
        output.append("\n")

    return "\n".join(output)
