
import click
from pathlib import Path

from . import config
from .scanner import scan_directory, _is_excluded
from .exporters.markdown import to_markdown
from .exporters.json_exporter import to_json
from .exporters.tree import to_tree


@click.group()
def main():
    """A CLI tool to generate a project map."""
    pass


@main.command()  # type: ignore[no-redef]
@click.argument("path", type=click.Path(exists=True, file_okay=False, path_type=Path))
@click.option("--ext", default=".py,.json,.md", help="Comma-separated list of file extensions to include.")
@click.option("--out", default="project_tree.md", help="Output file name.")
@click.option("--format", type=click.Choice(["md", "json"]), default="md", help="Output format.")
@click.option("--max-file-size", default="1MB", help="Maximum file size to include (e.g., 1MB, 2K).")
@click.option("--mode", type=click.Choice(["full", "ast"]), default="full", help="Default output mode.")
@click.option("--full", "full_exceptions", multiple=True, type=click.Path(path_type=Path), help="Files/paths to force full content output.")
@click.option("--ast", "ast_exceptions", multiple=True, type=click.Path(path_type=Path), help="Files/paths to force AST extraction.")
@click.option("--exclude", "exclude_patterns", multiple=True, help="Glob patterns to exclude from the scan.")
def map(
    path: Path,
    ext: str,
    out: str,
    format: str,
    max_file_size: str,
    mode: str,
    full_exceptions: tuple[Path, ...],
    ast_exceptions: tuple[Path, ...],
    exclude_patterns: tuple[str, ...],
):
    """Generate a project map from a directory."""
    try:
        size_in_bytes = config.parse_size(max_file_size)
    except ValueError as e:
        click.echo(f"Error: {e}", err=True)
        return

    cfg = config.Config(
        ext=ext.split(","),
        out=Path(out),
        format=format,
        max_file_size=size_in_bytes,
        default_mode=mode,
        full_exceptions=[path / p for p in full_exceptions],
        ast_exceptions=[path / p for p in ast_exceptions],
        exclude_patterns=list(exclude_patterns),
    )

    documents = list(scan_directory(path, cfg))

    if cfg.format == "md":
        output = to_markdown(documents, path)
    elif cfg.format == "json":
        output = to_json(documents, path)
    else:
        click.echo(f"Error: Unknown format {cfg.format}", err=True)
        return

    cfg.out.write_text(output, encoding="utf-8")
    click.echo(f"Project map saved to {cfg.out}")


@main.command()  # type: ignore[no-redef]
@click.argument("path", type=click.Path(exists=True, file_okay=False, path_type=Path))
@click.option("--out", default="project_tree.txt", help="Output file name.")
@click.option("--exclude", "exclude_patterns", multiple=True, help="Glob patterns to exclude from the tree.")
def tree(
    path: Path,
    out: str,
    exclude_patterns: tuple[str, ...],
):
    """Generate a directory tree."""
    cfg = config.Config(
        out=Path(out),
        exclude_patterns=list(exclude_patterns),
    )
    
    # For tree command, we only care about paths, not content or modes
    # so we can use a simplified scan that just yields paths
    
    all_paths = []
    for file_path in path.rglob("*"):
        if not file_path.is_file():
            continue

        if _is_excluded(file_path, path, cfg.exclude_patterns):
            continue
        
        all_paths.append(file_path)

    output = to_tree(all_paths, path)
    cfg.out.write_text(output, encoding="utf-8")
    click.echo(f"Project tree saved to {cfg.out}")


