from pprint import pprint
import uuid
import time
import requests
from .constants import get_api_url, get_api_key, get_docs_path, get_eval_dataset_file
from .utils import read_dataset
from .exceptions import SDKException, MissingEnvironmentVariableException
import os
import threading
from typing import Dict, Any, Optional
import pandas as pd
import json

class EvaluatorSDK:
    def __init__(self):

        print("Evaluator SDK initialized")
        print("API URL:", get_api_url())
        print("API Key:", get_api_key())
        if not get_api_url() or not get_api_key():
            raise MissingEnvironmentVariableException("EVAL_API_URL or EVAL_API_KEY is not set in the environment.")
        
        self.request_data = {}
        self.request_id = None
        self.request_uuid_mapping = {}  # Maps request_id to API request_uuid

    def get_request_data(self, request_id):
        """Retrieves the request data dictionary for a given request_id."""
        if request_id not in self.request_data:
            raise SDKException(f"Request ID {request_id} does not exist.")
        return self.request_data[request_id]

    def add_param(self, request_id, param_name, param_value):
        """Adds a parameter to the request data dictionary."""
        if request_id not in self.request_data:
            self.request_data[request_id] = {"request_uuid": request_id}
        
        self.request_data[request_id][param_name] = param_value
    
    def add_project_code(self, project_code):
        """Adds the project code to the request data."""
        if self.request_id is None:
            raise SDKException("Request ID is not set. Please start a request first.")
        
        self.add_param(self.request_id, 'project_code', project_code)

    def add_prompt(self, prompt):
        """Adds a prompt to the request data."""
        if self.request_id is None:
            raise SDKException("Request ID is not set. Please start a request first.")
        
        self.add_param(self.request_id, 'prompt', prompt)
    
    def add_no_of_docs_in_context(self, no_of_docs):
        """Adds the number of documents in context to the request data."""
        if self.request_id is None:
            raise SDKException("Request ID is not set. Please start a request first.")
        self.add_param(self.request_id, 'no_of_docs_in_context', no_of_docs)

    def add_top_k(self, top_k):
        """Adds the top_k parameter to the request data."""
        if self.request_id is None:
            raise SDKException("Request ID is not set. Please start a request first.")
        
        self.add_param(self.request_id, 'top_k', top_k)

    def add_no_of_docs_retrieved(self, no_of_docs):
        """Adds the number of documents retrieved to the request data."""
        if self.request_id is None:
            raise SDKException("Request ID is not set. Please start a request first.")
        
        self.add_param(self.request_id, 'no_of_docs_retrieved', no_of_docs)
    
    def add_retrieved_ans(self, retrieved_ans):
        """Adds the retrieved answers to the request data."""
        if self.request_id is None:
            raise SDKException("Request ID is not set. Please start a request first.")
        self.add_param(self.request_id, 'retrieved_ans', retrieved_ans)
    
    def add_final_ans(self, final_ans):
        """Adds the final answer to the request data."""
        if self.request_id is None:
            raise SDKException("Request ID is not set. Please start a request first.")
        
        self.add_param(self.request_id, 'final_ans', final_ans)

    def add_ground_truth(self, ground_truth):
        """Adds the ground truth to the request data."""
        if self.request_id is None:
            raise SDKException("Request ID is not set. Please start a request first.")
        
        self.add_param(self.request_id, 'ground_truth', ground_truth)
    
    def add_retrieved_docs_list(self, retrieved_docs_list):
        """Adds the list of retrieved documents to the request data."""
        if self.request_id is None:
            raise SDKException("Request ID is not set. Please start a request first.")
        self.add_param(self.request_id, 'retrieved_docs_list', retrieved_docs_list)
    
    def add_total_query_time(self, total_query_time):
        """Adds the total query time to the request data."""
        if self.request_id is None:
            raise SDKException("Request ID is not set. Please start a request first.")
        
        self.add_param(self.request_id, 'total_query_time', total_query_time)
    
    def add_retrieval_time(self, retrieval_time):
        """Adds the retrieval time to the request data."""
        if self.request_id is None:
            raise SDKException("Request ID is not set. Please start a request first.")
        
        self.add_param(self.request_id, 'retrieval_time', retrieval_time)
    
    def add_generation_time(self, generation_time):
        """Adds the generation time to the request data."""
        if self.request_id is None:
            raise SDKException("Request ID is not set. Please start a request first.")
        self.add_param(self.request_id, 'generation_time', generation_time)
    
    def add_throughput(self, throughput):
        """Adds the throughput to the request data."""
        if self.request_id is None:
            raise SDKException("Request ID is not set. Please start a request first.")
        
        self.add_param(self.request_id, 'throughput', throughput)

    def start_request(self):
        """Generates a unique request_id for a new request."""
        self.request_id = str(uuid.uuid4())
        return self.request_id

    def send(self, request_id):
        """Sends the collected data to the API as a POST request."""
        try:
            url = get_api_url()
            headers = {'Authorization': f"Bearer {get_api_key()}", 'Content-Type': 'application/json'}
            
            data = self.request_data.get(request_id)

            print("Sending data to URL:", url)
            # print("Data being sent:", data)

            if not data:
                raise SDKException("No data found for the provided request_id.")
            
            response = requests.post(url, json=data, headers=headers, timeout=120)
            print("Response from send:", response.json())

            if response.status_code != 200:
                raise SDKException(f"Request failed with status code {response.status_code}: {response.text}")
            return response.json()
        except requests.exceptions.RequestException as e:
            raise SDKException(f"Request failed, An error occurred while sending data: {e}")
        except SDKException as e:
            raise SDKException(f"SDK Exception, An error occurred: {e}")
        except Exception as e:
            raise SDKException(f"Send Exception, An error occurred: {e}")
    
    def send_in_background(self):
        """Sends the collected data to the API in a separate thread."""
        import threading
        req_id = self.request_id
        if req_id is None:
            raise SDKException("Request ID is not set. Please start a request first.")
        thread = threading.Thread(target=self.send, args=(req_id,))
        thread.start()

    def get_status(self, request_id: str) -> str:
        """Gets the status of an evaluation request."""
        try:
            # Get the API request_uuid for this request_id
            request_uuid = self.request_uuid_mapping.get(request_id)
            if not request_uuid:
                # If we don't have the uuid, assume the request was sent directly
                # and use the request_id as the uuid
                request_uuid = request_id
            
            url = f"{get_api_url().rstrip('/')}/get_status"
            headers = {'Authorization': f"Bearer {get_api_key()}", 'Content-Type': 'application/json'}
            
            data = {"request_uuid": request_uuid}
            response = requests.get(url, json=data, headers=headers, timeout=30)
            
            if response.status_code != 200:
                raise SDKException(f"Status request failed with status code {response.status_code}: {response.text}")
            
            response_data = response.json()
            return response_data.get('data', {}).get('status', 'Unknown')
            
        except requests.exceptions.RequestException as e:
            raise SDKException(f"Status request failed: {e}")
        except Exception as e:
            raise SDKException(f"Error getting status: {e}")

    def get_results(self, request_id: str) -> Dict[str, Any]:
        """Gets the results of an evaluation request."""
        try:
            # Get the API request_uuid for this request_id
            print("Get results called with request_id in sdk evaluator:", request_id)
            request_uuid = self.request_uuid_mapping.get(request_id)
            if not request_uuid:
                # If we don't have the uuid, assume the request was sent directly
                request_uuid = request_id
            
            url = f"{get_api_url().rstrip('/')}/results"
            headers = {'Authorization': f"Bearer {get_api_key()}", 'Content-Type': 'application/json'}
            
            data = {"request_uuid": request_uuid}
            response = requests.get(url, json=data, headers=headers, timeout=30)
            
            if response.status_code != 200:
                raise SDKException(f"Results request failed with status code {response.status_code}: {response.text}")

            result = response.json().get("data", {})
            if not result or result.get("status") == "Error":
                # Instead of crashing, return safe fallback
                raise SDKException(f"Evaluation failed: {response.json().get('message')}")
            
            return result
            
        except requests.exceptions.RequestException as e:
            raise SDKException(f"Results request failed: {e}")
        except Exception as e:
            raise SDKException(f"Error getting results: {e}")

    def wait_for_results(self, request_id: str, timeout: int = 300, poll_interval: int = 5) -> Dict[str, Any]:
        """Waits for the evaluation to complete and returns the results."""
        start_time = time.time()
        
        while time.time() - start_time < timeout:
            try:
                status = self.get_status(request_id)
                print(f"Current status for request {request_id} at time: {time.time()}: {status}")

                if status == "Completed":
                    print(f"Evaluation completed for request {request_id}")
                    return self.get_results(request_id)
                elif status  == "Pending":
                    time.sleep(poll_interval)
                    continue
                elif status == "Failed":
                    raise SDKException(f"Evaluation failed for request {request_id}")

                
                time.sleep(poll_interval)
                
            except SDKException:
                raise
            except Exception as e:
                raise SDKException(f"Error while waiting for results: {e}")
        
        raise SDKException(f"Timeout reached while waiting for evaluation to complete")

    def send_and_wait(self, timeout: int = 300) -> Dict[str, Any]:
        """Sends the request and waits for the results synchronously."""

        print("send_and_wait called with request_id in sdk evaluator:", self.request_id, "timeout:", timeout)
        
        # Send the request
        self.send(self.request_id)
        
        # Wait for results
        return self.wait_for_results(self.request_id, timeout)

    def process_dataset(self):
        """Reads the dataset and automatically sends requests for each row."""
        if not get_eval_dataset_file():
            raise MissingEnvironmentVariableException("EVAL_DATASET_FILE is not set.")
        
        dataset = read_dataset(get_eval_dataset_file())
        for index, row in dataset.iterrows():
            request_id = self.start_request()

            # Extract parameters from the row
            self.add_param(request_id, 'project_code', row.get('project_code'))
            self.add_param(request_id, 'prompt', row.get('prompt'))
            self.add_param(request_id, 'no_of_docs_in_context', len(os.listdir(get_docs_path())))  # Example logic
            self.add_param(request_id, 'top_k', row.get('top_k'))
            self.add_param(request_id, 'no_of_docs_retrieved', row.get('no_of_docs_retrieved'))
            self.add_param(request_id, 'retrieved_ans', row.get('retrieved_ans'))
            self.add_param(request_id, 'ground_truth', row.get('ground_truth'))
            self.add_param(request_id, 'retrieved_docs_list', row.get('retrieved_docs_list'))
            self.add_param(request_id, 'total_query_time', row.get('total_query_time'))
            self.add_param(request_id, 'retrieval_time', row.get('retrieval_time'))
            self.add_param(request_id, 'generation_time', row.get('generation_time'))

            # Send the request
            response = self.send(request_id)
            print(f"Request {request_id} response: {response}")
            time.sleep(1)  # To avoid hitting rate limits or overwhelming the server

    def send_bulk(self, file_path: str, request_id):
        """Uploads a dataset file (CSV, Excel, JSON) to the /bulk API for batch evaluation."""
        try:
            url = f"{get_api_url().rstrip('/')}/bulk"
            headers = {'Authorization': f"Bearer {get_api_key()}"}

            if not os.path.exists(file_path):
                raise SDKException(f"File not found: {file_path}")

            with open(file_path, 'rb') as f:
                files = {'file': (os.path.basename(file_path), f)}
                timeout_value = calculate_timeout(file_path, time_per_request=60)
                data = {"json_data": json.dumps({"request_uuid": str(request_id)})}

                print("url=", url, "headers=", headers, "files=", files, "timeout_value=", timeout_value, "data=", data)
                response = requests.post(url, headers=headers, files=files, timeout=timeout_value, data=data)

            # print("Response from send_bulk:", response.json())

            if response.status_code != 200:
                raise SDKException(f"Bulk request failed with status code {response.status_code}: {response.text}")

            return response.json()

        except requests.exceptions.RequestException as e:
            raise SDKException(f"Bulk request failed: {e}")
        except Exception as e:
            raise SDKException(f"Error in send_bulk: {e}")

    def send_bulk_in_background(self, file_path: str, request_id):
        """Sends the bulk dataset file to the /bulk API in a separate thread."""
        def _send_bulk_task():
            try:
                response = self.send_bulk(file_path, request_id)
                print(f"Background bulk request {request_id} response received")
            except Exception as e:
                print(f"Background bulk request {request_id} failed: {e}")

        thread = threading.Thread(target=_send_bulk_task, daemon=True)
        thread.start()
        return thread  # return threa


def calculate_timeout(file_path: str, time_per_request: int = 60) -> int:
    """
    Calculate total timeout dynamically based on number of rows in CSV.
    
    Args:
        file_path (str): Path to the CSV file
        time_per_request (int): Timeout per request in seconds (default 60)
    
    Returns:
        int: Total timeout in seconds
    """
    df = pd.read_csv(file_path)
    num_requests = len(df)
    total_timeout = num_requests * time_per_request
    return total_timeout
