import argparse
import logging
import sys

from . import transient
from . import __version__
from typing import List, Any, Optional


def parse_arguments() -> argparse.Namespace:
    # If we use the default 'usage' generated by argparse, it doesn't show
    # the '--' (though it does work). To avoid confusion, just use our own.
    parser = argparse.ArgumentParser(prog='transient',
                                     description='A simple libvirt/vagrant alternative',
                                     usage="transient [OPTIONS] -- [ARG [ARG ...]]")

    parser.add_argument('-version', '--version', action='version',
                        version='%(prog)s {}'.format(__version__))

    parser.add_argument('-v', '--verbose', action='count', default=0,
                        help="Verbosity level (repeat to be more verbose)")

    parser.add_argument('-name', help='Set the vm name', required=True)

    parser.add_argument('-image', metavar='IMG', nargs='+', action=ExtendAction,
                        help='Disk image to use (this option can be repeated)')
    parser.add_argument('-image-frontend', help='The location to place per-vm disk images')
    parser.add_argument('-image-backend',
                        help='The location to place the shared, read-only backing disk images')

    parser.add_argument('-ssh-console', '-ssh', action='store_const', const=True,
                        default=False, help='Use an ssh connection instead of the serial console')

    parser.add_argument('-ssh-user', '-u', default="vagrant", help='User to pass to SSH')
    parser.add_argument('-ssh-bin-name', default="ssh", help='SSH binary to use')
    parser.add_argument('-ssh-timeout', default=60, type=int,
                        help='Time to wait for SSH connection before failing')
    parser.add_argument('-ssh-port', help='Local port the 22 of the guest is forwarded to')
    parser.add_argument('-ssh-command', '-cmd', help='Run an ssh command instead of a console')

    parser.add_argument('-sync-before', '-b', nargs='+', action=ExtendAction,
                        help='Sync a host path to a guest path before starting the guest')

    parser.add_argument('-sync-after', '-a', nargs='+', action=ExtendAction,
                        help='Sync a guest path to a host path after stopping the guest')

    parser.add_argument('-shared-folder', '-s', nargs='+', action=ExtendAction, default=[],
                        help='Share a host directory with the guest (/path/on/host:/path/on/guest)')

    parser.add_argument('qemu_args', metavar='ARG', nargs='*',
                        help='Arguments passed directly to QEMU')
    return parser.parse_args()


class ExtendAction(argparse._AppendAction):
    def _copy_items(self, items: List[Any]) -> List[Any]:
        if items is None:
            return []
        import copy
        return copy.copy(items)

    # The 'extend' action is not available until 3.8. Backport that here
    # Mostly taken from https://github.com/python/cpython/blob/master/Lib/argparse.py#L1216
    def __call__(self, parser: argparse.ArgumentParser,
                 namespace: argparse.Namespace,
                 values: Any, option_string: Optional[str] = None) -> None:
        items = getattr(namespace, self.dest, None)
        items = self._copy_items(items)
        items.extend(values)
        setattr(namespace, self.dest, items)


def main() -> None:
    args = parse_arguments()

    log_level = logging.ERROR
    if args.verbose == 1:
        log_level = logging.WARNING
    elif args.verbose == 2:
        log_level = logging.INFO
    elif args.verbose >= 3:
        log_level = logging.DEBUG
    logging.basicConfig(level=log_level, format='%(asctime)s:%(levelname)s:%(message)s',
                        datefmt='%Y-%m-%d %H:%M:%S')

    logging.debug("Parsed arguments: {}".format(args))

    trans = transient.TransientVm(config=args)
    returncode = trans.run()
    sys.exit(returncode)
