# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pycmtensor', 'pycmtensor.models']

package_data = \
{'': ['*']}

install_requires = \
['aesara>2.7.4',
 'dill>=0.3.4,<0.4.0',
 'numpy>=1.19.0',
 'pandas>=1.3.5,<2.0.0',
 'pydot>=1.4.2,<2.0.0',
 'scipy>=1.7.1,<2.0.0']

extras_require = \
{'commit': ['pre-commit>=2.17.0,<3.0.0', 'commitizen>=2.21.0,<3.0.0'],
 'dev': ['black==22.6.0',
         'isort==5.10.1',
         'pydocstyle>=6.1.1,<7.0.0',
         'rstcheck>=3.3.1,<4.0.0',
         'Sphinx>=4.4.0,<5.0.0',
         'sphinx-book-theme>=0.2.0,<0.3.0',
         'sphinx-design>=0.3.0,<0.4.0',
         'sphinx-autoapi>=1.8.4,<2.0.0',
         'myst-parser>=0.18.0,<0.19.0',
         'pre-commit>=2.17.0,<3.0.0',
         'commitizen>=2.21.0,<3.0.0'],
 'docs': ['Sphinx>=4.4.0,<5.0.0',
          'sphinx-book-theme>=0.2.0,<0.3.0',
          'sphinx-design>=0.3.0,<0.4.0',
          'sphinx-autoapi>=1.8.4,<2.0.0',
          'myst-parser>=0.18.0,<0.19.0'],
 'lint': ['black==22.6.0',
          'isort==5.10.1',
          'pydocstyle>=6.1.1,<7.0.0',
          'rstcheck>=3.3.1,<4.0.0']}

setup_kwargs = {
    'name': 'pycmtensor',
    'version': '1.2.0',
    'description': 'Python Tensor based package for Deep neural net assisted Discrete Choice Modelling.',
    'long_description': '# PyCMTensor\n\n![Licence](https://img.shields.io/badge/Licence-MIT-blue)\n![](https://img.shields.io/pypi/pyversions/pycmtensor)\n[![PyPI version](https://badge.fury.io/py/pycmtensor.svg)](https://badge.fury.io/py/pycmtensor)\n[![Documentation Status](https://readthedocs.org/projects/pycmtensor/badge/?version=latest)](https://pycmtensor.readthedocs.io/en/latest/?badge=latest)\n[![codecov](https://codecov.io/gh/mwong009/pycmtensor/branch/master/graph/badge.svg?token=LFwgggDyjS)](https://codecov.io/gh/mwong009/pycmtensor)\n\n[![Tests](https://github.com/mwong009/pycmtensor/actions/workflows/tests.yml/badge.svg)](https://github.com/mwong009/pycmtensor/actions/workflows/tests.yml)\n[![CodeQL](https://github.com/mwong009/pycmtensor/actions/workflows/codeql-analysis.yml/badge.svg)](https://github.com/mwong009/pycmtensor/actions/workflows/codeql-analysis.yml)\n[![Publish](https://github.com/mwong009/pycmtensor/actions/workflows/publish.yml/badge.svg)](https://github.com/mwong009/pycmtensor/actions/workflows/publish.yml)\n\nA tensor-based choice modelling Python package with deep learning libraries. \n~~Documentation available at <https://pycmtensor.readthedocs.io>~~.\n\n- [PyCMTensor](#pycmtensor)\n\t- [About PyCMTensor](#about-pycmtensor)\n\t- [Features](#features)\n- [Quick start](#quick-start)\n\t- [Installation](#installation)\n- [Usage](#usage)\n\t- [Simple example: Swissmetro dataset](#simple-example-swissmetro-dataset)\n\t- [Results](#results)\n- [Development](#development)\n\t- [Installing the virtual environment](#installing-the-virtual-environment)\n\t- [Install the project and development dependencies](#install-the-project-and-development-dependencies)\n\n## About PyCMTensor\n\nPyCMTensor is a discrete choice modelling development tool on deep learning libraries, enabling development of complex models using deep neural networks.\nPyCMTensor is build on [Aesara](https://github.com/aesara-devs/aesara), a tensor library which uses features commonly found in deep learning packages such as ``Tensorflow`` and ``Keras``.\n``Aesara`` was chosen as the back end mathematical library because of its hackable, open-source nature.\nUsers of [Biogeme](https://biogeme.epfl.ch) would be familiar with the syntax of PyCMTensor.\n\nPyCMTensor improves on [Biogeme](https://biogeme.epfl.ch) in situations where much more complex models are necessary, for example, integrating neural networks into discrete choice models.\nPyCMTensor also include the ability to estimate models using 1st order stochastic gradient descent methods by default, such as Nesterov Accelerated Gradient (NAG), Adaptive momentum (ADAM), or RMSProp.\n\n## Features\n\n* Estimate complex choice models with neural networks using deep learning algorithms\n* Combines traditional econometric models (e.g. Multinomial Logit) with deep learning models (e.g. ResNets)\n* Shares similar programming syntax with ``Biogeme``, allowing easy transition between models\n* Uses tensor features found in the ``Aesara`` library\n\n---\n\n# Quick start\n\n## Installation\n\n1. Download and install [Miniconda](https://docs.conda.io/en/latest/miniconda.html)\n\n\tFull Anaconda works fine, but Miniconda is recommmended for a minimal installation. Ensure that Conda is using at least **Python 3.9**\n\n\tNext, install the required dependencies:\n\n\t**Windows**\n\n\t```\n\tconda install mkl-service conda-forge::cxx-compiler conda-forge::m2w64-toolchain\n\t```\n\t**Linux**\n\n\t```\n\tconda install mkl-service conda-forge::cxx-compiler\n\t```\n\n\t**Mac OSX**\n\n\t```\n\tconda install mkl-service Clang\n\t```\n\n2. Install the ``PyCMTensor`` package\n\n\tPyCMTensor is available on PyPi https://pypi.org/project/pycmtensor/. It can be installed with ``pip``\n\n\t```\n\tpip install -U pycmtensor==1.2.0\n\t```\n\n\tAlternatively, the latest development version is available via [Github](https://github.com/mwong009/pycmtensor). It can be installed via \n\n\t```\n\tpip install -U git+https://github.com/mwong009/pycmtensor.git\n\t```\n\nFor more information about installing, see [Installation](https://pycmtensor.readthedocs.io/en/latest/installation.html).\n\n# Usage\n\nPyCMTensor uses syntax very similar to ``Biogeme``. Users of ``Biogeme`` should be familiar with the syntax.\nMake sure you are using the correct Conda environment and/or the required packages are installed.\n\n## Simple example: Swissmetro dataset\n\n1. Start an interactive session (e.g. ``IPython`` or Jupyter Notebook) and import the ``PyCMTensor`` package:\n\t```python\n\timport pycmtensor as cmt\n\timport pandas as pd\n\t```\n\n\tSeveral submodules to include:\n\t```python\n\tfrom pycmtensor.expressions import Beta # Beta class for model parameters\n\tfrom pycmtensor.models import MNL  # MNL model\n\tfrom pycmtensor.statistics import elasticities  # For calculating elasticities\n\t```\n\n\tFor a full list of submodules and description, refer to [API Reference](https://pycmtensor.readthedocs.io/en/latest/autoapi/index.html).\n\tUsing the [swissmetro dataset](https://biogeme.epfl.ch/data.html), we define a simple MNL model. \n\n\n> :warning: Note: The following is a replication of the results from Biogeme using the ``Adam`` optimization method with constant learning rate.\n\n\n1. Import the dataset and perform some data cleaning\n\t```python\n\tswissmetro = pd.read_csv("swissmetro.dat", sep="\\t")\n\tswissmetro.drop(swissmetro[swissmetro["CHOICE"] == 0].index, inplace=True)\n\tswissmetro["CHOICE"] -= 1  # set the first choice index to 0\n\tdb = cmt.Data(df=swissmetro, choice="CHOICE")\n\tdb.autoscale_data(except_for=["ID", "ORIGIN", "DEST"])  # scales dataset\n\tdb.split_db(split_frac=0.8)  # split dataset into train/valid sets\n\t```\n\n2. Initialize the model parameters and specify the utility functions and availability conditions\n\t```python\n\tb_cost = Beta("b_cost", 0.0, None, None, 0)\n\tb_time = Beta("b_time", 0.0, None, None, 0)\n\tasc_train = Beta("asc_train", 0.0, None, None, 0)\n\tasc_car = Beta("asc_car", 0.0, None, None, 0)\n\tasc_sm = Beta("asc_sm", 0.0, None, None, 1)\n\n\tU_1 = b_cost * db["TRAIN_CO"] + b_time * db["TRAIN_TT"] + asc_train\n\tU_2 = b_cost * db["SM_CO"] + b_time * db["SM_TT"] + asc_sm\n\tU_3 = b_cost * db["CAR_CO"] + b_time * db["CAR_TT"] + asc_car\n\n\t# specify the utility function and the availability conditions\n\tU = [U_1, U_2, U_3]  # utility\n\tAV = [db["TRAIN_AV"], db["SM_AV"], db["CAR_AV"]]  # availability\n\t``` \n\n3. Define the Multinomial Logit model\n\t```python\n\tmymodel = MNL(U, AV, locals(), db, name="MNL")\n\t```\n\n4. Train the model and generate model statistics (Optionally, you can also set the training hyperparameters)\n\t```python\n\tmymodel.config.set_hyperparameter("max_steps", 200)  # set the max number of train steps\n\tmymodel.config.set_hyperparameter("batch_size", 128)  # set the training batch size\n\tmymodel.train(db)  # run the model training on the dataset `db`\n\t```\n\n## Results\nThe following model functions outputs the statistics, results of the model, and model training\n\n1. **Model estimates**\n\t```Python\n\tprint(mymodel.results.beta_statistics())\n\t```\n\n\tOutput:\n\t```\n\t              value   std err     t-test   p-value rob. std err rob. t-test rob. p-value\n\tasc_car   -0.665638  0.044783 -14.863615       0.0     0.176178    -3.77821     0.000158\n\tasc_sm          0.0         -          -         -            -           -            -\n\tasc_train -1.646826  0.048099 -34.238218       0.0     0.198978   -8.276443          0.0\n\tb_cost     0.024912   0.01943   1.282135  0.199795     0.016413    1.517851     0.129052\n\tb_time    -0.313186  0.049708  -6.300485       0.0     0.208239   -1.503979     0.132587\n\t```\n\n2. **Training results**\n\t```Python\n\tprint(mymodel.results.model_statistics())\n\t```\n\n\tOutput:\n\t```\n\t                                          value\n\tNumber of training samples used          8575.0\n\tNumber of validation samples used        2143.0\n\tInit. log likelihood               -8874.438875\n\tFinal log likelihood                -7513.22967\n\tAccuracy                                 59.26%\n\tLikelihood ratio test                2722.41841\n\tRho square                             0.153385\n\tRho square bar                         0.152822\n\tAkaike Information Criterion       15036.459339\n\tBayesian Information Criterion      15071.74237\n\tFinal gradient norm                    0.007164\n\t```\n\n3. **Correlation matrix**\n\t```Python\n\tprint(mymodel.results.model_correlation_matrix())\n\t```\n\n\tOutput:\n\t```\n\t             b_cost    b_time  asc_train   asc_car\n\tb_cost     1.000000  0.209979   0.226737 -0.028335\n\tb_time     0.209979  1.000000   0.731378  0.796144\n\tasc_train  0.226737  0.731378   1.000000  0.664478\n\tasc_car   -0.028335  0.796144   0.664478  1.000000\n\t```\n\n4. **Elasticities**\n\t```Python\n\tprint(elasticities(mymodel, db, 0, "TRAIN_TT"))  # CHOICE:TRAIN (0) wrt TRAIN_TT\n\t```\n\n\tOutput:\n\t```\n\t[-0.06813523 -0.01457346 -0.0555597  ... -0.03453162 -0.02809382 -0.02343637]\n\t```\n\n5. **Choice probability predictions**\n\t```Python\n\tprint(mymodel.predict(db, return_choices=False))\n\t```\n\n\tOutput:\n\t```\n\t[[0.12319342 0.54372904 0.33307754]\n\t[0.12267997 0.54499504 0.33232499]\n\t[0.12354587 0.54162143 0.3348327 ]\n\t...\n\t[0.12801816 0.5201341  0.35184774]\n\t[0.1271984  0.51681635 0.35598525]\n\t[0.12881032 0.51856181 0.35262787]]\n\t```\n\n---\n\n# Development\n\n(Optional) To develop PyCMTensor development package in a local environment, e.g. to modify, add features etc., you need to set up a virtual (Conda) environment and install the project requirements. Follow the instructions to install Conda (miniconda), then start a new virtual environment with the provided ``environment_<your OS>.yml`` file.\n\n1. Download the git project repository into a local directory\n\t```console\n\tgit clone git://github.com/mwong009/pycmtensor\n\tcd pycmtensor\n\t```\n\n## Installing the virtual environment\n\n**Windows**\n\n```\nconda env create -f environment_windows.yml\n```\n\n**Linux**\n\n```\nconda env create -f environment_linux.yml\n```\n\n**Mac OSX**\n\n```\nconda env create -f environment_macos.yml\n```\n\nNext, activate the virtual environment and install ``poetry`` dependency manager via ``pip``\n\n```\nconda activate pycmtensor-dev\npip install poetry\n```\n\n## Install the project and development dependencies\n\n```\npoetry install -E dev\n```\n',
    'author': 'Melvin Wong',
    'author_email': 'm.j.w.wong@tue.nl',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/mwong009/pycmtensor',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '==3.9.12',
}


setup(**setup_kwargs)
