

from machine import Pin, unique_id

import time 

import ubinascii
from umqtt.simple import MQTTClient 
import ujson

class HCSR04:
    """
    Driver to use the untrasonic sensor HC-SR04.
    The sensor range is between 2cm and 4m.
    The timeouts received listening to echo pin are converted to OSError('Out of range')
    """
    # echo_timeout_us is based in chip range limit (400cm)
    def __init__(self, trigger_pin, echo_pin, echo_timeout_us=500*2*30):
        """
        trigger_pin: Output pin to send pulses
        echo_pin: Readonly pin to measure the distance. The pin should be protected with 1k resistor
        echo_timeout_us: Timeout in microseconds to listen to echo pin. 
        By default is based in sensor limit range (4m)
        """
        self.echo_timeout_us = echo_timeout_us
        # Init trigger pin (out)
        self.trigger = Pin(trigger_pin, mode=Pin.OUT, pull=None)
        self.trigger.value(0)
 
        # Init echo pin (in)
        self.echo = Pin(echo_pin, mode=Pin.IN, pull=None)
 
    def _send_pulse_and_wait(self):
        """
        Send the pulse to trigger and listen on echo pin.
        We use the method `machine.time_pulse_us()` to get the microseconds until the echo is received.
        """
        self.trigger.value(0) # Stabilize the sensor
        time.sleep_us(5)
        self.trigger.value(1)
        # Send a 10us pulse.
        time.sleep_us(10)
        self.trigger.value(0)
        try:
            pulse_time = machine.time_pulse_us(self.echo, 1, self.echo_timeout_us)
            return pulse_time
        except OSError as ex:
            if ex.args[0] == 110: # 110 = ETIMEDOUT
                raise OSError('Out of range')
            raise ex
 
    def distance_mm(self):
        """
        Get the distance in milimeters without floating point operations.
        """
        pulse_time = self._send_pulse_and_wait()
 
        # To calculate the distance we get the pulse_time and divide it by 2 
        # (the pulse walk the distance twice) and by 29.1 becasue
        # the sound speed on air (343.2 m/s), that It's equivalent to
        # 0.34320 mm/us that is 1mm each 2.91us
        # pulse_time // 2 // 2.91 -> pulse_time // 5.82 -> pulse_time * 100 // 582 
        mm = pulse_time * 100 // 582
        return mm
 
    def distance_cm(self):
        """
        Get the distance in centimeters with floating point operations.
        It returns a float
        """
        pulse_time = self._send_pulse_and_wait()
 
        # To calculate the distance we get the pulse_time and divide it by 2 
        # (the pulse walk the distance twice) and by 29.1 becasue
        # the sound speed on air (343.2 m/s), that It's equivalent to
        # 0.034320 cm/us that is 1cm each 29.1us
        cms = (pulse_time / 2) / 29.1
        return cms





class Main:
	"""docstring for Main"""
	def __init__(self, echo_pin=None, trigger_pin=None, name='name', echo_timeout_us=500*2*30):
		super(Main, self).__init__()
		self.echo_pin = echo_pin
		self.trigger_pin = trigger_pin
		self.name = name 
		self.echo_timeout_us = echo_timeout_us
		self.machine_id = ubinascii.hexlify(unique_id()).decode('utf-8')
		self.client = MQTTClient(self.machine_id+'_'+self.name, 'broker.hivemq.com')


	def _publish_error_message(self, error = ''):
		data = {'success':0, 'error':error, 'distance_cm':0, 'distance_mm':0}
		self.client.connect()
		topic = self.machine_id+'/'+self.name
		self.client.publish(topic, ujson.dumps(data))
		return ujson.dumps(data) 



	def measure_data(self):

		try:
			sensor = HCSR04(self.trigger_pin, self.echo_pin, echo_timeout_us= self.echo_timeout_us)
			data = {'success':1, 'error':'', 'distance_cm':sensor.distance_cm(), 'distance_mm':sensor.distance_mm()}
			return ujson.dumps(data)

		except Exception as e:
			data = {'success':0, 'error':e, 'distance_cm':0, 'distance_mm':0}
			return ujson.dumps(data)



	def main(self):
		if not self.echo_pin or not self.trigger_pin:
			self.client.connect()
			topic = self.machine_id+'/'+self.name
			self.client.publish(topic, ujson.dumps({'success':0, 'error':'pins not valid.', 'distance_cm':0, 'distance_mm':0}))
			return ujson.dumps({'success':0, 'error':'pins not valid.', 'distance_cm':0, 'distance_mm':0})

		self.client.connect()

		while True:
			topic = self.machine_id+'/'+self.name
			self.client.publish(topic, self.measure_data())

		self.client.disconnect()





		



