"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.collectRuntimeInformation = void 0;
const path_1 = require("path");
const node_version_1 = require("./node-version");
// list of NPM scopes included in version reporting e.g. @aws-cdk and @aws-solutions-konstruk
const WHITELIST_SCOPES = ['@aws-cdk', '@aws-solutions-konstruk', '@aws-solutions-constructs'];
/**
 * Returns a list of loaded modules and their versions.
 */
function collectRuntimeInformation() {
    const libraries = {};
    for (const fileName of Object.keys(require.cache)) {
        const pkg = findNpmPackage(fileName);
        if (pkg && !pkg.private) {
            libraries[pkg.name] = pkg.version;
        }
    }
    // include only libraries that are in the whitelistLibraries list
    for (const name of Object.keys(libraries)) {
        let foundMatch = false;
        for (const scope of WHITELIST_SCOPES) {
            if (name.startsWith(`${scope}/`)) {
                foundMatch = true;
            }
        }
        if (!foundMatch) {
            delete libraries[name];
        }
    }
    // add jsii runtime version
    libraries['jsii-runtime'] = getJsiiAgentVersion();
    return { libraries };
}
exports.collectRuntimeInformation = collectRuntimeInformation;
/**
 * Determines which NPM module a given loaded javascript file is from.
 *
 * The only infromation that is available locally is a list of Javascript files,
 * and every source file is associated with a search path to resolve the further
 * ``require`` calls made from there, which includes its own directory on disk,
 * and parent directories - for example:
 *
 * [ '...repo/packages/aws-cdk-resources/lib/cfn/node_modules',
 *   '...repo/packages/aws-cdk-resources/lib/node_modules',
 *   '...repo/packages/aws-cdk-resources/node_modules',
 *   '...repo/packages/node_modules',
 *   // etc...
 * ]
 *
 * We are looking for ``package.json`` that is anywhere in the tree, except it's
 * in the parent directory, not in the ``node_modules`` directory. For this
 * reason, we strip the ``/node_modules`` suffix off each path and use regular
 * module resolution to obtain a reference to ``package.json``.
 *
 * @param fileName a javascript file name.
 * @returns the NPM module infos (aka ``package.json`` contents), or
 *      ``undefined`` if the lookup was unsuccessful.
 */
function findNpmPackage(fileName) {
    const mod = require.cache[fileName];
    if (!mod.paths) {
        // sometimes this can be undefined. for example when querying for .json modules
        // inside a jest runtime environment.
        // see https://github.com/aws/aws-cdk/issues/7657
        // potentially we can remove this if it turns out to be a bug in how jest implemented the 'require' module.
        return undefined;
    }
    // For any path in ``mod.paths`` that is a node_modules folder, use its parent directory instead.
    const paths = mod.paths.map((path) => path_1.basename(path) === 'node_modules' ? path_1.dirname(path) : path);
    try {
        const packagePath = require.resolve(
        // Resolution behavior changed in node 12.0.0 - https://github.com/nodejs/node/issues/27583
        node_version_1.major >= 12 ? './package.json' : 'package.json', { paths });
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(packagePath);
    }
    catch (e) {
        return undefined;
    }
}
function getJsiiAgentVersion() {
    let jsiiAgent = process.env.JSII_AGENT;
    // if JSII_AGENT is not specified, we will assume this is a node.js runtime
    // and plug in our node.js version
    if (!jsiiAgent) {
        jsiiAgent = `node.js/${process.version}`;
    }
    return jsiiAgent;
}
//# sourceMappingURL=data:application/json;base64,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