"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseProxy = exports.ProxyTarget = exports.SessionPinningFilter = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rds_generated_1 = require("./rds.generated");
/**
 * SessionPinningFilter
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy-pinning
 */
class SessionPinningFilter {
    constructor(
    /**
     * Filter name
     */
    filterName) {
        this.filterName = filterName;
    }
    /**
     * custom filter
     */
    static of(filterName) {
        return new SessionPinningFilter(filterName);
    }
}
exports.SessionPinningFilter = SessionPinningFilter;
/**
 * You can opt out of session pinning for the following kinds of application statements:
 *
 * - Setting session variables and configuration settings.
 */
SessionPinningFilter.EXCLUDE_VARIABLE_SETS = new SessionPinningFilter('EXCLUDE_VARIABLE_SETS');
/**
 * Proxy target: Instance or Cluster
 *
 * A target group is a collection of databases that the proxy can connect to.
 * Currently, you can specify only one RDS DB instance or Aurora DB cluster.
 */
class ProxyTarget {
    constructor(dbInstance, dbCluster) {
        this.dbInstance = dbInstance;
        this.dbCluster = dbCluster;
    }
    /**
     * From instance
     *
     * @param instance RDS database instance
     */
    static fromInstance(instance) {
        return new ProxyTarget(instance);
    }
    /**
     * From cluster
     *
     * @param cluster RDS database cluster
     */
    static fromCluster(cluster) {
        return new ProxyTarget(undefined, cluster);
    }
    /**
     * Bind this target to the specified database proxy.
     */
    bind(_) {
        let engine;
        if (this.dbCluster && this.dbInstance) {
            throw new Error('Proxy cannot target both database cluster and database instance.');
        }
        else if (this.dbCluster) {
            engine = this.dbCluster.node.defaultChild.engine;
        }
        else if (this.dbInstance) {
            engine = this.dbInstance.node.defaultChild.engine;
        }
        let engineFamily;
        switch (engine) {
            case 'aurora':
            case 'aurora-mysql':
            case 'mysql':
                engineFamily = 'MYSQL';
                break;
            case 'aurora-postgresql':
            case 'postgres':
                engineFamily = 'POSTGRESQL';
                break;
            default:
                throw new Error(`Unsupported engine type - ${engine}`);
        }
        return {
            engineFamily,
            dbClusters: this.dbCluster ? [this.dbCluster] : undefined,
            dbInstances: this.dbInstance ? [this.dbInstance] : undefined,
        };
    }
}
exports.ProxyTarget = ProxyTarget;
/**
 * RDS Database Proxy
 *
 * @resource AWS::RDS::DBProxy
 */
class DatabaseProxy extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id, { physicalName: props.dbProxyName || id });
        const role = props.role || new iam.Role(this, 'IAMRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        props.secret.grantRead(role);
        this.connections = new ec2.Connections({ securityGroups: props.securityGroups });
        const bindResult = props.proxyTarget.bind(this);
        this.resource = new rds_generated_1.CfnDBProxy(this, 'Resource', {
            auth: [
                {
                    authScheme: 'SECRETS',
                    iamAuth: props.iamAuth ? 'REQUIRED' : 'DISABLED',
                    secretArn: props.secret.secretArn,
                },
            ],
            dbProxyName: this.physicalName,
            debugLogging: props.debugLogging,
            engineFamily: bindResult.engineFamily,
            idleClientTimeout: (_a = props.idleClientTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            requireTls: (_b = props.requireTLS) !== null && _b !== void 0 ? _b : true,
            roleArn: role.roleArn,
            vpcSecurityGroupIds: (_c = props.securityGroups) === null || _c === void 0 ? void 0 : _c.map(_ => _.securityGroupId),
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds,
        });
        this.dbProxyName = this.resource.ref;
        this.dbProxyArn = this.resource.attrDbProxyArn;
        this.endpoint = this.resource.attrEndpoint;
        let dbInstanceIdentifiers;
        if (bindResult.dbClusters) {
            // support for only instances of a single cluster
            dbInstanceIdentifiers = bindResult.dbClusters[0].instanceIdentifiers;
        }
        else if (bindResult.dbInstances) {
            // support for only single instance
            dbInstanceIdentifiers = [bindResult.dbInstances[0].instanceIdentifier];
        }
        new rds_generated_1.CfnDBProxyTargetGroup(this, 'ProxyTargetGroup', {
            dbProxyName: this.dbProxyName,
            dbInstanceIdentifiers,
            dbClusterIdentifiers: (_d = bindResult.dbClusters) === null || _d === void 0 ? void 0 : _d.map((c) => c.clusterIdentifier),
            connectionPoolConfigurationInfo: toConnectionPoolConfigurationInfo(props),
        });
    }
    /**
     * Import an existing database proxy.
     */
    static fromDatabaseProxyAttributes(scope, id, attrs) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.dbProxyName = attrs.dbProxyName;
                this.dbProxyArn = attrs.dbProxyArn;
                this.endpoint = attrs.endpoint;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.dbProxyName,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_PROXY,
        };
    }
}
exports.DatabaseProxy = DatabaseProxy;
/**
 * ConnectionPoolConfiguration (L2 => L1)
 */
function toConnectionPoolConfigurationInfo(props) {
    var _a, _b;
    return {
        connectionBorrowTimeout: (_a = props.borrowTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
        initQuery: props.initQuery,
        maxConnectionsPercent: props.maxConnectionsPercent,
        maxIdleConnectionsPercent: props.maxIdleConnectionsPercent,
        sessionPinningFilters: (_b = props.sessionPinningFilters) === null || _b === void 0 ? void 0 : _b.map(_ => _.filterName),
    };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJveHkuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJwcm94eS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxxQ0FBcUMsQ0FBQyxtREFBbUQ7QUFDekYscUNBQXFDLENBQUMsbURBQW1EO0FBQ3pGLDJEQUEyRCxDQUFDLDhEQUE4RDtBQUMxSCxrQ0FBa0MsQ0FBQyxnREFBZ0Q7QUFHbkYsbURBQWlHO0FBQ2pHOzs7O0dBSUc7QUFDSCxNQUFhLG9CQUFvQjtJQWE3QjtJQUNBOztPQUVHO0lBQ2EsVUFBa0I7UUFBbEIsZUFBVSxHQUFWLFVBQVUsQ0FBUTtJQUFJLENBQUM7SUFWdkM7O09BRUc7SUFDSSxNQUFNLENBQUMsRUFBRSxDQUFDLFVBQWtCO1FBQy9CLE9BQU8sSUFBSSxvQkFBb0IsQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUNoRCxDQUFDOztBQVpMLG9EQWtCQztBQWpCRzs7OztHQUlHO0FBQ29CLDBDQUFxQixHQUFHLElBQUksb0JBQW9CLENBQUMsdUJBQXVCLENBQUMsQ0FBQztBQWFyRzs7Ozs7R0FLRztBQUNILE1BQWEsV0FBVztJQWlCcEIsWUFBcUMsVUFBOEIsRUFBbUIsU0FBNEI7UUFBN0UsZUFBVSxHQUFWLFVBQVUsQ0FBb0I7UUFBbUIsY0FBUyxHQUFULFNBQVMsQ0FBbUI7SUFBSSxDQUFDO0lBaEJ2SDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLFlBQVksQ0FBQyxRQUEyQjtRQUNsRCxPQUFPLElBQUksV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLFdBQVcsQ0FBQyxPQUF5QjtRQUMvQyxPQUFPLElBQUksV0FBVyxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxJQUFJLENBQUMsQ0FBZ0I7UUFDeEIsSUFBSSxNQUEwQixDQUFDO1FBQy9CLElBQUksSUFBSSxDQUFDLFNBQVMsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO1lBQ25DLE1BQU0sSUFBSSxLQUFLLENBQUMsa0VBQWtFLENBQUMsQ0FBQztTQUN2RjthQUNJLElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNyQixNQUFNLEdBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsWUFBNkIsQ0FBQyxNQUFNLENBQUM7U0FDdEU7YUFDSSxJQUFJLElBQUksQ0FBQyxVQUFVLEVBQUU7WUFDdEIsTUFBTSxHQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLFlBQThCLENBQUMsTUFBTSxDQUFDO1NBQ3hFO1FBQ0QsSUFBSSxZQUFZLENBQUM7UUFDakIsUUFBUSxNQUFNLEVBQUU7WUFDWixLQUFLLFFBQVEsQ0FBQztZQUNkLEtBQUssY0FBYyxDQUFDO1lBQ3BCLEtBQUssT0FBTztnQkFDUixZQUFZLEdBQUcsT0FBTyxDQUFDO2dCQUN2QixNQUFNO1lBQ1YsS0FBSyxtQkFBbUIsQ0FBQztZQUN6QixLQUFLLFVBQVU7Z0JBQ1gsWUFBWSxHQUFHLFlBQVksQ0FBQztnQkFDNUIsTUFBTTtZQUNWO2dCQUNJLE1BQU0sSUFBSSxLQUFLLENBQUMsNkJBQTZCLE1BQU0sRUFBRSxDQUFDLENBQUM7U0FDOUQ7UUFDRCxPQUFPO1lBQ0gsWUFBWTtZQUNaLFVBQVUsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUN6RCxXQUFXLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FDL0QsQ0FBQztJQUNOLENBQUM7Q0FDSjtBQXBERCxrQ0FvREM7QUF5TUQ7Ozs7R0FJRztBQUNILE1BQWEsYUFBYyxTQUFRLEdBQUcsQ0FBQyxRQUFRO0lBbUMzQyxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQXlCOztRQUNuRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxFQUFFLFlBQVksRUFBRSxLQUFLLENBQUMsV0FBVyxJQUFJLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDNUQsTUFBTSxJQUFJLEdBQUcsS0FBSyxDQUFDLElBQUksSUFBSSxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUNyRCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUM7U0FDM0QsQ0FBQyxDQUFDO1FBQ0gsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDN0IsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxXQUFXLENBQUMsRUFBRSxjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUM7UUFDakYsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDaEQsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLDBCQUFVLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUM3QyxJQUFJLEVBQUU7Z0JBQ0Y7b0JBQ0ksVUFBVSxFQUFFLFNBQVM7b0JBQ3JCLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLFVBQVU7b0JBQ2hELFNBQVMsRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVM7aUJBQ3BDO2FBQ0o7WUFDRCxXQUFXLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDOUIsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZO1lBQ2hDLFlBQVksRUFBRSxVQUFVLENBQUMsWUFBWTtZQUNyQyxpQkFBaUIsUUFBRSxLQUFLLENBQUMsaUJBQWlCLDBDQUFFLFNBQVMsRUFBRTtZQUN2RCxVQUFVLFFBQUUsS0FBSyxDQUFDLFVBQVUsbUNBQUksSUFBSTtZQUNwQyxPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87WUFDckIsbUJBQW1CLFFBQUUsS0FBSyxDQUFDLGNBQWMsMENBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLGVBQWUsQ0FBQztZQUN0RSxZQUFZLEVBQUUsS0FBSyxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDLFNBQVM7U0FDcEUsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQztRQUNyQyxJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDO1FBQy9DLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUM7UUFDM0MsSUFBSSxxQkFBMkMsQ0FBQztRQUNoRCxJQUFJLFVBQVUsQ0FBQyxVQUFVLEVBQUU7WUFDdkIsaURBQWlEO1lBQ2pELHFCQUFxQixHQUFHLFVBQVUsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsbUJBQW1CLENBQUM7U0FDeEU7YUFDSSxJQUFJLFVBQVUsQ0FBQyxXQUFXLEVBQUU7WUFDN0IsbUNBQW1DO1lBQ25DLHFCQUFxQixHQUFHLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1NBQzFFO1FBQ0QsSUFBSSxxQ0FBcUIsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7WUFDaEQsV0FBVyxFQUFFLElBQUksQ0FBQyxXQUFXO1lBQzdCLHFCQUFxQjtZQUNyQixvQkFBb0IsUUFBRSxVQUFVLENBQUMsVUFBVSwwQ0FBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxpQkFBaUIsQ0FBQztZQUM1RSwrQkFBK0IsRUFBRSxpQ0FBaUMsQ0FBQyxLQUFLLENBQUM7U0FDNUUsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQTdFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQywyQkFBMkIsQ0FBQyxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUE4QjtRQUN0RyxNQUFNLE1BQU8sU0FBUSxHQUFHLENBQUMsUUFBUTtZQUFqQzs7Z0JBQ29CLGdCQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztnQkFDaEMsZUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUM7Z0JBQzlCLGFBQVEsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDO1lBQzlDLENBQUM7U0FBQTtRQUNELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFvRUQ7O09BRUc7SUFDSSx3QkFBd0I7UUFDM0IsT0FBTztZQUNILFFBQVEsRUFBRSxJQUFJLENBQUMsV0FBVztZQUMxQixVQUFVLEVBQUUsY0FBYyxDQUFDLG9CQUFvQixDQUFDLFlBQVk7U0FDL0QsQ0FBQztJQUNOLENBQUM7Q0FDSjtBQXhGRCxzQ0F3RkM7QUFDRDs7R0FFRztBQUNILFNBQVMsaUNBQWlDLENBQUMsS0FBeUI7O0lBQ2hFLE9BQU87UUFDSCx1QkFBdUIsUUFBRSxLQUFLLENBQUMsYUFBYSwwQ0FBRSxTQUFTLEVBQUU7UUFDekQsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO1FBQzFCLHFCQUFxQixFQUFFLEtBQUssQ0FBQyxxQkFBcUI7UUFDbEQseUJBQXlCLEVBQUUsS0FBSyxDQUFDLHlCQUF5QjtRQUMxRCxxQkFBcUIsUUFBRSxLQUFLLENBQUMscUJBQXFCLDBDQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUM7S0FDN0UsQ0FBQztBQUNOLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBlYzIgZnJvbSBcIi4uLy4uL2F3cy1lYzJcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInXG5pbXBvcnQgKiBhcyBpYW0gZnJvbSBcIi4uLy4uL2F3cy1pYW1cIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nXG5pbXBvcnQgKiBhcyBzZWNyZXRzbWFuYWdlciBmcm9tIFwiLi4vLi4vYXdzLXNlY3JldHNtYW5hZ2VyXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3Mtc2VjcmV0c21hbmFnZXInXG5pbXBvcnQgKiBhcyBjZGsgZnJvbSBcIi4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBJRGF0YWJhc2VDbHVzdGVyIH0gZnJvbSAnLi9jbHVzdGVyLXJlZic7XG5pbXBvcnQgeyBJRGF0YWJhc2VJbnN0YW5jZSB9IGZyb20gJy4vaW5zdGFuY2UnO1xuaW1wb3J0IHsgQ2ZuREJDbHVzdGVyLCBDZm5EQkluc3RhbmNlLCBDZm5EQlByb3h5LCBDZm5EQlByb3h5VGFyZ2V0R3JvdXAgfSBmcm9tICcuL3Jkcy5nZW5lcmF0ZWQnO1xuLyoqXG4gKiBTZXNzaW9uUGlubmluZ0ZpbHRlclxuICpcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL3Jkcy1wcm94eS5odG1sI3Jkcy1wcm94eS1waW5uaW5nXG4gKi9cbmV4cG9ydCBjbGFzcyBTZXNzaW9uUGlubmluZ0ZpbHRlciB7XG4gICAgLyoqXG4gICAgICogWW91IGNhbiBvcHQgb3V0IG9mIHNlc3Npb24gcGlubmluZyBmb3IgdGhlIGZvbGxvd2luZyBraW5kcyBvZiBhcHBsaWNhdGlvbiBzdGF0ZW1lbnRzOlxuICAgICAqXG4gICAgICogLSBTZXR0aW5nIHNlc3Npb24gdmFyaWFibGVzIGFuZCBjb25maWd1cmF0aW9uIHNldHRpbmdzLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgRVhDTFVERV9WQVJJQUJMRV9TRVRTID0gbmV3IFNlc3Npb25QaW5uaW5nRmlsdGVyKCdFWENMVURFX1ZBUklBQkxFX1NFVFMnKTtcbiAgICAvKipcbiAgICAgKiBjdXN0b20gZmlsdGVyXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBvZihmaWx0ZXJOYW1lOiBzdHJpbmcpOiBTZXNzaW9uUGlubmluZ0ZpbHRlciB7XG4gICAgICAgIHJldHVybiBuZXcgU2Vzc2lvblBpbm5pbmdGaWx0ZXIoZmlsdGVyTmFtZSk7XG4gICAgfVxuICAgIHByaXZhdGUgY29uc3RydWN0b3IoXG4gICAgLyoqXG4gICAgICogRmlsdGVyIG5hbWVcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgZmlsdGVyTmFtZTogc3RyaW5nKSB7IH1cbn1cbi8qKlxuICogUHJveHkgdGFyZ2V0OiBJbnN0YW5jZSBvciBDbHVzdGVyXG4gKlxuICogQSB0YXJnZXQgZ3JvdXAgaXMgYSBjb2xsZWN0aW9uIG9mIGRhdGFiYXNlcyB0aGF0IHRoZSBwcm94eSBjYW4gY29ubmVjdCB0by5cbiAqIEN1cnJlbnRseSwgeW91IGNhbiBzcGVjaWZ5IG9ubHkgb25lIFJEUyBEQiBpbnN0YW5jZSBvciBBdXJvcmEgREIgY2x1c3Rlci5cbiAqL1xuZXhwb3J0IGNsYXNzIFByb3h5VGFyZ2V0IHtcbiAgICAvKipcbiAgICAgKiBGcm9tIGluc3RhbmNlXG4gICAgICpcbiAgICAgKiBAcGFyYW0gaW5zdGFuY2UgUkRTIGRhdGFiYXNlIGluc3RhbmNlXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBmcm9tSW5zdGFuY2UoaW5zdGFuY2U6IElEYXRhYmFzZUluc3RhbmNlKTogUHJveHlUYXJnZXQge1xuICAgICAgICByZXR1cm4gbmV3IFByb3h5VGFyZ2V0KGluc3RhbmNlKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogRnJvbSBjbHVzdGVyXG4gICAgICpcbiAgICAgKiBAcGFyYW0gY2x1c3RlciBSRFMgZGF0YWJhc2UgY2x1c3RlclxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbUNsdXN0ZXIoY2x1c3RlcjogSURhdGFiYXNlQ2x1c3Rlcik6IFByb3h5VGFyZ2V0IHtcbiAgICAgICAgcmV0dXJuIG5ldyBQcm94eVRhcmdldCh1bmRlZmluZWQsIGNsdXN0ZXIpO1xuICAgIH1cbiAgICBwcml2YXRlIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgZGJJbnN0YW5jZT86IElEYXRhYmFzZUluc3RhbmNlLCBwcml2YXRlIHJlYWRvbmx5IGRiQ2x1c3Rlcj86IElEYXRhYmFzZUNsdXN0ZXIpIHsgfVxuICAgIC8qKlxuICAgICAqIEJpbmQgdGhpcyB0YXJnZXQgdG8gdGhlIHNwZWNpZmllZCBkYXRhYmFzZSBwcm94eS5cbiAgICAgKi9cbiAgICBwdWJsaWMgYmluZChfOiBEYXRhYmFzZVByb3h5KTogUHJveHlUYXJnZXRDb25maWcge1xuICAgICAgICBsZXQgZW5naW5lOiBzdHJpbmcgfCB1bmRlZmluZWQ7XG4gICAgICAgIGlmICh0aGlzLmRiQ2x1c3RlciAmJiB0aGlzLmRiSW5zdGFuY2UpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignUHJveHkgY2Fubm90IHRhcmdldCBib3RoIGRhdGFiYXNlIGNsdXN0ZXIgYW5kIGRhdGFiYXNlIGluc3RhbmNlLicpO1xuICAgICAgICB9XG4gICAgICAgIGVsc2UgaWYgKHRoaXMuZGJDbHVzdGVyKSB7XG4gICAgICAgICAgICBlbmdpbmUgPSAodGhpcy5kYkNsdXN0ZXIubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuREJDbHVzdGVyKS5lbmdpbmU7XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSBpZiAodGhpcy5kYkluc3RhbmNlKSB7XG4gICAgICAgICAgICBlbmdpbmUgPSAodGhpcy5kYkluc3RhbmNlLm5vZGUuZGVmYXVsdENoaWxkIGFzIENmbkRCSW5zdGFuY2UpLmVuZ2luZTtcbiAgICAgICAgfVxuICAgICAgICBsZXQgZW5naW5lRmFtaWx5O1xuICAgICAgICBzd2l0Y2ggKGVuZ2luZSkge1xuICAgICAgICAgICAgY2FzZSAnYXVyb3JhJzpcbiAgICAgICAgICAgIGNhc2UgJ2F1cm9yYS1teXNxbCc6XG4gICAgICAgICAgICBjYXNlICdteXNxbCc6XG4gICAgICAgICAgICAgICAgZW5naW5lRmFtaWx5ID0gJ01ZU1FMJztcbiAgICAgICAgICAgICAgICBicmVhaztcbiAgICAgICAgICAgIGNhc2UgJ2F1cm9yYS1wb3N0Z3Jlc3FsJzpcbiAgICAgICAgICAgIGNhc2UgJ3Bvc3RncmVzJzpcbiAgICAgICAgICAgICAgICBlbmdpbmVGYW1pbHkgPSAnUE9TVEdSRVNRTCc7XG4gICAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5zdXBwb3J0ZWQgZW5naW5lIHR5cGUgLSAke2VuZ2luZX1gKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgZW5naW5lRmFtaWx5LFxuICAgICAgICAgICAgZGJDbHVzdGVyczogdGhpcy5kYkNsdXN0ZXIgPyBbdGhpcy5kYkNsdXN0ZXJdIDogdW5kZWZpbmVkLFxuICAgICAgICAgICAgZGJJbnN0YW5jZXM6IHRoaXMuZGJJbnN0YW5jZSA/IFt0aGlzLmRiSW5zdGFuY2VdIDogdW5kZWZpbmVkLFxuICAgICAgICB9O1xuICAgIH1cbn1cbi8qKlxuICogVGhlIHJlc3VsdCBvZiBiaW5kaW5nIGEgYFByb3h5VGFyZ2V0YCB0byBhIGBEYXRhYmFzZVByb3h5YC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQcm94eVRhcmdldENvbmZpZyB7XG4gICAgLyoqXG4gICAgICogVGhlIGVuZ2luZSBmYW1pbHkgb2YgdGhlIGRhdGFiYXNlIGluc3RhbmNlIG9yIGNsdXN0ZXIgdGhpcyBwcm94eSBjb25uZWN0cyB3aXRoLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGVuZ2luZUZhbWlseTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBkYXRhYmFzZSBpbnN0YW5jZXMgdG8gd2hpY2ggdGhpcyBwcm94eSBjb25uZWN0cy5cbiAgICAgKiBFaXRoZXIgdGhpcyBvciBgZGJDbHVzdGVyc2Agd2lsbCBiZSBzZXQgYW5kIHRoZSBvdGhlciBgdW5kZWZpbmVkYC5cbiAgICAgKiBAZGVmYXVsdCAtIGB1bmRlZmluZWRgIGlmIGBkYkNsdXN0ZXJzYCBpcyBzZXQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGJJbnN0YW5jZXM/OiBJRGF0YWJhc2VJbnN0YW5jZVtdO1xuICAgIC8qKlxuICAgICAqIFRoZSBkYXRhYmFzZSBjbHVzdGVycyB0byB3aGljaCB0aGlzIHByb3h5IGNvbm5lY3RzLlxuICAgICAqIEVpdGhlciB0aGlzIG9yIGBkYkluc3RhbmNlc2Agd2lsbCBiZSBzZXQgYW5kIHRoZSBvdGhlciBgdW5kZWZpbmVkYC5cbiAgICAgKiBAZGVmYXVsdCAtIGB1bmRlZmluZWRgIGlmIGBkYkluc3RhbmNlc2AgaXMgc2V0LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGRiQ2x1c3RlcnM/OiBJRGF0YWJhc2VDbHVzdGVyW107XG59XG4vKipcbiAqIE9wdGlvbnMgZm9yIGEgbmV3IERhdGFiYXNlUHJveHlcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEYXRhYmFzZVByb3h5T3B0aW9ucyB7XG4gICAgLyoqXG4gICAgICogVGhlIGlkZW50aWZpZXIgZm9yIHRoZSBwcm94eS5cbiAgICAgKiBUaGlzIG5hbWUgbXVzdCBiZSB1bmlxdWUgZm9yIGFsbCBwcm94aWVzIG93bmVkIGJ5IHlvdXIgQVdTIGFjY291bnQgaW4gdGhlIHNwZWNpZmllZCBBV1MgUmVnaW9uLlxuICAgICAqIEFuIGlkZW50aWZpZXIgbXVzdCBiZWdpbiB3aXRoIGEgbGV0dGVyIGFuZCBtdXN0IGNvbnRhaW4gb25seSBBU0NJSSBsZXR0ZXJzLCBkaWdpdHMsIGFuZCBoeXBoZW5zO1xuICAgICAqIGl0IGNhbid0IGVuZCB3aXRoIGEgaHlwaGVuIG9yIGNvbnRhaW4gdHdvIGNvbnNlY3V0aXZlIGh5cGhlbnMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEdlbmVyYXRlZCBieSBDbG91ZEZvcm1hdGlvbiAocmVjb21tZW5kZWQpXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGJQcm94eU5hbWU/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIGR1cmF0aW9uIGZvciBhIHByb3h5IHRvIHdhaXQgZm9yIGEgY29ubmVjdGlvbiB0byBiZWNvbWUgYXZhaWxhYmxlIGluIHRoZSBjb25uZWN0aW9uIHBvb2wuXG4gICAgICogT25seSBhcHBsaWVzIHdoZW4gdGhlIHByb3h5IGhhcyBvcGVuZWQgaXRzIG1heGltdW0gbnVtYmVyIG9mIGNvbm5lY3Rpb25zIGFuZCBhbGwgY29ubmVjdGlvbnMgYXJlIGJ1c3kgd2l0aCBjbGllbnRcbiAgICAgKiBzZXNzaW9ucy5cbiAgICAgKlxuICAgICAqIFZhbHVlIG11c3QgYmUgYmV0d2VlbiAxIHNlY29uZCBhbmQgMSBob3VyLCBvciBgRHVyYXRpb24uc2Vjb25kcygwKWAgdG8gcmVwcmVzZW50IHVubGltaXRlZC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGNkay5EdXJhdGlvbi5zZWNvbmRzKDEyMClcbiAgICAgKi9cbiAgICByZWFkb25seSBib3Jyb3dUaW1lb3V0PzogY2RrLkR1cmF0aW9uO1xuICAgIC8qKlxuICAgICAqIE9uZSBvciBtb3JlIFNRTCBzdGF0ZW1lbnRzIGZvciB0aGUgcHJveHkgdG8gcnVuIHdoZW4gb3BlbmluZyBlYWNoIG5ldyBkYXRhYmFzZSBjb25uZWN0aW9uLlxuICAgICAqIFR5cGljYWxseSB1c2VkIHdpdGggU0VUIHN0YXRlbWVudHMgdG8gbWFrZSBzdXJlIHRoYXQgZWFjaCBjb25uZWN0aW9uIGhhcyBpZGVudGljYWwgc2V0dGluZ3Mgc3VjaCBhcyB0aW1lIHpvbmVcbiAgICAgKiBhbmQgY2hhcmFjdGVyIHNldC5cbiAgICAgKiBGb3IgbXVsdGlwbGUgc3RhdGVtZW50cywgdXNlIHNlbWljb2xvbnMgYXMgdGhlIHNlcGFyYXRvci5cbiAgICAgKiBZb3UgY2FuIGFsc28gaW5jbHVkZSBtdWx0aXBsZSB2YXJpYWJsZXMgaW4gYSBzaW5nbGUgU0VUIHN0YXRlbWVudCwgc3VjaCBhcyBTRVQgeD0xLCB5PTIuXG4gICAgICpcbiAgICAgKiBub3QgY3VycmVudGx5IHN1cHBvcnRlZCBmb3IgUG9zdGdyZVNRTC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gbm8gaW5pdGlhbGl6YXRpb24gcXVlcnlcbiAgICAgKi9cbiAgICByZWFkb25seSBpbml0UXVlcnk/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIG1heGltdW0gc2l6ZSBvZiB0aGUgY29ubmVjdGlvbiBwb29sIGZvciBlYWNoIHRhcmdldCBpbiBhIHRhcmdldCBncm91cC5cbiAgICAgKiBGb3IgQXVyb3JhIE15U1FMLCBpdCBpcyBleHByZXNzZWQgYXMgYSBwZXJjZW50YWdlIG9mIHRoZSBtYXhfY29ubmVjdGlvbnMgc2V0dGluZyBmb3IgdGhlIFJEUyBEQiBpbnN0YW5jZSBvciBBdXJvcmEgREJcbiAgICAgKiBjbHVzdGVyIHVzZWQgYnkgdGhlIHRhcmdldCBncm91cC5cbiAgICAgKlxuICAgICAqIDEtMTAwXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAxMDBcbiAgICAgKi9cbiAgICByZWFkb25seSBtYXhDb25uZWN0aW9uc1BlcmNlbnQ/OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogQ29udHJvbHMgaG93IGFjdGl2ZWx5IHRoZSBwcm94eSBjbG9zZXMgaWRsZSBkYXRhYmFzZSBjb25uZWN0aW9ucyBpbiB0aGUgY29ubmVjdGlvbiBwb29sLlxuICAgICAqIEEgaGlnaCB2YWx1ZSBlbmFibGVzIHRoZSBwcm94eSB0byBsZWF2ZSBhIGhpZ2ggcGVyY2VudGFnZSBvZiBpZGxlIGNvbm5lY3Rpb25zIG9wZW4uXG4gICAgICogQSBsb3cgdmFsdWUgY2F1c2VzIHRoZSBwcm94eSB0byBjbG9zZSBpZGxlIGNsaWVudCBjb25uZWN0aW9ucyBhbmQgcmV0dXJuIHRoZSB1bmRlcmx5aW5nIGRhdGFiYXNlIGNvbm5lY3Rpb25zXG4gICAgICogdG8gdGhlIGNvbm5lY3Rpb24gcG9vbC5cbiAgICAgKiBGb3IgQXVyb3JhIE15U1FMLCBpdCBpcyBleHByZXNzZWQgYXMgYSBwZXJjZW50YWdlIG9mIHRoZSBtYXhfY29ubmVjdGlvbnMgc2V0dGluZyBmb3IgdGhlIFJEUyBEQiBpbnN0YW5jZVxuICAgICAqIG9yIEF1cm9yYSBEQiBjbHVzdGVyIHVzZWQgYnkgdGhlIHRhcmdldCBncm91cC5cbiAgICAgKlxuICAgICAqIGJldHdlZW4gMCBhbmQgTWF4Q29ubmVjdGlvbnNQZXJjZW50XG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCA1MFxuICAgICAqL1xuICAgIHJlYWRvbmx5IG1heElkbGVDb25uZWN0aW9uc1BlcmNlbnQ/OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogRWFjaCBpdGVtIGluIHRoZSBsaXN0IHJlcHJlc2VudHMgYSBjbGFzcyBvZiBTUUwgb3BlcmF0aW9ucyB0aGF0IG5vcm1hbGx5IGNhdXNlIGFsbCBsYXRlciBzdGF0ZW1lbnRzIGluIGEgc2Vzc2lvblxuICAgICAqIHVzaW5nIGEgcHJveHkgdG8gYmUgcGlubmVkIHRvIHRoZSBzYW1lIHVuZGVybHlpbmcgZGF0YWJhc2UgY29ubmVjdGlvbi5cbiAgICAgKiBJbmNsdWRpbmcgYW4gaXRlbSBpbiB0aGUgbGlzdCBleGVtcHRzIHRoYXQgY2xhc3Mgb2YgU1FMIG9wZXJhdGlvbnMgZnJvbSB0aGUgcGlubmluZyBiZWhhdmlvci5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gbm8gc2Vzc2lvbiBwaW5uaW5nIGZpbHRlcnNcbiAgICAgKi9cbiAgICByZWFkb25seSBzZXNzaW9uUGlubmluZ0ZpbHRlcnM/OiBTZXNzaW9uUGlubmluZ0ZpbHRlcltdO1xuICAgIC8qKlxuICAgICAqIFdoZXRoZXIgdGhlIHByb3h5IGluY2x1ZGVzIGRldGFpbGVkIGluZm9ybWF0aW9uIGFib3V0IFNRTCBzdGF0ZW1lbnRzIGluIGl0cyBsb2dzLlxuICAgICAqIFRoaXMgaW5mb3JtYXRpb24gaGVscHMgeW91IHRvIGRlYnVnIGlzc3VlcyBpbnZvbHZpbmcgU1FMIGJlaGF2aW9yIG9yIHRoZSBwZXJmb3JtYW5jZSBhbmQgc2NhbGFiaWxpdHkgb2YgdGhlIHByb3h5IGNvbm5lY3Rpb25zLlxuICAgICAqIFRoZSBkZWJ1ZyBpbmZvcm1hdGlvbiBpbmNsdWRlcyB0aGUgdGV4dCBvZiBTUUwgc3RhdGVtZW50cyB0aGF0IHlvdSBzdWJtaXQgdGhyb3VnaCB0aGUgcHJveHkuXG4gICAgICogVGh1cywgb25seSBlbmFibGUgdGhpcyBzZXR0aW5nIHdoZW4gbmVlZGVkIGZvciBkZWJ1Z2dpbmcsIGFuZCBvbmx5IHdoZW4geW91IGhhdmUgc2VjdXJpdHkgbWVhc3VyZXMgaW4gcGxhY2UgdG8gc2FmZWd1YXJkIGFueSBzZW5zaXRpdmVcbiAgICAgKiBpbmZvcm1hdGlvbiB0aGF0IGFwcGVhcnMgaW4gdGhlIGxvZ3MuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGRlYnVnTG9nZ2luZz86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogV2hldGhlciB0byByZXF1aXJlIG9yIGRpc2FsbG93IEFXUyBJZGVudGl0eSBhbmQgQWNjZXNzIE1hbmFnZW1lbnQgKElBTSkgYXV0aGVudGljYXRpb24gZm9yIGNvbm5lY3Rpb25zIHRvIHRoZSBwcm94eS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgaWFtQXV0aD86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogVGhlIG51bWJlciBvZiBzZWNvbmRzIHRoYXQgYSBjb25uZWN0aW9uIHRvIHRoZSBwcm94eSBjYW4gYmUgaW5hY3RpdmUgYmVmb3JlIHRoZSBwcm94eSBkaXNjb25uZWN0cyBpdC5cbiAgICAgKiBZb3UgY2FuIHNldCB0aGlzIHZhbHVlIGhpZ2hlciBvciBsb3dlciB0aGFuIHRoZSBjb25uZWN0aW9uIHRpbWVvdXQgbGltaXQgZm9yIHRoZSBhc3NvY2lhdGVkIGRhdGFiYXNlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgY2RrLkR1cmF0aW9uLm1pbnV0ZXMoMzApXG4gICAgICovXG4gICAgcmVhZG9ubHkgaWRsZUNsaWVudFRpbWVvdXQ/OiBjZGsuRHVyYXRpb247XG4gICAgLyoqXG4gICAgICogQSBCb29sZWFuIHBhcmFtZXRlciB0aGF0IHNwZWNpZmllcyB3aGV0aGVyIFRyYW5zcG9ydCBMYXllciBTZWN1cml0eSAoVExTKSBlbmNyeXB0aW9uIGlzIHJlcXVpcmVkIGZvciBjb25uZWN0aW9ucyB0byB0aGUgcHJveHkuXG4gICAgICogQnkgZW5hYmxpbmcgdGhpcyBzZXR0aW5nLCB5b3UgY2FuIGVuZm9yY2UgZW5jcnlwdGVkIFRMUyBjb25uZWN0aW9ucyB0byB0aGUgcHJveHkuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCB0cnVlXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmVxdWlyZVRMUz86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogSUFNIHJvbGUgdGhhdCB0aGUgcHJveHkgdXNlcyB0byBhY2Nlc3Mgc2VjcmV0cyBpbiBBV1MgU2VjcmV0cyBNYW5hZ2VyLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBBIHJvbGUgd2lsbCBhdXRvbWF0aWNhbGx5IGJlIGNyZWF0ZWRcbiAgICAgKi9cbiAgICByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuICAgIC8qKlxuICAgICAqIFRoZSBzZWNyZXQgdGhhdCB0aGUgcHJveHkgdXNlcyB0byBhdXRoZW50aWNhdGUgdG8gdGhlIFJEUyBEQiBpbnN0YW5jZSBvciBBdXJvcmEgREIgY2x1c3Rlci5cbiAgICAgKiBUaGVzZSBzZWNyZXRzIGFyZSBzdG9yZWQgd2l0aGluIEFtYXpvbiBTZWNyZXRzIE1hbmFnZXIuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIG5vIHNlY3JldFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNlY3JldDogc2VjcmV0c21hbmFnZXIuSVNlY3JldDtcbiAgICAvKipcbiAgICAgKiBPbmUgb3IgbW9yZSBWUEMgc2VjdXJpdHkgZ3JvdXBzIHRvIGFzc29jaWF0ZSB3aXRoIHRoZSBuZXcgcHJveHkuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIHNlY3VyaXR5IGdyb3Vwc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBzPzogZWMyLklTZWN1cml0eUdyb3VwW107XG4gICAgLyoqXG4gICAgICogVGhlIHN1Ym5ldHMgdXNlZCBieSB0aGUgcHJveHkuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIHRoZSBWUEMgZGVmYXVsdCBzdHJhdGVneSBpZiBub3Qgc3BlY2lmaWVkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHZwY1N1Ym5ldHM/OiBlYzIuU3VibmV0U2VsZWN0aW9uO1xuICAgIC8qKlxuICAgICAqIFRoZSBWUEMgdG8gYXNzb2NpYXRlIHdpdGggdGhlIG5ldyBwcm94eS5cbiAgICAgKi9cbiAgICByZWFkb25seSB2cGM6IGVjMi5JVnBjO1xufVxuLyoqXG4gKiBDb25zdHJ1Y3Rpb24gcHJvcGVydGllcyBmb3IgYSBEYXRhYmFzZVByb3h5XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRGF0YWJhc2VQcm94eVByb3BzIGV4dGVuZHMgRGF0YWJhc2VQcm94eU9wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIERCIHByb3h5IHRhcmdldDogSW5zdGFuY2Ugb3IgQ2x1c3RlclxuICAgICAqL1xuICAgIHJlYWRvbmx5IHByb3h5VGFyZ2V0OiBQcm94eVRhcmdldDtcbn1cbi8qKlxuICogUHJvcGVydGllcyB0aGF0IGRlc2NyaWJlIGFuIGV4aXN0aW5nIERCIFByb3h5XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRGF0YWJhc2VQcm94eUF0dHJpYnV0ZXMge1xuICAgIC8qKlxuICAgICAqIERCIFByb3h5IE5hbWVcbiAgICAgKi9cbiAgICByZWFkb25seSBkYlByb3h5TmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIERCIFByb3h5IEFSTlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGRiUHJveHlBcm46IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBFbmRwb2ludFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGVuZHBvaW50OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHNlY3VyaXR5IGdyb3VwcyBvZiB0aGUgaW5zdGFuY2UuXG4gICAgICovXG4gICAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cHM6IGVjMi5JU2VjdXJpdHlHcm91cFtdO1xufVxuLyoqXG4gKiBEQiBQcm94eVxuICovXG5leHBvcnQgaW50ZXJmYWNlIElEYXRhYmFzZVByb3h5IGV4dGVuZHMgY2RrLklSZXNvdXJjZSB7XG4gICAgLyoqXG4gICAgICogREIgUHJveHkgTmFtZVxuICAgICAqXG4gICAgICogQGF0dHJpYnV0ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGRiUHJveHlOYW1lOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogREIgUHJveHkgQVJOXG4gICAgICpcbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGJQcm94eUFybjogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIEVuZHBvaW50XG4gICAgICpcbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcmVhZG9ubHkgZW5kcG9pbnQ6IHN0cmluZztcbn1cbi8qKlxuICogUkRTIERhdGFiYXNlIFByb3h5XG4gKlxuICogQHJlc291cmNlIEFXUzo6UkRTOjpEQlByb3h5XG4gKi9cbmV4cG9ydCBjbGFzcyBEYXRhYmFzZVByb3h5IGV4dGVuZHMgY2RrLlJlc291cmNlIGltcGxlbWVudHMgSURhdGFiYXNlUHJveHksIGVjMi5JQ29ubmVjdGFibGUsIHNlY3JldHNtYW5hZ2VyLklTZWNyZXRBdHRhY2htZW50VGFyZ2V0IHtcbiAgICAvKipcbiAgICAgKiBJbXBvcnQgYW4gZXhpc3RpbmcgZGF0YWJhc2UgcHJveHkuXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBmcm9tRGF0YWJhc2VQcm94eUF0dHJpYnV0ZXMoc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBEYXRhYmFzZVByb3h5QXR0cmlidXRlcyk6IElEYXRhYmFzZVByb3h5IHtcbiAgICAgICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgY2RrLlJlc291cmNlIGltcGxlbWVudHMgSURhdGFiYXNlUHJveHkge1xuICAgICAgICAgICAgcHVibGljIHJlYWRvbmx5IGRiUHJveHlOYW1lID0gYXR0cnMuZGJQcm94eU5hbWU7XG4gICAgICAgICAgICBwdWJsaWMgcmVhZG9ubHkgZGJQcm94eUFybiA9IGF0dHJzLmRiUHJveHlBcm47XG4gICAgICAgICAgICBwdWJsaWMgcmVhZG9ubHkgZW5kcG9pbnQgPSBhdHRycy5lbmRwb2ludDtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBEQiBQcm94eSBOYW1lXG4gICAgICpcbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGRiUHJveHlOYW1lOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogREIgUHJveHkgQVJOXG4gICAgICpcbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGRiUHJveHlBcm46IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBFbmRwb2ludFxuICAgICAqXG4gICAgICogQGF0dHJpYnV0ZVxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBlbmRwb2ludDogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIEFjY2VzcyB0byBuZXR3b3JrIGNvbm5lY3Rpb25zLlxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBjb25uZWN0aW9uczogZWMyLkNvbm5lY3Rpb25zO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgcmVzb3VyY2U6IENmbkRCUHJveHk7XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBEYXRhYmFzZVByb3h5UHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkLCB7IHBoeXNpY2FsTmFtZTogcHJvcHMuZGJQcm94eU5hbWUgfHwgaWQgfSk7XG4gICAgICAgIGNvbnN0IHJvbGUgPSBwcm9wcy5yb2xlIHx8IG5ldyBpYW0uUm9sZSh0aGlzLCAnSUFNUm9sZScsIHtcbiAgICAgICAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdyZHMuYW1hem9uYXdzLmNvbScpLFxuICAgICAgICB9KTtcbiAgICAgICAgcHJvcHMuc2VjcmV0LmdyYW50UmVhZChyb2xlKTtcbiAgICAgICAgdGhpcy5jb25uZWN0aW9ucyA9IG5ldyBlYzIuQ29ubmVjdGlvbnMoeyBzZWN1cml0eUdyb3VwczogcHJvcHMuc2VjdXJpdHlHcm91cHMgfSk7XG4gICAgICAgIGNvbnN0IGJpbmRSZXN1bHQgPSBwcm9wcy5wcm94eVRhcmdldC5iaW5kKHRoaXMpO1xuICAgICAgICB0aGlzLnJlc291cmNlID0gbmV3IENmbkRCUHJveHkodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgICAgICAgYXV0aDogW1xuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgYXV0aFNjaGVtZTogJ1NFQ1JFVFMnLFxuICAgICAgICAgICAgICAgICAgICBpYW1BdXRoOiBwcm9wcy5pYW1BdXRoID8gJ1JFUVVJUkVEJyA6ICdESVNBQkxFRCcsXG4gICAgICAgICAgICAgICAgICAgIHNlY3JldEFybjogcHJvcHMuc2VjcmV0LnNlY3JldEFybixcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIGRiUHJveHlOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgICAgICAgIGRlYnVnTG9nZ2luZzogcHJvcHMuZGVidWdMb2dnaW5nLFxuICAgICAgICAgICAgZW5naW5lRmFtaWx5OiBiaW5kUmVzdWx0LmVuZ2luZUZhbWlseSxcbiAgICAgICAgICAgIGlkbGVDbGllbnRUaW1lb3V0OiBwcm9wcy5pZGxlQ2xpZW50VGltZW91dD8udG9TZWNvbmRzKCksXG4gICAgICAgICAgICByZXF1aXJlVGxzOiBwcm9wcy5yZXF1aXJlVExTID8/IHRydWUsXG4gICAgICAgICAgICByb2xlQXJuOiByb2xlLnJvbGVBcm4sXG4gICAgICAgICAgICB2cGNTZWN1cml0eUdyb3VwSWRzOiBwcm9wcy5zZWN1cml0eUdyb3Vwcz8ubWFwKF8gPT4gXy5zZWN1cml0eUdyb3VwSWQpLFxuICAgICAgICAgICAgdnBjU3VibmV0SWRzOiBwcm9wcy52cGMuc2VsZWN0U3VibmV0cyhwcm9wcy52cGNTdWJuZXRzKS5zdWJuZXRJZHMsXG4gICAgICAgIH0pO1xuICAgICAgICB0aGlzLmRiUHJveHlOYW1lID0gdGhpcy5yZXNvdXJjZS5yZWY7XG4gICAgICAgIHRoaXMuZGJQcm94eUFybiA9IHRoaXMucmVzb3VyY2UuYXR0ckRiUHJveHlBcm47XG4gICAgICAgIHRoaXMuZW5kcG9pbnQgPSB0aGlzLnJlc291cmNlLmF0dHJFbmRwb2ludDtcbiAgICAgICAgbGV0IGRiSW5zdGFuY2VJZGVudGlmaWVyczogc3RyaW5nW10gfCB1bmRlZmluZWQ7XG4gICAgICAgIGlmIChiaW5kUmVzdWx0LmRiQ2x1c3RlcnMpIHtcbiAgICAgICAgICAgIC8vIHN1cHBvcnQgZm9yIG9ubHkgaW5zdGFuY2VzIG9mIGEgc2luZ2xlIGNsdXN0ZXJcbiAgICAgICAgICAgIGRiSW5zdGFuY2VJZGVudGlmaWVycyA9IGJpbmRSZXN1bHQuZGJDbHVzdGVyc1swXS5pbnN0YW5jZUlkZW50aWZpZXJzO1xuICAgICAgICB9XG4gICAgICAgIGVsc2UgaWYgKGJpbmRSZXN1bHQuZGJJbnN0YW5jZXMpIHtcbiAgICAgICAgICAgIC8vIHN1cHBvcnQgZm9yIG9ubHkgc2luZ2xlIGluc3RhbmNlXG4gICAgICAgICAgICBkYkluc3RhbmNlSWRlbnRpZmllcnMgPSBbYmluZFJlc3VsdC5kYkluc3RhbmNlc1swXS5pbnN0YW5jZUlkZW50aWZpZXJdO1xuICAgICAgICB9XG4gICAgICAgIG5ldyBDZm5EQlByb3h5VGFyZ2V0R3JvdXAodGhpcywgJ1Byb3h5VGFyZ2V0R3JvdXAnLCB7XG4gICAgICAgICAgICBkYlByb3h5TmFtZTogdGhpcy5kYlByb3h5TmFtZSxcbiAgICAgICAgICAgIGRiSW5zdGFuY2VJZGVudGlmaWVycyxcbiAgICAgICAgICAgIGRiQ2x1c3RlcklkZW50aWZpZXJzOiBiaW5kUmVzdWx0LmRiQ2x1c3RlcnM/Lm1hcCgoYykgPT4gYy5jbHVzdGVySWRlbnRpZmllciksXG4gICAgICAgICAgICBjb25uZWN0aW9uUG9vbENvbmZpZ3VyYXRpb25JbmZvOiB0b0Nvbm5lY3Rpb25Qb29sQ29uZmlndXJhdGlvbkluZm8ocHJvcHMpLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmVuZGVycyB0aGUgc2VjcmV0IGF0dGFjaG1lbnQgdGFyZ2V0IHNwZWNpZmljYXRpb25zLlxuICAgICAqL1xuICAgIHB1YmxpYyBhc1NlY3JldEF0dGFjaG1lbnRUYXJnZXQoKTogc2VjcmV0c21hbmFnZXIuU2VjcmV0QXR0YWNobWVudFRhcmdldFByb3BzIHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIHRhcmdldElkOiB0aGlzLmRiUHJveHlOYW1lLFxuICAgICAgICAgICAgdGFyZ2V0VHlwZTogc2VjcmV0c21hbmFnZXIuQXR0YWNobWVudFRhcmdldFR5cGUuUkRTX0RCX1BST1hZLFxuICAgICAgICB9O1xuICAgIH1cbn1cbi8qKlxuICogQ29ubmVjdGlvblBvb2xDb25maWd1cmF0aW9uIChMMiA9PiBMMSlcbiAqL1xuZnVuY3Rpb24gdG9Db25uZWN0aW9uUG9vbENvbmZpZ3VyYXRpb25JbmZvKHByb3BzOiBEYXRhYmFzZVByb3h5UHJvcHMpOiBDZm5EQlByb3h5VGFyZ2V0R3JvdXAuQ29ubmVjdGlvblBvb2xDb25maWd1cmF0aW9uSW5mb0Zvcm1hdFByb3BlcnR5IHtcbiAgICByZXR1cm4ge1xuICAgICAgICBjb25uZWN0aW9uQm9ycm93VGltZW91dDogcHJvcHMuYm9ycm93VGltZW91dD8udG9TZWNvbmRzKCksXG4gICAgICAgIGluaXRRdWVyeTogcHJvcHMuaW5pdFF1ZXJ5LFxuICAgICAgICBtYXhDb25uZWN0aW9uc1BlcmNlbnQ6IHByb3BzLm1heENvbm5lY3Rpb25zUGVyY2VudCxcbiAgICAgICAgbWF4SWRsZUNvbm5lY3Rpb25zUGVyY2VudDogcHJvcHMubWF4SWRsZUNvbm5lY3Rpb25zUGVyY2VudCxcbiAgICAgICAgc2Vzc2lvblBpbm5pbmdGaWx0ZXJzOiBwcm9wcy5zZXNzaW9uUGlubmluZ0ZpbHRlcnM/Lm1hcChfID0+IF8uZmlsdGVyTmFtZSksXG4gICAgfTtcbn1cbiJdfQ==