"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OptionGroup = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rds_generated_1 = require("./rds.generated");
/**
 * An option group
 */
class OptionGroup extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The connections object for the options.
         */
        this.optionConnections = {};
        const optionGroup = new rds_generated_1.CfnOptionGroup(this, 'Resource', {
            engineName: props.engine.name,
            majorEngineVersion: props.majorEngineVersion,
            optionGroupDescription: props.description || `Option group for ${props.engine.name} ${props.majorEngineVersion}`,
            optionConfigurations: this.renderConfigurations(props.configurations),
        });
        this.optionGroupName = optionGroup.ref;
    }
    /**
     * Import an existing option group.
     */
    static fromOptionGroupName(scope, id, optionGroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.optionGroupName = optionGroupName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the option configurations specifications.
     */
    renderConfigurations(configurations) {
        const configs = [];
        for (const config of configurations) {
            let configuration = {
                optionName: config.name,
                optionSettings: config.settings && Object.entries(config.settings).map(([name, value]) => ({ name, value })),
                optionVersion: config.version,
            };
            if (config.port) {
                if (!config.vpc) {
                    throw new Error('`port` and `vpc` must be specified together.');
                }
                const securityGroup = new ec2.SecurityGroup(this, `SecurityGroup${config.name}`, {
                    description: `Security group for ${config.name} option`,
                    vpc: config.vpc,
                });
                this.optionConnections[config.name] = new ec2.Connections({
                    securityGroups: [securityGroup],
                    defaultPort: ec2.Port.tcp(config.port),
                });
                configuration = {
                    ...configuration,
                    port: config.port,
                    vpcSecurityGroupMemberships: [securityGroup.securityGroupId],
                };
            }
            configs.push(configuration);
        }
        return configs;
    }
}
exports.OptionGroup = OptionGroup;
//# sourceMappingURL=data:application/json;base64,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